/*
 * vectors.S -- Exception vector code for ARM(R) BSP
 *
 * Copyright (c) 1998, 1999, 2000 Cygnus Support
 *
 * The authors hereby grant permission to use, copy, modify, distribute,
 * and license this software and its documentation for any purpose, provided
 * that existing copyright notices are retained in all copies and that this
 * notice is included verbatim in any distributions. No written agreement,
 * license, or royalty fee is required for any of the authorized uses.
 * Modifications to this software may be copyrighted by their authors
 * and need not follow the licensing terms described here, provided that
 * the new terms are clearly indicated on the first page of each file where
 * they apply.
 *
 * ARM is a Registered Trademark of Advanced RISC Machines Limited.
 * Other Brands and Trademarks are the property of their respective owners.
 */

#include <bsp/cpu.h>

	.file	"vectors.S"

	.text

/***************************************
 * __usr_svc -- Handle user-mode SWI   *
 *                                     *
 * r0 points to our temp save area:    *
 *				       *
 *  +20:  pre-exception r1_priv	       *
 *  +16:  pre-exception r0_priv	       *
 *  +12:  pre-exception cpsr	       *
 *   +8:  pre-exception pc	       *
 *   +4:  exc_nr		       *
 *   +0:  pre-exception sp_svc         *
 **************************************/
__usr_svc:

	/* switch to exception stack */
	ldr	r1, =SYM_NAME(_ex_stack)
	cmp	sp, r1
	subhi	sp, r1, IMM(ARM_EX_REGS_T_SIZE)
	subls	sp, sp, IMM(ARM_EX_REGS_T_SIZE)

	/* save user mode regs */
	stmia	sp, {r0-r12,sp,lr}^
	nop

	/* fixup regs from temp save area */
	ldmfd	r0, {r0-r5}
	str	r0, [sp, IMM(spsvc_o)]
	str	r2, [sp, IMM(pc_o)]
	str	r3, [sp, IMM(cpsr_o)]

	/*
	 * void generic_exception_handler(ex_regs_t *regs,
	 *				  unsigned long vect_num)
	 */
	mov	r0, sp
	bl	SYM_NAME(generic_exception_handler)

	/* restore user regs */
	add	r0, sp, IMM(r0_o)
	ldmia	r0, {r0-r12,sp,lr}^
	nop
	
	/* return to usr mode */
	ldr	r0, [sp, IMM(spsvc_o)]
	str	r0, [sp, IMM(pc_o) - 4]
	ldr	r0, [sp, IMM(cpsr_o)]
	msr	spsr, r0
	add	sp, sp, IMM(pc_o) - 4
	ldmia	sp, {sp,pc}^

/********************************************
 * __svc_svc -- Handle supervisor-mode SWI  *
 *                                          *
 * r0 points to our temp save area:         *
 *				            *
 *  +20:  pre-exception r1	            *
 *  +16:  pre-exception r0	            *
 *  +12:  pre-exception cpsr	            *
 *   +8:  pre-exception pc	            *
 *   +4:  exc_nr		            *
 *   +0:  pre-exception sp_svc              *
 ********************************************/
__svc_svc:
	/* switch to exception stack */
	ldr	r1, =SYM_NAME(_ex_stack)
	cmp	sp, r1
	subhi	sp, r1, IMM(ARM_EX_REGS_T_SIZE)
	subls	sp, sp, IMM(ARM_EX_REGS_T_SIZE)

	stmia	sp, {r0-r12,sp,lr}

	/* fixup regs from temp save area */
	ldmfd	r0, {r0-r5}
	str	r0, [sp, IMM(sp_o)]
	str	r2, [sp, IMM(pc_o)]
	str	r3, [sp, IMM(cpsr_o)]
	str	r4, [sp, IMM(r0_o)]
	str	r5, [sp, IMM(r1_o)]

	/*
	 * void generic_exception_handler(ex_regs_t *regs,
	 *				  unsigned long vect_num)
	 */
	mov	r0, sp
	bl	SYM_NAME(generic_exception_handler)

	/* return to svc mode */
	ldr	r0, [sp, IMM(cpsr_o)]
	msr	spsr, r0
	ldmia	sp, {r0-r12,sp,lr,pc}^


__priv_svc:
	b __priv_svc	


/**********************************************
 * __usr_priv -- Handle user mode exceptions. *
 *				              *
 * r0 points to our temp save area:           *
 *				              *
 *  +16:  saved r1		              *
 *  +12:  saved r0		              *
 *   +8:  pre-exception cpsr	              *
 *   +4:  pre-exception pc	              *
 *   +0:  exc_nr		              *
 **********************************************/
__usr_priv:
	stmfd	r0!, {sp}    /* save svc sp in temp area */

	/* switch to exception stack */
	ldr	r1, =SYM_NAME(_ex_stack)
	cmp	sp, r1
	subhi	sp, r1, IMM(ARM_EX_REGS_T_SIZE)
	subls	sp, sp, IMM(ARM_EX_REGS_T_SIZE)

	/* regs from temp save area */
	ldmfd	r0, {r0-r3}
	str	r0, [sp, IMM(spsvc_o)]
	str	r2, [sp, IMM(pc_o)]
	str	r3, [sp, IMM(cpsr_o)]

	/* save user mode regs */
	add	r0, sp, IMM(r0_o)
	stmia	r0, {r0-r12,sp,lr}^
	nop

	/*
	 * void generic_exception_handler(ex_regs_t *regs,
	 *				  unsigned long vect_num)
	 */
	mov	r0, sp
	bl	SYM_NAME(generic_exception_handler)

	/* restore user regs */
	add	r0, sp, IMM(r0_o)
	ldmia	r0, {r0-r12,sp,lr}^
	nop

	/* return to user mode */
	ldr	r0, [sp, IMM(spsvc_o)]
	str	r0, [sp, IMM(pc_o) - 4]
	ldr	r0, [sp, IMM(cpsr_o)]
	msr	spsr, r0
	add	sp, sp, IMM(pc_o) - 4
	ldmia	sp, {sp,pc}^

/****************************************************
 * __svc_priv -- Handle supervisor mode exceptions. *
 *				                    *
 * r0 points to our temp save area:                 *
 *				                    *
 *  +16:  saved r1		                    *
 *  +12:  saved r0		                    *
 *   +8:  pre-exception cpsr	                    *
 *   +4:  pre-exception pc	                    *
 *   +0:  exc_nr		                    *
 ****************************************************/
__svc_priv:
	stmfd	r0!, {sp}    /* save svc sp in temp area */
	
	/* switch to exception stack */
	ldr	r1, =SYM_NAME(_ex_stack)
	cmp	sp, r1
	subhi	sp, r1, IMM(ARM_EX_REGS_T_SIZE)
	subls	sp, sp, IMM(ARM_EX_REGS_T_SIZE)

	stmia	sp, {r0-r12,sp,lr}

	/* regs from temp save area */
	ldmfd	r0, {r0-r5}
	str	r0, [sp, IMM(sp_o)]
	str	r2, [sp, IMM(pc_o)]
	str	r3, [sp, IMM(cpsr_o)]
	str	r4, [sp, IMM(r0_o)]
	str	r5, [sp, IMM(r1_o)]

	/*
	 * void generic_exception_handler(ex_regs_t *regs,
	 *				  unsigned long vect_num)
	 */
	mov	r0, sp
	bl	SYM_NAME(generic_exception_handler)

	/* return to svc mode */
	ldr	r0, [sp, IMM(cpsr_o)]
	msr	spsr, r0
	ldmia	sp, {r0-r12,sp,lr,pc}^

__priv_priv:
	b __priv_priv


/*
 * This is the Undefined instruction interrupt handler
 */
FUNC_START undefined_isr_asm
	/*
	 * Set up a temp stack to use
	 */
	ldr	sp, =SYM_NAME(_undefined_stack)
	
	stmfd	sp!, {r0-r1}
	mov	r0, lr	     /* pre-exception pc     */
	mrs	r1, spsr     /* pre-exception  cpsr  */
	stmfd	sp!, {r0-r1}
	ldr	r0, =BSP_CORE_EXC_UNDEFINED_INSTRUCTION
	stmfd	sp!, {r0}
	mov	r0, sp	     /* pointer to stuff we just saved */

	mov	r13, #ARM_PSR_IRQ | ARM_PSR_MODE_SVC
	msr	spsr, r13

	and	r1, r1, #15
	ldr	r1, [pc, r1, lsl #2]
	movs	pc, r1

	.word	__usr_priv	/* user */
	.word	__priv_priv	/* fiq  */
	.word	__priv_priv	/* irq  */
	.word	__svc_priv	/* svc  */
	.word	__priv_priv
	.word	__priv_priv
	.word	__priv_priv
	.word	__priv_priv	/* abort */
	.word	__priv_priv
	.word	__priv_priv
	.word	__priv_priv
	.word	__priv_priv	/* undef */
	.word	__priv_priv
	.word	__priv_priv
	.word	__priv_priv
	.word	__usr_priv	/* system */	
FUNC_END undefined_isr_asm


/*
 * This is the SWI interrupt handler
 */
FUNC_START swi_isr_asm
	stmfd	sp!, {r0-r1}
	mov	r0, lr			/* pre-exception pc     */
	mrs	r1, spsr     		/* pre-exception  cpsr  */
	stmfd	sp!, {r0-r1}
	ldr	r0, =BSP_CORE_EXC_SOFTWARE_INTERRUPT
	stmfd	sp!, {r0}
	add	r0, sp, IMM(4 * 4)	/* pre-exception sp */
	stmfd	sp!, {r0}
	mov	r0,sp

	and	r1, r1, #15
	ldr	r1, [pc, r1, lsl #2]
	mov	pc, r1

	.word	__usr_svc	/* user */
	.word	__priv_svc	/* fiq  */
	.word	__priv_svc	/* irq  */
	.word	__svc_svc	/* svc  */
	.word	__priv_svc
	.word	__priv_svc
	.word	__priv_svc
	.word	__priv_svc	/* abort */
	.word	__priv_svc
	.word	__priv_svc
	.word	__priv_svc
	.word	__priv_svc	/* undef */
	.word	__priv_svc
	.word	__priv_svc
	.word	__priv_svc
	.word	__usr_svc	/* system */	
FUNC_END swi_isr_asm


/*
 * This is the Prefetch Abort interrupt handler
 */
FUNC_START prefetch_abort_isr_asm
	/*
	 * Set up a temp stack to use
	 */
	ldr	sp, =SYM_NAME(_abort_stack)
	sub	lr, lr, IMM(4)
	
	stmfd	sp!, {r0-r1}
	mov	r0, lr	     /* pre-exception pc     */
	mrs	r1, spsr     /* pre-exception  cpsr  */
	stmfd	sp!, {r0-r1}
	ldr	r0, =BSP_CORE_EXC_PREFETCH_ABORT
	stmfd	sp!, {r0}
	mov	r0, sp	     /* pointer to stuff we just saved */

	mov	r13, #ARM_PSR_IRQ | ARM_PSR_MODE_SVC
	msr	spsr, r13

	and	r1, r1, #15
	ldr	r1, [pc, r1, lsl #2]
	movs	pc, r1

	.word	__usr_priv	/* user */
	.word	__priv_priv	/* fiq  */
	.word	__priv_priv	/* irq  */
	.word	__svc_priv	/* svc  */
	.word	__priv_priv
	.word	__priv_priv
	.word	__priv_priv
	.word	__priv_priv	/* abort */
	.word	__priv_priv
	.word	__priv_priv
	.word	__priv_priv
	.word	__priv_priv	/* undef */
	.word	__priv_priv
	.word	__priv_priv
	.word	__priv_priv
	.word	__usr_priv	/* system */	
FUNC_END prefetch_abort_isr_asm


/*
 * This is the Data Abort interrupt handler
 */
FUNC_START data_abort_isr_asm
	/*
	 * Set up a temp stack to use
	 */
	ldr	sp, =SYM_NAME(_abort_stack)
	sub	lr, lr, IMM(4)
	
	stmfd	sp!, {r0-r1}
	mov	r0, lr	     /* pre-exception pc     */
	mrs	r1, spsr     /* pre-exception  cpsr  */
	stmfd	sp!, {r0-r1}
	ldr	r0, =BSP_CORE_EXC_DATA_ABORT
	stmfd	sp!, {r0}
	mov	r0, sp	     /* pointer to stuff we just saved */

	mov	r13, #ARM_PSR_IRQ | ARM_PSR_MODE_SVC
	msr	spsr, r13

	and	r1, r1, #15
	ldr	r1, [pc, r1, lsl #2]
	movs	pc, r1

	.word	__usr_priv	/* user */
	.word	__priv_priv	/* fiq  */
	.word	__priv_priv	/* irq  */
	.word	__svc_priv	/* svc  */
	.word	__priv_priv
	.word	__priv_priv
	.word	__priv_priv
	.word	__priv_priv	/* abort */
	.word	__priv_priv
	.word	__priv_priv
	.word	__priv_priv
	.word	__priv_priv	/* undef */
	.word	__priv_priv
	.word	__priv_priv
	.word	__priv_priv
	.word	__usr_priv	/* system */	
FUNC_END data_abort_isr_asm


/*
 * This is the 26-bit mode Address Error interrupt handler
 */
FUNC_START address_err_isr_asm
	/*
	 * Set up a temp stack to use
	 */
	ldr	sp, =SYM_NAME(_address_err_stack)
	sub	lr, lr, IMM(4)
	
	stmfd	sp!, {r0-r1}
	mov	r0, lr	     /* pre-exception pc     */
	mrs	r1, spsr     /* pre-exception  cpsr  */
	stmfd	sp!, {r0-r1}
	ldr	r0, =BSP_CORE_EXC_ADDRESS_ERROR_26_BIT
	stmfd	sp!, {r0}
	mov	r0, sp	     /* pointer to stuff we just saved */

	mov	r13, #ARM_PSR_IRQ | ARM_PSR_MODE_SVC
	msr	spsr, r13

	and	r1, r1, #15
	ldr	r1, [pc, r1, lsl #2]
	movs	pc, r1

	.word	__usr_priv	/* user */
	.word	__priv_priv	/* fiq  */
	.word	__priv_priv	/* irq  */
	.word	__svc_priv	/* svc  */
	.word	__priv_priv
	.word	__priv_priv
	.word	__priv_priv
	.word	__priv_priv	/* abort */
	.word	__priv_priv
	.word	__priv_priv
	.word	__priv_priv
	.word	__priv_priv	/* undef */
	.word	__priv_priv
	.word	__priv_priv
	.word	__priv_priv
	.word	__usr_priv	/* system */	
FUNC_END address_err_isr_asm


/*
 * This is the IRQ interrupt handler
 */
FUNC_START irq_isr_asm
	/*
	 * Set up a temp stack to use
	 */
	ldr	sp, =SYM_NAME(_irq_stack)
	sub	lr, lr, IMM(4)
	
	stmfd	sp!, {r0-r1}
	mov	r0, lr	     /* pre-exception pc     */
	mrs	r1, spsr     /* pre-exception  cpsr  */
	stmfd	sp!, {r0-r1}
	ldr	r0, =BSP_CORE_EXC_IRQ
	stmfd	sp!, {r0}
	mov	r0, sp	     /* pointer to stuff we just saved */

	mov	r13, #ARM_PSR_IRQ | ARM_PSR_MODE_SVC
	msr	spsr, r13

	and	r1, r1, #15
	ldr	r1, [pc, r1, lsl #2]
	movs	pc, r1

	.word	__usr_priv	/* user */
	.word	__priv_priv	/* fiq  */
	.word	__priv_priv	/* irq  */
	.word	__svc_priv	/* svc  */
	.word	__priv_priv
	.word	__priv_priv
	.word	__priv_priv
	.word	__priv_priv	/* abort */
	.word	__priv_priv
	.word	__priv_priv
	.word	__priv_priv
	.word	__priv_priv	/* undef */
	.word	__priv_priv
	.word	__priv_priv
	.word	__priv_priv
	.word	__usr_priv	/* system */	
FUNC_END irq_isr_asm

/*
 * This is the FIQ interrupt handler
 */
FUNC_START fiq_isr_asm
	/*
	 * Set up a temp stack to use
	 */
	ldr	sp, =SYM_NAME(_fiq_stack)
	sub	lr, lr, IMM(4)
	
	stmfd	sp!, {r0-r1}
	mov	r0, lr	     /* pre-exception pc     */
	mrs	r1, spsr     /* pre-exception  cpsr  */
	stmfd	sp!, {r0-r1}
	ldr	r0, =BSP_CORE_EXC_FIQ
	stmfd	sp!, {r0}
	mov	r0, sp	     /* pointer to stuff we just saved */

	mov	r13, #ARM_PSR_IRQ | ARM_PSR_FIQ | ARM_PSR_MODE_SVC
	msr	spsr, r13

	and	r1, r1, #15
	ldr	r1, [pc, r1, lsl #2]
	movs	pc, r1

	.word	__usr_priv	/* user */
	.word	__priv_priv	/* fiq  */
	.word	__priv_priv	/* irq  */
	.word	__svc_priv	/* svc  */
	.word	__priv_priv
	.word	__priv_priv
	.word	__priv_priv
	.word	__priv_priv	/* abort */
	.word	__priv_priv
	.word	__priv_priv
	.word	__priv_priv
	.word	__priv_priv	/* undef */
	.word	__priv_priv
	.word	__priv_priv
	.word	__priv_priv
	.word	__usr_priv	/* system */	
FUNC_END fiq_isr_asm


/*
 * Assembly representing the exception vectors.
 */
	.section .ram_vectors

SYM_NAME(RESET_VECTOR):		  b	  SYM_NAME(RESET_VECTOR)
	.globl SYM_NAME(UNDEFINED_VECTOR)
SYM_NAME(UNDEFINED_VECTOR):	  ldr	  pc, SYM_NAME(ROM_UNDEFINED_ISR)
	.globl SYM_NAME(SWI_VECTOR)
SYM_NAME(SWI_VECTOR):		  ldr	  pc, SYM_NAME(ROM_SWI_ISR)
	.globl SYM_NAME(PREFETCH_ABORT_VECTOR)
SYM_NAME(PREFETCH_ABORT_VECTOR):  ldr	  pc, SYM_NAME(ROM_PREFETCH_ABORT_ISR)
	.globl SYM_NAME(DATA_ABORT_VECTOR)
SYM_NAME(DATA_ABORT_VECTOR):	  ldr	  pc, SYM_NAME(ROM_DATA_ABORT_ISR)
	.globl SYM_NAME(ADDR_ERROR_VECTOR)
SYM_NAME(ADDR_ERROR_VECTOR):	  ldr	  pc, SYM_NAME(ROM_ADDR_ERROR_ISR)
	.globl SYM_NAME(IRQ_VECTOR)
SYM_NAME(IRQ_VECTOR):		  ldr	  pc, SYM_NAME(ROM_IRQ_ISR)
	.globl SYM_NAME(FIQ_VECTOR)
SYM_NAME(FIQ_VECTOR):		  ldr	  pc, SYM_NAME(ROM_FIQ_ISR)

SYM_NAME(ROM_UNDEFINED_ISR):	  .word	  SYM_NAME(undefined_isr_asm)
SYM_NAME(ROM_SWI_ISR):		  .word	  SYM_NAME(swi_isr_asm)
SYM_NAME(ROM_PREFETCH_ABORT_ISR): .word	  SYM_NAME(prefetch_abort_isr_asm)
SYM_NAME(ROM_DATA_ABORT_ISR):	  .word	  SYM_NAME(data_abort_isr_asm)
SYM_NAME(ROM_ADDR_ERROR_ISR):	  .word	  SYM_NAME(address_err_isr_asm)
SYM_NAME(ROM_IRQ_ISR):		  .word	  SYM_NAME(irq_isr_asm)
SYM_NAME(ROM_FIQ_ISR):		  .word	  SYM_NAME(fiq_isr_asm)


#ifdef __YAVAPAI__
	/* we have to skip over Yavapai internal regs */
	.p2align 13
	.global SYM_NAME(__REAL_DATA_START)
SYM_NAME(__REAL_DATA_START):
#endif


#ifdef MMU
/*
 * These are the MMU page tables
 */
	.macro FL_SECTION_ENTRY base,x,ap,p,d,c,b
	.word (\base << 20) | (\x << 12) | (\ap << 10) | (\p << 9) | (\d << 5) | (\c << 3) | (\b << 2) | 2
	.endm

	/*
	 * I wanted to use logical operations here, but since I am using symbols later 
	 * to fill in the parameters, I had to use addition to force the assembler to
	 * do it right
	 */
	.macro FL_PT_ENTRY base,d
	.word \base + (\d << 5) + 1
	.endm

	.macro SL_SMPAGE_ENTRY base,ap3,ap2,ap1,ap0,c,b
	.word (\base << 12) | (\ap3 << 10) | (\ap2 << 8) | (\ap1 << 6) | (\ap0 << 4) | (\c << 3) | (\b << 2) | 2
	.endm

#if defined(__BOARD_XARET__)
#ifdef __ELF__
    /* "a" defines this section to be allocatable.  Without that, the binary objdump will strip this section */
	.section .rom_sec_table, "a"
#else
    /* "a" is an unknown attribute for coff targets. */
	.section .rom_sec_table
#endif
	.global SYM_NAME(page1)
SYM_NAME(page1):
	FL_SECTION_ENTRY 0x000,0,0,0,0,1,1
	FL_SECTION_ENTRY 0x001,0,3,0,0,0,0
	FL_SECTION_ENTRY 0x002,0,3,0,0,0,0
	FL_SECTION_ENTRY 0x003,0,3,0,0,0,0
	FL_SECTION_ENTRY 0x004,0,3,0,0,0,0
	FL_SECTION_ENTRY 0x005,0,3,0,0,0,0
	FL_SECTION_ENTRY 0x006,0,3,0,0,0,0
	FL_SECTION_ENTRY 0x007,0,3,0,0,0,0
	FL_SECTION_ENTRY 0x008,0,3,0,0,0,0
	FL_SECTION_ENTRY 0x009,0,3,0,0,0,0
	FL_SECTION_ENTRY 0x00A,0,3,0,0,0,0
	FL_SECTION_ENTRY 0x00B,0,3,0,0,0,0
	FL_SECTION_ENTRY 0x00C,0,3,0,0,0,0
	FL_SECTION_ENTRY 0x00D,0,3,0,0,0,0
	FL_SECTION_ENTRY 0x00E,0,3,0,0,0,0
	FL_SECTION_ENTRY 0x00F,0,3,0,0,0,0
	.set	__base,0x010
	.rept 0xA00 - 0x010
	FL_SECTION_ENTRY __base,0,3,0,0,0,0
	.set	__base,__base+1
	.endr
	.rept 0xA20 - 0xA00
	FL_SECTION_ENTRY __base,1,3,1,0,1,1
	.set	__base,__base+1
	.endr
	.rept 0x1000 - 0xA20
	FL_SECTION_ENTRY __base,0,3,0,0,0,0
	.set	__base,__base+1
	.endr
#elif defined(__BOARD_IQ80310__)
#ifdef __ELF__
    /* "a" defines this section to be allocatable.  Without that, the binary objdump will strip this section */
	.section .rom_sec_table, "a"
#else
    /* "a" is an unknown attribute for coff targets. */
	.section .rom_sec_table
#endif
	.global SYM_NAME(page1)

SYM_NAME(page1):
	/*
	 * 8MB of FLASH and i80307 MMRs mapped in using 4K small pages so we can
	 * set the access permission on flash and memory-mapped registers properly
	 */
	FL_PT_ENTRY SYM_NAME(page2),0
	
	.set	__base,1
	.rept	0x8 - 1
	FL_SECTION_ENTRY __base,0,3,0,0,1,0
	.set	__base,__base+1
	.endr	

	.rept	0xA00 - 0x8		/* nothing interesting here */
	FL_SECTION_ENTRY __base,0,3,0,0,0,0
	.set	__base,__base+1
	.endr

	.rept	0xC00 - 0xA00		/* 32MB ECC SDRAM */
	FL_SECTION_ENTRY __base,1,3,1,0,1,1	/* ECC, x=c=b=1 */
	.set	__base,__base+1
	.endr
	
	.rept	0x1000 - 0xC00		/* only I/O at 0xFE8xxxxx */
	FL_SECTION_ENTRY __base,0,3,0,0,0,0
	.set	__base,__base+1
	.endr

#ifdef __ELF__
    /* "a" defines this section to be allocatable.  Without that, the binary objdump will strip this section */
	.section .rom_smpage_table, "a"
#else
    /* "a" is an unknown attribute for coff targets. */
	.section .rom_smpage_table
#endif
	.global SYM_NAME(page2)
SYM_NAME(page2):
	/*
	 * This page table breaks the LO 8MB down into 4KB pages
	 */
	.set	__base,0x0

	SL_SMPAGE_ENTRY __base,3,3,3,3,1,0 /* Physical address 0 (Flash boot code). */
	.set	__base,__base+1		   /* Read-only, cacheable, non-bufferable  */

	SL_SMPAGE_ENTRY __base,3,3,3,3,0,0 /* Physical address 0x1000 (Memory mapped registers). */
	.set	__base,__base+1		   /* Read-write, non-cacheable, non-bufferable		 */

	.rept	0x800 - 0x2
	SL_SMPAGE_ENTRY __base,3,3,3,3,1,0 /* Physical address 0x2000-0x800000 (remainder of flash1). */
	.set	__base,__base+1		   /* Read-only, cacheable, non-bufferable		      */
	.endr
#else
	.section .arm_page_table
	.global SYM_NAME(page1)
SYM_NAME(page1): .rept ARM_FIRST_LEVEL_PAGE_TABLE_SIZE
		 .byte 0
		 .endr
#endif /* XARET */
#endif /* MMU */

/*
 * Macro Definition for an EXCEPTION stack
 */
.macro EXCEPTION_STACK label, size
	.bss
	.global SYM_NAME(\label)
	.align 4
	.rept	\size
	.byte	0
	.endr
SYM_NAME(\label):
.endm


/*
 * Provide a stack for use by exception processing
 */
	EXCEPTION_STACK _ex_stack, 16*1024

/*
 * Provide a simple stack for use in each of the
 * exception modes.  These are pretty small since
 * all we do is immediately switch to the exception
 * stack
 */
#define TMP_STKSIZE 32
	EXCEPTION_STACK _abort_stack, TMP_STKSIZE
	EXCEPTION_STACK _address_err_stack, TMP_STKSIZE
	EXCEPTION_STACK _undefined_stack, TMP_STKSIZE
	EXCEPTION_STACK _irq_stack, TMP_STKSIZE
	EXCEPTION_STACK _fiq_stack, TMP_STKSIZE
