/* Generate code from machine description to compute values of attributes.
   Copyright (C) 1991, 93-98, 1999 Free Software Foundation, Inc.
   Contributed by Richard Kenner (kenner@vlsi1.ultra.nyu.edu)

This file is part of GNU CC.

GNU CC is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU CC is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU CC; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

/* This program handles insn attributes and the DEFINE_DELAY and
   DEFINE_FUNCTION_UNIT definitions.

   It produces a series of functions named `get_attr_...', one for each insn
   attribute.  Each of these is given the rtx for an insn and returns a member
   of the enum for the attribute.

   These subroutines have the form of a `switch' on the INSN_CODE (via
   `recog_memoized').  Each case either returns a constant attribute value
   or a value that depends on tests on other attributes, the form of
   operands, or some random C expression (encoded with a SYMBOL_REF
   expression).

   If the attribute `alternative', or a random C expression is present,
   `constrain_operands' is called.  If either of these cases of a reference to
   an operand is found, `extract_insn' is called.

   The special attribute `length' is also recognized.  For this operand, 
   expressions involving the address of an operand or the current insn,
   (address (pc)), are valid.  In this case, an initial pass is made to
   set all lengths that do not depend on address.  Those that do are set to
   the maximum length.  Then each insn that depends on an address is checked
   and possibly has its length changed.  The process repeats until no further
   changed are made.  The resulting lengths are saved for use by
   `get_attr_length'.

   A special form of DEFINE_ATTR, where the expression for default value is a
   CONST expression, indicates an attribute that is constant for a given run
   of the compiler.  The subroutine generated for these attributes has no
   parameters as it does not depend on any particular insn.  Constant
   attributes are typically used to specify which variety of processor is
   used.
   
   Internal attributes are defined to handle DEFINE_DELAY and
   DEFINE_FUNCTION_UNIT.  Special routines are output for these cases.

   This program works by keeping a list of possible values for each attribute.
   These include the basic attribute choices, default values for attribute, and
   all derived quantities.

   As the description file is read, the definition for each insn is saved in a
   `struct insn_def'.   When the file reading is complete, a `struct insn_ent'
   is created for each insn and chained to the corresponding attribute value,
   either that specified, or the default.

   An optimization phase is then run.  This simplifies expressions for each
   insn.  EQ_ATTR tests are resolved, whenever possible, to a test that
   indicates when the attribute has the specified value for the insn.  This
   avoids recursive calls during compilation.

   The strategy used when processing DEFINE_DELAY and DEFINE_FUNCTION_UNIT
   definitions is to create arbitrarily complex expressions and have the
   optimization simplify them.

   Once optimization is complete, any required routines and definitions
   will be written.

   An optimization that is not yet implemented is to hoist the constant
   expressions entirely out of the routines and definitions that are written.
   A way to do this is to iterate over all possible combinations of values
   for constant attributes and generate a set of functions for that given
   combination.  An initialization function would be written that evaluates
   the attributes and installs the corresponding set of routines and
   definitions (each would be accessed through a pointer).

   We use the flags in an RTX as follows:
   `unchanging' (RTX_UNCHANGING_P): This rtx is fully simplified
      independent of the insn code.
   `in_struct' (MEM_IN_STRUCT_P): This rtx is fully simplified
      for the insn code currently being processed (see optimize_attrs).
   `integrated' (RTX_INTEGRATED_P): This rtx is permanent and unique
      (see attr_rtx).
   `volatil' (MEM_VOLATILE_P): During simplify_by_exploding the value of an
      EQ_ATTR rtx is true if !volatil and false if volatil.  */


#include "hconfig.h"
#include "system.h"
#include "rtl.h"
#include "ggc.h"

#ifdef HAVE_SYS_RESOURCE_H
# include <sys/resource.h>
#endif

/* We must include obstack.h after <sys/time.h>, to avoid lossage with
   /usr/include/sys/stdtypes.h on Sun OS 4.x.  */
#include "obstack.h"
#include "errors.h"

static struct obstack obstack, obstack1, obstack2;
struct obstack *rtl_obstack = &obstack;
struct obstack *hash_obstack = &obstack1;
struct obstack *temp_obstack = &obstack2;

#define obstack_chunk_alloc xmalloc
#define obstack_chunk_free free

/* enough space to reserve for printing out ints */
#define MAX_DIGITS (HOST_BITS_PER_INT * 3 / 10 + 3)

/* Define structures used to record attributes and values.  */

/* As each DEFINE_INSN, DEFINE_PEEPHOLE, or DEFINE_ASM_ATTRIBUTES is
   encountered, we store all the relevant information into a
   `struct insn_def'.  This is done to allow attribute definitions to occur
   anywhere in the file.  */

struct insn_def
{
  int insn_code;		/* Instruction number.  */
  int insn_index;		/* Expression numer in file, for errors.  */
  struct insn_def *next;	/* Next insn in chain.  */
  rtx def;			/* The DEFINE_...  */
  int num_alternatives;		/* Number of alternatives.  */
  int vec_idx;			/* Index of attribute vector in `def'.  */
};

/* Once everything has been read in, we store in each attribute value a list
   of insn codes that have that value.  Here is the structure used for the
   list.  */

struct insn_ent
{
  int insn_code;		/* Instruction number.  */
  int insn_index;		/* Index of definition in file */
  struct insn_ent *next;	/* Next in chain.  */
};

/* Each value of an attribute (either constant or computed) is assigned a
   structure which is used as the listhead of the insns that have that
   value.  */

struct attr_value
{
  rtx value;			/* Value of attribute.  */
  struct attr_value *next;	/* Next attribute value in chain.  */
  struct insn_ent *first_insn;	/* First insn with this value.  */
  int num_insns;		/* Number of insns with this value.  */
  int has_asm_insn;		/* True if this value used for `asm' insns */
};

/* Structure for each attribute.  */

struct attr_desc
{
  char *name;			/* Name of attribute.  */
  struct attr_desc *next;	/* Next attribute.  */
  unsigned is_numeric	: 1;	/* Values of this attribute are numeric.  */
  unsigned negative_ok	: 1;	/* Allow negative numeric values.  */
  unsigned unsigned_p	: 1;	/* Make the output function unsigned int.  */
  unsigned is_const	: 1;	/* Attribute value constant for each run.  */
  unsigned is_special	: 1;	/* Don't call `write_attr_set'.  */
  unsigned func_units_p	: 1;	/* this is the function_units attribute */
  unsigned blockage_p	: 1;	/* this is the blockage range function */
  struct attr_value *first_value; /* First value of this attribute.  */
  struct attr_value *default_val; /* Default value for this attribute.  */
};

#define NULL_ATTR (struct attr_desc *) NULL

/* A range of values.  */

struct range
{
  int min;
  int max;
};

/* Structure for each DEFINE_DELAY.  */

struct delay_desc
{
  rtx def;			/* DEFINE_DELAY expression.  */
  struct delay_desc *next;	/* Next DEFINE_DELAY.  */
  int num;			/* Number of DEFINE_DELAY, starting at 1.  */
};

/* Record information about each DEFINE_FUNCTION_UNIT.  */

struct function_unit_op
{
  rtx condexp;			/* Expression TRUE for applicable insn.  */
  struct function_unit_op *next; /* Next operation for this function unit.  */
  int num;			/* Ordinal for this operation type in unit.  */
  int ready;			/* Cost until data is ready.  */
  int issue_delay;		/* Cost until unit can accept another insn.  */
  rtx conflict_exp;		/* Expression TRUE for insns incurring issue delay.  */
  rtx issue_exp;		/* Expression computing issue delay.  */
};

/* Record information about each function unit mentioned in a
   DEFINE_FUNCTION_UNIT.  */

struct function_unit
{
  char *name;			/* Function unit name.  */
  struct function_unit *next;	/* Next function unit.  */
  int num;			/* Ordinal of this unit type.  */
  int multiplicity;		/* Number of units of this type.  */
  int simultaneity;		/* Maximum number of simultaneous insns
				   on this function unit or 0 if unlimited.  */
  rtx condexp;			/* Expression TRUE for insn needing unit.  */
  int num_opclasses;		/* Number of different operation types.  */
  struct function_unit_op *ops;	/* Pointer to first operation type.  */
  int needs_conflict_function;	/* Nonzero if a conflict function required.  */
  int needs_blockage_function;	/* Nonzero if a blockage function required.  */
  int needs_range_function;	/* Nonzero if blockage range function needed.*/
  rtx default_cost;		/* Conflict cost, if constant.  */
  struct range issue_delay;	/* Range of issue delay values.  */
  int max_blockage;		/* Maximum time an insn blocks the unit.  */
};

/* Listheads of above structures.  */

/* This one is indexed by the first character of the attribute name.  */
#define MAX_ATTRS_INDEX 256
static struct attr_desc *attrs[MAX_ATTRS_INDEX];
static struct insn_def *defs;
static struct delay_desc *delays;
static struct function_unit *units;

/* An expression where all the unknown terms are EQ_ATTR tests can be
   rearranged into a COND provided we can enumerate all possible
   combinations of the unknown values.  The set of combinations become the
   tests of the COND; the value of the expression given that combination is
   computed and becomes the corresponding value.  To do this, we must be
   able to enumerate all values for each attribute used in the expression
   (currently, we give up if we find a numeric attribute).
   
   If the set of EQ_ATTR tests used in an expression tests the value of N
   different attributes, the list of all possible combinations can be made
   by walking the N-dimensional attribute space defined by those
   attributes.  We record each of these as a struct dimension.

   The algorithm relies on sharing EQ_ATTR nodes: if two nodes in an
   expression are the same, the will also have the same address.  We find
   all the EQ_ATTR nodes by marking them MEM_VOLATILE_P.  This bit later
   represents the value of an EQ_ATTR node, so once all nodes are marked,
   they are also given an initial value of FALSE.

   We then separate the set of EQ_ATTR nodes into dimensions for each
   attribute and put them on the VALUES list.  Terms are added as needed by
   `add_values_to_cover' so that all possible values of the attribute are
   tested.

   Each dimension also has a current value.  This is the node that is
   currently considered to be TRUE.  If this is one of the nodes added by
   `add_values_to_cover', all the EQ_ATTR tests in the original expression
   will be FALSE.  Otherwise, only the CURRENT_VALUE will be true.

   NUM_VALUES is simply the length of the VALUES list and is there for
   convenience.

   Once the dimensions are created, the algorithm enumerates all possible
   values and computes the current value of the given expression.  */

struct dimension 
{
  struct attr_desc *attr;	/* Attribute for this dimension.  */
  rtx values;			/* List of attribute values used.  */
  rtx current_value;		/* Position in the list for the TRUE value.  */
  int num_values;		/* Length of the values list.  */
};

/* Other variables.  */

static int insn_code_number;
static int insn_index_number;
static int got_define_asm_attributes;
static int must_extract;
static int must_constrain;
static int address_used;
static int length_used;
static int num_delays;
static int have_annul_true, have_annul_false;
static int num_units, num_unit_opclasses;
/* CYGNUS LOCAL -- vmakarov/dfa */
static int num_dfa_decls;
/* END CYGNUS LOCAL -- vmakarov/dfa */
static int num_insn_ents;

/* Used as operand to `operate_exp':  */

enum operator {PLUS_OP, MINUS_OP, POS_MINUS_OP, EQ_OP, OR_OP, ORX_OP, MAX_OP, MIN_OP, RANGE_OP};

/* Stores, for each insn code, the number of constraint alternatives.  */

static int *insn_n_alternatives;

/* Stores, for each insn code, a bitmap that has bits on for each possible
   alternative.  */

static int *insn_alternatives;

/* If nonzero, assume that the `alternative' attr has this value.
   This is the hashed, unique string for the numeral
   whose value is chosen alternative.  */

static char *current_alternative_string;

/* Used to simplify expressions.  */

static rtx true_rtx, false_rtx;

/* Used to reduce calls to `strcmp' */

static char *alternative_name;

/* Indicate that REG_DEAD notes are valid if dead_or_set_p is ever
   called.  */

int reload_completed = 0;

/* Some machines test `optimize' in macros called from rtlanal.c, so we need
   to define it here.  */

int optimize = 0;

/* Simplify an expression.  Only call the routine if there is something to
   simplify.  */
#define SIMPLIFY_TEST_EXP(EXP,INSN_CODE,INSN_INDEX)	\
  (RTX_UNCHANGING_P (EXP) || MEM_IN_STRUCT_P (EXP) ? (EXP)	\
   : simplify_test_exp (EXP, INSN_CODE, INSN_INDEX))
  
/* Simplify (eq_attr ("alternative") ...)
   when we are working with a particular alternative.  */
#define SIMPLIFY_ALTERNATIVE(EXP)				\
  if (current_alternative_string				\
      && GET_CODE ((EXP)) == EQ_ATTR				\
      && XSTR ((EXP), 0) == alternative_name)			\
    (EXP) = (XSTR ((EXP), 1) == current_alternative_string	\
	    ? true_rtx : false_rtx);

/* These are referenced by rtlanal.c and hence need to be defined somewhere.
   They won't actually be used.  */

rtx global_rtl[GR_MAX];
rtx pic_offset_table_rtx;

static void attr_hash_add_rtx	PROTO((int, rtx));
static void attr_hash_add_string PROTO((int, char *));
static rtx attr_rtx		PVPROTO((enum rtx_code, ...));
static char *attr_printf	PVPROTO((int, const char *, ...))
  ATTRIBUTE_PRINTF_2;
static char *attr_string        PROTO((const char *, int));
static rtx check_attr_test	PROTO((rtx, int));
static rtx check_attr_value	PROTO((rtx, struct attr_desc *));
static rtx convert_set_attr_alternative PROTO((rtx, int, int));
static rtx convert_set_attr	PROTO((rtx, int, int));
static void check_defs		PROTO((void));
#if 0
static rtx convert_const_symbol_ref PROTO((rtx, struct attr_desc *));
#endif
static rtx make_canonical	PROTO((struct attr_desc *, rtx));
static struct attr_value *get_attr_value PROTO((rtx, struct attr_desc *, int));
static rtx copy_rtx_unchanging	PROTO((rtx));
static rtx copy_boolean		PROTO((rtx));
static void expand_delays	PROTO((void));
static rtx operate_exp		PROTO((enum operator, rtx, rtx));
static void expand_units	PROTO((void));
static rtx simplify_knowing	PROTO((rtx, rtx));
static rtx encode_units_mask	PROTO((rtx));
static void fill_attr		PROTO((struct attr_desc *));
/* dpx2 compiler chokes if we specify the arg types of the args.  */
static rtx substitute_address	PROTO((rtx, rtx (*) (rtx), rtx (*) (rtx)));
static void make_length_attrs	PROTO((void));
static rtx identity_fn		PROTO((rtx));
static rtx zero_fn		PROTO((rtx));
static rtx one_fn		PROTO((rtx));
static rtx max_fn		PROTO((rtx));
static void write_length_unit_log PROTO ((void));
static rtx simplify_cond	PROTO((rtx, int, int));
#if 0
static rtx simplify_by_alternatives PROTO((rtx, int, int));
#endif
static rtx simplify_by_exploding PROTO((rtx));
static int find_and_mark_used_attributes PROTO((rtx, rtx *, int *));
static void unmark_used_attributes PROTO((rtx, struct dimension *, int));
static int add_values_to_cover	PROTO((struct dimension *));
static int increment_current_value PROTO((struct dimension *, int));
static rtx test_for_current_value PROTO((struct dimension *, int));
static rtx simplify_with_current_value PROTO((rtx, struct dimension *, int));
static rtx simplify_with_current_value_aux PROTO((rtx));
static void clear_struct_flag PROTO((rtx));
static int count_sub_rtxs    PROTO((rtx, int));
static void remove_insn_ent  PROTO((struct attr_value *, struct insn_ent *));
static void insert_insn_ent  PROTO((struct attr_value *, struct insn_ent *));
static rtx insert_right_side	PROTO((enum rtx_code, rtx, rtx, int, int));
static rtx make_alternative_compare PROTO((int));
static int compute_alternative_mask PROTO((rtx, enum rtx_code));
static rtx evaluate_eq_attr	PROTO((rtx, rtx, int, int));
static rtx simplify_and_tree	PROTO((rtx, rtx *, int, int));
static rtx simplify_or_tree	PROTO((rtx, rtx *, int, int));
static rtx simplify_test_exp	PROTO((rtx, int, int));
static void optimize_attrs	PROTO((void));
static void gen_attr		PROTO((rtx));
static int count_alternatives	PROTO((rtx));
static int compares_alternatives_p PROTO((rtx));
static int contained_in_p	PROTO((rtx, rtx));
static void gen_insn		PROTO((rtx));
static void gen_delay		PROTO((rtx));
static void gen_unit		PROTO((rtx));
static void write_test_expr	PROTO((rtx, int));
static int max_attr_value	PROTO((rtx, int*));
static int or_attr_value	PROTO((rtx, int*));
static void walk_attr_value	PROTO((rtx));
static void write_attr_get	PROTO((struct attr_desc *));
static rtx eliminate_known_true PROTO((rtx, rtx, int, int));
static void write_attr_set	PROTO((struct attr_desc *, int, rtx,
				       const char *, const char *, rtx,
				       int, int));
static void write_attr_case	PROTO((struct attr_desc *, struct attr_value *,
				       int, const char *, const char *, int, rtx));
static void write_unit_name	PROTO((const char *, int, const char *));
static void write_attr_valueq	PROTO((struct attr_desc *, char *));
static void write_attr_value	PROTO((struct attr_desc *, rtx));
static void write_upcase	PROTO((const char *));
static void write_indent	PROTO((int));
static void write_eligible_delay PROTO((const char *));
static void write_function_unit_info PROTO((void));
static void write_complex_function PROTO((struct function_unit *, const char *,
					  const char *));
static int write_expr_attr_cache PROTO((rtx, struct attr_desc *));
static void write_toplevel_expr	PROTO((rtx));
static void write_const_num_delay_slots PROTO ((void));
static int n_comma_elts		PROTO((char *));
static char *next_comma_elt	PROTO((char **));
static struct attr_desc *find_attr PROTO((const char *, int));
static void make_internal_attr	PROTO((const char *, rtx, int));
static struct attr_value *find_most_used  PROTO((struct attr_desc *));
static rtx find_single_value	PROTO((struct attr_desc *));
static rtx make_numeric_value	PROTO((int));
static void extend_range	PROTO((struct range *, int, int));
static rtx attr_eq		PROTO((char *, char *));
static char *attr_numeral	PROTO((int));
static int attr_equal_p		PROTO((rtx, rtx));
static rtx attr_copy_rtx	PROTO((rtx));

/* CYGNUS LOCAL -- vmakarov/dfa */
static void gen_cpu_unit		PROTO((rtx));
static void gen_bypass			PROTO((rtx));
static void gen_excl_set		PROTO((rtx));
static void gen_req_set			PROTO((rtx));
static void gen_automaton		PROTO((rtx));
static void gen_reserv   		PROTO((rtx));
static void gen_insn_reserv     	PROTO((rtx));
static void initiate_automaton_gen	PROTO((int, char **));
static void expand_automata             PROTO((void));
static void write_automata              PROTO((void));
/* END CYGNUS LOCAL -- vmakarov/dfa */

#define oballoc(size) obstack_alloc (hash_obstack, size)


/* Hash table for sharing RTL and strings.  */

/* Each hash table slot is a bucket containing a chain of these structures.
   Strings are given negative hash codes; RTL expressions are given positive
   hash codes.  */

struct attr_hash
{
  struct attr_hash *next;	/* Next structure in the bucket.  */
  int hashcode;			/* Hash code of this rtx or string.  */
  union
    {
      char *str;		/* The string (negative hash codes) */
      rtx rtl;			/* or the RTL recorded here.  */
    } u;
};

/* Now here is the hash table.  When recording an RTL, it is added to
   the slot whose index is the hash code mod the table size.  Note
   that the hash table is used for several kinds of RTL (see attr_rtx)
   and for strings.  While all these live in the same table, they are
   completely independent, and the hash code is computed differently
   for each.  */

#define RTL_HASH_SIZE 4093
struct attr_hash *attr_hash_table[RTL_HASH_SIZE];

/* Here is how primitive or already-shared RTL's hash
   codes are made.  */
#define RTL_HASH(RTL) ((long) (RTL) & 0777777)

/* Add an entry to the hash table for RTL with hash code HASHCODE.  */

static void
attr_hash_add_rtx (hashcode, rtl)
     int hashcode;
     rtx rtl;
{
  register struct attr_hash *h;

  h = (struct attr_hash *) obstack_alloc (hash_obstack,
					  sizeof (struct attr_hash));
  h->hashcode = hashcode;
  h->u.rtl = rtl;
  h->next = attr_hash_table[hashcode % RTL_HASH_SIZE];
  attr_hash_table[hashcode % RTL_HASH_SIZE] = h;
}

/* Add an entry to the hash table for STRING with hash code HASHCODE.  */

static void
attr_hash_add_string (hashcode, str)
     int hashcode;
     char *str;
{
  register struct attr_hash *h;

  h = (struct attr_hash *) obstack_alloc (hash_obstack,
					  sizeof (struct attr_hash));
  h->hashcode = -hashcode;
  h->u.str = str;
  h->next = attr_hash_table[hashcode % RTL_HASH_SIZE];
  attr_hash_table[hashcode % RTL_HASH_SIZE] = h;
}

/* Generate an RTL expression, but avoid duplicates.
   Set the RTX_INTEGRATED_P flag for these permanent objects.

   In some cases we cannot uniquify; then we return an ordinary
   impermanent rtx with RTX_INTEGRATED_P clear.

   Args are like gen_rtx, but without the mode:

   rtx attr_rtx (code, [element1, ..., elementn])  */

/*VARARGS1*/
static rtx
attr_rtx VPROTO((enum rtx_code code, ...))
{
#ifndef ANSI_PROTOTYPES
  enum rtx_code code;
#endif
  va_list p;
  register int i;		/* Array indices...			*/
  register const char *fmt;		/* Current rtx's format...		*/
  register rtx rt_val;		/* RTX to return to caller...		*/
  int hashcode;
  register struct attr_hash *h;
  struct obstack *old_obstack = rtl_obstack;

  VA_START (p, code);

#ifndef ANSI_PROTOTYPES
  code = va_arg (p, enum rtx_code);
#endif

  /* For each of several cases, search the hash table for an existing entry.
     Use that entry if one is found; otherwise create a new RTL and add it
     to the table.  */

  if (GET_RTX_CLASS (code) == '1')
    {
      rtx arg0 = va_arg (p, rtx);

      /* A permanent object cannot point to impermanent ones.  */
      if (! RTX_INTEGRATED_P (arg0))
	{
	  rt_val = rtx_alloc (code);
	  XEXP (rt_val, 0) = arg0;
	  va_end (p);
	  return rt_val;
	}

      hashcode = ((HOST_WIDE_INT) code + RTL_HASH (arg0));
      for (h = attr_hash_table[hashcode % RTL_HASH_SIZE]; h; h = h->next)
	if (h->hashcode == hashcode
	    && GET_CODE (h->u.rtl) == code
	    && XEXP (h->u.rtl, 0) == arg0)
	  goto found;

      if (h == 0)
	{
	  rtl_obstack = hash_obstack;
	  rt_val = rtx_alloc (code);
	  XEXP (rt_val, 0) = arg0;
	}
    }
  else if (GET_RTX_CLASS (code) == 'c'
	   || GET_RTX_CLASS (code) == '2'
	   || GET_RTX_CLASS (code) == '<')
    {
      rtx arg0 = va_arg (p, rtx);
      rtx arg1 = va_arg (p, rtx);

      /* A permanent object cannot point to impermanent ones.  */
      if (! RTX_INTEGRATED_P (arg0) || ! RTX_INTEGRATED_P (arg1))
	{
	  rt_val = rtx_alloc (code);
	  XEXP (rt_val, 0) = arg0;
	  XEXP (rt_val, 1) = arg1;
	  va_end (p);
	  return rt_val;
	}

      hashcode = ((HOST_WIDE_INT) code + RTL_HASH (arg0) + RTL_HASH (arg1));
      for (h = attr_hash_table[hashcode % RTL_HASH_SIZE]; h; h = h->next)
	if (h->hashcode == hashcode
	    && GET_CODE (h->u.rtl) == code
	    && XEXP (h->u.rtl, 0) == arg0
	    && XEXP (h->u.rtl, 1) == arg1)
	  goto found;

      if (h == 0)
	{
	  rtl_obstack = hash_obstack;
	  rt_val = rtx_alloc (code);
	  XEXP (rt_val, 0) = arg0;
	  XEXP (rt_val, 1) = arg1;
	}
    }
  else if (GET_RTX_LENGTH (code) == 1
	   && GET_RTX_FORMAT (code)[0] == 's')
    {
      char * arg0 = va_arg (p, char *);

      if (code == SYMBOL_REF)
	arg0 = attr_string (arg0, strlen (arg0));

      hashcode = ((HOST_WIDE_INT) code + RTL_HASH (arg0));
      for (h = attr_hash_table[hashcode % RTL_HASH_SIZE]; h; h = h->next)
	if (h->hashcode == hashcode
	    && GET_CODE (h->u.rtl) == code
	    && XSTR (h->u.rtl, 0) == arg0)
	  goto found;

      if (h == 0)
	{
	  rtl_obstack = hash_obstack;
	  rt_val = rtx_alloc (code);
	  XSTR (rt_val, 0) = arg0;
	}
    }
  else if (GET_RTX_LENGTH (code) == 2
	   && GET_RTX_FORMAT (code)[0] == 's'
	   && GET_RTX_FORMAT (code)[1] == 's')
    {
      char *arg0 = va_arg (p, char *);
      char *arg1 = va_arg (p, char *);

      hashcode = ((HOST_WIDE_INT) code + RTL_HASH (arg0) + RTL_HASH (arg1));
      for (h = attr_hash_table[hashcode % RTL_HASH_SIZE]; h; h = h->next)
	if (h->hashcode == hashcode
	    && GET_CODE (h->u.rtl) == code
	    && XSTR (h->u.rtl, 0) == arg0
	    && XSTR (h->u.rtl, 1) == arg1)
	  goto found;

      if (h == 0)
	{
	  rtl_obstack = hash_obstack;
	  rt_val = rtx_alloc (code);
	  XSTR (rt_val, 0) = arg0;
	  XSTR (rt_val, 1) = arg1;
	}
    }
  else if (code == CONST_INT)
    {
      HOST_WIDE_INT arg0 = va_arg (p, HOST_WIDE_INT);
      if (arg0 == 0)
	{
	  va_end (p);
	  return false_rtx;
	}
      if (arg0 == 1)
	{
	  va_end (p);
	  return true_rtx;
	}
      goto nohash;
    }
  else
    {
    nohash:
      rt_val = rtx_alloc (code);	/* Allocate the storage space.  */
      
      fmt = GET_RTX_FORMAT (code);	/* Find the right format...  */
      for (i = 0; i < GET_RTX_LENGTH (code); i++)
	{
	  switch (*fmt++)
	    {
	    case '0':		/* Unused field.  */
	      break;

	    case 'i':		/* An integer?  */
	      XINT (rt_val, i) = va_arg (p, int);
	      break;

	    case 'w':		/* A wide integer? */
	      XWINT (rt_val, i) = va_arg (p, HOST_WIDE_INT);
	      break;

	    case 's':		/* A string?  */
	      XSTR (rt_val, i) = va_arg (p, char *);
	      break;

	    case 'e':		/* An expression?  */
	    case 'u':		/* An insn?  Same except when printing.  */
	      XEXP (rt_val, i) = va_arg (p, rtx);
	      break;

	    case 'E':		/* An RTX vector?  */
	      XVEC (rt_val, i) = va_arg (p, rtvec);
	      break;

	    default:
	      abort();
	    }
	}
      va_end (p);
      return rt_val;
    }

  rtl_obstack = old_obstack;
  va_end (p);
  attr_hash_add_rtx (hashcode, rt_val);
  RTX_INTEGRATED_P (rt_val) = 1;
  return rt_val;

 found:
  va_end (p);
  return h->u.rtl;
}

/* Create a new string printed with the printf line arguments into a space
   of at most LEN bytes:

   rtx attr_printf (len, format, [arg1, ..., argn])  */

/*VARARGS2*/
static char *
attr_printf VPROTO((register int len, const char *fmt, ...))
{
#ifndef ANSI_PROTOTYPES
  register int len;
  const char *fmt;
#endif
  va_list p;
  register char *str;

  VA_START (p, fmt);

#ifndef ANSI_PROTOTYPES
  len = va_arg (p, int);
  fmt = va_arg (p, const char *);
#endif

  /* Print the string into a temporary location.  */
  str = (char *) alloca (len);
  vsprintf (str, fmt, p);
  va_end (p);

  return attr_string (str, strlen (str));
}

static rtx
attr_eq (name, value)
     char *name, *value;
{
  return attr_rtx (EQ_ATTR, attr_string (name, strlen (name)),
		   attr_string (value, strlen (value)));
}

static char *
attr_numeral (n)
     int n;
{
  return XSTR (make_numeric_value (n), 0);
}

/* Return a permanent (possibly shared) copy of a string STR (not assumed
   to be null terminated) with LEN bytes.  */

static char *
attr_string (str, len)
     const char *str;
     int len;
{
  register struct attr_hash *h;
  int hashcode;
  int i;
  register char *new_str;

  /* Compute the hash code.  */
  hashcode = (len + 1) * 613 + (unsigned)str[0];
  for (i = 1; i <= len; i += 2)
    hashcode = ((hashcode * 613) + (unsigned)str[i]);
  if (hashcode < 0)
    hashcode = -hashcode;

  /* Search the table for the string.  */
  for (h = attr_hash_table[hashcode % RTL_HASH_SIZE]; h; h = h->next)
    if (h->hashcode == -hashcode && h->u.str[0] == str[0]
	&& !strncmp (h->u.str, str, len))
      return h->u.str;			/* <-- return if found.  */

  /* Not found; create a permanent copy and add it to the hash table.  */
  new_str = (char *) obstack_alloc (hash_obstack, len + 1);
  bcopy (str, new_str, len);
  new_str[len] = '\0';
  attr_hash_add_string (hashcode, new_str);

  return new_str;			/* Return the new string.  */
}

/* Check two rtx's for equality of contents,
   taking advantage of the fact that if both are hashed
   then they can't be equal unless they are the same object.  */

static int
attr_equal_p (x, y)
     rtx x, y;
{
  return (x == y || (! (RTX_INTEGRATED_P (x) && RTX_INTEGRATED_P (y))
		     && rtx_equal_p (x, y)));
}

/* Copy an attribute value expression,
   descending to all depths, but not copying any
   permanent hashed subexpressions.  */

static rtx
attr_copy_rtx (orig)
     register rtx orig;
{
  register rtx copy;
  register int i, j;
  register RTX_CODE code;
  register const char *format_ptr;

  /* No need to copy a permanent object.  */
  if (RTX_INTEGRATED_P (orig))
    return orig;

  code = GET_CODE (orig);

  switch (code)
    {
    case REG:
    case QUEUED:
    case CONST_INT:
    case CONST_DOUBLE:
    case SYMBOL_REF:
    case CODE_LABEL:
    case PC:
    case CC0:
      return orig;

    default:
      break;
    }

  copy = rtx_alloc (code);
  PUT_MODE (copy, GET_MODE (orig));
  copy->in_struct = orig->in_struct;
  copy->volatil = orig->volatil;
  copy->unchanging = orig->unchanging;
  copy->integrated = orig->integrated;
  
  format_ptr = GET_RTX_FORMAT (GET_CODE (copy));

  for (i = 0; i < GET_RTX_LENGTH (GET_CODE (copy)); i++)
    {
      switch (*format_ptr++)
	{
	case 'e':
	  XEXP (copy, i) = XEXP (orig, i);
	  if (XEXP (orig, i) != NULL)
	    XEXP (copy, i) = attr_copy_rtx (XEXP (orig, i));
	  break;

	case 'E':
	case 'V':
	  XVEC (copy, i) = XVEC (orig, i);
	  if (XVEC (orig, i) != NULL)
	    {
	      XVEC (copy, i) = rtvec_alloc (XVECLEN (orig, i));
	      for (j = 0; j < XVECLEN (copy, i); j++)
		XVECEXP (copy, i, j) = attr_copy_rtx (XVECEXP (orig, i, j));
	    }
	  break;

	case 'n':
	case 'i':
	  XINT (copy, i) = XINT (orig, i);
	  break;

	case 'w':
	  XWINT (copy, i) = XWINT (orig, i);
	  break;

	case 's':
	case 'S':
	  XSTR (copy, i) = XSTR (orig, i);
	  break;

	default:
	  abort ();
	}
    }
  return copy;
}

/* Given a test expression for an attribute, ensure it is validly formed.
   IS_CONST indicates whether the expression is constant for each compiler
   run (a constant expression may not test any particular insn).

   Convert (eq_attr "att" "a1,a2") to (ior (eq_attr ... ) (eq_attrq ..))
   and (eq_attr "att" "!a1") to (not (eq_attr "att" "a1")).  Do the latter
   test first so that (eq_attr "att" "!a1,a2,a3") works as expected.

   Update the string address in EQ_ATTR expression to be the same used
   in the attribute (or `alternative_name') to speed up subsequent
   `find_attr' calls and eliminate most `strcmp' calls.

   Return the new expression, if any.   */

static rtx
check_attr_test (exp, is_const)
     rtx exp;
     int is_const;
{
  struct attr_desc *attr;
  struct attr_value *av;
  char *name_ptr, *p;
  rtx orexp, newexp;

  switch (GET_CODE (exp))
    {
    case EQ_ATTR:
      /* Handle negation test.  */
      if (XSTR (exp, 1)[0] == '!')
	return check_attr_test (attr_rtx (NOT,
					  attr_eq (XSTR (exp, 0),
						   &XSTR (exp, 1)[1])),
				is_const);

      else if (n_comma_elts (XSTR (exp, 1)) == 1)
	{
	  attr = find_attr (XSTR (exp, 0), 0);
	  if (attr == NULL)
	    {
	      if (! strcmp (XSTR (exp, 0), "alternative"))
		{
		  XSTR (exp, 0) = alternative_name;
		  /* This can't be simplified any further.  */
		  RTX_UNCHANGING_P (exp) = 1;
		  return exp;
		}
	      else
		fatal ("Unknown attribute `%s' in EQ_ATTR", XSTR (exp, 0));
	    }

	  if (is_const && ! attr->is_const)
	    fatal ("Constant expression uses insn attribute `%s' in EQ_ATTR",
		   XSTR (exp, 0));

	  /* Copy this just to make it permanent,
	     so expressions using it can be permanent too.  */
	  exp = attr_eq (XSTR (exp, 0), XSTR (exp, 1));

	  /* It shouldn't be possible to simplify the value given to a
	     constant attribute, so don't expand this until it's time to
	     write the test expression.  */	       
	  if (attr->is_const)
	    RTX_UNCHANGING_P (exp) = 1;

	  if (attr->is_numeric)
	    {
	      for (p = XSTR (exp, 1); *p; p++)
		if (*p < '0' || *p > '9')
		   fatal ("Attribute `%s' takes only numeric values", 
			  XSTR (exp, 0));
	    }
	  else
	    {
	      for (av = attr->first_value; av; av = av->next)
		if (GET_CODE (av->value) == CONST_STRING
		    && ! strcmp (XSTR (exp, 1), XSTR (av->value, 0)))
		  break;

	      if (av == NULL)
		fatal ("Unknown value `%s' for `%s' attribute",
		       XSTR (exp, 1), XSTR (exp, 0));
	    }
	}
      else
	{
	  /* Make an IOR tree of the possible values.  */
	  orexp = false_rtx;
	  name_ptr = XSTR (exp, 1);
	  while ((p = next_comma_elt (&name_ptr)) != NULL)
	    {
	      newexp = attr_eq (XSTR (exp, 0), p);
	      orexp = insert_right_side (IOR, orexp, newexp, -2, -2);
	    }

	  return check_attr_test (orexp, is_const);
	}
      break;

    case ATTR_FLAG:
      break;

    case CONST_INT:
      /* Either TRUE or FALSE.  */
      if (XWINT (exp, 0))
	return true_rtx;
      else
	return false_rtx;

    case IOR:
    case AND:
      XEXP (exp, 0) = check_attr_test (XEXP (exp, 0), is_const);
      XEXP (exp, 1) = check_attr_test (XEXP (exp, 1), is_const);
      break;

    case NOT:
      XEXP (exp, 0) = check_attr_test (XEXP (exp, 0), is_const);
      break;

    case MATCH_INSN:
    case MATCH_OPERAND:
      if (is_const)
	fatal ("RTL operator \"%s\" not valid in constant attribute test",
	       GET_RTX_NAME (GET_CODE (exp)));
      /* These cases can't be simplified.  */
      RTX_UNCHANGING_P (exp) = 1;
      break;
 
    case LE:  case LT:  case GT:  case GE:
    case LEU: case LTU: case GTU: case GEU:
    case NE:  case EQ:
      if (GET_CODE (XEXP (exp, 0)) == SYMBOL_REF
	  && GET_CODE (XEXP (exp, 1)) == SYMBOL_REF)
	exp = attr_rtx (GET_CODE (exp),
			attr_rtx (SYMBOL_REF, XSTR (XEXP (exp, 0), 0)),
			attr_rtx (SYMBOL_REF, XSTR (XEXP (exp, 1), 0)));
      /* These cases can't be simplified.  */
      RTX_UNCHANGING_P (exp) = 1;
      break;

    case SYMBOL_REF:
      if (is_const)
	{
	  /* These cases are valid for constant attributes, but can't be
	     simplified.  */
	  exp = attr_rtx (SYMBOL_REF, XSTR (exp, 0));
	  RTX_UNCHANGING_P (exp) = 1;
	  break;
	}
    default:
      fatal ("RTL operator \"%s\" not valid in attribute test",
	     GET_RTX_NAME (GET_CODE (exp)));
    }

  return exp;
}

/* Given an expression, ensure that it is validly formed and that all named
   attribute values are valid for the given attribute.  Issue a fatal error
   if not.  If no attribute is specified, assume a numeric attribute.

   Return a perhaps modified replacement expression for the value.  */

static rtx
check_attr_value (exp, attr)
     rtx exp;
     struct attr_desc *attr;
{
  struct attr_value *av;
  char *p;
  int i;

  switch (GET_CODE (exp))
    {
    case CONST_INT:
      if (attr && ! attr->is_numeric)
	fatal ("CONST_INT not valid for non-numeric `%s' attribute",
	       attr->name);

      if (INTVAL (exp) < 0 && ! attr->negative_ok)
	fatal ("Negative numeric value specified for `%s' attribute",
	       attr->name);

      break;

    case CONST_STRING:
      if (! strcmp (XSTR (exp, 0), "*"))
	break;

      if (attr == 0 || attr->is_numeric)
	{
	  p = XSTR (exp, 0);
	  if (attr && attr->negative_ok && *p == '-')
	    p++;
	  for (; *p; p++)
	    if (*p > '9' || *p < '0')
	      fatal ("Non-numeric value for numeric `%s' attribute",
		     attr ? attr->name : "internal");
	  break;
	}

      for (av = attr->first_value; av; av = av->next)
	if (GET_CODE (av->value) == CONST_STRING
	    && ! strcmp (XSTR (av->value, 0), XSTR (exp, 0)))
	  break;

      if (av == NULL)
	fatal ("Unknown value `%s' for `%s' attribute",
	       XSTR (exp, 0), attr ? attr->name : "internal");

      break;

    case IF_THEN_ELSE:
      XEXP (exp, 0) = check_attr_test (XEXP (exp, 0),
				       attr ? attr->is_const : 0);
      XEXP (exp, 1) = check_attr_value (XEXP (exp, 1), attr);
      XEXP (exp, 2) = check_attr_value (XEXP (exp, 2), attr);
      break;

    case PLUS:
    case MINUS:
    case MULT:
    case DIV:
    case MOD:
      if (attr && !attr->is_numeric)
	fatal ("Invalid operation `%s' for non-numeric attribute value",
	       GET_RTX_NAME (GET_CODE (exp)));
      /* FALLTHRU */

    case IOR:
    case AND:
      XEXP (exp, 0) = check_attr_value (XEXP (exp, 0), attr);
      XEXP (exp, 1) = check_attr_value (XEXP (exp, 1), attr);
      break;

    case FFS:
      XEXP (exp, 0) = check_attr_value (XEXP (exp, 0), attr);
      break;

    case COND:
      if (XVECLEN (exp, 0) % 2 != 0)
	fatal ("First operand of COND must have even length");

      for (i = 0; i < XVECLEN (exp, 0); i += 2)
	{
	  XVECEXP (exp, 0, i) = check_attr_test (XVECEXP (exp, 0, i),
						 attr ? attr->is_const : 0);
	  XVECEXP (exp, 0, i + 1)
	    = check_attr_value (XVECEXP (exp, 0, i + 1), attr);
	}

      XEXP (exp, 1) = check_attr_value (XEXP (exp, 1), attr);
      break;

    case ATTR:
      {
	struct attr_desc *attr2 = find_attr (XSTR (exp, 0), 0);
	if (attr2 == NULL)
	  fatal ("Unknown attribute `%s' in ATTR", XSTR (exp, 0));
	else if ((attr && attr->is_const) && ! attr2->is_const)
	  fatal ("Non-constant attribute `%s' referenced from `%s'",
		 XSTR (exp, 0), attr->name);
	else if (attr 
		 && (attr->is_numeric != attr2->is_numeric
		     || (! attr->negative_ok && attr2->negative_ok)))
	  fatal ("Numeric attribute mismatch calling `%s' from `%s'",
		 XSTR (exp, 0), attr->name);
      }
      break;

    case SYMBOL_REF:
      /* A constant SYMBOL_REF is valid as a constant attribute test and
         is expanded later by make_canonical into a COND.  In a non-constant
         attribute test, it is left be.  */
      return attr_rtx (SYMBOL_REF, XSTR (exp, 0));

    default:
      fatal ("Invalid operation `%s' for attribute value",
	     GET_RTX_NAME (GET_CODE (exp)));
    }

  return exp;
}

/* Given an SET_ATTR_ALTERNATIVE expression, convert to the canonical SET.
   It becomes a COND with each test being (eq_attr "alternative "n") */

static rtx
convert_set_attr_alternative (exp, num_alt, insn_index)
     rtx exp;
     int num_alt;
     int insn_index;
{
  rtx condexp;
  int i;

  if (XVECLEN (exp, 1) != num_alt)
    fatal ("Bad number of entries in SET_ATTR_ALTERNATIVE for insn %d",
	   insn_index);

  /* Make a COND with all tests but the last.  Select the last value via the
     default.  */
  condexp = rtx_alloc (COND);
  XVEC (condexp, 0) = rtvec_alloc ((num_alt - 1) * 2);

  for (i = 0; i < num_alt - 1; i++)
    {
      char *p;
      p = attr_numeral (i);

      XVECEXP (condexp, 0, 2 * i) = attr_eq (alternative_name, p);
#if 0
      /* Sharing this EQ_ATTR rtl causes trouble.  */   
      XVECEXP (condexp, 0, 2 * i) = rtx_alloc (EQ_ATTR);
      XSTR (XVECEXP (condexp, 0, 2 * i), 0) = alternative_name;
      XSTR (XVECEXP (condexp, 0, 2 * i), 1) = p;
#endif
      XVECEXP (condexp, 0, 2 * i + 1) = XVECEXP (exp, 1, i);
    }

  XEXP (condexp, 1) = XVECEXP (exp, 1, i);

  return attr_rtx (SET, attr_rtx (ATTR, XSTR (exp, 0)), condexp);
}

/* Given a SET_ATTR, convert to the appropriate SET.  If a comma-separated
   list of values is given, convert to SET_ATTR_ALTERNATIVE first.  */

static rtx
convert_set_attr (exp, num_alt, insn_index)
     rtx exp;
     int num_alt;
     int insn_index;
{
  rtx newexp;
  char *name_ptr;
  char *p;
  int n;

  /* See how many alternative specified.  */
  n = n_comma_elts (XSTR (exp, 1));
  if (n == 1)
    return attr_rtx (SET,
		     attr_rtx (ATTR, XSTR (exp, 0)),
		     attr_rtx (CONST_STRING, XSTR (exp, 1)));

  newexp = rtx_alloc (SET_ATTR_ALTERNATIVE);
  XSTR (newexp, 0) = XSTR (exp, 0);
  XVEC (newexp, 1) = rtvec_alloc (n);

  /* Process each comma-separated name.  */
  name_ptr = XSTR (exp, 1);
  n = 0;
  while ((p = next_comma_elt (&name_ptr)) != NULL)
    XVECEXP (newexp, 1, n++) = attr_rtx (CONST_STRING, p);

  return convert_set_attr_alternative (newexp, num_alt, insn_index);
}

/* Scan all definitions, checking for validity.  Also, convert any SET_ATTR
   and SET_ATTR_ALTERNATIVE expressions to the corresponding SET
   expressions.  */

static void
check_defs ()
{
  struct insn_def *id;
  struct attr_desc *attr;
  int i;
  rtx value;

  for (id = defs; id; id = id->next)
    {
      if (XVEC (id->def, id->vec_idx) == NULL)
	continue;

      for (i = 0; i < XVECLEN (id->def, id->vec_idx); i++)
	{
	  value = XVECEXP (id->def, id->vec_idx, i);
	  switch (GET_CODE (value))
	    {
	    case SET:
	      if (GET_CODE (XEXP (value, 0)) != ATTR)
		fatal ("Bad attribute set in pattern %d", id->insn_index);
	      break;

	    case SET_ATTR_ALTERNATIVE:
	      value = convert_set_attr_alternative (value,
						    id->num_alternatives,
						    id->insn_index);
	      break;

	    case SET_ATTR:
	      value = convert_set_attr (value, id->num_alternatives,
					id->insn_index);
	      break;

	    default:
	      fatal ("Invalid attribute code `%s' for pattern %d",
		     GET_RTX_NAME (GET_CODE (value)), id->insn_index);
	    }

	  if ((attr = find_attr (XSTR (XEXP (value, 0), 0), 0)) == NULL)
	    fatal ("Unknown attribute `%s' for pattern number %d",
		   XSTR (XEXP (value, 0), 0), id->insn_index);

	  XVECEXP (id->def, id->vec_idx, i) = value;
	  XEXP (value, 1) = check_attr_value (XEXP (value, 1), attr);
	}
    }
}

#if 0
/* Given a constant SYMBOL_REF expression, convert to a COND that
   explicitly tests each enumerated value.  */

static rtx
convert_const_symbol_ref (exp, attr)
     rtx exp;
     struct attr_desc *attr;
{
  rtx condexp;
  struct attr_value *av;
  int i;
  int num_alt = 0;

  for (av = attr->first_value; av; av = av->next)
    num_alt++;

  /* Make a COND with all tests but the last, and in the original order.
     Select the last value via the default.  Note that the attr values
     are constructed in reverse order.  */

  condexp = rtx_alloc (COND);
  XVEC (condexp, 0) = rtvec_alloc ((num_alt - 1) * 2);
  av = attr->first_value;
  XEXP (condexp, 1) = av->value;

  for (i = num_alt - 2; av = av->next, i >= 0; i--)
    {
      char *p, *string;
      rtx value;

      string = p = (char *) oballoc (2
				     + strlen (attr->name)
				     + strlen (XSTR (av->value, 0)));
      strcpy (p, attr->name);
      strcat (p, "_");
      strcat (p, XSTR (av->value, 0));
      for (; *p != '\0'; p++)
	*p = TOUPPER (*p);

      value = attr_rtx (SYMBOL_REF, string);
      RTX_UNCHANGING_P (value) = 1;
      
      XVECEXP (condexp, 0, 2 * i) = attr_rtx (EQ, exp, value);

      XVECEXP (condexp, 0, 2 * i + 1) = av->value;
    }

  return condexp;
}
#endif

/* Given a valid expression for an attribute value, remove any IF_THEN_ELSE
   expressions by converting them into a COND.  This removes cases from this
   program.  Also, replace an attribute value of "*" with the default attribute
   value.  */

static rtx
make_canonical (attr, exp)
     struct attr_desc *attr;
     rtx exp;
{
  int i;
  rtx newexp;

  switch (GET_CODE (exp))
    {
    case CONST_INT:
      exp = make_numeric_value (INTVAL (exp));
      break;

    case CONST_STRING:
      if (! strcmp (XSTR (exp, 0), "*"))
	{
	  if (attr == 0 || attr->default_val == 0)
	    fatal ("(attr_value \"*\") used in invalid context.");
	  exp = attr->default_val->value;
	}

      break;

    case SYMBOL_REF:
      if (!attr->is_const || RTX_UNCHANGING_P (exp))
	break;
      /* The SYMBOL_REF is constant for a given run, so mark it as unchanging.
	 This makes the COND something that won't be considered an arbitrary
	 expression by walk_attr_value.  */
      RTX_UNCHANGING_P (exp) = 1;
#if 0
      /* ??? Why do we do this?  With attribute values { A B C D E }, this
         tends to generate (!(x==A) && !(x==B) && !(x==C) && !(x==D)) rather
	 than (x==E). */
      exp = convert_const_symbol_ref (exp, attr);
      RTX_UNCHANGING_P (exp) = 1;
      exp = check_attr_value (exp, attr);
      /* Goto COND case since this is now a COND.  Note that while the
         new expression is rescanned, all symbol_ref notes are marked as
	 unchanging.  */
      goto cond;
#else
      exp = check_attr_value (exp, attr);
      break;
#endif

    case IF_THEN_ELSE:
      newexp = rtx_alloc (COND);
      XVEC (newexp, 0) = rtvec_alloc (2);
      XVECEXP (newexp, 0, 0) = XEXP (exp, 0);
      XVECEXP (newexp, 0, 1) = XEXP (exp, 1);

      XEXP (newexp, 1) = XEXP (exp, 2);

      exp = newexp;
      /* Fall through to COND case since this is now a COND.  */

    case COND:
      {
	int allsame = 1;
	rtx defval;

	/* First, check for degenerate COND.  */
	if (XVECLEN (exp, 0) == 0)
	  return make_canonical (attr, XEXP (exp, 1));
	defval = XEXP (exp, 1) = make_canonical (attr, XEXP (exp, 1));

	for (i = 0; i < XVECLEN (exp, 0); i += 2)
	  {
	    XVECEXP (exp, 0, i) = copy_boolean (XVECEXP (exp, 0, i));
	    XVECEXP (exp, 0, i + 1)
	      = make_canonical (attr, XVECEXP (exp, 0, i + 1));
	    if (! rtx_equal_p (XVECEXP (exp, 0, i + 1), defval))
	      allsame = 0;
	  }
	if (allsame)
	  return defval;
      }
      break;

    default:
      break;
    }

  return exp;
}

static rtx
copy_boolean (exp)
     rtx exp;
{
  if (GET_CODE (exp) == AND || GET_CODE (exp) == IOR)
    return attr_rtx (GET_CODE (exp), copy_boolean (XEXP (exp, 0)),
		     copy_boolean (XEXP (exp, 1)));
  return exp;
}

/* Given a value and an attribute description, return a `struct attr_value *'
   that represents that value.  This is either an existing structure, if the
   value has been previously encountered, or a newly-created structure.

   `insn_code' is the code of an insn whose attribute has the specified
   value (-2 if not processing an insn).  We ensure that all insns for
   a given value have the same number of alternatives if the value checks
   alternatives.  */

static struct attr_value *
get_attr_value (value, attr, insn_code)
     rtx value;
     struct attr_desc *attr;
     int insn_code;
{
  struct attr_value *av;
  int num_alt = 0;

  value = make_canonical (attr, value);
  if (compares_alternatives_p (value))
    {
      if (insn_code < 0 || insn_alternatives == NULL)
	fatal ("(eq_attr \"alternatives\" ...) used in non-insn context");
      else
	num_alt = insn_alternatives[insn_code];
    }

  for (av = attr->first_value; av; av = av->next)
    if (rtx_equal_p (value, av->value)
	&& (num_alt == 0 || av->first_insn == NULL
	    || insn_alternatives[av->first_insn->insn_code]))
      return av;

  av = (struct attr_value *) oballoc (sizeof (struct attr_value));
  av->value = value;
  av->next = attr->first_value;
  attr->first_value = av;
  av->first_insn = NULL;
  av->num_insns = 0;
  av->has_asm_insn = 0;

  return av;
}

/* After all DEFINE_DELAYs have been read in, create internal attributes
   to generate the required routines.

   First, we compute the number of delay slots for each insn (as a COND of
   each of the test expressions in DEFINE_DELAYs).  Then, if more than one
   delay type is specified, we compute a similar function giving the
   DEFINE_DELAY ordinal for each insn.

   Finally, for each [DEFINE_DELAY, slot #] pair, we compute an attribute that
   tells whether a given insn can be in that delay slot.

   Normal attribute filling and optimization expands these to contain the
   information needed to handle delay slots.  */

static void
expand_delays ()
{
  struct delay_desc *delay;
  rtx condexp;
  rtx newexp;
  int i;
  char *p;

  /* First, generate data for `num_delay_slots' function.  */

  condexp = rtx_alloc (COND);
  XVEC (condexp, 0) = rtvec_alloc (num_delays * 2);
  XEXP (condexp, 1) = make_numeric_value (0);

  for (i = 0, delay = delays; delay; i += 2, delay = delay->next)
    {
      XVECEXP (condexp, 0, i) = XEXP (delay->def, 0);
      XVECEXP (condexp, 0, i + 1)
	= make_numeric_value (XVECLEN (delay->def, 1) / 3);
    }

  make_internal_attr ("*num_delay_slots", condexp, 0);

  /* If more than one delay type, do the same for computing the delay type.  */
  if (num_delays > 1)
    {
      condexp = rtx_alloc (COND);
      XVEC (condexp, 0) = rtvec_alloc (num_delays * 2);
      XEXP (condexp, 1) = make_numeric_value (0);

      for (i = 0, delay = delays; delay; i += 2, delay = delay->next)
	{
	  XVECEXP (condexp, 0, i) = XEXP (delay->def, 0);
	  XVECEXP (condexp, 0, i + 1) = make_numeric_value (delay->num);
	}

      make_internal_attr ("*delay_type", condexp, 1);
    }

  /* For each delay possibility and delay slot, compute an eligibility
     attribute for non-annulled insns and for each type of annulled (annul
     if true and annul if false).  */
 for (delay = delays; delay; delay = delay->next)
   {
     for (i = 0; i < XVECLEN (delay->def, 1); i += 3)
       {
	 condexp = XVECEXP (delay->def, 1, i);
	 if (condexp == 0) condexp = false_rtx;
	 newexp = attr_rtx (IF_THEN_ELSE, condexp,
			    make_numeric_value (1), make_numeric_value (0));

	 p = attr_printf (sizeof ("*delay__") + MAX_DIGITS*2, "*delay_%d_%d",
			  delay->num, i / 3);
	 make_internal_attr (p, newexp, 1);

	 if (have_annul_true)
	   {
	     condexp = XVECEXP (delay->def, 1, i + 1);
	     if (condexp == 0) condexp = false_rtx;
	     newexp = attr_rtx (IF_THEN_ELSE, condexp,
				make_numeric_value (1),
				make_numeric_value (0));
	     p = attr_printf (sizeof ("*annul_true__") + MAX_DIGITS*2,
			      "*annul_true_%d_%d", delay->num, i / 3);
	     make_internal_attr (p, newexp, 1);
	   }

	 if (have_annul_false)
	   {
	     condexp = XVECEXP (delay->def, 1, i + 2);
	     if (condexp == 0) condexp = false_rtx;
	     newexp = attr_rtx (IF_THEN_ELSE, condexp,
				make_numeric_value (1),
				make_numeric_value (0));
	     p = attr_printf (sizeof ("*annul_false__") + MAX_DIGITS*2,
			      "*annul_false_%d_%d", delay->num, i / 3);
	     make_internal_attr (p, newexp, 1);
	   }
       }
   }
}

/* This function is given a left and right side expression and an operator.
   Each side is a conditional expression, each alternative of which has a
   numerical value.  The function returns another conditional expression
   which, for every possible set of condition values, returns a value that is
   the operator applied to the values of the two sides.

   Since this is called early, it must also support IF_THEN_ELSE.  */

static rtx
operate_exp (op, left, right)
     enum operator op;
     rtx left, right;
{
  int left_value, right_value;
  rtx newexp;
  int i;

  /* If left is a string, apply operator to it and the right side.  */
  if (GET_CODE (left) == CONST_STRING)
    {
      /* If right is also a string, just perform the operation.  */
      if (GET_CODE (right) == CONST_STRING)
	{
	  left_value = atoi (XSTR (left, 0));
	  right_value = atoi (XSTR (right, 0));
	  switch (op)
	    {
	    case PLUS_OP:
	      i = left_value + right_value;
	      break;

	    case MINUS_OP:
	      i = left_value - right_value;
	      break;

	    case POS_MINUS_OP:  /* The positive part of LEFT - RIGHT.  */
	      if (left_value > right_value)
		i = left_value - right_value;
	      else
		i = 0;
	      break;

	    case OR_OP:
	    case ORX_OP:
	      i = left_value | right_value;
	      break;

	    case EQ_OP:
	      i = left_value == right_value;
	      break;

	    case RANGE_OP:
	      i = (left_value << (HOST_BITS_PER_INT / 2)) | right_value;
	      break;

	    case MAX_OP:
	      if (left_value > right_value)
		i = left_value;
	      else
		i = right_value;
	      break;

	    case MIN_OP:
	      if (left_value < right_value)
		i = left_value;
	      else
		i = right_value;
	      break;

	    default:
	      abort ();
	    }

	  if (i == left_value)
	    return left;
	  if (i == right_value)
	    return right;
	  return make_numeric_value (i);
	}
      else if (GET_CODE (right) == IF_THEN_ELSE)
	{
	  /* Apply recursively to all values within.  */
	  rtx newleft = operate_exp (op, left, XEXP (right, 1));
	  rtx newright = operate_exp (op, left, XEXP (right, 2));
	  if (rtx_equal_p (newleft, newright))
	    return newleft;
	  return attr_rtx (IF_THEN_ELSE, XEXP (right, 0), newleft, newright);
	}
      else if (GET_CODE (right) == COND)
	{
	  int allsame = 1;
	  rtx defval;

	  newexp = rtx_alloc (COND);
	  XVEC (newexp, 0) = rtvec_alloc (XVECLEN (right, 0));
	  defval = XEXP (newexp, 1) = operate_exp (op, left, XEXP (right, 1));

	  for (i = 0; i < XVECLEN (right, 0); i += 2)
	    {
	      XVECEXP (newexp, 0, i) = XVECEXP (right, 0, i);
	      XVECEXP (newexp, 0, i + 1)
		= operate_exp (op, left, XVECEXP (right, 0, i + 1));
	      if (! rtx_equal_p (XVECEXP (newexp, 0, i + 1),
				 defval))     
		allsame = 0;
	    }

	  /* If the resulting cond is trivial (all alternatives
	     give the same value), optimize it away.  */
	  if (allsame)
	    {
	      if (!ggc_p)
		obstack_free (rtl_obstack, newexp);
	      return operate_exp (op, left, XEXP (right, 1));
	    }

	  /* If the result is the same as the RIGHT operand,
	     just use that.  */
	  if (rtx_equal_p (newexp, right))
	    {
	      if (!ggc_p)
		obstack_free (rtl_obstack, newexp);
	      return right;
	    }

	  return newexp;
	}
      else
	fatal ("Badly formed attribute value");
    }

  /* A hack to prevent expand_units from completely blowing up: ORX_OP does
     not associate through IF_THEN_ELSE.  */
  else if (op == ORX_OP && GET_CODE (right) == IF_THEN_ELSE)
    {
      return attr_rtx (IOR, left, right);
    }

  /* Otherwise, do recursion the other way.  */
  else if (GET_CODE (left) == IF_THEN_ELSE)
    {
      rtx newleft = operate_exp (op, XEXP (left, 1), right);
      rtx newright = operate_exp (op, XEXP (left, 2), right);
      if (rtx_equal_p (newleft, newright))
	return newleft;
      return attr_rtx (IF_THEN_ELSE, XEXP (left, 0), newleft, newright);
    }
  else if (GET_CODE (left) == COND)
    {
      int allsame = 1;
      rtx defval;

      newexp = rtx_alloc (COND);
      XVEC (newexp, 0) = rtvec_alloc (XVECLEN (left, 0));
      defval = XEXP (newexp, 1) = operate_exp (op, XEXP (left, 1), right);

      for (i = 0; i < XVECLEN (left, 0); i += 2)
	{
	  XVECEXP (newexp, 0, i) = XVECEXP (left, 0, i);
	  XVECEXP (newexp, 0, i + 1)
	    = operate_exp (op, XVECEXP (left, 0, i + 1), right);
	  if (! rtx_equal_p (XVECEXP (newexp, 0, i + 1),
			     defval))     
	    allsame = 0;
	}

      /* If the cond is trivial (all alternatives give the same value),
	 optimize it away.  */
      if (allsame)
	{
	  if (!ggc_p)
	    obstack_free (rtl_obstack, newexp);
	  return operate_exp (op, XEXP (left, 1), right);
	}

      /* If the result is the same as the LEFT operand,
	 just use that.  */
      if (rtx_equal_p (newexp, left))
	{
	  if (!ggc_p)
	    obstack_free (rtl_obstack, newexp);
	  return left;
	}

      return newexp;
    }

  else
    fatal ("Badly formed attribute value.");
  /* NOTREACHED */
  return NULL;
}

/* Once all attributes and DEFINE_FUNCTION_UNITs have been read, we
   construct a number of attributes.

   The first produces a function `function_units_used' which is given an
   insn and produces an encoding showing which function units are required
   for the execution of that insn.  If the value is non-negative, the insn
   uses that unit; otherwise, the value is a one's compliment mask of units
   used.

   The second produces a function `result_ready_cost' which is used to
   determine the time that the result of an insn will be ready and hence
   a worst-case schedule.

   Both of these produce quite complex expressions which are then set as the
   default value of internal attributes.  Normal attribute simplification
   should produce reasonable expressions.

   For each unit, a `<name>_unit_ready_cost' function will take an
   insn and give the delay until that unit will be ready with the result
   and a `<name>_unit_conflict_cost' function is given an insn already
   executing on the unit and a candidate to execute and will give the
   cost from the time the executing insn started until the candidate
   can start (ignore limitations on the number of simultaneous insns).

   For each unit, a `<name>_unit_blockage' function is given an insn
   already executing on the unit and a candidate to execute and will
   give the delay incurred due to function unit conflicts.  The range of
   blockage cost values for a given executing insn is given by the
   `<name>_unit_blockage_range' function.  These values are encoded in
   an int where the upper half gives the minimum value and the lower
   half gives the maximum value.  */

static void
expand_units ()
{
  struct function_unit *unit, **unit_num;
  struct function_unit_op *op, **op_array, ***unit_ops;
  rtx unitsmask;
  rtx readycost;
  rtx newexp;
  const char *str;
  int i, j, u, num, nvalues;

  /* Rebuild the condition for the unit to share the RTL expressions.
     Sharing is required by simplify_by_exploding.  Build the issue delay
     expressions.  Validate the expressions we were given for the conditions
     and conflict vector.  Then make attributes for use in the conflict
     function.  */

  for (unit = units; unit; unit = unit->next)
    {
      unit->condexp = check_attr_test (unit->condexp, 0);

      for (op = unit->ops; op; op = op->next)
	{
	  rtx issue_delay = make_numeric_value (op->issue_delay);
	  rtx issue_exp = issue_delay;

	  /* Build, validate, and simplify the issue delay expression.  */
	  if (op->conflict_exp != true_rtx)
	    issue_exp = attr_rtx (IF_THEN_ELSE, op->conflict_exp,
				  issue_exp, make_numeric_value (0));
	  issue_exp = check_attr_value (make_canonical (NULL_ATTR,
							issue_exp),
					NULL_ATTR);
	  issue_exp = simplify_knowing (issue_exp, unit->condexp);
	  op->issue_exp = issue_exp;

	  /* Make an attribute for use in the conflict function if needed.  */
	  unit->needs_conflict_function = (unit->issue_delay.min
					   != unit->issue_delay.max);
	  if (unit->needs_conflict_function)
	    {
	      str = attr_printf (strlen (unit->name) + sizeof ("*_cost_") + MAX_DIGITS,
				 "*%s_cost_%d", unit->name, op->num);
	      make_internal_attr (str, issue_exp, 1);
	    }

	  /* Validate the condition.  */
	  op->condexp = check_attr_test (op->condexp, 0);
	}
    }

  /* Compute the mask of function units used.  Initially, the unitsmask is
     zero.   Set up a conditional to compute each unit's contribution.  */
  unitsmask = make_numeric_value (0);
  newexp = rtx_alloc (IF_THEN_ELSE);
  XEXP (newexp, 2) = make_numeric_value (0);

  /* If we have just a few units, we may be all right expanding the whole
     thing.  But the expansion is 2**N in space on the number of opclasses,
     so we can't do this for very long -- Alpha and MIPS in particular have
     problems with this.  So in that situation, we fall back on an alternate
     implementation method.  */
#define NUM_UNITOP_CUTOFF 20

  if (num_unit_opclasses < NUM_UNITOP_CUTOFF)
    {
      /* Merge each function unit into the unit mask attributes.  */
      for (unit = units; unit; unit = unit->next)
        {
          XEXP (newexp, 0) = unit->condexp;
          XEXP (newexp, 1) = make_numeric_value (1 << unit->num);
          unitsmask = operate_exp (OR_OP, unitsmask, newexp);
        }
    }
  else
    {
      /* Merge each function unit into the unit mask attributes.  */
      for (unit = units; unit; unit = unit->next)
        {
          XEXP (newexp, 0) = unit->condexp;
          XEXP (newexp, 1) = make_numeric_value (1 << unit->num);
          unitsmask = operate_exp (ORX_OP, unitsmask, attr_copy_rtx (newexp));
        }
    }

  /* Simplify the unit mask expression, encode it, and make an attribute
     for the function_units_used function.  */
  unitsmask = simplify_by_exploding (unitsmask);

  if (num_unit_opclasses < NUM_UNITOP_CUTOFF)
    unitsmask = encode_units_mask (unitsmask);
  else
    {
      /* We can no longer encode unitsmask at compile time, so emit code to
         calculate it at runtime.  Rather, put a marker for where we'd do
	 the code, and actually output it in write_attr_get().  */
      unitsmask = attr_rtx (FFS, unitsmask);
    }

  make_internal_attr ("*function_units_used", unitsmask, 10);

  /* Create an array of ops for each unit.  Add an extra unit for the
     result_ready_cost function that has the ops of all other units.  */
  unit_ops = (struct function_unit_op ***)
    alloca ((num_units + 1) * sizeof (struct function_unit_op **));
  unit_num = (struct function_unit **)
    alloca ((num_units + 1) * sizeof (struct function_unit *));

  unit_num[num_units] = unit = (struct function_unit *)
    alloca (sizeof (struct function_unit));
  unit->num = num_units;
  unit->num_opclasses = 0;

  for (unit = units; unit; unit = unit->next)
    {
      unit_num[num_units]->num_opclasses += unit->num_opclasses;
      unit_num[unit->num] = unit;
      unit_ops[unit->num] = op_array = (struct function_unit_op **)
	alloca (unit->num_opclasses * sizeof (struct function_unit_op *));

      for (op = unit->ops; op; op = op->next)
	op_array[op->num] = op;
    }

  /* Compose the array of ops for the extra unit.  */
  unit_ops[num_units] = op_array = (struct function_unit_op **)
    alloca (unit_num[num_units]->num_opclasses
	    * sizeof (struct function_unit_op *));

  for (unit = units, i = 0; unit; i += unit->num_opclasses, unit = unit->next)
    bcopy ((char *) unit_ops[unit->num], (char *) &op_array[i],
	   unit->num_opclasses * sizeof (struct function_unit_op *));

  /* Compute the ready cost function for each unit by computing the
     condition for each non-default value.  */
  for (u = 0; u <= num_units; u++)
    {
      rtx orexp;
      int value;

      unit = unit_num[u];
      op_array = unit_ops[unit->num];
      num = unit->num_opclasses;

      /* Sort the array of ops into increasing ready cost order.  */
      for (i = 0; i < num; i++)
	for (j = num - 1; j > i; j--)
	  if (op_array[j-1]->ready < op_array[j]->ready)
	    {
	      op = op_array[j];
	      op_array[j] = op_array[j-1];
	      op_array[j-1] = op;
	    }

      /* Determine how many distinct non-default ready cost values there
	 are.  We use a default ready cost value of 1.  */
      nvalues = 0; value = 1;
      for (i = num - 1; i >= 0; i--)
	if (op_array[i]->ready > value)
	  {
	    value = op_array[i]->ready;
	    nvalues++;
	  }

      if (nvalues == 0)
	readycost = make_numeric_value (1);
      else
	{
	  /* Construct the ready cost expression as a COND of each value from
	     the largest to the smallest.  */
	  readycost = rtx_alloc (COND);
	  XVEC (readycost, 0) = rtvec_alloc (nvalues * 2);
	  XEXP (readycost, 1) = make_numeric_value (1);

	  nvalues = 0; orexp = false_rtx; value = op_array[0]->ready;
	  for (i = 0; i < num; i++)
	    {
	      op = op_array[i];
	      if (op->ready <= 1)
		break;
	      else if (op->ready == value)
		orexp = insert_right_side (IOR, orexp, op->condexp, -2, -2);
	      else
		{
		  XVECEXP (readycost, 0, nvalues * 2) = orexp;
		  XVECEXP (readycost, 0, nvalues * 2 + 1)
		    = make_numeric_value (value);
		  nvalues++;
		  value = op->ready;
		  orexp = op->condexp;
		}
	    }
	  XVECEXP (readycost, 0, nvalues * 2) = orexp;
	  XVECEXP (readycost, 0, nvalues * 2 + 1) = make_numeric_value (value);
	}

      if (u < num_units)
	{
	  rtx max_blockage = 0, min_blockage = 0;

	  /* Simplify the readycost expression by only considering insns
	     that use the unit.  */
	  readycost = simplify_knowing (readycost, unit->condexp);

	  /* Determine the blockage cost the executing insn (E) given
	     the candidate insn (C).  This is the maximum of the issue
	     delay, the pipeline delay, and the simultaneity constraint.
	     Each function_unit_op represents the characteristics of the
	     candidate insn, so in the expressions below, C is a known
	     term and E is an unknown term.

	     We compute the blockage cost for each E for every possible C.
	     Thus OP represents E, and READYCOST is a list of values for
	     every possible C.

	     The issue delay function for C is op->issue_exp and is used to
	     write the `<name>_unit_conflict_cost' function.  Symbolicly
	     this is "ISSUE-DELAY (E,C)".

	     The pipeline delay results form the FIFO constraint on the
	     function unit and is "READY-COST (E) + 1 - READY-COST (C)".

	     The simultaneity constraint is based on how long it takes to
	     fill the unit given the minimum issue delay.  FILL-TIME is the
	     constant "MIN (ISSUE-DELAY (*,*)) * (SIMULTANEITY - 1)", and
	     the simultaneity constraint is "READY-COST (E) - FILL-TIME"
	     if SIMULTANEITY is non-zero and zero otherwise.

	     Thus, BLOCKAGE (E,C) when SIMULTANEITY is zero is

	         MAX (ISSUE-DELAY (E,C),
		      READY-COST (E) - (READY-COST (C) - 1))

	     and otherwise

	         MAX (ISSUE-DELAY (E,C),
		      READY-COST (E) - (READY-COST (C) - 1),
		      READY-COST (E) - FILL-TIME)

	     The `<name>_unit_blockage' function is computed by determining
	     this value for each candidate insn.  As these values are
	     computed, we also compute the upper and lower bounds for
	     BLOCKAGE (E,*).  These are combined to form the function
	     `<name>_unit_blockage_range'.  Finally, the maximum blockage
	     cost, MAX (BLOCKAGE (*,*)), is computed.  */

	  for (op = unit->ops; op; op = op->next)
	    {
	      rtx blockage = op->issue_exp;
	      blockage = simplify_knowing (blockage, unit->condexp);

	      /* Add this op's contribution to MAX (BLOCKAGE (E,*)) and
		 MIN (BLOCKAGE (E,*)).  */
	      if (max_blockage == 0)
		max_blockage = min_blockage = blockage;
	      else
		{
		  max_blockage
		    = simplify_knowing (operate_exp (MAX_OP, max_blockage,
						     blockage),
					unit->condexp);
		  min_blockage
		    = simplify_knowing (operate_exp (MIN_OP, min_blockage,
						     blockage),
					unit->condexp);
		}

	      /* Make an attribute for use in the blockage function.  */
	      str = attr_printf (strlen (unit->name) + sizeof ("*_block_") + MAX_DIGITS,
				 "*%s_block_%d", unit->name, op->num);
	      make_internal_attr (str, blockage, 1);
	    }

	  /* Record MAX (BLOCKAGE (*,*)).  */
	  {
	    int unknown;
	    unit->max_blockage = max_attr_value (max_blockage, &unknown);
	  }

	  /* See if the upper and lower bounds of BLOCKAGE (E,*) are the
	     same.  If so, the blockage function carries no additional
	     information and is not written.  */
	  newexp = operate_exp (EQ_OP, max_blockage, min_blockage);
	  newexp = simplify_knowing (newexp, unit->condexp);
	  unit->needs_blockage_function
	    = (GET_CODE (newexp) != CONST_STRING
	       || atoi (XSTR (newexp, 0)) != 1);

	  /* If the all values of BLOCKAGE (E,C) have the same value,
	     neither blockage function is written.  */	  
	  unit->needs_range_function
	    = (unit->needs_blockage_function
	       || GET_CODE (max_blockage) != CONST_STRING);

	  if (unit->needs_range_function)
	    {
	      /* Compute the blockage range function and make an attribute
		 for writing its value.  */
	      newexp = operate_exp (RANGE_OP, min_blockage, max_blockage);
	      newexp = simplify_knowing (newexp, unit->condexp);

	      str = attr_printf (strlen (unit->name) + sizeof ("*_unit_blockage_range"),
				 "*%s_unit_blockage_range", unit->name);
	      make_internal_attr (str, newexp, 20);
	    }

	  str = attr_printf (strlen (unit->name) + sizeof ("*_unit_ready_cost"),
			     "*%s_unit_ready_cost", unit->name);
	}
      else
	str = "*result_ready_cost";

      /* Make an attribute for the ready_cost function.  Simplifying
	 further with simplify_by_exploding doesn't win.  */
      make_internal_attr (str, readycost, 0);
    }

  /* For each unit that requires a conflict cost function, make an attribute
     that maps insns to the operation number.  */
  for (unit = units; unit; unit = unit->next)
    {
      rtx caseexp;

      if (! unit->needs_conflict_function
	  && ! unit->needs_blockage_function)
	continue;

      caseexp = rtx_alloc (COND);
      XVEC (caseexp, 0) = rtvec_alloc ((unit->num_opclasses - 1) * 2);

      for (op = unit->ops; op; op = op->next)
	{
	  /* Make our adjustment to the COND being computed.  If we are the
	     last operation class, place our values into the default of the
	     COND.  */
	  if (op->num == unit->num_opclasses - 1)
	    {
	      XEXP (caseexp, 1) = make_numeric_value (op->num);
	    }
	  else
	    {
	      XVECEXP (caseexp, 0, op->num * 2) = op->condexp;
	      XVECEXP (caseexp, 0, op->num * 2 + 1)
		= make_numeric_value (op->num);
	    }
	}

      /* Simplifying caseexp with simplify_by_exploding doesn't win.  */
      str = attr_printf (strlen (unit->name) + sizeof ("*_cases"),
			 "*%s_cases", unit->name);
      make_internal_attr (str, caseexp, 1);
    }
}

/* Simplify EXP given KNOWN_TRUE.  */

static rtx
simplify_knowing (exp, known_true)
     rtx exp, known_true;
{
  if (GET_CODE (exp) != CONST_STRING)
    {
      int unknown = 0, max;
      max = max_attr_value (exp, &unknown);
      if (! unknown)
	{
	  exp = attr_rtx (IF_THEN_ELSE, known_true, exp,
		          make_numeric_value (max));
          exp = simplify_by_exploding (exp);
	}
    }
  return exp;
}

/* Translate the CONST_STRING expressions in X to change the encoding of
   value.  On input, the value is a bitmask with a one bit for each unit
   used; on output, the value is the unit number (zero based) if one
   and only one unit is used or the one's compliment of the bitmask.  */

static rtx
encode_units_mask (x)
     rtx x;
{
  register int i;
  register int j;
  register enum rtx_code code;
  register const char *fmt;

  code = GET_CODE (x);

  switch (code)
    {
    case CONST_STRING:
      i = atoi (XSTR (x, 0));
      if (i < 0)
	abort (); /* The sign bit encodes a one's compliment mask.  */
      else if (i != 0 && i == (i & -i))
	/* Only one bit is set, so yield that unit number.  */
	for (j = 0; (i >>= 1) != 0; j++)
	  ;
      else
	j = ~i;
      return attr_rtx (CONST_STRING, attr_printf (MAX_DIGITS, "%d", j));

    case REG:
    case QUEUED:
    case CONST_INT:
    case CONST_DOUBLE:
    case SYMBOL_REF:
    case CODE_LABEL:
    case PC:
    case CC0:
    case EQ_ATTR:
      return x;
      
    default:
      break;
    }

  /* Compare the elements.  If any pair of corresponding elements
     fail to match, return 0 for the whole things.  */

  fmt = GET_RTX_FORMAT (code);
  for (i = GET_RTX_LENGTH (code) - 1; i >= 0; i--)
    {
      switch (fmt[i])
	{
	case 'V':
	case 'E':
	  for (j = 0; j < XVECLEN (x, i); j++)
	    XVECEXP (x, i, j) = encode_units_mask (XVECEXP (x, i, j));
	  break;

	case 'e':
	  XEXP (x, i) = encode_units_mask (XEXP (x, i));
	  break;
	}
    }
  return x;
}

/* Once all attributes and insns have been read and checked, we construct for
   each attribute value a list of all the insns that have that value for
   the attribute.  */

static void
fill_attr (attr)
     struct attr_desc *attr;
{
  struct attr_value *av;
  struct insn_ent *ie;
  struct insn_def *id;
  int i;
  rtx value;

  /* Don't fill constant attributes.  The value is independent of
     any particular insn.  */
  if (attr->is_const)
    return;

  for (id = defs; id; id = id->next)
    {
      /* If no value is specified for this insn for this attribute, use the
	 default.  */
      value = NULL;
      if (XVEC (id->def, id->vec_idx))
	for (i = 0; i < XVECLEN (id->def, id->vec_idx); i++)
	  if (! strcmp (XSTR (XEXP (XVECEXP (id->def, id->vec_idx, i), 0), 0), 
			attr->name))
	    value = XEXP (XVECEXP (id->def, id->vec_idx, i), 1);

      if (value == NULL)
	av = attr->default_val;
      else
	av = get_attr_value (value, attr, id->insn_code);

      ie = (struct insn_ent *) oballoc (sizeof (struct insn_ent));
      ie->insn_code = id->insn_code;
      ie->insn_index = id->insn_code;
      insert_insn_ent (av, ie);
    }
}

/* Given an expression EXP, see if it is a COND or IF_THEN_ELSE that has a
   test that checks relative positions of insns (uses MATCH_DUP or PC).
   If so, replace it with what is obtained by passing the expression to
   ADDRESS_FN.  If not but it is a COND or IF_THEN_ELSE, call this routine
   recursively on each value (including the default value).  Otherwise,
   return the value returned by NO_ADDRESS_FN applied to EXP.  */

static rtx
substitute_address (exp, no_address_fn, address_fn)
     rtx exp;
     rtx (*no_address_fn) PROTO ((rtx));
     rtx (*address_fn) PROTO ((rtx));
{
  int i;
  rtx newexp;

  if (GET_CODE (exp) == COND)
    {
      /* See if any tests use addresses.  */
      address_used = 0;
      for (i = 0; i < XVECLEN (exp, 0); i += 2)
	walk_attr_value (XVECEXP (exp, 0, i));

      if (address_used)
	return (*address_fn) (exp);

      /* Make a new copy of this COND, replacing each element.  */
      newexp = rtx_alloc (COND);
      XVEC (newexp, 0) = rtvec_alloc (XVECLEN (exp, 0));
      for (i = 0; i < XVECLEN (exp, 0); i += 2)
	{
	  XVECEXP (newexp, 0, i) = XVECEXP (exp, 0, i);
	  XVECEXP (newexp, 0, i + 1)
	    = substitute_address (XVECEXP (exp, 0, i + 1),
				  no_address_fn, address_fn);
	}

      XEXP (newexp, 1) = substitute_address (XEXP (exp, 1),
					     no_address_fn, address_fn);

      return newexp;
    }

  else if (GET_CODE (exp) == IF_THEN_ELSE)
    {
      address_used = 0;
      walk_attr_value (XEXP (exp, 0));
      if (address_used)
	return (*address_fn) (exp);

      return attr_rtx (IF_THEN_ELSE,
		       substitute_address (XEXP (exp, 0),
					   no_address_fn, address_fn),
		       substitute_address (XEXP (exp, 1),
					   no_address_fn, address_fn),
		       substitute_address (XEXP (exp, 2),
					   no_address_fn, address_fn));
    }

  return (*no_address_fn) (exp);
}

/* Make new attributes from the `length' attribute.  The following are made,
   each corresponding to a function called from `shorten_branches' or
   `get_attr_length':

   *insn_default_length		This is the length of the insn to be returned
				by `get_attr_length' before `shorten_branches'
				has been called.  In each case where the length
				depends on relative addresses, the largest
				possible is used.  This routine is also used
				to compute the initial size of the insn.

   *insn_variable_length_p	This returns 1 if the insn's length depends
				on relative addresses, zero otherwise.

   *insn_current_length		This is only called when it is known that the
				insn has a variable length and returns the
				current length, based on relative addresses.
  */

static void
make_length_attrs ()
{
  static const char *new_names[] = {"*insn_default_length",
				      "*insn_variable_length_p",
				      "*insn_current_length"};
  static rtx (*no_address_fn[]) PROTO((rtx)) = {identity_fn, zero_fn, zero_fn};
  static rtx (*address_fn[]) PROTO((rtx)) = {max_fn, one_fn, identity_fn};
  size_t i;
  struct attr_desc *length_attr, *new_attr;
  struct attr_value *av, *new_av;
  struct insn_ent *ie, *new_ie;

  /* See if length attribute is defined.  If so, it must be numeric.  Make
     it special so we don't output anything for it.  */
  length_attr = find_attr ("length", 0);
  if (length_attr == 0)
    return;

  if (! length_attr->is_numeric)
    fatal ("length attribute must be numeric.");

  length_attr->is_const = 0;
  length_attr->is_special = 1;

  /* Make each new attribute, in turn.  */
  for (i = 0; i < sizeof new_names / sizeof new_names[0]; i++)
    {
      make_internal_attr (new_names[i],
			  substitute_address (length_attr->default_val->value,
					      no_address_fn[i], address_fn[i]),
			  0);
      new_attr = find_attr (new_names[i], 0);
      for (av = length_attr->first_value; av; av = av->next)
	for (ie = av->first_insn; ie; ie = ie->next)
	  {
	    new_av = get_attr_value (substitute_address (av->value,
							 no_address_fn[i],
							 address_fn[i]),
				     new_attr, ie->insn_code);
	    new_ie = (struct insn_ent *) oballoc (sizeof (struct insn_ent));
	    new_ie->insn_code = ie->insn_code;
	    new_ie->insn_index = ie->insn_index;
	    insert_insn_ent (new_av, new_ie);
	  }
    }
}

/* Utility functions called from above routine.  */

static rtx
identity_fn (exp)
     rtx exp;
{
  return exp;
}

static rtx
zero_fn (exp)
     rtx exp ATTRIBUTE_UNUSED;
{
  return make_numeric_value (0);
}

static rtx
one_fn (exp)
     rtx exp ATTRIBUTE_UNUSED;
{
  return make_numeric_value (1);
}

static rtx
max_fn (exp)
     rtx exp;
{
  int unknown;
  return make_numeric_value (max_attr_value (exp, &unknown));
}

static void
write_length_unit_log ()
{
  struct attr_desc *length_attr = find_attr ("length", 0);
  struct attr_value *av;
  struct insn_ent *ie;
  unsigned int length_unit_log, length_or;
  int unknown = 0;

  if (length_attr == 0)
    return;
  length_or = or_attr_value (length_attr->default_val->value, &unknown);
  for (av = length_attr->first_value; av; av = av->next)
    for (ie = av->first_insn; ie; ie = ie->next)
      length_or |= or_attr_value (av->value, &unknown);

  if (unknown)
    length_unit_log = 0;
  else
    {
      length_or = ~length_or;
      for (length_unit_log = 0; length_or & 1; length_or >>= 1)
        length_unit_log++;
    }
  printf ("int length_unit_log = %u;\n", length_unit_log);
}

/* Take a COND expression and see if any of the conditions in it can be
   simplified.  If any are known true or known false for the particular insn
   code, the COND can be further simplified.

   Also call ourselves on any COND operations that are values of this COND.

   We do not modify EXP; rather, we make and return a new rtx.  */

static rtx
simplify_cond (exp, insn_code, insn_index)
     rtx exp;
     int insn_code, insn_index;
{
  int i, j;
  /* We store the desired contents here,
     then build a new expression if they don't match EXP.  */
  rtx defval = XEXP (exp, 1);
  rtx new_defval = XEXP (exp, 1);
  int len = XVECLEN (exp, 0);
  rtx *tests = (rtx *) alloca (len * sizeof (rtx));
  int allsame = 1;
  char *first_spacer;

  /* This lets us free all storage allocated below, if appropriate.  */
  first_spacer = (char *) obstack_finish (rtl_obstack);

  bcopy ((char *) XVEC (exp, 0)->elem, (char *) tests, len * sizeof (rtx));

  /* See if default value needs simplification.  */
  if (GET_CODE (defval) == COND)
    new_defval = simplify_cond (defval, insn_code, insn_index);

  /* Simplify the subexpressions, and see what tests we can get rid of.  */

  for (i = 0; i < len; i += 2)
    {
      rtx newtest, newval;

      /* Simplify this test.  */
      newtest = SIMPLIFY_TEST_EXP (tests[i], insn_code, insn_index);
      tests[i] = newtest;

      newval = tests[i + 1];
      /* See if this value may need simplification.  */
      if (GET_CODE (newval) == COND)
	newval = simplify_cond (newval, insn_code, insn_index);

      /* Look for ways to delete or combine this test.  */
      if (newtest == true_rtx)
	{
	  /* If test is true, make this value the default
	     and discard this + any following tests.  */
	  len = i;
	  defval = tests[i + 1];
	  new_defval = newval;
	}

      else if (newtest == false_rtx)
	{
	  /* If test is false, discard it and its value.  */
	  for (j = i; j < len - 2; j++)
	    tests[j] = tests[j + 2];
	  len -= 2;
	}

      else if (i > 0 && attr_equal_p (newval, tests[i - 1]))
	{
	  /* If this value and the value for the prev test are the same,
	     merge the tests.  */

	  tests[i - 2]
	    = insert_right_side (IOR, tests[i - 2], newtest,
				 insn_code, insn_index);

	  /* Delete this test/value.  */
	  for (j = i; j < len - 2; j++)
	    tests[j] = tests[j + 2];
	  len -= 2;
	}

      else
	tests[i + 1] = newval;
    }

  /* If the last test in a COND has the same value
     as the default value, that test isn't needed.  */

  while (len > 0 && attr_equal_p (tests[len - 1], new_defval))
    len -= 2;

  /* See if we changed anything.  */
  if (len != XVECLEN (exp, 0) || new_defval != XEXP (exp, 1))
    allsame = 0;
  else
    for (i = 0; i < len; i++)
      if (! attr_equal_p (tests[i], XVECEXP (exp, 0, i)))
	{
	  allsame = 0;
	  break;
	}

  if (len == 0)
    {
      if (!ggc_p)
	obstack_free (rtl_obstack, first_spacer);
      if (GET_CODE (defval) == COND)
	return simplify_cond (defval, insn_code, insn_index);
      return defval;
    }
  else if (allsame)
    {
      if (!ggc_p)
	obstack_free (rtl_obstack, first_spacer);
      return exp;
    }
  else
    {
      rtx newexp = rtx_alloc (COND);

      XVEC (newexp, 0) = rtvec_alloc (len);
      bcopy ((char *) tests, (char *) XVEC (newexp, 0)->elem,
	     len * sizeof (rtx));
      XEXP (newexp, 1) = new_defval;
      return newexp;
    }
}

/* Remove an insn entry from an attribute value.  */

static void
remove_insn_ent (av, ie)
     struct attr_value *av;
     struct insn_ent *ie;
{
  struct insn_ent *previe;

  if (av->first_insn == ie)
    av->first_insn = ie->next;
  else
    {
      for (previe = av->first_insn; previe->next != ie; previe = previe->next)
	;
      previe->next = ie->next;
    }

  av->num_insns--;
  if (ie->insn_code == -1)
    av->has_asm_insn = 0;

  num_insn_ents--;
}

/* Insert an insn entry in an attribute value list.  */

static void
insert_insn_ent (av, ie)
     struct attr_value *av;
     struct insn_ent *ie;
{
  ie->next = av->first_insn;
  av->first_insn = ie;
  av->num_insns++;
  if (ie->insn_code == -1)
    av->has_asm_insn = 1;

  num_insn_ents++;
}

/* This is a utility routine to take an expression that is a tree of either
   AND or IOR expressions and insert a new term.  The new term will be
   inserted at the right side of the first node whose code does not match
   the root.  A new node will be created with the root's code.  Its left
   side will be the old right side and its right side will be the new
   term.

   If the `term' is itself a tree, all its leaves will be inserted.  */

static rtx
insert_right_side (code, exp, term, insn_code, insn_index)
     enum rtx_code code;
     rtx exp;
     rtx term;
     int insn_code, insn_index;
{
  rtx newexp;

  /* Avoid consing in some special cases.  */
  if (code == AND && term == true_rtx)
    return exp;
  if (code == AND && term == false_rtx)
    return false_rtx;
  if (code == AND && exp == true_rtx)
    return term;
  if (code == AND && exp == false_rtx)
    return false_rtx;
  if (code == IOR && term == true_rtx)
    return true_rtx;
  if (code == IOR && term == false_rtx)
    return exp;
  if (code == IOR && exp == true_rtx)
    return true_rtx;
  if (code == IOR && exp == false_rtx)
    return term;
  if (attr_equal_p (exp, term))
    return exp;

  if (GET_CODE (term) == code)
    {
      exp = insert_right_side (code, exp, XEXP (term, 0),
			       insn_code, insn_index);
      exp = insert_right_side (code, exp, XEXP (term, 1),
			       insn_code, insn_index);

      return exp;
    }

  if (GET_CODE (exp) == code)
    {
      rtx new = insert_right_side (code, XEXP (exp, 1),
				   term, insn_code, insn_index);
      if (new != XEXP (exp, 1))
	/* Make a copy of this expression and call recursively.  */
	newexp = attr_rtx (code, XEXP (exp, 0), new);
      else
	newexp = exp;
    }
  else
    {
      /* Insert the new term.  */
      newexp = attr_rtx (code, exp, term);
    }

  return SIMPLIFY_TEST_EXP (newexp, insn_code, insn_index);
}

/* If we have an expression which AND's a bunch of
	(not (eq_attrq "alternative" "n"))
   terms, we may have covered all or all but one of the possible alternatives.
   If so, we can optimize.  Similarly for IOR's of EQ_ATTR.

   This routine is passed an expression and either AND or IOR.  It returns a
   bitmask indicating which alternatives are mentioned within EXP.  */

static int
compute_alternative_mask (exp, code)
     rtx exp;
     enum rtx_code code;
{
  char *string;
  if (GET_CODE (exp) == code)
    return compute_alternative_mask (XEXP (exp, 0), code)
	   | compute_alternative_mask (XEXP (exp, 1), code);

  else if (code == AND && GET_CODE (exp) == NOT
	   && GET_CODE (XEXP (exp, 0)) == EQ_ATTR
	   && XSTR (XEXP (exp, 0), 0) == alternative_name)
    string = XSTR (XEXP (exp, 0), 1);

  else if (code == IOR && GET_CODE (exp) == EQ_ATTR
	   && XSTR (exp, 0) == alternative_name)
    string = XSTR (exp, 1);

  else
    return 0;

  if (string[1] == 0)
    return 1 << (string[0] - '0');
  return 1 << atoi (string);
}

/* Given I, a single-bit mask, return RTX to compare the `alternative'
   attribute with the value represented by that bit.  */

static rtx
make_alternative_compare (mask)
     int mask;
{
  rtx newexp;
  int i;

  /* Find the bit.  */
  for (i = 0; (mask & (1 << i)) == 0; i++)
    ;

  newexp = attr_rtx (EQ_ATTR, alternative_name, attr_numeral (i));
  RTX_UNCHANGING_P (newexp) = 1;

  return newexp;
}

/* If we are processing an (eq_attr "attr" "value") test, we find the value
   of "attr" for this insn code.  From that value, we can compute a test
   showing when the EQ_ATTR will be true.  This routine performs that
   computation.  If a test condition involves an address, we leave the EQ_ATTR
   intact because addresses are only valid for the `length' attribute. 

   EXP is the EQ_ATTR expression and VALUE is the value of that attribute
   for the insn corresponding to INSN_CODE and INSN_INDEX.  */

static rtx
evaluate_eq_attr (exp, value, insn_code, insn_index)
     rtx exp;
     rtx value;
     int insn_code, insn_index;
{
  rtx orexp, andexp;
  rtx right;
  rtx newexp;
  int i;

  if (GET_CODE (value) == CONST_STRING)
    {
      if (! strcmp (XSTR (value, 0), XSTR (exp, 1)))
	newexp = true_rtx;
      else
	newexp = false_rtx;
    }
  else if (GET_CODE (value) == SYMBOL_REF)
    {
      char *p, *string;

      if (GET_CODE (exp) != EQ_ATTR)
	abort();

      string = (char *) alloca (2 + strlen (XSTR (exp, 0))
				+ strlen (XSTR (exp, 1)));
      strcpy (string, XSTR (exp, 0));
      strcat (string, "_");
      strcat (string, XSTR (exp, 1));
      for (p = string; *p ; p++)
	*p = TOUPPER (*p);
      
      newexp = attr_rtx (EQ, value,
			 attr_rtx (SYMBOL_REF,
				   attr_string(string, strlen(string))));
    }
  else if (GET_CODE (value) == COND)
    {
      /* We construct an IOR of all the cases for which the requested attribute
	 value is present.  Since we start with FALSE, if it is not present,
	 FALSE will be returned.

	 Each case is the AND of the NOT's of the previous conditions with the
	 current condition; in the default case the current condition is TRUE. 

	 For each possible COND value, call ourselves recursively.

	 The extra TRUE and FALSE expressions will be eliminated by another
	 call to the simplification routine.  */

      orexp = false_rtx;
      andexp = true_rtx;

      if (current_alternative_string)
	clear_struct_flag (value);

      for (i = 0; i < XVECLEN (value, 0); i += 2)
	{
	  rtx this = SIMPLIFY_TEST_EXP (XVECEXP (value, 0, i),
					insn_code, insn_index);

	  SIMPLIFY_ALTERNATIVE (this);

	  right = insert_right_side (AND, andexp, this,
				     insn_code, insn_index);
	  right = insert_right_side (AND, right,
				     evaluate_eq_attr (exp,
						       XVECEXP (value, 0,
								i + 1),
						       insn_code, insn_index),
				     insn_code, insn_index);
	  orexp = insert_right_side (IOR, orexp, right,
				     insn_code, insn_index);

	  /* Add this condition into the AND expression.  */
	  newexp = attr_rtx (NOT, this);
	  andexp = insert_right_side (AND, andexp, newexp,
				      insn_code, insn_index);
	}

      /* Handle the default case.  */
      right = insert_right_side (AND, andexp,
				 evaluate_eq_attr (exp, XEXP (value, 1),
						   insn_code, insn_index),
				 insn_code, insn_index);
      newexp = insert_right_side (IOR, orexp, right, insn_code, insn_index);
    }
  else
    abort ();

  /* If uses an address, must return original expression.  But set the
     RTX_UNCHANGING_P bit so we don't try to simplify it again.  */

  address_used = 0;
  walk_attr_value (newexp);

  if (address_used)
    {
      /* This had `&& current_alternative_string', which seems to be wrong.  */
      if (! RTX_UNCHANGING_P (exp))
	return copy_rtx_unchanging (exp);
      return exp;
    }
  else
    return newexp;
}

/* This routine is called when an AND of a term with a tree of AND's is
   encountered.  If the term or its complement is present in the tree, it
   can be replaced with TRUE or FALSE, respectively.

   Note that (eq_attr "att" "v1") and (eq_attr "att" "v2") cannot both
   be true and hence are complementary.  

   There is one special case:  If we see
	(and (not (eq_attr "att" "v1"))
	     (eq_attr "att" "v2"))
   this can be replaced by (eq_attr "att" "v2").  To do this we need to
   replace the term, not anything in the AND tree.  So we pass a pointer to
   the term.  */

static rtx
simplify_and_tree (exp, pterm, insn_code, insn_index)
     rtx exp;
     rtx *pterm;
     int insn_code, insn_index;
{
  rtx left, right;
  rtx newexp;
  rtx temp;
  int left_eliminates_term, right_eliminates_term;

  if (GET_CODE (exp) == AND)
    {
      left = simplify_and_tree (XEXP (exp, 0), pterm,  insn_code, insn_index);
      right = simplify_and_tree (XEXP (exp, 1), pterm, insn_code, insn_index);
      if (left != XEXP (exp, 0) || right != XEXP (exp, 1))
	{
	  newexp = attr_rtx (GET_CODE (exp), left, right);

	  exp = SIMPLIFY_TEST_EXP (newexp, insn_code, insn_index);
	}
    }

  else if (GET_CODE (exp) == IOR)
    {
      /* For the IOR case, we do the same as above, except that we can
         only eliminate `term' if both sides of the IOR would do so.  */
      temp = *pterm;
      left = simplify_and_tree (XEXP (exp, 0), &temp,  insn_code, insn_index);
      left_eliminates_term = (temp == true_rtx);

      temp = *pterm;
      right = simplify_and_tree (XEXP (exp, 1), &temp, insn_code, insn_index);
      right_eliminates_term = (temp == true_rtx);

      if (left_eliminates_term && right_eliminates_term)
	*pterm = true_rtx;

      if (left != XEXP (exp, 0) || right != XEXP (exp, 1))
	{
	  newexp = attr_rtx (GET_CODE (exp), left, right);

	  exp = SIMPLIFY_TEST_EXP (newexp, insn_code, insn_index);
	}
    }

  /* Check for simplifications.  Do some extra checking here since this
     routine is called so many times.  */

  if (exp == *pterm)
    return true_rtx;

  else if (GET_CODE (exp) == NOT && XEXP (exp, 0) == *pterm)
    return false_rtx;

  else if (GET_CODE (*pterm) == NOT && exp == XEXP (*pterm, 0))
    return false_rtx;

  else if (GET_CODE (exp) == EQ_ATTR && GET_CODE (*pterm) == EQ_ATTR)
    {
      if (XSTR (exp, 0) != XSTR (*pterm, 0))
	return exp;

      if (! strcmp (XSTR (exp, 1), XSTR (*pterm, 1)))
	return true_rtx;
      else
	return false_rtx;
    }

  else if (GET_CODE (*pterm) == EQ_ATTR && GET_CODE (exp) == NOT
	   && GET_CODE (XEXP (exp, 0)) == EQ_ATTR)
    {
      if (XSTR (*pterm, 0) != XSTR (XEXP (exp, 0), 0))
	return exp;

      if (! strcmp (XSTR (*pterm, 1), XSTR (XEXP (exp, 0), 1)))
	return false_rtx;
      else
	return true_rtx;
    }

  else if (GET_CODE (exp) == EQ_ATTR && GET_CODE (*pterm) == NOT
	   && GET_CODE (XEXP (*pterm, 0)) == EQ_ATTR)
    {
      if (XSTR (exp, 0) != XSTR (XEXP (*pterm, 0), 0))
	return exp;

      if (! strcmp (XSTR (exp, 1), XSTR (XEXP (*pterm, 0), 1)))
	return false_rtx;
      else
	*pterm = true_rtx;
    }

  else if (GET_CODE (exp) == NOT && GET_CODE (*pterm) == NOT)
    {
      if (attr_equal_p (XEXP (exp, 0), XEXP (*pterm, 0)))
	return true_rtx;
    }

  else if (GET_CODE (exp) == NOT)
    {
      if (attr_equal_p (XEXP (exp, 0), *pterm))
	return false_rtx;
    }

  else if (GET_CODE (*pterm) == NOT)
    {
      if (attr_equal_p (XEXP (*pterm, 0), exp))
	return false_rtx;
    }

  else if (attr_equal_p (exp, *pterm))
    return true_rtx;

  return exp;
}

/* Similar to `simplify_and_tree', but for IOR trees.  */

static rtx
simplify_or_tree (exp, pterm, insn_code, insn_index)
     rtx exp;
     rtx *pterm;
     int insn_code, insn_index;
{
  rtx left, right;
  rtx newexp;
  rtx temp;
  int left_eliminates_term, right_eliminates_term;

  if (GET_CODE (exp) == IOR)
    {
      left = simplify_or_tree (XEXP (exp, 0), pterm,  insn_code, insn_index);
      right = simplify_or_tree (XEXP (exp, 1), pterm, insn_code, insn_index);
      if (left != XEXP (exp, 0) || right != XEXP (exp, 1))
	{
	  newexp = attr_rtx (GET_CODE (exp), left, right);

	  exp = SIMPLIFY_TEST_EXP (newexp, insn_code, insn_index);
	}
    }

  else if (GET_CODE (exp) == AND)
    {
      /* For the AND case, we do the same as above, except that we can
         only eliminate `term' if both sides of the AND would do so.  */
      temp = *pterm;
      left = simplify_or_tree (XEXP (exp, 0), &temp,  insn_code, insn_index);
      left_eliminates_term = (temp == false_rtx);

      temp = *pterm;
      right = simplify_or_tree (XEXP (exp, 1), &temp, insn_code, insn_index);
      right_eliminates_term = (temp == false_rtx);

      if (left_eliminates_term && right_eliminates_term)
	*pterm = false_rtx;

      if (left != XEXP (exp, 0) || right != XEXP (exp, 1))
	{
	  newexp = attr_rtx (GET_CODE (exp), left, right);

	  exp = SIMPLIFY_TEST_EXP (newexp, insn_code, insn_index);
	}
    }

  if (attr_equal_p (exp, *pterm))
    return false_rtx;

  else if (GET_CODE (exp) == NOT && attr_equal_p (XEXP (exp, 0), *pterm))
    return true_rtx;

  else if (GET_CODE (*pterm) == NOT && attr_equal_p (XEXP (*pterm, 0), exp))
    return true_rtx;

  else if (GET_CODE (*pterm) == EQ_ATTR && GET_CODE (exp) == NOT
	   && GET_CODE (XEXP (exp, 0)) == EQ_ATTR
	   && XSTR (*pterm, 0) == XSTR (XEXP (exp, 0), 0))
    *pterm = false_rtx;

  else if (GET_CODE (exp) == EQ_ATTR && GET_CODE (*pterm) == NOT
	   && GET_CODE (XEXP (*pterm, 0)) == EQ_ATTR
	   && XSTR (exp, 0) == XSTR (XEXP (*pterm, 0), 0))
    return false_rtx;

  return exp;
}

/* Given an expression, see if it can be simplified for a particular insn
   code based on the values of other attributes being tested.  This can
   eliminate nested get_attr_... calls.

   Note that if an endless recursion is specified in the patterns, the 
   optimization will loop.  However, it will do so in precisely the cases where
   an infinite recursion loop could occur during compilation.  It's better that
   it occurs here!  */

static rtx
simplify_test_exp (exp, insn_code, insn_index)
     rtx exp;
     int insn_code, insn_index;
{
  rtx left, right;
  struct attr_desc *attr;
  struct attr_value *av;
  struct insn_ent *ie;
  int i;
  rtx newexp = exp;
  char *spacer = (char *) obstack_finish (rtl_obstack);

  /* Don't re-simplify something we already simplified.  */
  if (RTX_UNCHANGING_P (exp) || MEM_IN_STRUCT_P (exp))
    return exp;

  switch (GET_CODE (exp))
    {
    case AND:
      left = SIMPLIFY_TEST_EXP (XEXP (exp, 0), insn_code, insn_index);
      SIMPLIFY_ALTERNATIVE (left);
      if (left == false_rtx)
	{
	  if (!ggc_p)
	    obstack_free (rtl_obstack, spacer);
	  return false_rtx;
	}
      right = SIMPLIFY_TEST_EXP (XEXP (exp, 1), insn_code, insn_index);
      SIMPLIFY_ALTERNATIVE (right);
      if (left == false_rtx)
	{
	  if (!ggc_p)
	    obstack_free (rtl_obstack, spacer);
	  return false_rtx;
	}

      /* If either side is an IOR and we have (eq_attr "alternative" ..")
	 present on both sides, apply the distributive law since this will
	 yield simplifications.  */
      if ((GET_CODE (left) == IOR || GET_CODE (right) == IOR)
	  && compute_alternative_mask (left, IOR)
	  && compute_alternative_mask (right, IOR))
	{
	  if (GET_CODE (left) == IOR)
	    {
	      rtx tem = left;
	      left = right;
	      right = tem;
	    }

	  newexp = attr_rtx (IOR,
			     attr_rtx (AND, left, XEXP (right, 0)),
			     attr_rtx (AND, left, XEXP (right, 1)));

	  return SIMPLIFY_TEST_EXP (newexp, insn_code, insn_index);
	}

      /* Try with the term on both sides.  */
      right = simplify_and_tree (right, &left, insn_code, insn_index);
      if (left == XEXP (exp, 0) && right == XEXP (exp, 1))
	left = simplify_and_tree (left, &right, insn_code, insn_index);

      if (left == false_rtx || right == false_rtx)
	{
	  if (!ggc_p)
	    obstack_free (rtl_obstack, spacer);
	  return false_rtx;
	}
      else if (left == true_rtx)
	{
	  return right;
	}
      else if (right == true_rtx)
	{
	  return left;
	}
      /* See if all or all but one of the insn's alternatives are specified
	 in this tree.  Optimize if so.  */

      else if (insn_code >= 0
	       && (GET_CODE (left) == AND
		   || (GET_CODE (left) == NOT
		       && GET_CODE (XEXP (left, 0)) == EQ_ATTR
		       && XSTR (XEXP (left, 0), 0) == alternative_name)
		   || GET_CODE (right) == AND
		   || (GET_CODE (right) == NOT
		       && GET_CODE (XEXP (right, 0)) == EQ_ATTR
		       && XSTR (XEXP (right, 0), 0) == alternative_name)))
	{
	  i = compute_alternative_mask (exp, AND);
	  if (i & ~insn_alternatives[insn_code])
	    fatal ("Invalid alternative specified for pattern number %d",
		   insn_index);

	  /* If all alternatives are excluded, this is false.  */
	  i ^= insn_alternatives[insn_code];
	  if (i == 0)
	    return false_rtx;
	  else if ((i & (i - 1)) == 0 && insn_alternatives[insn_code] > 1)
	    {
	      /* If just one excluded, AND a comparison with that one to the
		 front of the tree.  The others will be eliminated by
		 optimization.  We do not want to do this if the insn has one
		 alternative and we have tested none of them!  */
	      left = make_alternative_compare (i);
	      right = simplify_and_tree (exp, &left, insn_code, insn_index);
	      newexp = attr_rtx (AND, left, right);

	      return SIMPLIFY_TEST_EXP (newexp, insn_code, insn_index);
	    }
	}

      if (left != XEXP (exp, 0) || right != XEXP (exp, 1))
	{
	  newexp = attr_rtx (AND, left, right);
	  return SIMPLIFY_TEST_EXP (newexp, insn_code, insn_index);
	}
      break;

    case IOR:
      left = SIMPLIFY_TEST_EXP (XEXP (exp, 0), insn_code, insn_index);
      SIMPLIFY_ALTERNATIVE (left);
      if (left == true_rtx)
	{
	  if (!ggc_p)
	    obstack_free (rtl_obstack, spacer);
	  return true_rtx;
	}
      right = SIMPLIFY_TEST_EXP (XEXP (exp, 1), insn_code, insn_index);
      SIMPLIFY_ALTERNATIVE (right);
      if (right == true_rtx)
	{
	  if (!ggc_p)
	    obstack_free (rtl_obstack, spacer);
	  return true_rtx;
	}

      right = simplify_or_tree (right, &left, insn_code, insn_index);
      if (left == XEXP (exp, 0) && right == XEXP (exp, 1))
	left = simplify_or_tree (left, &right, insn_code, insn_index);

      if (right == true_rtx || left == true_rtx)
	{
	  if (!ggc_p)
	    obstack_free (rtl_obstack, spacer);
	  return true_rtx;
	}
      else if (left == false_rtx)
	{
	  return right;
	}
      else if (right == false_rtx)
	{
	  return left;
	}

      /* Test for simple cases where the distributive law is useful.  I.e.,
	    convert (ior (and (x) (y))
			 (and (x) (z)))
	    to      (and (x)
			 (ior (y) (z)))
       */

      else if (GET_CODE (left) == AND && GET_CODE (right) == AND
	  && attr_equal_p (XEXP (left, 0), XEXP (right, 0)))
	{
	  newexp = attr_rtx (IOR, XEXP (left, 1), XEXP (right, 1));

	  left = XEXP (left, 0);
	  right = newexp;
	  newexp = attr_rtx (AND, left, right);
	  return SIMPLIFY_TEST_EXP (newexp, insn_code, insn_index);
	}

      /* See if all or all but one of the insn's alternatives are specified
	 in this tree.  Optimize if so.  */

      else if (insn_code >= 0
	  && (GET_CODE (left) == IOR
	      || (GET_CODE (left) == EQ_ATTR
		  && XSTR (left, 0) == alternative_name)
	      || GET_CODE (right) == IOR
	      || (GET_CODE (right) == EQ_ATTR
		  && XSTR (right, 0) == alternative_name)))
	{
	  i = compute_alternative_mask (exp, IOR);
	  if (i & ~insn_alternatives[insn_code])
	    fatal ("Invalid alternative specified for pattern number %d",
		   insn_index);

	  /* If all alternatives are included, this is true.  */
	  i ^= insn_alternatives[insn_code];
	  if (i == 0)
	    return true_rtx;
	  else if ((i & (i - 1)) == 0 && insn_alternatives[insn_code] > 1)
	    {
	      /* If just one excluded, IOR a comparison with that one to the
		 front of the tree.  The others will be eliminated by
		 optimization.  We do not want to do this if the insn has one
		 alternative and we have tested none of them!  */
	      left = make_alternative_compare (i);
	      right = simplify_and_tree (exp, &left, insn_code, insn_index);
	      newexp = attr_rtx (IOR, attr_rtx (NOT, left), right);

	      return SIMPLIFY_TEST_EXP (newexp, insn_code, insn_index);
	    }
	}

      if (left != XEXP (exp, 0) || right != XEXP (exp, 1))
	{
	  newexp = attr_rtx (IOR, left, right);
	  return SIMPLIFY_TEST_EXP (newexp, insn_code, insn_index);
	}
      break;

    case NOT:
      if (GET_CODE (XEXP (exp, 0)) == NOT)
	{
	  left = SIMPLIFY_TEST_EXP (XEXP (XEXP (exp, 0), 0),
				    insn_code, insn_index);
	  SIMPLIFY_ALTERNATIVE (left);
	  return left;
	}

      left = SIMPLIFY_TEST_EXP (XEXP (exp, 0), insn_code, insn_index);
      SIMPLIFY_ALTERNATIVE (left);
      if (GET_CODE (left) == NOT)
	return XEXP (left, 0);

      if (left == false_rtx)
	{
	  if (!ggc_p)
	    obstack_free (rtl_obstack, spacer);
	  return true_rtx;
	}
      else if (left == true_rtx)
	{
	  if (!ggc_p)
	    obstack_free (rtl_obstack, spacer);
	  return false_rtx;
	}

      /* Try to apply De`Morgan's laws.  */
      else if (GET_CODE (left) == IOR)
	{
	  newexp = attr_rtx (AND,
			     attr_rtx (NOT, XEXP (left, 0)),
			     attr_rtx (NOT, XEXP (left, 1)));

	  newexp = SIMPLIFY_TEST_EXP (newexp, insn_code, insn_index);
	}
      else if (GET_CODE (left) == AND)
	{
	  newexp = attr_rtx (IOR,
			     attr_rtx (NOT, XEXP (left, 0)),
			     attr_rtx (NOT, XEXP (left, 1)));

	  newexp = SIMPLIFY_TEST_EXP (newexp, insn_code, insn_index);
	}
      else if (left != XEXP (exp, 0))
	{
	  newexp = attr_rtx (NOT, left);
	}
      break;

    case EQ_ATTR:
      if (current_alternative_string && XSTR (exp, 0) == alternative_name)
	return (XSTR (exp, 1) == current_alternative_string
		? true_rtx : false_rtx);
	
      /* Look at the value for this insn code in the specified attribute.
	 We normally can replace this comparison with the condition that
	 would give this insn the values being tested for.   */
      if (XSTR (exp, 0) != alternative_name
	  && (attr = find_attr (XSTR (exp, 0), 0)) != NULL)
	for (av = attr->first_value; av; av = av->next)
	  for (ie = av->first_insn; ie; ie = ie->next)
	    if (ie->insn_code == insn_code)
	      return evaluate_eq_attr (exp, av->value, insn_code, insn_index);
      break;
      
    default:
      break;
    }

  /* We have already simplified this expression.  Simplifying it again
     won't buy anything unless we weren't given a valid insn code
     to process (i.e., we are canonicalizing something.).  */
  if (insn_code != -2 /* Seems wrong: && current_alternative_string.  */
      && ! RTX_UNCHANGING_P (newexp))
    return copy_rtx_unchanging (newexp);

  return newexp;
}

/* Optimize the attribute lists by seeing if we can determine conditional
   values from the known values of other attributes.  This will save subroutine
   calls during the compilation.  */

static void
optimize_attrs ()
{
  struct attr_desc *attr;
  struct attr_value *av;
  struct insn_ent *ie;
  rtx newexp;
  int something_changed = 1;
  int i;
  struct attr_value_list { struct attr_value *av;
			   struct insn_ent *ie;
			   struct attr_desc * attr;
			   struct attr_value_list *next; };
  struct attr_value_list **insn_code_values;
  struct attr_value_list *ivbuf;
  struct attr_value_list *iv;

  /* For each insn code, make a list of all the insn_ent's for it,
     for all values for all attributes.  */

  if (num_insn_ents == 0)
    return;

  /* Make 2 extra elements, for "code" values -2 and -1.  */
  insn_code_values
    = (struct attr_value_list **) alloca ((insn_code_number + 2)
					  * sizeof (struct attr_value_list *));
  bzero ((char *) insn_code_values,
	 (insn_code_number + 2) * sizeof (struct attr_value_list *));

  /* Offset the table address so we can index by -2 or -1.  */
  insn_code_values += 2;

  /* Allocate the attr_value_list structures using xmalloc rather than
     alloca, because using alloca can overflow the maximum permitted
     stack limit on SPARC Lynx.  */
  iv = ivbuf = ((struct attr_value_list *)
		xmalloc (num_insn_ents * sizeof (struct attr_value_list)));

  for (i = 0; i < MAX_ATTRS_INDEX; i++)
    for (attr = attrs[i]; attr; attr = attr->next)
      for (av = attr->first_value; av; av = av->next)
	for (ie = av->first_insn; ie; ie = ie->next)
	  {
	    iv->attr = attr;
	    iv->av = av;
	    iv->ie = ie;
	    iv->next = insn_code_values[ie->insn_code];
	    insn_code_values[ie->insn_code] = iv;
	    iv++;
	  }

  /* Sanity check on num_insn_ents.  */
  if (iv != ivbuf + num_insn_ents)
    abort ();

  /* Process one insn code at a time.  */
  for (i = -2; i < insn_code_number; i++)
    {
      /* Clear the MEM_IN_STRUCT_P flag everywhere relevant.
	 We use it to mean "already simplified for this insn".  */
      for (iv = insn_code_values[i]; iv; iv = iv->next)
	clear_struct_flag (iv->av->value);

      /* Loop until nothing changes for one iteration.  */
      something_changed = 1;
      while (something_changed)
	{
	  something_changed = 0;
	  for (iv = insn_code_values[i]; iv; iv = iv->next)
	    {
	      struct obstack *old = rtl_obstack;
	      char *spacer = (char *) obstack_finish (temp_obstack);

	      attr = iv->attr;
	      av = iv->av;
	      ie = iv->ie;
	      if (GET_CODE (av->value) != COND)
		continue;

	      rtl_obstack = temp_obstack;
#if 0 /* This was intended as a speed up, but it was slower.  */
	      if (insn_n_alternatives[ie->insn_code] > 6
		  && count_sub_rtxs (av->value, 200) >= 200)
		newexp = simplify_by_alternatives (av->value, ie->insn_code,
						   ie->insn_index);
	      else
#endif
		newexp = simplify_cond (av->value, ie->insn_code,
					ie->insn_index);

	      rtl_obstack = old;
	      if (newexp != av->value)
		{
		  newexp = attr_copy_rtx (newexp);
		  remove_insn_ent (av, ie);
		  av = get_attr_value (newexp, attr, ie->insn_code);
		  iv->av = av;
		  insert_insn_ent (av, ie);
		  something_changed = 1;
		}
	      if (!ggc_p)
		obstack_free (temp_obstack, spacer);
	    }
	}
    }

  free (ivbuf);
}

#if 0
static rtx
simplify_by_alternatives (exp, insn_code, insn_index)
     rtx exp;
     int insn_code, insn_index;
{
  int i;
  int len = insn_n_alternatives[insn_code];
  rtx newexp = rtx_alloc (COND);
  rtx ultimate;


  XVEC (newexp, 0) = rtvec_alloc (len * 2);

  /* It will not matter what value we use as the default value
     of the new COND, since that default will never be used.
     Choose something of the right type.  */
  for (ultimate = exp; GET_CODE (ultimate) == COND;)
    ultimate = XEXP (ultimate, 1);
  XEXP (newexp, 1) = ultimate;

  for (i = 0; i < insn_n_alternatives[insn_code]; i++)
    {
      current_alternative_string = attr_numeral (i);
      XVECEXP (newexp, 0, i * 2) = make_alternative_compare (1 << i);
      XVECEXP (newexp, 0, i * 2 + 1)
	= simplify_cond (exp, insn_code, insn_index);
    }

  current_alternative_string = 0;
  return simplify_cond (newexp, insn_code, insn_index);
}
#endif

/* If EXP is a suitable expression, reorganize it by constructing an
   equivalent expression that is a COND with the tests being all combinations
   of attribute values and the values being simple constants.  */

static rtx
simplify_by_exploding (exp)
     rtx exp;
{
  rtx list = 0, link, condexp, defval = NULL_RTX;
  struct dimension *space;
  rtx *condtest, *condval;
  int i, j, total, ndim = 0;
  int most_tests, num_marks, new_marks;

  /* Locate all the EQ_ATTR expressions.  */
  if (! find_and_mark_used_attributes (exp, &list, &ndim) || ndim == 0)
    {
      unmark_used_attributes (list, 0, 0);
      return exp;
    }

  /* Create an attribute space from the list of used attributes.  For each
     dimension in the attribute space, record the attribute, list of values
     used, and number of values used.  Add members to the list of values to
     cover the domain of the attribute.  This makes the expanded COND form
     order independent.  */

  space = (struct dimension *) alloca (ndim * sizeof (struct dimension));

  total = 1;
  for (ndim = 0; list; ndim++)
    {
      /* Pull the first attribute value from the list and record that
	 attribute as another dimension in the attribute space.  */
      char *name = XSTR (XEXP (list, 0), 0);
      rtx *prev;

      if ((space[ndim].attr = find_attr (name, 0)) == 0
	  || space[ndim].attr->is_numeric)
	{
	  unmark_used_attributes (list, space, ndim);
	  return exp;
	}

      /* Add all remaining attribute values that refer to this attribute.  */
      space[ndim].num_values = 0;
      space[ndim].values = 0;
      prev = &list;
      for (link = list; link; link = *prev)
	if (! strcmp (XSTR (XEXP (link, 0), 0), name))
	  {
	    space[ndim].num_values++;
	    *prev = XEXP (link, 1);
	    XEXP (link, 1) = space[ndim].values;
	    space[ndim].values = link;
	  }
	else
	  prev = &XEXP (link, 1);

      /* Add sufficient members to the list of values to make the list
	 mutually exclusive and record the total size of the attribute
	 space.  */
      total *= add_values_to_cover (&space[ndim]);
    }

  /* Sort the attribute space so that the attributes go from non-constant
     to constant and from most values to least values.  */
  for (i = 0; i < ndim; i++)
    for (j = ndim - 1; j > i; j--)
      if ((space[j-1].attr->is_const && !space[j].attr->is_const)
	  || space[j-1].num_values < space[j].num_values)
	{
	  struct dimension tmp;
	  tmp = space[j];
	  space[j] = space[j-1];
	  space[j-1] = tmp;
	}

  /* Establish the initial current value.  */
  for (i = 0; i < ndim; i++)
    space[i].current_value = space[i].values;

  condtest = (rtx *) alloca (total * sizeof (rtx));
  condval = (rtx *) alloca (total * sizeof (rtx));

  /* Expand the tests and values by iterating over all values in the
     attribute space.  */
  for (i = 0;; i++)
    {
      condtest[i] = test_for_current_value (space, ndim);
      condval[i] = simplify_with_current_value (exp, space, ndim);
      if (! increment_current_value (space, ndim))
	break;
    }
  if (i != total - 1)
    abort ();

  /* We are now finished with the original expression.  */
  unmark_used_attributes (0, space, ndim);

  /* Find the most used constant value and make that the default.  */
  most_tests = -1;
  for (i = num_marks = 0; i < total; i++)
    if (GET_CODE (condval[i]) == CONST_STRING
	&& ! MEM_VOLATILE_P (condval[i]))
      {
	/* Mark the unmarked constant value and count how many are marked.  */
	MEM_VOLATILE_P (condval[i]) = 1;
	for (j = new_marks = 0; j < total; j++)
	  if (GET_CODE (condval[j]) == CONST_STRING
	      && MEM_VOLATILE_P (condval[j]))
	    new_marks++;
	if (new_marks - num_marks > most_tests)
	  {
	    most_tests = new_marks - num_marks;
	    defval = condval[i];
	  }
	num_marks = new_marks;
      }
  /* Clear all the marks.  */
  for (i = 0; i < total; i++)
    MEM_VOLATILE_P (condval[i]) = 0;

  /* Give up if nothing is constant.  */
  if (num_marks == 0)
    return exp;

  /* If all values are the default, use that.  */
  if (total == most_tests)
    return defval;

  /* Make a COND with the most common constant value the default.  (A more
     complex method where tests with the same value were combined didn't
     seem to improve things.)  */
  condexp = rtx_alloc (COND);
  XVEC (condexp, 0) = rtvec_alloc ((total - most_tests) * 2);
  XEXP (condexp, 1) = defval;
  for (i = j = 0; i < total; i++)
    if (condval[i] != defval)
      {
	XVECEXP (condexp, 0, 2 * j) = condtest[i];
	XVECEXP (condexp, 0, 2 * j + 1) = condval[i];
	j++;
      }

  return condexp;
}

/* Set the MEM_VOLATILE_P flag for all EQ_ATTR expressions in EXP and
   verify that EXP can be simplified to a constant term if all the EQ_ATTR
   tests have known value.  */

static int
find_and_mark_used_attributes (exp, terms, nterms)
     rtx exp, *terms;
     int *nterms;
{
  int i;

  switch (GET_CODE (exp))
    {
    case EQ_ATTR:
      if (! MEM_VOLATILE_P (exp))
	{
	  rtx link = rtx_alloc (EXPR_LIST);
	  XEXP (link, 0) = exp;
	  XEXP (link, 1) = *terms;
	  *terms = link;
	  *nterms += 1;
	  MEM_VOLATILE_P (exp) = 1;
	}
      return 1;

    case CONST_STRING:
    case CONST_INT:
      return 1;

    case IF_THEN_ELSE:
      if (! find_and_mark_used_attributes (XEXP (exp, 2), terms, nterms))
	return 0;
    case IOR:
    case AND:
      if (! find_and_mark_used_attributes (XEXP (exp, 1), terms, nterms))
	return 0;
    case NOT:
      if (! find_and_mark_used_attributes (XEXP (exp, 0), terms, nterms))
	return 0;
      return 1;

    case COND:
      for (i = 0; i < XVECLEN (exp, 0); i++)
	if (! find_and_mark_used_attributes (XVECEXP (exp, 0, i), terms, nterms))
	  return 0;
      if (! find_and_mark_used_attributes (XEXP (exp, 1), terms, nterms))
	return 0;
      return 1;

    default:
      return 0;
    }
}

/* Clear the MEM_VOLATILE_P flag in all EQ_ATTR expressions on LIST and
   in the values of the NDIM-dimensional attribute space SPACE.  */

static void
unmark_used_attributes (list, space, ndim)
     rtx list;
     struct dimension *space;
     int ndim;
{
  rtx link, exp;
  int i;

  for (i = 0; i < ndim; i++)
    unmark_used_attributes (space[i].values, 0, 0);

  for (link = list; link; link = XEXP (link, 1))
    {
      exp = XEXP (link, 0);
      if (GET_CODE (exp) == EQ_ATTR)
	MEM_VOLATILE_P (exp) = 0;
    }
}

/* Update the attribute dimension DIM so that all values of the attribute
   are tested.  Return the updated number of values.  */

static int
add_values_to_cover (dim)
     struct dimension *dim;
{
  struct attr_value *av;
  rtx exp, link, *prev;
  int nalt = 0;

  for (av = dim->attr->first_value; av; av = av->next)
    if (GET_CODE (av->value) == CONST_STRING)
      nalt++;

  if (nalt < dim->num_values)
    abort ();
  else if (nalt == dim->num_values)
    ; /* Ok.  */
  else if (nalt * 2 < dim->num_values * 3)
    {
      /* Most all the values of the attribute are used, so add all the unused
	 values.  */
      prev = &dim->values;
      for (link = dim->values; link; link = *prev)
	prev = &XEXP (link, 1);

      for (av = dim->attr->first_value; av; av = av->next)
	if (GET_CODE (av->value) == CONST_STRING)
	  {
	    exp = attr_eq (dim->attr->name, XSTR (av->value, 0));
	    if (MEM_VOLATILE_P (exp))
	      continue;

	    link = rtx_alloc (EXPR_LIST);
	    XEXP (link, 0) = exp;
	    XEXP (link, 1) = 0;
	    *prev = link;
	    prev = &XEXP (link, 1);
	  }
      dim->num_values = nalt;
    }
  else
    {
      rtx orexp = false_rtx;

      /* Very few values are used, so compute a mutually exclusive
	 expression.  (We could do this for numeric values if that becomes
	 important.)  */
      prev = &dim->values;
      for (link = dim->values; link; link = *prev)
	{
	  orexp = insert_right_side (IOR, orexp, XEXP (link, 0), -2, -2);
	  prev = &XEXP (link, 1);
	}
      link = rtx_alloc (EXPR_LIST);
      XEXP (link, 0) = attr_rtx (NOT, orexp);
      XEXP (link, 1) = 0;
      *prev = link;
      dim->num_values++;
    }
  return dim->num_values;
}

/* Increment the current value for the NDIM-dimensional attribute space SPACE
   and return FALSE if the increment overflowed.  */

static int
increment_current_value (space, ndim)
     struct dimension *space;
     int ndim;
{
  int i;

  for (i = ndim - 1; i >= 0; i--)
    {
      if ((space[i].current_value = XEXP (space[i].current_value, 1)) == 0)
	space[i].current_value = space[i].values;
      else
	return 1;
    }
  return 0;
}

/* Construct an expression corresponding to the current value for the
   NDIM-dimensional attribute space SPACE.  */

static rtx
test_for_current_value (space, ndim)
     struct dimension *space;
     int ndim;
{
  int i;
  rtx exp = true_rtx;

  for (i = 0; i < ndim; i++)
    exp = insert_right_side (AND, exp, XEXP (space[i].current_value, 0),
			     -2, -2);

  return exp;
}

/* Given the current value of the NDIM-dimensional attribute space SPACE,
   set the corresponding EQ_ATTR expressions to that value and reduce
   the expression EXP as much as possible.  On input [and output], all
   known EQ_ATTR expressions are set to FALSE.  */

static rtx
simplify_with_current_value (exp, space, ndim)
     rtx exp;
     struct dimension *space;
     int ndim;
{
  int i;
  rtx x;

  /* Mark each current value as TRUE.  */
  for (i = 0; i < ndim; i++)
    {
      x = XEXP (space[i].current_value, 0);
      if (GET_CODE (x) == EQ_ATTR)
	MEM_VOLATILE_P (x) = 0;
    }

  exp = simplify_with_current_value_aux (exp);

  /* Change each current value back to FALSE.  */
  for (i = 0; i < ndim; i++)
    {
      x = XEXP (space[i].current_value, 0);
      if (GET_CODE (x) == EQ_ATTR)
	MEM_VOLATILE_P (x) = 1;
    }

  return exp;
}

/* Reduce the expression EXP based on the MEM_VOLATILE_P settings of
   all EQ_ATTR expressions.  */

static rtx
simplify_with_current_value_aux (exp)
     rtx exp;
{
  register int i;
  rtx cond;

  switch (GET_CODE (exp))
    {
    case EQ_ATTR:
      if (MEM_VOLATILE_P (exp))
	return false_rtx;
      else
	return true_rtx;
    case CONST_STRING:
    case CONST_INT:
      return exp;

    case IF_THEN_ELSE:
      cond = simplify_with_current_value_aux (XEXP (exp, 0));
      if (cond == true_rtx)
	return simplify_with_current_value_aux (XEXP (exp, 1));
      else if (cond == false_rtx)
	return simplify_with_current_value_aux (XEXP (exp, 2));
      else
	return attr_rtx (IF_THEN_ELSE, cond,
			 simplify_with_current_value_aux (XEXP (exp, 1)),
			 simplify_with_current_value_aux (XEXP (exp, 2)));

    case IOR:
      cond = simplify_with_current_value_aux (XEXP (exp, 1));
      if (cond == true_rtx)
	return cond;
      else if (cond == false_rtx)
	return simplify_with_current_value_aux (XEXP (exp, 0));
      else
	return attr_rtx (IOR, cond,
			 simplify_with_current_value_aux (XEXP (exp, 0)));

    case AND:
      cond = simplify_with_current_value_aux (XEXP (exp, 1));
      if (cond == true_rtx)
	return simplify_with_current_value_aux (XEXP (exp, 0));
      else if (cond == false_rtx)
	return cond;
      else
	return attr_rtx (AND, cond,
			 simplify_with_current_value_aux (XEXP (exp, 0)));

    case NOT:
      cond = simplify_with_current_value_aux (XEXP (exp, 0));
      if (cond == true_rtx)
	return false_rtx;
      else if (cond == false_rtx)
	return true_rtx;
      else
	return attr_rtx (NOT, cond);

    case COND:
      for (i = 0; i < XVECLEN (exp, 0); i += 2)
	{
	  cond = simplify_with_current_value_aux (XVECEXP (exp, 0, i));
	  if (cond == true_rtx)
	    return simplify_with_current_value_aux (XVECEXP (exp, 0, i + 1));
	  else if (cond == false_rtx)
	    continue;
	  else
	    abort (); /* With all EQ_ATTR's of known value, a case should
			 have been selected.  */
	}
      return simplify_with_current_value_aux (XEXP (exp, 1));

    default:
      abort ();
    }
}

/* Clear the MEM_IN_STRUCT_P flag in EXP and its subexpressions.  */

static void
clear_struct_flag (x)
     rtx x;
{
  register int i;
  register int j;
  register enum rtx_code code;
  register const char *fmt;

  MEM_IN_STRUCT_P (x) = 0;
  if (RTX_UNCHANGING_P (x))
    return;

  code = GET_CODE (x);

  switch (code)
    {
    case REG:
    case QUEUED:
    case CONST_INT:
    case CONST_DOUBLE:
    case SYMBOL_REF:
    case CODE_LABEL:
    case PC:
    case CC0:
    case EQ_ATTR:
    case ATTR_FLAG:
      return;
      
    default:
      break;
    }

  /* Compare the elements.  If any pair of corresponding elements
     fail to match, return 0 for the whole things.  */

  fmt = GET_RTX_FORMAT (code);
  for (i = GET_RTX_LENGTH (code) - 1; i >= 0; i--)
    {
      switch (fmt[i])
	{
	case 'V':
	case 'E':
	  for (j = 0; j < XVECLEN (x, i); j++)
	    clear_struct_flag (XVECEXP (x, i, j));
	  break;

	case 'e':
	  clear_struct_flag (XEXP (x, i));
	  break;
	}
    }
}

/* Return the number of RTX objects making up the expression X.
   But if we count more than MAX objects, stop counting.  */

static int
count_sub_rtxs (x, max)
     rtx x;
     int max;
{
  register int i;
  register int j;
  register enum rtx_code code;
  register const char *fmt;
  int total = 0;

  code = GET_CODE (x);

  switch (code)
    {
    case REG:
    case QUEUED:
    case CONST_INT:
    case CONST_DOUBLE:
    case SYMBOL_REF:
    case CODE_LABEL:
    case PC:
    case CC0:
    case EQ_ATTR:
    case ATTR_FLAG:
      return 1;
      
    default:
      break;
    }

  /* Compare the elements.  If any pair of corresponding elements
     fail to match, return 0 for the whole things.  */

  fmt = GET_RTX_FORMAT (code);
  for (i = GET_RTX_LENGTH (code) - 1; i >= 0; i--)
    {
      if (total >= max)
	return total;

      switch (fmt[i])
	{
	case 'V':
	case 'E':
	  for (j = 0; j < XVECLEN (x, i); j++)
	    total += count_sub_rtxs (XVECEXP (x, i, j), max);
	  break;

	case 'e':
	  total += count_sub_rtxs (XEXP (x, i), max);
	  break;
	}
    }
  return total;

}

/* Create table entries for DEFINE_ATTR.  */

static void
gen_attr (exp)
     rtx exp;
{
  struct attr_desc *attr;
  struct attr_value *av;
  char *name_ptr;
  char *p;

  /* Make a new attribute structure.  Check for duplicate by looking at
     attr->default_val, since it is initialized by this routine.  */
  attr = find_attr (XSTR (exp, 0), 1);
  if (attr->default_val)
    fatal ("Duplicate definition for `%s' attribute", attr->name);

  if (*XSTR (exp, 1) == '\0')
    attr->is_numeric = 1;
  else
    {
      name_ptr = XSTR (exp, 1);
      while ((p = next_comma_elt (&name_ptr)) != NULL)
	{
	  av = (struct attr_value *) oballoc (sizeof (struct attr_value));
	  av->value = attr_rtx (CONST_STRING, p);
	  av->next = attr->first_value;
	  attr->first_value = av;
	  av->first_insn = NULL;
	  av->num_insns = 0;
	  av->has_asm_insn = 0;
	}
    }

  if (GET_CODE (XEXP (exp, 2)) == CONST)
    {
      attr->is_const = 1;
      if (attr->is_numeric)
	fatal ("Constant attributes may not take numeric values");
      /* Get rid of the CONST node.  It is allowed only at top-level.  */
      XEXP (exp, 2) = XEXP (XEXP (exp, 2), 0);
    }

  if (! strcmp (attr->name, "length") && ! attr->is_numeric)
    fatal ("`length' attribute must take numeric values");

  /* Set up the default value.  */
  XEXP (exp, 2) = check_attr_value (XEXP (exp, 2), attr);
  attr->default_val = get_attr_value (XEXP (exp, 2), attr, -2);
}

/* Given a pattern for DEFINE_PEEPHOLE or DEFINE_INSN, return the number of
   alternatives in the constraints.  Assume all MATCH_OPERANDs have the same
   number of alternatives as this should be checked elsewhere.  */

static int
count_alternatives (exp)
     rtx exp;
{
  int i, j, n;
  const char *fmt;
  
  if (GET_CODE (exp) == MATCH_OPERAND)
    return n_comma_elts (XSTR (exp, 2));

  for (i = 0, fmt = GET_RTX_FORMAT (GET_CODE (exp));
       i < GET_RTX_LENGTH (GET_CODE (exp)); i++)
    switch (*fmt++)
      {
      case 'e':
      case 'u':
	n = count_alternatives (XEXP (exp, i));
	if (n)
	  return n;
	break;

      case 'E':
      case 'V':
	if (XVEC (exp, i) != NULL)
	  for (j = 0; j < XVECLEN (exp, i); j++)
	    {
	      n = count_alternatives (XVECEXP (exp, i, j));
	      if (n)
		return n;
	    }
      }

  return 0;
}

/* Returns non-zero if the given expression contains an EQ_ATTR with the
   `alternative' attribute.  */

static int
compares_alternatives_p (exp)
     rtx exp;
{
  int i, j;
  const char *fmt;

  if (GET_CODE (exp) == EQ_ATTR && XSTR (exp, 0) == alternative_name)
    return 1;

  for (i = 0, fmt = GET_RTX_FORMAT (GET_CODE (exp));
       i < GET_RTX_LENGTH (GET_CODE (exp)); i++)
    switch (*fmt++)
      {
      case 'e':
      case 'u':
	if (compares_alternatives_p (XEXP (exp, i)))
	  return 1;
	break;

      case 'E':
	for (j = 0; j < XVECLEN (exp, i); j++)
	  if (compares_alternatives_p (XVECEXP (exp, i, j)))
	    return 1;
	break;
      }

  return 0;
}

/* Returns non-zero is INNER is contained in EXP.  */

static int
contained_in_p (inner, exp)
     rtx inner;
     rtx exp;
{
  int i, j;
  const char *fmt;

  if (rtx_equal_p (inner, exp))
    return 1;

  for (i = 0, fmt = GET_RTX_FORMAT (GET_CODE (exp));
       i < GET_RTX_LENGTH (GET_CODE (exp)); i++)
    switch (*fmt++)
      {
      case 'e':
      case 'u':
	if (contained_in_p (inner, XEXP (exp, i)))
	  return 1;
	break;

      case 'E':
	for (j = 0; j < XVECLEN (exp, i); j++)
	  if (contained_in_p (inner, XVECEXP (exp, i, j)))
	    return 1;
	break;
      }

  return 0;
}
	
/* Process DEFINE_PEEPHOLE, DEFINE_INSN, and DEFINE_ASM_ATTRIBUTES.  */

static void
gen_insn (exp)
     rtx exp;
{
  struct insn_def *id;

  id = (struct insn_def *) oballoc (sizeof (struct insn_def));
  id->next = defs;
  defs = id;
  id->def = exp;

  switch (GET_CODE (exp))
    {
    case DEFINE_INSN:
      id->insn_code = insn_code_number++;
      id->insn_index = insn_index_number++;
      id->num_alternatives = count_alternatives (exp);
      if (id->num_alternatives == 0)
	id->num_alternatives = 1;
      id->vec_idx = 4;
      break;

    case DEFINE_PEEPHOLE:
      id->insn_code = insn_code_number++;
      id->insn_index = insn_index_number++;
      id->num_alternatives = count_alternatives (exp);
      if (id->num_alternatives == 0)
	id->num_alternatives = 1;
      id->vec_idx = 3;
      break;

    case DEFINE_ASM_ATTRIBUTES:
      id->insn_code = -1;
      id->insn_index = -1;
      id->num_alternatives = 1;
      id->vec_idx = 0;
      got_define_asm_attributes = 1;
      break;
      
    default:
      abort ();
    }
}

/* Process a DEFINE_DELAY.  Validate the vector length, check if annul
   true or annul false is specified, and make a `struct delay_desc'.  */

static void
gen_delay (def)
     rtx def;
{
  struct delay_desc *delay;
  int i;

  if (XVECLEN (def, 1) % 3 != 0)
    fatal ("Number of elements in DEFINE_DELAY must be multiple of three.");

  for (i = 0; i < XVECLEN (def, 1); i += 3)
    {
      if (XVECEXP (def, 1, i + 1))
	have_annul_true = 1;
      if (XVECEXP (def, 1, i + 2))
	have_annul_false = 1;
    }
  
  delay = (struct delay_desc *) oballoc (sizeof (struct delay_desc));
  delay->def = def;
  delay->num = ++num_delays;
  delay->next = delays;
  delays = delay;
}

/* Process a DEFINE_FUNCTION_UNIT.  

   This gives information about a function unit contained in the CPU.
   We fill in a `struct function_unit_op' and a `struct function_unit'
   with information used later by `expand_unit'.  */

static void
gen_unit (def)
     rtx def;
{
  struct function_unit *unit;
  struct function_unit_op *op;
  char *name = XSTR (def, 0);
  int multiplicity = XINT (def, 1);
  int simultaneity = XINT (def, 2);
  rtx condexp = XEXP (def, 3);
  int ready_cost = MAX (XINT (def, 4), 1);
  int issue_delay = MAX (XINT (def, 5), 1);

  /* See if we have already seen this function unit.  If so, check that
     the multiplicity and simultaneity values are the same.  If not, make
     a structure for this function unit.  */
  for (unit = units; unit; unit = unit->next)
    if (! strcmp (unit->name, name))
      {
	if (unit->multiplicity != multiplicity
	    || unit->simultaneity != simultaneity)
	  fatal ("Differing specifications given for `%s' function unit.",
		 unit->name);
	break;
      }

  if (unit == 0)
    {
      unit = (struct function_unit *) oballoc (sizeof (struct function_unit));
      unit->name = name;
      unit->multiplicity = multiplicity;
      unit->simultaneity = simultaneity;
      unit->issue_delay.min = unit->issue_delay.max = issue_delay;
      unit->num = num_units++;
      unit->num_opclasses = 0;
      unit->condexp = false_rtx;
      unit->ops = 0;
      unit->next = units;
      units = unit;
    }

  /* Make a new operation class structure entry and initialize it.  */
  op = (struct function_unit_op *) oballoc (sizeof (struct function_unit_op));
  op->condexp = condexp;
  op->num = unit->num_opclasses++;
  op->ready = ready_cost;
  op->issue_delay = issue_delay;
  op->next = unit->ops;
  unit->ops = op;
  num_unit_opclasses++;

  /* Set our issue expression based on whether or not an optional conflict
     vector was specified.  */
  if (XVEC (def, 6))
    {
      /* Compute the IOR of all the specified expressions.  */
      rtx orexp = false_rtx;
      int i;

      for (i = 0; i < XVECLEN (def, 6); i++)
	orexp = insert_right_side (IOR, orexp, XVECEXP (def, 6, i), -2, -2);

      op->conflict_exp = orexp;
      extend_range (&unit->issue_delay, 1, issue_delay);
    }
  else
    {
      op->conflict_exp = true_rtx;
      extend_range (&unit->issue_delay, issue_delay, issue_delay);
    }

  /* Merge our conditional into that of the function unit so we can determine
     which insns are used by the function unit.  */
  unit->condexp = insert_right_side (IOR, unit->condexp, op->condexp, -2, -2);
}

/* Given a piece of RTX, print a C expression to test its truth value.
   We use AND and IOR both for logical and bit-wise operations, so 
   interpret them as logical unless they are inside a comparison expression.
   The first bit of FLAGS will be non-zero in that case.

   Set the second bit of FLAGS to make references to attribute values use
   a cached local variable instead of calling a function.  */

static void
write_test_expr (exp, flags)
     rtx exp;
     int flags;
{
  int comparison_operator = 0;
  RTX_CODE code;
  struct attr_desc *attr;

  /* In order not to worry about operator precedence, surround our part of
     the expression with parentheses.  */

  printf ("(");
  code = GET_CODE (exp);
  switch (code)
    {
    /* Binary operators.  */
    case EQ: case NE:
    case GE: case GT: case GEU: case GTU:
    case LE: case LT: case LEU: case LTU:
      comparison_operator = 1;

    case PLUS:   case MINUS:  case MULT:     case DIV:      case MOD:
    case AND:    case IOR:    case XOR:
    case ASHIFT: case LSHIFTRT: case ASHIFTRT:
      write_test_expr (XEXP (exp, 0), flags | comparison_operator);
      switch (code)
        {
	case EQ:
	  printf (" == ");
	  break;
	case NE:
	  printf (" != ");
	  break;
	case GE:
	  printf (" >= ");
	  break;
	case GT:
	  printf (" > ");
	  break;
	case GEU:
	  printf (" >= (unsigned) ");
	  break;
	case GTU:
	  printf (" > (unsigned) ");
	  break;
	case LE:
	  printf (" <= ");
	  break;
	case LT:
	  printf (" < ");
	  break;
	case LEU:
	  printf (" <= (unsigned) ");
	  break;
	case LTU:
	  printf (" < (unsigned) ");
	  break;
	case PLUS:
	  printf (" + ");
	  break;
	case MINUS:
	  printf (" - ");
	  break;
	case MULT:
	  printf (" * ");
	  break;
	case DIV:
	  printf (" / ");
	  break;
	case MOD:
	  printf (" %% ");
	  break;
	case AND:
	  if (flags & 1)
	    printf (" & ");
	  else
	    printf (" && ");
	  break;
	case IOR:
	  if (flags & 1)
	    printf (" | ");
	  else
	    printf (" || ");
	  break;
	case XOR:
	  printf (" ^ ");
	  break;
	case ASHIFT:
	  printf (" << ");
	  break;
	case LSHIFTRT:
	case ASHIFTRT:
	  printf (" >> ");
	  break;
	default:
	  abort ();
        }

      write_test_expr (XEXP (exp, 1), flags | comparison_operator);
      break;

    case NOT:
      /* Special-case (not (eq_attrq "alternative" "x")) */
      if (! (flags & 1) && GET_CODE (XEXP (exp, 0)) == EQ_ATTR
	  && XSTR (XEXP (exp, 0), 0) == alternative_name)
	{
	  printf ("which_alternative != %s", XSTR (XEXP (exp, 0), 1));
	  break;
	}

      /* Otherwise, fall through to normal unary operator.  */

    /* Unary operators.  */   
    case ABS:  case NEG:
      switch (code)
	{
	case NOT:
	  if (flags & 1)
	    printf ("~ ");
	  else
	    printf ("! ");
	  break;
	case ABS:
	  printf ("abs ");
	  break;
	case NEG:
	  printf ("-");
	  break;
	default:
	  abort ();
	}

      write_test_expr (XEXP (exp, 0), flags);
      break;

    /* Comparison test of an attribute with a value.  Most of these will
       have been removed by optimization.   Handle "alternative"
       specially and give error if EQ_ATTR present inside a comparison.  */
    case EQ_ATTR:
      if (flags & 1)
	fatal ("EQ_ATTR not valid inside comparison");

      if (XSTR (exp, 0) == alternative_name)
	{
	  printf ("which_alternative == %s", XSTR (exp, 1));
	  break;
	}

      attr = find_attr (XSTR (exp, 0), 0);
      if (! attr) abort ();

      /* Now is the time to expand the value of a constant attribute.  */
      if (attr->is_const)
	{
	  write_test_expr (evaluate_eq_attr (exp, attr->default_val->value,
					     -2, -2),
			   flags);
	}
      else
	{
	  if (flags & 2)
	    printf ("attr_%s", attr->name);
	  else
	    printf ("get_attr_%s (insn)", attr->name);
	  printf (" == ");
	  write_attr_valueq (attr, XSTR (exp, 1));
	}
      break;

    /* Comparison test of flags for define_delays.  */
    case ATTR_FLAG:
      if (flags & 1)
	fatal ("ATTR_FLAG not valid inside comparison");
      printf ("(flags & ATTR_FLAG_%s) != 0", XSTR (exp, 0));
      break;

    /* See if an operand matches a predicate.  */
    case MATCH_OPERAND:
      /* If only a mode is given, just ensure the mode matches the operand.
	 If neither a mode nor predicate is given, error.  */
     if (XSTR (exp, 1) == NULL || *XSTR (exp, 1) == '\0')
	{
	  if (GET_MODE (exp) == VOIDmode)
	    fatal ("Null MATCH_OPERAND specified as test");
	  else
	    printf ("GET_MODE (operands[%d]) == %smode",
		    XINT (exp, 0), GET_MODE_NAME (GET_MODE (exp)));
	}
      else
	printf ("%s (operands[%d], %smode)",
		XSTR (exp, 1), XINT (exp, 0), GET_MODE_NAME (GET_MODE (exp)));
      break;

    case MATCH_INSN:
      printf ("%s (insn)", XSTR (exp, 0));
      break;

    /* Constant integer.  */
    case CONST_INT:
      printf (HOST_WIDE_INT_PRINT_DEC, XWINT (exp, 0));
      break;

    /* A random C expression.  */
    case SYMBOL_REF:
      printf ("%s", XSTR (exp, 0));
      break;

    /* The address of the branch target.  */
    case MATCH_DUP:
      printf ("insn_addresses[INSN_UID (GET_CODE (operands[%d]) == LABEL_REF ? XEXP (operands[%d], 0) : operands[%d])]",
	      XINT (exp, 0), XINT (exp, 0), XINT (exp, 0));
      break;

    case PC:
      /* The address of the current insn.  We implement this actually as the
	 address of the current insn for backward branches, but the last
	 address of the next insn for forward branches, and both with
	 adjustments that account for the worst-case possible stretching of
	 intervening alignments between this insn and its destination.  */
      printf("insn_current_reference_address (insn)");
      break;

    case CONST_STRING:
      printf ("%s", XSTR (exp, 0));
      break;

    case IF_THEN_ELSE:
      write_test_expr (XEXP (exp, 0), flags & 2);
      printf (" ? ");
      write_test_expr (XEXP (exp, 1), flags | 1);
      printf (" : ");
      write_test_expr (XEXP (exp, 2), flags | 1);
      break;

    default:
      fatal ("bad RTX code `%s' in attribute calculation\n",
	     GET_RTX_NAME (code));
    }

  printf (")");
}

/* Given an attribute value, return the maximum CONST_STRING argument
   encountered.  Set *UNKNOWNP and return INT_MAX if the value is unknown.  */

static int
max_attr_value (exp, unknownp)
     rtx exp;
     int *unknownp;
{
  int current_max;
  int i, n;

  switch (GET_CODE (exp))
    {
    case CONST_STRING:
      current_max = atoi (XSTR (exp, 0));
      break;

    case COND:
      current_max = max_attr_value (XEXP (exp, 1), unknownp);
      for (i = 0; i < XVECLEN (exp, 0); i += 2)
	{
	  n = max_attr_value (XVECEXP (exp, 0, i + 1), unknownp);
	  if (n > current_max)
	    current_max = n;
	}
      break;

    case IF_THEN_ELSE:
      current_max = max_attr_value (XEXP (exp, 1), unknownp);
      n = max_attr_value (XEXP (exp, 2), unknownp);
      if (n > current_max)
	current_max = n;
      break;

    default:
      *unknownp = 1;
      current_max = INT_MAX;
      break;
    }

  return current_max;
}

/* Given an attribute value, return the result of ORing together all
   CONST_STRING arguments encountered.  Set *UNKNOWNP and return -1
   if the numeric value is not known.  */

static int
or_attr_value (exp, unknownp)
     rtx exp;
     int *unknownp;
{
  int current_or;
  int i;

  switch (GET_CODE (exp))
    {
    case CONST_STRING:
      current_or = atoi (XSTR (exp, 0));
      break;

    case COND:
      current_or = or_attr_value (XEXP (exp, 1), unknownp);
      for (i = 0; i < XVECLEN (exp, 0); i += 2)
	current_or |= or_attr_value (XVECEXP (exp, 0, i + 1), unknownp);
      break;

    case IF_THEN_ELSE:
      current_or = or_attr_value (XEXP (exp, 1), unknownp);
      current_or |= or_attr_value (XEXP (exp, 2), unknownp);
      break;

    default:
      *unknownp = 1;
      current_or = -1;
      break;
    }

  return current_or;
}

/* Scan an attribute value, possibly a conditional, and record what actions
   will be required to do any conditional tests in it.

   Specifically, set
	`must_extract'	  if we need to extract the insn operands
	`must_constrain'  if we must compute `which_alternative'
	`address_used'	  if an address expression was used
	`length_used'	  if an (eq_attr "length" ...) was used
 */

static void
walk_attr_value (exp)
     rtx exp;
{
  register int i, j;
  register const char *fmt;
  RTX_CODE code;

  if (exp == NULL)
    return;

  code = GET_CODE (exp);
  switch (code)
    {
    case SYMBOL_REF:
      if (! RTX_UNCHANGING_P (exp))
	/* Since this is an arbitrary expression, it can look at anything.
	   However, constant expressions do not depend on any particular
	   insn.  */
	must_extract = must_constrain = 1;
      return;

    case MATCH_OPERAND:
      must_extract = 1;
      return;

    case EQ_ATTR:
      if (XSTR (exp, 0) == alternative_name)
	must_extract = must_constrain = 1;
      else if (strcmp (XSTR (exp, 0), "length") == 0)
	length_used = 1;
      return;

    case MATCH_DUP:
      must_extract = 1;
      address_used = 1;
      return;

    case PC:
      address_used = 1;
      return;

    case ATTR_FLAG:
      return;

    default:
      break;
    }

  for (i = 0, fmt = GET_RTX_FORMAT (code); i < GET_RTX_LENGTH (code); i++)
    switch (*fmt++)
      {
      case 'e':
      case 'u':
	walk_attr_value (XEXP (exp, i));
	break;

      case 'E':
	if (XVEC (exp, i) != NULL)
	  for (j = 0; j < XVECLEN (exp, i); j++)
	    walk_attr_value (XVECEXP (exp, i, j));
	break;
      }
}

/* Write out a function to obtain the attribute for a given INSN.  */

static void
write_attr_get (attr)
     struct attr_desc *attr;
{
  struct attr_value *av, *common_av;

  /* Find the most used attribute value.  Handle that as the `default' of the
     switch we will generate.  */
  common_av = find_most_used (attr);

  /* Write out prototype of function. */
  if (!attr->is_numeric)
    printf ("extern enum attr_%s ", attr->name);
  else if (attr->unsigned_p)
    printf ("extern unsigned int ");
  else
    printf ("extern int ");
  /* If the attribute name starts with a star, the remainder is the name of
     the subroutine to use, instead of `get_attr_...'.  */
  if (attr->name[0] == '*')
    printf ("%s PROTO ((rtx));\n", &attr->name[1]);
  else
    printf ("get_attr_%s PROTO ((%s));\n", attr->name,
	    (attr->is_const ? "void" : "rtx"));

  /* Write out start of function, then all values with explicit `case' lines,
     then a `default', then the value with the most uses.  */
  if (!attr->is_numeric)
    printf ("enum attr_%s\n", attr->name);
  else if (attr->unsigned_p)
    printf ("unsigned int\n");
  else
    printf ("int\n");

  /* If the attribute name starts with a star, the remainder is the name of
     the subroutine to use, instead of `get_attr_...'.  */
  if (attr->name[0] == '*')
    printf ("%s (insn)\n", &attr->name[1]);
  else if (attr->is_const == 0)
    printf ("get_attr_%s (insn)\n", attr->name);
  else
    {
      printf ("get_attr_%s ()\n", attr->name);
      printf ("{\n");

      for (av = attr->first_value; av; av = av->next)
	if (av->num_insns != 0)
	  write_attr_set (attr, 2, av->value, "return", ";",
			  true_rtx, av->first_insn->insn_code,
			  av->first_insn->insn_index);

      printf ("}\n\n");
      return;
    }

  printf ("     rtx insn;\n");
  printf ("{\n");

  if (GET_CODE (common_av->value) == FFS)
    {
      rtx p = XEXP (common_av->value, 0);

      /* No need to emit code to abort if the insn is unrecognized; the 
         other get_attr_foo functions will do that when we call them.  */

      write_toplevel_expr (p);

      printf ("\n  if (accum && accum == (accum & -accum))\n");
      printf ("    {\n");
      printf ("      int i;\n");
      printf ("      for (i = 0; accum >>= 1; ++i) continue;\n");
      printf ("      accum = i;\n");
      printf ("    }\n  else\n");
      printf ("    accum = ~accum;\n");
      printf ("  return accum;\n}\n\n");
    }
  else
    {
      printf ("  switch (recog_memoized (insn))\n");
      printf ("    {\n");

      for (av = attr->first_value; av; av = av->next)
	if (av != common_av)
	  write_attr_case (attr, av, 1, "return", ";", 4, true_rtx);

      write_attr_case (attr, common_av, 0, "return", ";", 4, true_rtx);
      printf ("    }\n}\n\n");
    }
}

/* Given an AND tree of known true terms (because we are inside an `if' with
   that as the condition or are in an `else' clause) and an expression,
   replace any known true terms with TRUE.  Use `simplify_and_tree' to do
   the bulk of the work.  */

static rtx
eliminate_known_true (known_true, exp, insn_code, insn_index)
     rtx known_true;
     rtx exp;
     int insn_code, insn_index;
{
  rtx term;

  known_true = SIMPLIFY_TEST_EXP (known_true, insn_code, insn_index);

  if (GET_CODE (known_true) == AND)
    {
      exp = eliminate_known_true (XEXP (known_true, 0), exp,
				  insn_code, insn_index);
      exp = eliminate_known_true (XEXP (known_true, 1), exp,
				  insn_code, insn_index);
    }
  else
    {
      term = known_true;
      exp = simplify_and_tree (exp, &term, insn_code, insn_index);
    }

  return exp;
}

/* Write out a series of tests and assignment statements to perform tests and
   sets of an attribute value.  We are passed an indentation amount and prefix
   and suffix strings to write around each attribute value (e.g., "return"
   and ";").  */

static void
write_attr_set (attr, indent, value, prefix, suffix, known_true,
		insn_code, insn_index)
     struct attr_desc *attr;
     int indent;
     rtx value;
     const char *prefix;
     const char *suffix;
     rtx known_true;
     int insn_code, insn_index;
{
  if (GET_CODE (value) == COND)
    {
      /* Assume the default value will be the default of the COND unless we
	 find an always true expression.  */
      rtx default_val = XEXP (value, 1);
      rtx our_known_true = known_true;
      rtx newexp;
      int first_if = 1;
      int i;

      for (i = 0; i < XVECLEN (value, 0); i += 2)
	{
	  rtx testexp;
	  rtx inner_true;

	  testexp = eliminate_known_true (our_known_true,
					  XVECEXP (value, 0, i),
					  insn_code, insn_index);
	  newexp = attr_rtx (NOT, testexp);
	  newexp  = insert_right_side (AND, our_known_true, newexp,
				       insn_code, insn_index);

	  /* If the test expression is always true or if the next `known_true'
	     expression is always false, this is the last case, so break
	     out and let this value be the `else' case.  */
	  if (testexp == true_rtx || newexp == false_rtx)
	    {
	      default_val = XVECEXP (value, 0, i + 1);
	      break;
	    }

	  /* Compute the expression to pass to our recursive call as being
	     known true.  */
	  inner_true = insert_right_side (AND, our_known_true,
					  testexp, insn_code, insn_index);

	  /* If this is always false, skip it.  */
	  if (inner_true == false_rtx)
	    continue;

	  write_indent (indent);
	  printf ("%sif ", first_if ? "" : "else ");
	  first_if = 0;
	  write_test_expr (testexp, 0);
	  printf ("\n");
	  write_indent (indent + 2);
	  printf ("{\n");

	  write_attr_set (attr, indent + 4,  
			  XVECEXP (value, 0, i + 1), prefix, suffix,
			  inner_true, insn_code, insn_index);
	  write_indent (indent + 2);
	  printf ("}\n");
	  our_known_true = newexp;
	}

      if (! first_if)
	{
	  write_indent (indent);
	  printf ("else\n");
	  write_indent (indent + 2);
	  printf ("{\n");
	}

      write_attr_set (attr, first_if ? indent : indent + 4, default_val,
		      prefix, suffix, our_known_true, insn_code, insn_index);

      if (! first_if)
	{
	  write_indent (indent + 2);
	  printf ("}\n");
	}
    }
  else
    {
      write_indent (indent);
      printf ("%s ", prefix);
      write_attr_value (attr, value);
      printf ("%s\n", suffix);
    }
}

/* Write out the computation for one attribute value.  */

static void
write_attr_case (attr, av, write_case_lines, prefix, suffix, indent,
		 known_true)
     struct attr_desc *attr;
     struct attr_value *av;
     int write_case_lines;
     const char *prefix, *suffix;
     int indent;
     rtx known_true;
{
  struct insn_ent *ie;

  if (av->num_insns == 0)
    return;

  if (av->has_asm_insn)
    {
      write_indent (indent);
      printf ("case -1:\n");
      write_indent (indent + 2);
      printf ("if (GET_CODE (PATTERN (insn)) != ASM_INPUT\n");
      write_indent (indent + 2);
      printf ("    && asm_noperands (PATTERN (insn)) < 0)\n");
      write_indent (indent + 2);
      printf ("  fatal_insn_not_found (insn);\n");
    }

  if (write_case_lines)
    {
      for (ie = av->first_insn; ie; ie = ie->next)
	if (ie->insn_code != -1)
	  {
	    write_indent (indent);
	    printf ("case %d:\n", ie->insn_code);
	  }
    }
  else
    {
      write_indent (indent);
      printf ("default:\n");
    }

  /* See what we have to do to output this value.  */
  must_extract = must_constrain = address_used = 0;
  walk_attr_value (av->value);

  if (must_extract)
    {
      write_indent (indent + 2);
      printf ("extract_insn (insn);\n");
    }

  if (must_constrain)
    {
      write_indent (indent + 2);
      printf ("if (! constrain_operands (reload_completed))\n");
      write_indent (indent + 2);
      printf ("  fatal_insn_not_found (insn);\n");
    }

  write_attr_set (attr, indent + 2, av->value, prefix, suffix,
		  known_true, av->first_insn->insn_code,
		  av->first_insn->insn_index);

  if (strncmp (prefix, "return", 6))
    {
      write_indent (indent + 2);
      printf ("break;\n");
    }
  printf ("\n");
}

/* Search for uses of non-const attributes and write code to cache them.  */

static int
write_expr_attr_cache (p, attr)
     rtx p;
     struct attr_desc *attr;
{
  const char *fmt;
  int i, ie, j, je;

  if (GET_CODE (p) == EQ_ATTR)
    {
      if (XSTR (p, 0) != attr->name)
	return 0;

      if (!attr->is_numeric)
	printf ("  register enum attr_%s ", attr->name);
      else if (attr->unsigned_p)
	printf ("  register unsigned int ");
      else
	printf ("  register int ");

      printf ("attr_%s = get_attr_%s (insn);\n", attr->name, attr->name);
      return 1;
    }

  fmt = GET_RTX_FORMAT (GET_CODE (p));
  ie = GET_RTX_LENGTH (GET_CODE (p));
  for (i = 0; i < ie; i++)
    {
      switch (*fmt++)
	{
	case 'e':
	  if (write_expr_attr_cache (XEXP (p, i), attr))
	    return 1;
	  break;

	case 'E':
	  je = XVECLEN (p, i);
	  for (j = 0; j < je; ++j)
	    if (write_expr_attr_cache (XVECEXP (p, i, j), attr))
	      return 1;
	  break;
	}
    }

  return 0;
}

/* Evaluate an expression at top level.  A front end to write_test_expr,
   in which we cache attribute values and break up excessively large
   expressions to cater to older compilers.  */

static void
write_toplevel_expr (p)
     rtx p;
{
  struct attr_desc *attr;
  int i;

  for (i = 0; i < MAX_ATTRS_INDEX; ++i)
    for (attr = attrs[i]; attr ; attr = attr->next)
      if (!attr->is_const)
	write_expr_attr_cache (p, attr);

  printf("  register unsigned long accum = 0;\n\n");

  while (GET_CODE (p) == IOR)
    {
      rtx e;
      if (GET_CODE (XEXP (p, 0)) == IOR)
	e = XEXP (p, 1), p = XEXP (p, 0);
      else
	e = XEXP (p, 0), p = XEXP (p, 1);

      printf ("  accum |= ");
      write_test_expr (e, 3);
      printf (";\n");
    }
  printf ("  accum |= ");
  write_test_expr (p, 3);
  printf (";\n");
}

/* Utilities to write names in various forms.  */

static void
write_unit_name (prefix, num, suffix)
     const char *prefix;
     int num;
     const char *suffix;
{
  struct function_unit *unit;

  for (unit = units; unit; unit = unit->next)
    if (unit->num == num)
      {
	printf ("%s%s%s", prefix, unit->name, suffix);
	return;
      }

  printf ("%s<unknown>%s", prefix, suffix);
}

static void
write_attr_valueq (attr, s)
     struct attr_desc *attr;
     char *s;
{
  if (attr->is_numeric)
    {
      int num = atoi (s);

      printf ("%d", num);

      /* Make the blockage range values and function units used values easier
         to read.  */
      if (attr->func_units_p)
	{
	  if (num == -1)
	    printf (" /* units: none */");
	  else if (num >= 0)
	    write_unit_name (" /* units: ", num, " */");
	  else
	    {
	      int i;
	      const char *sep = " /* units: ";
	      for (i = 0, num = ~num; num; i++, num >>= 1)
		if (num & 1)
		  {
		    write_unit_name (sep, i, (num == 1) ? " */" : "");
		    sep = ", ";
		  }
	    }
	}

      else if (attr->blockage_p)
	printf (" /* min %d, max %d */", num >> (HOST_BITS_PER_INT / 2),
		num & ((1 << (HOST_BITS_PER_INT / 2)) - 1));

      else if (num > 9 || num < 0)
	printf (" /* 0x%x */", num);
    }
  else
    {
      write_upcase (attr->name);
      printf ("_");
      write_upcase (s);
    }
}

static void
write_attr_value (attr, value)
     struct attr_desc *attr;
     rtx value;
{
  int op;

  switch (GET_CODE (value))
    {
    case CONST_STRING:
      write_attr_valueq (attr, XSTR (value, 0));
      break;

    case SYMBOL_REF:
      fputs (XSTR (value, 0), stdout);
      break;

    case ATTR:
      {
	struct attr_desc *attr2 = find_attr (XSTR (value, 0), 0);
	printf ("get_attr_%s (%s)", attr2->name, 
		(attr2->is_const ? "" : "insn"));
      }
      break;

    case PLUS:
      op = '+';
      goto do_operator;
    case MINUS:
      op = '-';
      goto do_operator;
    case MULT:
      op = '*';
      goto do_operator;
    case DIV:
      op = '/';
      goto do_operator;
    case MOD:
      op = '%';
      goto do_operator;

    do_operator:
      write_attr_value (attr, XEXP (value, 0));
      putchar (' ');
      putchar (op);
      putchar (' ');
      write_attr_value (attr, XEXP (value, 1));
      break;

    default:
      abort ();
    }
}

static void
write_upcase (str)
     const char *str;
{
  while (*str)
  {
    /* The argument of TOUPPER should not have side effects.  */
    putchar (TOUPPER(*str));
    str++;
  }
}

static void
write_indent (indent)
     int indent;
{
  for (; indent > 8; indent -= 8)
    printf ("\t");

  for (; indent; indent--)
    printf (" ");
}

/* Write a subroutine that is given an insn that requires a delay slot, a
   delay slot ordinal, and a candidate insn.  It returns non-zero if the
   candidate can be placed in the specified delay slot of the insn.

   We can write as many as three subroutines.  `eligible_for_delay'
   handles normal delay slots, `eligible_for_annul_true' indicates that
   the specified insn can be annulled if the branch is true, and likewise
   for `eligible_for_annul_false'.

   KIND is a string distinguishing these three cases ("delay", "annul_true",
   or "annul_false").  */

static void
write_eligible_delay (kind)
  const char *kind;
{
  struct delay_desc *delay;
  int max_slots;
  char str[50];
  struct attr_desc *attr;
  struct attr_value *av, *common_av;
  int i;

  /* Compute the maximum number of delay slots required.  We use the delay
     ordinal times this number plus one, plus the slot number as an index into
     the appropriate predicate to test.  */

  for (delay = delays, max_slots = 0; delay; delay = delay->next)
    if (XVECLEN (delay->def, 1) / 3 > max_slots)
      max_slots = XVECLEN (delay->def, 1) / 3;

  /* Write function prelude.  */

  printf ("int\n");
  printf ("eligible_for_%s (delay_insn, slot, candidate_insn, flags)\n", 
	   kind);
  printf ("     rtx delay_insn;\n");
  printf ("     int slot;\n");
  printf ("     rtx candidate_insn;\n");
  printf ("     int flags ATTRIBUTE_UNUSED;\n");
  printf ("{\n");
  printf ("  rtx insn;\n");
  printf ("\n");
  printf ("  if (slot >= %d)\n", max_slots);
  printf ("    abort ();\n");
  printf ("\n");

  /* If more than one delay type, find out which type the delay insn is.  */

  if (num_delays > 1)
    {
      attr = find_attr ("*delay_type", 0);
      if (! attr) abort ();
      common_av = find_most_used (attr);

      printf ("  insn = delay_insn;\n");
      printf ("  switch (recog_memoized (insn))\n");
      printf ("    {\n");

      sprintf (str, " * %d;\n      break;", max_slots);
      for (av = attr->first_value; av; av = av->next)
	if (av != common_av)
	  write_attr_case (attr, av, 1, "slot +=", str, 4, true_rtx);

      write_attr_case (attr, common_av, 0, "slot +=", str, 4, true_rtx);
      printf ("    }\n\n");

      /* Ensure matched.  Otherwise, shouldn't have been called.  */
      printf ("  if (slot < %d)\n", max_slots);
      printf ("    abort ();\n\n");
    }

  /* If just one type of delay slot, write simple switch.  */
  if (num_delays == 1 && max_slots == 1)
    {
      printf ("  insn = candidate_insn;\n");
      printf ("  switch (recog_memoized (insn))\n");
      printf ("    {\n");

      attr = find_attr ("*delay_1_0", 0);
      if (! attr) abort ();
      common_av = find_most_used (attr);

      for (av = attr->first_value; av; av = av->next)
	if (av != common_av)
	  write_attr_case (attr, av, 1, "return", ";", 4, true_rtx);

      write_attr_case (attr, common_av, 0, "return", ";", 4, true_rtx);
      printf ("    }\n");
    }

  else
    {
      /* Write a nested CASE.  The first indicates which condition we need to
	 test, and the inner CASE tests the condition.  */
      printf ("  insn = candidate_insn;\n");
      printf ("  switch (slot)\n");
      printf ("    {\n");

      for (delay = delays; delay; delay = delay->next)
	for (i = 0; i < XVECLEN (delay->def, 1); i += 3)
	  {
	    printf ("    case %d:\n",
		    (i / 3) + (num_delays == 1 ? 0 : delay->num * max_slots));
	    printf ("      switch (recog_memoized (insn))\n");
	    printf ("\t{\n");

	    sprintf (str, "*%s_%d_%d", kind, delay->num, i / 3);
	    attr = find_attr (str, 0);
	    if (! attr) abort ();
	    common_av = find_most_used (attr);

	    for (av = attr->first_value; av; av = av->next)
	      if (av != common_av)
		write_attr_case (attr, av, 1, "return", ";", 8, true_rtx);

	    write_attr_case (attr, common_av, 0, "return", ";", 8, true_rtx);
	    printf ("      }\n");
	  }

      printf ("    default:\n");
      printf ("      abort ();\n");     
      printf ("    }\n");
    }

  printf ("}\n\n");
}

/* Write routines to compute conflict cost for function units.  Then write a
   table describing the available function units.  */

static void
write_function_unit_info ()
{
  struct function_unit *unit;
  int i;

  /* Write out conflict routines for function units.  Don't bother writing
     one if there is only one issue delay value.  */

  for (unit = units; unit; unit = unit->next)
    {
      if (unit->needs_blockage_function)
	write_complex_function (unit, "blockage", "block");

      /* If the minimum and maximum conflict costs are the same, there
	 is only one value, so we don't need a function.  */
      if (! unit->needs_conflict_function)
	{
	  unit->default_cost = make_numeric_value (unit->issue_delay.max);
	  continue;
	}

      /* The function first computes the case from the candidate insn.  */
      unit->default_cost = make_numeric_value (0);
      write_complex_function (unit, "conflict_cost", "cost");
    }

  /* Now that all functions have been written, write the table describing
     the function units.   The name is included for documentation purposes
     only.  */

  printf ("struct function_unit_desc function_units[] = {\n");

  /* Write out the descriptions in numeric order, but don't force that order
     on the list.  Doing so increases the runtime of genattrtab.c.  */
  for (i = 0; i < num_units; i++)
    {
      for (unit = units; unit; unit = unit->next)
	if (unit->num == i)
	  break;

      printf ("  {\"%s\", %d, %d, %d, %s, %d, %s_unit_ready_cost, ",
	      unit->name, 1 << unit->num, unit->multiplicity,
	      unit->simultaneity, XSTR (unit->default_cost, 0),
	      unit->issue_delay.max, unit->name);

      if (unit->needs_conflict_function)
	printf ("%s_unit_conflict_cost, ", unit->name);
      else
	printf ("0, ");

      printf ("%d, ", unit->max_blockage);

      if (unit->needs_range_function)
	printf ("%s_unit_blockage_range, ", unit->name);
      else
	printf ("0, ");

      if (unit->needs_blockage_function)
	printf ("%s_unit_blockage", unit->name);
      else
	printf ("0");

      printf ("}, \n");
    }

  printf ("};\n\n");
}

static void
write_complex_function (unit, name, connection)
     struct function_unit *unit;
     const char *name, *connection;
{
  struct attr_desc *case_attr, *attr;
  struct attr_value *av, *common_av;
  rtx value;
  char *str;
  int using_case;
  int i;

  printf ("static int %s_unit_%s PROTO ((rtx, rtx));\n", unit->name, name);
  printf ("static int\n");
  printf ("%s_unit_%s (executing_insn, candidate_insn)\n",
	  unit->name, name);
  printf ("     rtx executing_insn;\n");
  printf ("     rtx candidate_insn;\n");
  printf ("{\n");
  printf ("  rtx insn;\n");
  printf ("  int casenum;\n\n");
  printf ("  insn = executing_insn;\n");
  printf ("  switch (recog_memoized (insn))\n");
  printf ("    {\n");

  /* Write the `switch' statement to get the case value.  */
  str = (char *) alloca (strlen (unit->name) + strlen (name) + strlen (connection) + 10);
  sprintf (str, "*%s_cases", unit->name);
  case_attr = find_attr (str, 0);
  if (! case_attr) abort ();
  common_av = find_most_used (case_attr);

  for (av = case_attr->first_value; av; av = av->next)
    if (av != common_av)
      write_attr_case (case_attr, av, 1,
		       "casenum =", ";", 4, unit->condexp);

  write_attr_case (case_attr, common_av, 0,
		   "casenum =", ";", 4, unit->condexp);
  printf ("    }\n\n");

  /* Now write an outer switch statement on each case.  Then write
     the tests on the executing function within each.  */
  printf ("  insn = candidate_insn;\n");
  printf ("  switch (casenum)\n");
  printf ("    {\n");

  for (i = 0; i < unit->num_opclasses; i++)
    {
      /* Ensure using this case.  */
      using_case = 0;
      for (av = case_attr->first_value; av; av = av->next)
	if (av->num_insns
	    && contained_in_p (make_numeric_value (i), av->value))
	  using_case = 1;

      if (! using_case)
	continue;

      printf ("    case %d:\n", i);
      sprintf (str, "*%s_%s_%d", unit->name, connection, i);
      attr = find_attr (str, 0);
      if (! attr) abort ();

      /* If single value, just write it.  */
      value = find_single_value (attr);
      if (value)
	write_attr_set (attr, 6, value, "return", ";\n", true_rtx, -2, -2);
      else
	{
	  common_av = find_most_used (attr);
	  printf ("      switch (recog_memoized (insn))\n");
	  printf ("\t{\n");

	  for (av = attr->first_value; av; av = av->next)
	    if (av != common_av)
	      write_attr_case (attr, av, 1,
			       "return", ";", 8, unit->condexp);

	  write_attr_case (attr, common_av, 0,
			   "return", ";", 8, unit->condexp);
	  printf ("      }\n\n");
	}
    }

  /* This default case should not be needed, but gcc's analysis is not
     good enough to realize that the default case is not needed for the
     second switch statement.  */
  printf ("    default:\n      abort ();\n");
  printf ("    }\n}\n\n");
}

/* This page contains miscellaneous utility routines.  */

/* Given a string, return the number of comma-separated elements in it.
   Return 0 for the null string.  */

static int
n_comma_elts (s)
     char *s;
{
  int n;

  if (*s == '\0')
    return 0;

  for (n = 1; *s; s++)
    if (*s == ',')
      n++;

  return n;
}

/* Given a pointer to a (char *), return a malloc'ed string containing the
   next comma-separated element.  Advance the pointer to after the string
   scanned, or the end-of-string.  Return NULL if at end of string.  */

static char *
next_comma_elt (pstr)
     char **pstr;
{
  char *out_str;
  char *p;

  if (**pstr == '\0')
    return NULL;

  /* Find end of string to compute length.  */
  for (p = *pstr; *p != ',' && *p != '\0'; p++)
    ;

  out_str = attr_string (*pstr, p - *pstr);
  *pstr = p;

  if (**pstr == ',')
    (*pstr)++;

  return out_str;
}

/* Return a `struct attr_desc' pointer for a given named attribute.  If CREATE
   is non-zero, build a new attribute, if one does not exist.  */

static struct attr_desc *
find_attr (name, create)
     const char *name;
     int create;
{
  struct attr_desc *attr;
  int index;

  /* Before we resort to using `strcmp', see if the string address matches
     anywhere.  In most cases, it should have been canonicalized to do so.  */
  if (name == alternative_name)
    return NULL;

  index = name[0] & (MAX_ATTRS_INDEX - 1);
  for (attr = attrs[index]; attr; attr = attr->next)
    if (name == attr->name)
      return attr;

  /* Otherwise, do it the slow way.  */
  for (attr = attrs[index]; attr; attr = attr->next)
    if (name[0] == attr->name[0] && ! strcmp (name, attr->name))
      return attr;

  if (! create)
    return NULL;

  attr = (struct attr_desc *) oballoc (sizeof (struct attr_desc));
  attr->name = attr_string (name, strlen (name));
  attr->first_value = attr->default_val = NULL;
  attr->is_numeric = attr->negative_ok = attr->is_const = attr->is_special = 0;
  attr->next = attrs[index];
  attrs[index] = attr;

  return attr;
}

/* Create internal attribute with the given default value.  */

static void
make_internal_attr (name, value, special)
     const char *name;
     rtx value;
     int special;
{
  struct attr_desc *attr;

  attr = find_attr (name, 1);
  if (attr->default_val)
    abort ();

  attr->is_numeric = 1;
  attr->is_const = 0;
  attr->is_special = (special & 1) != 0;
  attr->negative_ok = (special & 2) != 0;
  attr->unsigned_p = (special & 4) != 0;
  attr->func_units_p = (special & 8) != 0;
  attr->blockage_p = (special & 16) != 0;
  attr->default_val = get_attr_value (value, attr, -2);
}

/* Find the most used value of an attribute.  */

static struct attr_value *
find_most_used (attr)
     struct attr_desc *attr;
{
  struct attr_value *av;
  struct attr_value *most_used;
  int nuses;

  most_used = NULL;
  nuses = -1;

  for (av = attr->first_value; av; av = av->next)
    if (av->num_insns > nuses)
      nuses = av->num_insns, most_used = av;

  return most_used;
}

/* If an attribute only has a single value used, return it.  Otherwise
   return NULL.  */

static rtx
find_single_value (attr)
     struct attr_desc *attr;
{
  struct attr_value *av;
  rtx unique_value;

  unique_value = NULL;
  for (av = attr->first_value; av; av = av->next)
    if (av->num_insns)
      {
	if (unique_value)
	  return NULL;
	else
	  unique_value = av->value;
      }

  return unique_value;
}

/* Return (attr_value "n") */

static rtx
make_numeric_value (n)
     int n;
{
  static rtx int_values[20];
  rtx exp;
  char *p;

  if (n < 0)
    abort ();

  if (n < 20 && int_values[n])
    return int_values[n];

  p = attr_printf (MAX_DIGITS, "%d", n);
  exp = attr_rtx (CONST_STRING, p);

  if (n < 20)
    int_values[n] = exp;

  return exp;
}

static void
extend_range (range, min, max)
     struct range *range;
     int min;
     int max;
{
  if (range->min > min) range->min = min;
  if (range->max < max) range->max = max;
}

/* CYGNUS LOCAL -- vmakarov/dfa */
#if 0
/* END CYGNUS LOCAL -- vmakarov/dfa */
PTR
xrealloc (old, size)
  PTR old;
  size_t size;
{
  register PTR ptr;
  if (old)
    ptr = (PTR) realloc (old, size);
  else
    ptr = (PTR) malloc (size);
  if (!ptr)
    fatal ("virtual memory exhausted");
  return ptr;
}

PTR
xmalloc (size)
  size_t size;
{
  register PTR val = (PTR) malloc (size);

  if (val == 0)
    fatal ("virtual memory exhausted");
  return val;
}

/* CYGNUS LOCAL -- vmakarov/dfa */
#endif
/* END CYGNUS LOCAL -- vmakarov/dfa */

static rtx
copy_rtx_unchanging (orig)
     register rtx orig;
{
#if 0
  register rtx copy;
  register RTX_CODE code;
#endif

  if (RTX_UNCHANGING_P (orig) || MEM_IN_STRUCT_P (orig))
    return orig;

  MEM_IN_STRUCT_P (orig) = 1;
  return orig;

#if 0
  code = GET_CODE (orig);
  switch (code)
    {
    case CONST_INT:
    case CONST_DOUBLE:
    case SYMBOL_REF:
    case CODE_LABEL:
      return orig;
      
    default:
      break;
    }

  copy = rtx_alloc (code);
  PUT_MODE (copy, GET_MODE (orig));
  RTX_UNCHANGING_P (copy) = 1;
  
  bcopy ((char *) &XEXP (orig, 0), (char *) &XEXP (copy, 0),
	 GET_RTX_LENGTH (GET_CODE (copy)) * sizeof (rtx));
  return copy;
#endif
}

/* Determine if an insn has a constant number of delay slots, i.e., the
   number of delay slots is not a function of the length of the insn.  */

static void
write_const_num_delay_slots ()
{
  struct attr_desc *attr = find_attr ("*num_delay_slots", 0);
  struct attr_value *av;
  struct insn_ent *ie;

  if (attr)
    {
      printf ("int\nconst_num_delay_slots (insn)\n");
      printf ("     rtx insn;\n");
      printf ("{\n");
      printf ("  switch (recog_memoized (insn))\n");
      printf ("    {\n");

      for (av = attr->first_value; av; av = av->next)
	{
	  length_used = 0;
	  walk_attr_value (av->value);
	  if (length_used)
	    {
	      for (ie = av->first_insn; ie; ie = ie->next)
	      if (ie->insn_code != -1)
		printf ("    case %d:\n", ie->insn_code);
	      printf ("      return 0;\n");
	    }
	}

      printf ("    default:\n");
      printf ("      return 1;\n");
      printf ("    }\n}\n\n");
    }
}


extern int main PROTO ((int, char **));

int
main (argc, argv)
     int argc;
     char **argv;
{
  rtx desc;
  FILE *infile;
  register int c;
  struct attr_desc *attr;
  struct insn_def *id;
  rtx tem;
  int i;

  progname = "genattrtab";

#if defined (RLIMIT_STACK) && defined (HAVE_GETRLIMIT) && defined (HAVE_SETRLIMIT)
  /* Get rid of any avoidable limit on stack size.  */
  {
    struct rlimit rlim;

    /* Set the stack limit huge so that alloca does not fail.  */
    getrlimit (RLIMIT_STACK, &rlim);
    rlim.rlim_cur = rlim.rlim_max;
    setrlimit (RLIMIT_STACK, &rlim);
  }
#endif

  progname = "genattrtab";
  obstack_init (rtl_obstack);
  obstack_init (hash_obstack);
  obstack_init (temp_obstack);

  if (argc <= 1)
    fatal ("No input file name.");

  infile = fopen (argv[1], "r");
  if (infile == 0)
    {
      perror (argv[1]);
      return (FATAL_EXIT_CODE);
    }
  read_rtx_filename = argv[1];

  /* Set up true and false rtx's */
  true_rtx = rtx_alloc (CONST_INT);
  XWINT (true_rtx, 0) = 1;
  false_rtx = rtx_alloc (CONST_INT);
  XWINT (false_rtx, 0) = 0;
  RTX_UNCHANGING_P (true_rtx) = RTX_UNCHANGING_P (false_rtx) = 1;
  RTX_INTEGRATED_P (true_rtx) = RTX_INTEGRATED_P (false_rtx) = 1;

  alternative_name = attr_string ("alternative", strlen ("alternative"));

  printf ("/* Generated automatically by the program `genattrtab'\n\
from the machine description file `md'.  */\n\n");

  /* Read the machine description.  */

/* CYGNUS LOCAL -- vmakarov/dfa */
  initiate_automaton_gen (argc, argv);
/* END CYGNUS LOCAL -- vmakarov/dfa */

  while (1)
    {
      c = read_skip_spaces (infile);
      if (c == EOF)
	break;
      ungetc (c, infile);

      desc = read_rtx (infile);
      if (GET_CODE (desc) == DEFINE_INSN
	  || GET_CODE (desc) == DEFINE_PEEPHOLE
	  || GET_CODE (desc) == DEFINE_ASM_ATTRIBUTES)
	gen_insn (desc);

      else if (GET_CODE (desc) == DEFINE_EXPAND)
	insn_code_number++, insn_index_number++;

      else if (GET_CODE (desc) == DEFINE_SPLIT)
	insn_code_number++, insn_index_number++;

      else if (GET_CODE (desc) == DEFINE_PEEPHOLE2)
	insn_code_number++, insn_index_number++;

      else if (GET_CODE (desc) == DEFINE_ATTR)
	{
	  gen_attr (desc);
	  insn_index_number++;
	}

      else if (GET_CODE (desc) == DEFINE_DELAY)
	{
	  gen_delay (desc);
	  insn_index_number++;
	}

      else if (GET_CODE (desc) == DEFINE_FUNCTION_UNIT)
	{
	  gen_unit (desc);
	  insn_index_number++;
	}
/* CYGNUS LOCAL -- vmakarov/dfa */
      else if  (GET_CODE (desc) == DEFINE_CPU_UNIT)
	{
	  gen_cpu_unit (desc);
	  insn_index_number++;
	}

      else if  (GET_CODE (desc) == DEFINE_BYPASS)
	{
	  gen_bypass (desc);
	  insn_index_number++;
	}

      else if  (GET_CODE (desc) == EXCLUSION_SET)
	{
	  gen_excl_set (desc);
	  insn_index_number++;
	}

      else if  (GET_CODE (desc) == REQUIRE_SET)
	{
	  gen_req_set (desc);
	  insn_index_number++;
	}

      else if  (GET_CODE (desc) == DEFINE_AUTOMATON)
	{
	  gen_automaton (desc);
	  insn_index_number++;
	}

      else if  (GET_CODE (desc) == DEFINE_RESERVATION)
	{
	  gen_reserv (desc);
	  insn_index_number++;
	}

      else if  (GET_CODE (desc) == DEFINE_INSN_RESERVATION)
	{
	  gen_insn_reserv (desc);
	  insn_index_number++;
	}
/* END CYGNUS LOCAL -- vmakarov/dfa */
    }

  /* If we didn't have a DEFINE_ASM_ATTRIBUTES, make a null one.  */
  if (! got_define_asm_attributes)
    {
      tem = rtx_alloc (DEFINE_ASM_ATTRIBUTES);
      XVEC (tem, 0) = rtvec_alloc (0);
      gen_insn (tem);
    }

  /* Expand DEFINE_DELAY information into new attribute.  */
  if (num_delays)
    expand_delays ();

  /* Expand DEFINE_FUNCTION_UNIT information into new attributes.  */
  if (num_units)
    expand_units ();

/* CYGNUS LOCAL -- vmakarov/dfa */
  /* Build DFA, output some functions and expand DFA information into
     new attributes.  */
  if (num_dfa_decls)
    expand_automata ();
/* END CYGNUS LOCAL -- vmakarov/dfa */

  printf ("#include \"config.h\"\n");
  printf ("#include \"system.h\"\n");
  printf ("#include \"rtl.h\"\n");
  printf ("#include \"tm_p.h\"\n");
  printf ("#include \"insn-config.h\"\n");
  printf ("#include \"recog.h\"\n");
  printf ("#include \"regs.h\"\n");
  printf ("#include \"real.h\"\n");
  printf ("#include \"output.h\"\n");
  printf ("#include \"insn-attr.h\"\n");
  printf ("#include \"toplev.h\"\n");
  printf ("\n");  
  printf ("#define operands recog_data.operand\n\n");

  /* Make `insn_alternatives'.  */
  insn_alternatives = (int *) oballoc (insn_code_number * sizeof (int));
  for (id = defs; id; id = id->next)
    if (id->insn_code >= 0)
      insn_alternatives[id->insn_code] = (1 << id->num_alternatives) - 1;

  /* Make `insn_n_alternatives'.  */
  insn_n_alternatives = (int *) oballoc (insn_code_number * sizeof (int));
  for (id = defs; id; id = id->next)
    if (id->insn_code >= 0)
      insn_n_alternatives[id->insn_code] = id->num_alternatives;

  /* Prepare to write out attribute subroutines by checking everything stored
     away and building the attribute cases.  */

  check_defs ();
  for (i = 0; i < MAX_ATTRS_INDEX; i++)
    for (attr = attrs[i]; attr; attr = attr->next)
      {
	attr->default_val->value
	  = check_attr_value (attr->default_val->value, attr);
	fill_attr (attr);
      }

  /* Construct extra attributes for `length'.  */
  make_length_attrs ();

  /* Perform any possible optimizations to speed up compilation.  */
  optimize_attrs ();

  /* Now write out all the `gen_attr_...' routines.  Do these before the
     special routines (specifically before write_function_unit_info), so
     that they get defined before they are used.  */

  for (i = 0; i < MAX_ATTRS_INDEX; i++)
    for (attr = attrs[i]; attr; attr = attr->next)
      {
	if (! attr->is_special && ! attr->is_const)
	  write_attr_get (attr);
      }

  /* Write out delay eligibility information, if DEFINE_DELAY present.
     (The function to compute the number of delay slots will be written
     below.)  */
  if (num_delays)
    {
      write_eligible_delay ("delay");
      if (have_annul_true)
	write_eligible_delay ("annul_true");
      if (have_annul_false)
	write_eligible_delay ("annul_false");
    }

  /* Write out information about function units.  */
  if (num_units)
    {
/* CYGNUS LOCAL -- vmakarov/dfa */
      printf ("#if !AUTOMATON_PIPELINE_INTERFACE\n");
/* END CYGNUS LOCAL -- vmakarov/dfa */
      write_function_unit_info ();
/* CYGNUS LOCAL -- vmakarov/dfa */
      printf ("#endif /* #if !AUTOMATON_PIPELINE_INTERFACE */\n\n");
/* END CYGNUS LOCAL -- vmakarov/dfa */
    }

/* CYGNUS LOCAL -- vmakarov/dfa */
  if (num_dfa_decls)
    {
      /* Output code for pipeline hazards recognition based on
	 DFA (deterministic finite state automata. */
      printf ("#if AUTOMATON_PIPELINE_INTERFACE\n");
      write_automata ();
      printf ("#endif /* #if AUTOMATON_PIPELINE_INTERFACE */\n\n");
    }
/* END CYGNUS LOCAL -- vmakarov/dfa */

  /* Write out constant delay slot info */
  write_const_num_delay_slots ();

  write_length_unit_log ();

  fflush (stdout);
  return (ferror (stdout) != 0 ? FATAL_EXIT_CODE : SUCCESS_EXIT_CODE);
}

/* Define this so we can link with print-rtl.o to get debug_rtx function.  */
const char *
get_insn_name (code)
     int code ATTRIBUTE_UNUSED;
{
  return NULL;
}
/* CYGNUS LOCAL -- vmakarov/dfa */



/*====All the folowing code is pipeline hazard description translator.=======*/

#ifdef HAVE_ASSERT_H
#include <assert.h>
#else
#ifndef assert
#define assert(code) do { if ((code) == 0) abort ();} while (0)
#endif
#endif

#include <ctype.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "hashtab.h"
#include "varray.h"


#ifdef HAVE_LIMITS_H
#include <limits.h>
#else
#ifndef CHAR_BIT
#define CHAR_BIT 8
#endif
#endif

/* Positions in machine description file.  Now they are not used. */
typedef int pos_t;
typedef unsigned HOST_WIDE_INT set_el_t;
typedef set_el_t *reserv_sets_t;

/* The following structure repersents variable length array (vla) of
   pointers and HOST WIDE INTs.  See commentaries for macros working
   with vla. */
typedef struct {
  size_t length;      /* current size of vla. */
  varray_type varray; /* container for vla. */
} vla_ptr_t;

typedef vla_ptr_t vla_hwint_t;

/* The following structure describes a ticker. */
struct ticker
{
  /* The following member value is time of the ticker creation with
     taking into account time when the ticker is off.  Active time of
     the ticker is current time minus the value. */
  int modified_creation_time;
  /* The following member value is time (incremented by one) when the
     ticker was off.  Zero value means that now the ticker is on. */
  int incremented_off_time;
};

/* The ticker is represented by the following type. */
typedef struct ticker ticker_t;

/* The following type describes elements of output vectors. */
typedef HOST_WIDE_INT vect_el_t;

/* Forward declaration of structures of internal rpepresenation of
   pipeline description based on NDFA. */

struct unit_decl;
struct bypass_decl;
struct result_decl;
struct automaton_decl;
struct excl_decl;
struct req_decl;
struct reserv_decl;
struct insn_reserv_decl;
struct decl;
struct unit_regexp;
struct result_regexp;
struct reserv_regexp;
struct nothing_regexp;
struct sequence_regexp;
struct repeat_regexp;
struct allof_regexp;
struct oneof_regexp;
struct regexp;
struct description;
struct unit_set_el;
struct state;
struct alt_state;
struct arc;
struct ainsn;
struct automaton;
struct state_ainsn_table;

/* Prototypes of functions gen_cpu_unit, gen_bypass, gen_excl_set,
   gen_req_set, gen_automaton, gen_reserv, gen_insn_reserv,
   initiate_automaton_gen, expand_automata, write_automata are
   described on the file top because the functions are called from
   function `main'. */

static void *create_node                  PROTO((size_t));
static void *copy_node                    PROTO((void *, size_t));
static char *check_name                   PROTO((char *, pos_t));
static char *next_sep_el                  PROTO((char **, int, int));
static int n_sep_els                      PROTO((char *, int, int));
static char **get_str_vect                PROTO((char *, int *, int, int));
static struct regexp *gen_regexp_el       PROTO((char *));
static struct regexp *gen_regexp_repeat   PROTO((char *));
static struct regexp *gen_regexp_allof    PROTO((char *));
static struct regexp *gen_regexp_oneof    PROTO((char *));
static struct regexp *gen_regexp_sequence PROTO((char *));
static struct regexp *gen_regexp          PROTO((char *));

static unsigned string_hash               PROTO((const char *));
static unsigned automaton_decl_hash       PROTO((const void *));
static int automaton_decl_eq_p            PROTO((const void *, const void *));
static struct decl *insert_automaton_decl PROTO((struct decl *));
static struct decl *find_automaton_decl   PROTO((char *));
static void initiate_automaton_decl_table PROTO((void));
static void finish_automaton_decl_table   PROTO((void));

static unsigned insn_decl_hash            PROTO((const void *));
static int insn_decl_eq_p                 PROTO((const void *, const void *));
static struct decl *insert_insn_decl      PROTO((struct decl *));
static struct decl *find_insn_decl        PROTO((char *));
static void initiate_insn_decl_table      PROTO((void));
static void finish_insn_decl_table        PROTO((void));

static unsigned decl_hash                 PROTO((const void *));
static int decl_eq_p                      PROTO((const void *, const void *));
static struct decl *insert_decl           PROTO((struct decl *));
static struct decl *find_decl             PROTO((char *));
static void initiate_decl_table           PROTO((void));
static void finish_decl_table             PROTO((void));

static struct unit_set_el *process_excls PROTO((char **, int, pos_t));
static void add_excls                    PROTO((struct unit_set_el *,
                                                struct unit_set_el *, pos_t));
static void process_decls                PROTO((void));
static struct bypass_decl *find_bypass   PROTO((struct bypass_decl *,
						struct insn_reserv_decl *));
static void check_automaton_usage        PROTO((void));
static struct regexp *process_regexp     PROTO((struct regexp *));
static void process_regexp_decls         PROTO((void));
static void check_usage                  PROTO((void));
static int loop_in_regexp                PROTO((struct regexp *,
                                                struct decl *));
static void check_loops_in_regexps       PROTO((void));
static int process_regexp_cycles         PROTO((struct regexp *, int));
static void evaluate_max_reserv_cycles   PROTO((void));
static void check_all_description        PROTO((void));

static ticker_t create_ticker               PROTO((void));
static void ticker_off                      PROTO((ticker_t *));
static void ticker_on                       PROTO((ticker_t *));
static int active_time                      PROTO((ticker_t));
static void print_active_time               PROTO((FILE *, ticker_t));

static void add_advance_cycle_insn_decl       PROTO((void));

static struct alt_state *get_free_alt_state PROTO((void));
static void free_alt_state                PROTO((struct alt_state *));
static void free_alt_states               PROTO((struct alt_state *));
static int alt_state_cmp                  PROTO((const void *alt_state_ptr_1,
                                                 const void *alt_state_ptr_2));
static struct alt_state *uniq_sort_alt_states PROTO((struct alt_state *));
static int alt_states_eq                  PROTO((struct alt_state *, struct alt_state *));
static void initiate_alt_states           PROTO((void));
static void finish_alt_states             PROTO((void));

static reserv_sets_t alloc_empty_reserv_sets PROTO((void));
static unsigned reserv_sets_hash_value     PROTO((reserv_sets_t));
static int reserv_sets_cmp                 PROTO((reserv_sets_t,
                                                  reserv_sets_t));
static int reserv_sets_eq                  PROTO((reserv_sets_t,
                                                  reserv_sets_t));
static void set_unit_reserv                PROTO((reserv_sets_t, int, int));
static int it_is_empty_reserv_sets         PROTO((reserv_sets_t))
                                                ATTRIBUTE_UNUSED;
static int reserv_sets_are_intersected     PROTO((reserv_sets_t,
                                                  reserv_sets_t));
static void reserv_sets_shift              PROTO((reserv_sets_t, reserv_sets_t));
static void reserv_sets_or                 PROTO((reserv_sets_t, reserv_sets_t,
                                                  reserv_sets_t));
static void reserv_sets_and                PROTO((reserv_sets_t, reserv_sets_t,
                                                  reserv_sets_t))
                                                ATTRIBUTE_UNUSED;
static void output_cycle_reservs           PROTO((FILE *, reserv_sets_t,
                                                  int, int));
static void output_reserv_sets             PROTO((FILE *, reserv_sets_t));
static struct state *get_free_state        PROTO((int, struct automaton *));
static void free_state                     PROTO((struct state *));
static unsigned state_hash                 PROTO((const void *));
static int state_eq_p                      PROTO((const void *, const void *));
static struct state *insert_state          PROTO((struct state *));
static void set_state_reserv               PROTO((struct state *, int, int));
static int intersected_state_reservs_p     PROTO((struct state *,
                                                  struct state *));
static struct state *states_union          PROTO((struct state *,
                                                  struct state *));
static struct state *state_shift           PROTO((struct state *));
static void initiate_states                PROTO((void));
static void finish_states                  PROTO((void));

static void free_arc                 PROTO((struct arc *));
static void remove_arc               PROTO((struct state *, struct arc *));
static struct arc *find_arc          PROTO((struct state *, struct state *,
                                            struct ainsn *));
static struct arc *add_arc           PROTO((struct state *, struct state *,
                                            struct ainsn *, int));
static struct arc *first_out_arc     PROTO((struct state *));
static struct arc *next_out_arc      PROTO((struct arc *));
static void initiate_arcs            PROTO((void));
static void finish_arcs              PROTO((void));

static void initiate_excl_sets             PROTO((void));
static reserv_sets_t get_excl_set          PROTO((reserv_sets_t));

static void initiate_req_sets        PROTO((void));
static reserv_sets_t get_req_set     PROTO((reserv_sets_t));

static struct regexp *copy_insn_regexp     PROTO((struct regexp *));
static struct regexp *transform_1          PROTO((struct regexp *));
static struct regexp *transform_2          PROTO((struct regexp *));
static struct regexp *transform_3          PROTO((struct regexp *));
static struct regexp *regexp_transform_func
              PROTO((struct regexp *, struct regexp *(*) (struct regexp *)));
static struct regexp *transform_regexp     PROTO((struct regexp *));
static void transform_insn_regexps         PROTO((void));

static int process_seq_for_forming_states   PROTO((struct regexp *,
                                                   struct automaton *, int));
static void finish_forming_alt_state        PROTO((struct alt_state *,
                                                   struct automaton *));
static void process_alts_for_forming_states PROTO((struct regexp *,
                                                   struct automaton *, int));
static void create_alt_states               PROTO((struct automaton *));

static void form_ainsn_with_same_reservs   PROTO((struct automaton *));

static void make_automaton               PROTO((struct automaton *));
static void form_arcs_marked_by_insn     PROTO((struct state *));
static void create_composed_state        PROTO((struct state *, struct arc *,
                                                vla_ptr_t *));
static void NDFA_to_DFA                  PROTO((struct automaton *));
static void pass_state_graph             PROTO((struct state *,
                                                void (*) (struct state *)));
static void pass_states                  PROTO((struct automaton *,
                                                void (*) (struct state *)));
static void initiate_pass_states         PROTO((void));
static void add_achieved_state           PROTO((struct state *));
static int set_out_arc_insns_equiv_num   PROTO((struct state *, int));
static void clear_arc_insns_equiv_num    PROTO((struct state *));
static void copy_equiv_class             PROTO((vla_ptr_t *to,
                                                const vla_ptr_t *from));
static int state_is_differed             PROTO((struct state *, int, int));
static struct state *init_equiv_class    PROTO((struct state **states, int));
static void evaluate_equiv_classes       PROTO((struct automaton *,
                                                vla_ptr_t *));
static void merge_states                 PROTO((struct automaton *,
                                                vla_ptr_t *));
static void set_new_cycle_flags          PROTO((struct state *));
static void minimize_DFA                 PROTO((struct automaton *));
static void incr_states_and_arcs_nums    PROTO((struct state *));
static void count_states_and_arcs        PROTO((struct automaton *, int *,
                                                int *));
static void build_automaton              PROTO((struct automaton *));

static void set_order_state_num              PROTO((struct state *));
static void enumerate_states                 PROTO((struct automaton *));

static struct ainsn *insert_ainsn_into_equiv_class PROTO((struct ainsn *,
                                                          struct ainsn *));
static void delete_ainsn_from_equiv_class          PROTO((struct ainsn *));
static void process_insn_equiv_class               PROTO((struct ainsn *,
                                                          struct arc **));
static void process_state_for_insn_equiv_partition PROTO((struct state *));
static void set_insn_equiv_classes                 PROTO((struct automaton *));

static double estimate_one_automaton_bound     PROTO((void));
static int compare_max_occ_cycle_nums          PROTO((const void *,
                                                      const void *));
static void units_to_automata_heuristic_distr  PROTO((void));
static struct ainsn *create_ainsns             PROTO((void));
static void units_to_automata_distr            PROTO((void));
static void create_automata                    PROTO((void));

static void form_regexp                      PROTO((struct regexp *));
static const char *regexp_representation     PROTO((struct regexp *));
static void finish_regexp_representation     PROTO((void));

static void output_range_type              PROTO((FILE *, long int, long int));
static void output_vect                    PROTO((vect_el_t *, int));
static void output_chip_member_name        PROTO((FILE *, struct automaton *));
static void output_temp_chip_member_name   PROTO((FILE *, struct automaton *));
static void output_translate_vect_name     PROTO((FILE *, struct automaton *));
static void output_trans_full_vect_name    PROTO((FILE *, struct automaton *));
static void output_trans_comb_vect_name    PROTO((FILE *, struct automaton *));
static void output_trans_check_vect_name   PROTO((FILE *, struct automaton *));
static void output_trans_base_vect_name    PROTO((FILE *, struct automaton *));
static void output_state_alts_full_vect_name    PROTO((FILE *,
                                                       struct automaton *));
static void output_state_alts_comb_vect_name    PROTO((FILE *,
                                                       struct automaton *));
static void output_state_alts_check_vect_name   PROTO((FILE *,
                                                       struct automaton *));
static void output_state_alts_base_vect_name    PROTO((FILE *,
                                                       struct automaton *));
static void output_min_issue_delay_full_vect_name  PROTO((FILE *,
                                                          struct automaton *));
static void output_min_issue_delay_comb_vect_name  PROTO((FILE *,
                                                          struct automaton *));
static void output_min_issue_delay_check_vect_name PROTO((FILE *,
                                                          struct automaton *));
static void output_min_issue_delay_base_vect_name  PROTO((FILE *,
                                                          struct automaton *));
static void output_dead_lock_vect_name     PROTO((FILE *, struct automaton *));
static void output_state_member_type       PROTO((FILE *, struct automaton *));
static void output_chip_definitions        PROTO((void));
static void output_translate_vect          PROTO((struct automaton *));
static int comb_vect_p                     PROTO((struct state_ainsn_table *));
static struct state_ainsn_table *create_state_ainsn_table
                                           PROTO((struct automaton *));
static void output_state_ainsn_table
   PROTO((struct state_ainsn_table *, char *,
          void (*) (FILE *, struct automaton *),
          void (*) (FILE *, struct automaton *),
          void (*) (FILE *, struct automaton *),
          void (*) (FILE *, struct automaton *)));
static void add_vect                       PROTO((struct state_ainsn_table *,
                                                  int, vect_el_t *, int));
static int out_state_arcs_num              PROTO((struct state *));
static int compare_transition_els_num      PROTO((const void *, const void *));
static void add_vect_el 	           PROTO((vla_hwint_t *,
                                                  struct ainsn *, int));
static void add_states_vect_el             PROTO((struct state *));
static void output_trans_table             PROTO((struct automaton *));
static void output_state_alts_table        PROTO((struct automaton *));
static void min_issue_delay_pass_states    PROTO((struct state *,
                                                  struct ainsn *, int));
static int min_issue_delay                 PROTO((struct state *,
                                                  struct ainsn *));
static void initiate_min_issue_delay_pass_states PROTO((void));
static void output_min_issue_delay_table   PROTO((struct automaton *));
static void output_dead_lock_vect          PROTO((struct automaton *));
static void output_tables                  PROTO((void));
static void output_max_insn_queue_index_def PROTO((void));
static void output_internal_min_issue_delay_func PROTO((void));
static void output_internal_trans_func     PROTO((void));
static void output_internal_insn_code_evaluation PROTO((const char *,
							const char *));
static void output_trans_func              PROTO((void));
static void output_internal_state_alts_func PROTO((void));
static void output_state_alts_func         PROTO((void));
static void output_min_issue_delay_func    PROTO((void));
static void output_internal_dead_lock_func PROTO((void));
static void output_dead_lock_func          PROTO((void));
static void output_internal_reset_func     PROTO((void));
static void output_reset_func              PROTO((void));
static void output_min_insn_conflict_delay_func PROTO((void));
static void output_internal_insn_latency_func PROTO((void));
static void output_insn_latency_func       PROTO((void));
static void output_print_reservation_func  PROTO((void));
static void output_regexp                  PROTO((struct regexp *));
static void output_description             PROTO((void));
static void output_automaton_name          PROTO((FILE *, struct automaton *));
static void output_automaton_units         PROTO((struct automaton *));
static void add_state_reservs              PROTO((struct state *));
static void output_state_arcs              PROTO((struct state *));
static int state_reservs_cmp               PROTO((const void *, const void *));
static void remove_state_duplicate_reservs PROTO((void));
static void output_state                   PROTO((struct state *));
static void output_automaton_descriptions  PROTO((void));
static void output_statistics              PROTO((FILE *));
static void output_time_statistics         PROTO((FILE *));
static void generate                       PROTO((void));

static void make_insn_alts_attr               PROTO((void));
static void make_dfa_insn_code_attr           PROTO((void));
static void make_default_insn_latency_attr    PROTO((void));
static void make_bypass_attr                  PROTO((void));
static const char *file_name_suffix           PROTO((const char *));
static const char *base_file_name             PROTO((const char *));


/* Undefined position. */
static pos_t no_pos = 0;



/* Macros for work with internal representation (IR) storage
   manager. */

/* Start work with IR storage manager. */
#define IR_START_ALLOC()    obstack_init (&irp)

/* Finish work with the storage manager. */
#define IR_STOP_ALLOC()     obstack_free (&irp, NULL)

/* Allocate storage for IR of given size. */
#define IR_ALLOC(ptr, size, ptr_type)\
  do {\
    obstack_make_room (&irp, size); ptr = (ptr_type) obstack_finish (&irp);\
  } while (0);

/* Free storage of IR of given size. */
#define IR_FREE(ptr, size)

/* Start new IR object. */
#define IR_TOP_FINISH()  ((void) obstack_finish (&irp))

/* Nullify current IR object. */
#define IR_TOP_NULLIFY() obstack_blank_fast (&irp, -obstack_object_size (&irp))

/* Shorten current IR object on given number bytes. */
#define IR_TOP_SHORTEN(length) obstack_blank_fast (&irp, -(length))

/* Return start address of current IR object. */
#define IR_TOP_BEGIN()  obstack_base (&irp)

/* Return length of current IR object in bytes. */
#define IR_TOP_LENGTH()  obstack_object_size (&irp)

/* Expand current IR object. */
#define IR_TOP_EXPAND(length)  obstack_blank (&irp, length)

/* Add byte to the end of current IR object. */
#define IR_TOP_ADD_BYTE(b)  obstack_1grow (&irp, b)

/* Add string to the end of current IR object. */
#define IR_TOP_ADD_STRING(str)  obstack_grow (&irp, str, strlen (str) + 1)

/* Add memory of given length to the end of current IR object. */
#define IR_TOP_ADD_MEMORY(mem, length)  obstack_grow (&irp, mem, length)

/* All IR is stored in the following obstack. */
static struct obstack irp;



/* This page contains code for work with variable length array (vla)
   of pointers.  We could be use only varray.  But we add new lay
   because we add elements very frequently and this could stress OS
   allocator when varray is used only. */

/* Start work with vla. */
#define VLA_PTR_CREATE(vla, allocated_length, name)                   \
  do {                                                                \
    vla_ptr_t *vla_ptr = &(vla);                                      \
                                                                      \
    VARRAY_GENERIC_PTR_INIT (vla_ptr->varray, allocated_length, name);\
    vla_ptr->length = 0;                                              \
  } while (0)

/* Finish work with the vla. */
#define VLA_PTR_DELETE(vla) VARRAY_FREE ((vla).varray)

/* Return start address of the vla. */
#define VLA_PTR_BEGIN(vla) ((void *) &VARRAY_GENERIC_PTR ((vla).varray, 0))

/* Address of the last element of the vla.  Do not use side effects in
   the macro argument. */
#define VLA_PTR_LAST(vla) (&VARRAY_GENERIC_PTR ((vla).varray,         \
                                                (vla).length - 1))
/* Nullify the vla. */
#define VLA_PTR_NULLIFY(vla)  ((vla).length = 0)

/* Shorten the vla on given number bytes. */
#define VLA_PTR_SHORTEN(vla, n)  ((vla).length -= (n))

/* Expand the vla on N elements.  The values of new elements are
   undefined. */
#define VLA_PTR_EXPAND(vla, n)                                        \
  do {                                                                \
    vla_ptr_t *expand_vla_ptr = &(vla);                               \
    size_t new_length = (n) + expand_vla_ptr->length;                 \
                                                                      \
    if (VARRAY_SIZE (expand_vla_ptr->varray) < new_length)            \
      VARRAY_GROW (expand_vla_ptr->varray,                            \
                   (new_length - expand_vla_ptr->length < 128         \
                    ? expand_vla_ptr->length + 128 : new_length));    \
    expand_vla_ptr->length = new_length;                              \
  } while (0)

/* Add element to the end of the vla. */
#define VLA_PTR_ADD(vla, ptr)                                         \
  do {                                                                \
    vla_ptr_t *vla_ptr = &(vla);                                      \
                                                                      \
    VLA_PTR_EXPAND (*vla_ptr, 1);                                     \
    VARRAY_GENERIC_PTR (vla_ptr->varray, vla_ptr->length - 1) = (ptr);\
  } while (0)

/* Length of the vla in elements. */
#define VLA_PTR_LENGTH(vla) ((vla).length)

/* N-th element of the vla. */
#define VLA_PTR(vla, n) VARRAY_GENERIC_PTR ((vla).varray, n)


/* The following macros are analogous to the previous ones but for
   VLAs of HOST WIDE INTs. */

#define VLA_HWINT_CREATE(vla, allocated_length, name)                 \
  do {                                                                \
    vla_hwint_t *vla_ptr = &(vla);                                    \
                                                                      \
    VARRAY_WIDE_INT_INIT (vla_ptr->varray, allocated_length, name);   \
    vla_ptr->length = 0;                                              \
  } while (0)

#define VLA_HWINT_DELETE(vla) VARRAY_FREE ((vla).varray)

#define VLA_HWINT_BEGIN(vla) (&VARRAY_WIDE_INT ((vla).varray, 0))

/* Do not use side effects in the macro argument. */
#define VLA_HWINT_LAST(vla) (&VARRAY_WIDE_INT ((vla).varray,          \
                                              (vla).length - 1))

#define VLA_HWINT_NULLIFY(vla)  ((vla).length = 0)

#define VLA_HWINT_SHORTEN(vla, n)  ((vla).length -= (n))

#define VLA_HWINT_EXPAND(vla, n)                                      \
  do {                                                                \
    vla_hwint_t *expand_vla_ptr = &(vla);                             \
    size_t new_length = (n) + expand_vla_ptr->length;                 \
                                                                      \
    if (VARRAY_SIZE (expand_vla_ptr->varray) < new_length)            \
      VARRAY_GROW (expand_vla_ptr->varray,                            \
                   (new_length - expand_vla_ptr->length < 128         \
                    ? expand_vla_ptr->length + 128 : new_length));    \
    expand_vla_ptr->length = new_length;                              \
  } while (0)

#define VLA_HWINT_ADD(vla, ptr)                                       \
  do {                                                                \
    vla_hwint_t *vla_ptr = &(vla);                                    \
                                                                      \
    VLA_HWINT_EXPAND (*vla_ptr, 1);                                   \
    VARRAY_WIDE_INT (vla_ptr->varray, vla_ptr->length - 1) = (ptr);   \
  } while (0)

#define VLA_HWINT_LENGTH(vla) ((vla).length)

#define VLA_HWINT(vla, n) VARRAY_WIDE_INT ((vla).varray, n)



/* Standard designators for true and false values. */
#define FALSE 0
#define TRUE  1

/* The following flags are set up by function
   `initiate_automaton_gen'. */

/* Make automata with nondetermenistic reservation by insns (`-ndfa'). */
static int ndfa_flag;

/* Do not make minimization of DFA (`-no-minimization'). */
static int no_minimization_flag;

/* Value of this variable is number of automata being generated.  The
   actual number of automata may be less this value if there is not
   sufficient number of units.  This value is defined by argument of
   option `-split' or by constructions automaton if the value is zero
   (it is default value of the argument). */
static int split_argument;

/* Flag of output time statistics (`-time'). */
static int time_flag;

/* Flag of creation of description file which contains description of
   result automaton and statistics information (`-v'). */
static int v_flag;


/* Output file for pipeline hazard recognizer (PHR) being generated.
   The value is NULL if the file is not defined. */
static FILE *output_file;

/* Description file of PHR.  The value is NULL if the file is not
   created. */
static FILE *output_description_file;

/* PHR description file name. */
static char *output_description_file_name;

/* Value of the following variable is node representing description
   being processed.  This is start point of IR. */
static struct description *description;



/* This page contains description of IR structure (nodes). */

enum decl_mode
{
  dm_unit,
  dm_bypass,
  dm_automaton,
  dm_excl,
  dm_req,
  dm_reserv,
  dm_insn_reserv
};

/* This describes define_cpu_unit (see file rtl.def). */
struct unit_decl
{
  char *name;
  /* NULL if the automaton name is absent. */
  char *automaton_name;

  /* The following fields are defined by checker. */

  /* The following field value is TRUE if the unit is used in an
     regexp. */
  char unit_is_used;
  /* The following field value is order number (0, 1, ...) of given
     unit. */
  int unit_num;
  /* The following field value is corresponding declaration of
     automaton which was given in description.  If the field value is
     NULL then automaton in the unit declaration was absent. */
  struct automaton_decl *automaton_decl;
  /* The following field value is maximal cycle number (1, ...) on
     which given unit occurs in insns.  Zero value means that given
     unit is not used in insns. */
  int max_occ_cycle_num;
  /* The following list contains units which conflict with given
     unit. */
  struct unit_set_el *excl_list;
  /* The following list contains units which are required to
     reservation of given unit. */
  struct unit_set_el *req_list;

  /* The following fields are defined by automaton generator. */

  /* The following field value is number of the automaton to which
     given unit belongs. */
  int corresponding_automaton_num;
};

/* This describes define_bypass (see file rtl.def). */
struct bypass_decl
{
  int latency;
  char *out_insn_name;
  char *in_insn_name;

  /* The following fields are defined by checker. */

  /* output and input insns of given bypass. */
  struct insn_reserv_decl *out_insn_reserv;
  struct insn_reserv_decl *in_insn_reserv;
  /* The next bypass for given output insn. */
  struct bypass_decl *next;
};

/* This describes define_automaton (see file rtl.def). */
struct automaton_decl
{
  char *name;

  /* The following fields are defined by automaton generator. */

  /* The following field value is TRUE if the automaton is used in an
     regexp definition. */
  char automaton_is_used;

  /* The following fields are defined by checker. */

  /* The following field value is the corresponding automaton.  This
     field is not NULL only if the automaton is present in unit
     declarations and the automatic partition on automata is not
     used. */
  struct automaton *corresponding_automaton;
};

/* This describes exclusion_set (see file rtl.def). */
struct excl_decl
{
  int names_num;
  int first_list_length;
  char *names [1];
};

/* This describes require_set (see file rtl.def). */
struct req_decl
{
  int names_num;
  int first_list_length;
  char *names [1];
};

/* This describes define_reservation (see file rtl.def). */
struct reserv_decl
{
  char *name;
  struct regexp *regexp;

  /* The following fields are defined by checker. */

  /* The following field value is TRUE if the unit is used in an
     regexp. */
  char reserv_is_used;
  /* The following field is used to check up cycle in expression
     definition. */
  int loop_pass_num;
};

/* This describes define_insn_reservartion (see file rtl.def). */
struct insn_reserv_decl
{
  rtx condexp;
  int default_latency;
  struct regexp *regexp;
  char *name;

  /* The following fields are defined by checker. */

  /* The following field value is order number (0, 1, ...) of given
     insn. */
  int insn_num;
  /* The following field value is list of bypasses in which given insn
     is output insn. */
  struct bypass_decl *bypass_list;

  /* The following fields are defined by automaton generator. */

  /* The following field is the insn regexp transformed that
     the regexp has not optional regexp, repetition regexp, and an
     reservation name (i.e. reservation identifiers are changed by the
     corresponding regexp) and all alternations are the topest level
     of the regexp.  The value can be NULL only if it is special
     insn `cycle advancing'. */
  struct regexp *transformed_regexp;
  /* The following field value is list of arcs marked given
     insn.  The field is used in transfromation NDFA -> DFA. */
  struct arc *arcs_marked_by_insn;
  /* The two following fields are used during minimization of a finite state
     automaton. */
  /* The field value is number of equivalence class of state into
     which arc marked by given insn enters from a state (fixed during
     an automaton minimization). */
  int equiv_class_num;
  /* The field value is state_alts of arc leaving a state (fixed
     during an automaton minimization) and marked by given insn
     enters. */
  int state_alts;
};

/* This contains a declaration mentioned above. */
struct decl
{
  /* What node in the union? */
  enum decl_mode mode;
  pos_t pos;
  union
  {
    struct unit_decl unit;
    struct bypass_decl bypass;
    struct automaton_decl automaton;
    struct excl_decl excl;
    struct req_decl req;
    struct reserv_decl reserv;
    struct insn_reserv_decl insn_reserv;
  } decl;
};

/* The following structures represent parsed reservation strings. */
enum regexp_mode
{
  rm_unit,
  rm_reserv,
  rm_nothing,
  rm_sequence,
  rm_repeat,
  rm_allof,
  rm_oneof
};

/* Cpu unit in reservation. */
struct unit_regexp
{
  char *name;
  struct unit_decl *unit_decl;
};

/* Define_reservation in a reservation. */
struct reserv_regexp
{
  char *name;
  struct reserv_decl *reserv_decl;
};

/* Absence of reservation (represented by string `nothing'). */
struct nothing_regexp
{
  /* This used to be empty but ISO C doesn't allow that.  */
  char unused;
};

/* Representation of reservations separated by ',' (see file
   rtl.def). */
struct sequence_regexp
{
  int regexps_num;
  struct regexp *regexps [1];
};

/* Representation of construction `repeat' (see file rtl.def). */
struct repeat_regexp
{
  int repeat_num;
  struct regexp *regexp;
};

/* Representation of reservations separated by '+' (see file
   rtl.def). */
struct allof_regexp
{
  int regexps_num;
  struct regexp *regexps [1];
};

/* Representation of reservations separated by '|' (see file
   rtl.def). */
struct oneof_regexp
{
  int regexps_num;
  struct regexp *regexps [1];
};

/* Representation of a reservation string. */
struct regexp
{
  /* What node in the union? */
  enum regexp_mode mode;
  pos_t pos;
  union
  {
    struct unit_regexp unit;
    struct reserv_regexp reserv;
    struct nothing_regexp nothing;
    struct sequence_regexp sequence;
    struct repeat_regexp repeat;
    struct allof_regexp allof;
    struct oneof_regexp oneof;
  } regexp;
};

/* Reperesents description of pipeline hazard description based on
   NDFA. */
struct description
{
  int decls_num;

  /* The following fields are defined by checker. */

  /* The following fields values are correspondingly number of units
     and insns in the description. */
  int units_num;
  /* The following field is the number of insns. */
  int insns_num;
  /* The following field value is max length (in cycles) of
     reservations of insns.  The field value is defined only for
     correct programs. */
  int max_insn_reserv_cycles;

  /* The following fields are defined by automaton generator. */

  /* The following field value is the first automaton. */
  struct automaton *first_automaton;

  /* The following field is created by pipeline hazard parser and
     contains all declarations.  We allocate additional entry for
     special insn "cycle advancing" which is added by the automaton
     generator. */
  struct decl *decls [1];
};



/* The following nodes are created in automaton checker. */

/* The following nodes represent exclusion & require set for cpu
   units.  Each element are accessed through only one excl_list or
   req_list. */
struct unit_set_el
{
  struct unit_decl *unit_decl;
  struct unit_set_el *next_unit_set_el;
};



/* The following nodes are created in automaton generator. */

/* The following node type describes state automaton.  The state may
   be deterministic or non-deterministic.  Non-deterministic state has
   several component states which represent alternative cpu units
   reservations.  The state also is used for describing a
   deterministic reservation of automaton insn. */
struct state
{
  /* The following member is TRUE if there is a transition by cycle
     advancing. */
  int new_cycle_p;
  /* The following field is list of processor unit reservations on
     each cycle. */
  reserv_sets_t reservs;
  /* The following field is unique number of given state between other
     states. */
  int unique_num;
  /* The following field value is automaton to which given state
     belongs. */
  struct automaton *automaton;
  /* The following field value is the first arc output from given
     state. */
  struct arc *first_out_arc;
  /* The following field is used to form NDFA. */
  char it_was_placed_in_stack_for_NDFA_forming;
  /* The following field is used to form DFA. */
  char it_was_placed_in_stack_for_DFA_forming;
  /* The following field is used to transform NDFA to DFA.  The field
     value is not NULL if the state is a compound state.  In this case
     the value of field `unit_sets_list' is NULL.  All states in the
     list are in the hash table.  The list is formed through field
     `next_sorted_alt_state'. */
  struct alt_state *component_states;
  /* The following field is used for passing graph of states. */
  int pass_num;
  /* The list of states belonging to one equivalence class is formed
     with the aid of the following field. */
  struct state *next_equiv_class_state;
  /* The two following fields are used during minimization of a finite
     state automaton. */
  int equiv_class_num_1, equiv_class_num_2;
  /* The following field is used during minimization of a finite state
     automaton.  The field value is state corresponding to equivalence
     class to which given state belongs. */
  struct state *equiv_class_state;
  /* The following field value is the order number of given state.
     The states in final DFA is enumerated with the aid of the
     following field. */
  int order_state_num;
};

/* Automaton arc. */
struct arc
{
  /* The following field refers for the state into which given arc
     enters. */
  struct state *to_state;
  /* The following field describes that the insn issue (with cycle
     advancing for special insn `cycle advancing' and without cycle
     advancing for others) makes transition from given state to
     another given state. */
  struct ainsn *insn;
  /* The following field value is the next arc output from the same
     state. */
  struct arc *next_out_arc;
  /* List of arcs marked given insn is formed with the following
     field.  The field is used in transfromation NDFA -> DFA. */
  struct arc *next_arc_marked_by_insn;
  /* The following field is defined if NDFA_FLAG is FALSE.  The member
     value is number of alternative reservations which can be used for
     transition for given state by given insn. */
  int state_alts;
  /* This member is used for passing states for searching minimal
     delay time. */
  int pass_number;
};

/* The following node type describes a deterministic alternative in
   non-deterministic state which characterizes cpu unit reservations
   of automaton insn or which is part of NDFA. */
struct alt_state
{
  /* The following field is a determinist state which characterizes
     unit reservations of the instruction. */
  struct state *state;
  /* The following field refers to the next state which characterizes
     unit reservations of the instruction. */
  struct alt_state *next_alt_state;
  /* The following field refers to the next state in sorted list. */
  struct alt_state *next_sorted_alt_state;
};

/* The following node type describes insn of automaton.  They are
   labels of FA arcs. */
struct ainsn
{
  /* The following field value is the corresponding insn declaration
     of description. */
  struct insn_reserv_decl *insn_reserv_decl;
  /* The following field value is the next insn declaration for an
     automaton. */
  struct ainsn *next_ainsn;
  /* The following field is states which characterize automaton unit
     reservations of the instruction.  The value can be NULL only if it
     is special insn `cycle advancing'. */
  struct alt_state *alt_states;
  /* The following field is sorted list of states which characterize
     automaton unit reservations of the instruction.  The value can be
     NULL only if it is special insn `cycle advancing'. */
  struct alt_state *sorted_alt_states;
  /* The following field refers the next automaton insn with
     the same reservations. */
  struct ainsn *next_same_reservs_insn;
  /* The following field is flag of the first automaton insn with the
     same reservations in the declaration list.  Only arcs marked such
     insn is present in the automaton.  This significantly decreases
     memory requirements especially when several automata are
     formed. */
  char first_insn_with_same_reservs;
  /* Cyclic list of insns of a equivalence class is formed with the
     aid of the following field. */
  struct ainsn *next_equiv_class_insn;
  /* The following field value is TRUE if the insn
     declaration is the first insn declaration with given
     equivalence number. */
  char first_ainsn_with_given_equialence_num;
  /* The following field is number of class of equivalence of insns.
     It is necessary because many insns may be equivalent with the
     point of view of pipeline hazards. */
  int insn_equiv_class_num;
};

/* The folowing describes an automaton for PHR. */
struct automaton
{
  /* The following field value is the list of insn declarations for
     given automaton. */
  struct ainsn *ainsn_list;
  /* The following field value is the corresponding automaton
     declaration.  This field is not NULL only if the automatic
     partition on automata is not used. */
  struct automaton_decl *corresponding_automaton_decl;
  /* The following field value is the next automaton. */
  struct automaton *next_automaton;
  /* The following field is start state of FA.  There are not unit
     reservations in the state. */
  struct state *start_state;
  /* The following field value is number of equivalence classes of
     insns (see field `insn_equiv_class_num' in
     `insn_reserv_decl'). */
  int insn_equiv_classes_num;
  /* The following field value is number of states of final DFA. */
  int achieved_states_num;
  /* The following field value is the order number (0, 1, ...) of
     given automaton. */
  int automaton_order_num;
  /* The following fields contain statistics information about
     building automaton. */
  int NDFA_states_num, DFA_states_num;
  /* The following field value is defined only if minimization of DFA
     is used. */
  int minimal_DFA_states_num;
  int NDFA_arcs_num, DFA_arcs_num;
  /* The following field value is defined only if minimization of DFA
     is used. */
  int minimal_DFA_arcs_num;
  /* The following three members refer for three table state x ainsn
     -> int. */
  struct state_ainsn_table *trans_table;
  struct state_ainsn_table *state_alts_table;
  struct state_ainsn_table *min_issue_delay_table;
};

/* The following structure describes a table state X ainsn -> int(>= 0). */
struct state_ainsn_table
{
  /* Automaton to which given table belongs. */
  struct automaton *automaton;
  /* The following tree vectors for comb vector implementation of the
     table. */
  vla_hwint_t comb_vect;
  vla_hwint_t check_vect;
  vla_hwint_t base_vect;
  /* This is simple implementation of the table. */
  vla_hwint_t full_vect;
  /* Minimal and maximal values of the previous vectors. */
  int min_comb_vect_el_value, max_comb_vect_el_value;
  int min_base_vect_el_value, max_base_vect_el_value;
};

/* Create IR structure (node). */
static void *
create_node (size)
     size_t size;
{
  void *result;

  IR_TOP_EXPAND (size);
  result = IR_TOP_BEGIN ();
  IR_TOP_FINISH ();
  /* Default values of members are NULL and zero. */
  memset (result, 0, size);
  return result;
}

/* Copy IR structure (node). */
static void *
copy_node (from, size)
     void *from;
     size_t size;
{
  void *result;
  result = create_node (size);
  memcpy (result, from, size);
  return result;
}

/* The function checks that NAME does not contain ". */
static char *
check_name (name, pos)
     char * name;
     pos_t pos ATTRIBUTE_UNUSED;
{
  char *str;

  for (str = name; *str != '\0'; str++)
    if (*str == '\"')
      error ("Name `%s' contains quotes", name);
  return name;
}

/* Pointers top all declartions during IR generation are stored in the
   following. */
static vla_ptr_t decls;

/* Given a pointer to a (char *) and a separator, return a alloc'ed
   string containing the next separated element, taking parentheses
   into account if PAR_FLAG is TRUE.  Advance the pointer to after the
   string scanned, or the end-of-string.  Return NULL if at end of
   string.  */
static char *
next_sep_el (pstr, sep, par_flag)
     char **pstr;
     int sep;
     int par_flag;
{
  char *out_str;
  char *p;
  int pars_num;
  int n_spaces;

  /* Remove leading whitespaces. */
  while (isspace ((int) **pstr))
    (*pstr)++;

  if (**pstr == '\0')
    return NULL;

  n_spaces = 0;
  for (pars_num = 0, p = *pstr; *p != '\0'; p++)
    {
      if (par_flag && *p == '(')
	pars_num++;
      else if (par_flag && *p == ')')
	pars_num--;
      else if (pars_num == 0 && *p == sep)
	break;
      if (pars_num == 0 && isspace ((int) *p))
	n_spaces++;
      else
	{
	  for (; n_spaces != 0; n_spaces--)
	    IR_TOP_ADD_BYTE (p [-n_spaces]);
	  IR_TOP_ADD_BYTE (*p);
	}
    }
  IR_TOP_ADD_BYTE ('\0');
  out_str = IR_TOP_BEGIN ();
  IR_TOP_FINISH ();

  *pstr = p;
  if (**pstr == sep)
    (*pstr)++;

  return out_str;
}

/* Given a string and a separator, return the number of separated
   elements in it, taking parentheses into account if PAR_FLAG is
   TRUE.  Return 0 for the null string, -1 if parantheses is not
   balanced.  */
static int
n_sep_els (s, sep, par_flag)
     char *s;
     int sep;
     int par_flag;
{
  int n;
  int pars_num;

  if (*s == '\0')
    return 0;

  for (pars_num = 0, n = 1; *s; s++)
    if (par_flag && *s == '(')
      pars_num++;
    else if (par_flag && *s == ')')
      pars_num--;
    else if (pars_num == 0 && *s == sep)
      n++;

  return (pars_num != 0 ? -1 : n);
}

/* Given a string and a separator, return vector of strings which are
   elements in the string and number of elements through els_num.
   Take parentheses into account if PAR_FLAG is TRUE.  Return 0 for
   the null string, -1 if parantheses are not balanced.  */
static char **
get_str_vect (str, els_num, sep, par_flag)
     char *str;
     int *els_num;
     int sep;
     int par_flag;
{
  int i;
  char **vect;
  char **pstr;

  *els_num = n_sep_els (str, sep, par_flag);
  if (*els_num <= 0)
    return NULL;
  IR_TOP_EXPAND (sizeof (char *) * (*els_num));
  vect = (char **) IR_TOP_BEGIN ();
  IR_TOP_FINISH ();
  pstr = &str;
  for (i = 0; i < *els_num; i++)
    vect [i] = next_sep_el (pstr, sep, par_flag);
  assert (next_sep_el (pstr, sep, par_flag) == NULL);
  return vect;
}

/* Process a DEFINE_CPU_UNIT.  

   This gives information about a unit contained in CPU.  We fill a
   struct unit_decl with information used later by `expand_automata'.  */
static void
gen_cpu_unit (def)
     rtx def;
{
  struct decl *decl;
  char **str_cpu_units;
  int vect_length;
  int i;

  str_cpu_units = get_str_vect (XSTR (def, 0), &vect_length, ',', FALSE);
  if (str_cpu_units == NULL)
    fatal ("invalid string `%s' in define_cpu_unit", XSTR (def, 0));
  for (i = 0; i < vect_length; i++)
    {
      decl = create_node (sizeof (struct decl));
      decl->mode = dm_unit;
      decl->pos = 0;
      decl->decl.unit.name = check_name (str_cpu_units [i], decl->pos);
      decl->decl.unit.automaton_name = XSTR (def, 1);
      VLA_PTR_ADD (decls, decl);
      num_dfa_decls++;
    }
}

/* Process a DEFINE_BYPASS.  

   This gives information about a unit contained in the CPU.  We fill
   in a struct bypass_decl with information used later by
   `expand_automata'.  */
static void
gen_bypass (def)
     rtx def;
{
  struct decl *decl;
  char **out_insns;
  int out_length;
  char **in_insns;
  int in_length;
  int i, j;

  out_insns = get_str_vect (XSTR (def, 1), &out_length, ',', FALSE);
  if (out_insns == NULL)
    fatal ("invalid string `%s' in define_bypass", XSTR (def, 1));
  in_insns = get_str_vect (XSTR (def, 2), &in_length, ',', FALSE);
  if (in_insns == NULL)
    fatal ("invalid string `%s' in define_bypass", XSTR (def, 2));
  for (i = 0; i < out_length; i++)
    for (j = 0; j < in_length; j++)
      {
	decl = create_node (sizeof (struct decl));
	decl->mode = dm_bypass;
	decl->pos = 0;
	decl->decl.bypass.latency = XINT (def, 0);
	decl->decl.bypass.out_insn_name = out_insns [i];
	decl->decl.bypass.in_insn_name = in_insns [j];
	VLA_PTR_ADD (decls, decl);
	num_dfa_decls++;
      }
}

/* Process a EXCLUSION_SET.  

   This gives information about a cpu unit conflicts.  We fill a
   struct excl_decl with information used later by `expand_automata'.  */
static void
gen_excl_set (def)
     rtx def;
{
  struct decl *decl;
  char **first_str_cpu_units;
  char **second_str_cpu_units;
  int first_vect_length;
  int length;
  int i;

  first_str_cpu_units
    = get_str_vect (XSTR (def, 0), &first_vect_length, ',', FALSE);
  if (first_str_cpu_units == NULL)
    fatal ("invalid first string `%s' in exclusion_set", XSTR (def, 0));
  second_str_cpu_units = get_str_vect (XSTR (def, 1), &length, ',', FALSE);
  if (second_str_cpu_units == NULL)
    fatal ("invalid second string `%s' in exclusion_set", XSTR (def, 1));
  length += first_vect_length;
  decl = create_node (sizeof (struct decl) + (length - 1) * sizeof (char *));
  decl->mode = dm_excl;
  decl->pos = 0;
  decl->decl.excl.names_num = length;
  decl->decl.excl.first_list_length = first_vect_length;
  for (i = 0; i < length; i++)
    if (i < first_vect_length)
      decl->decl.excl.names [i] = first_str_cpu_units [i];
    else
      decl->decl.excl.names [i]
        = second_str_cpu_units [i - first_vect_length];
  VLA_PTR_ADD (decls, decl);
  num_dfa_decls++;
}

/* Process a REQUIRE_SET.  

   This gives information about a cpu unit reservation requirements.
   We fill a struct req_decl with information used later by
   `expand_automata'.  */
static void
gen_req_set (def)
     rtx def;
{
  struct decl *decl;
  char **first_str_cpu_units;
  char **second_str_cpu_units;
  int first_vect_length;
  int length;
  int i;

  first_str_cpu_units
    = get_str_vect (XSTR (def, 0), &first_vect_length, ',', FALSE);
  if (first_str_cpu_units == NULL)
    fatal ("invalid first string `%s' in require_set", XSTR (def, 0));
  second_str_cpu_units = get_str_vect (XSTR (def, 1), &length, ',', FALSE);
  if (second_str_cpu_units == NULL)
    fatal ("invalid second string `%s' in require_set", XSTR (def, 1));
  length += first_vect_length;
  decl = create_node (sizeof (struct decl) + (length - 1) * sizeof (char *));
  decl->mode = dm_req;
  decl->pos = 0;
  decl->decl.req.names_num = length;
  decl->decl.req.first_list_length = first_vect_length;
  for (i = 0; i < length; i++)
    if (i < first_vect_length)
      decl->decl.req.names [i] = first_str_cpu_units [i];
    else
      decl->decl.req.names [i]
        = second_str_cpu_units [i - first_vect_length];
  VLA_PTR_ADD (decls, decl);
  num_dfa_decls++;
}

/* Process a DEFINE_AUTOMATON.  

   This gives information about a finite state automaton used for
   recognizing pipeline hazards.  We fill a struct automaton_decl
   with information used later by `expand_automata'.  */
static void
gen_automaton (def)
     rtx def;
{
  struct decl *decl;
  char **str_automata;
  int vect_length;
  int i;

  str_automata = get_str_vect (XSTR (def, 0), &vect_length, ',', FALSE);
  if (str_automata == NULL)
    fatal ("invalid string `%s' in define_automaton", XSTR (def, 0));
  for (i = 0; i < vect_length; i++)
    {
      decl = create_node (sizeof (struct decl));
      decl->mode = dm_automaton;
      decl->pos = 0;
      decl->decl.automaton.name = check_name (str_automata [i], decl->pos);
      VLA_PTR_ADD (decls, decl);
      num_dfa_decls++;
    }
}

/* Name in reservation to denote absence reservation. */
#define NOTHING_NAME "nothing"

/* The following string contains original reservation string being
   parsed. */
static char *reserv_str;

/* Parse an element in STR. */
static struct regexp *
gen_regexp_el (str)
     char *str;
{
  struct regexp *regexp;
  int len;

  if (*str == '(')
    {
      len = strlen (str);
      if (str [len - 1] != ')')
	fatal ("garbage after ) in reservation `%s'", reserv_str);
      str [len - 1] = '\0';
      regexp = gen_regexp_sequence (str + 1);
    }
  else if (strcmp (str, NOTHING_NAME) == 0)
    {
      regexp = create_node (sizeof (struct decl));
      regexp->mode = rm_nothing;
    }
  else
    {
      regexp = create_node (sizeof (struct decl));
      regexp->mode = rm_unit;
      regexp->regexp.unit.name = str;
    }
  return regexp;
}

/* Parse construction `repeat' in STR. */
static struct regexp *
gen_regexp_repeat (str)
     char *str;
{
  struct regexp *regexp;
  struct regexp *repeat;
  char **repeat_vect;
  int els_num;
  int i;

  repeat_vect = get_str_vect (str, &els_num, '*', TRUE);
  if (repeat_vect == NULL)
    fatal ("invalid `%s' in reservation `%s'", str, reserv_str);
  if (els_num > 1)
    {
      regexp = gen_regexp_el (repeat_vect [0]);
      for (i = 1; i < els_num; i++)
	{
	  repeat = create_node (sizeof (struct regexp));
	  repeat->mode = rm_repeat;
	  repeat->regexp.repeat.regexp = regexp;
	  repeat->regexp.repeat.repeat_num = atoi (repeat_vect [i]);
          if (repeat->regexp.repeat.repeat_num <= 1)
            fatal ("repetition `%s' <= 1 in reservation `%s'",
                   str, reserv_str);
          regexp = repeat;
	}
      return regexp;
    }
  else
    return gen_regexp_el (str);
}

/* Parse reservation STR which possibly contains separator '+'. */
static struct regexp *
gen_regexp_allof (str)
     char *str;
{
  struct regexp *allof;
  char **allof_vect;
  int els_num;
  int i;

  allof_vect = get_str_vect (str, &els_num, '+', TRUE);
  if (allof_vect == NULL)
    fatal ("invalid `%s' in reservation `%s'", str, reserv_str);
  if (els_num > 1)
    {
      allof = create_node (sizeof (struct regexp));
      allof->mode = rm_allof;
      allof->regexp.allof.regexps_num = els_num;
      for (i = 0; i < els_num; i++)
	allof->regexp.allof.regexps [i] = gen_regexp_repeat (allof_vect [i]);
      return allof;
    }
  else
    return gen_regexp_repeat (str);
}

/* Parse reservation STR which possibly contains separator '|'. */
static struct regexp *
gen_regexp_oneof (str)
     char *str;
{
  struct regexp *oneof;
  char **oneof_vect;
  int els_num;
  int i;

  oneof_vect = get_str_vect (str, &els_num, '|', TRUE);
  if (oneof_vect == NULL)
    fatal ("invalid `%s' in reservation `%s'", str, reserv_str);
  if (els_num > 1)
    {
      oneof = create_node (sizeof (struct regexp));
      oneof->mode = rm_oneof;
      oneof->regexp.oneof.regexps_num = els_num;
      for (i = 0; i < els_num; i++)
	oneof->regexp.oneof.regexps [i] = gen_regexp_allof (oneof_vect [i]);
      return oneof;
    }
  else
    return gen_regexp_allof (str);
}

/* Parse reservation STR which possibly contains separator ','. */
static struct regexp *
gen_regexp_sequence (str)
     char *str;
{
  struct regexp *sequence;
  char **sequence_vect;
  int els_num;
  int i;

  sequence_vect = get_str_vect (str, &els_num, ',', TRUE);
  if (els_num > 1)
    {
      sequence = create_node (sizeof (struct regexp));
      sequence->mode = rm_sequence;
      sequence->regexp.sequence.regexps_num = els_num;
      for (i = 0; i < els_num; i++)
	sequence->regexp.sequence.regexps [i]
          = gen_regexp_oneof (sequence_vect [i]);
      return sequence;
    }
  else
    return gen_regexp_oneof (str);
}

/* Parse construction reservation STR. */
static struct regexp *
gen_regexp (str)
     char *str;
{
  reserv_str = str;
  return gen_regexp_sequence (str);;
}

/* Process a DEFINE_RESERVATION.

   This gives information about a reservation of cpu units.  We fill
   in a struct reserv_decl with information used later by
   `expand_automata'.  */
static void
gen_reserv (def)
     rtx def;
{
  struct decl *decl;

  decl = create_node (sizeof (struct decl));
  decl->mode = dm_reserv;
  decl->pos = 0;
  decl->decl.reserv.name = check_name (XSTR (def, 0), decl->pos);
  decl->decl.reserv.regexp = gen_regexp (XSTR (def, 1));
  VLA_PTR_ADD (decls, decl);
  num_dfa_decls++;
}

/* Process a DEFINE_INSN_RESERVATION.

   This gives information about the reservation of cpu units by an
   insn.  We fill a struct insn_reserv_decl with information used
   later by `expand_automata'.  */
static void
gen_insn_reserv (def)
     rtx def;
{
  struct decl *decl;

  decl = create_node (sizeof (struct decl));
  decl->mode = dm_insn_reserv;
  decl->pos = 0;
  decl->decl.insn_reserv.name = check_name (XSTR (def, 0), decl->pos);
  decl->decl.insn_reserv.default_latency = XINT (def, 1);
  decl->decl.insn_reserv.condexp = XEXP (def, 2);
  decl->decl.insn_reserv.regexp = gen_regexp (XSTR (def, 3));
  VLA_PTR_ADD (decls, decl);
  num_dfa_decls++;
}



/* The function evaluates hash value (0..UINT_MAX) of string. */
static unsigned
string_hash (string)
     const char *string;
{
  unsigned result, i;

  for (result = i = 0;*string++ != '\0'; i++)
    result += ((unsigned char) *string << (i % CHAR_BIT));
  return result;
}



/* This page contains abstract data `table of automaton declarations'.
   Elements of the table is nodes representing automaton declarations.
   Key of the table elements is name of given automaton.  Rememeber
   that automaton names have own space. */

/* The function evaluates hash value of a automaton declaration.  The
   function is used by abstract data `hashtab'.  The function returns
   hash value (0..UINT_MAX) of given automaton declaration. */
static unsigned
automaton_decl_hash (automaton_decl)
     const void *automaton_decl;
{
  const struct decl *decl = (struct decl *) automaton_decl;

  assert (decl->mode == dm_automaton && decl->decl.automaton.name != NULL);
  return string_hash (decl->decl.automaton.name);
}

/* The function tests automaton declarations on equality of their
   keys.  The function is used by abstract data `hashtab'.  The
   function returns 1 if the declarations have the same key, 0
   otherwise.  */
static int
automaton_decl_eq_p (automaton_decl_1, automaton_decl_2)
     const void* automaton_decl_1;
     const void* automaton_decl_2;
{
  const struct decl *decl1 = (struct decl *) automaton_decl_1;
  const struct decl *decl2 = (struct decl *) automaton_decl_2;

  assert (decl1->mode == dm_automaton && decl1->decl.automaton.name != NULL
	  && decl2->mode == dm_automaton
          && decl2->decl.automaton.name != NULL);
  return strcmp (decl1->decl.automaton.name, decl2->decl.automaton.name) == 0;
}

/* The automaton declaration table itself is represented by the
   following variable. */
static htab_t automaton_decl_table;

/* The function inserts automaton declaration into the table.  The
   function does nothing if an automaton declaration with the same key
   exists already in the table.  The function returns automaton
   declaration node in the table with the same key as given automaton
   declaration node. */
static struct decl *
insert_automaton_decl (automaton_decl)
     struct decl *automaton_decl;
{
  void **entry_ptr;

  entry_ptr = htab_find_slot (automaton_decl_table, automaton_decl, TRUE);
  if (*entry_ptr == NULL)
    *entry_ptr = (void *) automaton_decl;
  return (struct decl *) *entry_ptr;
}

/* The following variable value is node representing automaton
   declaration.  The node used for searching automaton declaration
   with given name. */
static struct decl work_automaton_decl;

/* The function searches for automaton declaration in the table with
   the same key as node representing name of the automaton
   declaration.  The function returns node found in the table, NULL if
   such node does not exist in the table. */
static struct decl *
find_automaton_decl (name)
     char *name;
{
  void *entry;

  work_automaton_decl.decl.automaton.name = name;
  entry = htab_find (automaton_decl_table, &work_automaton_decl);
  return (struct decl *) entry;
}

/* The function creates empty automaton declaration table and node
   representing automaton declaration and used for searching automaton
   declaration with given name.  The function must be called only once
   before any work with the automaton declaration table. */
static void
initiate_automaton_decl_table ()
{
  work_automaton_decl.mode = dm_automaton;
  automaton_decl_table = htab_create (10, automaton_decl_hash,
				      automaton_decl_eq_p, (htab_del) 0);
}

/* The function deletes the automaton declaration table.  Only call of
   function `initiate_automaton_decl_table' is possible immediately
   after this function call. */
static void
finish_automaton_decl_table ()
{
  htab_delete (automaton_decl_table);
}



/* This page contains abstract data `table of insn declarations'.
   Elements of the table is nodes representing insn declarations.  Key
   of the table elements is name of given insn (in corresponding
   define_insn_reservation).  Rememeber that insn names have own
   space. */

/* The function evaluates hash value of a insn declaration.  The
   function is used by abstract data `hashtab'.  The function returns
   hash value (0..UINT_MAX) of given insn declaration. */
static unsigned
insn_decl_hash (insn_decl)
     const void *insn_decl;
{
  const struct decl *decl = (struct decl *) insn_decl;

  assert (decl->mode == dm_insn_reserv && decl->decl.insn_reserv.name != NULL);
  return string_hash (decl->decl.insn_reserv.name);
}

/* The function tests insn declarations on equality of their keys.
   The function is used by abstract data `hashtab'.  The function
   returns 1 if declarations have the same key, 0 otherwise.  */
static int
insn_decl_eq_p (insn_decl_1, insn_decl_2)
     const void *insn_decl_1;
     const void *insn_decl_2;
{
  const struct decl *decl1 = (struct decl *) insn_decl_1;
  const struct decl *decl2 = (struct decl *) insn_decl_2;

  assert (decl1->mode == dm_insn_reserv && decl1->decl.insn_reserv.name != NULL
	  && decl2->mode == dm_insn_reserv
	  && decl2->decl.insn_reserv.name != NULL);
  return strcmp (decl1->decl.insn_reserv.name,
                 decl2->decl.insn_reserv.name) == 0;
}

/* The insn declaration table itself is represented by the following
   variable.  The table does not contain insn reservation
   declarations. */
static htab_t insn_decl_table;

/* The function inserts insn declaration into the table.  The function
   does nothing if an insn declaration with the same key exists
   already in the table.  The function returns insn declaration node
   in the table with the same key as given insn declaration node. */
static struct decl *
insert_insn_decl (insn_decl)
     struct decl *insn_decl;
{
  void **entry_ptr;

  entry_ptr = htab_find_slot (insn_decl_table, insn_decl, TRUE);
  if (*entry_ptr == NULL)
    *entry_ptr = (void *) insn_decl;
  return (struct decl *) *entry_ptr;
}

/* The following variable value is node representing insn reservation
   declaration.  The node used for searching insn reservation
   declaration with given name. */
static struct decl work_insn_decl;

/* The function searches for insn reservation declaration in the table
   with the same key as node representing name of the insn reservation
   declaration.  The function returns node found in the table, NULL if
   such node does not exist in the table. */
static struct decl *
find_insn_decl (name)
     char *name;
{
  void *entry;

  work_insn_decl.decl.insn_reserv.name = name;
  entry = htab_find (insn_decl_table, &work_insn_decl);
  return (struct decl *) entry;
}

/* The function creates empty insn declaration table and node
   representing insn declaration and used for searching insn
   declaration with given name.  The function must be called only once
   before any work with the insn declaration table. */
static void
initiate_insn_decl_table ()
{
  work_insn_decl.mode = dm_insn_reserv;
  insn_decl_table = htab_create (10, insn_decl_hash, insn_decl_eq_p,
				 (htab_del) 0);
}

/* The function deletes the insn declaration table.  Only call of
   function `initiate_insn_decl_table' is possible immediately after
   this function call. */
static void
finish_insn_decl_table ()
{
  htab_delete (insn_decl_table);
}



/* This page contains abstract data `table of declarations'.  Elements
   of the table is nodes representing declarations (of units and
   reservations).  Key of the table elements is names of given
   declarations. */

/* The function evaluates hash value of a declaration.  The function
   is used by abstract data `hashtab'.  The function returns hash
   value (0..UINT_MAX) of given declaration.  */
static unsigned
decl_hash (decl)
     const void *decl;
{
  const struct decl *d = (const struct decl *) decl;

  assert ((d->mode == dm_unit && d->decl.unit.name)
	  || (d->mode == dm_reserv && d->decl.reserv.name));
  return string_hash (d->mode == dm_unit
		      ? d->decl.unit.name : d->decl.reserv.name);
}

/* The function tests declarations on equality of their keys.  The
   function is used by abstract data `hashtab'.  The function
   returns 1 if the declarations have the same key, 0 otherwise.  */
static int
decl_eq_p (decl_1, decl_2)
     const void *decl_1;
     const void *decl_2;
{
  const struct decl *d1 = (const struct decl *) decl_1;
  const struct decl *d2 = (const struct decl *) decl_2;

  assert (((d1->mode == dm_unit && d1->decl.unit.name)
	   || (d1->mode == dm_reserv && d1->decl.reserv.name))
	  && ((d2->mode == dm_unit && d2->decl.unit.name)
	      || (d2->mode == dm_reserv && d2->decl.reserv.name)));
  return strcmp ((d1->mode == dm_unit
                  ? d1->decl.unit.name : d1->decl.reserv.name),
                 (d2->mode == dm_unit
                  ? d2->decl.unit.name : d2->decl.reserv.name)) == 0;
}

/* The declaration table itself is represented by the following
   variable. */
static htab_t decl_table;

/* The function inserts declaration into the table.  The function does
   nothing if a declaration with the same key exists already in the
   table.  The function returns declaration node in the table with the
   same key as given declaration node. */

static struct decl *
insert_decl (decl)
     struct decl *decl;
{
  void **entry_ptr;

  entry_ptr = htab_find_slot (decl_table, decl, TRUE);
  if (*entry_ptr == NULL)
    *entry_ptr = (void *) decl;
  return (struct decl *) *entry_ptr;
}

/* The following variable value is node representing declaration.  The
   node used for searching declaration with given name. */
static struct decl work_decl;

/* The function searches for declaration in the table with the same
   key as node representing name of the declaration.  The function
   returns node found in the table, NULL if such node does not exist
   in the table. */
static struct decl *
find_decl (name)
     char *name;
{
  void *entry;

  work_decl.decl.unit.name = name;
  entry = htab_find (decl_table, &work_decl);
  return (struct decl *) entry;
}

/* The function creates empty declaration table and node representing
   declaration and used for searching declaration with given name.
   The function must be called only once before any work with the
   declaration table.  */
static void
initiate_decl_table ()
{
  work_decl.mode = dm_unit;
  decl_table = htab_create (10, decl_hash, decl_eq_p, (htab_del) 0);
}

/* The function deletes the declaration table.  Only call of function
   `initiate_declaration_table' is possible immediately after this
   function call. */
static void
finish_decl_table ()
{
  htab_delete (decl_table);
}



/* This page contains checker of pipeline hazard description. */

/* Checking NAMES in an exclusion clause vector and returning formed
   unit_set_el_list. */
static struct unit_set_el *
process_excls (names, num, excl_pos)
     char **names;
     int num;
     pos_t excl_pos ATTRIBUTE_UNUSED;
{
  struct unit_set_el *el_list;
  struct unit_set_el *last_el;
  struct unit_set_el *new_el;
  struct decl *decl_in_table;
  int i;

  el_list = NULL;
  last_el = NULL;
  for (i = 0; i < num; i++)
    {
      decl_in_table = find_decl (names [i]);
      if (decl_in_table == NULL)
	error ("unit `%s' in exclusion is not declared", names [i]);
      else if (decl_in_table->mode != dm_unit)
	error ("`%s' in exclusion is not unit", names [i]);
      else
	{
	  new_el = create_node (sizeof (struct unit_set_el));
	  new_el->unit_decl = &decl_in_table->decl.unit;
	  new_el->next_unit_set_el = NULL;
	  if (last_el == NULL)
	    el_list = last_el = new_el;
	  else
	    {
	      last_el->next_unit_set_el = new_el;
	      last_el = last_el->next_unit_set_el;
	    }
	}
    }
  return el_list;
}

/* The function adds each element from SOURCE_LIST to the exclusion
   list of the each element from DEST_LIST.  Checking situation "unit
   excludes itself". */
static void
add_excls (dest_list, source_list, excl_pos)
     struct unit_set_el *dest_list;
     struct unit_set_el *source_list;
     pos_t excl_pos ATTRIBUTE_UNUSED;
{
  struct unit_set_el *curr_dest_el;
  struct unit_set_el *curr_source_el;
  struct unit_set_el *curr_excl_list_unit;
  struct unit_set_el *prev_excl_list_unit;
  struct unit_set_el *copy;

  for (curr_dest_el = dest_list;
       curr_dest_el != NULL;
       curr_dest_el = curr_dest_el->next_unit_set_el)
    for (curr_source_el = source_list;
	 curr_source_el != NULL;
	 curr_source_el = curr_source_el->next_unit_set_el)
      {
	if (curr_dest_el->unit_decl == curr_source_el->unit_decl)
	  {
	    error ("unit `%s' excludes itself",
		   curr_source_el->unit_decl->name);
	    continue;
	  }
	for (curr_excl_list_unit = curr_dest_el->unit_decl->excl_list,
	     prev_excl_list_unit = NULL;
	     curr_excl_list_unit != NULL;
	     prev_excl_list_unit = curr_excl_list_unit,
	     curr_excl_list_unit = curr_excl_list_unit->next_unit_set_el)
	  if (curr_excl_list_unit->unit_decl == curr_source_el->unit_decl)
	    break;
	if (curr_excl_list_unit == NULL)
	  {
	    /* Element not found - insert. */
	    copy = copy_node (curr_source_el, sizeof (*curr_source_el));
	    copy->next_unit_set_el = NULL;
	    if (prev_excl_list_unit == NULL)
	      curr_dest_el->unit_decl->excl_list = copy;
	    else
	      prev_excl_list_unit->next_unit_set_el = copy;
	}
    }
}

/* Checking NAMES in an require clause vector and returning formed
   unit_set_el_list.  The function is called only after processing
   all exclusion sets. */
static struct unit_set_el *
process_reqs (names, num, req_pos)
     char **names;
     int num;
     pos_t req_pos ATTRIBUTE_UNUSED;
{
  struct unit_set_el *el_list;
  struct unit_set_el *last_el;
  struct unit_set_el *new_el;
  struct decl *decl_in_table;
  int i;

  el_list = NULL;
  last_el = NULL;
  for (i = 0; i < num; i++)
    {
      decl_in_table = find_decl (names [i]);
      if (decl_in_table == NULL)
	error ("unit `%s' in require set is not declared", names [i]);
      else if (decl_in_table->mode != dm_unit)
	error ("`%s' in require set is not unit", names [i]);
      else
	{
	  new_el = create_node (sizeof (struct unit_set_el));
	  new_el->unit_decl = &decl_in_table->decl.unit;
	  new_el->next_unit_set_el = NULL;
	  if (last_el == NULL)
	    el_list = last_el = new_el;
	  else
	    {
	      last_el->next_unit_set_el = new_el;
	      last_el = last_el->next_unit_set_el;
	    }
	}
    }
  return el_list;
}

/* The function adds each element from SOURCE_LIST to the require list
   of the each element from DEST_LIST.  Checking situations "unit
   requires itself" and "unit is required and excluded". */
static void
add_reqs (dest_list, source_list, req_pos)
     struct unit_set_el *dest_list;
     struct unit_set_el *source_list;
     pos_t req_pos ATTRIBUTE_UNUSED;
{
  struct unit_set_el *curr_dest_el;
  struct unit_set_el *curr_source_el;
  struct unit_set_el *curr_req_list_unit;
  struct unit_set_el *prev_req_list_unit;
  struct unit_set_el *copy;

  for (curr_dest_el = dest_list;
       curr_dest_el != NULL;
       curr_dest_el = curr_dest_el->next_unit_set_el)
    for (curr_source_el = source_list;
	 curr_source_el != NULL;
	 curr_source_el = curr_source_el->next_unit_set_el)
      {
	if (curr_dest_el->unit_decl == curr_source_el->unit_decl)
	  {
	    error ("unit `%s' requires itself",
		   curr_source_el->unit_decl->name);
	    continue;
	  }
	for (curr_req_list_unit = curr_dest_el->unit_decl->req_list,
	     prev_req_list_unit = NULL;
	     curr_req_list_unit != NULL;
	     prev_req_list_unit = curr_req_list_unit,
	     curr_req_list_unit = curr_req_list_unit->next_unit_set_el)
	  if (curr_req_list_unit->unit_decl == curr_source_el->unit_decl)
	    break;
	if (curr_req_list_unit == NULL)
	  {
	    /* Element not found - insert if there is no error. */
	    struct unit_set_el *curr_excl_list_unit;
	    int no_error_flag = TRUE;

	    for (curr_excl_list_unit = curr_dest_el->unit_decl->excl_list;
		 curr_excl_list_unit != NULL;
		 curr_excl_list_unit = curr_excl_list_unit->next_unit_set_el)
	      if (curr_source_el->unit_decl == curr_excl_list_unit->unit_decl)
		{
		  error ("unit `%s' requires and excludes `%s'",
			 curr_dest_el->unit_decl->name,
			 curr_source_el->unit_decl->name);
		  no_error_flag = FALSE;
		}
	    if (no_error_flag)
	      {
		copy = copy_node (curr_source_el, sizeof (*curr_source_el));
		copy->next_unit_set_el = NULL;
		if (prev_req_list_unit == NULL)
		  curr_dest_el->unit_decl->req_list = copy;
		else
		  prev_req_list_unit->next_unit_set_el = copy;
	      }
	}
    }
}

/* The function searches for bypass with given IN_INSN_RESERV in given
   BYPASS_LIST. */
static struct bypass_decl *
find_bypass (bypass_list, in_insn_reserv)
     struct bypass_decl *bypass_list;
     struct insn_reserv_decl *in_insn_reserv;
{
  struct bypass_decl *curr_bypass;

  for (curr_bypass = bypass_list;
       curr_bypass != NULL;
       curr_bypass = curr_bypass->next)
    if (curr_bypass->in_insn_reserv == in_insn_reserv)
      break;
  return curr_bypass;
}

/* The function fixes errors `... already declared' and warnings
   `... repeated declaration'.  The function also fixes occurences of
   undeclared automata name in unit declarations or absence of insn
   name in unit declarations if there are automaton declarations.  The
   function checks bypasses and forms list of bypasses for each
   (output) insn. */
static void
process_decls ()
{
  struct decl *curr_decl;
  struct decl *automaton_decl;
  struct decl *decl_in_table;
  struct decl *out_insn_reserv;
  struct decl *in_insn_reserv;
  struct bypass_decl *bypass;
  int automaton_presence;
  int i;

  automaton_presence = FALSE;
  for (i = 0; i < description->decls_num; i++)
    {
      curr_decl = description->decls [i];
      if (curr_decl->mode == dm_automaton)
	{
	  automaton_presence = TRUE;
	  decl_in_table = insert_automaton_decl (curr_decl);
	  if (decl_in_table != curr_decl)
	    warning ("repeated declaration of automaton `%s'",
		     curr_decl->decl.automaton.name);
	}
    }
  for (i = 0; i < description->decls_num; i++)
    {
      curr_decl = description->decls [i];
      if (curr_decl->mode == dm_insn_reserv)
	{
          curr_decl->decl.insn_reserv.condexp
	    = check_attr_test (curr_decl->decl.insn_reserv.condexp, 0);
	  if (curr_decl->decl.insn_reserv.default_latency < 0)
	    error ("define_insn_reservation `%s' has negative latency time",
		   curr_decl->decl.insn_reserv.name);
	  curr_decl->decl.insn_reserv.insn_num = description->insns_num;
	  description->insns_num++;
	  decl_in_table = insert_insn_decl (curr_decl);
	  if (decl_in_table != curr_decl)
	    error ("`%s' is already used as insn reservation name",
		   curr_decl->decl.insn_reserv.name);
	}
      else if (curr_decl->mode == dm_bypass)
	{
	  if (curr_decl->decl.bypass.latency < 0)
	    error ("define_bypass `%s - %s' has negative latency time",
		   curr_decl->decl.bypass.out_insn_name,
		   curr_decl->decl.bypass.in_insn_name);
	}
      else if (curr_decl->mode == dm_unit || curr_decl->mode == dm_reserv)
	{
	  if (curr_decl->mode == dm_unit)
	    {
	      curr_decl->decl.unit.automaton_decl = NULL;
	      if (curr_decl->decl.unit.automaton_name != NULL)
		{
		  automaton_decl
                    = find_automaton_decl
                      (curr_decl->decl.unit.automaton_name);
		  if (automaton_decl == NULL)
		    error ("automaton `%s' is not declared",
			   curr_decl->decl.unit.automaton_name);
		  else
		    {
		      automaton_decl->decl.automaton.automaton_is_used = TRUE;
		      curr_decl->decl.unit.automaton_decl
			= &automaton_decl->decl.automaton;
		    }
		}
	      else if (automaton_presence)
		error ("define_unit `%s' without automaton when one defined",
		       curr_decl->decl.unit.name);
	      curr_decl->decl.unit.unit_num = description->units_num;
	      description->units_num++;
	      if (strcmp (curr_decl->decl.unit.name, NOTHING_NAME) == 0)
		{
		  error ("`%s' is declared as cpu unit", NOTHING_NAME);
		  continue;
		}
	      decl_in_table = find_decl (curr_decl->decl.unit.name);
	    }
	  else
	    {
	      if (strcmp (curr_decl->decl.reserv.name, NOTHING_NAME) == 0)
		{
		  error ("`%s' is declared as cpu reservation", NOTHING_NAME);
		  continue;
		}
	      decl_in_table = find_decl (curr_decl->decl.reserv.name);
	    }
	  if (decl_in_table == NULL)
	    decl_in_table = insert_decl (curr_decl);
	  else
	    {
	      if (curr_decl->mode == dm_unit)
		error ("repeated declaration of unit `%s'",
		       curr_decl->decl.unit.name);
	      else
		error ("repeated declaration of reservation `%s'",
		       curr_decl->decl.reserv.name);
	    }
	}
    }
  for (i = 0; i < description->decls_num; i++)
    {
      curr_decl = description->decls [i];
      if (curr_decl->mode == dm_bypass)
	{
	  out_insn_reserv
	    = find_insn_decl (curr_decl->decl.bypass.out_insn_name);
	  in_insn_reserv
	    = find_insn_decl (curr_decl->decl.bypass.in_insn_name);
	  if (out_insn_reserv == NULL)
	    error ("there is no insn reservation `%s'",
		   curr_decl->decl.bypass.out_insn_name);
	  else if (in_insn_reserv == NULL)
	    error ("there is no insn reservation `%s'",
		   curr_decl->decl.bypass.in_insn_name);
	  else
	    {
	      curr_decl->decl.bypass.out_insn_reserv
		= &out_insn_reserv->decl.insn_reserv;
	      curr_decl->decl.bypass.in_insn_reserv
		= &in_insn_reserv->decl.insn_reserv;
	      bypass
		= find_bypass (out_insn_reserv->decl.insn_reserv.bypass_list,
			       curr_decl->decl.bypass.in_insn_reserv);
	      if (bypass != NULL)
		{
		  if (curr_decl->decl.bypass.latency == bypass->latency)
		    warning ("the same bypass `%s - %s' is already defined",
			     curr_decl->decl.bypass.out_insn_name,
			     curr_decl->decl.bypass.in_insn_name);
		  else
		    error ("bypass `%s - %s' is already defined",
			   curr_decl->decl.bypass.out_insn_name,
			   curr_decl->decl.bypass.in_insn_name);
		}
	      else
		{
		  curr_decl->decl.bypass.next
		    = out_insn_reserv->decl.insn_reserv.bypass_list;
		  out_insn_reserv->decl.insn_reserv.bypass_list
		    = &curr_decl->decl.bypass;
		}
	    }
	}
    }

  for (i = 0; i < description->decls_num; i++)
    {
      curr_decl = description->decls [i];
      if (curr_decl->mode == dm_excl)
	{
	  struct unit_set_el *unit_set_el_list;
	  struct unit_set_el *unit_set_el_list_2;
	  
	  unit_set_el_list
            = process_excls (curr_decl->decl.excl.names,
                             curr_decl->decl.excl.first_list_length,
                             curr_decl->pos);
	  unit_set_el_list_2
	    = process_excls (&curr_decl->decl.excl.names
                             [curr_decl->decl.excl.first_list_length],
                             curr_decl->decl.excl.names_num
                             - curr_decl->decl.excl.first_list_length,
                             curr_decl->pos);
	  add_excls (unit_set_el_list, unit_set_el_list_2, curr_decl->pos);
	  add_excls (unit_set_el_list_2, unit_set_el_list, curr_decl->pos);
	}
    }

  for (i = 0; i < description->decls_num; i++)
    {
      curr_decl = description->decls [i];
      if (curr_decl->mode == dm_req)
	{
	  struct unit_set_el *unit_set_el_list;
	  struct unit_set_el *unit_set_el_list_2;
	  
	  unit_set_el_list
            = process_reqs (curr_decl->decl.req.names,
                             curr_decl->decl.req.first_list_length,
                             curr_decl->pos);
	  unit_set_el_list_2
	    = process_reqs (&curr_decl->decl.req.names
			    [curr_decl->decl.req.first_list_length],
			    curr_decl->decl.req.names_num
			    - curr_decl->decl.req.first_list_length,
			    curr_decl->pos);
	  add_reqs (unit_set_el_list, unit_set_el_list_2, curr_decl->pos);
	}
    }
}

/* The following function checks that declared automaton is used.  If
   the automaton is not used, the function fixes error.  The following
   function must be called only after `process_decls'. */
static void
check_automaton_usage ()
{
  struct decl *curr_decl;
  int i;

  for (i = 0; i < description->decls_num; i++)
    {
      curr_decl = description->decls [i];
      if (curr_decl->mode == dm_automaton
          && !curr_decl->decl.automaton.automaton_is_used)
	warning ("automaton `%s' is not used", curr_decl->decl.automaton.name);
    }
}

/* The following recursive function processes all regexp in order to
   fix usage of units or reservations and to fix errors of undeclared
   name.  The function may change unit_regexp onto reserv_regexp.
   Remember that reserv_regexp does not exist before the function
   call. */
static struct regexp *
process_regexp (struct regexp *regexp)
{
  struct decl *decl_in_table;
  struct regexp *new_regexp;
  int i;
    
  if (regexp->mode == rm_unit)
    {
      decl_in_table = find_decl (regexp->regexp.unit.name);
      if (decl_in_table == NULL)
        error ("undeclared unit or reservation `%s'",
	       regexp->regexp.unit.name);
      else if (decl_in_table->mode == dm_unit)
	{
	  decl_in_table->decl.unit.unit_is_used = TRUE;
	  regexp->regexp.unit.unit_decl = &decl_in_table->decl.unit;
	}
      else if (decl_in_table->mode == dm_reserv)
	{
	  decl_in_table->decl.reserv.reserv_is_used = TRUE;
	  new_regexp = create_node (sizeof (struct regexp));
	  new_regexp->mode = rm_reserv;
	  new_regexp->pos = regexp->pos;
	  new_regexp->regexp.reserv.name = regexp->regexp.unit.name;
	  new_regexp->regexp.reserv.reserv_decl = &decl_in_table->decl.reserv;
	  regexp = new_regexp;
	}
      else
	assert (FALSE);
    }
  else if (regexp->mode == rm_sequence)
    for (i = 0; i < regexp->regexp.sequence.regexps_num; i++)
      regexp->regexp.sequence.regexps [i]
	= process_regexp (regexp->regexp.sequence.regexps [i]);
  else if (regexp->mode == rm_allof)
    for (i = 0; i < regexp->regexp.allof.regexps_num; i++)
      regexp->regexp.allof.regexps [i]
        = process_regexp (regexp->regexp.allof.regexps [i]);
  else if (regexp->mode == rm_oneof)
    for (i = 0; i < regexp->regexp.oneof.regexps_num; i++)
      regexp->regexp.oneof.regexps [i]
	= process_regexp (regexp->regexp.oneof.regexps [i]);
  else if (regexp->mode == rm_repeat)
    regexp->regexp.repeat.regexp
      = process_regexp (regexp->regexp.repeat.regexp);
  else
    assert (regexp->mode == rm_nothing);
  return regexp;
}

/* The following function processes regexp of define_reservation and
   define_insn_reservation with the aid of function
   `process_regexp'. */
static void
process_regexp_decls ()
{
  struct decl *curr_decl;
  int i;

  for (i = 0; i < description->decls_num; i++)
    {
      curr_decl = description->decls [i];
      if (curr_decl->mode == dm_reserv)
	curr_decl->decl.reserv.regexp
          = process_regexp (curr_decl->decl.reserv.regexp);
      else if (curr_decl->mode == dm_insn_reserv)
	curr_decl->decl.insn_reserv.regexp
	  = process_regexp (curr_decl->decl.insn_reserv.regexp);
    }
}

/* The following function checks that declared unit is used.  If the
   unit is not used, the function fixes warning.  The following
   function must be called only after `process_decls',
   `process_regexp_decls'. */
static void
check_usage ()
{
  struct decl *curr_decl;
  int i;

  for (i = 0; i < description->decls_num; i++)
    {
      curr_decl = description->decls [i];
      if (curr_decl->mode == dm_unit && !curr_decl->decl.unit.unit_is_used)
	warning ("unit `%s' is not used", curr_decl->decl.unit.name);
      else if (curr_decl->mode == dm_reserv
               && !curr_decl->decl.reserv.reserv_is_used)
	warning ("reservation `%s' is not used", curr_decl->decl.reserv.name);
    }
}

/* The following variable value is number of reservation being
   processed on loop recognition. */
static int current_loop_pass_num;

/* The following recursive function returns TRUE if REGEXP contains
   given decl or reservations in given regexp refers for given
   decl. */
static int
loop_in_regexp (regexp, start_decl)
     struct regexp *regexp;
     struct decl *start_decl;
{
  int i;

  if (regexp == NULL)
    return FALSE;
  if (regexp->mode == rm_unit)
    return FALSE;
  else if (regexp->mode == rm_reserv)
    {
      if (start_decl->mode == dm_reserv
          && regexp->regexp.reserv.reserv_decl == &start_decl->decl.reserv)
        return TRUE;
      else if (regexp->regexp.reserv.reserv_decl->loop_pass_num
	       == current_loop_pass_num)
        /* declaration has been processed. */
        return FALSE;
      else
        {
	  regexp->regexp.reserv.reserv_decl->loop_pass_num
            = current_loop_pass_num;
          return loop_in_regexp (regexp->regexp.reserv.reserv_decl->regexp,
                                 start_decl);
        }
    }
  else if (regexp->mode == rm_sequence)
    {
      for (i = 0; i < regexp->regexp.sequence.regexps_num; i++)
	if (loop_in_regexp (regexp->regexp.sequence.regexps [i], start_decl))
	  return TRUE;
      return FALSE;
    }
  else if (regexp->mode == rm_allof)
    {
      for (i = 0; i < regexp->regexp.allof.regexps_num; i++)
	if (loop_in_regexp (regexp->regexp.allof.regexps [i], start_decl))
	  return TRUE;
      return FALSE;
    }
  else if (regexp->mode == rm_oneof)
    {
      for (i = 0; i < regexp->regexp.oneof.regexps_num; i++)
	if (loop_in_regexp (regexp->regexp.oneof.regexps [i], start_decl))
	  return TRUE;
      return FALSE;
    }
  else if (regexp->mode == rm_repeat)
    return loop_in_regexp (regexp->regexp.repeat.regexp, start_decl);
  else
    {
      assert (regexp->mode == rm_nothing);
      return FALSE;
    }
}

/* The following function fixes errors "cycle in definition ...".  The
   function uses function `loop_in_regexp' for that. */
static void
check_loops_in_regexps ()
{
  struct decl *curr_decl;
  int i;

  for (i = 0; i < description->decls_num; i++)
    {
      curr_decl = description->decls [i];
      if (curr_decl->mode == dm_reserv)
	curr_decl->decl.reserv.loop_pass_num = 0;
    }
  for (i = 0; i < description->decls_num; i++)
    {
      curr_decl = description->decls [i];
      current_loop_pass_num = i;
      
      if (curr_decl->mode == dm_reserv)
	  {
	    curr_decl->decl.reserv.loop_pass_num = current_loop_pass_num;
	    if (loop_in_regexp (curr_decl->decl.reserv.regexp, curr_decl))
	      {
		assert (curr_decl->decl.reserv.regexp != NULL);
		error ("cycle in definition of reservation `%s'",
		       curr_decl->decl.reserv.name);
	      }
	  }
    }
}

/* The function recursively processes IR of reservation and defines
   max and min cycle for reservation of unit and for result in the
   reservation. */
static int
process_regexp_cycles (regexp, start_cycle)
     struct regexp *regexp;
     int start_cycle;
{
  int i;

  if (regexp->mode == rm_unit)
    {
      if (regexp->regexp.unit.unit_decl->max_occ_cycle_num < start_cycle)
	regexp->regexp.unit.unit_decl->max_occ_cycle_num = start_cycle;
      return start_cycle;
    }
  else if (regexp->mode == rm_reserv)
    return process_regexp_cycles (regexp->regexp.reserv.reserv_decl->regexp,
                                  start_cycle);
  else if (regexp->mode == rm_repeat)
    {
      int current_repetition_num;

      for (current_repetition_num = 0;
           current_repetition_num < regexp->regexp.repeat.repeat_num;
           current_repetition_num++)
        start_cycle = process_regexp_cycles (regexp->regexp.repeat.regexp,
					     start_cycle) + 1;
      return start_cycle;
    }
  else if (regexp->mode == rm_sequence)
    {
      for (i = 0; i < regexp->regexp.sequence.regexps_num; i++)
	start_cycle
          = process_regexp_cycles (regexp->regexp.sequence.regexps [i],
				   start_cycle) + 1;
      return start_cycle;
    }
  else if (regexp->mode == rm_allof)
    {
      int finish_cycle = 0;
      int cycle;

      for (i = 0; i < regexp->regexp.allof.regexps_num; i++)
	{
	  cycle = process_regexp_cycles (regexp->regexp.allof.regexps [i],
					 start_cycle);
	  if (finish_cycle < cycle)
	    finish_cycle = cycle;
	}
      return finish_cycle;
    }
  else if (regexp->mode == rm_oneof)
    {
      int finish_cycle = 0;
      int cycle;

      for (i = 0; i < regexp->regexp.oneof.regexps_num; i++)
	{
	  cycle = process_regexp_cycles (regexp->regexp.oneof.regexps [i],
					 start_cycle);
	  if (finish_cycle < cycle)
	    finish_cycle = cycle;
	}
      return finish_cycle;
    }
  else
    {
      assert (regexp->mode == rm_nothing);
      return start_cycle;
    }
}

/* The following function is called only for correct program.  The
   function defines max reservation of insns in cycles.  */
static void
evaluate_max_reserv_cycles ()
{
  int max_insn_cycles_num;
  struct decl *curr_decl;
  int i;

  description->max_insn_reserv_cycles = 0;
  for (i = 0; i < description->decls_num; i++)
    {
      curr_decl = description->decls [i];
      if (curr_decl->mode == dm_insn_reserv)
      {
        max_insn_cycles_num
          = process_regexp_cycles (curr_decl->decl.insn_reserv.regexp, 0);
        if (description->max_insn_reserv_cycles < max_insn_cycles_num)
	  description->max_insn_reserv_cycles = max_insn_cycles_num;
      }
    }
}

/* The following function calls functions for checking all
   description. */
static void
check_all_description ()
{
  process_decls ();
  check_automaton_usage ();
  process_regexp_decls ();
  check_usage ();
  check_loops_in_regexps ();
  if (!have_error)
    evaluate_max_reserv_cycles ();
}



/* The page contains abstract data `ticker'. */

/* The following function creates ticker and makes it active. */
static ticker_t
create_ticker ()
{
  ticker_t ticker;

  ticker.modified_creation_time = get_run_time ();
  ticker.incremented_off_time = 0;
  return ticker;
}

/* The following function switches off given ticker. */
static void
ticker_off (ticker)
     ticker_t *ticker;
{
  if (ticker->incremented_off_time == 0)
    ticker->incremented_off_time = get_run_time () + 1;
}

/* The following function switches on given ticker. */
static void
ticker_on (ticker)
     ticker_t *ticker;
{
  if (ticker->incremented_off_time != 0)
    {
      ticker->modified_creation_time
        += get_run_time () - ticker->incremented_off_time + 1;
      ticker->incremented_off_time = 0;
    }
}

/* The following function returns current time in milliseconds since
   the moment when given ticker was created. */
static int
active_time (ticker)
     ticker_t ticker;
{
  if (ticker.incremented_off_time != 0)
    return ticker.incremented_off_time - 1 - ticker.modified_creation_time;
  else
    return get_run_time () - ticker.modified_creation_time;
}

/* The following function returns string representation of active time
   of given ticker.  The result is string representation of seconds
   with accuracy of 1/100 second.  Only result of the last call of the
   function exists.  Therefore the following code is not correct

      printf ("parser time: %s\ngeneration time: %s\n",
              active_time_string (parser_ticker),
              active_time_string (generation_ticker));

   Correct code has to be the following

      printf ("parser time: %s\n", active_time_string (parser_ticker));
      printf ("generation time: %s\n",
              active_time_string (generation_ticker));

*/
static void
print_active_time (f, ticker)
     FILE *f;
     ticker_t ticker;
{
  int msecs;

  msecs = active_time (ticker);
  fprintf (f, "%d.%06d", msecs / 1000000, msecs % 1000000);
}



/* The following variable value is number of automaton which are
   really being created.  This value is defined on the base of
   argument of option `-split'.  If the variable has zero value the
   number of automata is defined by the constructions `%automaton'.
   This case occures when option `-split' is absent or has zero
   argument.  If constructions `define_automaton' is absent only one
   automaton is created. */
static int automata_num;

/* The following variable values are times of
       o transformation of regular expressions
       o building NDFA (DFA if !ndfa_flag)
       o NDFA -> DFA   (simply the same automaton if !ndfa_flag)
       o DFA minimization
       o building insn equivalence classes
       o all previous ones
       o code output */
static ticker_t transform_time;
static ticker_t NDFA_time;
static ticker_t NDFA_to_DFA_time;
static ticker_t minimize_time;
static ticker_t equiv_time;
static ticker_t automaton_generation_time;
static ticker_t output_time;

/* The following variable values are times of
       all checking
       all generation
       all pipeline hazard translator work */
static ticker_t check_time;
static ticker_t generation_time;
static ticker_t all_time;



/* Pseudo insn decl which denotes advancing cycle. */
static struct decl *advance_cycle_insn_decl;
static void
add_advance_cycle_insn_decl ()
{
  advance_cycle_insn_decl = create_node (sizeof (struct decl));
  advance_cycle_insn_decl->mode = dm_insn_reserv;
  advance_cycle_insn_decl->pos = no_pos;
  advance_cycle_insn_decl->decl.insn_reserv.regexp = NULL;
  advance_cycle_insn_decl->decl.insn_reserv.name = "$advance_cycle";
  advance_cycle_insn_decl->decl.insn_reserv.insn_num = description->insns_num;
  description->decls [description->decls_num] = advance_cycle_insn_decl;
  description->decls_num++;
  description->insns_num++;
  num_dfa_decls++;
}


/* Abstract data `alternative states' which reperesents
   nondeterministic nature of the description (see comments for
   structures alt_state and state). */

/* List of free states. */
static struct alt_state *first_free_alt_state;

#ifndef NDEBUG
/* The following variables is maximal number of allocated nodes
   alt_state. */
static int allocated_alt_states_num = 0;
#endif

/* The following function returns free node alt_state.  It may be new
   allocated node or node freed eralier. */
static struct alt_state *
get_free_alt_state ()
{
  struct alt_state *result;

  if (first_free_alt_state != NULL)
    {
      result = first_free_alt_state;
      first_free_alt_state = first_free_alt_state->next_alt_state;
    }
  else
    {
#ifndef NDEBUG
      allocated_alt_states_num++;
#endif
      result = create_node (sizeof (struct alt_state));
    }
  result->state = NULL;
  result->next_alt_state = NULL;
  result->next_sorted_alt_state = NULL;
  return result;
}

/* The function frees node ALT_STATE. */
static void
free_alt_state (alt_state)
     struct alt_state *alt_state;
{
  if (alt_state == NULL)
    return;
  alt_state->next_alt_state = first_free_alt_state;
  first_free_alt_state = alt_state;
}

/* The function frees list started with node ALT_STATE_LIST. */
static void
free_alt_states (alt_states_list)
     struct alt_state *alt_states_list;
{
  struct alt_state *current_alt_state;
  struct alt_state *next_alt_state;

  for (current_alt_state = alt_states_list;
       current_alt_state != NULL;
       current_alt_state = next_alt_state)
    {
      next_alt_state = current_alt_state->next_alt_state;
      free_alt_state (current_alt_state);
    }
}

/* The function compares unique numbers of alt states. */
static int
alt_state_cmp (alt_state_ptr_1, alt_state_ptr_2)
     const void *alt_state_ptr_1;
     const void *alt_state_ptr_2;
{
  if ((*(struct alt_state **) alt_state_ptr_1)->state->unique_num
      == (*(struct alt_state **) alt_state_ptr_2)->state->unique_num)
    return 0;
  else if ((*(struct alt_state **) alt_state_ptr_1)->state->unique_num
	   < (*(struct alt_state **) alt_state_ptr_2)->state->unique_num)
    return -1;
  else
    return 1;
}

/* The function sorts ALT_STATES_LIST and removes duplicated alt
   states from the list.  The comparison key is alt state unique
   number. */
static struct alt_state *
uniq_sort_alt_states (alt_states_list)
     struct alt_state *alt_states_list;
{
  struct alt_state *curr_alt_state;
  vla_ptr_t alt_states;
  size_t curr_index;
  size_t prev_unique_state_index;
  struct alt_state *result;
  struct alt_state **result_ptr;

  VLA_PTR_CREATE (alt_states, 150, "alt_states");
  for (curr_alt_state = alt_states_list;
       curr_alt_state != NULL;
       curr_alt_state = curr_alt_state->next_alt_state)
    VLA_PTR_ADD (alt_states, curr_alt_state);
  qsort (VLA_PTR_BEGIN (alt_states), VLA_PTR_LENGTH (alt_states),
	 sizeof (struct alt_state *), alt_state_cmp);
  if (VLA_PTR_LENGTH (alt_states) == 0)
    result = NULL;
  else
    {
      result_ptr = VLA_PTR_BEGIN (alt_states);
      for (prev_unique_state_index = 0, curr_index = 1;
           curr_index < VLA_PTR_LENGTH (alt_states);
           curr_index++)
        if (result_ptr [prev_unique_state_index]->state
            != result_ptr [curr_index]->state)
          {
            prev_unique_state_index++;
            result_ptr [prev_unique_state_index] = result_ptr [curr_index];
          }
#if 0
      for (curr_index = prev_unique_state_index + 1;
           curr_index < VLA_PTR_LENGTH (alt_states);
           curr_index++)
        free_alt_state (result_ptr [curr_index]);
#endif
      VLA_PTR_SHORTEN (alt_states, curr_index - prev_unique_state_index - 1);
      result_ptr = VLA_PTR_BEGIN (alt_states);
      for (curr_index = 1;
           curr_index < VLA_PTR_LENGTH (alt_states);
           curr_index++)
        result_ptr [curr_index - 1]->next_sorted_alt_state
	  = result_ptr [curr_index];
      result_ptr [curr_index - 1]->next_sorted_alt_state = NULL;
      result = *result_ptr;
    }
  VLA_PTR_DELETE (alt_states);
  return result;
}

/* The function checks equality of alt state lists.  Remember that the
   lists must be already sorted by the previous function. */
static int
alt_states_eq (alt_states_1, alt_states_2)
     struct alt_state *alt_states_1;
     struct alt_state *alt_states_2;
{
  while (alt_states_1 != NULL && alt_states_2 != NULL
         && alt_state_cmp (&alt_states_1, &alt_states_2) == 0)
    {
      alt_states_1 = alt_states_1->next_sorted_alt_state;
      alt_states_2 = alt_states_2->next_sorted_alt_state;
    }
  return alt_states_1 == alt_states_2;
}

/* Initialization of the abstract data. */
static void
initiate_alt_states ()
{
  first_free_alt_state = NULL;
}

/* Finishing work with the abstract data. */
static void
finish_alt_states ()
{
}



/* The page contains macros for work with bits strings. */

/* Set bit number bitno in the bit string.  The macro is not side
   effect proof. */
#define SET_BIT(bitstring, bitno)					  \
  (((char *) (bitstring)) [(bitno) / CHAR_BIT] |= 1 << (bitno) % CHAR_BIT)

/* Test if bit number bitno in the bitstring is set.  The macro is not
   side effect proof. */
#define TEST_BIT(bitstring, bitno)                                        \
  (((char *) (bitstring)) [(bitno) / CHAR_BIT] >> (bitno) % CHAR_BIT & 1)



/* This page contains abstract data `state'. */

/* Maximal length of reservations in cycles (> 1). */
static int max_cycles_num;

/* Number of set elements (see type set_el_t) needed for
   representation of one cycle reservation.  It is depended on units
   number. */
static int els_in_cycle_reserv;

/* Number of set elements (see type set_el_t) needed for
   representation of maximal length reservation.  Deterministic
   reservation is stored as set (bit string) of length equal to the
   variable value * number of bits in set_el_t. */
static int els_in_reservs;

/* VLA for representation of array of pointers to unit
   declarations. */
static vla_ptr_t units_container;

/* The start address of the array. */
static struct unit_decl **units_array;

/* Empty reservation of maximal length. */
static reserv_sets_t empty_reserv;

/* The state table itself is represented by the following variable. */
static htab_t state_table;

/* VLA for representation of array of pointers to free nodes
   `state'. */
static vla_ptr_t free_states;

static int curr_unique_state_num;

#ifndef NDEBUG
/* The following variables is maximal number of allocated nodes
   `state'. */
static int allocated_states_num = 0;
#endif

/* Allocate new reservation set. */
static reserv_sets_t
alloc_empty_reserv_sets ()
{
  reserv_sets_t result;

  IR_TOP_EXPAND (els_in_reservs * sizeof (set_el_t));
  result = (reserv_sets_t) IR_TOP_BEGIN ();
  IR_TOP_FINISH ();
  memset (result, 0, els_in_reservs * sizeof (set_el_t));
  return result;
}

/* Hash value of reservation set. */
static unsigned
reserv_sets_hash_value (reservs)
     reserv_sets_t reservs;
{
  unsigned int hash_value;
  int unit_reserv_set_els_num;
  set_el_t *unit_reserv_set_el_ptr;

  hash_value = 0;
  unit_reserv_set_els_num = els_in_reservs;
  unit_reserv_set_el_ptr = reservs;
  while (unit_reserv_set_els_num != 0)
    {
      unit_reserv_set_els_num--;
      hash_value = ((hash_value >> (sizeof (unsigned) - 1) * CHAR_BIT)
                    | (hash_value << CHAR_BIT)) + *unit_reserv_set_el_ptr;
      unit_reserv_set_el_ptr++;
    }
  return hash_value;
}

/* Comparison of given reservation sets. */
static int
reserv_sets_cmp (reservs_1, reservs_2)
     reserv_sets_t reservs_1;
     reserv_sets_t reservs_2;
{
  int unit_reserv_set_els_num;
  set_el_t *unit_reserv_set_el_ptr_1;
  set_el_t *unit_reserv_set_el_ptr_2;

  assert (reservs_1 != NULL && reservs_2 != NULL);
  unit_reserv_set_els_num = els_in_reservs;
  unit_reserv_set_el_ptr_1 = reservs_1;
  unit_reserv_set_el_ptr_2 = reservs_2;
  while (unit_reserv_set_els_num != 0
         && *unit_reserv_set_el_ptr_1 == *unit_reserv_set_el_ptr_2)
    {
      unit_reserv_set_els_num--;
      unit_reserv_set_el_ptr_1++;
      unit_reserv_set_el_ptr_2++;
    }
  if (unit_reserv_set_els_num == 0)
    return 0;
  else if (*unit_reserv_set_el_ptr_1 < *unit_reserv_set_el_ptr_2)
    return -1;
  else
    return 1;
}

/* The function checks equality of the reservation sets. */
static int
reserv_sets_eq (reservs_1, reservs_2)
     reserv_sets_t reservs_1;
     reserv_sets_t reservs_2;
{
  return reserv_sets_cmp (reservs_1, reservs_2) == 0;
}

/* Set up in the reservation set that unit with UNIT_NUM is used on
   CYCLE_NUM. */
static void
set_unit_reserv (reservs, cycle_num, unit_num)
     reserv_sets_t reservs;
     int cycle_num;
     int unit_num;
{
  assert (cycle_num < max_cycles_num);
  SET_BIT (reservs, cycle_num * els_in_cycle_reserv
           * sizeof (set_el_t) * CHAR_BIT + unit_num);
}

/* The function checks that the reservation set represents no one unit
   reservation. */
static int
it_is_empty_reserv_sets (operand)
     reserv_sets_t operand;
{
  set_el_t *unit_reserv_set_el_ptr;
  int unit_reserv_set_els_num;

  assert (operand != NULL);
  for (unit_reserv_set_els_num = els_in_reservs,
       unit_reserv_set_el_ptr = operand;
       unit_reserv_set_els_num != 0;
       unit_reserv_set_el_ptr++, unit_reserv_set_els_num--)
    if (*unit_reserv_set_el_ptr != 0)
      return FALSE;
  return TRUE;
}

/* The function checks that the reservation sets are intersected,
   i.e. there is a unit reservation on a cycle in both reservation
   sets. */
static int
reserv_sets_are_intersected (operand_1, operand_2)
     reserv_sets_t operand_1;
     reserv_sets_t operand_2;
{
  set_el_t *curr_unit_set_el_ptr_1;
  set_el_t *curr_unit_set_el_ptr_2;
  set_el_t *curr_cycle_ptr_1;
  set_el_t *curr_cycle_ptr_2;

  assert (operand_1 != NULL && operand_2 != NULL);
  for (curr_unit_set_el_ptr_1 = operand_1, curr_unit_set_el_ptr_2 = operand_2;
       curr_unit_set_el_ptr_1 < operand_1 + els_in_reservs;
       curr_unit_set_el_ptr_1++, curr_unit_set_el_ptr_2++)
    if (*curr_unit_set_el_ptr_1 & *curr_unit_set_el_ptr_2)
      return TRUE;
  for (curr_cycle_ptr_1 = operand_1, curr_cycle_ptr_2 = operand_2;
       curr_cycle_ptr_1 < operand_1 + els_in_reservs;
       curr_cycle_ptr_1 += els_in_cycle_reserv,
       curr_cycle_ptr_2 += els_in_cycle_reserv)
    {
      for (curr_unit_set_el_ptr_1 = curr_cycle_ptr_1,
	     curr_unit_set_el_ptr_2 = get_excl_set (curr_cycle_ptr_2);
	   curr_unit_set_el_ptr_1 < curr_cycle_ptr_1 + els_in_cycle_reserv;
	   curr_unit_set_el_ptr_1++, curr_unit_set_el_ptr_2++)
	if (*curr_unit_set_el_ptr_1 & *curr_unit_set_el_ptr_2)
	  return TRUE;
      for (curr_unit_set_el_ptr_1 = curr_cycle_ptr_1,
	     curr_unit_set_el_ptr_2 = get_req_set (curr_cycle_ptr_2);
	   curr_unit_set_el_ptr_1 < curr_cycle_ptr_1 + els_in_cycle_reserv;
	   curr_unit_set_el_ptr_1++, curr_unit_set_el_ptr_2++)
	if ((*curr_unit_set_el_ptr_1 | *curr_unit_set_el_ptr_2)
	    != *curr_unit_set_el_ptr_1)
	  return TRUE;
    }
  return FALSE;
}

/* The function sets up RESULT bits by bits of OPERAND shifted on one
   cpu cycle.  The remaining bits of OPERAND (representing the last
   cycle unit reservations) are not chenged. */
static void
reserv_sets_shift (result, operand)
     reserv_sets_t result;
     reserv_sets_t operand;
{
  int i;

  assert (result != NULL && operand != NULL && result != operand);
  for (i = els_in_cycle_reserv; i < els_in_reservs; i++)
    result [i - els_in_cycle_reserv] = operand [i];
}

/* OR of the reservation sets. */
static void
reserv_sets_or (result, operand_1, operand_2)
     reserv_sets_t result;
     reserv_sets_t operand_1;
     reserv_sets_t operand_2;
{
  set_el_t *curr_unit_set_el_ptr_1;
  set_el_t *curr_unit_set_el_ptr_2;
  set_el_t *curr_result_unit_set_el_ptr;

  assert (result != NULL && operand_1 != NULL && operand_2 != NULL);
  for (curr_unit_set_el_ptr_1 = operand_1, curr_unit_set_el_ptr_2 = operand_2,
       curr_result_unit_set_el_ptr = result;
       curr_unit_set_el_ptr_1 < operand_1 + els_in_reservs;
       curr_unit_set_el_ptr_1++, curr_unit_set_el_ptr_2++,
         curr_result_unit_set_el_ptr++)
    *curr_result_unit_set_el_ptr
      = *curr_unit_set_el_ptr_1 | *curr_unit_set_el_ptr_2;
}

/* AND of the reservation sets. */
static void
reserv_sets_and (result, operand_1, operand_2)
     reserv_sets_t result;
     reserv_sets_t operand_1;
     reserv_sets_t operand_2;
{
  set_el_t *curr_unit_set_el_ptr_1;
  set_el_t *curr_unit_set_el_ptr_2;
  set_el_t *curr_result_unit_set_el_ptr;

  assert (result != NULL && operand_1 != NULL && operand_2 != NULL);
  for (curr_unit_set_el_ptr_1 = operand_1, curr_unit_set_el_ptr_2 = operand_2,
       curr_result_unit_set_el_ptr = result;
       curr_unit_set_el_ptr_1 < operand_1 + els_in_reservs;
       curr_unit_set_el_ptr_1++, curr_unit_set_el_ptr_2++,
       curr_result_unit_set_el_ptr++)
    *curr_result_unit_set_el_ptr
      = *curr_unit_set_el_ptr_1 & *curr_unit_set_el_ptr_2;
}

/* The function outputs string representation of units reservation on
   cycle START_CYCLE in the reservation set.  The function uses repeat
   construction if REPETITION_NUM > 1. */
static void
output_cycle_reservs (f, reservs, start_cycle, repetition_num)
     FILE *f;
     reserv_sets_t reservs;
     int start_cycle;
     int repetition_num;
{
  int curr_unit_num;
  int reserved_units_num;

  reserved_units_num = 0;
  for (curr_unit_num = 0;
       curr_unit_num < description->units_num;
       curr_unit_num++)
    if (TEST_BIT (reservs, start_cycle * els_in_cycle_reserv
                  * sizeof (set_el_t) * CHAR_BIT + curr_unit_num))
      reserved_units_num++;
  assert (repetition_num > 0);
  if (repetition_num != 1 && reserved_units_num > 1)
    fprintf (f, "(");
  reserved_units_num = 0;
  for (curr_unit_num = 0;
       curr_unit_num < description->units_num;
       curr_unit_num++)
    if (TEST_BIT (reservs, start_cycle * els_in_cycle_reserv
		  * sizeof (set_el_t) * CHAR_BIT + curr_unit_num))
      {
        if (reserved_units_num != 0)
          fprintf (f, "+");
        reserved_units_num++;
        fprintf (f, "%s", units_array [curr_unit_num]->name);
      }
  if (reserved_units_num == 0)
    fprintf (f, NOTHING_NAME);
  assert (repetition_num > 0);
  if (reserved_units_num != 0 && repetition_num != 1)
    {
      if (reserved_units_num > 1)
        fprintf (f, ")");
      fprintf (f, "*%d", repetition_num);
    }
}

/* The function outputs string representation of units reservation in
   the reservation set. */
static void
output_reserv_sets (f, reservs)
     FILE *f;
     reserv_sets_t reservs;
{
  int start_cycle;
  int curr_cycle_num;
  int repetition_num;
  int next_cycle_output_flag;

  repetition_num = 0;
  next_cycle_output_flag = FALSE;
  for (curr_cycle_num = 0; curr_cycle_num < max_cycles_num; curr_cycle_num++)
    if (repetition_num == 0)
      {
        repetition_num++;
        start_cycle = curr_cycle_num;
      }
    else if (memcmp
             ((char *) reservs + start_cycle * els_in_cycle_reserv
              * sizeof (set_el_t),
              (char *) reservs + curr_cycle_num * els_in_cycle_reserv
	      * sizeof (set_el_t), els_in_cycle_reserv * sizeof (set_el_t))
	     == 0)
      repetition_num++;
    else
      {
        if (next_cycle_output_flag)
          fprintf (f, ", ");
        output_cycle_reservs (f, reservs, start_cycle, repetition_num);
        next_cycle_output_flag = TRUE;
        repetition_num = 1;
        start_cycle = curr_cycle_num;
      }
  if (!next_cycle_output_flag)
    {
      if (next_cycle_output_flag)
        fprintf (f, ", ");
      output_cycle_reservs (f, reservs, start_cycle, repetition_num);
    }
}

/* The following function returns free node state for AUTOMATON.  It
   may be new allocated node or node freed eralier.  The function also
   allocates reservation set if WITH_RESERVS==TRUE. */
static struct state *
get_free_state (with_reservs, automaton)
     int with_reservs;
     struct automaton *automaton;
{
  struct state *result;

  assert (max_cycles_num > 0 && automaton != NULL);
  if (VLA_PTR_LENGTH (free_states) != 0)
    {
      result = VLA_PTR (free_states, VLA_PTR_LENGTH (free_states) - 1);
      VLA_PTR_SHORTEN (free_states, 1);
      result->automaton = automaton;
      result->first_out_arc = NULL;
      result->it_was_placed_in_stack_for_NDFA_forming = FALSE;
      result->it_was_placed_in_stack_for_DFA_forming = FALSE;
      result->component_states = NULL;
    }
  else
    {
#ifndef NDEBUG
      allocated_states_num++;
#endif
      result = create_node (sizeof (struct state));
      result->automaton = automaton;
      result->first_out_arc = NULL;
      result->unique_num = curr_unique_state_num;
      curr_unique_state_num++;
    }
  if (with_reservs)
    {
      if (result->reservs == NULL)
        result->reservs = alloc_empty_reserv_sets ();
      else
        memset (result->reservs, 0, els_in_reservs * sizeof (set_el_t));
    }
  return result;
}

/* The function frees node STATE. */
static void
free_state (state)
     struct state *state;
{
  free_alt_states (state->component_states);
  VLA_PTR_ADD (free_states, state);
}

/* Hash value of STATE.  If STATE represents deterministic state it is
   simply hash value of the corresponding reservation set.  Otherwise
   it is formed from hash values of the component deterministic
   states.  One more key is order number of state automaton. */
static unsigned
state_hash (state)
     const void *state;
{
  unsigned int hash_value;
  struct alt_state *curr_alt_state;

  if (((struct state *) state)->component_states == NULL)
    hash_value = reserv_sets_hash_value (((struct state *) state)->reservs);
  else
    {
      hash_value = 0;
      for (curr_alt_state = ((struct state *) state)->component_states;
           curr_alt_state != NULL;
           curr_alt_state = curr_alt_state->next_sorted_alt_state)
        hash_value = (((hash_value >> (sizeof (unsigned) - 1) * CHAR_BIT)
                       | (hash_value << CHAR_BIT))
                      + curr_alt_state->state->unique_num);
    }
  hash_value = (((hash_value >> (sizeof (unsigned) - 1) * CHAR_BIT)
                 | (hash_value << CHAR_BIT))
                + ((struct state *) state)->automaton->automaton_order_num);
  return hash_value;
}

/* Return true if the states are the same. */
static int
state_eq_p (state_1, state_2)
     const void *state_1;
     const void *state_2;
{
  struct alt_state *curr_alt_state_1;
  struct alt_state *curr_alt_state_2;

  if (((struct state *) state_1)->automaton
      != ((struct state *) state_2)->automaton)
    return FALSE;
  else if (((struct state *) state_1)->component_states == NULL
           && ((struct state *) state_2)->component_states == NULL)
    return reserv_sets_eq (((struct state *) state_1)->reservs,
                           ((struct state *) state_2)->reservs);
  else if (((struct state *) state_1)->component_states != NULL
           && ((struct state *) state_2)->component_states != NULL)
    {
      for (curr_alt_state_1 = ((struct state *) state_1)->component_states,
           curr_alt_state_2 = ((struct state *) state_2)->component_states;
           curr_alt_state_1 != NULL && curr_alt_state_2 != NULL;
           curr_alt_state_1 = curr_alt_state_1->next_sorted_alt_state,
	   curr_alt_state_2 = curr_alt_state_2->next_sorted_alt_state)
        /* All state in the list must be already in the hash table.
           Also the lists must be sorted. */
        if (curr_alt_state_1->state != curr_alt_state_2->state)
          return FALSE;
      return curr_alt_state_1 == curr_alt_state_2;
    }
  else
    return FALSE;
}

/* Insert STATE into the state table. */
static struct state *
insert_state (state)
     struct state *state;
{
  void **entry_ptr;

  entry_ptr = htab_find_slot (state_table, (void *) state, TRUE);
  if (*entry_ptr == NULL)
    *entry_ptr = (void *) state;
  return (struct state *) *entry_ptr;
}

/* Add reservation of unit with UNIT_NUM on cycle CYCLE_NUM to
   deterministic STATE. */
static void
set_state_reserv (state, cycle_num, unit_num)
     struct state *state;
     int cycle_num;
     int unit_num;
{
  set_unit_reserv (state->reservs, cycle_num, unit_num);
}

/* Return TRUE if the deterministic states contains a reservation of
   the same cpu unit on the same cpu cycle. */
static int
intersected_state_reservs_p (state1, state2)
     struct state *state1;
     struct state *state2;
{
  assert (state1->automaton == state2->automaton);
  return reserv_sets_are_intersected (state1->reservs, state2->reservs);
}

/* Return deterministic state (inserted into the table) which
   representing the automaton state whic is union of reservations of
   deterministic states. */
static struct state *
states_union (state1, state2)
     struct state *state1;
     struct state *state2;
{
  struct state *result;
  struct state *state_in_table;

  assert (state1->automaton == state2->automaton);
  result = get_free_state (TRUE, state1->automaton);
  reserv_sets_or (result->reservs, state1->reservs, state2->reservs);
  state_in_table = insert_state (result);
  if (result != state_in_table)
    {
      free_state (result);
      result = state_in_table;
    }
  return result;
}

/* Return deterministic state (inserted into the table) which
   represent the automaton state is obtained from deterministic STATE
   by advancing cpu cycle. */
static struct state *
state_shift (state)
     struct state *state;
{
  struct state *result;
  struct state *state_in_table;

  result = get_free_state (TRUE, state->automaton);
  reserv_sets_shift (result->reservs, state->reservs);
  state_in_table = insert_state (result);
  if (result != state_in_table)
    {
      free_state (result);
      result = state_in_table;
    }
  return result;
}

/* Initialization of the abstract data. */
static void
initiate_states ()
{
  struct decl *curr_decl;
  int i;

  VLA_PTR_CREATE (units_container, description->units_num, "units_container");
  units_array = VLA_PTR_BEGIN (units_container);
  for (i = 0; i < description->decls_num; i++)
    {
      curr_decl = description->decls [i];
      if (curr_decl->mode == dm_unit)
	units_array [curr_decl->decl.unit.unit_num] = &curr_decl->decl.unit;
    }
  max_cycles_num = description->max_insn_reserv_cycles;
  if (max_cycles_num == 0)
    max_cycles_num++;
  els_in_cycle_reserv
    = ((description->units_num + sizeof (set_el_t) * CHAR_BIT - 1)
       / (sizeof (set_el_t) * CHAR_BIT));
  els_in_reservs = els_in_cycle_reserv * max_cycles_num;
  curr_unique_state_num = 0;
  initiate_alt_states ();
  VLA_PTR_CREATE (free_states, 1500, "free states");
  state_table = htab_create (1500, state_hash, state_eq_p, (htab_del) 0);
  empty_reserv = alloc_empty_reserv_sets ();
}

/* Finisging work with the abstract data. */
static void
finish_states ()
{
  VLA_PTR_DELETE (units_container);
  htab_delete (state_table);
  VLA_PTR_DELETE (free_states);
  finish_alt_states ();
}



/* Abstract data `arcs'. */

/* List of free arcs. */
static struct arc *first_free_arc;

#ifndef NDEBUG
/* The following variables is maximal number of allocated nodes
   `arc'. */
static int allocated_arcs_num = 0;
#endif

/* The function frees node ARC. */
static void
free_arc (arc)
     struct arc *arc;
{
  arc->next_out_arc = first_free_arc;
  first_free_arc = arc;
}

/* The function removes and frees ARC staring from FROM_STATE. */
static void
remove_arc (from_state, arc)
     struct state *from_state;
     struct arc *arc;
{
  struct arc *prev_arc;
  struct arc *curr_arc;

  assert (arc != NULL);
  for (prev_arc = NULL, curr_arc = from_state->first_out_arc;
       curr_arc != NULL;
       prev_arc = curr_arc, curr_arc = curr_arc->next_out_arc)
    if (curr_arc == arc)
      break;
  assert (curr_arc != NULL);
  if (prev_arc == NULL)
    from_state->first_out_arc = arc->next_out_arc;
  else
    prev_arc->next_out_arc = arc->next_out_arc;
  free_arc (arc);
}

/* The functions returns arc with given characteristics (or NULL if
   the arc does not exist). */
static struct arc *
find_arc (from_state, to_state, insn)
     struct state *from_state;
     struct state *to_state;
     struct ainsn *insn;
{
  struct arc *curr_arc;

  for (curr_arc = first_out_arc (from_state);
       curr_arc != NULL;
       curr_arc = next_out_arc (curr_arc))
    {
      if (curr_arc->to_state == to_state && curr_arc->insn == insn)
        return curr_arc;
    }
  return NULL;
}

/* The function adds arc from FROM_STATE to TO_STATE marked by AINSN
   and with given STATE_ALTS.  The function returns added arc (or
   already existing arc). */
static struct arc *
add_arc (from_state, to_state, ainsn, state_alts)
     struct state *from_state;
     struct state *to_state;
     struct ainsn *ainsn;
     int state_alts;
{
  struct arc *new_arc;

  new_arc = find_arc (from_state, to_state, ainsn);
  if (new_arc != NULL)
    return new_arc;
  if (first_free_arc == NULL)
    {
#ifndef NDEBUG
      allocated_arcs_num++;
#endif
      new_arc = create_node (sizeof (struct arc));
      new_arc->to_state = NULL;
      new_arc->insn = NULL;
      new_arc->next_out_arc = NULL;
    }
  else
    {
      new_arc = first_free_arc;
      first_free_arc =  first_free_arc->next_out_arc;
    }
  new_arc->to_state = to_state;
  new_arc->insn = ainsn;
  new_arc->next_out_arc = from_state->first_out_arc;
  from_state->first_out_arc = new_arc;
  new_arc->next_arc_marked_by_insn = NULL;
  new_arc->state_alts = state_alts;
  return new_arc;
}

/* The function returns the first arc starting from STATE. */
static struct arc *
first_out_arc (state)
     struct state *state;
{
  return state->first_out_arc;
}

/* The function returns next out arc after ARC. */
static struct arc *
next_out_arc (arc)
     struct arc *arc;
{
  return arc->next_out_arc;
}

/* Initialization of the abstract data. */
static void
initiate_arcs ()
{
  first_free_arc = NULL;
}

/* Finishing work with the abstract data. */
static void
finish_arcs ()
{
}



/* The page contains abstract data for work with exclusion sets (see
   exclusion_set in file rtl.def). */

/* The following variable refers to an exclusion set returned by
   get_excl_set.  This is bit string of length equal to cpu units
   number.  If exclusion set for given unit contains 1 for a unit,
   then simultaneous reservation of the units is prohibited. */
static reserv_sets_t excl_set;

/* The array contains exclusion sets for each unit. */
static reserv_sets_t *unit_excl_set_table;

/* The following function forms the array containing exclusion sets
   for each unit. */
static void
initiate_excl_sets ()
{
  struct decl *curr_decl;
  reserv_sets_t unit_excl_set;
  struct unit_set_el *curr_unit_set_el;
  int i;

  IR_TOP_EXPAND (els_in_cycle_reserv * sizeof (set_el_t));
  excl_set = (reserv_sets_t) IR_TOP_BEGIN ();
  IR_TOP_FINISH ();
  IR_TOP_EXPAND (description->units_num * sizeof (reserv_sets_t));
  unit_excl_set_table = (reserv_sets_t *) IR_TOP_BEGIN ();
  IR_TOP_FINISH ();
  /* Evaluate unit exclusion sets. */
  for (i = 0; i < description->decls_num; i++)
    {
      curr_decl = description->decls [i];
      if (curr_decl->mode == dm_unit)
	{
	  IR_TOP_EXPAND (els_in_cycle_reserv * sizeof (set_el_t));
	  unit_excl_set = (reserv_sets_t) IR_TOP_BEGIN ();
	  IR_TOP_FINISH ();
	  memset (unit_excl_set, 0, els_in_cycle_reserv * sizeof (set_el_t));
	  for (curr_unit_set_el = curr_decl->decl.unit.excl_list;
	       curr_unit_set_el != NULL;
	       curr_unit_set_el = curr_unit_set_el->next_unit_set_el)
            SET_BIT (unit_excl_set, curr_unit_set_el->unit_decl->unit_num);
          unit_excl_set_table [curr_decl->decl.unit.unit_num] = unit_excl_set;
        }
    }
}

/* The function sets up and return EXCL_SET which is union of
   exclusion sets for each unit in IN_SET. */
static reserv_sets_t
get_excl_set (in_set)
     reserv_sets_t in_set;
{
  int excl_char_num;
  int chars_num;
  int i;
  int unit_num;
  int curr_unit_num;

  chars_num = els_in_cycle_reserv * sizeof (set_el_t);
  memset (excl_set, 0, chars_num);
  for (excl_char_num = 0; excl_char_num < chars_num; excl_char_num++)
    if (((unsigned char *) in_set) [excl_char_num])
      for (i = CHAR_BIT - 1; i >= 0; i--)
	if ((((unsigned char *) in_set) [excl_char_num] >> i) & 1)
	  {
	    unit_num = excl_char_num * CHAR_BIT + i;
	    if (unit_num >= description->units_num)
	      return excl_set;
	    for (curr_unit_num = 0;
                 curr_unit_num < els_in_cycle_reserv;
                 curr_unit_num++)
	      {
		excl_set [curr_unit_num]
                  |= unit_excl_set_table [unit_num] [curr_unit_num];
	      }
	  }
  return excl_set;
}



/* The page contains abstract data for work with require sets (see
   require_set in file rtl.def). */

/* The following variable refers to an require set returned by
   get_req_set.  This is bit string of length equal to cpu units
   number.  If require set for given unit contains 1 for a unit, then
   reservation of the unit is possibly only if given unit is reserved
   already. */
static reserv_sets_t req_set;

/* The array contains require sets for each unit. */
static reserv_sets_t *unit_req_set_table;

/* The following function forms the array containing require sets
   for each unit */
static void
initiate_req_sets ()
{
  struct decl *curr_decl;
  reserv_sets_t unit_req_set;
  struct unit_set_el *curr_unit_set_el;
  int i;

  IR_TOP_EXPAND (els_in_cycle_reserv * sizeof (set_el_t));
  req_set = (reserv_sets_t) IR_TOP_BEGIN ();
  IR_TOP_FINISH ();
  IR_TOP_EXPAND (description->units_num * sizeof (reserv_sets_t));
  unit_req_set_table = (reserv_sets_t *) IR_TOP_BEGIN ();
  IR_TOP_FINISH ();
  /* Evaluate unit require sets. */
  for (i = 0; i < description->decls_num; i++)
    {
      curr_decl = description->decls [i];
      if (curr_decl->mode == dm_unit)
	{
	  IR_TOP_EXPAND (els_in_cycle_reserv * sizeof (set_el_t));
	  unit_req_set = (reserv_sets_t) IR_TOP_BEGIN ();
	  IR_TOP_FINISH ();
	  memset (unit_req_set, 0, els_in_cycle_reserv * sizeof (set_el_t));
	  for (curr_unit_set_el = curr_decl->decl.unit.req_list;
	       curr_unit_set_el != NULL;
	       curr_unit_set_el = curr_unit_set_el->next_unit_set_el)
            SET_BIT (unit_req_set, curr_unit_set_el->unit_decl->unit_num);
          unit_req_set_table [curr_decl->decl.unit.unit_num] = unit_req_set;
        }
    }
}

/* The function sets up and return REQ_SET which is union of require
   sets for each unit in IN_SET. */
static reserv_sets_t
get_req_set (in_set)
     reserv_sets_t in_set;
{
  int req_char_num;
  int chars_num;
  int i;
  int unit_num;
  int curr_unit_num;

  chars_num = els_in_cycle_reserv * sizeof (set_el_t);
  memset (req_set, 0, chars_num);
  for (req_char_num = 0; req_char_num < chars_num; req_char_num++)
    if (((unsigned char *) in_set) [req_char_num])
      for (i = CHAR_BIT - 1; i >= 0; i--)
	if ((((unsigned char *) in_set) [req_char_num] >> i) & 1)
	  {
	    unit_num = req_char_num * CHAR_BIT + i;
	    if (unit_num >= description->units_num)
	      return req_set;
	    for (curr_unit_num = 0;
                 curr_unit_num < els_in_cycle_reserv;
                 curr_unit_num++)
	      {
		req_set [curr_unit_num]
		  |= unit_req_set_table [unit_num] [curr_unit_num];
	      }
	  }
  return req_set;
}



/* This page contains code for transformation of original reservations
   described in .md file.  The main goal of transformations is
   simplifying reservation and lifting up all `|' on the top of IR
   reservation representation. */


/* The following function makes copy of IR representation of
   reservation.  The function also substitutes all reservations
   defined by define_reservation by corresponding value during making
   the copy. */
static struct regexp *
copy_insn_regexp (regexp)
     struct regexp *regexp;
{
  struct regexp * result;
  int i;

  if (regexp->mode == rm_reserv)
    result = copy_insn_regexp (regexp->regexp.reserv.reserv_decl->regexp);
  else if (regexp->mode == rm_unit)
    result = copy_node (regexp, sizeof (struct regexp));
  else if (regexp->mode == rm_repeat)
    {
      result = copy_node (regexp, sizeof (struct regexp));
      result->regexp.repeat.regexp
        = copy_insn_regexp (regexp->regexp.repeat.regexp);
    }
  else if (regexp->mode == rm_sequence)
    {
      result = copy_node (regexp,
                          sizeof (struct regexp) + sizeof (struct regexp *)
			  * (regexp->regexp.sequence.regexps_num - 1));
      for (i = 0; i < regexp->regexp.sequence.regexps_num; i++)
	result->regexp.sequence.regexps [i]
	  = copy_insn_regexp (regexp->regexp.sequence.regexps [i]);
    }
  else if (regexp->mode == rm_allof)
    {
      result = copy_node (regexp,
                          sizeof (struct regexp) + sizeof (struct regexp *)
			  * (regexp->regexp.allof.regexps_num - 1));
      for (i = 0; i < regexp->regexp.allof.regexps_num; i++)
	result->regexp.allof.regexps [i]
	  = copy_insn_regexp (regexp->regexp.allof.regexps [i]);
    }
  else if (regexp->mode == rm_oneof)
    {
      result = copy_node (regexp,
                          sizeof (struct regexp) + sizeof (struct regexp *)
			  * (regexp->regexp.oneof.regexps_num - 1));
      for (i = 0; i < regexp->regexp.oneof.regexps_num; i++)
	result->regexp.oneof.regexps [i]
	  = copy_insn_regexp (regexp->regexp.oneof.regexps [i]);
    }
  else
    {
      assert (regexp->mode == rm_nothing);
      result = copy_node (regexp, sizeof (struct regexp));
    }
  return result;
}

/* The following variable is set up TRUE if a transformation has been
   applied. */
static int regexp_transformed_p;

/* The function makes transformation
   A*N -> A, A, ... */
static struct regexp *
transform_1 (regexp)
     struct regexp *regexp;
{
  int curr_regexp_num;
  int repeat_num;
  struct regexp *operand;
  pos_t pos;

  if (regexp->mode == rm_repeat)
    {
      repeat_num = regexp->regexp.repeat.repeat_num;
      assert (repeat_num > 1);
      operand = regexp->regexp.repeat.regexp;
      pos = regexp->mode;
      regexp = create_node (sizeof (struct regexp) + sizeof (struct regexp *)
			    * (repeat_num - 1));
      regexp->mode = rm_sequence;
      regexp->pos = pos;
      regexp->regexp.sequence.regexps_num = repeat_num;
      for (curr_regexp_num = 0;
	   curr_regexp_num < repeat_num;
	   curr_regexp_num++)
	regexp->regexp.sequence.regexps [curr_regexp_num]
          = copy_insn_regexp (operand);
      regexp_transformed_p = TRUE;
    }
  return regexp;
}

/* The function makes transformations
   ...,(A,B,...),C,... -> ...,A,B,...,C,...
   ...+(A+B+...)+C+... -> ...+A+B+...+C+...
   ...|(A|B|...)|C|... -> ...|A|B|...|C|... */
static struct regexp *
transform_2 (regexp)
     struct regexp *regexp;
{
  if (regexp->mode == rm_sequence)
    {
      struct regexp *sequence;
      struct regexp *result;
      int sequence_index;
      int i, j;

      for (i = 0; i < regexp->regexp.sequence.regexps_num; i++)
	if (regexp->regexp.sequence.regexps [i]->mode == rm_sequence)
	  {
	    sequence_index = i;
	    sequence = regexp->regexp.sequence.regexps [i];
	    break;
	  }
      if (i < regexp->regexp.sequence.regexps_num)
	{
	  assert (sequence->regexp.sequence.regexps_num > 1
		  && regexp->regexp.sequence.regexps_num > 1);
	  result = create_node (sizeof (struct regexp)
                                + sizeof (struct regexp *)
				* (regexp->regexp.sequence.regexps_num
                                   + sequence->regexp.sequence.regexps_num
                                   - 2));
	  result->mode = rm_sequence;
	  result->pos = regexp->pos;
	  result->regexp.sequence.regexps_num
            = (regexp->regexp.sequence.regexps_num
               + sequence->regexp.sequence.regexps_num - 1);
	  for (i = 0; i < regexp->regexp.sequence.regexps_num; i++)
            if (i < sequence_index)
              result->regexp.sequence.regexps [i]
                = copy_insn_regexp (regexp->regexp.sequence.regexps [i]);
            else if (i > sequence_index)
              result->regexp.sequence.regexps
                [i + sequence->regexp.sequence.regexps_num - 1]
                = copy_insn_regexp (regexp->regexp.sequence.regexps [i]);
            else
              for (j = 0; j < sequence->regexp.sequence.regexps_num; j++)
                result->regexp.sequence.regexps [i + j]
                  = copy_insn_regexp (sequence->regexp.sequence.regexps [j]);
	  regexp_transformed_p = TRUE;
	  regexp = result;
	}
    }
  else if (regexp->mode == rm_allof)
    {
      struct regexp *allof;
      struct regexp *result;
      int allof_index;
      int i, j;

      for (i = 0; i < regexp->regexp.allof.regexps_num; i++)
	if (regexp->regexp.allof.regexps [i]->mode == rm_allof)
	  {
	    allof_index = i;
	    allof = regexp->regexp.allof.regexps [i];
	    break;
	  }
      if (i < regexp->regexp.allof.regexps_num)
	{
	  assert (allof->regexp.allof.regexps_num > 1
		  && regexp->regexp.allof.regexps_num > 1);
	  result = create_node (sizeof (struct regexp)
                                + sizeof (struct regexp *)
				* (regexp->regexp.allof.regexps_num
                                   + allof->regexp.allof.regexps_num - 2));
	  result->mode = rm_allof;
	  result->pos = regexp->pos;
	  result->regexp.allof.regexps_num
            = (regexp->regexp.allof.regexps_num
               + allof->regexp.allof.regexps_num - 1);
	  for (i = 0; i < regexp->regexp.allof.regexps_num; i++)
            if (i < allof_index)
              result->regexp.allof.regexps [i]
                = copy_insn_regexp (regexp->regexp.allof.regexps [i]);
            else if (i > allof_index)
              result->regexp.allof.regexps
                [i + allof->regexp.allof.regexps_num - 1]
                = copy_insn_regexp (regexp->regexp.allof.regexps [i]);
            else
              for (j = 0; j < allof->regexp.allof.regexps_num; j++)
                result->regexp.allof.regexps [i + j]
                  = copy_insn_regexp (allof->regexp.allof.regexps [j]);
	  regexp_transformed_p = TRUE;
	  regexp = result;
	}
    }
  else if (regexp->mode == rm_oneof)
    {
      struct regexp *oneof;
      struct regexp *result;
      int oneof_index;
      int i, j;

      for (i = 0; i < regexp->regexp.oneof.regexps_num; i++)
	if (regexp->regexp.oneof.regexps [i]->mode == rm_oneof)
	  {
	    oneof_index = i;
	    oneof = regexp->regexp.oneof.regexps [i];
	    break;
	  }
      if (i < regexp->regexp.oneof.regexps_num)
	{
	  assert (oneof->regexp.oneof.regexps_num > 1
		  && regexp->regexp.oneof.regexps_num > 1);
	  result = create_node (sizeof (struct regexp)
				+ sizeof (struct regexp *)
				* (regexp->regexp.oneof.regexps_num
                                   + oneof->regexp.oneof.regexps_num - 2));
	  result->mode = rm_oneof;
	  result->pos = regexp->pos;
	  result->regexp.oneof.regexps_num
	    = (regexp->regexp.oneof.regexps_num
               + oneof->regexp.oneof.regexps_num - 1);
	  for (i = 0; i < regexp->regexp.oneof.regexps_num; i++)
            if (i < oneof_index)
              result->regexp.oneof.regexps [i]
                = copy_insn_regexp (regexp->regexp.oneof.regexps [i]);
            else if (i > oneof_index)
              result->regexp.oneof.regexps
                [i + oneof->regexp.oneof.regexps_num - 1]
                = copy_insn_regexp (regexp->regexp.oneof.regexps [i]);
            else
              for (j = 0; j < oneof->regexp.oneof.regexps_num; j++)
                result->regexp.oneof.regexps [i + j]
                  = copy_insn_regexp (oneof->regexp.oneof.regexps [j]);
	  regexp_transformed_p = TRUE;
	  regexp = result;
	}
    }
  return regexp;
}

/* The function makes transformations
   ...,A|B|...,C,... -> (...,A,C,...)|(...,B,C,...)|...
   ...+(A|B|...)+C+... -> (...+A+C+...)|(...+B+C+...)|... */
static struct regexp *
transform_3 (regexp)
     struct regexp *regexp;
{
  if (regexp->mode == rm_sequence)
    {
      struct regexp *oneof;
      int oneof_index;
      struct regexp *result;
      struct regexp *sequence;
      int i, j;

      for (i = 0; i < regexp->regexp.sequence.regexps_num; i++)
	if (regexp->regexp.sequence.regexps [i]->mode == rm_oneof)
	  {
	    oneof_index = i;
	    oneof = regexp->regexp.sequence.regexps [i];
	    break;
	  }
      if (i < regexp->regexp.sequence.regexps_num)
	{
	  assert (oneof->regexp.oneof.regexps_num > 1
		  && regexp->regexp.sequence.regexps_num > 1);
	  result = create_node (sizeof (struct regexp)
				+ sizeof (struct regexp *)
				* (oneof->regexp.oneof.regexps_num - 1));
	  result->mode = rm_oneof;
	  result->pos = regexp->pos;
	  result->regexp.oneof.regexps_num = oneof->regexp.oneof.regexps_num;
	  for (i = 0; i < result->regexp.oneof.regexps_num; i++)
	    {
	      sequence
                = create_node (sizeof (struct regexp)
                               + sizeof (struct regexp *)
                               * (regexp->regexp.sequence.regexps_num - 1));
	      sequence->mode = rm_sequence;
	      sequence->pos = regexp->pos;
	      sequence->regexp.sequence.regexps_num
                = regexp->regexp.sequence.regexps_num;
              result->regexp.oneof.regexps [i] = sequence;
	      for (j = 0; j < sequence->regexp.sequence.regexps_num; j++)
		if (j != oneof_index)
		  sequence->regexp.sequence.regexps [j]
		    = copy_insn_regexp (regexp->regexp.sequence.regexps [j]);
		else
		  sequence->regexp.sequence.regexps [j]
		    = copy_insn_regexp (oneof->regexp.oneof.regexps [i]);
	    }
	  regexp_transformed_p = TRUE;
	  regexp = result;
	}
    }
  else if (regexp->mode == rm_allof)
    {
      struct regexp *oneof;
      int oneof_index;
      struct regexp *result;
      struct regexp *allof;
      int i, j;

      for (i = 0; i < regexp->regexp.allof.regexps_num; i++)
	if (regexp->regexp.allof.regexps [i]->mode == rm_oneof)
	  {
	    oneof_index = i;
	    oneof = regexp->regexp.allof.regexps [i];
	    break;
	  }
      if (i < regexp->regexp.allof.regexps_num)
	{
	  assert (oneof->regexp.oneof.regexps_num > 1
		  && regexp->regexp.allof.regexps_num > 1);
	  result = create_node (sizeof (struct regexp)
				+ sizeof (struct regexp *)
				* (oneof->regexp.oneof.regexps_num - 1));
	  result->mode = rm_oneof;
	  result->pos = regexp->pos;
	  result->regexp.oneof.regexps_num = oneof->regexp.oneof.regexps_num;
	  for (i = 0; i < result->regexp.oneof.regexps_num; i++)
	    {
	      allof
		= create_node (sizeof (struct regexp)
                               + sizeof (struct regexp *)
			       * (regexp->regexp.allof.regexps_num - 1));
	      allof->mode = rm_allof;
	      allof->pos = regexp->pos;
	      allof->regexp.allof.regexps_num
                = regexp->regexp.allof.regexps_num;
              result->regexp.oneof.regexps [i] = allof;
	      for (j = 0; j < allof->regexp.allof.regexps_num; j++)
		if (j != oneof_index)
		  allof->regexp.allof.regexps [j]
		    = copy_insn_regexp (regexp->regexp.allof.regexps [j]);
		else
		  allof->regexp.allof.regexps [j]
		    = copy_insn_regexp (oneof->regexp.oneof.regexps [i]);
	    }
	  regexp_transformed_p = TRUE;
	  regexp = result;
	}
    }
  return regexp;
}

/* The function traverses IR of reservation and applies transformations
   implemented by FUNC. */
static struct regexp *
regexp_transform_func (regexp, func)
     struct regexp *regexp;
     struct regexp *(*func) (struct regexp *regexp);
{
  int i;

  if (regexp->mode == rm_sequence)
    for (i = 0; i < regexp->regexp.sequence.regexps_num; i++)
      regexp->regexp.sequence.regexps [i]
	= regexp_transform_func (regexp->regexp.sequence.regexps [i], func);
  else if (regexp->mode == rm_allof)
    for (i = 0; i < regexp->regexp.allof.regexps_num; i++)
      regexp->regexp.allof.regexps [i]
	= regexp_transform_func (regexp->regexp.allof.regexps [i], func);
  else if (regexp->mode == rm_oneof)
    for (i = 0; i < regexp->regexp.oneof.regexps_num; i++)
      regexp->regexp.oneof.regexps [i]
	= regexp_transform_func (regexp->regexp.oneof.regexps [i], func);
  else if (regexp->mode == rm_repeat)
    regexp->regexp.repeat.regexp
      = regexp_transform_func (regexp->regexp.repeat.regexp, func);
  else
    assert (regexp->mode == rm_nothing || regexp->mode == rm_unit);
  return (*func) (regexp);
}

/* The function applies all transformations for IR representation of
   reservation REGEXP. */
static struct regexp *
transform_regexp (regexp)
     struct regexp *regexp;
{
  regexp = regexp_transform_func (regexp, transform_1);  
  do
    {
      regexp_transformed_p = FALSE;
      regexp = regexp_transform_func (regexp, transform_2);
      regexp = regexp_transform_func (regexp, transform_3);
    }
  while (regexp_transformed_p);
  return regexp;
}

/* The function applys all transformations for reservations of all
   insn declarations. */
static void
transform_insn_regexps ()
{
  struct decl *curr_decl;
  int i;

  for (i = 0; i < description->decls_num; i++)
    {
      curr_decl = description->decls [i];
      if (curr_decl->mode == dm_insn_reserv
          && curr_decl != advance_cycle_insn_decl)
	curr_decl->decl.insn_reserv.transformed_regexp
	  = transform_regexp (copy_insn_regexp
                              (curr_decl->decl.insn_reserv.regexp));
    }
}



/* The page contains code for building alt_states (see comments for
   IR) describing all possible insns reservations of an automaton. */

/* Current state being formed for which the current alt_state
   refers. */
static struct state *state_being_formed;

/* Current alt_state being formed. */
static struct alt_state *alt_state_being_formed;
 
/* This recursive function processes `,' and units in reservation
   REGEXP for forming alt_states of AUTOMATON.  It is believed that
   CURR_CYCLE_NUM is start cycle of all reservation REGEXP. */
static int
process_seq_for_forming_states (regexp, automaton, curr_cycle_num)
     struct regexp *regexp;
     struct automaton *automaton;
     int curr_cycle_num;
{
  int i;

  if (regexp == NULL)
    return curr_cycle_num;
  else if (regexp->mode == rm_unit)
    {
      if (regexp->regexp.unit.unit_decl->corresponding_automaton_num
          == automaton->automaton_order_num)
        set_state_reserv (state_being_formed, curr_cycle_num,
                          regexp->regexp.unit.unit_decl->unit_num);
      return curr_cycle_num;
    }
  else if (regexp->mode == rm_sequence)
    {
      for (i = 0; i < regexp->regexp.sequence.regexps_num; i++)
	curr_cycle_num
	  = process_seq_for_forming_states
            (regexp->regexp.sequence.regexps [i],
             automaton, curr_cycle_num) + 1;
      return curr_cycle_num;
    }
  else if (regexp->mode == rm_allof)
    {
      int finish_cycle = 0;
      int cycle;

      for (i = 0; i < regexp->regexp.allof.regexps_num; i++)
	{
	  cycle
            = process_seq_for_forming_states (regexp->regexp.allof.regexps [i],
                                              automaton, curr_cycle_num);
	  if (finish_cycle < cycle)
	    finish_cycle = cycle;
	}
      return finish_cycle;
    }
  else
    {
      assert (regexp->mode == rm_nothing);
      return curr_cycle_num;
    }
}

/* This recursive function finishes forming ALT_STATE of AUTOMATON and
   inserts alt_state into the table. */
static void
finish_forming_alt_state (alt_state, automaton)
     struct alt_state *alt_state;
     struct automaton *automaton ATTRIBUTE_UNUSED;
{
  struct state *state_in_table;
  struct state *corresponding_state;

  corresponding_state = alt_state->state;
  state_in_table = insert_state (corresponding_state);
  if (state_in_table != corresponding_state)
    {
      free_state (corresponding_state);
      alt_state->state = state_in_table;
    }
}

/* The following variable value is current automaton insn for whose
   reservation the alt states are created. */
static struct ainsn *curr_ainsn;

/* This recursive function processes `|' in reservation REGEXP for
   forming alt_states of AUTOMATON.  List of the alt states should
   have the same order as in the description. */
static void
process_alts_for_forming_states (regexp, automaton, inside_oneof_p)
     struct regexp *regexp;
     struct automaton *automaton;
     int inside_oneof_p;
{
  int i;

  if (regexp->mode != rm_oneof)
    {
      alt_state_being_formed = get_free_alt_state ();
      state_being_formed = get_free_state (TRUE, automaton);
      alt_state_being_formed->state = state_being_formed;
      /* We inserts in reverse order but we process alternatives also
         in reverse order.  So we have the same order of alternative
         as in the description. */
      alt_state_being_formed->next_alt_state = curr_ainsn->alt_states;
      curr_ainsn->alt_states = alt_state_being_formed;
      (void) process_seq_for_forming_states (regexp, automaton, 0);
      finish_forming_alt_state (alt_state_being_formed, automaton);
    }
  else
    {
      if (inside_oneof_p)
	abort ();
      /* We processes it in reverse order to get list with the same
	 order as in the description.  See also the previous
	 commentary. */
      for (i = regexp->regexp.oneof.regexps_num - 1; i >= 0; i--)
	process_alts_for_forming_states (regexp->regexp.oneof.regexps [i],
					 automaton, TRUE);
    }
}

/* Create nodes alt_state for all AUTOMATON insns. */
static void
create_alt_states (automaton)
     struct automaton *automaton;
{
  struct insn_reserv_decl *curr_insn_reserv_decl;

  for (curr_ainsn = automaton->ainsn_list;
       curr_ainsn != NULL;
       curr_ainsn = curr_ainsn->next_ainsn)
    {
      curr_insn_reserv_decl = curr_ainsn->insn_reserv_decl;
      if (curr_insn_reserv_decl != &advance_cycle_insn_decl->decl.insn_reserv)
        {
          curr_ainsn->alt_states = NULL;
          process_alts_for_forming_states
            (curr_insn_reserv_decl->transformed_regexp, automaton, FALSE);
          curr_ainsn->sorted_alt_states
            = uniq_sort_alt_states (curr_ainsn->alt_states);
        }
    }
}



/* The page contains major code for building DFA(s) for fast pipeline
   hazards recognition. */

/* The function forms list of ainsns of AUTOMATON with the same
   reservation. */
static void
form_ainsn_with_same_reservs (automaton)
     struct automaton *automaton;
{
  struct ainsn *curr_ainsn;
  size_t curr_index;
  vla_ptr_t first_insns_with_same_reservs;
  vla_ptr_t last_insns_with_same_reservs;

  VLA_PTR_CREATE (first_insns_with_same_reservs, 150,
		  "first insns with the same reservs");
  VLA_PTR_CREATE (last_insns_with_same_reservs, 150,
		  "last insns with the same reservs");
  for (curr_ainsn = automaton->ainsn_list;
       curr_ainsn != NULL;
       curr_ainsn = curr_ainsn->next_ainsn)
    if (curr_ainsn->insn_reserv_decl
        == &advance_cycle_insn_decl->decl.insn_reserv)
      {
        curr_ainsn->next_same_reservs_insn = NULL;
        curr_ainsn->first_insn_with_same_reservs = TRUE;
      }
    else
      {
        for (curr_index = 0;
             curr_index < VLA_PTR_LENGTH (first_insns_with_same_reservs);
             curr_index++)
          if (alt_states_eq
              (curr_ainsn->sorted_alt_states,
               ((struct ainsn *) VLA_PTR (first_insns_with_same_reservs,
                                          curr_index))->sorted_alt_states))
            break;
        curr_ainsn->next_same_reservs_insn = NULL;
        if (curr_index < VLA_PTR_LENGTH (first_insns_with_same_reservs))
          {
            curr_ainsn->first_insn_with_same_reservs = FALSE;
	    ((struct ainsn *) VLA_PTR (last_insns_with_same_reservs,
                                       curr_index))->next_same_reservs_insn
	      = curr_ainsn;
            VLA_PTR (last_insns_with_same_reservs, curr_index) = curr_ainsn;
          }
        else
          {
            VLA_PTR_ADD (first_insns_with_same_reservs, curr_ainsn);
            VLA_PTR_ADD (last_insns_with_same_reservs, curr_ainsn);
            curr_ainsn->first_insn_with_same_reservs = TRUE;
          }
      }
  VLA_PTR_DELETE (first_insns_with_same_reservs);
  VLA_PTR_DELETE (last_insns_with_same_reservs);
}

/* The following function creates all states of nondeterministic (if
   NDFA_FLAG is TRUE) or deterministic AUTOMATON. */
static void
make_automaton (automaton)
     struct automaton *automaton;
{
  struct ainsn *curr_ainsn;
  struct insn_reserv_decl *curr_insn_reserv_decl;
  struct alt_state *curr_alt_state;
  struct state *state;
  struct state *start_state;
  struct state *state2;
  struct ainsn *advance_cycle_ainsn;
  struct arc *added_arc;
  vla_ptr_t state_stack;

  VLA_PTR_CREATE (state_stack, 150, "state stack");
  /* Create the start state (empty state). */
  start_state = insert_state (get_free_state (TRUE, automaton));
  automaton->start_state = start_state;
  start_state->it_was_placed_in_stack_for_NDFA_forming = TRUE;
  VLA_PTR_ADD (state_stack, start_state);
  while (VLA_PTR_LENGTH (state_stack) != 0)
    {
      state = VLA_PTR (state_stack, VLA_PTR_LENGTH (state_stack) - 1);
      VLA_PTR_SHORTEN (state_stack, 1);
      advance_cycle_ainsn = NULL;
      for (curr_ainsn = automaton->ainsn_list;
           curr_ainsn != NULL;
           curr_ainsn = curr_ainsn->next_ainsn)
        if (curr_ainsn->first_insn_with_same_reservs)
          {
            curr_insn_reserv_decl = curr_ainsn->insn_reserv_decl;
            if (curr_insn_reserv_decl
		!= &advance_cycle_insn_decl->decl.insn_reserv)
              {
		/* We process alt_states in the same order as they are
                   present in the description. */
		added_arc = NULL;
                for (curr_alt_state = curr_ainsn->alt_states;
                     curr_alt_state != NULL;
                     curr_alt_state = curr_alt_state->next_alt_state)
                  {
                    state2 = curr_alt_state->state;
                    if (!intersected_state_reservs_p (state, state2))
                      {
                        state2 = states_union (state, state2);
                        if (!state2->it_was_placed_in_stack_for_NDFA_forming)
                          {
                            state2->it_was_placed_in_stack_for_NDFA_forming
			      = TRUE;
                            VLA_PTR_ADD (state_stack, state2);
                          }
			added_arc = add_arc (state, state2, curr_ainsn, 1);
			if (!ndfa_flag)
			  break;
                      }
                  }
		if (!ndfa_flag && added_arc != NULL)
		  {
		    added_arc->state_alts = 0;
		    for (curr_alt_state = curr_ainsn->alt_states;
			 curr_alt_state != NULL;
			 curr_alt_state = curr_alt_state->next_alt_state)
		      {
			state2 = curr_alt_state->state;
			if (!intersected_state_reservs_p (state, state2))
			  added_arc->state_alts++;
		      }
		  }
              }
            else
              advance_cycle_ainsn = curr_ainsn;
          }
      /* Add transition to advance cycle. */
      state2 = state_shift (state);
      if (!state2->it_was_placed_in_stack_for_NDFA_forming)
        {
          state2->it_was_placed_in_stack_for_NDFA_forming = TRUE;
          VLA_PTR_ADD (state_stack, state2);
        }
      assert (advance_cycle_ainsn != NULL);
      add_arc (state, state2, advance_cycle_ainsn, 1);
    }
  VLA_PTR_DELETE (state_stack);
}

/* Foms lists of all arcs of STATE marked by the same ainsn. */
static void
form_arcs_marked_by_insn (state)
     struct state *state;
{
  struct decl *curr_decl;
  struct arc *curr_arc;
  int i;

  for (i = 0; i < description->decls_num; i++)
    {
      curr_decl = description->decls [i];
      if (curr_decl->mode == dm_insn_reserv)
	curr_decl->decl.insn_reserv.arcs_marked_by_insn = NULL;
    }
  for (curr_arc = first_out_arc (state);
       curr_arc != NULL;
       curr_arc = next_out_arc (curr_arc))
    {
      assert (curr_arc->insn != NULL);
      curr_arc->next_arc_marked_by_insn
	= curr_arc->insn->insn_reserv_decl->arcs_marked_by_insn;
      curr_arc->insn->insn_reserv_decl->arcs_marked_by_insn = curr_arc;
    }
}

/* The function creates composed state (see comments for IR) from
   ORIGINAL_STATE and list of arcs ARCS_MARKED_BY_INSN marked by the
   same insn.  If the composed state is not in STATE_STACK yet, it is
   popped to STATE_STACK. */
static void
create_composed_state (original_state, arcs_marked_by_insn, state_stack)
     struct state *original_state;
     struct arc *arcs_marked_by_insn;
     vla_ptr_t *state_stack;
{
  struct state *curr_state;
  struct alt_state *curr_alt_state;
  struct alt_state *new_alt_state;
  struct arc *curr_arc;
  struct arc *next_arc;
  struct state *state_in_table;
  struct state *temp_state;
  struct alt_state *canonical_alt_states_list;
  int alts_number;

  if (arcs_marked_by_insn == NULL)
    return;
  if (arcs_marked_by_insn->next_arc_marked_by_insn == NULL)
    curr_state = arcs_marked_by_insn->to_state;
  else
    {
      assert (ndfa_flag);
      /* Create composed state. */
      curr_state = get_free_state (FALSE,
                                   arcs_marked_by_insn->to_state->automaton);
      curr_alt_state = NULL;
      for (curr_arc = arcs_marked_by_insn;
           curr_arc != NULL;
           curr_arc = curr_arc->next_arc_marked_by_insn)
        {
          new_alt_state = get_free_alt_state ();
          new_alt_state->next_alt_state = curr_alt_state;
          new_alt_state->state = curr_arc->to_state;
	  assert (curr_arc->to_state->component_states == NULL);
          curr_alt_state = new_alt_state;
        }
      /* There are not identical sets in the alt state list. */
      canonical_alt_states_list = uniq_sort_alt_states (curr_alt_state);
      if (canonical_alt_states_list->next_sorted_alt_state == NULL)
        {
          temp_state = curr_state;
          curr_state = canonical_alt_states_list->state;
          free_state (temp_state);
        }
      else
        {
          curr_state->component_states = canonical_alt_states_list;
          state_in_table = insert_state (curr_state);
          if (state_in_table != curr_state)
            {
              assert (state_in_table->it_was_placed_in_stack_for_DFA_forming);
              free_state (curr_state);
              curr_state = state_in_table;
            }
          else
            {
              assert (!curr_state->it_was_placed_in_stack_for_DFA_forming);
              for (curr_alt_state = curr_state->component_states;
                   curr_alt_state != NULL;
                   curr_alt_state = curr_alt_state->next_sorted_alt_state)
                for (curr_arc = first_out_arc (curr_alt_state->state);
                     curr_arc != NULL;
                     curr_arc = next_out_arc (curr_arc))
		  add_arc (curr_state, curr_arc->to_state, curr_arc->insn, 1);
            }
          arcs_marked_by_insn->to_state = curr_state;
          for (alts_number = 0,
	       curr_arc = arcs_marked_by_insn->next_arc_marked_by_insn;
               curr_arc != NULL;
               curr_arc = next_arc)
            {
              next_arc = curr_arc->next_arc_marked_by_insn;
              remove_arc (original_state, curr_arc);
	      alts_number++;
            }
	  arcs_marked_by_insn->state_alts = alts_number;
        }
    }
  if (!curr_state->it_was_placed_in_stack_for_DFA_forming)
    {
      curr_state->it_was_placed_in_stack_for_DFA_forming = TRUE;
      VLA_PTR_ADD (*state_stack, curr_state);
    }
}

/* The function transformes nondeterminstic AUTOMATON into
   determenistic. */
static void
NDFA_to_DFA (automaton)
     struct automaton *automaton;
{
  struct state *start_state;
  struct state *state;
  struct decl *curr_decl;
  vla_ptr_t state_stack;
  int i;

  VLA_PTR_CREATE (state_stack, 150, "state stack");
  /* Create the start state (empty state). */
  start_state = automaton->start_state;
  start_state->it_was_placed_in_stack_for_DFA_forming = TRUE;
  VLA_PTR_ADD (state_stack, start_state);
  while (VLA_PTR_LENGTH (state_stack) != 0)
    {
      state = VLA_PTR (state_stack, VLA_PTR_LENGTH (state_stack) - 1);
      VLA_PTR_SHORTEN (state_stack, 1);
      form_arcs_marked_by_insn (state);
      for (i = 0; i < description->decls_num; i++)
	{
	  curr_decl = description->decls [i];
	  if (curr_decl->mode == dm_insn_reserv)
	    create_composed_state
              (state, curr_decl->decl.insn_reserv.arcs_marked_by_insn,
               &state_stack);
	}
    }
  VLA_PTR_DELETE (state_stack);
}

/* The following variable value is current number (1, 2, ...) of passing
   graph of states. */
static int curr_state_graph_pass_num;

/* This recursive function passes all states achieved from START_STATE
   and applies APPLIED_FUNC to them. */
static void
pass_state_graph (start_state, applied_func)
     struct state *start_state;
     void (*applied_func) (struct state *state);
{
  struct arc *curr_arc;

  if (start_state->pass_num == curr_state_graph_pass_num)
    return;
  start_state->pass_num = curr_state_graph_pass_num;
  (*applied_func) (start_state);
  for (curr_arc = first_out_arc (start_state);
       curr_arc != NULL;
       curr_arc = next_out_arc (curr_arc))
    pass_state_graph (curr_arc->to_state, applied_func);
}

/* This recursive function passes all states of AUTOMATON and applies
   APPLIED_FUNC to them. */
static void
pass_states (automaton, applied_func)
     struct automaton *automaton;
     void (*applied_func) (struct state *state);
{
  curr_state_graph_pass_num++;
  pass_state_graph (automaton->start_state, applied_func);
}

/* The function initializes code for passing of all states. */
static void
initiate_pass_states ()
{
  curr_state_graph_pass_num = 0;
}

/* The following vla is used for storing pointers to all achieved
   states. */
static vla_ptr_t all_achieved_states;

/* This function is called by function pass_states to add an achieved
   STATE. */
static void
add_achieved_state (state)
     struct state *state;
{
  VLA_PTR_ADD (all_achieved_states, state);
}

/* The function sets up equivalence numbers of insns which mark all
   out arcs of STATE by equiv_class_num_1 (if ODD_ITERATION_FLAG ==
   TRUE) or by equiv_class_num_2 of the destination state.  The
   function returns number of out arcs of STATE. */
static int
set_out_arc_insns_equiv_num (state, odd_iteration_flag)
     struct state *state;
     int odd_iteration_flag;
{
  int state_out_arcs_num;
  struct arc *curr_arc;

  state_out_arcs_num = 0;
  for (curr_arc = first_out_arc (state);
       curr_arc != NULL;
       curr_arc = next_out_arc (curr_arc))
    {
      assert (curr_arc->insn->insn_reserv_decl->equiv_class_num == 0
	      && curr_arc->insn->insn_reserv_decl->state_alts == 0);
      state_out_arcs_num++;
      curr_arc->insn->insn_reserv_decl->equiv_class_num
	= (odd_iteration_flag
           ? curr_arc->to_state->equiv_class_num_1
	   : curr_arc->to_state->equiv_class_num_2);
      curr_arc->insn->insn_reserv_decl->state_alts = curr_arc->state_alts;
      assert (curr_arc->insn->insn_reserv_decl->equiv_class_num != 0
	      && curr_arc->insn->insn_reserv_decl->state_alts > 0);
    }
  return state_out_arcs_num;
}

/* The function clears equivalence numbers and alt_states in all insns
   which mark all out arcs of STATE. */
static void
clear_arc_insns_equiv_num (state)
     struct state *state;
{
  struct arc *curr_arc;

  for (curr_arc = first_out_arc (state);
       curr_arc != NULL;
       curr_arc = next_out_arc (curr_arc))
    {
      curr_arc->insn->insn_reserv_decl->equiv_class_num = 0;
      curr_arc->insn->insn_reserv_decl->state_alts = 0;
    }
}

/* The function copies pointers to equivalent states from vla FROM
   into vla TO. */
static void
copy_equiv_class (to, from)
     vla_ptr_t *to;
     const vla_ptr_t *from;
{
  struct state **curr_class_ptr;

  VLA_PTR_NULLIFY (*to);
  for (curr_class_ptr = VLA_PTR_BEGIN (*from);
       curr_class_ptr <= (struct state **) VLA_PTR_LAST (*from);
       curr_class_ptr++)
    VLA_PTR_ADD (*to, *curr_class_ptr);
}

/* The function returns TRUE if STATE is not equivalent to another
   state from the same current partition on equivalence classes
   Another state has ORIGINAL_STATE_OUT_ARCS_NUM number of output
   arcs.  Iteration of making equivalence partition is defined by
   ODD_ITERATION_FLAG. */
static int
state_is_differed (state, original_state_out_arcs_num, odd_iteration_flag)
     struct state *state;
     int original_state_out_arcs_num;
     int odd_iteration_flag;
{
  struct arc *curr_arc;
  int state_out_arcs_num;

  state_out_arcs_num = 0;
  for (curr_arc = first_out_arc (state);
       curr_arc != NULL;
       curr_arc = next_out_arc (curr_arc))
    {
      state_out_arcs_num++;
      if ((odd_iteration_flag
           ? curr_arc->to_state->equiv_class_num_1
           : curr_arc->to_state->equiv_class_num_2)
          != curr_arc->insn->insn_reserv_decl->equiv_class_num
	  || (curr_arc->insn->insn_reserv_decl->state_alts
	      != curr_arc->state_alts))
        return TRUE;
    }
  return state_out_arcs_num != original_state_out_arcs_num;
}

/* The function makes initial partition of STATES on equivalent
   classes. */
static struct state *
init_equiv_class (states, states_num)
     struct state **states;
     int states_num;
{
  struct state **curr_state_ptr;
  struct state *result_equiv_class;

  result_equiv_class = NULL;
  for (curr_state_ptr = states;
       curr_state_ptr < states + states_num;
       curr_state_ptr++)
    {
      (*curr_state_ptr)->equiv_class_num_1 = 1;
      (*curr_state_ptr)->next_equiv_class_state = result_equiv_class;
      result_equiv_class = *curr_state_ptr;
    }
  return result_equiv_class;
}

/* The function finds equivalent states of AUTOMATON. */
static void
evaluate_equiv_classes (automaton, equiv_classes)
     struct automaton *automaton;
     vla_ptr_t *equiv_classes;
{
  struct state *new_equiv_class;
  int new_equiv_class_num;
  int odd_iteration_flag;
  int finish_flag;
  vla_ptr_t next_iteration_classes;
  struct state *first_state;
  struct state *curr_state;
  struct state *prev_state;
  struct state *next_state;
  struct state **curr_equiv_class_ptr;
  struct state **curr_state_ptr;
  int first_state_out_arcs_num;
  
  VLA_PTR_CREATE (all_achieved_states, 1500, "all achieved states");
  pass_states (automaton, add_achieved_state);
  new_equiv_class = init_equiv_class (VLA_PTR_BEGIN (all_achieved_states),
                                      VLA_PTR_LENGTH (all_achieved_states));
  odd_iteration_flag = FALSE;
  new_equiv_class_num = 1;
  VLA_PTR_CREATE (next_iteration_classes, 150, "next iteration classes");
  VLA_PTR_ADD (next_iteration_classes, new_equiv_class);
  do
    {
      odd_iteration_flag = !odd_iteration_flag;
      finish_flag = TRUE;
      copy_equiv_class (equiv_classes, &next_iteration_classes);
      /* Transfer equiv numbers for the next iteration. */
      for (curr_state_ptr = VLA_PTR_BEGIN (all_achieved_states);
	   curr_state_ptr
	     <= (struct state **) VLA_PTR_LAST (all_achieved_states);
           curr_state_ptr++)
	if (odd_iteration_flag)
	  (*curr_state_ptr)->equiv_class_num_2
	    = (*curr_state_ptr)->equiv_class_num_1;
	else
	  (*curr_state_ptr)->equiv_class_num_1
	    = (*curr_state_ptr)->equiv_class_num_2;
      for (curr_equiv_class_ptr = VLA_PTR_BEGIN (*equiv_classes);
           curr_equiv_class_ptr
             <= (struct state **) VLA_PTR_LAST (*equiv_classes);
           curr_equiv_class_ptr++)
        {
          assert (*curr_equiv_class_ptr != NULL);
          for (first_state = *curr_equiv_class_ptr;
               first_state != NULL;
               first_state = new_equiv_class)
            {
              new_equiv_class = NULL;
              if (first_state->next_equiv_class_state != NULL)
                {
                  /* There are more one states in the class equivalence. */
		  first_state_out_arcs_num
		    = set_out_arc_insns_equiv_num (first_state,
						   odd_iteration_flag);
                  for (prev_state = first_state,
                       curr_state = first_state->next_equiv_class_state;
                       curr_state != NULL;
                       curr_state = next_state)
                    {
                      next_state = curr_state->next_equiv_class_state;
                      if (state_is_differed (curr_state,
					     first_state_out_arcs_num,
					     odd_iteration_flag))
                        {
                          /* Remove curr state from the class
                             equivalence. */
                          prev_state->next_equiv_class_state = next_state;
                          /* Add curr state to the new class equivalence. */
                          curr_state->next_equiv_class_state = new_equiv_class;
                          if (new_equiv_class == NULL)
                            new_equiv_class_num++;
                          if (odd_iteration_flag)
                            curr_state->equiv_class_num_2
                              = new_equiv_class_num;
                          else
                            curr_state->equiv_class_num_1
                              = new_equiv_class_num;
                          new_equiv_class = curr_state;
                          finish_flag = FALSE;
                        }
                      else
                        prev_state = curr_state;
                    }
                  clear_arc_insns_equiv_num (first_state);
                }
              if (new_equiv_class != NULL)
                VLA_PTR_ADD  (next_iteration_classes, new_equiv_class);
            }
        }
    }
  while (!finish_flag);
  VLA_PTR_DELETE (next_iteration_classes);
  VLA_PTR_DELETE (all_achieved_states);
}

/* The function merges equivalent states of AUTOMATON. */
static void
merge_states (automaton, equiv_classes)
     struct automaton *automaton;
     vla_ptr_t *equiv_classes;
{
  struct state **curr_equiv_class_ptr;
  struct state *curr_state;
  struct state *new_state;
  struct state *first_class_state;
  struct alt_state *alt_states;
  struct alt_state *new_alt_state;
  struct arc *curr_arc;
  struct arc *next_arc;

  /* Create states corresponding to equivalence classes containing two
     or more states. */
  for (curr_equiv_class_ptr = VLA_PTR_BEGIN (*equiv_classes);
       curr_equiv_class_ptr <= (struct state **) VLA_PTR_LAST (*equiv_classes);
       curr_equiv_class_ptr++)
    if ((*curr_equiv_class_ptr)->next_equiv_class_state != NULL)
      {
        /* There are more one states in the class equivalence. */
        /* Create new compound state. */
        new_state = get_free_state (FALSE, automaton);
        alt_states = NULL;
        first_class_state = *curr_equiv_class_ptr;
        for (curr_state = first_class_state;
             curr_state != NULL;
             curr_state = curr_state->next_equiv_class_state)
          {
            curr_state->equiv_class_state = new_state;
            new_alt_state = get_free_alt_state ();
            new_alt_state->state = curr_state;
            new_alt_state->next_sorted_alt_state = alt_states;
            alt_states = new_alt_state;
          }
        new_state->component_states = alt_states;
      }
    else
      (*curr_equiv_class_ptr)->equiv_class_state = *curr_equiv_class_ptr;
  for (curr_equiv_class_ptr = VLA_PTR_BEGIN (*equiv_classes);
       curr_equiv_class_ptr <= (struct state **) VLA_PTR_LAST (*equiv_classes);
       curr_equiv_class_ptr++)
    if ((*curr_equiv_class_ptr)->next_equiv_class_state != NULL)
      {
        first_class_state = *curr_equiv_class_ptr;
        /* Create new arcs output from the state corresponding to
           equiv class. */
        for (curr_arc = first_out_arc (first_class_state);
             curr_arc != NULL;
             curr_arc = next_out_arc (curr_arc))
          add_arc (first_class_state->equiv_class_state,
                   curr_arc->to_state->equiv_class_state,
		   curr_arc->insn, curr_arc->state_alts);
        /* Delete output arcs from states of given class equivalence. */
        for (curr_state = first_class_state;
             curr_state != NULL;
             curr_state = curr_state->next_equiv_class_state)
          {
            if (automaton->start_state == curr_state)
              automaton->start_state = curr_state->equiv_class_state;
            /* Delete the state and its output arcs. */
            for (curr_arc = first_out_arc (curr_state);
                 curr_arc != NULL;
                 curr_arc = next_arc)
              {
                next_arc = next_out_arc (curr_arc);
                free_arc (curr_arc);
              }
          }
      }
    else
      {
        /* Change `to_state' of arcs output from the state of given
           equivalence class. */
        for (curr_arc = first_out_arc (*curr_equiv_class_ptr);
             curr_arc != NULL;
             curr_arc = next_out_arc (curr_arc))
          curr_arc->to_state = curr_arc->to_state->equiv_class_state;
      }
}

/* The function sets up new_cycle_p for states if there is arc to the
   state marked by advance_cycle_insn_decl. */
static void
set_new_cycle_flags (state)
     struct state *state;
{
  struct arc *curr_arc;

  for (curr_arc = first_out_arc (state);
       curr_arc != NULL;
       curr_arc = next_out_arc (curr_arc))
    if (curr_arc->insn->insn_reserv_decl
        == &advance_cycle_insn_decl->decl.insn_reserv)
      curr_arc->to_state->new_cycle_p = TRUE;
}

/* The top level function for minimization of deterministic
   AUTOMATON. */
static void
minimize_DFA (automaton)
     struct automaton *automaton;
{
  vla_ptr_t equiv_classes;

  VLA_PTR_CREATE (equiv_classes, 1500, "equivalence classes");
  evaluate_equiv_classes (automaton, &equiv_classes);
  merge_states (automaton, &equiv_classes);
  pass_states (automaton, set_new_cycle_flags);
  VLA_PTR_DELETE (equiv_classes);
}

/* Values of two variables are counted number of states and arcs in an
   automaton.  */
static int curr_counted_states_num;
static int curr_counted_arcs_num;

/* The function is called by function `pass_states' to count states
   and arcs of an automaton. */
static void
incr_states_and_arcs_nums (state)
     struct state *state;
{
  struct arc *curr_arc;

  curr_counted_states_num++;
  for (curr_arc = first_out_arc (state);
       curr_arc != NULL;
       curr_arc = next_out_arc (curr_arc))
    curr_counted_arcs_num++;
}

/* The function counts states and arcs of AUTOMATON. */
static void
count_states_and_arcs (automaton, states_num, arcs_num)
     struct automaton *automaton;
     int *states_num;
     int *arcs_num;
{
  curr_counted_states_num = 0;
  curr_counted_arcs_num = 0;
  pass_states (automaton, incr_states_and_arcs_nums);
  *states_num = curr_counted_states_num;
  *arcs_num = curr_counted_arcs_num;
}

/* The function builds one DFA AUTOMATON for fast pipeline hazards
   recognition after checking and simplifying IR of the
   description. */
static void
build_automaton (automaton)
     struct automaton *automaton;
{
  int states_num;
  int arcs_num;

  ticker_on (&NDFA_time);
  make_automaton (automaton);
  ticker_off (&NDFA_time);
  count_states_and_arcs (automaton, &states_num, &arcs_num);
  automaton->NDFA_states_num = states_num;
  automaton->NDFA_arcs_num = arcs_num;
  ticker_on (&NDFA_to_DFA_time);
  NDFA_to_DFA (automaton);
  ticker_off (&NDFA_to_DFA_time);
  count_states_and_arcs (automaton, &states_num, &arcs_num);
  automaton->DFA_states_num = states_num;
  automaton->DFA_arcs_num = arcs_num;
  if (!no_minimization_flag)
    {
      ticker_on (&minimize_time);
      minimize_DFA (automaton);
      ticker_off (&minimize_time);
      count_states_and_arcs (automaton, &states_num, &arcs_num);
      automaton->minimal_DFA_states_num = states_num;
      automaton->minimal_DFA_arcs_num = arcs_num;
    }
}



/* The page contains code for enumeration  of all states of an automaton. */

/* Variable used for enumeration of all states of an automaton.  Its
   value is current number of automaton states. */
static int curr_state_order_num;

/* The function is called by function `pass_states' for enumerating
   states. */
static void
set_order_state_num (state)
     struct state *state;
{
  state->order_state_num = curr_state_order_num;
  curr_state_order_num++;
}

/* The function enumerates all states of AUTOMATON. */
static void
enumerate_states (automaton)
     struct automaton *automaton;
{
  curr_state_order_num = 0;
  pass_states (automaton, set_order_state_num);
  automaton->achieved_states_num = curr_state_order_num;
}



/* The page contains code for finding equivalent automaton insns
   (ainsns). */

/* The function inserts AINSN into cyclic list
   CYCLIC_EQUIV_CLASS_INSN_LIST of ainsns. */
static struct ainsn *
insert_ainsn_into_equiv_class (ainsn, cyclic_equiv_class_insn_list)
     struct ainsn *ainsn;
     struct ainsn *cyclic_equiv_class_insn_list;
{
  if (cyclic_equiv_class_insn_list == NULL)
    ainsn->next_equiv_class_insn = ainsn;
  else
    {
      ainsn->next_equiv_class_insn
        = cyclic_equiv_class_insn_list->next_equiv_class_insn;
      cyclic_equiv_class_insn_list->next_equiv_class_insn = ainsn;
    }
  return ainsn;
}

/* The function deletes equiv_class_insn into cyclic list of
   equivalent ainsns. */
static void
delete_ainsn_from_equiv_class (equiv_class_insn)
     struct ainsn *equiv_class_insn;
{
  struct ainsn *curr_equiv_class_insn;
  struct ainsn *prev_equiv_class_insn;

  prev_equiv_class_insn = equiv_class_insn;
  for (curr_equiv_class_insn = equiv_class_insn->next_equiv_class_insn;
       curr_equiv_class_insn != equiv_class_insn;
       curr_equiv_class_insn = curr_equiv_class_insn->next_equiv_class_insn)
    prev_equiv_class_insn = curr_equiv_class_insn;
  if (prev_equiv_class_insn != equiv_class_insn)
    prev_equiv_class_insn->next_equiv_class_insn
      = equiv_class_insn->next_equiv_class_insn;
}

/* The function processes AINSN of a state in order to find equivalent
   ainsns.  INSN_ARCS_ARRAY is table: code of insn -> out arc of the
   state. */
static void
process_insn_equiv_class (ainsn, insn_arcs_array)
     struct ainsn *ainsn;
     struct arc **insn_arcs_array;
{
  struct ainsn *next_equiv_class_insn;
  struct ainsn *curr_equiv_class_insn;
  struct ainsn *cyclic_equiv_class_insn_list;
  struct arc *temp_arc;

  assert (insn_arcs_array [ainsn->insn_reserv_decl->insn_num] != NULL);
  curr_equiv_class_insn = ainsn;
  /* New class of ainsns which are not equivalent to given ainsn. */
  cyclic_equiv_class_insn_list = NULL;
  do
    {
      next_equiv_class_insn = curr_equiv_class_insn->next_equiv_class_insn;
      temp_arc
	= insn_arcs_array [curr_equiv_class_insn->insn_reserv_decl->insn_num];
      if (temp_arc == NULL
          || (insn_arcs_array [ainsn->insn_reserv_decl->insn_num]->to_state
              != temp_arc->to_state))
        {
          delete_ainsn_from_equiv_class (curr_equiv_class_insn);
          cyclic_equiv_class_insn_list
            = insert_ainsn_into_equiv_class (curr_equiv_class_insn,
                                             cyclic_equiv_class_insn_list);
        }
      curr_equiv_class_insn = next_equiv_class_insn;
    }
  while (curr_equiv_class_insn != ainsn);
}

/* The function processes STATE in order to find equivalent ainsns. */
static void
process_state_for_insn_equiv_partition (state)
     struct state *state;
{
  struct arc *curr_arc;
  struct arc **insn_arcs_array;
  int curr_insn_num;
  vla_ptr_t insn_arcs_vect;

  VLA_PTR_CREATE (insn_arcs_vect, 500, "insn arcs vector");
  VLA_PTR_EXPAND (insn_arcs_vect, description->insns_num);
  insn_arcs_array = VLA_PTR_BEGIN (insn_arcs_vect);
  /* Process insns of the arcs. */
  for (curr_insn_num = 0;
       curr_insn_num < description->insns_num;
       curr_insn_num++)
    insn_arcs_array [curr_insn_num] = NULL;
  for (curr_arc = first_out_arc (state);
       curr_arc != NULL;
       curr_arc = next_out_arc (curr_arc))
    insn_arcs_array [curr_arc->insn->insn_reserv_decl->insn_num] = curr_arc;
  for (curr_arc = first_out_arc (state);
       curr_arc != NULL;
       curr_arc = next_out_arc (curr_arc))
    process_insn_equiv_class (curr_arc->insn, insn_arcs_array);
  VLA_PTR_DELETE (insn_arcs_vect);
}

/* The function searches for equivalent ainsns of AUTOMATON. */
static void
set_insn_equiv_classes (automaton)
     struct automaton *automaton;
{
  struct ainsn *curr_ainsn;
  struct ainsn *first_equiv_class_insn;
  struct ainsn *curr_equiv_class_insn;
  struct ainsn *cyclic_equiv_class_insn_list;
  struct ainsn *curr_insn_with_same_reservs;
  int equiv_classes_num;

  /* All insns are included in one equivalence class. */
  cyclic_equiv_class_insn_list = NULL;
  for (curr_ainsn = automaton->ainsn_list;
       curr_ainsn != NULL;
       curr_ainsn = curr_ainsn->next_ainsn)
    if (curr_ainsn->first_insn_with_same_reservs)
      cyclic_equiv_class_insn_list
        = insert_ainsn_into_equiv_class (curr_ainsn,
                                         cyclic_equiv_class_insn_list);
  /* Process insns in order to make equivalence partition. */
  pass_states (automaton, process_state_for_insn_equiv_partition);
  /* Enumerate equiv classes. */
  for (curr_ainsn = automaton->ainsn_list;
       curr_ainsn != NULL;
       curr_ainsn = curr_ainsn->next_ainsn)
    /* Set undefined value. */
    curr_ainsn->insn_equiv_class_num = -1;
  equiv_classes_num = 0;
  for (curr_ainsn = automaton->ainsn_list;
       curr_ainsn != NULL;
       curr_ainsn = curr_ainsn->next_ainsn)
    if (curr_ainsn->insn_equiv_class_num < 0)
      {
        first_equiv_class_insn = curr_ainsn;
        assert (first_equiv_class_insn->first_insn_with_same_reservs);
        first_equiv_class_insn->first_ainsn_with_given_equialence_num = TRUE;
        curr_equiv_class_insn = first_equiv_class_insn;
        do
          {
            for (curr_insn_with_same_reservs = curr_equiv_class_insn;
                 curr_insn_with_same_reservs != NULL;
                 curr_insn_with_same_reservs
                   = curr_insn_with_same_reservs->next_same_reservs_insn)
              curr_insn_with_same_reservs->insn_equiv_class_num
                = equiv_classes_num;
            curr_equiv_class_insn
              = curr_equiv_class_insn->next_equiv_class_insn;
          }
        while (curr_equiv_class_insn != first_equiv_class_insn);
        equiv_classes_num++;
      }
  automaton->insn_equiv_classes_num = equiv_classes_num;
}



/* This page contains code for creating DFA(s) and calls functions
   building them. */


/* The following value is used to prevent floating point overflow for
   estimating an automaton bound.  The value should be less DBL_MAX on
   the host machine.  We use here approximate minimum of maximal
   double floating point value required by ANSI C standard.  It
   will work for non ANSI sun compiler too. */

#define MAX_FLOATING_POINT_VALUE_FOR_AUTOMATON_BOUND  1.0E37

/* The function estimate size of the single DFA used by PHR (pipeline
   hazards recognizer). */
static double
estimate_one_automaton_bound ()
{
  struct decl *curr_decl;
  double one_automaton_estimation_bound;
  double root_value;
  int i;

  one_automaton_estimation_bound = 1.0;
  for (i = 0; i < description->decls_num; i++)
    {
      curr_decl = description->decls [i];
      if (curr_decl->mode == dm_unit)
	{
	  root_value = exp (log (curr_decl->decl.unit.max_occ_cycle_num + 1.0)
                            / automata_num);
	  if (MAX_FLOATING_POINT_VALUE_FOR_AUTOMATON_BOUND / root_value
	      > one_automaton_estimation_bound)
	    one_automaton_estimation_bound *= root_value;
	}
    }
  return one_automaton_estimation_bound;
}

/* The function compares unit declarations acoording to their maximal
   cycle in reservations. */
static int
compare_max_occ_cycle_nums (unit_decl_1, unit_decl_2)
     const void *unit_decl_1;
     const void *unit_decl_2;
{
  if (((*(struct decl **) unit_decl_1)->decl.unit.max_occ_cycle_num)
      < ((*(struct decl **) unit_decl_2)->decl.unit.max_occ_cycle_num))
    return 1;
  else if (((*(struct decl **) unit_decl_1)->decl.unit.max_occ_cycle_num)
	   == ((*(struct decl **) unit_decl_2)->decl.unit.max_occ_cycle_num))
    return 0;
  else
    return -1;
}

/* The function makes heuristic assigning automata to units.  Actually
   efficacy of the algorithm has been checked yet??? */
static void
units_to_automata_heuristic_distr ()
{
  double one_automaton_estimation_bound;
  struct decl *curr_decl;
  struct decl **curr_unit_decl_ptr;
  int curr_automaton_num;
  int rest_units_num;
  double curr_bound_value;
  vla_ptr_t unit_decls;
  int i;

  if (description->units_num == 0)
    return;
  one_automaton_estimation_bound = estimate_one_automaton_bound ();
  VLA_PTR_CREATE (unit_decls, 150, "unit decls");
  for (i = 0; i < description->decls_num; i++)
    {
      curr_decl = description->decls [i];
      if (curr_decl->mode == dm_unit)
	VLA_PTR_ADD (unit_decls, curr_decl);
    }
  qsort (VLA_PTR_BEGIN (unit_decls), VLA_PTR_LENGTH (unit_decls),
         sizeof (struct decl *), compare_max_occ_cycle_nums);
  curr_automaton_num = 0;
  curr_unit_decl_ptr = VLA_PTR_BEGIN (unit_decls);
  curr_bound_value = (*curr_unit_decl_ptr)->decl.unit.max_occ_cycle_num;
  (*curr_unit_decl_ptr)->decl.unit.corresponding_automaton_num
    = curr_automaton_num;
  for (curr_unit_decl_ptr++;
       curr_unit_decl_ptr <= (struct decl **) VLA_PTR_LAST (unit_decls);
       curr_unit_decl_ptr++)
    {
      rest_units_num = ((struct decl **) VLA_PTR_LAST (unit_decls)
                        - curr_unit_decl_ptr + 1);
      assert (automata_num - curr_automaton_num - 1 <= rest_units_num);
      if (curr_automaton_num < automata_num - 1
          && ((automata_num - curr_automaton_num - 1 == rest_units_num)
              || (curr_bound_value
                  > (one_automaton_estimation_bound
                     / ((*curr_unit_decl_ptr)->decl.unit.max_occ_cycle_num)))))
        {
          curr_bound_value
            = (*curr_unit_decl_ptr)->decl.unit.max_occ_cycle_num;
          curr_automaton_num++;
        }
      else
        curr_bound_value *= (*curr_unit_decl_ptr)->decl.unit.max_occ_cycle_num;
      (*curr_unit_decl_ptr)->decl.unit.corresponding_automaton_num
	= curr_automaton_num;
    }
  assert (curr_automaton_num == automata_num - 1);
  VLA_PTR_DELETE (unit_decls);
}

/* The functions creates automaton insns for each automata.  Automaton
   insn is simply insn for given automaton which makes reservation
   only of units of the automaton. */
static struct ainsn *
create_ainsns ()
{
  struct decl *curr_decl;
  struct ainsn *first_ainsn;
  struct ainsn *curr_ainsn;
  struct ainsn *prev_ainsn;
  int i;

  first_ainsn = NULL;
  prev_ainsn = NULL;
  for (i = 0; i < description->decls_num; i++)
    {
      curr_decl = description->decls [i];
      if (curr_decl->mode == dm_insn_reserv)
	{
	  curr_ainsn = create_node (sizeof (struct ainsn));
	  curr_ainsn->insn_reserv_decl = &curr_decl->decl.insn_reserv;
	  curr_ainsn->next_ainsn = NULL;
	  if (prev_ainsn == NULL)
	    first_ainsn = curr_ainsn;
	  else
	    prev_ainsn->next_ainsn = curr_ainsn;
	  prev_ainsn = curr_ainsn;
	}
    }
  return first_ainsn;
}

/* The function assigns automata to units according to constructions
   `define_automaton' in the description. */
static void
units_to_automata_distr ()
{
  struct decl *curr_decl;
  int i;
  
  for (i = 0; i < description->decls_num; i++)
    {
      curr_decl = description->decls [i];
      if (curr_decl->mode == dm_unit)
	{
	  if (curr_decl->decl.unit.automaton_decl == NULL
	      || (curr_decl->decl.unit.automaton_decl->corresponding_automaton
		  == NULL))
	    /* Distribute to the first automaton. */
	    curr_decl->decl.unit.corresponding_automaton_num = 0;
	  else
	    curr_decl->decl.unit.corresponding_automaton_num
	      = (curr_decl->decl.unit.automaton_decl
                 ->corresponding_automaton->automaton_order_num);
	}
    }
}

/* The function creates DFA(s) for fast pipeline hazards recognition
   after checking and simplifying IR of the description. */
static void
create_automata ()
{
  struct automaton *curr_automaton;
  struct automaton *prev_automaton;
  struct decl *curr_decl;
  int curr_automaton_num;
  int i;

  if (automata_num != 0)
    {
      units_to_automata_heuristic_distr ();
      for (prev_automaton = NULL, curr_automaton_num = 0;
           curr_automaton_num < automata_num;
           curr_automaton_num++, prev_automaton = curr_automaton)
        {
	  curr_automaton = create_node (sizeof (struct automaton));
	  curr_automaton->ainsn_list = create_ainsns ();
	  curr_automaton->corresponding_automaton_decl = NULL;
	  curr_automaton->next_automaton = NULL;
          curr_automaton->automaton_order_num = curr_automaton_num;
          if (prev_automaton == NULL)
            description->first_automaton = curr_automaton;
          else
            prev_automaton->next_automaton = curr_automaton;
        }
    }
  else
    {
      curr_automaton_num = 0;
      prev_automaton = NULL;
      for (i = 0; i < description->decls_num; i++)
	{
	  curr_decl = description->decls [i];
	  if (curr_decl->mode == dm_automaton
              && curr_decl->decl.automaton.automaton_is_used)
	    {
	      curr_automaton = create_node (sizeof (struct automaton));
	      curr_automaton->ainsn_list = create_ainsns ();
	      curr_automaton->corresponding_automaton_decl
                = &curr_decl->decl.automaton;
	      curr_automaton->next_automaton = NULL;
	      curr_decl->decl.automaton.corresponding_automaton
                = curr_automaton;
	      curr_automaton->automaton_order_num = curr_automaton_num;
	      if (prev_automaton == NULL)
		description->first_automaton = curr_automaton;
	      else
		prev_automaton->next_automaton = curr_automaton;
	      curr_automaton_num++;
	      prev_automaton = curr_automaton;
	    }
	}
      if (curr_automaton_num == 0)
	{
	  curr_automaton = create_node (sizeof (struct automaton));
	  curr_automaton->ainsn_list = create_ainsns ();
	  curr_automaton->corresponding_automaton_decl = NULL;
	  curr_automaton->next_automaton = NULL;
	  description->first_automaton = curr_automaton;
	}
      units_to_automata_distr ();
    }
  NDFA_time = create_ticker ();
  ticker_off (&NDFA_time);
  NDFA_to_DFA_time = create_ticker ();
  ticker_off (&NDFA_to_DFA_time);
  minimize_time = create_ticker ();
  ticker_off (&minimize_time);
  equiv_time = create_ticker ();
  ticker_off (&equiv_time);
  for (curr_automaton = description->first_automaton;
       curr_automaton != NULL;
       curr_automaton = curr_automaton->next_automaton)
    {
      create_alt_states (curr_automaton);
      form_ainsn_with_same_reservs (curr_automaton);
      build_automaton (curr_automaton);
      enumerate_states (curr_automaton);
      ticker_on (&equiv_time);
      set_insn_equiv_classes (curr_automaton);
      ticker_off (&equiv_time);
    }
}



/* This page contains code for forming string representation of
   regexp.  The representation is formed on IR obstack.  So you should
   not work with IR obstack between regexp_representation and
   finish_regexp_representation calls. */

/* This recursive function forms string representation of regexp
   (without tailing '\0'). */
static void
form_regexp (regexp)
     struct regexp *regexp;
{
  int i;
    
  if (regexp->mode == rm_unit || regexp->mode == rm_reserv)
    {
      const char *name = (regexp->mode == rm_unit
                          ? regexp->regexp.unit.name
                          : regexp->regexp.reserv.name);

      IR_TOP_ADD_MEMORY (name, strlen (name));
    }
  else if (regexp->mode == rm_sequence)
    for (i = 0; i < regexp->regexp.sequence.regexps_num; i++)
      {
	if (i != 0)
          IR_TOP_ADD_BYTE (',');
	form_regexp (regexp->regexp.sequence.regexps [i]);
      }
  else if (regexp->mode == rm_allof)
    {
      IR_TOP_ADD_BYTE ('(');
      for (i = 0; i < regexp->regexp.allof.regexps_num; i++)
	{
	  if (i != 0)
            IR_TOP_ADD_BYTE ('+');
	  if (regexp->regexp.allof.regexps[i]->mode == rm_sequence
              || regexp->regexp.oneof.regexps[i]->mode == rm_oneof)
            IR_TOP_ADD_BYTE ('(');
	  form_regexp (regexp->regexp.allof.regexps [i]);
	  if (regexp->regexp.allof.regexps[i]->mode == rm_sequence
              || regexp->regexp.oneof.regexps[i]->mode == rm_oneof)
            IR_TOP_ADD_BYTE (')');
        }
      IR_TOP_ADD_BYTE (')');
    }
  else if (regexp->mode == rm_oneof)
    for (i = 0; i < regexp->regexp.oneof.regexps_num; i++)
      {
	if (i != 0)
          IR_TOP_ADD_BYTE ('|');
	if (regexp->regexp.oneof.regexps[i]->mode == rm_sequence)
          IR_TOP_ADD_BYTE ('(');
        form_regexp (regexp->regexp.oneof.regexps [i]);
	if (regexp->regexp.oneof.regexps[i]->mode == rm_sequence)
          IR_TOP_ADD_BYTE (')');
      }
  else if (regexp->mode == rm_repeat)
    {
      char digits [20];

      if (regexp->regexp.repeat.regexp->mode == rm_sequence
	  || regexp->regexp.repeat.regexp->mode == rm_allof
	  || regexp->regexp.repeat.regexp->mode == rm_oneof)
        IR_TOP_ADD_BYTE ('(');
      form_regexp (regexp->regexp.repeat.regexp);
      if (regexp->regexp.repeat.regexp->mode == rm_sequence
	  || regexp->regexp.repeat.regexp->mode == rm_allof
	  || regexp->regexp.repeat.regexp->mode == rm_oneof)
        IR_TOP_ADD_BYTE (')');
      sprintf (digits, "*%d", regexp->regexp.repeat.repeat_num);
      IR_TOP_ADD_MEMORY (digits, strlen (digits));
    }
  else if (regexp->mode == rm_nothing)
    IR_TOP_ADD_MEMORY (NOTHING_NAME, strlen (NOTHING_NAME));
  else
    assert (FALSE);
}

/* The function returns string representation of REGEXP on IR
   obstack. */
static const char *
regexp_representation (regexp)
     struct regexp *regexp;
{
  form_regexp (regexp);
  IR_TOP_ADD_BYTE ('\0');
  return IR_TOP_BEGIN ();
}

/* The function frees memory allocated for last formed string
   representation of regexp. */
static void
finish_regexp_representation ()
{
  IR_TOP_NULLIFY ();
}



/* This page contains code for output PHR (pipeline hazards recognizer). */

/* The function outputs minimal C type which is sufficient for
   representation numbers in range min_range_value and
   max_range_value.  Because host machine and build machine may be
   different, we use here minimal values required by ANSI C standard
   instead of UCHAR_MAX, SHRT_MAX, SHRT_MIN, etc.  This is a good
   approximation. */

static void
output_range_type (f, min_range_value, max_range_value)
     FILE *f;
     long int min_range_value;
     long int max_range_value;
{
  if (min_range_value >= 0 && max_range_value <= 255)
    fprintf (f, "unsigned char");
  else if (min_range_value >= -127 && max_range_value <= 127)
    fprintf (f, "signed char");
  else if (min_range_value >= 0 && max_range_value <= 65535)
    fprintf (f, "unsigned short");
  else if (min_range_value >= -32767 && max_range_value <= 32767)
    fprintf (f, "short");
  else
    fprintf (f, "int");
}

/* The function outputs all initialization values of VECT with length
   vect_length. */
static void
output_vect (vect, vect_length)
     vect_el_t *vect;
     int vect_length;
{
  int els_on_line;

  els_on_line = 1;
  if (vect_length == 0)
    fprintf (output_file,
             "0 /* This is dummy el because the vect is empty */");
  else
    {
      do
        {
          fprintf (output_file, "%5d", *vect);
          vect_length--;
          if (els_on_line == 10)
	    {
	      els_on_line = 0;
	      fprintf (output_file, ",\n");
	    }
          else if (vect_length != 0)
            fprintf (output_file, ", ");
          els_on_line++;
          vect++;
        }
      while (vect_length != 0);
    }
}

/* The following is name of the structure which represents DFA(s) for
   PHR. */
#define CHIP_NAME "DFA_chip"

/* The following is name of member which represents state of a DFA for
   PHR. */
static void
output_chip_member_name (f, automaton)
     FILE *f;
     struct automaton *automaton;
{
  if (automaton->corresponding_automaton_decl == NULL)
    fprintf (f, "automaton_state_%d", automaton->automaton_order_num);
  else
    fprintf (f, "%s_automaton_state",
             automaton->corresponding_automaton_decl->name);
}

/* The following is name of temporary variable which stores state of a
   DFA for PHR. */
static void
output_temp_chip_member_name (f, automaton)
     FILE *f;
     struct automaton *automaton;
{
  fprintf (f, "_");
  output_chip_member_name (f, automaton);
}

/* This is name of macro value which is code of pseudo_insn
   representing advancing cpu cycle.  Its value is used as internal
   code unknown insn. */
#define ADVANCE_CYCLE_VALUE_NAME "DFA__ADVANCE_CYCLE"

/* Output name of translate vector for given automaton. */
static void
output_translate_vect_name (f, automaton)
     FILE *f;
     struct automaton *automaton;
{
  if (automaton->corresponding_automaton_decl == NULL)
    fprintf (f, "translate_%d", automaton->automaton_order_num);
  else
    fprintf (f, "%s_translate", automaton->corresponding_automaton_decl->name);
}

/* Output name for simple transition table representation. */
static void
output_trans_full_vect_name (f, automaton)
     FILE *f;
     struct automaton *automaton;
{
  if (automaton->corresponding_automaton_decl == NULL)
    fprintf (f, "transitions_%d", automaton->automaton_order_num);
  else
    fprintf (f, "%s_transitions",
             automaton->corresponding_automaton_decl->name);
}

/* Output name of comb vector of the transition table for given
   automaton. */
static void
output_trans_comb_vect_name (f, automaton)
     FILE *f;
     struct automaton *automaton;
{
  if (automaton->corresponding_automaton_decl == NULL)
    fprintf (f, "transitions_%d", automaton->automaton_order_num);
  else
    fprintf (f, "%s_transitions",
             automaton->corresponding_automaton_decl->name);
}

/* Output name of check vector of the transition table for given
   automaton. */
static void
output_trans_check_vect_name (f, automaton)
     FILE *f;
     struct automaton *automaton;
{
  if (automaton->corresponding_automaton_decl == NULL)
    fprintf (f, "check_%d", automaton->automaton_order_num);
  else
    fprintf (f, "%s_check", automaton->corresponding_automaton_decl->name);
}

/* Output name of base vector of the transition table for given
   automaton. */
static void
output_trans_base_vect_name (f, automaton)
     FILE *f;
     struct automaton *automaton;
{
  if (automaton->corresponding_automaton_decl == NULL)
    fprintf (f, "base_%d", automaton->automaton_order_num);
  else
    fprintf (f, "%s_base", automaton->corresponding_automaton_decl->name);
}

/* Output name for simple alternatives number representation. */
static void
output_state_alts_full_vect_name (f, automaton)
     FILE *f;
     struct automaton *automaton;
{
  if (automaton->corresponding_automaton_decl == NULL)
    fprintf (f, "state_alts_%d", automaton->automaton_order_num);
  else
    fprintf (f, "%s_state_alts",
             automaton->corresponding_automaton_decl->name);
}

/* Output name of comb vector of the alternatives number table for given
   automaton. */
static void
output_state_alts_comb_vect_name (f, automaton)
     FILE *f;
     struct automaton *automaton;
{
  if (automaton->corresponding_automaton_decl == NULL)
    fprintf (f, "state_alts_%d", automaton->automaton_order_num);
  else
    fprintf (f, "%s_state_alts",
             automaton->corresponding_automaton_decl->name);
}

/* Output name of check vector of the alternatives number table for given
   automaton. */
static void
output_state_alts_check_vect_name (f, automaton)
     FILE *f;
     struct automaton *automaton;
{
  if (automaton->corresponding_automaton_decl == NULL)
    fprintf (f, "check_state_alts_%d", automaton->automaton_order_num);
  else
    fprintf (f, "%s_check_state_alts",
	     automaton->corresponding_automaton_decl->name);
}

/* Output name of base vector of the alternatives number table for given
   automaton. */
static void
output_state_alts_base_vect_name (f, automaton)
     FILE *f;
     struct automaton *automaton;
{
  if (automaton->corresponding_automaton_decl == NULL)
    fprintf (f, "base_state_alts_%d", automaton->automaton_order_num);
  else
    fprintf (f, "%s_base_state_alts",
	     automaton->corresponding_automaton_decl->name);
}

/* Output name of simple min issue delay table representation. */
static void
output_min_issue_delay_full_vect_name (f, automaton)
     FILE *f;
     struct automaton *automaton;
{
  if (automaton->corresponding_automaton_decl == NULL)
    fprintf (f, "min_issue_delay_%d", automaton->automaton_order_num);
  else
    fprintf (f, "%s_min_issue_delay",
             automaton->corresponding_automaton_decl->name);
}

/* Output name of comb vector of min issue delay table (state x isns
   -> delay) for given automaton. */
static void
output_min_issue_delay_comb_vect_name (f, automaton)
     FILE *f;
     struct automaton *automaton;
{
  if (automaton->corresponding_automaton_decl == NULL)
    fprintf (f, "min_issue_delay_%d", automaton->automaton_order_num);
  else
    fprintf (f, "%s_min_issue_delay",
             automaton->corresponding_automaton_decl->name);
}

/* Output name of check vector of the min issue delay table for given
   automaton. */
static void
output_min_issue_delay_check_vect_name (f, automaton)
     FILE *f;
     struct automaton *automaton;
{
  if (automaton->corresponding_automaton_decl == NULL)
    fprintf (f, "check_min_issue_delay_%d", automaton->automaton_order_num);
  else
    fprintf (f, "%s_check_min_issue_delay",
	     automaton->corresponding_automaton_decl->name);
}

/* Output name of base vector of the min issue delay table for given
   automaton. */
static void
output_min_issue_delay_base_vect_name (f, automaton)
     FILE *f;
     struct automaton *automaton;
{
  if (automaton->corresponding_automaton_decl == NULL)
    fprintf (f, "base_min_issue_delay_%d", automaton->automaton_order_num);
  else
    fprintf (f, "%s_base_min_issue_delay",
	     automaton->corresponding_automaton_decl->name);
}

/* Output name of deadlock vector for given automaton. */
static void
output_dead_lock_vect_name (f, automaton)
     FILE *f;
     struct automaton *automaton;
{
  if (automaton->corresponding_automaton_decl == NULL)
    fprintf (f, "dead_lock_%d", automaton->automaton_order_num);
  else
    fprintf (f, "%s_dead_lock", automaton->corresponding_automaton_decl->name);
}

/* Name of the PHR interface macro. */
#define AUTOMATON_STATE_ALTS_MACRO_NAME "AUTOMATON_STATE_ALTS"

/* Names of an internal functions: */
#define INTERNAL_MIN_ISSUE_DELAY_FUNC_NAME "internal_min_issue_delay"

#define INTERNAL_TRANSITION_FUNC_NAME "internal_state_transition"

#define INTERNAL_STATE_ALTS_FUNC_NAME "internal_state_alts"

#define INTERNAL_RESET_FUNC_NAME "internal_reset"

#define INTERNAL_DEAD_LOCK_FUNC_NAME "internal_state_dead_lock_p"

#define INTERNAL_INSN_LATENCY_FUNC_NAME "internal_insn_latency"

/* Names of the PHR interface functions: */
#define TRANSITION_FUNC_NAME "state_transition"

#define STATE_ALTS_FUNC_NAME "state_alts"

#define MIN_ISSUE_DELAY_FUNC_NAME "min_issue_delay"

#define MIN_INSN_CONFLICT_DELAY_FUNC_NAME "min_insn_conflict_delay"

#define DEAD_LOCK_FUNC_NAME "state_dead_lock_p"

#define RESET_FUNC_NAME "state_reset"

#define INSN_LATENCY_FUNC_NAME "insn_latency"

#define PRINT_RESERVATION_FUNC_NAME "print_reservation"

/* Names of parameters of the PHR interface functions. */
#define INSN_PARAMETER_NAME "insn"

#define INSN2_PARAMETER_NAME "insn2"

#define CHIP_PARAMETER_NAME "chip"

#define FILE_PARAMETER_NAME "f"

/* Names of the variables whose values are internal insn code of rtx
   insn. */
#define INTERNAL_INSN_CODE_NAME "insn_code"

#define INTERNAL_INSN2_CODE_NAME "insn2_code"

/* Name of temporary variable in some functions. */
#define TEMPORARY_VARIABLE_NAME "temp"

/* Name of result variable in some functions. */
#define RESULT_VARIABLE_NAME "res"

/* Name of function (attribute) to translate insn into number of insn
   alternatives reservation. */
#define INSN_ALTS_FUNC_NAME "insn_alts"

/* Name of function (attribute) to translate insn into internal insn
   code. */
#define DFA_INSN_CODE_FUNC_NAME "dfa_insn_code"

/* Name of function (attribute) to translate insn into internal insn
   code. */
#define INSN_DEFAULT_LATENCY_FUNC_NAME "insn_default_latency"

/* Name of function (attribute) to translate insn into internal insn
   code. */
#define BYPASS_P_FUNC_NAME "bypass_p"

/* Output C type which is used for representation of codes of states
   of AUTOMATON. */
static void
output_state_member_type (f, automaton)
     FILE *f;
     struct automaton *automaton;
{
  output_range_type (f, 0, automaton->achieved_states_num);
}

/* Output definition of the structure representing current DFA(s)
   state(s). */
static void
output_chip_definitions ()
{
  struct automaton *curr_automaton;

  fprintf (output_file, "struct %s\n{\n", CHIP_NAME);
  for (curr_automaton = description->first_automaton;
       curr_automaton != NULL;
       curr_automaton = curr_automaton->next_automaton)
    {
      fprintf (output_file, "  ");
      output_state_member_type (output_file, curr_automaton);
      fprintf (output_file, " ");
      output_chip_member_name (output_file, curr_automaton);
      fprintf (output_file, ";\n");
    }
  fprintf (output_file, "};\n\n");
  fprintf (output_file, "static struct %s %s;\n\n", CHIP_NAME, CHIP_NAME);
}


/* The function outputs translate vector of internal insn code into
   insn equivalence class number.  The equivalence class number is
   used to access to table and vectors reprewsenting DFA(s). */
static void
output_translate_vect (automaton)
     struct automaton *automaton;
{
  struct ainsn *curr_ainsn;
  int curr_insn_value;
  vla_hwint_t translate_vect;

  VLA_HWINT_CREATE (translate_vect, 250, "translate vector");
  VLA_HWINT_EXPAND (translate_vect, description->insns_num);
  for (curr_insn_value = 0;
       curr_insn_value <= description->insns_num;
       curr_insn_value++)
    /* Undefined value */
    VLA_HWINT (translate_vect, curr_insn_value)
      = automaton->insn_equiv_classes_num;
  for (curr_ainsn = automaton->ainsn_list;
       curr_ainsn != NULL;
       curr_ainsn = curr_ainsn->next_ainsn)
    VLA_HWINT (translate_vect, curr_ainsn->insn_reserv_decl->insn_num)
      = curr_ainsn->insn_equiv_class_num;
  fprintf (output_file,
           "/* Vector translating external insn codes to internal ones.*/\n");
  fprintf (output_file, "static const ");
  output_range_type (output_file, 0, automaton->insn_equiv_classes_num);
  fprintf (output_file, " ");
  output_translate_vect_name (output_file, automaton);
  fprintf (output_file, "[] = {\n");
  output_vect (VLA_HWINT_BEGIN (translate_vect),
               VLA_HWINT_LENGTH (translate_vect));
  fprintf (output_file, "};\n\n");
  VLA_HWINT_DELETE (translate_vect);
}

/* The value in a table state x ainsn -> something which represents
   undefined value. */
static int undefined_vect_el_value;

/* The following function returns TRUE if the best representation of
   the table is comb vector. */
static int
comb_vect_p (tab)
     struct state_ainsn_table *tab;
{
  return  (2 * VLA_HWINT_LENGTH (tab->full_vect)
           > 5 * VLA_HWINT_LENGTH (tab->comb_vect));
}

/* The following function creates new table for AUTOMATON. */
static struct state_ainsn_table *
create_state_ainsn_table (automaton)
     struct automaton *automaton;
{
  struct state_ainsn_table *tab;
  int full_vect_length;
  int i;

  tab = create_node (sizeof (struct state_ainsn_table));
  tab->automaton = automaton;
  VLA_HWINT_CREATE (tab->comb_vect, 10000, "comb vector");
  VLA_HWINT_CREATE (tab->check_vect, 10000, "check vector");
  VLA_HWINT_CREATE (tab->base_vect, 1000, "base vector");
  VLA_HWINT_EXPAND (tab->base_vect, automaton->achieved_states_num);
  VLA_HWINT_CREATE (tab->full_vect, 10000, "full vector");
  full_vect_length = (automaton->insn_equiv_classes_num
                      * automaton->achieved_states_num);
  VLA_HWINT_EXPAND (tab->full_vect, full_vect_length);
  for (i = 0; i < full_vect_length; i++)
    VLA_HWINT (tab->full_vect, i) = undefined_vect_el_value;
  tab->min_base_vect_el_value = 0;
  tab->max_base_vect_el_value = 0;
  tab->min_comb_vect_el_value = 0;
  tab->max_comb_vect_el_value = 0;
  return tab;
}

/* The following function outputs the best C representation of the
   table TAB of given TABLE_NAME. */
static void
output_state_ainsn_table (tab, table_name, output_full_vect_name_func,
                          output_comb_vect_name_func,
                          output_check_vect_name_func,
                          output_base_vect_name_func)
     struct state_ainsn_table *tab;
     char *table_name;
     void (*output_full_vect_name_func) (FILE *f, struct automaton *automaton);
     void (*output_comb_vect_name_func) (FILE *f, struct automaton *automaton);
     void (*output_check_vect_name_func) (FILE *f,
                                          struct automaton *automaton);
     void (*output_base_vect_name_func) (FILE *f, struct automaton *automaton);
{
  if (!comb_vect_p (tab))
    {
      fprintf (output_file, "/* Vector for %s. */\n", table_name);
      fprintf (output_file, "static const ");
      output_range_type (output_file, tab->min_comb_vect_el_value,
                         tab->max_comb_vect_el_value);
      fprintf (output_file, " ");
      (*output_full_vect_name_func) (output_file, tab->automaton);
      fprintf (output_file, "[] = {\n");
      output_vect (VLA_HWINT_BEGIN (tab->full_vect),
                   VLA_HWINT_LENGTH (tab->full_vect));
      fprintf (output_file, "};\n\n");
    }
  else
    {
      fprintf (output_file, "/* Comb vector for %s. */\n", table_name);
      fprintf (output_file, "static const ");
      output_range_type (output_file, tab->min_comb_vect_el_value,
                         tab->max_comb_vect_el_value);
      fprintf (output_file, " ");
      (*output_comb_vect_name_func) (output_file, tab->automaton);
      fprintf (output_file, "[] = {\n");
      output_vect (VLA_HWINT_BEGIN (tab->comb_vect),
                   VLA_HWINT_LENGTH (tab->comb_vect));
      fprintf (output_file, "};\n\n");
      fprintf (output_file, "/* Check vector for %s. */\n", table_name);
      fprintf (output_file, "static const ");
      output_range_type (output_file, 0, tab->automaton->achieved_states_num);
      fprintf (output_file, " ");
      (*output_check_vect_name_func) (output_file, tab->automaton);
      fprintf (output_file, "[] = {\n");
      output_vect (VLA_HWINT_BEGIN (tab->check_vect),
                   VLA_HWINT_LENGTH (tab->check_vect));
      fprintf (output_file, "};\n\n");
      fprintf (output_file, "/* Base vector for %s. */\n", table_name);
      fprintf (output_file, "static const ");
      output_range_type (output_file, tab->min_base_vect_el_value,
                         tab->max_base_vect_el_value);
      fprintf (output_file, " ");
      (*output_base_vect_name_func) (output_file, tab->automaton);
      fprintf (output_file, "[] = {\n");
      output_vect (VLA_HWINT_BEGIN (tab->base_vect),
                   VLA_HWINT_LENGTH (tab->base_vect));
      fprintf (output_file, "};\n\n");
    }
}

static void
add_vect (tab, vect_num, vect, vect_length)
     struct state_ainsn_table *tab;
     int vect_num;
     vect_el_t *vect;
     int vect_length;
{
  int real_vect_length;
  vect_el_t *comb_vect_start;
  vect_el_t *check_vect_start;
  int comb_vect_index;
  int comb_vect_els_num;
  int vect_index;
  int first_unempty_vect_index;
  int additional_els_num;
  int no_state_value;
  vect_el_t vect_el;
  int i;

  assert (vect_length != 0);
  real_vect_length = tab->automaton->insn_equiv_classes_num;
  assert (vect [vect_length - 1] != undefined_vect_el_value);
  /* Form full vector: */
  for (i = 0; i < vect_length; i++)
    VLA_HWINT (tab->full_vect,
               i + tab->automaton->insn_equiv_classes_num * vect_num)
      = vect [i];
  /* Form comb vector: */
  assert (VLA_HWINT_LENGTH (tab->comb_vect)
          == VLA_HWINT_LENGTH (tab->check_vect));
  comb_vect_start = VLA_HWINT_BEGIN (tab->comb_vect);
  comb_vect_els_num = VLA_HWINT_LENGTH (tab->comb_vect);
  for (first_unempty_vect_index = 0;
       first_unempty_vect_index < vect_length;
       first_unempty_vect_index++)
    if (vect [first_unempty_vect_index] != undefined_vect_el_value)
      break;
  /* Search for the place in comb vect for the inserted vect. */
  for (comb_vect_index = 0;
       comb_vect_index < comb_vect_els_num;
       comb_vect_index++)
    {
      for (vect_index = first_unempty_vect_index;
           vect_index < vect_length
             && vect_index + comb_vect_index < comb_vect_els_num;
           vect_index++)
        if (vect [vect_index] != undefined_vect_el_value
            && (comb_vect_start [vect_index + comb_vect_index]
		!= undefined_vect_el_value))
          break;
      if (vect_index >= vect_length
          || vect_index + comb_vect_index >= comb_vect_els_num)
        break;
    }
  /* Slot was found. */
  additional_els_num = comb_vect_index + real_vect_length - comb_vect_els_num;
  if (additional_els_num < 0)
    additional_els_num = 0;
  /* Expand comb and check vectors. */
  vect_el = undefined_vect_el_value;
  no_state_value = tab->automaton->achieved_states_num;
  while (additional_els_num > 0)
    {
      VLA_HWINT_ADD (tab->comb_vect, vect_el);
      VLA_HWINT_ADD (tab->check_vect, no_state_value);
      additional_els_num--;
    }
  comb_vect_start = VLA_HWINT_BEGIN (tab->comb_vect);
  check_vect_start = VLA_HWINT_BEGIN (tab->check_vect);
  assert (VLA_HWINT_LENGTH (tab->comb_vect)
          >= (size_t) (comb_vect_index + real_vect_length));
  /* Fill comb and check vectors. */
  for (vect_index = 0; vect_index < vect_length; vect_index++)
    if (vect [vect_index] != undefined_vect_el_value)
      {
        assert (comb_vect_start [comb_vect_index + vect_index]
		== undefined_vect_el_value);
        comb_vect_start [comb_vect_index + vect_index] = vect [vect_index];
        assert (vect [vect_index] >= 0);
        if (tab->max_comb_vect_el_value < vect [vect_index])
          tab->max_comb_vect_el_value = vect [vect_index];
        if (tab->min_comb_vect_el_value > vect [vect_index])
          tab->min_comb_vect_el_value = vect [vect_index];
        check_vect_start [comb_vect_index + vect_index] = vect_num;
      }
  if (tab->max_base_vect_el_value < comb_vect_index)
    tab->max_base_vect_el_value = comb_vect_index;
  if (tab->min_base_vect_el_value > comb_vect_index)
    tab->min_base_vect_el_value = comb_vect_index;
  VLA_HWINT (tab->base_vect, vect_num) = comb_vect_index;
}

/* Return number of out arcs of STATE. */
static int
out_state_arcs_num (state)
     struct state *state;
{
  int result;
  struct arc *curr_arc;

  result = 0;
  for (curr_arc = first_out_arc (state);
       curr_arc != NULL;
       curr_arc = next_out_arc (curr_arc))
    {
      assert (curr_arc->insn != NULL);
      if (curr_arc->insn->first_ainsn_with_given_equialence_num)
        result++;
    }
  return result;
}

/* Compare number of possible transitions from the states. */
static int
compare_transition_els_num (state_ptr_1, state_ptr_2)
     const void *state_ptr_1;
     const void *state_ptr_2;
{
  int transition_els_num_1;
  int transition_els_num_2;

  transition_els_num_1 = out_state_arcs_num (*(struct state **) state_ptr_1);
  transition_els_num_2 = out_state_arcs_num (*(struct state **) state_ptr_2);
  if (transition_els_num_1 < transition_els_num_2)
    return 1;
  else if (transition_els_num_1 == transition_els_num_2)
    return 0;
  else
    return -1;
}

/* The function adds element EL_VALUE to vector VECT for a table state
   x AINSN. */
static void
add_vect_el (vect, ainsn, el_value)
     vla_hwint_t *vect;
     struct ainsn *ainsn;
     int el_value;
{
  int equiv_class_num;
  int vect_index;

  assert (ainsn != NULL);
  equiv_class_num = ainsn->insn_equiv_class_num;
  for (vect_index = VLA_HWINT_LENGTH (*vect);
       vect_index <= equiv_class_num;
       vect_index++)
    VLA_HWINT_ADD (*vect, undefined_vect_el_value);
  VLA_HWINT (*vect, equiv_class_num) = el_value;
}

/* This is for forming vector of states of an automaton. */
static vla_ptr_t output_states_vect;

/* The function is called by function pass_states.  The function adds
   STATE to `output_states_vect'. */
static void
add_states_vect_el (state)
     struct state *state;
{
  VLA_PTR_ADD (output_states_vect, state);
}

/* Form and output vectors (comb, check, base or full vector)
   representing transition table of AUTOMATON. */
static void
output_trans_table (automaton)
     struct automaton *automaton;
{
  struct state **curr_state_ptr;
  struct arc *curr_arc;
  vla_hwint_t transition_vect;

  undefined_vect_el_value = automaton->achieved_states_num;
  automaton->trans_table = create_state_ainsn_table (automaton);
  /* Create vect of pointers to states ordered by num of transitions
     from the state (state with the maximum num is the first). */
  VLA_PTR_CREATE (output_states_vect, 1500, "output states vector");
  pass_states (automaton, add_states_vect_el);
  qsort (VLA_PTR_BEGIN (output_states_vect),
         VLA_PTR_LENGTH (output_states_vect),
         sizeof (struct state *), compare_transition_els_num);
  VLA_HWINT_CREATE (transition_vect, 500, "transition vector");
  for (curr_state_ptr = VLA_PTR_BEGIN (output_states_vect);
       curr_state_ptr <= (struct state **) VLA_PTR_LAST (output_states_vect);
       curr_state_ptr++)
    {
      VLA_HWINT_NULLIFY (transition_vect);
      for (curr_arc = first_out_arc (*curr_state_ptr);
           curr_arc != NULL;
           curr_arc = next_out_arc (curr_arc))
        {
          assert (curr_arc->insn != NULL);
          if (curr_arc->insn->first_ainsn_with_given_equialence_num)
            add_vect_el (&transition_vect, curr_arc->insn,
                         curr_arc->to_state->order_state_num);
        }
      add_vect (automaton->trans_table, (*curr_state_ptr)->order_state_num,
                VLA_HWINT_BEGIN (transition_vect),
                VLA_HWINT_LENGTH (transition_vect));
    }
  output_state_ainsn_table (automaton->trans_table, "state transitions",
                            output_trans_full_vect_name,
                            output_trans_comb_vect_name,
                            output_trans_check_vect_name,
                            output_trans_base_vect_name);
  VLA_PTR_DELETE (output_states_vect);
  VLA_HWINT_DELETE (transition_vect);
}

/* Form and output vectors (comb, check, base or simple vect)
   representing alts number table of AUTOMATON.  The table is state x
   ainsn -> number of possible alternative reservations by the
   ainsn. */
static void
output_state_alts_table (automaton)
     struct automaton *automaton;
{
  struct state **curr_state_ptr;
  struct arc *curr_arc;
  vla_hwint_t state_alts_vect;

  undefined_vect_el_value = 0; /* no alts when transition is not possible */
  automaton->state_alts_table = create_state_ainsn_table (automaton);
  /* Create vect of pointers to states ordered by num of transitions
     from the state (state with the maximum num is the first). */
  VLA_PTR_CREATE (output_states_vect, 1500, "output states vector");
  pass_states (automaton, add_states_vect_el);
  qsort (VLA_PTR_BEGIN (output_states_vect),
         VLA_PTR_LENGTH (output_states_vect),
         sizeof (struct state *), compare_transition_els_num);
  /* Create base, comb, and check vectors. */
  VLA_HWINT_CREATE (state_alts_vect, 500, "state alts vector");
  for (curr_state_ptr = VLA_PTR_BEGIN (output_states_vect);
       curr_state_ptr <= (struct state **) VLA_PTR_LAST (output_states_vect);
       curr_state_ptr++)
    {
      VLA_HWINT_NULLIFY (state_alts_vect);
      for (curr_arc = first_out_arc (*curr_state_ptr);
           curr_arc != NULL;
           curr_arc = next_out_arc (curr_arc))
        {
          assert (curr_arc->insn != NULL);
          if (curr_arc->insn->first_ainsn_with_given_equialence_num)
            add_vect_el (&state_alts_vect, curr_arc->insn,
                         curr_arc->state_alts);
        }
      add_vect (automaton->state_alts_table,
                (*curr_state_ptr)->order_state_num,
                VLA_HWINT_BEGIN (state_alts_vect),
                VLA_HWINT_LENGTH (state_alts_vect));
    }
  output_state_ainsn_table (automaton->state_alts_table,
                            "state insn alternatives",
                            output_state_alts_full_vect_name,
                            output_state_alts_comb_vect_name,
                            output_state_alts_check_vect_name,
                            output_state_alts_base_vect_name);
  VLA_PTR_DELETE (output_states_vect);
  VLA_HWINT_DELETE (state_alts_vect);
}

/* The following variable value is the current minimal issue delay
   value found for an ainsn and state. Negative value means undefined
   value. */
static int current_min_insn_issue_delay;

/* The current number of passing states to find minimal issue delay
   value for an ainsn and state. */
static int current_arc_pass_number;

/* This recursive function passes states to find minimal issue delay
   value for AINSN.  The state being visited is STATE.
   PATH_LENGTH_BEFORE is passed path length from original state to
   STATE. */
static void
min_issue_delay_pass_states (state, ainsn, path_length_before)
     struct state *state;
     struct ainsn *ainsn;
     int path_length_before;
{
  struct arc *curr_arc;

  if (current_min_insn_issue_delay >= 0
      && path_length_before >= current_min_insn_issue_delay)
    return;
  for (curr_arc = first_out_arc (state);
       curr_arc != NULL;
       curr_arc = next_out_arc (curr_arc))
    if (curr_arc->insn == ainsn)
      current_min_insn_issue_delay = path_length_before;
    else if (curr_arc->pass_number != current_arc_pass_number)
      {
        curr_arc->pass_number = current_arc_pass_number;
        min_issue_delay_pass_states
          (curr_arc->to_state, ainsn,
           path_length_before + (curr_arc->insn->insn_reserv_decl
                                 == &advance_cycle_insn_decl->decl.insn_reserv
                                 ? 1 : 0));
      }
}

/* The function searches minimal issue delay value for AINSN in STATE. */
static int
min_issue_delay (state, ainsn)
     struct state *state;
     struct ainsn *ainsn;
{
  current_min_insn_issue_delay = -1;
  current_arc_pass_number++;
  min_issue_delay_pass_states (state, ainsn, 0);
  assert (current_min_insn_issue_delay >= 0);
  return current_min_insn_issue_delay;
}

/* The function initiates code for finding minimal issue delay values.
   It should be called only once. */
static void
initiate_min_issue_delay_pass_states ()
{
  current_arc_pass_number = 0;
}

/* Form and output vectors (comb, check, base or simple vect)
   representing minimal issue delay table of AUTOMATON.  The table is
   state x ainsn -> minimal issue delay of the ainsn. */
static void
output_min_issue_delay_table (automaton)
     struct automaton *automaton;
{
  vect_el_t min_delay;
  struct ainsn *curr_ainsn;
  struct state **curr_state_ptr;
  vla_hwint_t state_min_issue_delay;

  /* Actually all elements will be defined.  So we can use negative
     value here. */
  undefined_vect_el_value = -1;
  automaton->min_issue_delay_table = create_state_ainsn_table (automaton);
  /* Create vect of pointers to states ordered by num of transitions
     from the state (state with the maximum num is the first). */
  VLA_PTR_CREATE (output_states_vect, 1500, "output states vector");
  pass_states (automaton, add_states_vect_el);
  /* Create base, comb, and check vectors. */
  VLA_HWINT_CREATE (state_min_issue_delay, 500,
                    "min issue delay state vector");
  for (curr_state_ptr = VLA_PTR_BEGIN (output_states_vect);
       curr_state_ptr <= (struct state **) VLA_PTR_LAST (output_states_vect);
       curr_state_ptr++)
    {
      VLA_HWINT_NULLIFY (state_min_issue_delay);
      for (curr_ainsn = automaton->ainsn_list;
           curr_ainsn != NULL;
           curr_ainsn = curr_ainsn->next_ainsn)
        if (curr_ainsn->first_ainsn_with_given_equialence_num)
          {
            min_delay = min_issue_delay (*curr_state_ptr, curr_ainsn);
            add_vect_el (&state_min_issue_delay, curr_ainsn, min_delay);
          }
      add_vect (automaton->min_issue_delay_table,
                (*curr_state_ptr)->order_state_num,
                VLA_HWINT_BEGIN (state_min_issue_delay),
                VLA_HWINT_LENGTH (state_min_issue_delay));
    }
  output_state_ainsn_table (automaton->min_issue_delay_table,
                            "min issue delay of insns",
                            output_min_issue_delay_full_vect_name,
                            output_min_issue_delay_comb_vect_name,
                            output_min_issue_delay_check_vect_name,
                            output_min_issue_delay_base_vect_name);
  VLA_PTR_DELETE (output_states_vect);
  VLA_HWINT_DELETE (state_min_issue_delay);
}

#ifndef NDEBUG
/* Number of states which contains transition only by advancing cpu
   cycle. */
static int locked_states_num;
#endif

/* Form and output vector representing the locked states of
   AUTOMATON. */
static void
output_dead_lock_vect (automaton)
     struct automaton *automaton;
{
  struct state **curr_state_ptr;
  struct arc *curr_arc;
  vla_hwint_t dead_lock_vect;

  /* Create vect of pointers to states ordered by num of
     transitions from the state (state with the maximum num is the
     first). */
  VLA_PTR_CREATE (output_states_vect, 1500, "output states vector");
  pass_states (automaton, add_states_vect_el);
  /* Create base, transition, and check vectors. */
  VLA_PTR_CREATE (dead_lock_vect, 1500, "is dead locked vector");
  VLA_PTR_EXPAND (dead_lock_vect, VLA_HWINT_LENGTH (output_states_vect));
  for (curr_state_ptr = VLA_PTR_BEGIN (output_states_vect);
       curr_state_ptr <= (struct state **) VLA_PTR_LAST (output_states_vect);
       curr_state_ptr++)
    {
      curr_arc = first_out_arc (*curr_state_ptr);
      assert (curr_arc != NULL);
      VLA_HWINT (dead_lock_vect, (*curr_state_ptr)->order_state_num)
        = (next_out_arc (curr_arc) == NULL
           && (curr_arc->insn->insn_reserv_decl
               == &advance_cycle_insn_decl->decl.insn_reserv)
           ? 1 : 0);
#ifndef NDEBUG
      if (VLA_HWINT (dead_lock_vect, (*curr_state_ptr)->order_state_num))
        locked_states_num++;
#endif
    }
  fprintf (output_file, "/* Vector for locked state flags. */\n");
  fprintf (output_file, "static const ");
  output_range_type (output_file, 0, 1);
  fprintf (output_file, " ");
  output_dead_lock_vect_name (output_file, automaton);
  fprintf (output_file, "[] = {\n");
  output_vect (VLA_HWINT_BEGIN (dead_lock_vect),
               VLA_HWINT_LENGTH (dead_lock_vect));
  fprintf (output_file, "};\n\n");
  VLA_HWINT_DELETE (dead_lock_vect);
}

/* The function outputs all tables representing DFA(s) used for fast
   pipeline hazards recognition.  */
static void
output_tables ()
{
  struct automaton *curr_automaton;

#ifndef NDEBUG
  locked_states_num = 0;
#endif
  initiate_min_issue_delay_pass_states ();
  for (curr_automaton = description->first_automaton;
       curr_automaton != NULL;
       curr_automaton = curr_automaton->next_automaton)
    {
      output_translate_vect (curr_automaton);
      output_trans_table (curr_automaton);
      fprintf (output_file, "\n#if %s\n", AUTOMATON_STATE_ALTS_MACRO_NAME);
      output_state_alts_table (curr_automaton);
      fprintf (output_file, "\n#endif /* #if %s */\n\n",
	       AUTOMATON_STATE_ALTS_MACRO_NAME);
      output_min_issue_delay_table (curr_automaton);
      output_dead_lock_vect (curr_automaton);
    }
  fprintf (output_file, "\n#define %s %d\n\n", ADVANCE_CYCLE_VALUE_NAME,
           advance_cycle_insn_decl->decl.insn_reserv.insn_num);
}

/* The function outputs definition and value of PHR interface variable
   `max_insn_queue_index' */
static void
output_max_insn_queue_index_def ()
{
  int i;

  for (i = 0; (1 << i) <= description->max_insn_reserv_cycles; i++)
    ;
  assert (i >= 0);
  fprintf (output_file, "\nint max_insn_queue_index = %d;\n\n", (1 << i) - 1);
}


/* Output function `internal_min_issue_delay'. */
static void
output_internal_min_issue_delay_func ()
{
  struct automaton *curr_automaton;

  fprintf (output_file,
	   "static int\n%s (%s, %s)\n\tint %s;\n\tstruct %s *%s;\n",
	   INTERNAL_MIN_ISSUE_DELAY_FUNC_NAME, INTERNAL_INSN_CODE_NAME,
	   CHIP_PARAMETER_NAME, INTERNAL_INSN_CODE_NAME, CHIP_NAME,
	   CHIP_PARAMETER_NAME);
  fprintf (output_file, "{\n  int %s;\n  int %s;\n",
	   TEMPORARY_VARIABLE_NAME, RESULT_VARIABLE_NAME);
  for (curr_automaton = description->first_automaton;
       curr_automaton != NULL;
       curr_automaton = curr_automaton->next_automaton)
    {
      if (comb_vect_p (curr_automaton->min_issue_delay_table))
	{
	  fprintf (output_file, "\n  %s = ", TEMPORARY_VARIABLE_NAME);
	  output_min_issue_delay_base_vect_name (output_file, curr_automaton);
	  fprintf (output_file, " [%s->", CHIP_PARAMETER_NAME);
	  output_chip_member_name (output_file, curr_automaton);
	  fprintf (output_file, "] + ");
	  output_translate_vect_name (output_file, curr_automaton);
	  fprintf (output_file, " [%s];\n", INTERNAL_INSN_CODE_NAME);
	  fprintf (output_file, "  if (");
	  output_min_issue_delay_check_vect_name (output_file, curr_automaton);
	  fprintf (output_file, " [%s] != %s->",
		   TEMPORARY_VARIABLE_NAME, CHIP_PARAMETER_NAME);
	  output_chip_member_name (output_file, curr_automaton);
	  fprintf (output_file, ")\n");
	  fprintf (output_file, "    abort ();\n");
	  fprintf (output_file, "  else\n");
	  fprintf (output_file, "    %s = ", TEMPORARY_VARIABLE_NAME);
	  output_min_issue_delay_comb_vect_name (output_file, curr_automaton);
	  fprintf (output_file, " [%s];\n", TEMPORARY_VARIABLE_NAME);
	}
      else
	{
	  fprintf (output_file, "\n  %s = ", TEMPORARY_VARIABLE_NAME);
	  output_min_issue_delay_full_vect_name (output_file, curr_automaton);
	  fprintf (output_file, " [");
	  output_translate_vect_name (output_file, curr_automaton);
	  fprintf (output_file, " [%s] + ", INTERNAL_INSN_CODE_NAME);
	  fprintf (output_file, "%s->", CHIP_PARAMETER_NAME);
	  output_chip_member_name (output_file, curr_automaton);
	  fprintf (output_file, " * %d];\n",
		   curr_automaton->insn_equiv_classes_num);
	}
      if (curr_automaton == description->first_automaton)
	fprintf (output_file, "  %s = %s;\n",
		 RESULT_VARIABLE_NAME, TEMPORARY_VARIABLE_NAME);
      else
	{
	  fprintf (output_file, "  if (%s > %s)\n",
		   TEMPORARY_VARIABLE_NAME, RESULT_VARIABLE_NAME);
	  fprintf (output_file, "    %s = %s;\n",
		   RESULT_VARIABLE_NAME, TEMPORARY_VARIABLE_NAME);
	}
    }
  fprintf (output_file, "  return %s;\n", RESULT_VARIABLE_NAME);
  fprintf (output_file, "}\n\n");
}

/* Output function `internal_state_transition'. */
static void
output_internal_trans_func ()
{
  struct automaton *curr_automaton;
  struct automaton *next_automaton;

  fprintf (output_file,
	   "static int\n%s (%s, %s)\n\tint %s;\n\tstruct %s *%s;\n",
	   INTERNAL_TRANSITION_FUNC_NAME, INTERNAL_INSN_CODE_NAME,
	   CHIP_PARAMETER_NAME, INTERNAL_INSN_CODE_NAME,
	   CHIP_NAME, CHIP_PARAMETER_NAME);
  fprintf (output_file, "{\n  int %s;\n", TEMPORARY_VARIABLE_NAME);
  if (description->first_automaton != NULL)
    for (curr_automaton = description->first_automaton;;
         curr_automaton = next_automaton)
      {
        next_automaton = curr_automaton->next_automaton;
        if (next_automaton == NULL)
          break;
        fprintf (output_file, "  ");
        output_state_member_type (output_file, curr_automaton);
	fprintf (output_file, " ");
        output_temp_chip_member_name (output_file, curr_automaton);
        fprintf (output_file, ";\n");
      }
  for (curr_automaton = description->first_automaton;
       curr_automaton != NULL;
       curr_automaton = curr_automaton->next_automaton)
    if (comb_vect_p (curr_automaton->trans_table))
      {
	fprintf (output_file, "\n  %s = ", TEMPORARY_VARIABLE_NAME);
	output_trans_base_vect_name (output_file, curr_automaton);
	fprintf (output_file, " [%s->", CHIP_PARAMETER_NAME);
	output_chip_member_name (output_file, curr_automaton);
	fprintf (output_file, "] + ");
	output_translate_vect_name (output_file, curr_automaton);
	fprintf (output_file, " [%s];\n", INTERNAL_INSN_CODE_NAME);
	fprintf (output_file, "  if (");
	output_trans_check_vect_name (output_file, curr_automaton);
	fprintf (output_file, " [%s] != %s->",
		 TEMPORARY_VARIABLE_NAME, CHIP_PARAMETER_NAME);
	output_chip_member_name (output_file, curr_automaton);
	fprintf (output_file, ")\n");
	fprintf (output_file, "    return %s (%s, %s);\n",
		 INTERNAL_MIN_ISSUE_DELAY_FUNC_NAME, INTERNAL_INSN_CODE_NAME,
		 CHIP_PARAMETER_NAME);
	fprintf (output_file, "  else\n");
	fprintf (output_file, "    ");
	if (curr_automaton->next_automaton != NULL)
	  output_temp_chip_member_name (output_file, curr_automaton);
	else
	  {
	    fprintf (output_file, "%s->", CHIP_PARAMETER_NAME);
	    output_chip_member_name (output_file, curr_automaton);
	  }
	fprintf (output_file, " = ");
	output_trans_comb_vect_name (output_file, curr_automaton);
	fprintf (output_file, " [%s];\n", TEMPORARY_VARIABLE_NAME);
      }
    else
      {
	fprintf (output_file, "\n  %s = ", TEMPORARY_VARIABLE_NAME);
	output_trans_full_vect_name (output_file, curr_automaton);
	fprintf (output_file, " [");
	output_translate_vect_name (output_file, curr_automaton);
	fprintf (output_file, " [%s] + ", INTERNAL_INSN_CODE_NAME);
	fprintf (output_file, "%s->", CHIP_PARAMETER_NAME);
	output_chip_member_name (output_file, curr_automaton);
	fprintf (output_file, " * %d];\n",
		 curr_automaton->insn_equiv_classes_num);
	fprintf (output_file, "  if (%s >= %d)\n", TEMPORARY_VARIABLE_NAME,
		 curr_automaton->achieved_states_num);
	fprintf (output_file, "    return %s (%s, %s);\n",
		 INTERNAL_MIN_ISSUE_DELAY_FUNC_NAME, INTERNAL_INSN_CODE_NAME,
		 CHIP_PARAMETER_NAME);
	fprintf (output_file, "  else\n    ");
	if (curr_automaton->next_automaton != NULL)
	  output_temp_chip_member_name (output_file, curr_automaton);
	else
	  {
	    fprintf (output_file, "%s->", CHIP_PARAMETER_NAME);
	    output_chip_member_name (output_file, curr_automaton);
	  }
	fprintf (output_file, " = %s;\n", TEMPORARY_VARIABLE_NAME);
      }
  if (description->first_automaton != NULL)
    for (curr_automaton = description->first_automaton;;
         curr_automaton = next_automaton)
      {
        next_automaton = curr_automaton->next_automaton;
        if (next_automaton == NULL)
          break;
        fprintf (output_file, "  %s->", CHIP_PARAMETER_NAME);
        output_chip_member_name (output_file, curr_automaton);
        fprintf (output_file, " = ");
        output_temp_chip_member_name (output_file, curr_automaton);
        fprintf (output_file, ";\n");
      }
  fprintf (output_file, "  return 0;\n");
  fprintf (output_file, "}\n\n");
}

/* Output code

  if (insn != 0)
    {
      insn_code = dfa_insn_code (insn);
      if (insn_code > DFA__ADVANCE_CYCLE)
        return 0;
    }
  else
    insn_code = DFA__ADVANCE_CYCLE;

  where insn denotes INSN_NAME and insn_code denotes INSN_CODE_NAME. */
static void
output_internal_insn_code_evaluation (insn_name, insn_code_name)
     const char *insn_name;
     const char *insn_code_name;
{
  fprintf (output_file, "\n  if (%s != 0)\n    {\n", insn_name);
  fprintf (output_file, "      %s = %s (%s);\n", insn_code_name,
	   DFA_INSN_CODE_FUNC_NAME, insn_name);
  fprintf (output_file, "      if (%s > %s)\n        return 0;\n",
	   insn_code_name, ADVANCE_CYCLE_VALUE_NAME);
  fprintf (output_file, "    }\n  else\n    %s = %s;\n\n",
	   insn_code_name, ADVANCE_CYCLE_VALUE_NAME);
}


/* The function outputs PHR interface function `state_transition'. */
static void
output_trans_func ()
{
  fprintf (output_file, "int\n%s (%s)\n\trtx %s;\n", TRANSITION_FUNC_NAME,
	   INSN_PARAMETER_NAME, INSN_PARAMETER_NAME);
  fprintf (output_file, "{\n  int %s;\n", INTERNAL_INSN_CODE_NAME);
  output_internal_insn_code_evaluation (INSN_PARAMETER_NAME,
					INTERNAL_INSN_CODE_NAME);
  fprintf (output_file, "  return %s (%s, &%s);\n}\n\n",
	   INTERNAL_TRANSITION_FUNC_NAME, INTERNAL_INSN_CODE_NAME, CHIP_NAME);
}

/* Output function `internal_state_alts'. */
static void
output_internal_state_alts_func ()
{
  struct automaton *curr_automaton;

  fprintf (output_file,
	   "static int\n%s (%s, %s)\n\tint %s;\n\tstruct %s *%s;\n",
	   INTERNAL_STATE_ALTS_FUNC_NAME, INTERNAL_INSN_CODE_NAME,
	   CHIP_PARAMETER_NAME, INTERNAL_INSN_CODE_NAME, CHIP_NAME,
	   CHIP_PARAMETER_NAME);
  fprintf (output_file, "{\n  int %s;\n", RESULT_VARIABLE_NAME);
  for (curr_automaton = description->first_automaton;
       curr_automaton != NULL;
       curr_automaton = curr_automaton->next_automaton)
    if (comb_vect_p (curr_automaton->state_alts_table))
      {
	fprintf (output_file, "  int %s;\n", TEMPORARY_VARIABLE_NAME);
	break;
      }
  for (curr_automaton = description->first_automaton;
       curr_automaton != NULL;
       curr_automaton = curr_automaton->next_automaton)
    if (comb_vect_p (curr_automaton->state_alts_table))
      {
	fprintf (output_file, "\n  %s = ", TEMPORARY_VARIABLE_NAME);
	output_state_alts_base_vect_name (output_file, curr_automaton);
	fprintf (output_file, " [%s->", CHIP_PARAMETER_NAME);
	output_chip_member_name (output_file, curr_automaton);
	fprintf (output_file, "] + ");
	output_translate_vect_name (output_file, curr_automaton);
	fprintf (output_file, " [%s];\n", INTERNAL_INSN_CODE_NAME);
	fprintf (output_file, "  if (");
	output_state_alts_check_vect_name (output_file, curr_automaton);
	fprintf (output_file, " [%s] != %s->",
		 TEMPORARY_VARIABLE_NAME, CHIP_PARAMETER_NAME);
	output_chip_member_name (output_file, curr_automaton);
	fprintf (output_file, ")\n");
	fprintf (output_file, "    return 0;\n");
	fprintf (output_file, "  else\n");
	fprintf (output_file,
		 (curr_automaton == description->first_automaton
		  ? "    %s = " : "    %s += "), RESULT_VARIABLE_NAME);
	output_state_alts_comb_vect_name (output_file, curr_automaton);
	fprintf (output_file, " [%s];\n", TEMPORARY_VARIABLE_NAME);
      }
    else
      {
	fprintf (output_file,
		 (curr_automaton == description->first_automaton
		  ? "\n  %s = " : "  %s += "), RESULT_VARIABLE_NAME);
	output_state_alts_full_vect_name (output_file, curr_automaton);
	fprintf (output_file, " [");
	output_translate_vect_name (output_file, curr_automaton);
	fprintf (output_file, " [%s] + ", INTERNAL_INSN_CODE_NAME);
	fprintf (output_file, "%s->", CHIP_PARAMETER_NAME);
	output_chip_member_name (output_file, curr_automaton);
	fprintf (output_file, " * %d];\n",
		 curr_automaton->insn_equiv_classes_num);
      }
  fprintf (output_file, "  return %s;\n", RESULT_VARIABLE_NAME);
  fprintf (output_file, "}\n\n");
}

/* The function outputs PHR interface function `state_alts'. */
static void
output_state_alts_func ()
{
  fprintf (output_file, "int\n%s (%s)\n\trtx %s;\n", STATE_ALTS_FUNC_NAME,
	   INSN_PARAMETER_NAME, INSN_PARAMETER_NAME);
  fprintf (output_file, "{\n  int %s;\n", INTERNAL_INSN_CODE_NAME);
  output_internal_insn_code_evaluation (INSN_PARAMETER_NAME,
					INTERNAL_INSN_CODE_NAME);
  fprintf (output_file, "  return %s (%s, &%s);\n}\n\n",
	   INTERNAL_STATE_ALTS_FUNC_NAME, INTERNAL_INSN_CODE_NAME, CHIP_NAME);
}

/* Output function `min_issue_delay'. */
static void
output_min_issue_delay_func ()
{
  fprintf (output_file, "int\n%s (%s)\n\trtx %s;\n", MIN_ISSUE_DELAY_FUNC_NAME,
	   INSN_PARAMETER_NAME, INSN_PARAMETER_NAME);
  fprintf (output_file, "{\n  int %s;\n", INTERNAL_INSN_CODE_NAME);
  fprintf (output_file, "\n  if (%s != 0)\n    {\n", INSN_PARAMETER_NAME);
  fprintf (output_file, "      %s = %s (%s);\n", INTERNAL_INSN_CODE_NAME,
	   DFA_INSN_CODE_FUNC_NAME, INSN_PARAMETER_NAME);
  fprintf (output_file, "      if (%s > %s)\n        return 0;\n",
	   INTERNAL_INSN_CODE_NAME, ADVANCE_CYCLE_VALUE_NAME);
  fprintf (output_file, "    }\n  else\n    %s = %s;\n",
	   INTERNAL_INSN_CODE_NAME, ADVANCE_CYCLE_VALUE_NAME);
  fprintf (output_file, "\n  return %s (%s, &%s);\n",
	   INTERNAL_MIN_ISSUE_DELAY_FUNC_NAME, INTERNAL_INSN_CODE_NAME,
	   CHIP_NAME);
  fprintf (output_file, "}\n\n");
}

/* Output function `internal_dead_lock'. */
static void
output_internal_dead_lock_func ()
{
  struct automaton *curr_automaton;

  fprintf (output_file, "static int\n%s (%s)\n\tstruct %s *%s;\n",
	   INTERNAL_DEAD_LOCK_FUNC_NAME, CHIP_PARAMETER_NAME, CHIP_NAME,
	   CHIP_PARAMETER_NAME);
  fprintf (output_file, "{\n");
  for (curr_automaton = description->first_automaton;
       curr_automaton != NULL;
       curr_automaton = curr_automaton->next_automaton)
    {
      fprintf (output_file, "  if (");
      output_dead_lock_vect_name (output_file, curr_automaton);
      fprintf (output_file, " [%s->", CHIP_PARAMETER_NAME);
      output_chip_member_name (output_file, curr_automaton);
      fprintf (output_file, "])\n    return 1/* TRUE */;\n");
    }
  fprintf (output_file, "  return 0/* FALSE */;\n}\n\n");
}

/* The function outputs PHR interface function `state_dead_lock_p'. */
static void
output_dead_lock_func ()
{
  fprintf (output_file, "int\n%s ()\n", DEAD_LOCK_FUNC_NAME);
  fprintf (output_file, "{\n  return %s (&%s);\n}\n\n",
	   INTERNAL_DEAD_LOCK_FUNC_NAME, CHIP_NAME);
}

/* Output function `internal_reset'. */
static void
output_internal_reset_func ()
{
  struct automaton *curr_automaton;

  fprintf (output_file, "static void\n%s (%s)\n\tstruct %s *%s;\n",
	   INTERNAL_RESET_FUNC_NAME, CHIP_PARAMETER_NAME,
	   CHIP_NAME, CHIP_PARAMETER_NAME);
  fprintf (output_file, "{\n");
  for (curr_automaton = description->first_automaton;
       curr_automaton != NULL;
       curr_automaton = curr_automaton->next_automaton)
    {
      fprintf (output_file, "  %s->", CHIP_PARAMETER_NAME);
      output_chip_member_name (output_file, curr_automaton);
      fprintf (output_file, " = 0;\n");
    }
  fprintf (output_file, "}\n\n");
}

/* The function outputs PHR interface function `state_reset'. */
static void
output_reset_func ()
{
  fprintf (output_file, "void\n%s ()\n", RESET_FUNC_NAME);
  fprintf (output_file, "{\n  %s (&%s);\n}\n\n", INTERNAL_RESET_FUNC_NAME,
	   CHIP_NAME);
}

/* Output function `min_insn_conflict_delay'. */
static void
output_min_insn_conflict_delay_func ()
{
  fprintf (output_file, "int\n%s (%s, %s)\n\trtx %s;\n\trtx %s;\n",
	   MIN_INSN_CONFLICT_DELAY_FUNC_NAME,
	   INSN_PARAMETER_NAME, INSN2_PARAMETER_NAME,
	   INSN_PARAMETER_NAME, INSN2_PARAMETER_NAME);
  fprintf (output_file, "{\n  struct %s %s;\n  int %s, %s;\n",
	   CHIP_NAME, CHIP_NAME, INTERNAL_INSN_CODE_NAME,
	   INTERNAL_INSN2_CODE_NAME);
  output_internal_insn_code_evaluation (INSN_PARAMETER_NAME,
					INTERNAL_INSN_CODE_NAME);
  output_internal_insn_code_evaluation (INSN2_PARAMETER_NAME,
					INTERNAL_INSN2_CODE_NAME);
  fprintf (output_file, "  %s (&%s);\n", INTERNAL_RESET_FUNC_NAME, CHIP_NAME);
  fprintf (output_file, "  if (%s (%s, &%s))\n    abort ();\n",
	   INTERNAL_TRANSITION_FUNC_NAME, INTERNAL_INSN_CODE_NAME, CHIP_NAME);
  fprintf (output_file, "  return %s (%s, &%s);\n",
	   INTERNAL_MIN_ISSUE_DELAY_FUNC_NAME, INTERNAL_INSN2_CODE_NAME,
	   CHIP_NAME);
  fprintf (output_file, "}\n\n");
}

/* Output function `internal_insn_latency'. */
static void
output_internal_insn_latency_func ()
{
  struct decl *curr_decl;
  struct bypass_decl *curr_bypass;
  int i;

  fprintf (output_file, "static int\n%s (%s, %s)\n\tint %s;\n\tint %s;\n",
	   INTERNAL_INSN_LATENCY_FUNC_NAME, INTERNAL_INSN_CODE_NAME,
	   INTERNAL_INSN2_CODE_NAME, INTERNAL_INSN_CODE_NAME,
	   INTERNAL_INSN2_CODE_NAME);
  fprintf (output_file, "{\n  switch (%s)\n    {\n", INTERNAL_INSN_CODE_NAME);
  for (i = 0; i < description->decls_num; i++)
    {
      curr_decl = description->decls [i];
      if (curr_decl->mode == dm_insn_reserv)
	{
	  fprintf (output_file, "    case %d:\n",
		   curr_decl->decl.insn_reserv.insn_num);
	  if (curr_decl->decl.insn_reserv.bypass_list == NULL)
	    fprintf (output_file, "      return (%s != %s ? %d : 0);\n",
		     INTERNAL_INSN2_CODE_NAME, ADVANCE_CYCLE_VALUE_NAME,
		     curr_decl->decl.insn_reserv.default_latency);
	  else
	    {
	      fprintf (output_file, "      switch (%s)\n        {\n",
		       INTERNAL_INSN2_CODE_NAME);
	      for (curr_bypass = curr_decl->decl.insn_reserv.bypass_list;
		   curr_bypass != NULL;
		   curr_bypass = curr_bypass->next)
		{
		  fprintf (output_file, "        case %d:\n",
			   curr_bypass->in_insn_reserv->insn_num);
		  fprintf (output_file, "          return %d;\n",
			   curr_bypass->latency);
		}
	      fprintf (output_file, "        default:\n");
	      fprintf (output_file,
		       "          return (%s != %s ? %d : 0);\n        }\n",
		       INTERNAL_INSN2_CODE_NAME, ADVANCE_CYCLE_VALUE_NAME,
		       curr_decl->decl.insn_reserv.default_latency);
	      
	    }
	}
    }
  fprintf (output_file, "    default:\n      return 0;\n    }\n}\n\n");
}

/* The function outputs PHR interface function `insn_latency'. */
static void
output_insn_latency_func ()
{
  fprintf (output_file, "int\n%s (%s, %s)\n\trtx %s;\n\trtx %s;\n",
	   INSN_LATENCY_FUNC_NAME, INSN_PARAMETER_NAME, INSN2_PARAMETER_NAME,
	   INSN_PARAMETER_NAME, INSN2_PARAMETER_NAME);
  fprintf (output_file, "{\n  int %s, %s;\n",
	   INTERNAL_INSN_CODE_NAME, INTERNAL_INSN2_CODE_NAME);
  output_internal_insn_code_evaluation (INSN_PARAMETER_NAME,
					INTERNAL_INSN_CODE_NAME);
  output_internal_insn_code_evaluation (INSN2_PARAMETER_NAME,
					INTERNAL_INSN2_CODE_NAME);
  fprintf (output_file, "  return %s (%s, %s);\n}\n\n",
	   INTERNAL_INSN_LATENCY_FUNC_NAME,
	   INTERNAL_INSN_CODE_NAME, INTERNAL_INSN_CODE_NAME);
}

/* The function outputs PHR interface function `print_reservation'. */
static void
output_print_reservation_func ()
{
  struct decl *curr_decl;
  int i;

  fprintf (output_file, "void\n%s (%s, %s)\n\tFILE *%s;\n\trtx %s;\n",
           PRINT_RESERVATION_FUNC_NAME, FILE_PARAMETER_NAME,
           INSN_PARAMETER_NAME, FILE_PARAMETER_NAME,
           INSN_PARAMETER_NAME);
  fprintf (output_file, "{\n  int %s;\n", INTERNAL_INSN_CODE_NAME);
  fprintf (output_file, "\n  if (%s != 0)\n    {\n", INSN_PARAMETER_NAME);
  fprintf (output_file, "      %s = %s (%s);\n",
	   INTERNAL_INSN_CODE_NAME, DFA_INSN_CODE_FUNC_NAME,
	   INSN_PARAMETER_NAME);
  fprintf (output_file, "      if (%s > %s)\n",
	   INTERNAL_INSN_CODE_NAME, ADVANCE_CYCLE_VALUE_NAME);
  fprintf (output_file, "        {\n          fprintf (%s, \"%s\");\n",
           FILE_PARAMETER_NAME, NOTHING_NAME);
  fprintf (output_file, "          return;\n        }\n");
  fprintf (output_file, "    }\n  else\n");
  fprintf (output_file,
           "    {\n      fprintf (%s, \"%s\");\n      return;\n    }\n",
           FILE_PARAMETER_NAME, NOTHING_NAME);
  fprintf (output_file, "  switch (%s)\n    {\n", INTERNAL_INSN_CODE_NAME);
  for (i = 0; i < description->decls_num; i++)
    {
      curr_decl = description->decls [i];
      if (curr_decl->mode == dm_insn_reserv
          && curr_decl != advance_cycle_insn_decl)
	{
          fprintf (output_file,
                   "    case %d:\n", curr_decl->decl.insn_reserv.insn_num);
          fprintf (output_file,
                   "      fprintf (%s, \"%s\");\n",
                   FILE_PARAMETER_NAME,
                   regexp_representation (curr_decl->decl.insn_reserv.regexp));
          finish_regexp_representation ();
        }
    }
  fprintf (output_file, "    default:\n      fprintf (%s, \"%s\");\n    }\n",
           FILE_PARAMETER_NAME, NOTHING_NAME);
  fprintf (output_file, "}\n\n");
}



/* The page contains code for output description file (readable
   representation of original description and generated DFA(s). */

/* The function outputs string representation of IR reservation. */
static void
output_regexp (regexp)
     struct regexp *regexp;
{
  fprintf (output_description_file, "%s", regexp_representation (regexp));
  finish_regexp_representation ();
}

/* Output names of units in LIST separated by comma. */
static void
output_unit_set_el_list (list)
     struct unit_set_el *list;
{
  struct unit_set_el *curr_el;

  for (curr_el = list; curr_el != NULL; curr_el = curr_el->next_unit_set_el)
    {
      if (curr_el != list)
	fprintf (output_description_file, ",");
      fprintf (output_description_file, "%s", curr_el->unit_decl->name);
    }
}

/* The function outputs string representation of IR define_reservation
   and define_insn_reservation. */
static void
output_description ()
{
  struct decl *curr_decl;
  int i;

  for (i = 0; i < description->decls_num; i++)
    {
      curr_decl = description->decls [i];
      if (curr_decl->mode == dm_unit)
	{
	  if (curr_decl->decl.unit.excl_list != NULL)
	    {
	      fprintf (output_description_file, "unit %s exlusion_set: ",
		       curr_decl->decl.unit.name);
	      output_unit_set_el_list (curr_decl->decl.unit.excl_list);
	      fprintf (output_description_file, "\n");
	    }
	  if (curr_decl->decl.unit.req_list != NULL)
	    {
	      fprintf (output_description_file, "unit %s require_set: ",
		       curr_decl->decl.unit.name);
	      output_unit_set_el_list (curr_decl->decl.unit.req_list);
	      fprintf (output_description_file, "\n");
	    }
	}
    }
  fprintf (output_description_file, "\n");
  for (i = 0; i < description->decls_num; i++)
    {
      curr_decl = description->decls [i];
      if (curr_decl->mode == dm_reserv)
	{
          fprintf (output_description_file, "reservation ");
          fprintf (output_description_file, curr_decl->decl.reserv.name);
          fprintf (output_description_file, ": ");
          output_regexp (curr_decl->decl.reserv.regexp);
          fprintf (output_description_file, "\n");
        }
      else if (curr_decl->mode == dm_insn_reserv
	       && curr_decl != advance_cycle_insn_decl)
        {
          fprintf (output_description_file, "insn reservation %s ",
		   curr_decl->decl.insn_reserv.name);
          print_rtl (output_description_file,
                     curr_decl->decl.insn_reserv.condexp);
          fprintf (output_description_file, ": ");
          output_regexp (curr_decl->decl.insn_reserv.regexp);
          fprintf (output_description_file, "\n");
        }
      else if (curr_decl->mode == dm_bypass)
	fprintf (output_description_file, "bypass %d %s %s\n",
		 curr_decl->decl.bypass.latency,
		 curr_decl->decl.bypass.out_insn_name,
		 curr_decl->decl.bypass.in_insn_name);
    }
  fprintf (output_description_file, "\n\f\n");
}

/* The function outputs name of AUTOMATON. */
static void
output_automaton_name (f, automaton)
     FILE *f;
     struct automaton *automaton;
{
  if (automaton->corresponding_automaton_decl == NULL)
    fprintf (f, "#%d", automaton->automaton_order_num);
  else
    fprintf (f, "`%s'", automaton->corresponding_automaton_decl->name);
}

/* Maximal length of line for pretty printing into description
   file. */
#define MAX_LINE_LENGTH 70

/* The function outputs units name belonging to AUTOMATON. */
static void
output_automaton_units (automaton)
     struct automaton *automaton;
{
  struct decl *curr_decl;
  char *name;
  int curr_line_length;
  int there_is_an_automaton_unit;
  int i;

  fprintf (output_description_file, "\n  Coresponding units:\n");
  fprintf (output_description_file, "    ");
  curr_line_length = 4;
  there_is_an_automaton_unit = FALSE;
  for (i = 0; i < description->decls_num; i++)
    {
      curr_decl = description->decls [i];
      if (curr_decl->mode == dm_unit
          && (curr_decl->decl.unit.corresponding_automaton_num
	      == automaton->automaton_order_num))
	{
	  there_is_an_automaton_unit = TRUE;
	  name = curr_decl->decl.unit.name;
	  if (curr_line_length + strlen (name) + 1 > MAX_LINE_LENGTH )
	    {
	      curr_line_length = strlen (name) + 4;
	      fprintf (output_description_file, "\n    ");
	    }
	  else
	    {
	      curr_line_length += strlen (name) + 1;
	      fprintf (output_description_file, " ");
	    }
	  fprintf (output_description_file, name);
	}
    }
  if (!there_is_an_automaton_unit)
    fprintf (output_description_file, "<None>");
  fprintf (output_description_file, "\n\n");
}

/* The following variable is used for forming array of all possible cpu unit
   reservations described by the current DFA state. */
static vla_ptr_t state_reservs;

/* The function forms `state_reservs' for STATE. */
static void
add_state_reservs (state)
     struct state *state;
{
  struct alt_state *curr_alt_state;
  reserv_sets_t reservs;

  if (state->component_states != NULL)
    for (curr_alt_state = state->component_states;
         curr_alt_state != NULL;
         curr_alt_state = curr_alt_state->next_sorted_alt_state)
      add_state_reservs (curr_alt_state->state);
  else
    {
      reservs = state->reservs;
      VLA_PTR_ADD (state_reservs, reservs);
    }
}

/* The function outputs readable represenatation of all out arcs of
   STATE. */
static void
output_state_arcs (state)
     struct state *state;
{
  struct arc *curr_arc;
  struct ainsn *curr_ainsn;
  char *insn_name;
  int curr_line_length;

  for (curr_arc = first_out_arc (state);
       curr_arc != NULL;
       curr_arc = next_out_arc (curr_arc))
    {
      curr_ainsn = curr_arc->insn;
      assert (curr_ainsn->first_insn_with_same_reservs);
      fprintf (output_description_file, "    ");
      curr_line_length = 4;
      while (curr_ainsn != NULL)
        {
          insn_name = curr_ainsn->insn_reserv_decl->name;
          if (curr_line_length + strlen (insn_name) > MAX_LINE_LENGTH)
            {
              if (curr_ainsn != curr_arc->insn)
                {
                  fprintf (output_description_file, ",\n      ");
                  curr_line_length = strlen (insn_name) + 6;
                }
              else
                curr_line_length += strlen (insn_name);
            }
          else
            {
              curr_line_length += strlen (insn_name);
              if (curr_ainsn != curr_arc->insn)
                {
                  curr_line_length += 2;
                  fprintf (output_description_file, ", ");
                }
            }
          fprintf (output_description_file, insn_name);
          curr_ainsn = curr_ainsn->next_same_reservs_insn;
        }
      fprintf (output_description_file, "    %d (%d)\n",
	       curr_arc->to_state->order_state_num,
	       curr_arc->state_alts);
    }
  fprintf (output_description_file, "\n");
}

/* The following function is used for sorting possible cpu unit
   reservation of a DFA state. */
static int
state_reservs_cmp (reservs_ptr_1, reservs_ptr_2)
     const void *reservs_ptr_1;
     const void *reservs_ptr_2;
{
  return reserv_sets_cmp (*(reserv_sets_t *) reservs_ptr_1,
                          *(reserv_sets_t *) reservs_ptr_2);
}

/* The following function is used for sorting possible cpu unit
   reservation of a DFA state. */
static void
remove_state_duplicate_reservs ()
{
  reserv_sets_t *curr_reservs_ptr;
  reserv_sets_t *last_formed_reservs_ptr;

  last_formed_reservs_ptr = NULL;
  for (curr_reservs_ptr = VLA_PTR_BEGIN (state_reservs);
       curr_reservs_ptr <= (reserv_sets_t *) VLA_PTR_LAST (state_reservs);
       curr_reservs_ptr++)
    if (last_formed_reservs_ptr == NULL)
      last_formed_reservs_ptr = curr_reservs_ptr;
    else if (reserv_sets_cmp (*last_formed_reservs_ptr, *curr_reservs_ptr)
             != 0)
      {
        ++last_formed_reservs_ptr;
        *last_formed_reservs_ptr = *curr_reservs_ptr;
      }
  VLA_PTR_SHORTEN (state_reservs,
                   curr_reservs_ptr - last_formed_reservs_ptr - 1);
}

/* The following function output readable representation of DFA(s)
   state used for fast recognition of pipeline hazards.  State is
   described by possible (current and scehduled) cpu unit
   reservations. */
static void
output_state (state)
     struct state *state;
{
  reserv_sets_t *curr_reservs_ptr;

  VLA_PTR_CREATE (state_reservs, 150, "state reservations");
  fprintf (output_description_file, "  State #%d", state->order_state_num);
  fprintf (output_description_file,
	   state->new_cycle_p ? " (new cycle)\n" : "\n");
  add_state_reservs (state);
  qsort (VLA_PTR_BEGIN (state_reservs), VLA_PTR_LENGTH (state_reservs),
         sizeof (reserv_sets_t), state_reservs_cmp);
  remove_state_duplicate_reservs ();
  for (curr_reservs_ptr = VLA_PTR_BEGIN (state_reservs);
       curr_reservs_ptr <= (reserv_sets_t *) VLA_PTR_LAST (state_reservs);
       curr_reservs_ptr++)
    {
      fprintf (output_description_file, "    ");
      output_reserv_sets (output_description_file, *curr_reservs_ptr);
      fprintf (output_description_file, "\n");
    }
  fprintf (output_description_file, "\n");
  output_state_arcs (state);
  VLA_PTR_DELETE (state_reservs);
}

/* The following function output readable representation of
   DFAs used for fast recognition of pipeline hazards. */
static void
output_automaton_descriptions ()
{
  struct automaton *curr_automaton;

  for (curr_automaton = description->first_automaton;
       curr_automaton != NULL;
       curr_automaton = curr_automaton->next_automaton)
    {
      fprintf (output_description_file, "\nAutomaton ");
      output_automaton_name (output_description_file, curr_automaton);
      fprintf (output_description_file, "\n");
      output_automaton_units (curr_automaton);
      pass_states (curr_automaton, output_state);
    }
}



/* The page contains top level function for generation DFA(s) used for
   PHR. */

/* The function outputs statistics about work of different phases of
   DFA generator. */
static void
output_statistics (f)
     FILE *f;
{
  struct automaton *curr_automaton;
#ifndef NDEBUG
  int transition_comb_vect_els = 0;
  int transition_full_vect_els = 0;
  int state_alts_comb_vect_els = 0;
  int state_alts_full_vect_els = 0;
  int min_issue_delay_comb_vect_els = 0;
  int min_issue_delay_full_vect_els = 0;
#endif

  for (curr_automaton = description->first_automaton;
       curr_automaton != NULL;
       curr_automaton = curr_automaton->next_automaton)
    {
      fprintf (f, "\nAutomaton ");
      output_automaton_name (f, curr_automaton);
      fprintf (f, "\n    %5d NDFA states,          %5d NDFA arcs\n",
	       curr_automaton->NDFA_states_num, curr_automaton->NDFA_arcs_num);
      fprintf (f, "    %5d DFA states,           %5d DFA arcs\n",
	       curr_automaton->DFA_states_num, curr_automaton->DFA_arcs_num);
      if (!no_minimization_flag)
	fprintf (f, "    %5d minimal DFA states,   %5d minimal DFA arcs\n",
		 curr_automaton->minimal_DFA_states_num,
		 curr_automaton->minimal_DFA_arcs_num);
      fprintf (f, "    %5d all insns      %5d insn equivalence classes\n",
	       description->insns_num, curr_automaton->insn_equiv_classes_num);
#ifndef NDEBUG
      fprintf (f, "%5d transition comb vector els, %5d trans table els: %s\n",
               VLA_HWINT_LENGTH (curr_automaton->trans_table->comb_vect),
               VLA_HWINT_LENGTH (curr_automaton->trans_table->full_vect),
               (comb_vect_p (curr_automaton->trans_table)
                ? "use comb vect" : "use simple vect"));
      fprintf
        (f, "%5d state alts comb vector els, %5d state alts table els: %s\n",
         VLA_HWINT_LENGTH (curr_automaton->state_alts_table->comb_vect),
         VLA_HWINT_LENGTH (curr_automaton->state_alts_table->full_vect),
         (comb_vect_p (curr_automaton->state_alts_table)
          ? "use comb vect" : "use simple vect"));
      fprintf
        (f, "%5d min delay comb vector els, %5d min delay table els: %s\n",
         VLA_HWINT_LENGTH (curr_automaton->min_issue_delay_table->comb_vect),
         VLA_HWINT_LENGTH (curr_automaton->min_issue_delay_table->full_vect),
         (comb_vect_p (curr_automaton->min_issue_delay_table)
          ? "use comb vect" : "use simple vect"));
      transition_comb_vect_els
        += VLA_HWINT_LENGTH (curr_automaton->trans_table->comb_vect);
      transition_full_vect_els 
        += VLA_HWINT_LENGTH (curr_automaton->trans_table->full_vect);
      state_alts_comb_vect_els
        += VLA_HWINT_LENGTH (curr_automaton->state_alts_table->comb_vect);
      state_alts_full_vect_els
        += VLA_HWINT_LENGTH (curr_automaton->state_alts_table->full_vect);
      min_issue_delay_comb_vect_els
        += VLA_HWINT_LENGTH (curr_automaton->min_issue_delay_table->comb_vect);
      min_issue_delay_full_vect_els
        += VLA_HWINT_LENGTH (curr_automaton->min_issue_delay_table->full_vect);
#endif
    }
#ifndef NDEBUG
  fprintf (f, "\n%5d all allocated states,     %5d all allocated arcs\n",
	   allocated_states_num, allocated_arcs_num);
  fprintf (f, "%5d all allocated alternative states\n",
           allocated_alt_states_num);
  fprintf (f, "%5d all transition comb vector els, %5d all trans table els\n",
	   transition_comb_vect_els, transition_full_vect_els);
  fprintf
    (f, "%5d all state alts comb vector els, %5d all state alts table els\n",
     state_alts_comb_vect_els, state_alts_full_vect_els);
  fprintf
    (f, "%5d all min delay comb vector els, %5d all min delay table els\n",
     min_issue_delay_comb_vect_els, min_issue_delay_full_vect_els);
  fprintf (f, "%5d locked states num\n", locked_states_num);
#endif
}

/* The function output times of work of different phases of DFA
   generator. */
static void
output_time_statistics (f)
     FILE *f;
{
  fprintf (f, "\n  transformation: ");
  print_active_time (f, transform_time);
  fprintf (f, (!ndfa_flag ? ", building DFA: " : ", building NDFA: "));
  print_active_time (f, NDFA_time);
  if (ndfa_flag)
    {
      fprintf (f, ", NDFA -> DFA: ");
      print_active_time (f, NDFA_to_DFA_time);
    }
  fprintf (f, "\n  DFA minimization: ");
  print_active_time (f, minimize_time);
  fprintf (f, ", making insn equivalence: ");
  print_active_time (f, equiv_time);
  fprintf (f, "\n all automaton generation: ");
  print_active_time (f, automaton_generation_time);
  fprintf (f, ", output: ");
  print_active_time (f, output_time);
  fprintf (f, "\n");
}

/* The function generates DFA (deterministic finate state automaton)
   for fast recognition of pipeline hazards.  No errors during
   checking must be fixed before this function call. */
static void
generate ()
{
  automata_num = split_argument;
  if (description->units_num < automata_num)
    automata_num = description->units_num;
  initiate_states ();
  initiate_arcs ();
  initiate_pass_states ();
  initiate_excl_sets ();
  initiate_req_sets ();
  automaton_generation_time = create_ticker ();
  transform_time = create_ticker ();
  add_advance_cycle_insn_decl ();
  transform_insn_regexps ();
  ticker_off (&transform_time);
  create_automata ();
  ticker_off (&automaton_generation_time);
}



/* The following function creates attribute which order number of insn
   in pipeline hazard description translator. */
static void
make_insn_alts_attr ()
{
  int i, insn_num;
  struct decl *curr_decl;
  rtx condexp;

  condexp = rtx_alloc (COND);
  XVEC (condexp, 0) = rtvec_alloc ((description->insns_num - 1) * 2);
  XEXP (condexp, 1) = make_numeric_value (0);
  for (i = insn_num = 0; i < description->decls_num; i++)
    {
      curr_decl = description->decls [i];
      if (curr_decl->mode == dm_insn_reserv
          && curr_decl != advance_cycle_insn_decl)
	{
          XVECEXP (condexp, 0, 2 * insn_num)
            = curr_decl->decl.insn_reserv.condexp;
          XVECEXP (condexp, 0, 2 * insn_num + 1)
            = make_numeric_value (curr_decl->decl.insn_reserv
				  .transformed_regexp
				  ->regexp.oneof.regexps_num);
          insn_num++;
        }
    }
  assert (description->insns_num == insn_num + 1);
  make_internal_attr (attr_printf (sizeof ("*")
				   + strlen (INSN_ALTS_FUNC_NAME) + 1,
				   "*%s", INSN_ALTS_FUNC_NAME),
		      condexp, 0);
}



/* The following function creates attribute which is order number of
   insn in pipeline hazard description translator. */
static void
make_dfa_insn_code_attr ()
{
  int i, insn_num;
  struct decl *curr_decl;
  rtx condexp;

  condexp = rtx_alloc (COND);
  XVEC (condexp, 0) = rtvec_alloc ((description->insns_num - 1) * 2);
  XEXP (condexp, 1) = make_numeric_value (advance_cycle_insn_decl
					  ->decl.insn_reserv.insn_num + 1);
  for (i = insn_num = 0; i < description->decls_num; i++)
    {
      curr_decl = description->decls [i];
      if (curr_decl->mode == dm_insn_reserv
          && curr_decl != advance_cycle_insn_decl)
	{
          XVECEXP (condexp, 0, 2 * insn_num)
            = curr_decl->decl.insn_reserv.condexp;
          XVECEXP (condexp, 0, 2 * insn_num + 1)
            = make_numeric_value (curr_decl->decl.insn_reserv.insn_num);
          insn_num++;
        }
    }
  assert (description->insns_num == insn_num + 1);
  make_internal_attr (attr_printf (sizeof ("*")
				   + strlen (DFA_INSN_CODE_FUNC_NAME) + 1,
				   "*%s", DFA_INSN_CODE_FUNC_NAME),
		      condexp, 0);
}



/* The following function creates attribute which order number of insn
   in pipeline hazard description translator. */
static void
make_default_insn_latency_attr ()
{
  int i, insn_num;
  struct decl *curr_decl;
  rtx condexp;

  condexp = rtx_alloc (COND);
  XVEC (condexp, 0) = rtvec_alloc ((description->insns_num - 1) * 2);
  XEXP (condexp, 1) = make_numeric_value (0);
  for (i = insn_num = 0; i < description->decls_num; i++)
    {
      curr_decl = description->decls [i];
      if (curr_decl->mode == dm_insn_reserv
          && curr_decl != advance_cycle_insn_decl)
	{
          XVECEXP (condexp, 0, 2 * insn_num)
            = curr_decl->decl.insn_reserv.condexp;
          XVECEXP (condexp, 0, 2 * insn_num + 1)
            = make_numeric_value (curr_decl->decl.insn_reserv.default_latency);
          insn_num++;
        }
    }
  assert (description->insns_num == insn_num + 1);
  make_internal_attr (attr_printf (sizeof ("*")
				   + strlen (INSN_DEFAULT_LATENCY_FUNC_NAME)
				   + 1, "*%s", INSN_DEFAULT_LATENCY_FUNC_NAME),
		      condexp, 0);
}



/* The following function creates attribute which returns 1 if given
   output insn has bypassing and 0 otherwise.. */
static void
make_bypass_attr ()
{
  int i, bypass_insn;
  int bypass_insns = 0;
  struct decl *curr_decl;
  rtx result_rtx;
  
  for (i = 0; i < description->decls_num; i++)
    {
      curr_decl = description->decls [i];
      if (curr_decl->mode == dm_insn_reserv
	  && curr_decl->decl.insn_reserv.condexp != NULL
	  && curr_decl->decl.insn_reserv.bypass_list != NULL)
	bypass_insns++;
    }
  if (bypass_insns == 0)
    result_rtx = make_numeric_value (0);
  else
    {
      result_rtx = rtx_alloc (COND);
      XVEC (result_rtx, 0) = rtvec_alloc (bypass_insns * 2);
      XEXP (result_rtx, 1) = make_numeric_value (0);
    }
  for (i = bypass_insn = 0; i < description->decls_num; i++)
    {
      curr_decl = description->decls [i];
      if (curr_decl->mode == dm_insn_reserv)
	{
          XVECEXP (result_rtx, 0, 2 * bypass_insn)
	    = curr_decl->decl.insn_reserv.condexp;
          XVECEXP (result_rtx, 0, 2 * bypass_insn + 1)
	    = make_numeric_value (1);
          bypass_insn++;
        }
    }
  make_internal_attr (attr_printf (sizeof ("*")
				   + strlen (BYPASS_P_FUNC_NAME) + 1,
				   "*%s", BYPASS_P_FUNC_NAME),
		      result_rtx, 0);
}



/* This page mainly contains top level functions of pipeline hazards
   description translator. */

/* The following macro value is suffix of name of description file of
   pipeline hazards description translator. */
#define STANDARD_OUTPUT_DESCRIPTION_FILE_SUFFIX ".dfa"

/* The function returns suffix of given file name.  The returned
   string can not be changed.  */
static const char *
file_name_suffix (file_name)
     const char *file_name;
{
  const char *last_period;

  for (last_period = NULL; *file_name != '\0'; file_name++)
    if (*file_name == '.')
      last_period = file_name;
  return (last_period == NULL ? file_name : last_period);
}

/* The function returns base name of given file name, i.e. pointer to
   first char after last `/' (or `\' for WIN32) in given file name,
   given file name itself if the directory name is absent.  The
   returned string can not be changed. */
static const char *
base_file_name (file_name)
     const char *file_name;
{
  int directory_name_length;

  directory_name_length = strlen (file_name);
#ifdef WIN32
  while (directory_name_length >= 0 && file_name[directory_name_length] != '/'
         && file_name[directory_name_length] != '\\')
#else
  while (directory_name_length >= 0 && file_name[directory_name_length] != '/')
#endif
    directory_name_length--;
  return file_name + directory_name_length + 1;
}

/* The following is top level function to initialize the work of
   pipeline hazards description translator. */
static void
initiate_automaton_gen (argc, argv)
     int argc;
     char **argv;
{
  const char *base_name;
  int i;

  ndfa_flag = FALSE;
  split_argument = 0;  /* default value */
  no_minimization_flag = FALSE;
  time_flag = FALSE;
  v_flag = FALSE;
  for (i = 2; i < argc; i++)
    if (strcmp (argv [i], "-no-minimization") == 0)
      no_minimization_flag = TRUE;
    else if (strcmp (argv [i], "-time") == 0)
      time_flag = TRUE;
    else if (strcmp (argv [i], "-v") == 0)
      v_flag = TRUE;
    else if (strcmp (argv [i], "-ndfa") == 0)
      ndfa_flag = TRUE;
    else if (strcmp (argv [i], "-split") == 0)
      {
	if (i + 1 >= argc)
	  fatal ("-split has no argument.");
	fatal ("option `-split' has not been implemented yet\n");
	/* split_argument = atoi (argument_vect [i + 1]); */
      }
  VLA_PTR_CREATE (decls, 150, "decls");
  IR_START_ALLOC ();
  initiate_automaton_decl_table ();
  initiate_insn_decl_table ();
  initiate_decl_table ();
  output_file = stdout;
  output_description_file = NULL;
  base_name = base_file_name (argv[1]);
  IR_TOP_ADD_MEMORY (base_name,
                     strlen (base_name)
                     - strlen (file_name_suffix (base_name)));
  IR_TOP_ADD_STRING (STANDARD_OUTPUT_DESCRIPTION_FILE_SUFFIX);
  IR_TOP_ADD_BYTE ('\0');
  output_description_file_name = IR_TOP_BEGIN ();
  IR_TOP_FINISH ();
}

/* The following is top level function to generate automat(a,on) for
   fast recognition of pipeline hazards. */
static void
expand_automata ()
{
  int i;

  if (VLA_PTR_LENGTH (decls) != 0)
    {
      description = create_node (sizeof (struct description)
				 + sizeof (struct decl *)
				 /* One entry for cycle advancing insn. */
				 * VLA_PTR_LENGTH (decls));
      description->decls_num = VLA_PTR_LENGTH (decls);
      for (i = 0; i < description->decls_num; i++)
	description->decls [i] = VLA_PTR (decls, i);
      all_time = create_ticker ();
      check_time = create_ticker ();
      check_all_description ();
      ticker_off (&check_time);
      generation_time = create_ticker ();
      if (!have_error)
	{
	  generate ();
	  make_dfa_insn_code_attr ();
	  make_insn_alts_attr ();
	  make_default_insn_latency_attr ();
	  make_bypass_attr ();
	}
      ticker_off (&generation_time);
      ticker_off (&all_time);
    }
}

/* The following is top level function to output PHR and to finish
   work with pipeline description translator. */
static void
write_automata ()
{
  if (have_error)
    fatal ("Errors in DFA description");
  ticker_on (&all_time);
  output_time = create_ticker ();
  output_tables ();
  output_chip_definitions ();
  output_max_insn_queue_index_def ();
  output_internal_min_issue_delay_func ();
  output_internal_trans_func ();
  output_trans_func ();
  output_internal_state_alts_func ();
  output_state_alts_func ();
  output_min_issue_delay_func ();
  output_internal_dead_lock_func ();
  output_dead_lock_func ();
  output_internal_reset_func ();
  output_reset_func ();
  output_min_insn_conflict_delay_func ();
  output_internal_insn_latency_func ();
  output_insn_latency_func ();
  output_print_reservation_func ();
  if (v_flag)
    {
      output_description_file = fopen (output_description_file_name, "w");
      if (output_description_file == NULL)
	{
	  perror (output_description_file_name);
	  exit (FATAL_EXIT_CODE);
	}
      output_description ();
      output_automaton_descriptions ();
      output_statistics (output_description_file);
    }
  output_statistics (stderr);
  ticker_off (&output_time);
  output_time_statistics (stderr);
  finish_states ();
  finish_arcs ();
  if (time_flag)
    {
      fprintf (stderr, "Summary:\n");
      fprintf (stderr, "  check time ");
      print_active_time (stderr, check_time);
      fprintf (stderr, ", generation time ");
      print_active_time (stderr, generation_time);
      fprintf (stderr, ", all time ");
      print_active_time (stderr, all_time);
      fprintf (stderr, "\n");
    }
  /* Finish all work. */
  if (output_description_file != NULL)
    {
      fflush (output_description_file);
      if (ferror (stdout) != 0)
	fatal ("Error in writing DFA description file %s",
               output_description_file_name);
      fclose (output_description_file);
    }
  finish_automaton_decl_table ();
  finish_insn_decl_table ();
  finish_decl_table ();
  IR_STOP_ALLOC();
  if (have_error && output_description_file != NULL)
    remove (output_description_file_name);

}
/* END CYGNUS LOCAL -- vmakarov/dfa */
