#!/usr/bin/perl
#
# Copyright Hewlett Packard 2012 - 2015
#
# version = 1.26
# This script analyzes captured strings in specific files. 
# See def/capture_definitions.txt for the search criteria and file specifications.
#
# Modified: 25-Aug-2014 V1.23 Gary Sachs (HP)
# Modified: 20-Nov-2014 V1.25-0 Gary Sachs (HP)
# Modified: 29-Apr-2015 V1.25-1 Gary Sachs (HP)
# 
 
use strict;
use warnings;
use Data::Dumper;
    $Data::Dumper::Sortkeys = 1;


my %search_strings = (
   "ct_mgmt_get_port_info"								=> \&analyze_ct_mgmt_get_port_info,
   "sdt_cdb_retry_func"									=> \&analyze_sdt_cdb_retry_func,
   "ELS\/CT timeout"									=> \&analyze_els_ct_timeout,
   "els_ct timeout"									=> \&analyze_els_ct_timeout,
   "scsi_send_cmnd_async"								=> \&analyze_scsi_send_cmnd_async,
   "scsi_cmnd_retry"									=> \&analyze_scsi_cmnd_retry,
   "Failed to block IO to the volumes in group" 					=> \&analyze_rc_vv_locks,
   'kernel: BUG: soft lockup \- CPU.\d+ stuck for \d+s\!\s+\[sdt_getlbastatus\:\d+\]'	=> \&analyze_getlbastatus,
   'RTPG SC \S+ EVT ID \d+ NOT FOUND"'							=> \&analyze_RTPG_count,
   'Qlen \d+ exceeding threshold of \d+'						=> \&analyze_Qlen_events,
   'End run on DDS'                                                                     => \&analyze_gc_abandonment,
   'Eagle link error hw_eagle:\d+ Link bringup fatal error from node \d+ to \d+'        => \&analyze_link_errors,
   'Eagle link error hw_eagle:\d+ Fatal link error between member nodes \d+ to \d+'     => \&analyze_link_errors,
   'CM Error: Link \d+ to node \d+ \(step \d+ DOWN\):'                                  => \&analyze_cm_errors,
   #'Unresponsive IOCTL sw_os\s+IOCTL sccmd_portop unresponsive for \d+ seconds'         => \&analyze_sccmd_portop,
   #'Unresponsive IOCTL sw_os\s+IOCTL \w+ unresponsive for \d+ seconds'                  => \&analyze_sccmd_portop,
   'Unresponsive IOCTL sw_os\s+IOCTL \w+ unresponsive for \d+ seconds'                  => \&analyze_unresponsive_ioctls,
   'Too many events are being generated'                                                => \&analyze_too_many_events,
   'failure: drive capacity is unsupported'                                             => \&analyze_drive_capacity,
   'Failed\s*\(Preserved'                             			                => \&analyze_failed_preserved,
   'ocs_hal_command: MQ command issued sent in non-active state, HAL state='            => \&analyze_unresponsive_hal,
   'NEMOE Get (?:[\w\s]+) failed for node \d+: RC_TIMEOUT'                              => \&analyze_rc_timeouts,
   '\{config_lock_tattler \}\s+.+?\s+for an extended period of time \(\d+ seconds\)'    => \&analyze_config_lock_tattler,
   'Took \d+ seconds to acquire the config lock \(\S+\)'                                => \&analyze_config_lock_tattler,
   'TOC update required \d+ seconds'                                                    => \&analyze_toc_delays,
   'event handling appears to be unresponsive'                                          => \&analyze_unresponsive_event_handling,
   #'End run on DDS .* status 1'                                                         => \&analyze_dedup_issues,
);

# Read in input values
my $common      = $ARGV[0];		# inex_common object
my $type        = $ARGV[1];		# Type of configuration objects
my $ptr         = $ARGV[2];		# Hash of configuration objects of type $ARGV[1]
my $syssn       = $ARGV[3];		# Serial Number of the system
my $model       = $ARGV[4];		# Model of the system (F200, V400, etc)
my $version     = $ARGV[5];		# Baselevel currently running
my $baselevel   = $ARGV[6];		# Baselevel currently running
my $patches     = $ARGV[7];		# List of installed patches
my $insploredir = $ARGV[8];		# Directory name containing the decompressed Insplore

# Define local variables, global in this program though
my %local_hash  = ();			# Hash to store local stuff
my $lptr	= \%local_hash;
my $save_debug  = $common->{debug}->{level};
my $string      = undef;

# In order to facilitate debugging and see the structure of the hash, uncomment the following 2 lines. 
# $common->{debug}->{level} = TRACE_MEMORY;
# $common->dump_memory($ptr, "cfganal_captures: ptr");

# Report issues using the following method:
# $common->report($syssn, <analysis_code>, <analysis_code_p1>, <analysis_code_p2>, ..);

$common->trace(TRACE_ROUTINES, "---> cfganal_captures $type, $syssn, $model, $version, $baselevel, $patches");

%{$lptr->{models}} = ();		# Init the local PD Model hash
@{$lptr->{reports}} = ();
%{$lptr->{nemoe}} = ();
%{$lptr->{ioctls}} = ();

foreach $string (sort keys(%search_strings)) {
	if (exists($ptr->{strings}->{$string})) {
	    $search_strings{$string}($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $string);
	}
}

#printf STDERR "CAPTURE STRINGS:\n%s\n", Dumper($ptr->{strings});

$common->trace(TRACE_ROUTINES, "<--- cfganal_captures");
$common->{debug}->{level} = $save_debug;
return 1;

sub analyze_unresponsive_event_handling {
    my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $string) = @_;
    $common->trace(TRACE_ROUTINES, "---> cfganal_captures::analyze_unresponsive_event_handling");
    
    @{$lptr->{reports}} = ();
    
    my $report = string_count_occurrences ($common, $ptr, $string, "showeventlog_-d_-debug_-oneline.out");
    #
    if (scalar(@{$lptr->{reports}}) > 0) {
	foreach my $rstring (@{$lptr->{reports}})
	{
	    $common->report($syssn, 2016, $rstring);
	}
    }
    #
    $common->trace(TRACE_ROUTINES, "<--- cfganal_captures::analyze_unresponsive_event_handling");
    return;
}

sub analyze_toc_delays {
    my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $string) = @_;
    $common->trace(TRACE_ROUTINES, "---> cfganal_captures::analyze_toc_delays");
    
    @{$lptr->{reports}} = ();
    
    foreach my $fname (sort keys%{$ptr->{strings}->{$string}})
    {
	if ($fname !~ m/sysmgr/i){next;}
	#
	if (!$ptr->{strings}->{$string}->{$fname}->{count}){next;}
	#
	my $report = string_count_occurrences ($common, $ptr, $string, $fname);
	#
	
    }
    if (scalar(@{$lptr->{reports}}) > 0) {
	foreach (@{$lptr->{reports}}) { $common->report($syssn, 2015, $_); }
    }
    #
    $common->trace(TRACE_ROUTINES, "<--- cfganal_captures::analyze_toc_delays");
    return;
}

sub analyze_config_lock_tattler {
    my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $string) = @_;
    $common->trace(TRACE_ROUTINES, "---> cfganal_captures::analyze_config_lock_tattler");
    
    my $msgcode = 2013;
    if ($string =~ m/acquire/) { $msgcode = 2014;  }

    
    @{$lptr->{reports}} = ();
    
    foreach my $fname (sort keys%{$ptr->{strings}->{$string}})
    {
	if ($fname !~ m/sysmgr/i){next;}
	#
	if (!$ptr->{strings}->{$string}->{$fname}->{count}){next;}
	#
	my $report = string_count_occurrences ($common, $ptr, $string, $fname);
	#
    }
    #
    if (scalar(@{$lptr->{reports}}) > 0) {
	foreach my $rstring (@{$lptr->{reports}})
	{
	    my $msgcode = 2013;
	    if ($rstring =~ m/acquire/) { $msgcode = 2014;  }
	    $common->report($syssn, $msgcode, $rstring);
	}
    }
    #
    $common->trace(TRACE_ROUTINES, "<--- cfganal_captures::analyze_config_lock_tattler");
    return;
}

sub analyze_rc_timeouts {
    my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $string) = @_;
    $common->trace(TRACE_ROUTINES, "---> cfganal_captures::analyze_rc_timeouts");

    @{$lptr->{reports}} = ();
    #
    #printf STDERR "CFGANAL_CPATURES::analysze_rc_timeouts: $string|\n%s\n", Dumper($ptr->{strings}->{$string});
    #printf STDERR "CFGANAL_CPATURES::analysze_rc_timeouts2: $string|\n%s\n", Dumper($common->{captures}->{$syssn}->{observed}->{ts});
    #
    my $rccnt = 0;
    foreach my $fname (sort keys%{$ptr->{strings}->{$string}})
    {
	if ($fname !~ m/sysmgr/i){next;}
	#
	if (!$ptr->{strings}->{$string}->{$fname}->{count}){next;}
	#
	$rccnt += $ptr->{strings}->{$string}->{$fname}->{count};
    }
    #
    if ($rccnt) { $common->report($syssn, 2019, $rccnt); }

    $common->trace(TRACE_ROUTINES, "<--- cfganal_captures::analyze_rc_timeouts");
    return 1;
}

sub analyze_failed_preserved
{
    my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $string) = @_;
    $common->trace(TRACE_ROUTINES, "---> cfganal_captures::analyze_failed_preserved");
    #
    %{$lptr->{fpreserved}} = ();
    #
    foreach my $fn  (sort keys %{$ptr->{strings}->{$string}})
    {
	#printf "ANALYZE_FAILED_PRESERVED: sn: !$syssn! | $string | $fn | Count: %s\n", $ptr->{strings}->{$string}->{$fn}->{count};
	if (!$ptr->{strings}->{$string}->{$fn}->{count}){next;}
	#
	foreach my $xepoch (sort keys %{$ptr->{strings}->{$string}->{$fn}->{ts}})
	{
	    #print Dumper($common->{captures}->{$syssn}->{observed}->{ts}->{$xepoch});

	    for (my $j=0; $j < scalar(@{$common->{captures}->{$syssn}->{observed}->{ts}->{$xepoch}}); $j++)
	    {
		
		if ($common->{captures}->{$syssn}->{observed}->{ts}->{$xepoch}[$j] !~ m/Virtual Volume (\d+)\((\S+)\) Failed \(Preserved \{(\S+)\}\)/) { next; }
		#
		my $node = $common->{captures}->{$syssn}->{observed}->{ts}->{$xepoch}[$j-3];
		my $vvid = $1;
		my $vvname = $2;
		my $pstatus = $3;
		#
		if (!exists($lptr->{fpreserved}->{$xepoch}) ||
		    !exists($lptr->{fpreserved}->{$xepoch}->{$vvid}) )
		{
		    $lptr->{fpreserved}->{$xepoch}->{$vvid}->{node} = $node;
		    $lptr->{fpreserved}->{$xepoch}->{$vvid}->{name} = $vvname;
		    $lptr->{fpreserved}->{$xepoch}->{$vvid}->{status} = $pstatus;
		}
	    }
	}
	
    }
    #
    foreach my $xepoch (sort keys %{$lptr->{fpreserved}})
    {
	foreach my $vid (sort keys %{$lptr->{fpreserved}->{$xepoch}})
	{
	    $common->report($syssn, 2004, $lptr->{fpreserved}->{$xepoch}->{$vid}->{node},
					  $common->convert_epoch_to_external_datetime($xepoch),
					  $vid,
					  $lptr->{fpreserved}->{$xepoch}->{$vid}->{name},
					  $lptr->{fpreserved}->{$xepoch}->{$vid}->{status});
	}
    }
    $common->trace(TRACE_ROUTINES, "<--- cfganal_captures::analyze_failed_preserved");
    return 1;
}

sub analyze_too_many_events {
    my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $string) = @_;
    #
    $common->trace(TRACE_ROUTINES, "---> cfganal_captures::analyze_too_many_events");
    #
    @{$lptr->{reports}} = ();
    foreach my $fn  (sort keys %{$ptr->{strings}->{$string}})
    {
	#printf "CM_LINK_ERRORS: sn: !$syssn! | $string | $fn | Count: %s\n", $ptr->{strings}->{$string}->{$fn}->{count};
	if (!$ptr->{strings}->{$string}->{$fn}->{count}){next;}
	#
	my $report = analyze_string($common, $ptr, $string, $fn, 2, (24*60));	# Check for 2 occurences in 24 hours (24 x 60 minutes)
	#if (defined($report)) {
	if ($report) { foreach (@{$lptr->{reports}}) { $common->report($syssn, 3000, $_); } }
    }
    $common->trace(TRACE_ROUTINES, "<--- cfganal_captures::analyze_too_many_events");
    return 1;
}

sub analyze_unresponsive_ioctls {
    my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $string) = @_;
    
    $common->trace(TRACE_ROUTINES, "---> cfganal_captures::analyze_unresponsive_ioctls");
    #printf STDERR "UNRESPONSIVE:\n%s\n", Dumper($ptr->{strings}->{$string});
    
    @{$lptr->{reports}} = ();
    foreach my $fn  (sort keys %{$ptr->{strings}->{$string}})
    {
	#printf "CM_LINK_ERRORS: sn: !$syssn! | $string | $fn | Count: %s\n", $ptr->{strings}->{$string}->{$fn}->{count};
	if (!$ptr->{strings}->{$string}->{$fn}->{count}){next;}
	
	#
	foreach my $xepoch (sort keys %{$ptr->{strings}->{$string}->{$fn}->{ts}})
	{
	    for (my $j=0; $j < scalar(@{$common->{captures}->{$syssn}->{observed}->{ts}->{$xepoch}}); $j++)
	    {
		if ($common->{captures}->{$syssn}->{observed}->{ts}->{$xepoch}[$j] !~ m/Unresponsive IOCTL sw_os\s+IOCTL (\w+) unresponsive for \d+ seconds/){next;}
		#
		if (!exists($ptr->{strings}->{$1}))
		{
		    $lptr->{ioctls}->{$1} = 0;
		    $ptr->{strings}->{$1}->{$fn}->{count} = 0;
		    %{$ptr->{strings}->{$1}->{$fn}->{ts}} = ();
		}
		#
		$ptr->{strings}->{$1}->{$fn}->{count}++;
		#
		if (!exists($ptr->{strings}->{$1}->{$fn}->{ts}->{$xepoch})) { $ptr->{strings}->{$1}->{$fn}->{ts}->{$xepoch} = 0; }
		$ptr->{strings}->{$1}->{$fn}->{ts}->{$xepoch}++;
	    }
	}
    }
    #
    #
    #
    foreach my $iname (sort keys %{$lptr->{ioctls}})
    {
	#print "UNRESPONSIVE IOCT: $iname...\n";
	my $report = 0;
	 @{$lptr->{reports}} = ();
	if ($iname eq 'sccmd_portop')
	{ 
	    $report = analyze_string($common, $ptr, $iname, "showeventlog_-d_-debug_-oneline.out", 50, (24*60));	# Check for 50 occurences in 24 hours (24 x 60 minutes)
	    #if (defined($report)) {
	    if ($report)  
	    {
		if ($version >= 313) {
		    #
		    # Added 313.MU1 20-Nov-2014 Gary Sachs, the problem still exists on 313.MU1
		    # 3-Mar-2015: Spoke with Vish, this seems to be a problem with 313 and onward for the time being.
		    #
		    #$common->report($syssn, 1012, "enable_skip_fabric_ioctls.sh", 5988, $report);
		    foreach (@{$lptr->{reports}}) { $common->report($syssn, 1012, "enable_skip_fabric_ioctls.sh", 5988, $_); }
		}
	    }
	}
	else
	{
	    #printf STDERR "UNRESPONSIVE IOCT: $iname\n%s\n", Dumper($ptr->{strings}->{$iname});
	    $report = string_count_occurrences ($common, $ptr, $iname, "showeventlog_-d_-debug_-oneline.out");
	    #
	    if (scalar(@{$lptr->{reports}}) > 0)
	    {
		foreach my $rstring (@{$lptr->{reports}})
		{
		    $common->report($syssn, 3001, $iname, $rstring);
		}
	    }
	}
    }
    $common->trace(TRACE_ROUTINES, "<--- cfganal_captures::analyze_unresponsive_ioctls");
    return 1;
}

sub analyze_sccmd_portop {
    my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $string) = @_;
    $common->trace(TRACE_ROUTINES, "---> cfganal_captures::analyze_sccmd_portop");

    @{$lptr->{reports}} = ();
    my $report = analyze_string($common, $ptr, $string, "showeventlog_-d_-debug_-oneline.out", 50, (24*60));	# Check for 50 occurences in 24 hours (24 x 60 minutes)
    #if (defined($report)) {
    if ($report) {
	if ($version >= 313) {
	    #
	    # Added 313.MU1 20-Nov-2014 Gary Sachs, the problem still exists on 313.MU1
	    # 3-Mar-2015: Spoke with Vish, this seems to be a problem with 313 and onward for the time being.
	    #
	    #$common->report($syssn, 1012, "enable_skip_fabric_ioctls.sh", 5988, $report);
	    foreach (@{$lptr->{reports}}) { $common->report($syssn, 1012, "enable_skip_fabric_ioctls.sh", 5988, $_); }
	}
    }
    $common->trace(TRACE_ROUTINES, "<--- cfganal_captures::analyze_sccmd_portop");
    return 1;
}

sub analyze_cm_errors {
    my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $string) = @_;
    $common->trace(TRACE_ROUTINES, "---> cfganal_captures::analyze_cm_errors");

    foreach my $fn  (sort keys %{$ptr->{strings}->{$string}})
    {
	#printf "CM_LINK_ERRORS: sn: !$syssn! | $string | $fn | Count: %s\n", $ptr->{strings}->{$string}->{$fn}->{count};
	if (!$ptr->{strings}->{$string}->{$fn}->{count}){next;}
	
	#
	foreach my $xepoch (sort keys %{$ptr->{strings}->{$string}->{$fn}->{ts}})
	{
	    #print Dumper($common->{captures}->{$syssn}->{observed}->{ts}->{$xepoch});

	#    foreach my $ctext (@{$common->{captures}->{$syssn}->{observed}->{ts}->{$xepoch}})
	#    {
	#	if ($ctext !~ m/CM Error\:/){next;}
	#	printf ": $ctext\n";
	#	my $node = '<unk>';
	#	if ($ctext =~ m/$syssn-(\d+)\s+kernel/){$node = $1;}
	#	#printf "cfganal_captures::analyze_cm_link_errors: $node | $ctext\n";
	#	$common->report($syssn, 49, $node, $common->convert_epoch_to_external_datetime($xepoch), $ctext);
	#    }
	    for (my $j=0; $j < scalar(@{$common->{captures}->{$syssn}->{observed}->{ts}->{$xepoch}}); $j++)
	    {
		
		if ($common->{captures}->{$syssn}->{observed}->{ts}->{$xepoch}[$j] !~ m/CM Error\:/){next;}
		my $node = $common->{captures}->{$syssn}->{observed}->{ts}->{$xepoch}[$j-3];
		#printf "cfganal_captures::analyze_cm_link_errors: $node | $ctext\n";
		$common->report($syssn, 49, $node, $common->convert_epoch_to_external_datetime($xepoch), $common->{captures}->{$syssn}->{observed}->{ts}->{$xepoch}[$j]);
	    }
	}
    }
    $common->trace(TRACE_ROUTINES, "<--- cfganal_captures::analyze_cm_errors");
    return 1;
}

sub analyze_link_errors
{
    my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $string) = @_;
    #
    $common->trace(TRACE_ROUTINES, "---> cfganal_captures::analyze_link_errors");
    #
    foreach my $fn  (sort keys %{$ptr->{strings}->{$string}})
    {
	#printf "CM_LINK_ERRORS: $string | $fn | Count: %s\n", $ptr->{strings}->{$string}->{$fn}->{count};
	if (!$ptr->{strings}->{$string}->{$fn}->{count}){next;}
	#
	foreach my $xepoch (sort keys %{$ptr->{strings}->{$string}->{$fn}->{ts}})
	{
	    foreach my $ctext (@{$common->{captures}->{$syssn}->{observed}->{ts}->{$xepoch}})
	    {
		if ($ctext !~ m/Eagle link error hw_eagle:(\d+) Link bringup fatal error from node (\d+) to (\d+)/ &&
		    $ctext !~ m/Eagle link error hw_eagle:(\d+) Fatal link error between member nodes (\d+) to (\d+)/){next;}
		my $node = $1;
		#printf "cfganal_captures::analyze_cm_link_errors: $ctext\n";
		$common->report($syssn, 48, $node, $common->convert_epoch_to_external_datetime($xepoch), $ctext);
	    }
	}
	
    }
    #if ($ptr->{strings}->{$string}->{$fn}->{count} < $minimum) { return $report; }
    #print Dumper($ptr->{strings}->{$string});
    $common->trace(TRACE_ROUTINES, "<--- cfganal_captures::analyze_link_errors");
    return 1;
}

sub analyze_gc_abandonment
{
    #
    my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $string) = @_;
    #
    my %ddcscan = ();
    my $ddsptr = \%ddcscan;
    #
    $common->trace(TRACE_ROUTINES, "---> cfganal_captures::analyze_gc_abandonment");
    my $logname = "showeventlog_-d_-debug_-oneline.out";
    #@{$common->{captures}->{$syssn}->{observed}->{ts}->{$epoch}}
    foreach my $xepoch (sort keys %{$common->{captures}->{$syssn}->{observed}->{ts}})
    {
	foreach my $ctext (@{$common->{captures}->{$syssn}->{observed}->{ts}->{$xepoch}})
	{
	    if ($ctext !~ m/End run on DDS ([\.\w]+) \(id (\d+)\) status (\d+)/){next;}
	    #printf "cfganal_captures::analyze_dedupe_issues: Name: %s  ID: %s  Status: %s\n", $1, $2, $3;
	    my $ddsid = sprintf("%08d", $2);
	    my $ddsname = $1;
	    my $ddsstatus = $3;
	    if (!exists($ddsptr->{$ddsid}))
	    {
		$ddsptr->{$ddsid}->{name} = $ddsname;
		$ddsptr->{$ddsid}->{status}->{$ddsstatus} = 1;
	    }
	    else
	    {
		$ddsptr->{$ddsid}->{status}->{$ddsstatus}++;
	    }
	}
    }
    #
    # Now report and calculate on each DDS...
    #
    foreach my $ddsid (sort keys %{$ddsptr})
    {
	#
	# Add up our statuses, we are going to assume that only 1 and 0 are really
	# the only status values. However, if there are more, we will use them in the total.
	# And the numerator will just be the Status total for 1.
	#
	my $stattotal = 0;
	foreach my $statval (sort keys %{$ddsptr->{$ddsid}->{status}})
	{
	    $stattotal += $ddsptr->{$ddsid}->{status}->{$statval};
	}
	#
	# The percentage...
	#
	my $abandon_percentage = 0;
	if ($stattotal > 0){ $abandon_percentage = ($ddsptr->{$ddsid}->{status}->{1}/$stattotal) * 100.0;}
	#
	# If the abandonment percentage is greater or equal to 50% we need to raise the alarm.
	#
	if ($abandon_percentage >= 50.0)
	{
	    $common->report_cfganal_issue($syssn, 'vv', $ddsid, 2, "DDS $ddsid\:$ddsptr->{$ddsid}->{name} has an abandonment percentage >= 50%\!", 41,
					      $ddsid, $ddsptr->{$ddsid}->{name}, sprintf("% 5.2f", $abandon_percentage));
	}
    }
    #
    $common->trace(TRACE_ROUTINES, "<--- cfganal_captures::analyze_gc_abandonment");
    $ddsptr = undef;
    %ddcscan = ();
    return;
    
}

sub analyze_drive_capacity
{
    my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $string) = @_;
    #
    $common->trace(TRACE_ROUTINES, "---> cfganal_captures::analyze_drive_capacity");
    #
    foreach my $fn  (sort keys %{$ptr->{strings}->{$string}})
    {
	#printf "CM_LINK_ERRORS: $string | $fn | Count: %s\n", $ptr->{strings}->{$string}->{$fn}->{count};
	if (!$ptr->{strings}->{$string}->{$fn}->{count}){next;}
	#
	foreach my $xepoch (sort keys %{$ptr->{strings}->{$string}->{$fn}->{ts}})
	{
	    foreach my $ctext (@{$common->{captures}->{$syssn}->{observed}->{ts}->{$xepoch}})
	    {
		#
		##This message is really a single line:
		##
		#Degraded       Disk fail alert hw_disk:5000CCA045029A3B pd wwn 5000CCA045029A3B failure: drive capacity is unsupported- Internal reason:-
		#drive (wwn: 0x5000CCA045029A3B, p.rev.: 3P00, devid: HKCF0300S5xeN015) reported total chunklet numbers (272) not equal to our
		#scsi database chksize (1117)- This disk will not be admitted into the system.
		#
		if ($ctext !~ m/\(wwn: 0x(\w+), p\.rev\.: (\w+), devid: (\w+)\) reported total chunklet numbers \((\d+)\)/) {next;}
		#my $node = $1;
		#printf "cfganal_captures::analyze_cm_link_errors: $ctext\n";
		my $pdmodel = $3;
		my $pdchunks = $4;
		my $fpdwwn = $common->format_wwn($1);
		my $pptr = undef;
		my $pdid = 'unk';
		my $pdpos = 'unk';
		#
		if (exists($common->{refs}) &&
		    exists($common->{refs}->{$syssn}) &&
		    exists($common->{refs}->{$syssn}->{wwn}) &&
		    exists($common->{refs}->{$syssn}->{wwn}->{$fpdwwn}))
		{
		    $pptr = $common->{refs}->{$syssn}->{wwn}->{$fpdwwn};
		    if (exists($pptr->{id}) &&
		        defined($pptr->{id}) &&
			$pptr->{id} =~ m/\d+/)
		    {
			$pdid = $pptr->{id};
		    }
		    #
		    if (exists($pptr->{pos}) &&
		        defined($pptr->{pos}) &&
			$pptr->{pos} =~ m/\d+:\d+:\d+/)
		    {
			$pdpos = $pptr->{pos};
		    }
		}
		#
		
		#
		# Check and see if we have a King-Cobra drive...
		#
		my $acode = 55;
		if ($pdmodel eq 'HKCF0300S5xeN015') {$acode = 54;}
		#
		$common->report($syssn, $acode, $fpdwwn, $pdid, $pdpos, $pdmodel, $pdchunks, $common->convert_epoch_to_external_datetime($xepoch));
	    } #end of ctext loop
	    #
	} #end of xepoch loop
	#
    } #end of filename loop
    #
    #if ($ptr->{strings}->{$string}->{$fn}->{count} < $minimum) { return $report; }
    #print Dumper($ptr->{strings}->{$string});
    $common->trace(TRACE_ROUTINES, "<--- cfganal_captures::analyze_drive_capacity");
    return 1;
}

sub analyze_unresponsive_hal {
    my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $string) = @_;
    
    $common->trace(TRACE_ROUTINES, "---> cfganal_captures::analyze_unresponsive_hal");

    @{$lptr->{reports}} = ();
    #printf "ANALYZE_UNRESPONSIVE_HAL: sn: !$syssn! | $string | $fn | Count: %s\n", $ptr->{strings}->{$string}->{$fn}->{count};
    #
    
    #printf STDERR "CFGANAL_CAPTURES::HAL: $syssn\n%s\n", Dumper($common->{captures}->{$syssn}->{observed}->{ts});
    foreach my $fn  (sort keys %{$ptr->{strings}->{$string}})
    {
	#printf "ANALYZE_UNRESPONSIVE_HAL: sn: !$syssn! | $string | $fn | Count: %s\n", $ptr->{strings}->{$string}->{$fn}->{count};
	if (!$ptr->{strings}->{$string}->{$fn}->{count}) { next; }
	#
	foreach my $xepoch (sort keys %{$ptr->{strings}->{$string}->{$fn}->{ts}})
	{
	    

	    for (my $j=0; $j < scalar(@{$common->{captures}->{$syssn}->{observed}->{ts}->{$xepoch}}); $j++)
	    {
		
		if ($common->{captures}->{$syssn}->{observed}->{ts}->{$xepoch}[$j] !~ m/(\d+):(\d+):(\d+) ocs_hal_command: MQ command issued sent in non-active state, HAL state=(\d+)/) { next; }
		#
		my $node = $common->{captures}->{$syssn}->{observed}->{ts}->{$xepoch}[$j-3];
		my $nspnode = $1;
		my $nspslot = $2;
		my $nspport = $3;
		my $halstate = $4;
		#
		if ($halstate ne 3) { next; }
		my $portid = sprintf ("%02d%02d%02d", $nspnode, $nspslot, $nspport);
		$common->report_cfganal_issue($syssn, 
				  "ports", 
				  $portid, 
				  "model", 
				  "See CFI 7839, there is potential for poor performance, hung IOs or unresponsive array.",
				  62,
				  $nspnode.":".$nspslot.":".$nspport,
				  $common->{config}->{$syssn}->{ports}->{$portid}->{model},
				  $common->{config}->{$syssn}->{ports}->{$portid}->{vendor},
				  $common->{config}->{$syssn}->{ports}->{$portid}->{firmware},
				  $halstate,
				  $node,
				  $common->convert_epoch_to_external_datetime($xepoch)
				 );
	    }
	}
    }
    #
    $common->trace(TRACE_ROUTINES, "<--- cfganal_captures::analyze_unresponsive_hal");
    #
    return 1;
}


#sub orig_analyze_gc_abandonment
#{
#    #
#    my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $string) = @_;
#    #
#    my %ddcscan = ();
#    my $ddsptr = \%ddcscan;
#    #
#    $common->trace(TRACE_ROUTINES, "---> cfganal_captures::analyze_gc_abandonment");
#    my $logname = "showeventlog_-d_-debug_-oneline.out";
#    #
#    foreach my $ctext (@{$common->{captures}->{$syssn}->{observed}->{strings}->{$string}->{$logname}->{captured}})
#    {
#	if ($ctext !~ m/End run on DDS ([\.\w]+) \(id (\d+)\) status (\d+)/){next;}
#	#printf "cfganal_captures::analyze_dedupe_issues: Name: %s  ID: %s  Status: %s\n", $1, $2, $3;
#	my $ddsid = sprintf("%08d", $2);
#	my $ddsname = $1;
#	my $ddsstatus = $3;
#	if (!exists($ddsptr->{$ddsid}))
#	{
#	    $ddsptr->{$ddsid}->{name} = $ddsname;
#	    $ddsptr->{$ddsid}->{status}->{$ddsstatus} = 1;
#	}
#	else
#	{
#	    $ddsptr->{$ddsid}->{status}->{$ddsstatus}++;
#	}
#    }
#    #
#    # Now report and calculate on each DDS...
#    #
#    foreach my $ddsid (sort keys %{$ddsptr})
#    {
#	#
#	# Add up our statuses, we are going to assume that only 1 and 0 are really
#	# the only status values. However, if there are more, we will use them in the total.
#	# And the numerator will just be the Status total for 1.
#	#
#	my $stattotal = 0;
#	foreach my $statval (sort keys %{$ddsptr->{$ddsid}->{status}})
#	{
#	    $stattotal += $ddsptr->{$ddsid}->{status}->{$statval};
#	}
#	#
#	# The percentage...
#	#
#	my $abandon_percentage = ($ddsptr->{$ddsid}->{status}->{1}/$stattotal) * 100.0;
#	#
#	# If the abandonment percentage is greater or equal to 50% we need to raise the alarm.
#	#
#	if ($abandon_percentage >= 50.0)
#	{
#	    $common->report_cfganal_issue($syssn, 'vv', $ddsid, 2, "DDS $ddsid\:$ddsptr->{$ddsid}->{name} has an abandonment percentage >= 50%\!", 41,
#					      $ddsid, $ddsptr->{$ddsid}->{name}, sprintf("% 5.2f", $abandon_percentage));
#	}
#    }
#    #
#    $common->trace(TRACE_ROUTINES, "<--- cfganal_captures::analyze_gc_abandonment");
#    $ddsptr = undef;
#    %ddcscan = ();
#    return;
#    
#}

sub analyze_getlbastatus {
        my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $string) = @_;

	@{$lptr->{reports}} = ();
	foreach my $fname (sort keys%{$ptr->{strings}->{$string}})
	{
	    if ($fname !~ m/syslog/i){next;}
	    
	    my $report = analyze_string($common, $ptr, $string, $fname, 1, 24*60);	# Check for 1 occurences 24 hours
	    #if (defined($report)) {
	    if ($report) {
		#3.1.2.278 -> /export/ws/releases/3.1.2.GA
		#3.1.2.322 -> /export/ws/releases/3.1.2.MU1
		#3.1.2.540 -> /export/ws/releases/3.1.2.MU1.P25
		#3.1.2.422 -> /export/ws/releases/3.1.2.MU2
		#3.1.2.430 -> /export/ws/releases/3.1.2.MU2-git
		#3.1.2.544 -> /export/ws/releases/3.1.2.MU2.P25

		if ($version == 312 &&
		    ($baselevel == 278 ||
		     $baselevel == 322 ||
		     $baselevel == 540 ||
		     $baselevel == 422 ||
		     $baselevel == 430 ||
		     #$baselevel == 544 ) ) {  $common->report($syssn, 1012, "disable_get_lba_status.sh", 4232, $report); }	#
		     $baselevel == 544 ) ) {  foreach (@{$lptr->{reports}}) { $common->report($syssn, 1012, "disable_get_lba_status.sh", 4232, $_); } }	#
	    }
	}
	
        return;
}

sub analyze_rc_vv_locks {
        my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $string) = @_;

	@{$lptr->{reports}} = ();
	foreach my $fname (sort keys%{$ptr->{strings}->{$string}})
	{
	    if ($fname !~ m/sysmgr/i){next;}
	    my $report = analyze_string($common, $ptr, $string, $fname, 1, 24*60);	# Check for 1 occurences 24 hours
	#    if (defined($report)) {
	#	if ($version == 313 && $baselevel == 230) {  $common->report($syssn, 1011, "P06", 5496, $report); }	#
	#    }
	    if ($report) {
		if ($version == 313 && $baselevel == 230) {  foreach (@{$lptr->{reports}}) { $common->report($syssn, 1011, "P06", 5496, $_); } }	#
	    }
	}
	
        return;
}

sub analyze_ct_mgmt_get_port_info {
        my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $string) = @_;
	$common->trace(TRACE_ROUTINES, "---> cfganal_captures::analyze_ct_mgmt_get_port_info");
	@{$lptr->{reports}} = ();
	#
	my $report = analyze_string($common, $ptr, $string, "showeventlog_-d_-debug_-oneline.out", 50, (24*60));	# Check for 50 occurences in 24 hours (24 x 60 minutes)
	#foreach (@{$lptr->{reports}}){print "CT_MGMT_: $_\n";}
#	if (defined($report)) {
#	    if ($version == 312) {
#			
#			$common->report($syssn, 1011, "P04", 3918, $report);
#			
#     	} elsif ($version >= 313) {
#		#
#		# Added 313.MU1 20-Nov-2014 Gary Sachs, the problem still exists on 313.MU1
#		# 3-Mar-2015: Spoke with Vish, this seems to be a problem with 313 and onward for the time being.
#		#  The script has been updated to run on any version 313 and above.
#		# 
##		if ($baselevel == 202 || $baselevel == 230) {
##		     # For 3.1.3.GA only
##		     $common->report($syssn, 1012, "enable_skip_fabric_ioctls.sh", 4952, $report);
##                }
#			$common->report($syssn, 1012, "enable_skip_fabric_ioctls.sh", 4952, $report);
#     	}
#	}
	if ($report) {
	    if ($version == 312) {
			
			foreach (@{$lptr->{reports}}) { $common->report($syssn, 1011, "P04", 3918, $_);}
			
     	} elsif ($version >= 313) {
		#
		# Added 313.MU1 20-Nov-2014 Gary Sachs, the problem still exists on 313.MU1
		# 3-Mar-2015: Spoke with Vish, this seems to be a problem with 313 and onward for the time being.
		#  The script has been updated to run on any version 313 and above.
		# 
#		if ($baselevel == 202 || $baselevel == 230) {
#		     # For 3.1.3.GA only
#		     $common->report($syssn, 1012, "enable_skip_fabric_ioctls.sh", 4952, $report);
#                }
			foreach (@{$lptr->{reports}}) { $common->report($syssn, 1012, "enable_skip_fabric_ioctls.sh", 4952, $_); }
     	}
	}
	$common->trace(TRACE_ROUTINES, "<--- cfganal_captures::analyze_ct_mgmt_get_port_info");
        return;
}


sub analyze_sdt_cdb_retry_func {
        my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $string) = @_;

	@{$lptr->{reports}} = ();
	my $report = analyze_string($common, $ptr, $string, "showeventlog_-d_-debug_-oneline.out", 50, (24*60));	# Check for 50 occurences in 24 hours (24 x 60 minutes)
	#if (defined($report)) {
	if ($report) {
	    if ($version == 312) {
	        #$common->report($syssn, 1012, "scsi_debug_events.sh", 4627, $report);
	        #$common->report($syssn, 1012, "scsi_debug_events.sh", "5074 (AIX Hosts only)", $report);
		#
		foreach (@{$lptr->{reports}}) {$common->report($syssn, 1012, "scsi_debug_events.sh", 4627, $_); }
	        foreach (@{$lptr->{reports}}) {$common->report($syssn, 1012, "scsi_debug_events.sh", "5074 (AIX Hosts only)", $_);}
            }
	}
	
        return;
}

sub analyze_els_ct_timeout {
        my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $string) = @_;

	@{$lptr->{reports}} = ();
	my $report = analyze_string($common, $ptr, $string, "showeventlog_-d_-debug_-oneline.out", 10, (24*60));	# Check for 10 occurences in 24 hours (24 x 60 minutes)
#	if (defined($report)) {
#	    if      ( ($version == 312) && ($baselevel == 422) ) { $common->report($syssn, 1011, "P38", 5294, $report);	# Baselevel 3.1.2-422 is 3.1.2.MU2
#     	    } elsif ( ($version == 312) && ($baselevel == 484) ) { $common->report($syssn, 1011, "P33", 5294, $report); # Baselevel 3.1.2-484 is 3.1.2.MU3
#     	    } elsif   ($version == 312)                          { $common->report($syssn, 1012, "nop_els_ct_timer.sh", 5294, $report);
#     	    }
#	}
	if ($report) {
	    if      ( ($version == 312) && ($baselevel == 422) ) { foreach (@{$lptr->{reports}}) {$common->report($syssn, 1011, "P38", 5294, $_);}	# Baselevel 3.1.2-422 is 3.1.2.MU2
     	    } elsif ( ($version == 312) && ($baselevel == 484) ) { foreach (@{$lptr->{reports}}) {$common->report($syssn, 1011, "P33", 5294, $_);} # Baselevel 3.1.2-484 is 3.1.2.MU3
     	    } elsif   ($version == 312)                          { foreach (@{$lptr->{reports}}) {$common->report($syssn, 1012, "nop_els_ct_timer.sh", 5294, $_);}
     	    }
	}
	
        return;
}

sub analyze_scsi_send_cmnd_async {
        my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $string) = @_;

	@{$lptr->{reports}} = ();
	my $report = analyze_string($common, $ptr, $string, "showeventlog_-d_-debug_-oneline.out", 2, 1);	# Check for 2 occurences in 1 minute, as usually this leads to a FWcore
	#if (defined($report)) {
	#    if ($version == 312) {  $common->report($syssn, 1013, "3.1.3", 4981, $report); }	#
	#}
	if ($report) {
	    if ($version == 312) {  foreach (@{$lptr->{reports}}) { $common->report($syssn, 1013, "3.1.3", 4981, $_);} }	#
	}
	
        return;
}

sub analyze_scsi_cmnd_retry {
        my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $string) = @_;

	@{$lptr->{reports}} = ();
	my $report = analyze_string($common, $ptr, $string, "showeventlog_-d_-debug_-oneline.out", 2, 1);	# Check for 2 occurences in 1 minute, as usually this leads to a FWcore
	#if (defined($report)) {
	#    if ($version == 312) {  $common->report($syssn, 1013, "3.1.3", 4981, $report); }	#
	#}
	if ($report) {
	    if ($version == 312) {  foreach (@{$lptr->{reports}}) { $common->report($syssn, 1013, "3.1.3", 4981, $_); } }	#
	}
	
        return;
}

sub analyze_RTPG_count {
        my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $string) = @_;

	@{$lptr->{reports}} = ();
	if ($version == 313) {
	    my $report = analyze_string($common, $ptr, $string, "showeventlog_-d_-debug_-oneline.out", 50, 1, 5);	# Check for 50 occurences in 1 minute and 5 occurences, as this indicates sysmgr is seriously behind in responding
	    #if (defined($report)) { $common->report($syssn, 1016, $string, 5804, $report); }
	    if ($report) { foreach (@{$lptr->{reports}}) { $common->report($syssn, 1016, $string, 5804, $_); } }
	}
	
        return;
}

sub analyze_Qlen_events {
        my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $string) = @_;

	my @hosts = ();
	my ($vvid, $hostname, $hostid, $report);
	@{$lptr->{reports}} = ();
	if ($version == 313) {
	    if ( ($baselevel == 202) || ($baselevel == 230) ) {	# 202 = 3.1.3.GA,    230 = 3.1.3.MU1
	        if ($patches !~ /P10/) {
		    if (exists($common->{config}->{$syssn}->{rcopy})) {
	    	        $report = analyze_string($common, $ptr, $string, "showeventlog_-d_-debug_-oneline.out", 2, 5);	# Check for 2 occurences in 5 minutes
	                #if (defined($report)) { $common->report($syssn, 1018, "3.1.3.".$baselevel, "P10", "RemoteCopy", 5906, $report); }
			if ($report) { foreach (@{$lptr->{reports}}) { $common->report($syssn, 1018, "3.1.3.".$baselevel, "P10", "RemoteCopy", 5906, $_); } }
                    }
                    $report = undef;
                    foreach $vvid (keys(%{$common->{config}->{$syssn}->{vv}})) {
                        if (exists($common->{config}->{$syssn}->{vv}->{$vvid}->{qos})) {
	                    @hosts = ();
	                    foreach $hostname (keys(%{$common->{config}->{$syssn}->{vv}->{$vvid}->{hosts}})) {
	                        if ($hostname =~ /^set:/) { 
  		                    @hosts = keys(%{$common->{config}->{$syssn}->{hostset}->{substr($hostname,4)}->{members}}); 
		                } else { 
		                    push (@hosts, $common->{parse_config_host_obj}->name2id($hostname)); 
                                }
	                        foreach $hostid (@hosts) {
	                            unless ($common->{config}->{$syssn}->{host}->{$hostid}->{persona}->{id}) { next; }
	                            if ($common->{config}->{$syssn}->{host}->{$hostid}->{persona}->{id} == 15) {
	    	                        $report = analyze_string($common, $ptr, $string, "showeventlog_-d_-debug_-oneline.out", 2, 5);	# Check for 2 occurences in 5 minutes
	                                 #if (defined($report)) { $common->report($syssn, 1018, "3.1.3.".$baselevel, "P10", "QoS", 5876, $report); }
					 if ($report) { foreach (@{$lptr->{reports}}) { $common->report($syssn, 1018, "3.1.3.".$baselevel, "P10", "QoS", 5876, $_); } }
	                                 last;
                                    }
                                }
                            }
                            #if (defined($report)) { last; }
			    if ($report) { last; }
			}
		    }	        
                    #if (defined($report)) { last; }
		    if ($report) { last; }
                }
            }
	}
	
        return;
}

sub gary2_analyze_string {
        my ($common, $ptr, $string, $fn, $minimum, $min, $observedcount) = @_;

	my $count   = 0;
	my $report  = undef;
	my $nrtimes = 1;
	my $occcnt = 0;
	#
	if (defined($observedcount)) { $nrtimes = $observedcount; }

	# If we have less then the minimum, let us immediately return;
        if ($ptr->{strings}->{$string}->{$fn}->{count} < $minimum) { return $report; }
	#
	my @ts = sort keys(%{$ptr->{strings}->{$string}->{$fn}->{ts}});
	#
	printf STDERR "ANALYZE_STRING: string: $string fn: $fn minimum: $minimum min: $min obscnt: $observedcount\n%s\n", Dumper($ptr->{strings}->{$string}->{$fn}->{ts});
	 
	my ($i, $j);
	for $i (0..(@ts-1)) {
	    if ( ($ptr->{strings}->{$string}->{$fn}->{ts}->{$ts[$i]} + $count) < $minimum ) {  
	        $count   = 0;
	        $nrtimes = 1;
	        if (defined($observedcount)) { $nrtimes = $observedcount; }
	        for $j (($i+1)..(@ts-1)) {
	            if ($ts[$j] > ($ts[$i]+($min*60))) #{ last; }		# Stop if $ts[$j] is beyond the current timestamp ($ts[$i]) + ($min*60)
		    {
			$count = 0;
			last;
		    }
		    $count += $ptr->{strings}->{$string}->{$fn}->{ts}->{$ts[$j]};
		    #
		    printf STDERR "$string | i-time: %s   j-time: %s  basecnt: %s count: $count   min: $minimum\n",
			$common->convert_epoch_to_external_datetime($ts[$i]),
			$common->convert_epoch_to_external_datetime($ts[$j]),
			$ptr->{strings}->{$string}->{$fn}->{ts}->{$ts[$i]};
		    #
		    if ( ($ptr->{strings}->{$string}->{$fn}->{ts}->{$ts[$i]} + $count) < $minimum ) { next; }
		    if ($nrtimes == 1) {
                        $report  = "Between ".$common->convert_epoch_to_external_datetime($ts[$i])." and ".$common->convert_epoch_to_external_datetime($ts[$j])." observed ".($ptr->{strings}->{$string}->{$fn}->{ts}->{$ts[$i]} + $count)." occurences of \'$string\'.";
			#print "TEST_REPORT: $report\n";
			push (@{$lptr->{reports}}, $report);
			$occcnt++;
			$count = 0;
			last;
		    } else {
		        $nrtimes -= 1;
		    } 
	        }
            } else {
		if ($nrtimes == 1) {
		    $report  = "On ".$common->convert_epoch_to_external_datetime($ts[$i])." observed ".$ptr->{strings}->{$string}->{$fn}->{ts}->{$ts[$i]}." occurences of \'$string\".";
		    push (@{$lptr->{reports}}, $report);
		    $count = 0;
		    $occcnt++;
		} else {
		    $nrtimes -= 1;
		}
            }
	    #if (defined($report)) { last; }
	}

	#	if (defined($report)) { 
#	    unless  ($observedcount     ) { $report .= " Should not have exceeded $minimum occurences per $min minutes"; 
#	    } elsif ($observedcount == 1) { $report .= " Should not have exceeded $minimum occurences per $min minutes";
#	    } else {                        $report .= " Should not have exceeded $minimum occurences per $min minutes, and that $observedcount times";
#	    }
#        }
	if ($occcnt) {
	    for (my $i = 0; $i < scalar(@{$lptr->{reports}}); $i++)
	    {
		$report = $lptr->{reports}[$i];
		unless  ($observedcount     ) { $report .= " Should not have exceeded $minimum occurences per $min minutes"; 
		} elsif ($observedcount == 1) { $report .= " Should not have exceeded $minimum occurences per $min minutes";
		} else {                        $report .= " Should not have exceeded $minimum occurences per $min minutes, and that $observedcount times";
		}
		$lptr->{reports}[$i] = $report;
	    }
	    #
	    if (!defined($common->{procmode}->{analysis_overview_lists}) || !$common->{procmode}->{analysis_overview_lists})
	    {
		$#{$lptr->{reports}} = 0;
		$occcnt = 1;
	    }
        }
	#
        return $occcnt; #$report;
}

#sub gary1_analyze_string {
sub analyze_string {
        my ($common, $ptr, $string, $fn, $minimum, $min, $observedcount) = @_;

	my $count   = 0;
	my $report  = undef;
	my $nrtimes = 1;
	my $occcnt = 0;
	#
	if (defined($observedcount)) { $nrtimes = $observedcount; }

	#
	# If we have less then the minimum, let us immediately return;
	#
        if ($ptr->{strings}->{$string}->{$fn}->{count} < $minimum) { return $report; }
	#printf STDERR "ANALYZE_STRING: string: $string fn: $fn minimum: $minimum min: $min obscnt: $observedcount\n%s\n", Dumper($ptr->{strings}->{$string}->{$fn}->{ts});

	my @ts = sort keys(%{$ptr->{strings}->{$string}->{$fn}->{ts}});
	
	my ($i, $j);
	
	#for $i (0..(@ts-1)) {
	$i = 0;
ILOOP:	while ($i < scalar(@ts)){
	    #if ( !exists($ts[$i])) { last; }
	    if ( ($ptr->{strings}->{$string}->{$fn}->{ts}->{$ts[$i]} + $count) < $minimum ) {  
	        $count   = 0;
	        $nrtimes = 1;
	        if (defined($observedcount)) { $nrtimes = $observedcount; }
	        for $j (($i+1)..(@ts-1)) {
		    #
		    # Stop if $ts[$j] is beyond the current timestamp ($ts[$i]) + ($min*60)
		    #
	            if ($ts[$j] > ($ts[$i]+($min*60)))
		    {
			#$i = $j;
			$count = 0;
			#next ILOOP;
			last;
		    }
		    $count += $ptr->{strings}->{$string}->{$fn}->{ts}->{$ts[$j]};
		    #
			#printf STDERR "$string | i-time: %s   j-time: %s  basecnt: %s count: $count   min: $minimum\n",
			#    $common->convert_epoch_to_external_datetime($ts[$i]),
			#    $common->convert_epoch_to_external_datetime($ts[$j]),
			#    $ptr->{strings}->{$string}->{$fn}->{ts}->{$ts[$i]};
		    #
		    if ( ($ptr->{strings}->{$string}->{$fn}->{ts}->{$ts[$i]} + $count) < $minimum ) { next; }
		    if ($nrtimes == 1) {
                        $report  = "Between ".$common->convert_epoch_to_external_datetime($ts[$i])." and ".$common->convert_epoch_to_external_datetime($ts[$j])." observed ".($ptr->{strings}->{$string}->{$fn}->{ts}->{$ts[$i]} + $count)." occurences of \'$string\'.";
			#print "TEST_REPORT: $report\n";
			push (@{$lptr->{reports}}, $report);
			$occcnt++;
			$i = ++$j;
			$count = 0;
			next ILOOP;
		        last;
		    } else {
		        $nrtimes -= 1;
		    } 
	        }
            } else {
		if ($nrtimes == 1) {
		    $report  = "On ".$common->convert_epoch_to_external_datetime($ts[$i])." observed ".($ptr->{strings}->{$string}->{$fn}->{ts}->{$ts[$i]})." occurences of \'$string\".";
		    #print "TEST_REPORT: $report\n";
		    push (@{$lptr->{reports}}, $report);
		    $occcnt++;
		} else {
		    $nrtimes -= 1;
		}
            }
	    #if (defined($report)) { last; }
	    #if ($occcnt){ last; }
	    $i++;
	}


#	if (defined($report)) { 
#	    unless  ($observedcount     ) { $report .= " Should not have exceeded $minimum occurences per $min minutes"; 
#	    } elsif ($observedcount == 1) { $report .= " Should not have exceeded $minimum occurences per $min minutes";
#	    } else {                        $report .= " Should not have exceeded $minimum occurences per $min minutes, and that $observedcount times";
#	    }
#        }
	if ($occcnt) {
	    for (my $i = 0; $i < scalar(@{$lptr->{reports}}); $i++)
	    {
		$report = $lptr->{reports}[$i];
		unless  ($observedcount     ) { $report .= " Should not have exceeded $minimum occurences per $min minutes"; 
		} elsif ($observedcount == 1) { $report .= " Should not have exceeded $minimum occurences per $min minutes";
		} else {                        $report .= " Should not have exceeded $minimum occurences per $min minutes, and that $observedcount times";
		}
		$lptr->{reports}[$i] = $report;
	    }
	    #
	    if (!defined($common->{procmode}->{analysis_overview_lists}) || !$common->{procmode}->{analysis_overview_lists})
	    {
		$#{$lptr->{reports}} = 0;
		$occcnt = 1;
	    }
        }
	#
        return $occcnt; #$report;
}

sub printed_analyze_string {
        my ($common, $ptr, $string, $fn, $minimum, $min, $observedcount) = @_;

	my $count   = 0;
	my $report  = undef;
	my $nrtimes = 1;
	if (defined($observedcount)) { $nrtimes = $observedcount; }

	# If we have less then the minimum, let us immediately return;
        if ($ptr->{strings}->{$string}->{$fn}->{count} < $minimum) { return $report; }
printf STDERR "ANALYZE_STRING: string: $string fn: $fn minimum: $minimum min: $min obscnt: $observedcount\n%s\n", Dumper($ptr->{strings}->{$string}->{$fn}->{ts});
	my @ts = sort keys(%{$ptr->{strings}->{$string}->{$fn}->{ts}});
	
	my ($i, $j);
	for $i (0..(@ts-1)) {
	    if ( ($ptr->{strings}->{$string}->{$fn}->{ts}->{$ts[$i]} + $count) < $minimum ) {  
	        $count   = 0;
	        $nrtimes = 1;
	        if (defined($observedcount)) { $nrtimes = $observedcount; }
	        for $j (($i+1)..(@ts-1)) {
	            if ($ts[$j] > ($ts[$i]+($min*60))) { last; }		# Stop if $ts[$j] is beyond the current timestamp ($ts[$i]) + ($min*60)
		    $count += $ptr->{strings}->{$string}->{$fn}->{ts}->{$ts[$j]};
		    printf STDERR "$string | i-time: %s   j-time: %s  basecnt: %s count: $count   min: $minimum\n",
			$common->convert_epoch_to_external_datetime($ts[$i]),
			$common->convert_epoch_to_external_datetime($ts[$j]),
			$ptr->{strings}->{$string}->{$fn}->{ts}->{$ts[$i]};
		    if ( ($ptr->{strings}->{$string}->{$fn}->{ts}->{$ts[$i]} + $count) < $minimum ) { next; }
		    if ($nrtimes == 1) {
                        $report  = "Between ".$common->convert_epoch_to_external_datetime($ts[$i])." and ".$common->convert_epoch_to_external_datetime($ts[$j])." observed ".($ptr->{strings}->{$string}->{$fn}->{ts}->{$ts[$i]} + $count)." occurences of \'$string\'.";
			last;
		    } else {
		        $nrtimes -= 1;
		    } 
	        }
            } else {
                 if ($nrtimes == 1) {
                     $report  = "On ".$common->convert_epoch_to_external_datetime($ts[$i])." observed ".$ptr->{strings}->{$string}->{$fn}->{ts}->{$ts[$i]}." occurences of \'$string\".";
                 } else {
		     $nrtimes -= 1;
                 }
            }
	    if (defined($report)) { last; }
	}

	if (defined($report)) { 
	    unless  ($observedcount     ) { $report .= " Should not have exceeded $minimum occurences per $min minutes"; 
	    } elsif ($observedcount == 1) { $report .= " Should not have exceeded $minimum occurences per $min minutes";
	    } else {                        $report .= " Should not have exceeded $minimum occurences per $min minutes, and that $observedcount times";
	    }
        } 
        return $report;
}

sub origi_orig_analyze_string {
        my ($common, $ptr, $string, $fn, $minimum, $min, $observedcount) = @_;

	my $count   = 0;
	my $report  = undef;
	my $nrtimes = 1;
	if (defined($observedcount)) { $nrtimes = $observedcount; }

	# If we have less then the minimum, let us immediately return;
        if ($ptr->{strings}->{$string}->{$fn}->{count} < $minimum) { return $report; }
	printf STDERR "ANALYZE_STRING: string: $string fn: $fn minimum: $minimum min: $min obscnt: $observedcount\n%s\n", Dumper($ptr->{strings}->{$string}->{$fn}->{ts});
	my @ts = sort keys(%{$ptr->{strings}->{$string}->{$fn}->{ts}});
	 
	my ($i, $j);
	for $i (0..(@ts-1)) {
	    if ( ($ptr->{strings}->{$string}->{$fn}->{ts}->{$ts[$i]} + $count) < $minimum ) {  
	        $count   = 0;
	        $nrtimes = 1;
	        if (defined($observedcount)) { $nrtimes = $observedcount; }
	        for $j (($i+1)..(@ts-1)) {
	            if ($ts[$j] > ($ts[$i]+($min*60))) { last; }		# Stop if $ts[$j] is beyond the current timestamp ($ts[$i]) + ($min*60)
		    $count += $ptr->{strings}->{$string}->{$fn}->{ts}->{$ts[$j]};
		    if ( ($ptr->{strings}->{$string}->{$fn}->{ts}->{$ts[$i]} + $count) < $minimum ) { next; }
		    if ($nrtimes == 1) {
                        $report  = "Between ".$common->convert_epoch_to_external_datetime($ts[$i])." and ".$common->convert_epoch_to_external_datetime($ts[$j])." observed ".($ptr->{strings}->{$string}->{$fn}->{ts}->{$ts[$i]} + $count)." occurences of \'$string\'.";
		        last;
		    } else {
		        $nrtimes -= 1;
		    } 
	        }
            } else {
                 if ($nrtimes == 1) {
                     $report  = "On ".$common->convert_epoch_to_external_datetime($ts[$i])." observed ".$ptr->{strings}->{$string}->{$fn}->{ts}->{$ts[$i]}." occurences of \'$string\".";
                 } else {
		     $nrtimes -= 1;
                 }
            }
	    if (defined($report)) { last; }
	}

	if (defined($report)) { 
	    unless  ($observedcount     ) { $report .= " Should not have exceeded $minimum occurences per $min minutes"; 
	    } elsif ($observedcount == 1) { $report .= " Should not have exceeded $minimum occurences per $min minutes";
	    } else {                        $report .= " Should not have exceeded $minimum occurences per $min minutes, and that $observedcount times";
	    }
        } 
        return $report;
}


sub string_count_occurrences {
        my ($common, $ptr, $string, $fn) = @_;

	my $report  = undef;
	my $occcnt = 0;
	#

	#
	# If we have less then the minimum, let us immediately return;
	#
        if ($ptr->{strings}->{$string}->{$fn}->{count} == 0) { return $report; }
	#printf STDERR "STRING_COUNT_OCCURRENCES: string: $string fn: $fn\n%s\n", Dumper($ptr->{strings}->{$string});

	my @ts = sort keys(%{$ptr->{strings}->{$string}->{$fn}->{ts}});
	
	$report  = "Between ".$common->convert_epoch_to_external_datetime($ts[0])." and ".$common->convert_epoch_to_external_datetime($ts[-1])." observed ".$ptr->{strings}->{$string}->{$fn}->{count}." occurences of \'$string\'.";
	push (@{$lptr->{reports}}, $report);
	$occcnt++;
	#
        return $occcnt; #$report;
}