#!/usr/bin/perl
#
# Copyright Hewlett Packard 2012 - 2015
#
# version = 1.26
# This script analyzes configuration related issues for specific disk drive models. 
#
 
use strict;
use warnings;
use Data::Dumper;
    $Data::Dumper::Sortkeys = 1;

my %pd_models = (
   "SEGLE0300GBFC15K"	=> \&analyze_Eagle_PDs,
   "SEGLE0600GBFC15K"	=> \&analyze_Eagle_PDs,
   #"HMRSK2000GBAS07K"	=> \&analyze_MarsK_PDs,
   "HMRSK3000GBAS07K"	=> \&analyze_MarsK_PDs,
   #
   "HCBRE0450GBAS10K"	=> \&analyze_CobraE_PDs,
   "HCBRE0600GBAS10K"	=> \&analyze_CobraE_PDs,
   "HCBRE0900GBAS10K"	=> \&analyze_CobraE_PDs,
   #
   "HKCF0300S5xeN015"   => \&analyze_KingCobra_PDs,
   #"HVIPC0600GBFC15K"   => \&analyze_KingCobra_PDs,
   #
   "DOPE0480S5xnNMRI"   => \&analyze_Optimus_ECO,
   "DOPE1920S5xnNMRI"   => \&analyze_Optimus_ECO,
   #
   "DOPA0480S5xnNMRI"   => \&analyze_Optimus_ASCEND,
   "DOPA0920S5xnNMRI"   => \&analyze_Optimus_ASCEND,
);



# Read in input values
my $common      = $ARGV[0];		# inex_common object
my $type        = $ARGV[1];		# Type of configuration objects
my $ptr         = $ARGV[2];		# Hash of configuration objects of type $ARGV[1]
my $syssn       = $ARGV[3];		# Serial Number of the system
my $model       = $ARGV[4];		# Model of the system (F200, V400, etc)
my $version     = $ARGV[5];		# Baselevel currently running
my $baselevel   = $ARGV[6];		# Baselevel currently running
my $patches     = $ARGV[7];		# List of installed patches
my $insploredir = $ARGV[8];		# Directory name containing the decompressed Insplore

# Define local variables, global in this program though
my %local_hash  = ();			# Hash to store local stuff
my $lptr	= \%local_hash;
my $save_debug  = $common->{debug}->{level};

# In order to facilitate debugging and see the structure of the hash, uncomment the following 2 lines. 
# $common->{debug}->{level} = TRACE_MEMORY;
# $common->dump_memory($ptr, "cfganal_pd: ptr");

# Report issues using the following method:
# $common->report_cfganal_issue($syssn, $type, $pdid, <column>, "Comments go here", <analysis_code>, $cageid, <analysis_code_p1>, <analysis_code_p2>, ..);

$common->trace(TRACE_ROUTINES, "---> cfganal_pd $type, $syssn, $model, $version, $baselevel, $patches");

%{$lptr->{models}} = ();		# Init the local PD Model hash

#printf STDERR "CFGANAL_PD:: hdwfw dump:\n%s\nPDs:\n%s\n", Dumper($common->{database}->{$syssn}->{hddfw}), Dumper($ptr);

foreach my $pdid (sort keys(%{$ptr})) {
	#
	# Look at specific vendors and their specific models...
	# 
	if (exists($ptr->{$pdid}->{vendor}) &&
	    exists($ptr->{$pdid}->{vendor}->{model}) &&
	    exists($pd_models{$ptr->{$pdid}->{vendor}->{model}}))
	{ 
	    $pd_models{$ptr->{$pdid}->{vendor}->{model}}($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $pdid, $ptr->{$pdid}->{vendor}->{model}, $lptr); 
        }
	#
	# Look for the invalid label issue, look for state = failed and detailed state includes "invalid_header"... 3.1.2.MU2 and above...
	#
	if (exists($ptr->{$pdid}->{state}) && exists($ptr->{$pdid}->{detailedstate}))
	{
	    if ($version >= 312)
	    {
		if ($version == 312 && $baselevel < 422) { next; } #below 3.1.2.MU2
		#
		# Look at the states...
		#
		#print STDERR "cfganal_pd:: PDID: $pdid > state: !%s!  detail: !%s!\n", $ptr->{$pdid}->{state}, $ptr->{$pdid}->{detailedstate};
		if ($ptr->{$pdid}->{state} =~ m/failed/i && $ptr->{$pdid}->{detailedstate} =~ m/invalid_label/i)
		{
		    $common->report_cfganal_issue($syssn, $type, $pdid, 28, "See CFI 7677", 31, $ptr->{$pdid}->{id}, "rewrite_pd_label.sh", 7677);
		}
	    }
	}
	#
	# Look for the SEDstate that indicates "owned_by_other_system".
	#
	if (exists($ptr->{$pdid}->{SedState}))
	{
	    if ($version >= 321)
	    {
		if ($ptr->{$pdid}->{SedState} =~ m/owned_by_other_system/i)
		{
		    my $t1 = "controlpd cryptoerase $ptr->{$pdid}->{wwn}";
		    $t1 =~ s/\-//g;
		    my $t2 = '';
		    if ($ptr->{$pdid}->{vendor}->{name} =~ m/seagate/i)
		    {
			$t2 = "\nAdditionally because SEAGATE may need:\ncontrolpd revert <PSID> $ptr->{$pdid}->{wwn}\nPSID located on physical exterior label of disk\nOr replace with a new drive"
		    }
		    #
		    $common->report_cfganal_issue($syssn, $type, $pdid, 29, "To clear: $t1 $t2", 66, $ptr->{$pdid}->{id});
		}
	    }
	}
}

foreach my $pdtype (sort keys(%{$lptr->{models}})) {
	if ($lptr->{models}->{$pdtype}->{count}->{affected}) {$common->report($syssn, 99, $lptr->{models}->{$pdtype}->{count}->{total}, $pdtype, $lptr->{models}->{$pdtype}->{count}->{affected}); }
}

$common->trace(TRACE_ROUTINES, "<--- cfganal_pd");
$common->{debug}->{level} = $save_debug;

return 1;

sub analyze_KingCobra_PDs {
        my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $pdid, $pdtype, $lptr) = @_;
	#
	#printf "KING COBRA... PD: $pdid id: %s  model: %s\n", $ptr->{$pdid}->{id}, $ptr->{$pdid}->{vendor}->{model};
	# 
	if ($version eq "321") {
	    if ($baselevel eq "200") { # Baselevel 200 = MU2
		if ($patches !~ /P13/i) {
		    $common->report_cfganal_issue($syssn, $type, $pdid, 21, "Requires P13, see SAW doc_id: c04700171", 53, $ptr->{$pdid}->{id}, $ptr->{$pdid}->{vendor}->{model}, "not installed");
		}
		else
		{
		    $common->report_cfganal_issue($syssn, $type, $pdid, 21, "See SAW doc_id: c04700171", 53, $ptr->{$pdid}->{id}, $ptr->{$pdid}->{vendor}->{model}, "installed");
		}
	    }
	}
	 
	return;
}

sub analyze_Optimus_ECO {
        my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $pdid, $pdtype, $lptr) = @_;
    	#
	# If the drive hase firmware 3P02 or newer, the drive needs to be reformatted to address any
	#  earlier conditions were not left behind in the LBAs. The drive needs to be vacated before re-formatting.
	# 
	if ($ptr->{$pdid}->{fw} ge "3P02") {
	    $common->report_cfganal_issue($syssn, $type, $pdid, 24, "See CFI_5872 and critical information within CFI.\nSAW: http://saw.cce.hp.com/km/saw/view.do?docId=emr_na-c04780380", 63, $ptr->{$pdid}->{id}, $ptr->{$pdid}->{vendor}->{model}, $ptr->{$pdid}->{fw}, "ECO");
	}
	#
	# If thefirmware on the drive is below 3P02, then the drive is susceptible to the issue.
	# Once updated the drive needs to be reformatted.The drive needs to be vacated before re-formatting.
	#
	if ($ptr->{$pdid}->{fw} lt "3P02") {
	    $common->report_cfganal_issue($syssn, $type, $pdid, 24, "See CFI_5872 and critical information within CFI.\nSAW: http://saw.cce.hp.com/km/saw/view.do?docId=emr_na-c04780380", 64, $ptr->{$pdid}->{id}, $ptr->{$pdid}->{vendor}->{model}, $ptr->{$pdid}->{fw}, "ECO");
	}
	
	return;
}

sub analyze_Optimus_ASCEND {
        my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $pdid, $pdtype, $lptr) = @_;
    	#
	# If the drive hase firmware 3P02 or newer, the drive needs to be reformatted to address any
	#  earlier conditions were not left behind in the LBAs. The drive needs to be vacated before re-formatting.
	# 
	if ($ptr->{$pdid}->{fw} ge "3P02") {
	    $common->report_cfganal_issue($syssn, $type, $pdid, 24, "See CFI_5872 and critical information within CFI.\nSAW: http://saw.cce.hp.com/km/saw/view.do?docId=emr_na-c04780380", 63, $ptr->{$pdid}->{id}, $ptr->{$pdid}->{vendor}->{model}, $ptr->{$pdid}->{fw}, "ASCEND");
	}
	#
	# If thefirmware on the drive is below 3P02, then the drive is susceptible to the issue.
	# Once updated the drive needs to be reformatted.The drive needs to be vacated before re-formatting.
	#
	if ($ptr->{$pdid}->{fw} lt "3P02") {
	    $common->report_cfganal_issue($syssn, $type, $pdid, 24, "See CFI_5872 and critical information within CFI.\nSAW: http://saw.cce.hp.com/km/saw/view.do?docId=emr_na-c04780380", 64, $ptr->{$pdid}->{id}, $ptr->{$pdid}->{vendor}->{model}, $ptr->{$pdid}->{fw}, "ASCEND");
	}
	#
	#
	#
	
	return;
}

sub analyze_Eagle_PDs {
        my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $pdid, $pdtype, $lptr) = @_;

	my $sn    = $ptr->{$pdid}->{sn};
	my $count = 1;

	read_pd_serialnrs ($common, $pdtype, $lptr);
	
	if (!(exists($lptr->{models}->{$pdtype}->{count}->{total}))) { 
	    $lptr->{models}->{$pdtype}->{count}->{affected} = 0;
	    $lptr->{models}->{$pdtype}->{count}->{total}    = 0;
	}
	
	# Check the serial number range first (CFI 3129)
	if (exists($lptr->{models}->{$pdtype}->{sn}->{$sn})) {
	    $common->report_cfganal_issue($syssn, $type, $pdid, 24, "May experience high drive failure rate, see CFI 3129", 27, $ptr->{$pdid}->{id}, "3129");
	    $lptr->{models}->{$pdtype}->{count}->{affected} += $count; $count = 0;
	}

	$lptr->{models}->{$pdtype}->{count}->{total} += 1;

	return;
}
 
sub analyze_MarsK_PDs {
        my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $pdid, $pdtype, $lptr) = @_;
	
	my $sn    = $ptr->{$pdid}->{sn};
	my $count = 1;
	
	read_pd_serialnrs ($common, $pdtype, $lptr);
	
	if (!(exists($lptr->{models}->{$pdtype}->{count}->{total}))) { 
	    $lptr->{models}->{$pdtype}->{count}->{affected} = 0;
	    $lptr->{models}->{$pdtype}->{count}->{total}    = 0;
	}
	
	# Check the serial number range first (CFI 4212)
	if (exists($lptr->{models}->{$pdtype}->{sn}->{$sn})) {
	    if ($version eq "312") {
	        if      ($baselevel eq "422") { # Baselevel 422 = MU2
	            if ($patches !~ /P37/i) {
		        $common->report_cfganal_issue($syssn, $type, $pdid, 24, "Requires P37, see CFI 4212", 23, $ptr->{$pdid}->{id}, "P37", "4212");
		        $lptr->{models}->{$pdtype}->{count}->{affected} += $count; $count = 0;
	            }
	        } elsif ($baselevel eq "484") { # Baselevel 484 = MU3
                    if ($patches !~ /P35/i) {
		        $common->report_cfganal_issue($syssn, $type, $pdid, 24, "Requires P35, see CFI 4212", 23, $ptr->{$pdid}->{id}, "P35", "4212");
		        $lptr->{models}->{$pdtype}->{count}->{affected} += $count; $count = 0;
	            }
	        }
		$common->report_cfganal_issue($syssn, $type, $pdid, 25, "Drive in suspect serial number range, see CFI 4613", 31, $ptr->{$pdid}->{id}, "hgstbgscan", "4613");
		$lptr->{models}->{$pdtype}->{count}->{affected} += $count; $count = 0;
	        
	    }	    
	}
	
	$lptr->{models}->{$pdtype}->{count}->{total} += 1;
	 
	return;
}

sub analyze_CobraE_PDs {
        my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $pdid, $pdtype, $lptr) = @_;
	
	my $sn    = $ptr->{$pdid}->{sn};
	my $count = 1;
	
	read_pd_serialnrs ($common, $pdtype, $lptr);
	
	if (!(exists($lptr->{models}->{$pdtype}->{count}->{total}))) { 
	    $lptr->{models}->{$pdtype}->{count}->{affected} = 0;
	    $lptr->{models}->{$pdtype}->{count}->{total}    = 0;
	}

	# Check the firmware level of the drive
	if (exists($ptr->{$pdid}->{fw})) {
	    if ($ptr->{$pdid}->{fw} eq "3P00") {
		$common->report_cfganal_issue($syssn, $type, $pdid, 24, "Requires 3P02 firmware, see CFI 4212. Addressed in 3.1.3.MU1", 27, $ptr->{$pdid}->{id}, "4212");
		$lptr->{models}->{$pdtype}->{count}->{affected} += $count; $count = 0;
	    }
	}
	# Check the serial number range first (CFI xxxx)
	if (exists($lptr->{models}->{$pdtype}->{sn}->{$sn})) {
	    $common->report_cfganal_issue($syssn, $type, $pdid, 25, "See SIE60709, Bug 117907", 36, $ptr->{$pdid}->{id}, 60709, 117907);
	    $lptr->{models}->{$pdtype}->{count}->{affected} += $count; $count = 0;
        }
        
	$lptr->{models}->{$pdtype}->{count}->{total} += 1;
	 
	return;
}

sub read_pd_serialnrs {
        my ($common, $pdtype, $lptr) = @_;

	# Filename format: %INEX_HOME/defs/cfganal/serialnrs/$pdtype.<nr>
	        
        local *SNDIR;
	local *SNFH;

        if (exists($lptr->{models}->{$pdtype})) { return; }
        
        %{$lptr->{models}->{$pdtype}} = ();

	my $sndir       = $common->{os_obj}->make_filename_platform_dependant($common->{files}->{cfganaldir}."/serialnrs");         
	my @dircontents = ();
	my $snfn        = undef;
	
      	opendir (SNDIR,$sndir);
	@dircontents = grep !/^\.\.?$/, readdir SNDIR;
	closedir SNDIR;
	foreach $snfn (@dircontents) {
	    if (-d $sndir."/".$snfn ) { next; }	# Skip subdirectories

            if ($snfn =~ /^$pdtype.\d+$/) {
                 unless (open(SNFH, "< ".$common->{os_obj}->make_filename_platform_dependant($sndir."/$snfn"))) {
                     $common->trace(TRACE_ERROR, "Unable to open file ".$common->{os_obj}->make_filename_platform_dependant($sndir."/$snfn"));
                     next;
                 }
                 while (<SNFH>) {
                     chop;	# remove CRLF
		     if ($_ !~ /\S+/)    { next; }	# skip empty lines
		     if ($_ =~ /^\s*\#/) { next; }	# skip comment lines
		     # For the rest, right now it is only one serialnr per line. 
		     # We could make this such that we can have multiple serialnrs per line, comma seperated
                     $lptr->{models}->{$pdtype}->{sn}->{$_} = 1;
                 }
                 close (SNFH);
            }
        }

	return;        
}

 
