#!/usr/bin/perl
#
# Copyright Hewlett Packard 2012 - 2015
#
# version = 1.25-1
# This script analyzes configuration related issues for ports. 
#
 
use strict;
use warnings;
use Data::Dumper;

my %port_types = (
   "iscsi"	=> \&check_iscsi_ports,
);

my @node2partner = (1, 0, 3, 2, 5, 4, 7, 6);
#                   N0 N1 N2 N3 N4 N5 N6 N7

# Read in input values
my $common      = $ARGV[0];		# inex_common object
my $type        = $ARGV[1];		# Type of configuration objects
my $ptr         = $ARGV[2];		# Hash of configuration objects of type $ARGV[1]
my $syssn       = $ARGV[3];		# Serial Number of the system
my $model       = $ARGV[4];		# Model of the system (F200, V400, etc)
my $version     = $ARGV[5];		# Baselevel currently running
my $baselevel   = $ARGV[6];		# Baselevel currently running
my $patches     = $ARGV[7];		# List of installed patches
my $insploredir = $ARGV[8];		# Directory name containing the decompressed Insplore

# Define local variables, global in this program though
my %local_hash  = ();			# Hash to store local stuff
my $lptr	= \%local_hash;
my $save_debug  = $common->{debug}->{level};

my $portid      = undef;

# In order to facilitate debugging and see the structure of the hash, uncomment the following 2 lines. 
# $common->{debug}->{level} = TRACE_MEMORY;
# $common->dump_memory($ptr, "cfganal_port: ptr");

# Report issues using the following method:
# $common->report_cfganal_issue($syssn, "ports", $portid, "<comma seperated list of keys>, "Comments go here", <analysis_code>, $cageid, <analysis_code_p1>, <analysis_code_p2>, ..);

$common->trace(TRACE_ROUTINES, "---> cfganal_port $type, $syssn, $model, $version, $baselevel, $patches");

$lptr->{porttypes}->{host}  = 0;
$lptr->{porttypes}->{free}  = 0;
$lptr->{porttypes}->{disk}  = 0;
$lptr->{porttypes}->{rcfc}  = 0;
$lptr->{porttypes}->{rcip}  = 0;
$lptr->{porttypes}->{peer}  = 0;
$lptr->{porttypes}->{fcoe}  = 0;
$lptr->{porttypes}->{iport} = 0;
$lptr->{porttypes}->{iscsi} = 0;
$lptr->{porttypes}->{fs}    = 0;

foreach $portid (sort keys(%{$ptr})) {
        if (exists($port_types{$ptr->{$portid}->{type}})) { 
	    $port_types{$ptr->{$portid}->{type}}($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $portid, $ptr->{$portid}->{type}, $lptr); 
        }
	#
        check_persistent_port($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $portid, $ptr->{$portid}->{type}, $lptr);
        check_intcoal_port_setting($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $portid, $ptr->{$portid}->{type}, $lptr);
	check_16GB_emulex ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $portid, $ptr->{$portid}->{type}, $lptr);
        #
	$lptr->{porttypes}->{$ptr->{$portid}->{type}} += 1;
}

check_limitations_on_port_types($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $lptr);

$common->trace(TRACE_ROUTINES, "<--- cfganal_port");
$common->{debug}->{level} = $save_debug;

return 1;

sub check_iscsi_ports {
        my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $portid, $porttype, $lptr) = @_;

	if (exists($ptr->{$portid}->{iscsi})) {
	    if (exists($ptr->{$portid}->{iscsi}->{dhcp})) {
	        if (defined($ptr->{$portid}->{iscsi}->{dhcp})) {
		    if ($ptr->{$portid}->{iscsi}->{dhcp} eq 1) {
	                $common->report_cfganal_issue($syssn, $type, $portid, "iscsi,dhcp", "iSCSI uses DHCP, must be static IP address", 30, $ptr->{$portid}->{node}.":".$ptr->{$portid}->{slot}.":".$ptr->{$portid}->{port}, "3129");
		    }
	        }
	    }
	}
	return;
}

sub check_persistent_port {
        my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $portid, $porttype, $lptr) = @_;

	my ($partnerport, $partnerportid);
	
	if ($porttype eq "host") {
 	     $partnerport   = $node2partner[$ptr->{$portid}->{node}].":".$ptr->{$portid}->{slot}.":".$ptr->{$portid}->{port};
 	     $partnerportid = sprintf("%02d",$node2partner[$ptr->{$portid}->{node}]).substr($portid, 2, 4);
 	     if (exists($ptr->{$partnerportid})) {
	         if ($ptr->{$partnerportid}->{type} ne "host") {
		     if ($ptr->{$partnerportid}->{type} eq "free") {
		         # For now, lt us only complain if the partner port is listed as free, while this port is listed as host. Both should be listed as 'host'
 		         $common->report_cfganal_issue($syssn, 
			                               "ports", 
						       $partnerportid, 
						       "type", 
						       "No hosts logged in into partner port, which is listed as \'free\'",
						       32, 
			                               $ptr->{$portid}->{node}.":".$ptr->{$portid}->{slot}.":".$ptr->{$portid}->{port},
			                               $ptr->{$portid}->{type},
			                               $partnerport,
			                               $ptr->{$partnerportid}->{type}
			                              ); 
		     } else {
 		         $common->report_cfganal_issue($syssn, 
			                               "ports", 
						       $partnerportid, 
						       "type", 
						       "Partner port is configured as different port type",
						       32, 
			                               $ptr->{$portid}->{node}.":".$ptr->{$portid}->{slot}.":".$ptr->{$portid}->{port},
			                               $ptr->{$portid}->{type},
			                               $partnerport,
			                               $ptr->{$partnerportid}->{type}
			                              ); 
	             }
	         } 
 	     }
	}
        return;
}

sub check_limitations_on_port_types {
        my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $lptr) = @_;
        
        my $nrkeyshost = $lptr->{porttypes}->{host};
        my $nrkeysfree = $lptr->{porttypes}->{free};
        
        if      ($version == 312) {
	    if  (($nrkeyshost+$nrkeysfree)  >= 135) { $common->report($syssn, 34, 135, "3.1.2", $nrkeyshost, $nrkeysfree); }   
        } elsif ($version == 313) {
	    if  (($nrkeyshost+$nrkeysfree)  >= 270) { $common->report($syssn, 34, 270, "3.1.3", $nrkeyshost, $nrkeysfree); }   
        }
        return;
}

sub check_intcoal_port_setting {
        my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $portid, $porttype, $lptr) = @_;

	if ($version >= 312) {
	    if (exists($ptr->{$portid}->{intcoal})) {
	        # "intcoal" does not exist for "iscsi", "rcip"
	        if ($ptr->{$portid}->{connection}->{mode} ne "disk") {
                    # So "host", "rcfc", "peer", "free"
                    if ($ptr->{$portid}->{state} !~ /loss_sync/i) {
                        # Only check if the port is in use, so not if there is no FC cable
                        if ($ptr->{$portid}->{intcoal} !~ /disabled/i) {
			    $common->report_cfganal_issue($syssn, 
			                                  "ports", 
			   			          $portid, 
						          "intcoal", 
						          "For improved performance during production, this setting needs to be disabled",
						          37, 
			                                  $ptr->{$portid}->{node}.":".$ptr->{$portid}->{slot}.":".$ptr->{$portid}->{port},
			                                  "intcoal",
			                                  $ptr->{$portid}->{intcoal},
			                                  "disabled"
			                                 );
			} 
		    } 
	        }
            }
	}
	
        return;
}

sub check_16GB_emulex {
    my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $portid, $porttype, $lptr) = @_;
    #
    my $currversion;
    #
    if (!exists($ptr->{$portid}->{model}) ||
	!exists($ptr->{$portid}->{vendor}) ||
	!exists($ptr->{$portid}->{firmware})) { return 1; }
    #
    #
    #
    if ($ptr->{$portid}->{model}  !~ m/lpe1600\w/i) { return 1; }
    if ($ptr->{$portid}->{vendor} !~ m/emulex/i) { return 1; }
    if ($ptr->{$portid}->{firmware} =~ m/(\d+)\.(\d+)\.(\d+)\.(\d+)/)
    {
	$currversion = sprintf("%04d%04d%04d%04d", $1, $2, $3, $4);
	#print "PORT FW: $currversion\n";
	#
	# Check the version if it greater than 1.1.65.16 we should be good.
	#
	if ($currversion gt "0001000100650016") { return 1; }
    } 

#    printf "PORT 16GB FOUND: %s  %s  %s\n",
#	$ptr->{$portid}->{node}.":".$ptr->{$portid}->{slot}.":".$ptr->{$portid}->{port},
#	$ptr->{$portid}->{model},
#	$currversion;
    $common->report_cfganal_issue($syssn, 
				  "ports", 
				  $portid, 
				  "firmware", 
				  "See CFI 7839, there is potential for poor performance, hung IOs or unresponsive array.",
				  61, 
				  $ptr->{$portid}->{node}.":".$ptr->{$portid}->{slot}.":".$ptr->{$portid}->{port},
				  $ptr->{$portid}->{model},
				  $ptr->{$portid}->{vendor},
				 $ptr->{$portid}->{firmware}
				 );
    
    
    return 1;
}

