#!/usr/bin/perl
#
# (C) Copyright 2012-2015 Hewlett Packard Enterprise Development LP
#
# version = 1.20
# This script analyzes configuration related issues for CPGs. 
#
 
use strict;
use warnings;

# Read in input values
my $common      = $ARGV[0];		# inex_common object
my $type        = $ARGV[1];		# Type of configuration objects
my $ptr         = $ARGV[2];		# Hash of configuration objects of type $ARGV[1]
my $syssn       = $ARGV[3];		# Serial Number of the system
my $model       = $ARGV[4];		# Model of the system (F200, V400, etc)
my $version     = $ARGV[5];		# Baselevel currently running
my $baselevel   = $ARGV[6];		# Baselevel currently running
my $patches     = $ARGV[7];		# List of installed patches
my $insploredir = $ARGV[8];		# Directory name containing the decompressed Insplore

# Define local variables, global in this program though
my %local_hash  = ();			# Hash to store local stuff
my $lptr	= \%local_hash;
my $save_debug  = $common->{debug}->{level};

# In order to facilitate debugging and see the structure of the hash, uncomment the following 2 lines. 
# $common->{debug}->{level} = TRACE_MEMORY;
# $common->dump_memory($ptr, "cfganal_cpg: ptr");

# Report issues using the following method:
# $common->report_cfganal_issue($syssn, $type, $cpgid, <column>, "Comments go here", <analysis_code>, $cageid, <analysis_code_p1>, <analysis_code_p2>, ..);

$common->trace(TRACE_ROUTINES, "---> cfganal_cpg $type, $syssn, $model, $version, $baselevel, $patches");

foreach my $cpgid (sort keys(%{$ptr})) { 
        check_growth_values($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $cpgid);
}

$common->trace(TRACE_ROUTINES, "<--- cfganal_cpg");
$common->{debug}->{level} = $save_debug;

return 1;

sub check_growth_values {
        my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $cpgid) = @_;

        # Minimum CPG grow size.
        # Nr of Nodes   Default  Minimum
        # 1-2             32 GB     8 GB
        # 3-4             64 GB    16 GB
        # 5-6             96 GB    24 GB
        # 7-8            128 GB    32 GB

	my $nrnodes = keys(%{$common->{config}->{$syssn}->{nodes}});
	my ($minimum, $default);
	
	# Note: values need to be in MB         
        if      ($nrnodes >= 7) { $default = 131072;		# 131072 = 1024 * 128
	} elsif ($nrnodes >= 5) { $default =  98304;		#  98304 = 1024 *  96
	} elsif ($nrnodes >= 3) { $default =  65536;		#  65536 = 1024 *  64
	} else                  { $default =  32768;		#  32768 = 1024 *  32
	}
	$minimum = $default/4;
	
	if ($ptr->{$cpgid}->{sdg}->{growth} < $minimum) {
	    $common->report_cfganal_issue($syssn, $type, $cpgid, 17, "Minimum CPG growth must be ".($minimum/1024)."G", 28, $ptr->{$cpgid}->{name}, $nrnodes, ($minimum/1024));
	}

	if ($ptr->{$cpgid}->{devtype} eq "SSD") {
	    if ( int($ptr->{$cpgid}->{sdg}->{growth}/32)*32 != $ptr->{$cpgid}->{sdg}->{growth} ) {
	        # This is not a multiple of 32M
	        $common->report_cfganal_issue($syssn, $type, $cpgid, 2, "CPG growth not multiple of 32M", 29, $ptr->{$cpgid}->{name}, $nrnodes, (int($ptr->{$cpgid}->{sdg}->{growth}/32)*32));
	    }    
	} else {
	    # FC and NL: check if the default values are used. 
	    if ($ptr->{$cpgid}->{sdg}->{growth} != $default) {
	        $common->report_cfganal_issue($syssn, $type, $cpgid, 2, "CPG growth not default value (".($default/1024)."G)", 29, $ptr->{$cpgid}->{name}, $nrnodes, ($default/1024));
	    }
	}
        return;
}
