#!/usr/bin/perl
#
# (C) Copyright 2012-2015 Hewlett Packard Enterprise Development LP
#
# version = 1.26-5
# This script analyzes configuration related issues for nodes (memory, battery, etc). 
#
# version = 1.26-8
# Added t10-DIF checks.
#
 
use strict;
use warnings;
use Data::Dumper;
    $Data::Dumper::Sortkeys = 1;

# Read in input values
my $common      = $ARGV[0];		# inex_common object
my $type        = $ARGV[1];		# Type of configuration objects
my $ptr         = $ARGV[2];		# Hash of configuration objects of type $ARGV[1]
my $syssn       = $ARGV[3];		# Serial Number of the system
my $model       = $ARGV[4];		# Model of the system (F200, V400, etc)
my $version     = $ARGV[5];		# Baselevel currently running
my $baselevel   = $ARGV[6];		# Baselevel currently running
my $patches     = $ARGV[7];		# List of installed patches
my $insploredir = $ARGV[8];		# Directory name containing the decompressed Insplore

# Define local variables, global in this program though
my %local_hash  = ();			# Hash to store local stuff
my $lptr	= \%local_hash;
my $save_debug  = $common->{debug}->{level};
my $nodeid	= undef;

my %node_models = (
       "InServ F200"	=> \&analyze_F_Class,
       "InServ F400"	=> \&analyze_F_Class,
       "InServ T400"	=> \&analyze_T_Class,
       "InServ T800"	=> \&analyze_T_Class,
       "InServ V400"	=> \&analyze_V_Class,
       "InServ V800"	=> \&analyze_V_Class,
       "HP_3PAR 7200"	=> \&analyze_SS7000,
       "HP_3PAR 7400"	=> \&analyze_SS7000,
       "HP_3PAR 7450"	=> \&analyze_SS7000,
);

my %ss7000_ps_models = (
       "726237-001"	=> "726237-001",	# comma seperated list of compatible partnumbers with 726237-001
       "682372-001"	=> "682372-001",	# comma seperated list of compatible partnumbers with 682372-001
);

# In order to facilitate debugging and see the structure of the hash, uncomment the following 2 lines. 
#$common->{debug}->{level} = TRACE_MEMORY;
# $common->dump_memory($ptr, "cfganal_nodes: ptr");

# Report issues using the following method:
# $common->report_cfganal_issue($syssn, $type, $nodeid, "<comma seperated list of keys>, "Comments go here", <analysis_code>, $nodeid, <analysis_code_p1>, <analysis_code_p2>, ..);

$common->trace(TRACE_ROUTINES, "---> cfganal_nodes $type, $syssn, $model, $version, $baselevel, $patches");

if (exists($node_models{$model})) {
        $node_models{$model}($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $insploredir, $lptr);
}
#
foreach $nodeid (sort keys(%{$ptr})) {
	check_cache_status ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $insploredir, $lptr, $nodeid);
    check_dif_errors($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $insploredir, $lptr, $nodeid);
    check_node_slot ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $insploredir, $lptr, $nodeid);
    #
    #
    #
    my $code257cnt = 0;
    foreach my $msgid (0..3)
    {
        if ($common->{config}->{$syssn}->{nodes}->{$nodeid}->{eeprom}->{$msgid}->{Message} !~ m/Uncorrectable Error in Data Cache Memory/) { next; }
        $code257cnt++;
    }
    if ($code257cnt) { $common->report($syssn, 1050, $nodeid, $code257cnt); }
}

$common->trace(TRACE_ROUTINES, "<--- cfganal_nodes");
$common->{debug}->{level} = $save_debug;

return 1;

sub analyze_F_Class {
        my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $insploredir, $lptr) = @_;

	analyze_F_T_Data_DIMMs_MicronTechnology($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $insploredir, $lptr);

	return;
}
 
sub analyze_T_Class {
        my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $insploredir, $lptr) = @_;

	analyze_F_T_Data_DIMMs_MicronTechnology($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $insploredir, $lptr);

	return;
}
 
sub analyze_V_Class {
        my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $insploredir, $lptr) = @_;

	check_ipl5_issue($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $insploredir, $lptr);
	
	return;
}
 
sub analyze_SS7000 {
        my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $insploredir, $lptr) = @_;

	# Check if both power supplies are compatible in the SS7000 Node cage. 
	foreach my $nodeid (keys(%{$ptr})) {
	   if (exists($ptr->{$nodeid}->{power})) {
	       # Only exists for one of the 2 nodes
	       my $ps0type = $ptr->{$nodeid}->{power}->{0}->{AssemblyPartNr};
	       my $ps1type = $ptr->{$nodeid}->{power}->{1}->{AssemblyPartNr};
               #
               # If either of the PS part numbers do not show up treat them as possibly good,
               # because we cannot tell. The PS could have failed or is being upgraded, etc. so
               # still be a compatible part number.
               # 
               if ($ps0type =~ m/--/) { next;  }
               if ($ps1type =~ m/--/) { next;  }
               #
	       if ($ps0type ne $ps1type) {
	           # So the power supplies are not equal. Let us see if we can mix them. 
	           if      (exists($ss7000_ps_models{$ps0type})) {
	                if ($ss7000_ps_models{$ps0type} !~ /$ps1type/) {
		            $common->report_cfganal_issue($syssn, $type, $nodeid, "power,0,AssemblyPartNr", "Mixture of Power Supply PartNrs", 35, $ptr->{$nodeid}->{id}, $ptr->{$nodeid}->{id}+1, $ps0type, $ps1type);
                        }
	           } elsif (exists($ss7000_ps_models{$ps1type})) {
	                if ($ss7000_ps_models{$ps1type} !~ /$ps0type/) {
		            $common->report_cfganal_issue($syssn, $type, $nodeid, "power,1,AssemblyPartNr", "Mixture of Power Supply PartNrs", 35, $ptr->{$nodeid}->{id}, $ptr->{$nodeid}->{id}+1, $ps0type, $ps1type);
                        }
	           } else {
		            $common->report_cfganal_issue($syssn, $type, $nodeid, "power,1,AssemblyPartNr", "Mixture of Power Supply PartNrs", 35, $ptr->{$nodeid}->{id}, $ptr->{$nodeid}->{id}+1, $ps0type, $ps1type);
	           }
	       }
	   }
	}
	return;
}

sub analyze_F_T_Data_DIMMs_MicronTechnology {
        my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $insploredir, $lptr) = @_;

	my ($controldata, $slot);
	my ($first2ch, $mysn, $goodsn);
	
	foreach my $nodeid (keys(%{$ptr})) {
	    foreach $controldata (keys(%{$ptr->{$nodeid}->{memory}})) {
	        foreach $slot (sort keys(%{$ptr->{$nodeid}->{memory}->{$controldata}})) {
                    #
                    # Have to take into account the "slab" memory structure that may be present and would not have
                    # any "part number" data.
                    # 
                    if (!exists($ptr->{$nodeid}->{memory}->{$controldata}->{$slot}->{PartNumber})) { next; }
                    #
	            if      ($ptr->{$nodeid}->{memory}->{$controldata}->{$slot}->{PartNumber} =~ /-335F3$/i) {
 		        $common->report_cfganal_issue($syssn, $type, $nodeid, "memory,$controldata,$slot,PartNumber", "Suspect memory DIMM [-335F3]. Requires Replacement", 25, $ptr->{$nodeid}->{id}, ucfirst($controldata)." Cache", $ptr->{$nodeid}->{memory}->{$controldata}->{$slot}->{SlotID});
		    } elsif ($ptr->{$nodeid}->{memory}->{$controldata}->{$slot}->{PartNumber} =~ /-40BF3$/i) {
		        $first2ch = uc(substr($ptr->{$nodeid}->{memory}->{$controldata}->{$slot}->{SerialNumber}, 0, 2));
		        if      ($first2ch eq "4C") { $goodsn = "0x4C4EE89D";
		        } elsif ($first2ch eq "50") { $goodsn = "0x50000000";
		        } elsif ($first2ch eq "5C") { $goodsn = "0x5C004A04";
		        } elsif ($first2ch eq "5D") { $goodsn = "0x5D140010";
		        } elsif ($first2ch eq "9A") { $goodsn = "0x9AEAD100";
		        } elsif ($first2ch eq "B1") { $goodsn = "0xB1ECDA4D";
		        } elsif ($first2ch eq "D2") { $goodsn = "0xD2362760";
		        } elsif ($first2ch eq "D3") { $goodsn = "0xD3675D74";
		        } elsif ($first2ch eq "D4") { $goodsn = "0xD465FA1A";
		        } elsif ($first2ch eq "D9") { $goodsn = "0xD9495563";
 		        } elsif ($first2ch eq "DF") { $goodsn = "0xDF6F4503";
		        } elsif ($first2ch eq "E0") { $goodsn = "0xE07859FD";
		        } elsif ($first2ch eq "E1") { $goodsn = "0xE16850B7";
		        } elsif ($first2ch eq "E2") { $goodsn = "0xE2728603";
		        } elsif ($first2ch eq "EC") { $goodsn = "0xEC005707";
		        } else { next; 
		        }
		        $mysn = "0x".$ptr->{$nodeid}->{memory}->{$controldata}->{$slot}->{SerialNumber};
                        #printf STDERR "CFGANAL_NODES::analyze_F_T_Data_DIMMs_MicronTechnology: Node: $nodeid CD: $controldata NSP: $slot F2: $first2ch DSN: $mysn [%s] GSN: $goodsn [%s] BAD: %s\n", hex($mysn), hex($goodsn), (hex($mysn) < hex($goodsn));
		        if (hex($mysn) < hex($goodsn)) {
		            $common->report_cfganal_issue($syssn, $type, $nodeid, "memory,$controldata,$slot,SerialNumber", "Suspect memory DIMM [SerialNr]. Requires Replacement", 25, $ptr->{$nodeid}->{id}, ucfirst($controldata)." Cache", $ptr->{$nodeid}->{memory}->{$controldata}->{$slot}->{SlotID});
          		}
		    }
	        }
	    }
	}
	
	return;
}

sub check_cache_status {
        my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $insploredir, $lptr, $nodeid) = @_;
        
        if (exists($ptr->{$nodeid}->{general}->{CacheAvailable})) {
	    if (defined($ptr->{$nodeid}->{general}->{CacheAvailable})) {
	        if ($ptr->{$nodeid}->{general}->{CacheAvailable} ne '100%') {
	            $common->report_cfganal_issue($syssn, $type, $nodeid, "general,CacheAvailable", "No 100% Cache Availability", 33, $ptr->{$nodeid}->{id}, $ptr->{$nodeid}->{general}->{CacheAvailable});
		} 
	    } else {
                $common->report_cfganal_issue($syssn, $type, $nodeid, "general,CacheAvailable", "No 100% Cache Availability", 33, $ptr->{$nodeid}->{id}, "Undefined");
	    }
    }
    return;
}

sub check_dif_errors {
    my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $insploredir, $lptr, $nodeid) = @_;
    
    my $nrkeys = 0;    
    if (exists($ptr->{$nodeid}->{dif})) {
        if (exists($ptr->{$nodeid}->{dif}->{detectors})) {
            $nrkeys = keys(%{$ptr->{$nodeid}->{dif}->{detectors}});
            if ($nrkeys > 0) {
                 $common->report_cfganal_issue($syssn, $type, $nodeid, "general,AssemblySerialNr", "T10 DIF errors observed", 1020, $ptr->{$nodeid}->{id});
	        }
        }
    }
    return;
}

sub check_ipl5_issue {
        my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $insploredir, $lptr) = @_;

	local *INSPLOREDIR;
	local *INTERRUPTSFH;
	
	my ($node, $sfn, $fn);
	my @dircontents	= ();

	if ($version eq "312") {
	    if ($baselevel < 592) {	        # 3.1.2.592 = 3.1.2.MU5
	        if ($baselevel != 562) { 	# 3.1.2.562 = 3.1.2.MU4
	            opendir (INSPLOREDIR,$insploredir);
	            @dircontents = ();
	            @dircontents = grep !/^\.\.?$/, readdir INSPLOREDIR;
	            closedir INSPLOREDIR;
	            foreach $sfn (@dircontents) {
	                if (-d $insploredir."/".$sfn ) { 
		            # Is the the *.node\d+.* directory ?
		            if ($sfn =~ /InSplore\.node\d+./i) {
		                ($node) = $sfn =~ /InSplore\.node(\d+)/i;
			        $fn  = $insploredir.'/'.$sfn.'/proc/interrupts';
			        unless (open(INTERRUPTSFH, "< $fn")) {
			            $common->trace(TRACE_DETAILED, "UNable to open file $fn");
			        } else {
			            $common->trace(TRACE_DETAILED, "File \"$sfn/proc/interrupts\" opened for READONLY");
			            while (<INTERRUPTSFH>) {
			                # Search for: emfc n:s:p  0x5\w
			                if ($_ =~ /emfc\s+\d+:\d+:\d+\s+0x5\w$/i) {
			                    my ($hba) = $_ =~ /emfc\s+\d+:(\d+):\d+/i;
			                    $common->report_cfganal_issue($syssn, 
					                                  $type, 
				                                          sprintf("%02d", $node),
					                                  "general,InFormOS",
							   	          "Node exposed to interrupt stack overflow issue",
								          1010,
								          $node,
								          $hba
								         ); 
		                            last;
			                }
			            }
			            close (INTERRUPTSFH);
			            $common->trace(TRACE_DETAILED, "File \"$sfn\" closed");
			        }
			    }
		        }
	            }
                }        
	    }
	}
	
	return;
}

sub check_node_slot {
    my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $insploredir, $lptr, $nodeid) = @_;
    #
    if (!exists($ptr->{$nodeid}->{slot})) { return; }
    if (scalar(keys %{$ptr->{$nodeid}->{slot}}) != 3) { return; }
    #
    if (!defined($ptr->{$nodeid}->{slot}->{id}) ||
        !defined($ptr->{$nodeid}->{slot}->{date}) ||
        !defined($ptr->{$nodeid}->{slot}->{tz}) ) { return; }
    #
    # So we have valid data...
    # 
    if ($ptr->{$nodeid}->{slot}->{id} != $nodeid) {
        #printf "SLOT ID MIS-MATCH Node: $nodeid is in Slot %s\n", $ptr->{$nodeid}->{slot}->{id};
        $common->report_cfganal_issue ($syssn, "nodes", $nodeid, "id", "Node $nodeid is in the wrong slot", 9101, $nodeid, $ptr->{$nodeid}->{slot}->{id}, $ptr->{$nodeid}->{slot}->{date}." ".$ptr->{$nodeid}->{slot}->{tz});
    } 
    
    return;
} #End of check_node_slot
