#!/usr/bin/perl
#
# (C) Copyright 2012-2015 Hewlett Packard Enterprise Development LP
#
# version = 1.20
# This script analyzes configuration related issues for Remote Copy groups. 
#
 
use strict;
use warnings;

# Read in input values
my $common      = $ARGV[0];		# inex_common object
my $type        = $ARGV[1];		# Type of configuration objects
my $ptr         = $ARGV[2];		# Hash of configuration objects of type $ARGV[1]
my $syssn       = $ARGV[3];		# Serial Number of the system
my $model       = $ARGV[4];		# Model of the system (F200, V400, etc)
my $version     = $ARGV[5];		# Baselevel currently running
my $baselevel   = $ARGV[6];		# Baselevel currently running
my $patches     = $ARGV[7];		# List of installed patches
my $insploredir = $ARGV[8];		# Directory name containing the decompressed Insplore

# Define local variables, global in this program though
my %local_hash  = ();			# Hash to store local stuff
my $lptr	    = \%local_hash;
my $save_debug  = $common->{debug}->{level};

my $rcpyid      = undef;

# In order to facilitate debugging and see the structure of the hash, uncomment the following 2 lines. 
# $common->{debug}->{level} = TRACE_MEMORY;
# $common->dump_memory($ptr, "cfganal_rcopy: ptr");

# Report issues using the following method:
# $common->report_cfganal_issue($syssn, $type, $rcpyid, <column>, "Comments go here", <analysis_code>, $cageid, <analysis_code_p1>, <analysis_code_p2>, ..);

$common->trace(TRACE_ROUTINES, "---> cfganal_rcopy $type, $syssn, $model, $version, $baselevel, $patches");

my $rcgroupcnt = scalar(keys %$ptr);
#
$lptr->{minimums}->{minutes} = 30;
$lptr->{minimums}->{seconds} = 1800;
$lptr->{minimums}->{hours}   = 0.5;
$lptr->{minimums}->{days}    = 0.21;

%{$lptr->{suspectvv}} = ();
#
if ($rcgroupcnt > 1)
{
    $lptr->{minimums}->{minutes} = 60;
    $lptr->{minimums}->{seconds} = 3600;
    $lptr->{minimums}->{hours}   = 1;
    $lptr->{minimums}->{days}    = 0.42;
}

foreach $rcpyid (keys(%{$ptr})) {
    
    if ($rcpyid !~ /\d+/) { next; }		# skip non Id keys
	check_peer_persistence_VMware       ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $rcpyid, $lptr);
	check_peer_persistence_WindowsServer($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $rcpyid, $lptr);
	if ($version ge "321"){
	    collect_dedup_periodcodity ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $rcpyid, $lptr); 
	}
}

check_dedup_periodcodity ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $lptr);
check_target_policies ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $lptr);

$common->trace(TRACE_ROUTINES, "<--- cfganal_rcopy");
$common->{debug}->{level} = $save_debug;

return 1;

sub check_peer_persistence_VMware {
    my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $rcpyid, $lptr) = @_;
        
    my ($vvid, $hostname, $hostid, $nrkeys, $target);

	my @hosts                         = ();
	my %suspect_hosts_auto_failover   = ();
	my %suspect_hosts_path_management = ();
	
	foreach $target (keys(%{$ptr->{$rcpyid}})) {
	    if ($ptr->{$rcpyid}->{$target}->{mode} !~ /^sync/i) { next; }       # Peer Persistence is only valid in Sync Mode. 
	    %suspect_hosts_auto_failover   = ();
	    %suspect_hosts_path_management = ();
	    foreach $vvid (keys(%{$ptr->{$rcpyid}->{$target}->{vv}})) {
	        @hosts = ();
	        foreach $hostname (keys(%{$common->{config}->{$syssn}->{vv}->{$vvid}->{hosts}})) {
	            if ($hostname =~ /^set:/) { 
		            @hosts = keys(%{$common->{config}->{$syssn}->{hostset}->{substr($hostname,4)}->{members}}); 
		        } else { 
		            push (@hosts, $common->{parse_config_host_obj}->name2id($hostname)); 
                }
	            foreach $hostid (@hosts) {
	                unless ($common->{config}->{$syssn}->{host}->{$hostid}->{persona}->{id}) { next; }
	                if ($common->{config}->{$syssn}->{host}->{$hostid}->{persona}->{id} == 11) {
   		                # VV presented to host with persona 11, so Peer Persistence is possible
                        if ($ptr->{$rcpyid}->{$target}->{options}->{auto_failover} eq "N") {
				            if (($version == 312 && $baselevel >= 422) || $version >= 313) {
			                    # Auto Failover policy is introduced in 3.1.2.MU2, so do not complain on prior to 3.1.2.MU2 versions
				                $suspect_hosts_auto_failover{$common->{config}->{$syssn}->{host}->{$hostid}->{name}} = 1;
			                }
		                }
		                if ($version >= 313) {
		                    if ($ptr->{$rcpyid}->{$target}->{options}->{path_management} eq "N") {
				                $suspect_hosts_path_management{$common->{config}->{$syssn}->{host}->{$hostid}->{name}} = 1;
	                        }
	                    }
                    }
                }
	        }
        }

	    $nrkeys = keys(%suspect_hosts_auto_failover);
	    if ($nrkeys > 0) {
  	        @hosts = ();
	        @hosts = keys(%suspect_hosts_auto_failover);
	        $common->report_cfganal_issue($syssn, $type, $rcpyid, 28, "Member(s) presented to hosts with persona 11 require \'auto_failover = Y\', see CFI 5107", 26, $ptr->{$rcpyid}->{$target}->{name}, $nrkeys, "ESX", join(",",@hosts), "Auto_Failover", "5107");
        }

	    $nrkeys = keys(%suspect_hosts_path_management);
	    if ($nrkeys > 0) {
	        @hosts = ();
	        @hosts = keys(%suspect_hosts_path_management);
	        $common->report_cfganal_issue($syssn, $type, $rcpyid, 29, "Member(s) presented to hosts with persona 11 require \'path_management = Y\' if on InFormOS 3.1.3 or higher, see CFI 5107", 26, $ptr->{$rcpyid}->{$target}->{name}, $nrkeys, "ESX", join(",",@hosts), "Path_Management (InFormOS 3.1.3 and higher)", "5107");
        }
    }
	 
    return;
}

sub check_peer_persistence_WindowsServer {
    my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $rcpyid, $lptr) = @_;
    #
	# Peer Persistance support for WINDOWS SERVERs was not until 321.
	# CLX can be recognized as "auto_failover" policy not being set. So we should not alarm if that policy is not set. 
	# However, if "path_management" is set and "auto_failover" is not set, then we will raise our voice. 
	#
    if ($version < 321) { return; }
        
    my ($vvid, $hostname, $hostid, $nrkeys, $target, $report);

	my @hosts                         = ();
	my %suspect_hosts_auto_failover   = ();
	my %suspect_hosts_path_management = ();
	
	foreach $target (keys(%{$ptr->{$rcpyid}})) {
        if (!exists($ptr->{$rcpyid}->{$target}->{mode}) || !defined($ptr->{$rcpyid}->{$target}->{mode})) { next; }
	    if ($ptr->{$rcpyid}->{$target}->{mode} !~ /^sync/i) { next; }       # Peer Persistence is only valid in Sync Mode. 
	    %suspect_hosts_auto_failover   = ();
	    %suspect_hosts_path_management = ();
	    foreach $vvid (keys(%{$ptr->{$rcpyid}->{$target}->{vv}})) {
	        @hosts = ();
	        foreach $hostname (keys(%{$common->{config}->{$syssn}->{vv}->{$vvid}->{hosts}})) {
	            if ($hostname =~ /^set:/) { 
		            @hosts = keys(%{$common->{config}->{$syssn}->{hostset}->{substr($hostname,4)}->{members}}); 
		        } else { 
 		            push (@hosts, $common->{parse_config_host_obj}->name2id($hostname)); 
                }
	            foreach $hostid (@hosts) {
	                unless ($common->{config}->{$syssn}->{host}->{$hostid}->{persona}->{id}) { next; }
	                if ($common->{config}->{$syssn}->{host}->{$hostid}->{persona}->{id} == 15) {
   		                # VV presented to host with persona 15, so Peer Persistence is possible
                        if ($ptr->{$rcpyid}->{$target}->{options}->{auto_failover} eq "Y") {
		                    $suspect_hosts_auto_failover{$common->{config}->{$syssn}->{host}->{$hostid}->{name}} = 1;
					    }
		                if ($ptr->{$rcpyid}->{$target}->{options}->{path_management} eq "Y") {
			                $suspect_hosts_path_management{$common->{config}->{$syssn}->{host}->{$hostid}->{name}} = 1;
			            }
		            }
                }
            }
	    }

	    $nrkeys = keys(%suspect_hosts_auto_failover);
	    if ($nrkeys > 0) {
	        @hosts = ();
		    foreach $hostname (keys(%suspect_hosts_auto_failover)) {
		        if (!(exists($suspect_hosts_path_management{$hostname}))) { push (@hosts, $hostname); }
		    }
		    if (@hosts >  0) {
	            $common->report_cfganal_issue($syssn, $type, $rcpyid, 28, "Member(s) presented to hosts with persona 15 with \'auto_failover = Y\' also require \'path_management = Y\', see CFI 5107", 
				26, $ptr->{$rcpyid}->{$target}->{name}, $nrkeys, "Windows", join(",",@hosts), "Auto_Failover and Path_Management", "5107");
		    }
        }

	    $nrkeys = keys(%suspect_hosts_path_management);
	    if ($nrkeys > 0) {
	        @hosts = ();
		    foreach $hostname (keys(%suspect_hosts_path_management)) {
		        if (!(exists($suspect_hosts_auto_failover{$hostname}))) { push (@hosts, $hostname); }
		    }
		    if (@hosts >  0) {
	            $common->report_cfganal_issue($syssn, $type, $rcpyid, 28, "Member(s) presented to hosts with persona 15 with \'path_management = Y\' also require \'auto_failover = Y\', see CFI 5107", 
				26,$ptr->{$rcpyid}->{$target}->{name}, $nrkeys, "Windows", join(",",@hosts), "Path_Management and Auto_Failover", "5107");
		    }
        }
    }
	    
}

sub collect_dedup_periodcodity {
    my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $rcpyid, $lptr) = @_;
    
    my ($target, $vvid);

    my %suspect_vvs = ();
    
    foreach $target (keys(%{$ptr->{$rcpyid}})) {
	    if ($target eq 'cfganal') { next; }
	    if ($ptr->{$rcpyid}->{$target}->{mode} !~ /^periodic/i) { next; }       # Only concerned with RC groups that are set periodic 
	    #
	    %suspect_vvs = ();
	    #
	    foreach $vvid (sort keys(%{$ptr->{$rcpyid}->{$target}->{vv}})) {
	       #
	       # We are looking for VVs which are TDVVs...
	       if ($common->{config}->{$syssn}->{vv}->{$vvid}->{prov} ne 'tdvv') { next; }
	       $lptr->{suspectvv}->{$rcpyid}->{$target}->{$vvid} = $common->{config}->{$syssn}->{vv}->{$vvid}->{name};
	       #
	    }
    }
    return;
} #End of collect_dedup_periodcodity

sub check_dedup_periodcodity {
    my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $lptr) = @_;
    
    my ($target, $vvid);
    #
    # If no TDVVs in the RC group, then nothing to be concerned about.
    #
    if (!scalar(keys %{$lptr->{suspectvv}})) { return; }
	$lptr->{minimums}->{minutes} = 60;
	$lptr->{minimums}->{seconds} = 3600;
	$lptr->{minimums}->{hours}   = 1;
	$lptr->{minimums}->{days}    = 0.42;

    #
    # Now loop through the rc groups with TDVVs to report on them if need be...
    foreach my $rcpyid (sort keys %{$lptr->{suspectvv}}) {
	    foreach my $target (sort keys %{$lptr->{suspectvv}->{$rcpyid}}) {
	        #
	        # Okay we have a TDVV in the RC group...
	        # Now check the interval.
	        # Since interval values are integer only, we can ignore the intervals
	        # that are based on hour or day. If the interval is minute or seconds
	        # based then we need to look at them a little further. If we have a zero (0) value,
	        # it means that auto-synch has been disabled, so we can ignore that.
	        #
	        if ($ptr->{$rcpyid}->{$target}->{options}->{period} =~ m/(\d+)m/i) {
		        if ($1 == 0 || $1 >= $lptr->{minimums}->{minutes}) { next; }
		        $common->report_cfganal_issue($syssn, $type, $rcpyid, 27, "Due to RCCopyGroup containing a TDVV, Asynch Period must be at least $lptr->{minimums}->{minutes} minutes", 40, $ptr->{$rcpyid}->{$target}->{name}, "$lptr->{minimums}->{minutes}m");
	        } elsif ($ptr->{$rcpyid}->{$target}->{options}->{period} =~ m/(\d+)s/i) {
		        if ($1 == 0 || $1 >= $lptr->{minimums}->{seconds}) { next; }
		        $common->report_cfganal_issue($syssn, $type, $rcpyid, 27, "Due to RCCopyGroup containing a TDVV, Asynch Period must be at least $lptr->{minimums}->{seconds} seconds", 40, $ptr->{$rcpyid}->{$target}->{name}, "$lptr->{minimums}->{seconds}s");
			}
	    } #end of target loop
    } #end of rcpyid loop
	 
    return;
} #End of check_dedup_periodcodity


sub check_target_policies {
    my ($common, $type, $ptr, $syssn, $model, $version, $baselevel, $patches, $lptr) = @_;
    
    my ($target, $vvid);
    #
    # If there are no remote systems then there is nothing to do!
    #
    if (!exists($common->{config}->{$syssn}->{remsystems})) { return; }
    #
    # Now loop through the remote rc systems...
    #
    foreach my $remid (sort keys %{$common->{config}->{$syssn}->{remsystems}}) {
	    #
	    # Look at the policies and check for mirror_config, if there we are okay.
	    # If we see no_mirror_config, we need to alert the masses!
	    #
	    if ($common->{config}->{$syssn}->{remsystems}->{$remid}->{policy} !~ m/no_mirror_config/) { next; }
	    #
	    # OH MY, mirror_config is disabled, flag it!
	    #
	    $common->report_cfganal_issue($syssn, 'remsystems', $remid, 6, "mirror_config should be enabled. Enable with command:\nsetrcopytarget pol mirror_config $common->{config}->{$syssn}->{remsystems}->{$remid}->{name}",
				      9300,
				      $common->{config}->{$syssn}->{cluster}->{name},
				      $syssn,
				      $common->{config}->{$syssn}->{remsystems}->{$remid}->{name},
				      $common->{config}->{$syssn}->{remsystems}->{$remid}->{sysid});
	
    } #end of remid loop
	 
    return;
} #End of check_target_policies