// (C) Copyright 2020 Hewlett-Packard Enterprise Company, L.P.

define([
    'hp/core/NotificationFormatter',
    'fs/view/settings/restore/RestoreAlertNotificationFormatter',
    'hp/core/StatusState',
    'hp/services/Log'],
function(notificationFormatter, restoreAlertFormatter, statusState, log) {"use strict";

    // Generic alert notification formatter. Takes an alert from the index service or
    // an actual alert resource and populates the notification object appropriately.
    var AlertNotificationFormatter = ( function() {

        // This is a singleton module
        function AlertNotificationFormatter() {

            // Add resolution text to the notification
            function addResolution(text, notification) {
                if (text) {
                    notificationFormatter.addResolution(notification,
                        notificationFormatter.replaceInlineLinks(text));
                }
            }

            // Format an alert resource
            function formatAlert(notification, alert) {
                notification.uri = alert.uri;
                notification.status = statusState.getHealthStatus(alert.severity);
                notification.state = alert.alertstate;
                notification.summary =
                    notificationFormatter.replaceInlineLinks(alert.description);
                notification.timestamp = alert.created;
                notification.owner = alert.owner;
                notification.sourceUri = alert.associatedResourceUri;
                notification.sourceCategory = alert.associatedResourceCategory;
                addResolution(alert.correctiveAction, notification);
                if (alert.actions) {
                    $.each(alert.actions, function (index, action) {
                        notificationFormatter.addAction(notification,
                        notificationFormatter.formatLink(action));
                    });
                }
            }

            // Format an index result
            function formatIndex(notification, indexResult) {
                notification.uri = indexResult.uri;
                notification.status = statusState.getHealthStatus(indexResult.status);
                notification.state =
                    indexResult.state || indexResult.attributes.state;
                // work around alerts not using 'name' until mapping changes come in
                notification.summary =
                    notificationFormatter.replaceInlineLinks(
                        indexResult.name || indexResult.description);
                notification.timestamp = indexResult.created;
                notification.owner = indexResult.attributes.owner;
                notification.sourceUri =
                    indexResult.attributes.associatedResourceUri;
                notification.sourceName =
                    indexResult.attributes.associatedResourceName;
                notification.sourceCategory =
                    indexResult.attributes.associatedResourceCategory;
                notification.progress = indexResult.attributes.percentComplete;
                notification.details = indexResult.details;
            }

            // Figure out whether we have an index item or a full resource and
            // format appropriately.
            function format(notification, item) {
                if ('IndexResource' === item.type) {
                    formatIndex(notification, item);
                } else {
                    formatAlert(notification, item);
                }
            }

            // ### register()
            // Register this formatter with the NotificationFormatter.
            this.register = function() {
                // Need to re-evaluate restore alert formatter post Atlas 2.0
                restoreAlertFormatter.register(this);
                notificationFormatter.register({
                    category: 'alerts',
                    associatedResourceCategory: 'any',
                    index: true,
                    format: format});
            };

            // ### formatAlert()
            // Format an alert resource. Used by custom formatter to do generic formatting.
            this.formatAlert = formatAlert;

            // ### formatIndex()
            // Format an alert index item. Used by custom formatter to do generic formatting.
            this.formatIndex = formatIndex;

            // ### format()
            // Format an item.
            this.format = format;
        }

        return new AlertNotificationFormatter();
    }());

    return AlertNotificationFormatter;
});
