// (C) Copyright 2020 Hewlett-Packard Enterprise Company, L.P.
//
// Returns a singleton instance of an AlertsPanelPresenter
//
define(['hp/core/EventDispatcher',
    'fs/model/alerts/AlertPanelResource'],
function(EventDispatcher, AlertPanelResource) {"use strict";

    var AlertsPanelPresenter = (function() {

        /**
         * @constructor
         * @type {AlertsPanelPresenter}
         */
        function AlertsPanelPresenter() {
            var resource = new AlertPanelResource(),
                dispatcher = new EventDispatcher();

            // public
            // gets the from the AlertResource's severity
            // used for displaying the status icon
            this.getStatus = function(severity) {
                var status;
                if (severity) {
                    switch(severity) {
                        case "OK":
                            status = "ok";
                            break;
                        case "Unknown":
                            status =  "unknown";
                            break;
                        case "Disabled":
                            status =  "disabled";
                            break;
                        case "Warning":
                            status =  "warning";
                            break;
                        case "Critical":
                            status = "error";
                            break;
                    }
                }
                return status;
            };
            
            this.getAlertData = function() {
                // calculate counts
                var allAlerts = resource.getAlerts();
                var alertCount = resource.getAlertCount();
                var returnAlertData = null;

                if (allAlerts){
                    returnAlertData = {
                        alerts: [],
                        critCount: 0,
                        majorCount: 0,
                        okCount: 0
                    };

                    returnAlertData.critCount = alertCount.criticalCount;
                    returnAlertData.majorCount = alertCount.warningCount;
                    returnAlertData.okCount = alertCount.okCount;

                    // Build return array
                    $.each(allAlerts, function (idx, singleAlert) {
                        returnAlertData.alerts.push({
                            severity: singleAlert.severity,
                            description: singleAlert.description,
                            alertDate: singleAlert.created
                        });
                    });
                }
                return returnAlertData;
            };

            this.setUri = function (newUri) {
                resource.setUri(newUri);
            };

            /**
             * @public
             * Initialize this presenter. Should only be called once.
             */
            this.init = function () {
                // Register for the resource's alertChange event
                resource.on("alertChange", function(alertObj) {
                    dispatcher.fire("alertChange", alertObj);
                });
            };
            /**
             * Add a listener for a specified event. (For the view)
             * @public
             * @param {string} eventName The name of the event.
             * @param {function(...)}
             */
            this.on = function(eventName, callback) {
                // Register with dispatcher (presenter's list of listeners)
                dispatcher.on(eventName, callback);
                if (eventName === 'alertChange'){
                    dispatcher.fire('alertChange', resource.getAlerts());
                }
            };

            this.off = function(eventName, callback) {
                dispatcher.off(eventName, callback);
            };
        }

        return AlertsPanelPresenter;
    }());

    return AlertsPanelPresenter;
});
