// (C) Copyright 2020 Hewlett-Packard Enterprise Company, L.P.

define(['fs/services/backup/BackupService',
        'hp/model/TaskFollower'],
function(backupService, taskFollower) { "use strict";

    var BackupActionPresenter = (function() {
    
        var IN_PROGRESS = 'IN_PROGRESS';
        var SUCCEEDED = 'SUCCEEDED';

        /**
         * @constructor
         */
        function BackupActionPresenter() {
            
            function trackBackupTaskStatus(taskResource, handlers) {
                taskFollower.watchTask(taskResource.uri, handlers);
            }
            
            /**
             * Start backup. 
             *
             * @param handlers The changed and error handlers
             */
            this.startBackup = function(handlers) {
                backupService.startBackup({
                            success: function(taskResource) {
                                trackBackupTaskStatus(taskResource, handlers);
                            },
                            error: handlers.error
                });
            };
            
            /**
             * Cancel the specified backup. 
             *
             * @param backupUri The uri of the backup to cancel
             * @param handlers The success and error handlers
             */
            this.cancelBackup = function(backupUri, handlers) {
                backupService.cancelBackup(backupUri, handlers);     
            };
            
            /**
             * Get the most recent backup resource in order to download the backup 
             *
             * @param handlers The success and error handlers
             */
            this.getLastBackupForDownload = function(handlers) {
                backupService.getLastBackup({
                    success: 
                        function(backupResources) {
                            var backupResource = (backupResources.count > 0) ? 
                                backupResources.members[0] : null;
                            handlers.success(backupResource);                     
                        }, 
                     error: 
                        handlers.error
                });
            };
            
            /**
             * Return true if the requested backup is in progress
             *
             * @param backupResource The resource of the backup to check
             */
            this.isBackupInProgress = function(backupResource) {
                return backupResource.status === IN_PROGRESS;
            };
            
            /**
             * Return true if the requested backup completed successfully
             *
             * @param backupResource The resource of the backup to check
             */
            this.isGoodBackup = function(backupResource) {
                return backupResource.status === SUCCEEDED;
            };
            
            /**
             * Return the download URI for the requested backup
             *
             * @param backupResource The resource of the requested backup
             */
            this.getDownloadUri = function(backupResource) {
                return backupResource.downloadUri;
            };
            
            /**
             * Initialize
             */
            this.init = function() {
               // nothing
            };

        }

        return new BackupActionPresenter();
    }());

    return BackupActionPresenter;
});
