// (C) Copyright 2020 Hewlett-Packard Enterprise Company, L.P.
//
// Returns a singleton instance of an WelcomePresenter
//
define(['hp/core/EventDispatcher',
        'fs/services/welcome/WelcomeService', 
        'hp/core/Localizer',
        'hp/lib/date'],
function(EventDispatcher, welcomeService, localizer) { "use strict";

    var WelcomePresenter = (function() {
        /**
         * @constructor Singleton class containing presentation
         *              logic for the welcome page
         * @type {WelcomePresenter}
         */
        function WelcomePresenter() {
            /*
             * Given a value that may be below minimums, return the properly formatted number and units.
             * See comments below for the conversion and formatting logic.
             */
            function rescaleUINodeStatus(statusError, kilo, value, smallUnits, bigUnits) {
                var vu = {};
                
                if (value >= kilo) {
                    if (statusError) {
                        // If the value is big enough to scale but not big
                        // enough to meet the minimum, we have two choices. We
                        // could return the un-scaled value ...
                        // vu.value = value.toString();
                        // vu.units = smallUnits
                        // ... or, scale but make sure we round down ...
                        vu.value = ((value / kilo) - 0.051).toFixed(1);
                        vu.units = bigUnits;
                    } else {
                        // Re-scale to one decimal place in larger units.
                        vu.value = (value / kilo).toFixed(1);
                        vu.units = bigUnits;
                    }
                } else {
                    vu.value = value.toString();
                    vu.units = smallUnits;
                }
                vu.value = vu.value.replace(".0", "");
                return vu;
            }
  
            /*
             * Convert units to more readable form i.e. MGz to GHz if needed
             */
            function getUINodeStatus(nodeStatus) {
                var lanVU = rescaleUINodeStatus(nodeStatus.lanStatusError,
                        1000, nodeStatus.lan, nodeStatus.lanUnits,
                        localizer.getString('fs.welcome.nodeStatus.lanSpeedUnitsGb'));
                var cpuVU = rescaleUINodeStatus(nodeStatus.cpuStatusError,
                        1000, nodeStatus.cpuSpeed, nodeStatus.cpuSpeedUnits,
                        localizer.getString('fs.welcome.nodeStatus.cpuSpeedUnitsGHz'));
                var memVU = rescaleUINodeStatus(nodeStatus.memStatusError,
                        1024, nodeStatus.memory, nodeStatus.memoryUnits,
                        localizer.getString('fs.welcome.nodeStatus.memorySizeUnitsTb'));
                var lanValue = ( nodeStatus.lanStatusError === null && lanVU.value == 0 ) ? '&mdash;' : lanVU.value;
                var lanError = ( nodeStatus.lanStatusError === null && nodeStatus.lanLinkStatusError === null ) ? null : true;
                var viewNodeStatus = {
                    cpu: nodeStatus.cpu,
                    memory: memVU.value,
                    memoryUnits : memVU.units,
                    lan : lanValue,
                    lanUnits : lanVU.units,
                    cpuSpeed: cpuVU.value,
                    cpuSpeedUnits : cpuVU.units,
                    lanStatusError : lanError,
                    cpuStatusError : nodeStatus.cpuStatusError,
                    memoryStatusError : nodeStatus.memoryStatusError
                    
                };
                return  viewNodeStatus;
            }
            /**
             * Retrieve the current network configuration.
             *
             * @param {{success:function(NodeStatus):void, error:function(ErrorInfo):void}} handlers
             *     Handlers for success and error return
             */
            this.getNodeStatus= function(handlers) {
                welcomeService.getNodeStatus({
                    success: function(nodeStatus) {
                        handlers.success(getUINodeStatus(nodeStatus));           
                    },
                    error: handlers.error
                });
            };
 
            
            /* The helper functions for minimum configuration requirements
             */
            this.isLANMinConfigMet = function(nodeStatusObject){
                return (nodeStatusObject.lanStatusError == null);
            };
  
            this.isCPUMinConfigMet = function(nodeStatusObject){
                return (nodeStatusObject.cpuStatusError == null);
            };
            
            this.isMemoryMinConfigMet = function(nodeStatusObject){
                return (nodeStatusObject.memoryStatusError == null);
            };
        }

        return new WelcomePresenter();

    }());

    return WelcomePresenter;

});
