//(C) Copyright 2020 Hewlett-Packard Enterprise Company, L.P.

define(['hp/services/REST', 'jquery'], function(REST) { "use strict";

    var AlertsService = (function() {
        
        function AlertsService() {

            /**
             * Retrieves a list of alerts based on the filters passed in
             * @param filters the filters used to retrieve the specific set of alerts
             * @param queryStrings the query (broken up into an array) used to retrieve the specific set of alerts
             * @param severityCount boolean value for whether a count based on severity should be returned in addition
             *  to the list of alerts
             * @param count the number of alerts to retrieve
             * @param sort an array containing the order the alerts should be sorted
             * @param {Object} handlers success and error handlers to call upon success or failure
             *  of the REST request.
             */
            function getAlerts(filters, queryStrings, severityCount, count, sort, handlers) {
                var uri = '/alerts/rest/resources?';
                var params = ['start=0'];
                
                // check if count was passed in
                if(count !== null && count !== undefined){
                    params.push('count=' + count);
                }

                // add the filters
                if (filters) {
                    for(var i = 0;i<filters.length;i++) {
                        params.push('filter="' + filters[i] + '"');
                    }
                }

                if(queryStrings && queryStrings.length > 0) {
                    var expression = $.map(queryStrings, function(query, index) {
                        return '(' + query + ')';
                    }).join(' AND ');
                    params.push('query="' + expression + '"');
                }
                
                // add sort parameter
                if(sort !== null && sort !== undefined){
                    params.push('sort=' + sort.join('%2C'));
                }

                if (severityCount) {
                    params.push('view=alertSummary');
                }

                REST.getURI(uri + params.join('&'), {
                    success: function(result) {
                        if (result && result.members) {
                            // make sure count matches members.length since at this point /rest/alerts doesn't set it correctly
                            result.count = result.members.length;
                        }

                        if (handlers && handlers.success) {
                            handlers.success(result);
                        }
                    },
                    error: handlers ? handlers.error: null
                }, {apiVersion: 2});
            }
            
            this.getAllUsers = function(handlers) {
                REST.getURI("/security-base/rest/user", handlers, {apiVersion: 2});
            };
            
            this.getOwners = function(handlers) {
                REST.getURI("/alerts/rest/resources/OwnerList", handlers,  {apiVersion: 2});
            };

            /**
             * Retrieves the event resource with the given uri
             * @param The unique identifier for a given event
             * @param {Object} handlers success and error handlers to call upon success or failure
             *          of the REST request.
             */
            this.getEventByUri = function(uri, handlers) {
                REST.getURI(uri + '?view=viewDescription', handlers, {apiVersion: 2});
            };

            /**
             * Deletes the selected resource from the database
             * @param uri The unique identifier of the resource to be deleted
             * @param {Object} handlers success and error handlers to call upon success or failure
             *  of the REST request.
             */
            this.deleteResourceByUri = function(uri, handlers) {
                REST.deleteURI(uri, handlers);
            };

            /**
             * Sets Alert state for the selected alert
             * @param uri The unique identifier of the alert to be updated
             * @param state The new state
             * @param {Object} handlers success and error handlers to call upon success or failure
             *  of the REST request.
             */
            this.setAlertState = function(uri, state, handlers) {
                var jsonObject = {"alertState":state};
                REST.putObject(uri, jsonObject, null, handlers, false);
            };

            /**
             * Adds a note to the selected alert
             * @param uri the uri of the alert to be updated
             * @param notes the notes to be added to the alert
             * @param {Object} handlers success and error handlers to call upon success or failure
             *  of the REST request.
             */
            this.addAlertNote = function(uri, note, handlers) {
                var jsonObject = {"notes":note};
                REST.putObject(uri, jsonObject, null, handlers, false);
            };
            
            this.deleteAlertNote = function(uri, handlers) {
                REST.deleteURI(uri, handlers);
            };

            /**
             * Sets the assignedToUser of an alert
             * @param uri The unique identifier of the alert to be updated
             * @param user the user to set the assignedToUser field of the alert to
             * @param {Object} handlers success and error handlers to call upon success or failure
             *  of the REST request.
             */
            this.assignUserToAlertByUri = function(uri, user, handlers) {
                var jsonObject = {"assignedToUser":user};
                REST.putObject(uri, jsonObject, null, handlers, false);
            };

            /**
             * Retrieves a list of alerts based on the filters passed in
             * @param filters the filters used to retrieve the specific set of alerts
             * @param severityCount boolean value for whether a count based on severity should be returned in addition
             *  to the list of alerts
             * @param count the number of alerts to retrieve
             * @param sort an array containing the order the alerts should be sorted
             * @param {Object} handlers success and error handlers to call upon success or failure
             *  of the REST request.
             */
            this.getAlertsByFilter = function(filters, severityCount, count, sort, handlers) {
                getAlerts(filters, null, severityCount, count, sort, handlers);
            };

            /**
             * Retrieves a list of alerts based on the query strings passed in
             * @param queryStrings the query (broken up into an array) used to retrieve the specific set of alerts
             * @param severityCount boolean value for whether a count based on severity should be returned in addition
             *  to the list of alerts
             * @param count the number of alerts to retrieve
             * @param sort an array containing the order the alerts should be sorted
             * @param {Object} handlers success and error handlers to call upon success or failure
             *  of the REST request.
             */
            this.getAlertsByQuery = function(queryStrings, count, sort, handlers) {
                getAlerts(null, queryStrings, false, count, sort, handlers);
            };
        }

        return new AlertsService();
    }());

    return AlertsService;
});
