// (C) Copyright 2020 Hewlett-Packard Enterprise Company, L.P.

define(['hp/core/Localizer',
        'hp/core/UrlFragment',
        'hp/view/FormStateView',
        'hp/core/Router',
        'fs/view/settings/backup/BackupTaskNotificationFormatter',
        'fs/presenter/settings/BackupActionPresenter',
        'fs/view/settings/backup/BackupStatusView',
        'jquery'],
function(localizer, urlFragment, FormStateView, router, 
    BackupTaskNotificationFormatter, presenter, statusView) {"use strict";

    var BackupStopView = ( function() {
    
        /**
         * Constants
         */
         var SETTINGS_SHOW_SECTION = '#hp-settings-show';
         var NOTIFY_STATUS_ERROR = 'error';
         var ADD_NOTIFICATION = 'addNotification';
         var STOP_BUTTON = '#fs-settings-backup-stop-button';
         var CANCEL_BUTTON = '#fs-settings-backup-stop-cancel-button';
         var BACKUP_STOP_FORM = '#fs-settings-backup-stop-form';
        
        /**
         * Constructor
         */
        function BackupStopView() {
        
            var taskFormatter = new BackupTaskNotificationFormatter();
            var cancelErrorReported = false;
            var formStateView = null;
            var params = null;
            var backupUri = null;
            
            function addErrorNotification(summary, error, resolution) {
                var notification = {
                    summary    : summary + " " + error,
                    status     : NOTIFY_STATUS_ERROR,
                    details    : " ",
                    resolution : resolution
                };
                
                $(SETTINGS_SHOW_SECTION).trigger(ADD_NOTIFICATION, notification);
            }
            
            function handlerErrorFromPresenter(message, errorInfo) {
                addErrorNotification(message, errorInfo.message, errorInfo.resolution);
            }
            
            function reportBackupCancelError(errorInfo) {
                handlerErrorFromPresenter(
                    localizer.getString(
                        'fs.settings.backup.errorCancellingBackup'
                    ), 
                    errorInfo
                );
            }
            
            function addNotification(notification) {
                $(SETTINGS_SHOW_SECTION).trigger(ADD_NOTIFICATION, notification);
            }
            
            function updateBackupStatusView() {
                statusView.update();
            }
            
            function onBackupCancelError(errorInfo) {
                if (!cancelErrorReported) {
                    reportBackupCancelError(errorInfo);
                    cancelErrorReported = true;
                }
            }
        
            function onBackupStatusChanged(taskResource) {
                var notification = taskFormatter.getNotification(taskResource);
                addNotification(notification);
                updateBackupStatusView();
            }
            
            function cancelBackup(backupUri) {
                cancelErrorReported = false;
                presenter.cancelBackup(
                    backupUri,
                    {success: onBackupStatusChanged,
                     error: onBackupCancelError
                    }
                );
            }
            
            /**
             * Initialization.
             */
            this.init = function() {
               
                formStateView = new FormStateView();
                formStateView.init({form: BACKUP_STOP_FORM});
               
                params = urlFragment.getParameters(router.location()); 
                backupUri = params.uri;
            
                $(CANCEL_BUTTON).on('click', function(){
                   router.go('/settings/'); 
                });

                $(STOP_BUTTON).on('click', function(){
                    cancelBackup(backupUri);
                    router.go('/settings/'); 
                 });
                 
            };
            
            this.pause = function() {
                // Nothing to do
            };

            this.resume = function() {
            
                formStateView.reset();
                
                params = urlFragment.getParameters(router.location());
                backupUri = params.uri;
                         
            };
        }

        return new BackupStopView();
    }());

    return BackupStopView;
});
