// (C) Copyright 2020 Hewlett-Packard Enterprise Company, L.P.

define(['hp/view/TaskNotificationFormatter',
        'hp/model/TaskFollower',
        'hp/core/LinkTargetBuilder',
        'hp/core/Environment',
        'hp/core/Localizer',
        'hp/model/Session'],
function(TaskNotificationFormatter, taskFollower, linkTargetBuilder, environment, localizer, session) {"use strict";

    var BackupTaskNotificationFormatter = ( function() {

        var STOP_LINK = '#/settings/backup/stop?uri=';
        var TASK_STATE_COMPLETED = 'Completed';
        var TASK_STATE_TERMINATED = 'Terminated';
        var TASK_STATE_STOPPING = 'Stopping';
        var BACKUP_FILE_SUFFIX = '.bkp';
        var AUTH_CATEGORY = 'backups';

        /**
         * @constructor
         * @type {NotificationsPresenter}
         */
        function BackupTaskNotificationFormatter() {

            /**
             * Inherit from TaskNotificationFormatter
             */
            TaskNotificationFormatter.call(this); 
            
            /**
             * Save a reference to getDetails in base class
             */
            var getDetailsBase = this.getDetails;
            
            function getBackupId(backupUri) {
                var elements = backupUri.split('/');
                if (elements.length > 0) {
                    return elements[elements.length - 1];
                } else {
                    return "";
                }
            }
            
            function getBackupFileName(backupUri) {
                return getBackupId(backupUri) + BACKUP_FILE_SUFFIX;
            }
            
            /**
             * Override getDetails
             */
            this.getDetails = function(taskResource) {
                var details = getDetailsBase(taskResource);
                if (taskFollower.isTaskFinished(taskResource)) {
                    if (taskResource.taskState === TASK_STATE_COMPLETED) {
                        details = localizer.getString(
                            'fs.settings.backup.completedDetails', 
                            [getBackupFileName(
                                taskResource.associatedResourceUri),
                            taskResource.owner]);
                    } else if (taskResource.taskState === TASK_STATE_TERMINATED) {
                        details = localizer.getString(
                            'fs.settings.backup.canceledDetails', 
                            [getBackupFileName(
                                taskResource.associatedResourceUri)]);
                    } 
                } else {
                    details = localizer.getString(
                        'fs.settings.backup.backupCreateDetails', 
                        [taskResource.owner]);
                }
                return details;                
            };
            
            /**
             * Override getActions
             */
            this.getActions = function(taskResource) {
                var actions = [];
                if (taskFollower.isTaskFinished(taskResource)) {
                    if (taskResource.taskState === TASK_STATE_COMPLETED) {
                        if ((session.canPerformAction(AUTH_CATEGORY, 'Read')) &&
                            (!environment.isKioskMode())) {  
                            var backupIdUri = taskResource.associatedResourceUri;
                            var backupId = getBackupId(backupIdUri);
                            var downloadLinkName = localizer.getString('fs.settings.backup.backupDownloadLinkLabel');
                            var downloadLink = '<a class="' + backupId + '" ' + '>' + downloadLinkName + '</a>';
                            actions = [downloadLink];
                        }
                    } 
                } else /* task not finished */ {
                    if (taskResource.taskState !== TASK_STATE_STOPPING) {
                        if (session.canPerformAction(AUTH_CATEGORY, 'Delete')) {
                            var backupUri = taskResource.associatedResourceUri;
                            var stopLinkName = localizer.getString('fs.settings.backup.backupStopLinkLabel');
                            var stopLink = '<a href="'+ STOP_LINK + backupUri + '" >' + stopLinkName + '</a>';
                            actions = [stopLink];
                        }
                    }
                }
                return actions;
            };
            
            /**
             * Override getSourceName
             */
            this.getSourceName = function(taskResource) {
                return localizer.getString(
                            'fs.settings.backup.backupSourceName');
            };

            this.getBackupId = getBackupId;
            
            this.canFormatTask = function(task) {
                var match,
                    canFormat = false;
                if ((task.category === AUTH_CATEGORY) && task.associatedResourceUri) {
                    match = task.associatedResourceUri.match('^/rest/backups/.+');
                    if (match && match.length > 0) {
                        canFormat = true;
                    }
                }
                return canFormat;
            };
        }
        return BackupTaskNotificationFormatter;
    }());

    return BackupTaskNotificationFormatter;
});