// (C) Copyright 2020 Hewlett-Packard Enterprise Company, L.P.
define(['fs/presenter/settings/NetworkSetupPresenter',
        'hp/core/Notifications',
        'hp/core/Localizer',
        'fs/view/common/CommonDisplayView'],
function(presenter, notifications, localizer, commonDisplayView) {"use strict";

    var NetworkMoreView = ( function() {

        /**
         * @class Detail view script for the Network Settings panel on the Settings page.
         */
        function NetworkMoreView() {

            // Entity and class references.
            var
                CONTROLLER_NAME = '#cic-settings-net-controller-name',
                IP_ADDRESS = '#cic-settings-net-ip-address',
                IP_ADDRESS_LABEL = '#cic-settings-net-ip-address-label',
                SUBNET_MASK = '#cic-settings-net-subnet-mask',
                SUBNET_MASK_LABEL = '#cic-settings-net-subnet-mask-label',
                DEFAULT_GATEWAY = '#cic-settings-net-default-gateway',
                DEFAULT_GATEWAY_LABEL = '#cic-settings-net-default-gateway-label',
                WEB_PROXY = '#cic-settings-net-web-proxy',
                DNS_PREFERRED = '#cic-settings-net-dns-preferred',
                DNS_ALTERNATE = '#cic-settings-net-dns-alternate',
                NOTIFY_BOX = '#cic-settings-details-notify',
                NOTIFY_MSG = "#cic-settings-details-notify-message-id",
                NOTIFICATION_CLASS = "cic-details-notify",
                NOTIFICATION_COLOR_PREFIX = "cic-details-notify-",
                IPV6_SLAAC_IP_ADDRESS = '#cic-settings-net-ipv6-slaac-ip-address',
                IPV6_SLAAC_IP_ADDRESS_FIELD ='#cic-settings-net-ipv6-slaac-ip-address-field',
                HA_FIELDS = '#cic-settings-net-ha-fieldset',
                HA_NODE_1_IP_ADDR = '#cic-settings-net-node1-ip-addr',
                HA_NODE_2_IP_ADDR = '#cic-settings-net-node2-ip-addr',
                HA_IPV4_CONFIGURATION = '#cic-settings-net-ha-ipv4-config',
                HA_IPV6_CONFIGURATION = '#cic-settings-net-ha-ipv6-config',
                HA_IPV6_CONFIGURATION_FIELD = '#cic-settings-net-ha-ipv6-config-field',
                HA_IPV4_CONFIGURATION_FIELD = '#cic-settings-net-ha-ipv4-config-field';

            /**
             * Formats error information from a REST request
             * @param errorInfo {object} Rest request error information
             * @returns {String} Formatted error information to display
             */
            function formatNotificationDetails(errorInfo) {
                var failureNotificationDetails = "</br>" +
                    errorInfo.errorMessage +
                    (errorInfo.resolution === null || errorInfo.resolution == "" ? ""
                        : "</br>" + errorInfo.resolution);
                return failureNotificationDetails;
            }

            /**
             * Displays a detailed error message
             * @param {string} message Message to display
             * @param {string} messageType Type of notification such as error
             */
            function displaySettingsNotificationDetails(message,
                messageType) {
                $(NOTIFY_MSG).append(message);
                $(NOTIFY_BOX).attr(
                    'class',
                    NOTIFICATION_CLASS +  ' ' + NOTIFICATION_COLOR_PREFIX + messageType);
                $(NOTIFY_BOX).slideDown('slow');
            }

            /**
             * Displays an error notification as well as details about the error
             * @param {string} message Message containing high level error message
             * @param {string} details Message containing error details
             * @param {string} messageType indicates the type of notification such as error
             */
            function displayNotifications(message, details, type) {
                notifications.show(message);
                displaySettingsNotificationDetails(details, type);
            }

            /**
             * Displays Appliance configuration settings
             * @param {object} configuration Network object returned by the service
             */
            function displayApplianceInfo(configuration) {
                var notset = localizer.getString('core.common.notset');
                $(CONTROLLER_NAME).text(configuration.hostname);

                if (configuration.ipv6Type == presenter.UNCONFIGURE) {
                    $(IP_ADDRESS).text(configuration.addrs.node1.ipv4);
                } else {
                    $(IP_ADDRESS).html(configuration.addrs.node1.ipv4 + ', &nbsp;' + configuration.addrs.node1.ipv6);
                    $(IP_ADDRESS_LABEL).text(localizer.getString('fs.settings.network.fields.ipaddr.one-node.both'));
                }

                if (configuration.ipv6Type == presenter.UNCONFIGURE || configuration.ipv6Subnet == null) {
                    $(SUBNET_MASK).text(configuration.ipv4Subnet);
                } else {
                    $(SUBNET_MASK).html(configuration.ipv4Subnet + ', &nbsp;' + configuration.ipv6Subnet);
                    $(SUBNET_MASK_LABEL).text(localizer.getString('fs.settings.network.fields.subnet.both'));
                }

                if (configuration.ipv6Type == presenter.UNCONFIGURE || configuration.ipv6Gateway == null) {
                    $(DEFAULT_GATEWAY).text(configuration.ipv4Gateway);
                } else {
                    $(DEFAULT_GATEWAY).html(configuration.ipv4Gateway + ', &nbsp;' + configuration.ipv6Gateway);
                    $(DEFAULT_GATEWAY_LABEL).text(localizer.getString('fs.settings.network.fields.gateway.both'));
                }

                if (configuration.ipv6SlaacAddr != null) {
                    $(IPV6_SLAAC_IP_ADDRESS).text(configuration.ipv6SlaacAddr);
                    $(IPV6_SLAAC_IP_ADDRESS_FIELD).show();
                } else {
                    $(IPV6_SLAAC_IP_ADDRESS_FIELD).hide();
                }

                $(WEB_PROXY).text(notset);
                $(WEB_PROXY).addClass("cic-unset");
            }

            /**
             * Displays HA configuration settings
             * @param {object} configuration Network object returned by the service
             */
            function displayHaInfo(configuration) {
                var dhcp = localizer.getString('fs.settings.network.iptype.dhcp');
                $(HA_FIELDS).show();
                if (configuration.ipv6Type == presenter.UNCONFIGURE) {
                    $(IP_ADDRESS).text(configuration.addrs.cluster.ipv4);
                    $(IP_ADDRESS_LABEL).text(localizer.getString('fs.settings.network.fields.ipaddr.two-node.short'));
                    $(HA_NODE_1_IP_ADDR).text(configuration.addrs.node1.ipv4);
                    $(HA_NODE_2_IP_ADDR).text(configuration.addrs.node2.ipv4);
                } else {
                    $(IP_ADDRESS).html(configuration.addrs.cluster.ipv4 + ', &nbsp;' + configuration.addrs.cluster.ipv6);
                    $(IP_ADDRESS_LABEL).text(localizer.getString('fs.settings.network.fields.ipaddr.two-node.both'));
                    $(HA_NODE_1_IP_ADDR).html(configuration.addrs.node1.ipv4 + ', &nbsp;' + configuration.addrs.node1.ipv6);
                    $(HA_NODE_2_IP_ADDR).html(configuration.addrs.node2.ipv4 + ', &nbsp;' + configuration.addrs.node2.ipv6);
                }

                if (configuration.ipv4Type == presenter.DHCP) {
                    $(HA_IPV4_CONFIGURATION).text(dhcp);
                    $(HA_IPV4_CONFIGURATION_FIELD).show();
                } else {
                    $(HA_IPV4_CONFIGURATION_FIELD).hide();
                }

                if (configuration.ipv6Type == presenter.DHCP) {
                    $(HA_IPV6_CONFIGURATION).text(dhcp);
                    $(HA_IPV6_CONFIGURATION_FIELD).show();
                } else {
                    $(HA_IPV6_CONFIGURATION_FIELD).hide();
                }

            }

            /**
             * Displays name server configuration settings
             * @param {object} configuration Network object returned by the service
             */
            function displayNameServerInfo(configuration) {
                var notset = localizer.getString('core.common.notset');

                if (configuration.priDns != null) {
                    $(DNS_PREFERRED).removeClass("cic-unset");
                    $(DNS_PREFERRED).text(configuration.priDns);
                } else {
                    $(DNS_PREFERRED).addClass("cic-unset");
                    $(DNS_PREFERRED).text(notset);
                }

                if (configuration.altDns != null) {
                    $(DNS_ALTERNATE).removeClass("cic-unset");
                    $(DNS_ALTERNATE).text(configuration.altDns);
                } else {
                    $(DNS_ALTERNATE).addClass("cic-unset");
                    $(DNS_ALTERNATE).text(notset);
                }
            }

            /**
             * @public Handle successful retrieval of network configuration
             */
            function onGetNetworkConfigurationSuccess(configuration) {
                displayApplianceInfo(configuration);
                if (configuration.confOneNode) {
                    $(HA_FIELDS).hide();
                } else {
                    displayHaInfo(configuration);
                }
                displayNameServerInfo(configuration);
            }

            /**
             * @public Handle failed validation by printing error messages
             */
            function onGetNetworkConfigurationError(errorInfo) {
                // TODO consistent error handling
                // TODO add exception processing

                var failureNotificationDetails = formatNotificationDetails(errorInfo);
                displayNotifications(errorInfo.errorMessage, failureNotificationDetails, 'error');
            }

            this.init = function() {
                $(NOTIFY_BOX).slideUp('slow');
                this.resume();
            };

            this.pause = function() {

            };

            this.resume = function() {
                presenter.getNetworkConfiguration({
                    success : onGetNetworkConfigurationSuccess,
                    error : onGetNetworkConfigurationError
                });
                commonDisplayView.manageDisplayInOverviewDropdown();
            };

        }

        return new NetworkMoreView();

    }());

    return NetworkMoreView;

});