// (C) Copyright 2020 Hewlett-Packard Enterprise Company, L.P.

define(['hp/core/EventDispatcher',
        'hp/services/Log',
        'hp/core/Localizer', 
        'jquery'],

function(EventDispatcher, log, localizer) {"use strict";
    var UploadManager = (function() {

        /**
         * @class UploadManager
         * @constructor
         */
        function UploadManager(options) {
            // Helper variable to create unique ids for the upload:
            var uploadCounter = 0;
            // A map of uploadID:upload
            var uploadList = {};
            var dispatcher = new EventDispatcher();

            /**
             * Add an entry of an'upload' instance to the list managed
             * by the UploadManager.
             * 
             * @param {Upload} upload The instance to be added
             */
            this.add = function(upload) {
                var uploadID = 'upload_' + (++uploadCounter);
                upload.setId(uploadID);
                uploadList[uploadID] = upload;
                dispatcher.fire("uploadAdd", upload);
                return uploadID;
            };
            
            /**
             * Return the upload instance specified by the id supplied.
             * @param {object} id An opaque object returned by the add function.
             * @return Return the upload instance specified by id if found, or undefined if not.
             */
            this.getUpload = function(id) {
                return uploadList[id];
            };

            /**
             * Return all upload instances currently maintained.
             * @return Return a list of upload instances currently maintained by this class.
             */
            this.getUploadList = function() {
                return uploadList;
            };
            
            /**
             * Remove all inactive upload instances maintained by this class.
             * An upload instance if inactive if all the following conditions
             * are met:
             * 1- the instance in not in "uploading" state.
             * 2- the instance's stopTime is not equal to 0, which means it is
             *    not pending.
             */
            this.clean = function() {
                $.each(uploadList, function(index, upload) {
                    // if loading is not in progress, and it has a non zero
                    // stop time, then this is an old upload that can be
                    // cleaned.
                    if (!upload.isUploading() && (upload.getStopTime() !== 0)) {
                        delete uploadList[index];
                    }
                });
            };
            
            /**
             * Add a listener for a specified event.
             * Supported event:
             *   'uploadAdd'
             * @param {string} eventName The name of the event.
             * @param {function(...)}
             * 
             */
            this.on = function (eventName, callback) {
                dispatcher.on(eventName, callback);
            };

            this.off = function (eventName, callback) {
                dispatcher.off(eventName, callback);
            };

        }
        
        return new UploadManager();
    }());
    
    return UploadManager;
});

