// (C) Copyright 2020 Hewlett-Packard Enterprise Company, L.P.
define(
        [ 'hp/services/REST', 'hp/services/Log', 'jquery' ],
        function(REST, log) {
            "use strict";

            var AuthnService = (function() {

                function AuthnService() {

                    /**
                     * @public Login to the appliance
                     * @param {string}
                     *            username The username
                     * @param {string}
                     *            password The password for that user
                     * @param {string}
                     *            authnProvider Authentication Provider ID
                     * @param {Object}
                     *            handlers Must contain a "success" function and
                     *            an "error" function. e.g.
                     */
                    this.login = function(username, password, authnProvider,
                            handlers) {
                        var loginInfoDto = {
                            userName : username,
                            password : password,
                            authLoginDomain : authnProvider
                        };
                        REST.postObject(
                                  '/rest/login-sessions',loginInfoDto, handlers);
                    };

                    /**
                     * @public Logout of the appliance
                     * @param {string}
                     *            authToken The authtoken from login. parameter
                     *            is ignored as token is available in session.
                     * @param {Object}
                     *            handlers Must contain a "success" function and
                     *            an "error" function. e.g.
                     */
                    this.logout = function(authToken, handlers) {
                        REST.deleteURI('/rest/login-sessions', handlers);
                    };

                    /**
                     * @public Get value for allow local login and default
                     *         provider
                     * @param {Object}
                     *            handlers Must contain a "success" function and
                     *            an "error" function. e.g.
                     */
                    this.getGlobalSettings = function(handlers) {
                        REST.getURI(
                                '/rest/logindomains/global-settings',
                                handlers);
                    };

                    /**
                     * @public Set values for global settings
                     * @param {boolean}
                     *            allowLocalLogin indicates the boolean flag to
                     *            allow or not allow Local login
                     * @param {string}
                     *            defaultLoginDomain indicates the default
                     *            provider to be set for the appliance
                     * @param {Object}
                     *            handlers Must contain a "success" function and
                     *            an "error" function. e.g.
                     */
                    this.setGlobalSettings = function(allowLocalLogin,
                            defaultLoginDomain, handlers) {
                        REST.postObject(
                                '/rest/logindomains/global-settings', {
                                    "allowLocalLogin" : allowLocalLogin,
                                    "defaultLoginDomain" : defaultLoginDomain
                                }, handlers, null);
                    };

                    /**
                     * @public Get full details on a provider
                     * @param {string}
                     *            URI is use to get full details on the provider
                     * @param {Object}
                     *            handlers Must contain a "success" function and
                     *            an "error" function. e.g.
                     */
                    this.getResource = function(URI, handlers) {
                        REST.getURI(URI, handlers);
                    };

                    /**
                     * @public
                     * Delete a provider
                     * @param {string} URI is use to delete a provider
                     * @param {Object} handlers Must contain a "success"
                     *                 function and an "error" function. e.g.
                     */
                    this.deleteProviderConfig = function(URI, handlers) {
                        REST.deleteURI(URI, handlers);
                    };

                    /**
                     * @public addGroupToRoleMappings
                     * @param {string}
                     *            loginDomain indicates the selected
                     *            login domain
                     * @param {list}
                     *            roles - list of roles selected
                     * @param {Object}
                     *            handlers Must contain a "success" function and
                     *            an "error" function. e.g.
                     */
                    this.addGroupToRoleMappings = function(loginDomain,
                            group, roles, userName, password, handlers) {
                        var group2RolesMappingPerGroupValidationDto = {
                                "type":"Group2RolesMappingPerGroupValidationDto",
                                "group2rolesPerGroup":{
                                    "type":"Group2RolesMappingPerGroupDto",
                                    "loginDomain":loginDomain,
                                    "egroup" : group,
                                    "roles" : roles
                            },
                                "credentials" : {
                                    "userName":userName,
                                    "password":password
                            }
                        };

                        REST.postObject(
                                '/rest/logindomains/grouptorolemapping', group2RolesMappingPerGroupValidationDto, handlers);
                    };

                    /**
                     * @public Create a new provider
                     * @param {Object}
                     *            LoginDomainConfigVersion2Dto object with details to be
                     *            created
                     * @param {Object}
                     *            handlers Must contain a "success" function and
                     *            an "error" function. e.g.
                     */
                    this.addDirectory = function(config, handlers) {
                        var LoginDomainConfigVersion2Dto = {
                            "type" : "LoginDomainConfigVersion2Dto",
                            "name" : config.name,
                            "credential" : {
                                "userName" : config.userName,
                                "password" : config.password
                            },
                            "authProtocol" : config.authProtocol,
                            "org" : config.org,
                            "top" : config.top,
                            "userNameField" : config.userNameField,
                            "useSsl" : false,
                            "directoryServers" : config.directoryServers
                        };

                        REST.postObject('/rest/logindomains',
                            LoginDomainConfigVersion2Dto,
                            handlers);
                    };

                    /**
                     * @public Validate a directory server before adding
                     * @param {Object}
                     *            LoginDomainConfigVersion2Dto object with details to be
                     *            validated
                     * @param {Object}
                     *            handlers Must contain a "success" function and
                     *            an "error" function. e.g.
                     */
                    this.validateDirectoryServer = function(config, handlers) {
                        var LoginDomainConfigVersion2Dto = {
                            "type" : "LoginDomainConfigVersion2Dto",
                            "name" : config.name,
                            "loginDomain":config.loginDomain,
                            "credential" : {
                                "userName" : config.userName,
                                "password" : config.password
                            },
                            "authProtocol" : config.authProtocol,
                            "org" : config.org,
                            "top" : config.top,
                            "userNameField" : config.userNameField,
                            "useSsl" : false,
                            "directoryServers" : config.directoryServers
                        };

                        REST.postObject('/rest/logindomains/validator',
                            LoginDomainConfigVersion2Dto,
                            handlers);
                    };
                    /**
                     * @public Update the directory
                     *@param {string} URI is use to update the directory
                     * @param {Object}
                     *            LoginDomainConfigVersion2Dto object with details to be
                     *            created
                     * @param {Object}
                     *            handlers Must contain a "success" function and
                     *            an "error" function. e.g.
                     */
                    this.updateDirectory = function(uri ,config, handlers) {
                        var LoginDomainConfigVersion2Dto = {
                            "type" : "LoginDomainConfigVersion2Dto",
                            "name" : config.name,
                            "credential" : {
                                "userName" : config.userName,
                                "password" : config.password
                            },
                            "authProtocol" : config.authProtocol,
                            "org" : config.org,
                            "top" : config.top,
                            "userNameField" : config.userNameField,
                            "useSsl" : false,
                            "directoryServers" : config.directoryServers
                        };
                       
                        REST.putObject(uri,
                            LoginDomainConfigVersion2Dto,null,
                            handlers);
                    };

                }
                return new AuthnService();
            }());

            return AuthnService;
        });
