// (C) Copyright 2020 Hewlett-Packard Enterprise Company, L.P.
define(['hp/core/Style', 'jquery', 'hp/lib/jquery.hpEllipsis'],
function (style) {
    "use strict";
   
    // Manages the layout of a dialog's header.
    //
    // Use directly only when not using DialogView.
    // Assumption - In a dialog's header, the 'Action' element is always present.

    var DialogHeader = (function () {

        var NAME = '.hp-name'; // contains variable text
        var ACTION = '.hp-action'; // contains static text
        var HEADER_CONTROL = '.hp-details-header-controls';
        var PANEL_SELECTOR = '.hp-panel-selector';
        var PANEL_SELECTOR_VALUE = '.hp-panel-selector > .hp-value';

        // ### new DialogHeader()
        // Create a new DialogHeader.
        function DialogHeader() {

            var element = '';
            var layoutTimer = null;

            // Given 2 elements, set its max-width
            function setElementMaxWidth(elemA, elemB, elemAWidth, elemBWidth, maxAvailableWidth) {
                var halfMaxAvailableWidth = maxAvailableWidth / 2;

                if (elemAWidth >= halfMaxAvailableWidth && elemBWidth >= halfMaxAvailableWidth) {
                    elemA.css('max-width', halfMaxAvailableWidth);
                    elemB.css('max-width', halfMaxAvailableWidth);
                }
                else if (elemAWidth > halfMaxAvailableWidth) {
                    elemA.css('max-width', maxAvailableWidth - elemBWidth);
                }
                else if (elemBWidth > halfMaxAvailableWidth) {
                    elemB.css('max-width', maxAvailableWidth - elemAWidth);
                }
            }

            // Returns the outer width of a given element
            function getElementOuterWidth(elem) {
                return elem.length > 0 ? elem.outerWidth() : 0;
            }
            
            // Reset max-width of a given element
            function resetMaxWidth(elems) {
                $.each(elems, function (index, elem) {
                    if (elem.length > 0) {
                        elem.css('max-width', 'none');
                    }
                });
            }

            // Adjust header elements to current size
            function layout() {
                var headerControlWidth = $(HEADER_CONTROL, element)[0] ? $(HEADER_CONTROL, element).outerWidth() : 0;
                var maxAvailableWidth = Math.max((element.outerWidth() - headerControlWidth) - 100, 0); // leave a little space
                var halfMaxAvailableWidth = maxAvailableWidth / 2;
                
                // reset the max-width to its original value
                resetMaxWidth([$(ACTION, element), $(NAME, element), $(PANEL_SELECTOR_VALUE, element)]);
                
                var actionWidth = getElementOuterWidth($(ACTION, element));
                var nameWidth = getElementOuterWidth($(NAME, element));
                var panelSelectorValueWidth = getElementOuterWidth($(PANEL_SELECTOR_VALUE, element));
                var hasName = nameWidth > 0;
                var hasPanelSelector = panelSelectorValueWidth > 0;
                
                //set min-width for header
                element.css('min-width', 400);

                if (!hasName && !hasPanelSelector) {
                    if (actionWidth > maxAvailableWidth) {
                        $(ACTION, element).css('max-width', maxAvailableWidth);
                    }
                }
                else if (!hasName && hasPanelSelector) {
                    setElementMaxWidth($(ACTION, element), 
                            $(PANEL_SELECTOR_VALUE, element), actionWidth, 
                            panelSelectorValueWidth, maxAvailableWidth);
                }
                else if (hasName && !hasPanelSelector) {
                    setElementMaxWidth($(ACTION, element), $(NAME, element), 
                            actionWidth, nameWidth, maxAvailableWidth);
                }
                else if (hasName && hasPanelSelector) {
                    var otherWidth = actionWidth + panelSelectorValueWidth;
                    var newMaxAvailableWidth = 0;

                    if (nameWidth >= halfMaxAvailableWidth) {
                        var nameMaxWidth = 0;
                        if (otherWidth >= halfMaxAvailableWidth) {
                            nameMaxWidth = halfMaxAvailableWidth;
                            newMaxAvailableWidth = halfMaxAvailableWidth;
                        }
                        else {
                            nameMaxWidth = maxAvailableWidth - otherWidth;
                            newMaxAvailableWidth = otherWidth;
                        }
                        $(NAME, element).css('max-width', nameMaxWidth);
                    }
                    else {
                        newMaxAvailableWidth = maxAvailableWidth - nameWidth;
                    }
                    setElementMaxWidth($(ACTION, element), 
                            $(PANEL_SELECTOR_VALUE, element), actionWidth, 
                            panelSelectorValueWidth, newMaxAvailableWidth);
                }
            }

            // Calls layout after a set time
            function onResize(event) {
                if (event.target === window) {
                    clearTimeout(layoutTimer);
                    layoutTimer = setTimeout(layout, style.animationDelay());
                }
            }

            // Called on various events to the Panel Selector
            function onPanelSelectorEvent() {
                $(PANEL_SELECTOR_VALUE, element).css('max-width', 'none');
                setTimeout(layout, style.animationDelay());
            }

            // ### init()
            // Call after `new` to initialize.
            //
            // Takes a single parameter which is a jquery object or selector
            // corresponding to the header element in the dialog.
            this.init = function (elementArg) {
                element = $(elementArg);

                $(ACTION, element).hpEllipsis();
                $(NAME, element).hpEllipsis();

                // ensure there is an element before calling resize
                if (element.length > 0) {
                    $(window).on('resize', onResize);
                    $(PANEL_SELECTOR, element).on('change', onPanelSelectorEvent);
                    $(PANEL_SELECTOR, element).on('click', onPanelSelectorEvent);
                    $(PANEL_SELECTOR, element).on('blur', onPanelSelectorEvent);
                    // we delay so any CSS animation can finish
                    setTimeout(layout, style.animationDelay());
                }
            };

            // ### reset()
            // Call when resetting view, or
            // call after loading data lazily
            this.reset = function() {
                setTimeout(layout, style.animationDelay());
            };

            // ### pause()
            // Call when pausing view
            this.pause = function () {
                $(window).off('resize', onResize);
                $(PANEL_SELECTOR, element).off('change', onPanelSelectorEvent);
                $(PANEL_SELECTOR, element).off('click', onPanelSelectorEvent);
                $(PANEL_SELECTOR, element).off('blur', onPanelSelectorEvent);
            };

            // ### resume()
            // Call when resuming view
            this.resume = function () {
                setTimeout(layout, style.animationDelay());
                $(window).on('resize', onResize);
                $(PANEL_SELECTOR, element).on('change', onPanelSelectorEvent);
                $(PANEL_SELECTOR, element).on('click', onPanelSelectorEvent);
                $(PANEL_SELECTOR, element).on('blur', onPanelSelectorEvent);
            };
        }

        return DialogHeader;
    }());

    return DialogHeader;
});
