// (C) Copyright 2020 Hewlett-Packard Enterprise Company, L.P.
define(['hp/view/NotificationDetailsView',
    'hp/core/Localizer',
    'text!hpPages/core/dialog_notification.html',
    'jquery', 'hp/lib/jquery.hpStatus'],
function (notificationDetailsView, localizer, dialogNotificationHtml) {
    "use strict";
   
    // Manages notification messages at the top of dialog contents.
    //
    // Use directly only when not using DialogView.

    var DialogNotification = (function () {

        var VARIABLE_CONTENTS = '.hp-dialog-variable-contents';
        var STACKED_PANELS = '.hp-stacked-panels';
        var FORM_CONTENTS = ".hp-form-contents";
        var NOTIFICATION = '.hp-dialog-notification';
        var STATUS = NOTIFICATION + ' > .hp-status, ' +
            NOTIFICATION + ' .hp-full header > .hp-status';
        var MESSAGE = NOTIFICATION + ' .hp-message';
        var FULL = NOTIFICATION + ' .hp-full';
        var DETAILS = NOTIFICATION + ' .hp-details';

        // ### new DialogNotification()
        // Create a new DialogNotification.
        function DialogNotification() {
          
            var dialog;
            
            // Adjust header elements to current size
            function onResize() {
                if ($(NOTIFICATION, dialog).length > 0) {
                    $(NOTIFICATION, dialog).css('max-height',
                        dialog.parents('.hp-dialog-container-wrapper').height() - 100);
                }
            }
            
            // ### init()
            // Call after `new` to initialize.
            //
            // Takes a single parameter which is a jquery object or selector
            // corresponding to the hp-dialog element.
            this.init = function (dialogArg) {
                dialog = $(dialogArg);
            };
            
            // ### pause()
            // Call when pausing view
            this.pause = function () {
            };

            // ### resume()
            // Call when resuming view
            this.resume = function () {
                onResize();
            };

            // ### resume()
            // Call to re-layout
            this.resize = function () {
                onResize();
            };
            
            // ### setMessage(notif)
            //
            // Sets one or more notification messages using a single object argument with
            // the following properties:
            //
            // - **status** one of 'ok', 'error', 'warning', 'info'
            // - **summary** minimal text summary, should be one line
            // - **details** longer description that can have line breaks and links
            //     Will use `hpTextFormat` to format Markdown text.
            // - **resolution** multi-line recommendation on how the user should proceed
            //     Will use `hpTextFormat` to format Markdown text.
            // - **actions** Array of action links
            // - **children** Array of sub-notification objects
            //
            // The only property that is required is the summary.
            this.setMessage = function(notif, expand) {
                var notificationHtml;
                if ($(NOTIFICATION, dialog).length === 0) {
                    notificationHtml = $(dialogNotificationHtml);
                    localizer.localizeDom(notificationHtml);
                    if ($(STACKED_PANELS, dialog).length > 0) {
                        var li = $('<li></li>').
                            addClass('hp-stacked-panel hp-stacked-panel-notification').
                            append(notificationHtml);
                        $(STACKED_PANELS, dialog).prepend(li);
                    } else {
                        // add variable contents if not there
                        if ($(VARIABLE_CONTENTS, dialog).length === 0) {
                            if ($(FORM_CONTENTS, dialog).length > 0) {
                                $(FORM_CONTENTS , dialog).prepend($('<div></div>').
                                        addClass('hp-dialog-variable-contents'));
                            } else {
                                $('header', dialog).after($('<div></div>').
                                        addClass('hp-dialog-variable-contents'));
                            }
                        }
                        // prepend variable contents
                        $(VARIABLE_CONTENTS, dialog).prepend(notificationHtml);
                    }
                }
                if (notif.status) {
                    $(STATUS, dialog).hpStatus(notif.status, notif.changing);
                } else {
                    $(STATUS, dialog).attr('class', 'hp-status').text('');
                }
                // Don't replace if it hasn't changed, allows user selection to persist.
                if ($(MESSAGE, dialog).text() !== notif.summary) {
                    $(MESSAGE, dialog).text(notif.summary);
                }
                if (!notificationDetailsView.update($(DETAILS, dialog), notif, 
                    true /* hide empty details */)) {
                    // If there aren't any details, hide the margin of the full
                    // element which contains the details
                    $(FULL, dialog).hide();
                } else {
                    $(FULL, dialog).show();
                }
            };
            
            // ### clearMessage()
            // Clear any notification messages.
            this.clearMessage = function() {
                notificationDetailsView.reset($(DETAILS, dialog));
                $(NOTIFICATION, dialog).remove();
            };
        }

        return DialogNotification;
    }());

    return DialogNotification;
});
