// (C) Copyright 2020 Hewlett-Packard Enterprise Company, L.P.
/**
 * @type {MasterPaneView}
 */
define(['hp/presenter/IndexPresenter',
    'hp/view/IndexTableView',
    'hp/core/LinkTargetBuilder',
    'hp/core/Localizer',
    'text!hpPages/core/no_details.html',
    'text!hpPages/core/no_items.html',
    'jquery',
    'hp/lib/jquery.hpSelect',
    'hp/lib/jquery.hpEllipsis',
    'hp/lib/jquery.hpCollapsible'],
function (IndexPresenter, IndexTableView,
    linkTargetBuilder, localizer,
    noDetailsHtml, noItemsHtml) {
"use strict";

    var IndexView = (function () {

        var RESET_FILTERS = '.hp-filter-reset';
        var FILTER = '.hp-filter[data-filter-property]';
        var META_FILTER = '.hp-meta-filter';

        /**
         * @constructor
         */
        function IndexView() {

            //
            // private class instance variables
            //
            var presenter = new IndexPresenter();
            var container;
            var table;
            var active = false;
            var activeView;
            var noDetails = $(noDetailsHtml);
            var noItems = $(noItemsHtml);
            
            // Presenter Events
            
            function onFilterChange(filter) {
                var metaOption = null;
                if ($(META_FILTER, container).length > 0) {
                    var metaOption = $(META_FILTER + ' li[data-query="' +
                        presenter.getSearchText() + '"]', container);
                    if (metaOption.length > 0) {
                        $(META_FILTER, container).hpSelect('set',
                            metaOption.attr('data-id'));
                    } else {
                        $(META_FILTER, container).hpSelect('set',
                            localizer.getString('fs.activity.filter.meta.custom'));
                    }
                }
                // disable reset if we are on the defalt meta filter
                $(RESET_FILTERS, container).toggleClass('hp-disabled',
                    (metaOption !== null && metaOption.hasClass('hp-default')));
                
                $(FILTER, container).each(function (index, elem) {
                    if (! $(elem).hasClass('.hp-meta-filter')) {
                        var name = $(elem).attr('data-filter-property');
                        var value = presenter.getFilterValue(name);
                        $(elem).hpSelect('set', value);
                    }
                });
            }

            // DOM Events
            
            function onResetFilters() {
                var names = [];
                $(FILTER, container).each(function (index, elem) {
                    var name = $(elem).attr('data-filter-property');
                    names.push(name);
                });
                presenter.resetFilters(names);
            }

            function onPropertyFilterChange(ev, value) {
                var name = $(this).attr('data-filter-property');
                if (name) {
                    presenter.setFilterValue(name, value);
                }
            }
            
            function onMetaFilterChange(ev, value) {
                var query = $(META_FILTER + ' li[data-id="' + value + '"]').
                    attr('data-query');
                presenter.setMetaFilter(query);
            }
            
            function registerListeners() {
                presenter.on("filterChange", onFilterChange);
            }
            
            function unregisterListeners() {
                presenter.off("filterChange", onFilterChange);
            }

            /**
             * @public
             * Stop the timer polling of the index service.
             */
            this.pause = function () {
                if (active) {
                    active = false;
                    presenter.pause();
                    unregisterListeners();
                    if (activeView) {
                        activeView.pause();
                    }
                }
            };

            /**
             * @public
             * Resume the timer polling of the index service.
             */
            this.resume = function () {
                if (! active) {
                    active = true;
                    registerListeners();
                    if (activeView) {
                        activeView.resume();
                    }
                    // Resume the presenter after everyone is listening again.
                    // This ensures they hear about the latest state.
                    presenter.resume();
                }
            };

            /**
             * @public
             * Intialize the view.
             */
            this.init = function (args) {
                active = true;
                container = args.container;

                localizer.localizeDom(noDetails);
                localizer.localizeDom(noItems);
                
                var defaultMetaQuery =
                    $('.hp-filters .hp-meta-filter li.hp-default', container).
                        attr('data-query');
                if (defaultMetaQuery) {
                    args = $.extend({defaultMetaQuery: defaultMetaQuery}, args);
                }

                presenter.init(args);

                $('.hp-filters .hp-select.hp-filter', container).hpSelect();
                
                $(RESET_FILTERS, container).on('click', onResetFilters);

                $(FILTER, container).change(onPropertyFilterChange);
                $(META_FILTER, container).change(onMetaFilterChange);

                if (args.dataTableOptions) {
                    table = new IndexTableView();
                    table.init(presenter, args);
                    activeView = table;
                }
                
                registerListeners();
            };
            
            this.scrollToUriAndExpand = function (uri) {
                table.scrollToUriAndExpand(uri);
            };
        }

        return IndexView;

    }());

    return IndexView;
});
