/*
 * Decompiled with CFR 0.152.
 */
package net.sf.ehcache.store;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.ObjectStreamClass;
import java.io.RandomAccessFile;
import java.io.Serializable;
import java.io.StreamCorruptedException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.ConcurrentModificationException;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;
import java.util.logging.Logger;
import net.sf.ehcache.CacheException;
import net.sf.ehcache.Ehcache;
import net.sf.ehcache.Element;
import net.sf.ehcache.Status;
import net.sf.ehcache.config.CacheConfiguration;
import net.sf.ehcache.event.RegisteredEventListeners;
import net.sf.ehcache.store.AbstractPolicy;
import net.sf.ehcache.store.LfuPolicy;
import net.sf.ehcache.store.Policy;
import net.sf.ehcache.store.Store;
import net.sf.ehcache.util.MemoryEfficientByteArrayOutputStream;

public class DiskStore
implements Store {
    public static final String AUTO_DISK_PATH_DIRECTORY_PREFIX = "ehcache_auto_created";
    private static final Logger LOG = Logger.getLogger(DiskStore.class.getName());
    private static final int MS_PER_SECOND = 1000;
    private static final int SPOOL_THREAD_INTERVAL = 200;
    private static final int ESTIMATED_MINIMUM_PAYLOAD_SIZE = 512;
    private static final int ONE_MEGABYTE = 0x100000;
    private static final int QUARTER_OF_A_SECOND = 250;
    private long expiryThreadInterval;
    private final String name;
    private boolean active;
    private RandomAccessFile randomAccessFile;
    private ConcurrentHashMap diskElements = new ConcurrentHashMap();
    private List freeSpace = Collections.synchronizedList(new ArrayList());
    private volatile ConcurrentHashMap spool = new ConcurrentHashMap();
    private Thread spoolAndExpiryThread;
    private Ehcache cache;
    private final boolean persistent;
    private final String diskPath;
    private File dataFile;
    private File indexFile;
    private Status status = Status.STATUS_UNINITIALISED;
    private long totalSize;
    private final long maxElementsOnDisk;
    private boolean eternal;
    private int lastElementSize;
    private int diskSpoolBufferSizeBytes;
    private final AtomicBoolean writeIndexFlag;
    private final Object writeIndexFlagLock;
    private volatile boolean spoolAndExpiryThreadActive;

    public DiskStore(Ehcache cache, String diskPath) {
        this.cache = cache;
        this.name = cache.getName();
        this.diskPath = diskPath;
        CacheConfiguration config = cache.getCacheConfiguration();
        this.expiryThreadInterval = config.getDiskExpiryThreadIntervalSeconds();
        this.persistent = config.isDiskPersistent();
        this.maxElementsOnDisk = config.getMaxElementsOnDisk();
        this.eternal = config.isEternal();
        this.diskSpoolBufferSizeBytes = cache.getCacheConfiguration().getDiskSpoolBufferSizeMB() * 0x100000;
        this.writeIndexFlag = new AtomicBoolean(false);
        this.writeIndexFlagLock = new Object();
        try {
            this.initialiseFiles();
            this.active = true;
            this.spoolAndExpiryThread = new SpoolAndExpiryThread();
            this.spoolAndExpiryThread.start();
            this.status = Status.STATUS_ALIVE;
        }
        catch (Exception e) {
            this.dispose();
            throw new CacheException(this.name + "Cache: Could not create disk store. " + "Initial cause was " + e.getMessage(), e);
        }
    }

    private void initialiseFiles() throws Exception {
        File diskDir = new File(this.diskPath);
        if (diskDir.exists() && !diskDir.isDirectory()) {
            throw new Exception("Store directory \"" + diskDir.getCanonicalPath() + "\" exists and is not a directory.");
        }
        if (!diskDir.exists() && !diskDir.mkdirs()) {
            throw new Exception("Could not create cache directory \"" + diskDir.getCanonicalPath() + "\".");
        }
        this.dataFile = new File(diskDir, this.getDataFileName());
        this.indexFile = new File(diskDir, this.getIndexFileName());
        this.deleteIndexIfNoData();
        if (this.persistent) {
            if (this.diskPath.indexOf(AUTO_DISK_PATH_DIRECTORY_PREFIX) != -1) {
                LOG.log(Level.WARNING, "Data in persistent disk stores is ignored for stores from automatically created directories (they start with ehcache_auto_created).\nRemove diskPersistent or resolve the conflicting disk paths in cache configuration.\nDeleting data file " + this.getDataFileName());
                this.dataFile.delete();
            } else if (!this.readIndex()) {
                if (LOG.isLoggable(Level.FINE)) {
                    LOG.log(Level.FINE, "Index file dirty or empty. Deleting data file " + this.getDataFileName());
                }
                this.dataFile.delete();
            }
        } else {
            if (LOG.isLoggable(Level.FINE)) {
                LOG.log(Level.FINE, "Deleting data file " + this.getDataFileName());
            }
            this.dataFile.delete();
            this.indexFile = null;
        }
        this.randomAccessFile = new RandomAccessFile(this.dataFile, "rw");
    }

    private void deleteIndexIfNoData() {
        boolean dataFileExists = this.dataFile.exists();
        boolean indexFileExists = this.indexFile.exists();
        if (!dataFileExists && indexFileExists) {
            if (LOG.isLoggable(Level.FINE)) {
                LOG.log(Level.FINE, "Matching data file missing for index file. Deleting index file " + this.getIndexFileName());
            }
            this.indexFile.delete();
        }
    }

    private void checkActive() throws CacheException {
        if (!this.active) {
            throw new CacheException(this.name + " Cache: The Disk store is not active.");
        }
    }

    public final synchronized Element get(Object key) {
        try {
            this.checkActive();
            Element element = (Element)this.spool.remove(key);
            if (element != null) {
                element.updateAccessStatistics();
                return element;
            }
            DiskElement diskElement = (DiskElement)this.diskElements.get(key);
            if (diskElement == null) {
                return null;
            }
            element = this.loadElementFromDiskElement(diskElement);
            if (element != null) {
                element.updateAccessStatistics();
            }
            return element;
        }
        catch (Exception exception) {
            LOG.log(Level.SEVERE, this.name + "Cache: Could not read disk store element for key " + key + ". Error was " + exception.getMessage(), exception);
            return null;
        }
    }

    public final boolean containsKey(Object key) {
        return this.diskElements.containsKey(key) || this.spool.containsKey(key);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Element loadElementFromDiskElement(DiskElement diskElement) throws IOException, ClassNotFoundException {
        byte[] buffer;
        RandomAccessFile randomAccessFile = this.randomAccessFile;
        synchronized (randomAccessFile) {
            this.randomAccessFile.seek(diskElement.position);
            buffer = new byte[diskElement.payloadSize];
            this.randomAccessFile.readFully(buffer);
        }
        ByteArrayInputStream instr = new ByteArrayInputStream(buffer);
        ObjectInputStream objstr = new ObjectInputStream(instr){

            protected Class resolveClass(ObjectStreamClass clazz) throws ClassNotFoundException, IOException {
                try {
                    ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
                    return Class.forName(clazz.getName(), false, classLoader);
                }
                catch (ClassNotFoundException e) {
                    return super.resolveClass(clazz);
                }
            }
        };
        Element element = (Element)objstr.readObject();
        objstr.close();
        return element;
    }

    public final synchronized Element getQuiet(Object key) {
        try {
            this.checkActive();
            Element element = (Element)this.spool.remove(key);
            if (element != null) {
                return element;
            }
            DiskElement diskElement = (DiskElement)this.diskElements.get(key);
            if (diskElement == null) {
                return null;
            }
            element = this.loadElementFromDiskElement(diskElement);
            return element;
        }
        catch (Exception e) {
            LOG.log(Level.SEVERE, this.name + "Cache: Could not read disk store element for key " + key + ". Initial cause was " + e.getMessage(), e);
            return null;
        }
    }

    public final synchronized Object[] getKeyArray() {
        Set elementKeySet = this.diskElements.keySet();
        Set spoolKeySet = this.spool.keySet();
        HashSet allKeysSet = new HashSet(elementKeySet.size() + spoolKeySet.size());
        allKeysSet.addAll(elementKeySet);
        allKeysSet.addAll(spoolKeySet);
        return allKeysSet.toArray();
    }

    public final synchronized int getSize() {
        try {
            this.checkActive();
            int spoolSize = this.spool.size();
            int diskSize = this.diskElements.size();
            return spoolSize + diskSize;
        }
        catch (Exception e) {
            LOG.log(Level.SEVERE, this.name + "Cache: Could not determine size of disk store.. Initial cause was " + e.getMessage(), e);
            return 0;
        }
    }

    public long getSizeInBytes() {
        return this.getDataFileSize();
    }

    public final Status getStatus() {
        return this.status;
    }

    public final void put(Element element) {
        try {
            this.checkActive();
            if (this.spoolAndExpiryThread.isAlive()) {
                this.spool.put(element.getObjectKey(), element);
            } else {
                LOG.log(Level.SEVERE, this.name + "Cache: Elements cannot be written to disk store because the" + " spool thread has died.");
                this.spool.clear();
            }
        }
        catch (Exception e) {
            LOG.log(Level.SEVERE, this.name + "Cache: Could not write disk store element for " + element.getObjectKey() + ". Initial cause was " + e.getMessage(), e);
        }
    }

    public boolean bufferFull() {
        boolean backedUp;
        long estimatedSpoolSize = this.spool.size() * this.lastElementSize;
        boolean bl = backedUp = estimatedSpoolSize > (long)this.diskSpoolBufferSizeBytes;
        if (backedUp && LOG.isLoggable(Level.FINE)) {
            LOG.log(Level.FINE, "A back up on disk store puts occurred. Consider increasing diskSpoolBufferSizeMB for cache " + this.name);
        }
        return backedUp;
    }

    public final synchronized Element remove(Object key) {
        Element element;
        try {
            this.checkActive();
            element = (Element)this.spool.remove(key);
            DiskElement diskElement = (DiskElement)this.diskElements.remove(key);
            if (diskElement != null) {
                element = this.loadElementFromDiskElement(diskElement);
                this.freeBlock(diskElement);
            }
        }
        catch (Exception exception) {
            String message = this.name + "Cache: Could not remove disk store entry for key " + key + ". Error was " + exception.getMessage();
            LOG.log(Level.SEVERE, message, exception);
            throw new CacheException(message);
        }
        return element;
    }

    private void freeBlock(DiskElement diskElement) {
        this.totalSize -= (long)diskElement.payloadSize;
        diskElement.payloadSize = 0;
        diskElement.key = null;
        diskElement.hitcount = 0L;
        diskElement.expiryTime = 0L;
        this.freeSpace.add(diskElement);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public final synchronized void removeAll() {
        try {
            this.checkActive();
            this.spool = new ConcurrentHashMap();
            this.diskElements = new ConcurrentHashMap();
            this.freeSpace = Collections.synchronizedList(new ArrayList());
            this.totalSize = 0L;
            RandomAccessFile randomAccessFile = this.randomAccessFile;
            synchronized (randomAccessFile) {
                this.randomAccessFile.setLength(0L);
            }
            if (this.persistent) {
                this.indexFile.delete();
                this.indexFile.createNewFile();
            }
        }
        catch (Exception e) {
            LOG.log(Level.SEVERE, this.name + " Cache: Could not rebuild disk store. Initial cause was " + e.getMessage(), e);
            this.dispose();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public final void dispose() {
        if (!this.active) {
            return;
        }
        try {
            this.flush();
            this.spoolAndExpiryThreadActive = false;
            this.spoolAndExpiryThread.interrupt();
            if (this.spoolAndExpiryThread != null) {
                this.spoolAndExpiryThread.join();
            }
            this.spool.clear();
            this.diskElements.clear();
            this.freeSpace.clear();
            RandomAccessFile randomAccessFile = this.randomAccessFile;
            synchronized (randomAccessFile) {
                if (this.randomAccessFile != null) {
                    this.randomAccessFile.close();
                }
            }
            this.deleteFilesInAutoGeneratedDirectory();
            if (!this.persistent) {
                LOG.log(Level.FINE, "Deleting file " + this.dataFile.getName());
                this.dataFile.delete();
            }
        }
        catch (Exception e) {
            LOG.log(Level.SEVERE, this.name + "Cache: Could not shut down disk cache. Initial cause was " + e.getMessage(), e);
        }
        finally {
            this.active = false;
            this.randomAccessFile = null;
            this.cache = null;
        }
    }

    protected void deleteFilesInAutoGeneratedDirectory() {
        if (this.diskPath.indexOf(AUTO_DISK_PATH_DIRECTORY_PREFIX) != -1) {
            File dataDirectory;
            if (this.dataFile != null && this.dataFile.exists()) {
                LOG.log(Level.FINE, "Deleting file " + this.dataFile.getName());
                this.dataFile.delete();
            }
            if (this.indexFile != null && this.indexFile.exists()) {
                LOG.log(Level.FINE, "Deleting file " + this.indexFile.getName());
                this.indexFile.delete();
            }
            if ((dataDirectory = new File(this.diskPath)) != null && dataDirectory.exists() && dataDirectory.delete()) {
                LOG.log(Level.FINE, "Deleted directory " + dataDirectory.getName());
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public final void flush() {
        if (this.persistent) {
            Object object = this.writeIndexFlagLock;
            synchronized (object) {
                this.writeIndexFlag.set(true);
            }
        }
    }

    private void spoolAndExpiryThreadMain() {
        long nextExpiryTime = System.currentTimeMillis();
        while (this.spoolAndExpiryThreadActive || this.writeIndexFlag.get()) {
            try {
                if (!this.writeIndexFlag.get()) {
                    Thread.sleep(200L);
                }
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
            this.throwableSafeFlushSpoolIfRequired();
            if (!this.spoolAndExpiryThreadActive) {
                return;
            }
            nextExpiryTime = this.throwableSafeExpireElementsIfRequired(nextExpiryTime);
        }
    }

    private long throwableSafeExpireElementsIfRequired(long nextExpiryTime) {
        long updatedNextExpiryTime = nextExpiryTime;
        if (!this.eternal && System.currentTimeMillis() > nextExpiryTime) {
            try {
                updatedNextExpiryTime += this.expiryThreadInterval * 1000L;
                this.expireElements();
            }
            catch (Throwable e) {
                LOG.log(Level.SEVERE, this.name + " Cache: Could not expire elements from disk due to " + e.getMessage() + ". Continuing...", e);
            }
        }
        return updatedNextExpiryTime;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void throwableSafeFlushSpoolIfRequired() {
        Object object = this.writeIndexFlagLock;
        synchronized (object) {
            block9: {
                if (this.spool != null && (this.spool.size() != 0 || this.writeIndexFlag.get())) {
                    try {
                        this.flushSpool();
                        if (!this.persistent || !this.writeIndexFlag.get()) break block9;
                        try {
                            this.writeIndex();
                        }
                        finally {
                            this.writeIndexFlag.set(false);
                        }
                    }
                    catch (Throwable e) {
                        LOG.log(Level.SEVERE, this.name + " Cache: Could not flush elements to disk due to " + e.getMessage() + ". Continuing...", e);
                    }
                }
            }
        }
    }

    private synchronized void flushSpool() throws IOException {
        if (this.spool.size() == 0) {
            return;
        }
        Map copyOfSpool = this.swapSpoolReference();
        Iterator valuesIterator = copyOfSpool.values().iterator();
        while (valuesIterator.hasNext()) {
            this.writeOrReplaceEntry(valuesIterator.next());
            valuesIterator.remove();
        }
    }

    private Map swapSpoolReference() {
        ConcurrentHashMap copyOfSpool = null;
        copyOfSpool = this.spool;
        this.spool = new ConcurrentHashMap();
        return copyOfSpool;
    }

    private void writeOrReplaceEntry(Object object) throws IOException {
        Element element = (Element)object;
        if (element == null) {
            return;
        }
        Serializable key = (Serializable)element.getObjectKey();
        this.removeOldEntryIfAny(key);
        if (this.maxElementsOnDisk > 0L && (long)this.diskElements.size() >= this.maxElementsOnDisk) {
            this.evictLfuDiskElement();
        }
        this.writeElement(element, key);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void writeElement(Element element, Serializable key) throws IOException {
        try {
            MemoryEfficientByteArrayOutputStream buffer = this.serializeElement(element);
            if (buffer == null) {
                return;
            }
            int bufferLength = buffer.size();
            try {
                DiskElement diskElement = this.checkForFreeBlock(bufferLength);
                RandomAccessFile randomAccessFile = this.randomAccessFile;
                synchronized (randomAccessFile) {
                    this.randomAccessFile.seek(diskElement.position);
                    this.randomAccessFile.write(buffer.toByteArray(), 0, bufferLength);
                }
                buffer = null;
                diskElement.payloadSize = bufferLength;
                diskElement.key = key;
                diskElement.expiryTime = element.getExpirationTime();
                diskElement.hitcount = element.getHitCount();
                this.totalSize += (long)bufferLength;
                this.lastElementSize = bufferLength;
                this.diskElements.put(key, diskElement);
            }
            catch (OutOfMemoryError e) {
                LOG.log(Level.SEVERE, "OutOfMemoryError on serialize: " + key);
            }
        }
        catch (Exception e) {
            LOG.log(Level.SEVERE, this.name + "Cache: Failed to write element to disk '" + key + "'. Initial cause was " + e.getMessage(), e);
        }
    }

    private MemoryEfficientByteArrayOutputStream serializeElement(Element element) throws IOException, InterruptedException {
        ConcurrentModificationException exception = null;
        for (int retryCount = 0; retryCount < 2; ++retryCount) {
            try {
                return MemoryEfficientByteArrayOutputStream.serialize(element, this.estimatedPayloadSize());
            }
            catch (ConcurrentModificationException e) {
                exception = e;
                Thread.sleep(250L);
                continue;
            }
        }
        if (LOG.isLoggable(Level.FINE)) {
            LOG.log(Level.FINE, "Gave up trying to Serialize " + element.getObjectKey(), exception);
        }
        return null;
    }

    private int estimatedPayloadSize() {
        int size = 0;
        try {
            size = (int)(this.totalSize / (long)this.diskElements.size());
        }
        catch (Exception exception) {
            // empty catch block
        }
        if (size <= 0) {
            size = 512;
        }
        return size;
    }

    private void removeOldEntryIfAny(Serializable key) {
        DiskElement oldBlock = (DiskElement)this.diskElements.remove(key);
        if (oldBlock != null) {
            this.freeBlock(oldBlock);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private DiskElement checkForFreeBlock(int bufferLength) throws IOException {
        DiskElement diskElement = this.findFreeBlock(bufferLength);
        if (diskElement == null) {
            diskElement = new DiskElement();
            RandomAccessFile randomAccessFile = this.randomAccessFile;
            synchronized (randomAccessFile) {
                diskElement.position = this.randomAccessFile.length();
            }
            diskElement.blockSize = bufferLength;
        }
        return diskElement;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private synchronized void writeIndex() throws IOException {
        FileOutputStream fout = null;
        ObjectOutputStream objectOutputStream = null;
        try {
            fout = new FileOutputStream(this.indexFile);
            objectOutputStream = new ObjectOutputStream(fout);
            objectOutputStream.writeObject(this.diskElements);
            objectOutputStream.writeObject(this.freeSpace);
        }
        finally {
            if (objectOutputStream != null) {
                objectOutputStream.close();
            }
            if (fout != null) {
                fout.close();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private synchronized boolean readIndex() throws IOException {
        ObjectInputStream objectInputStream = null;
        FileInputStream fin = null;
        boolean success = false;
        if (this.indexFile.exists()) {
            try {
                fin = new FileInputStream(this.indexFile);
                objectInputStream = new ObjectInputStream(fin);
                Map diskElementsMap = (Map)objectInputStream.readObject();
                this.diskElements = diskElementsMap instanceof ConcurrentHashMap ? (ConcurrentHashMap)diskElementsMap : new ConcurrentHashMap(diskElementsMap);
                this.freeSpace = (List)objectInputStream.readObject();
                success = true;
                return success;
            }
            catch (StreamCorruptedException e) {
                LOG.log(Level.SEVERE, "Corrupt index file. Creating new index.");
                return success;
            }
            catch (IOException e) {
                if (!LOG.isLoggable(Level.FINE)) return success;
                LOG.log(Level.FINE, "IOException reading index. Creating new index. ");
                return success;
            }
            catch (ClassNotFoundException e) {
                LOG.log(Level.SEVERE, "Class loading problem reading index. Creating new index. Initial cause was " + e.getMessage(), e);
                return success;
            }
            finally {
                try {
                    if (objectInputStream != null) {
                        objectInputStream.close();
                    }
                    if (fin != null) {
                        fin.close();
                    }
                }
                catch (IOException e) {
                    LOG.log(Level.SEVERE, "Problem closing the index file.");
                }
                if (!success) {
                    this.createNewIndexFile();
                }
            }
        }
        this.createNewIndexFile();
        return success;
    }

    private void createNewIndexFile() throws IOException {
        if (this.indexFile.exists()) {
            if (this.indexFile.delete()) {
                if (LOG.isLoggable(Level.FINE)) {
                    LOG.log(Level.FINE, "Index file " + this.indexFile + " deleted.");
                }
            } else {
                throw new IOException("Index file " + this.indexFile + " could not deleted.");
            }
        }
        if (this.indexFile.createNewFile()) {
            if (LOG.isLoggable(Level.FINE)) {
                LOG.log(Level.FINE, "Index file " + this.indexFile + " created successfully");
            }
        } else {
            throw new IOException("Index file " + this.indexFile + " could not created.");
        }
    }

    public void expireElements() {
        long now = System.currentTimeMillis();
        Iterator iterator = this.spool.values().iterator();
        while (iterator.hasNext()) {
            Element element = (Element)iterator.next();
            if (!element.isExpired()) continue;
            if (LOG.isLoggable(Level.FINE)) {
                LOG.log(Level.FINE, this.name + "Cache: Removing expired spool element " + element.getObjectKey());
            }
            iterator.remove();
            this.notifyExpiryListeners(element);
        }
        Element element = null;
        RegisteredEventListeners listeners = this.cache.getCacheEventNotificationService();
        Iterator iterator2 = this.diskElements.entrySet().iterator();
        while (iterator2.hasNext()) {
            Map.Entry entry = iterator2.next();
            DiskElement diskElement = (DiskElement)entry.getValue();
            if (now < diskElement.expiryTime) continue;
            if (LOG.isLoggable(Level.FINE)) {
                LOG.log(Level.FINE, this.name + "Cache: Removing expired spool element " + entry.getKey() + " from Disk Store");
            }
            iterator2.remove();
            if (listeners.hasCacheEventListeners()) {
                try {
                    element = this.loadElementFromDiskElement(diskElement);
                    this.notifyExpiryListeners(element);
                }
                catch (Exception exception) {
                    LOG.log(Level.SEVERE, this.name + "Cache: Could not remove disk store entry for " + entry.getKey() + ". Error was " + exception.getMessage(), exception);
                }
            }
            this.freeBlock(diskElement);
        }
    }

    private void notifyExpiryListeners(Element element) {
        this.cache.getCacheEventNotificationService().notifyElementExpiry(element, false);
    }

    private DiskElement findFreeBlock(int length) {
        for (int i = 0; i < this.freeSpace.size(); ++i) {
            DiskElement element = (DiskElement)this.freeSpace.get(i);
            if (element.blockSize < length) continue;
            this.freeSpace.remove(i);
            return element;
        }
        return null;
    }

    public final String toString() {
        StringBuffer sb = new StringBuffer();
        sb.append("[ dataFile = ").append(this.dataFile.getAbsolutePath()).append(", active=").append(this.active).append(", totalSize=").append(this.totalSize).append(", status=").append(this.status).append(", expiryThreadInterval = ").append(this.expiryThreadInterval).append(" ]");
        return sb.toString();
    }

    public static String generateUniqueDirectory() {
        return "ehcache_auto_created_" + System.currentTimeMillis();
    }

    public final long getTotalFileSize() {
        return this.getDataFileSize() + this.getIndexFileSize();
    }

    public final long getDataFileSize() {
        return this.dataFile.length();
    }

    public final float calculateDataFileSparseness() {
        return 1.0f - (float)this.getUsedDataSize() / (float)this.getDataFileSize();
    }

    public final long getUsedDataSize() {
        return this.totalSize;
    }

    public final long getIndexFileSize() {
        if (this.indexFile == null) {
            return 0L;
        }
        return this.indexFile.length();
    }

    public final String getDataFileName() {
        String safeName = this.name.replace('/', '_');
        return safeName + ".data";
    }

    public final String getDataFilePath() {
        return this.diskPath;
    }

    public final String getIndexFileName() {
        return this.name + ".index";
    }

    public final boolean isSpoolThreadAlive() {
        if (this.spoolAndExpiryThread == null) {
            return false;
        }
        return this.spoolAndExpiryThread.isAlive();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void evictLfuDiskElement() {
        ConcurrentHashMap concurrentHashMap = this.diskElements;
        synchronized (concurrentHashMap) {
            DiskElement diskElement = this.findRelativelyUnused();
            this.diskElements.remove(diskElement.key);
            this.notifyEvictionListeners(diskElement);
            this.freeBlock(diskElement);
        }
    }

    private DiskElement findRelativelyUnused() {
        DiskElement[] elements = this.sampleElements(this.diskElements);
        return DiskStore.leastHit(elements, null);
    }

    public static DiskElement leastHit(DiskElement[] sampledElements, DiskElement justAdded) {
        if (sampledElements.length == 1 && justAdded != null) {
            return justAdded;
        }
        DiskElement lowestElement = null;
        for (DiskElement diskElement : sampledElements) {
            if (lowestElement == null) {
                if (diskElement.equals(justAdded)) continue;
                lowestElement = diskElement;
                continue;
            }
            if (diskElement.getHitCount() >= lowestElement.getHitCount() || diskElement.equals(justAdded)) continue;
            lowestElement = diskElement;
        }
        return lowestElement;
    }

    private DiskElement[] sampleElements(Map map) {
        int[] offsets = AbstractPolicy.generateRandomSample(map.size());
        DiskElement[] elements = new DiskElement[offsets.length];
        Iterator iterator = map.values().iterator();
        for (int i = 0; i < offsets.length; ++i) {
            for (int j = 0; j < offsets[i]; ++j) {
                iterator.next();
            }
            elements[i] = (DiskElement)iterator.next();
        }
        return elements;
    }

    private void notifyEvictionListeners(DiskElement diskElement) {
        RegisteredEventListeners listeners = this.cache.getCacheEventNotificationService();
        if (listeners.hasCacheEventListeners()) {
            Element element = null;
            try {
                element = this.loadElementFromDiskElement(diskElement);
                this.cache.getCacheEventNotificationService().notifyElementEvicted(element, false);
            }
            catch (Exception exception) {
                LOG.log(Level.SEVERE, this.name + "Cache: Could not notify disk store eviction of " + element.getObjectKey() + ". Error was " + exception.getMessage(), exception);
            }
        }
    }

    public Policy getEvictionPolicy() {
        return new LfuPolicy();
    }

    public void setEvictionPolicy(Policy policy) {
        throw new UnsupportedOperationException("Disk store only uses LFU.");
    }

    private final class SpoolAndExpiryThread
    extends Thread {
        public SpoolAndExpiryThread() {
            super("Store " + DiskStore.this.name + " Spool Thread");
            this.setDaemon(true);
            this.setPriority(5);
            DiskStore.this.spoolAndExpiryThreadActive = true;
        }

        public final void run() {
            DiskStore.this.spoolAndExpiryThreadMain();
        }
    }

    private static final class DiskElement
    implements Serializable {
        private static final long serialVersionUID = -717310932566592289L;
        private long position;
        private int payloadSize;
        private int blockSize;
        private Object key;
        private long expiryTime;
        private long hitcount;

        private DiskElement() {
        }

        public Object getObjectKey() {
            return this.key;
        }

        public long getHitCount() {
            return this.hitcount;
        }
    }
}

