#!/usr/bin/perl
# Author: Maik Ernst
# History:
# 2013-07-09 - initial script
#
# LICENSE
# Hitachi, Ltd. All rights reserved
# Hitachi Data Systems products and services can be ordered only under the terms and conditions
# of the applicable Hitachi Data Systems agreements.
#
# NAME
# ppcollect.pl
# Hitachi Confoederatio Helvetica performance package collect tool
# Part of HiCHperfpkg
#
# SYNOPSIS
# see usage function below
#
# DESCRIPTION
#
# OPTIONS 
# see "usage" function below.
#
# NOTES
#
# AUTHOR
#   Maik Ernst (maik.ernst@hds.com) - July,2013 
#
# MODIFICATIONS
#   2013-07-09 - v1.0	- initial script
#   2013-07-09 - v1.1	- changed from only tar output file to tar/gzip
#   2013-07-10 - v1.2	- Adding VSP/USPV/HUSVM switch for correct java export tool command
#						- Deleting existing outpath before export tool start to avoid issue if the script got aborted (e.g. strg-c)
#						- Adding script version information
#						- Set -w automatically when osname is *Win*
#						- Set script start time to variable SHORTRANGE if getting Shortrange was not possible from logfile
#						- Bugfix sort on modified time of export tool log file
#  2013-07-23 - v1.3	- Changing default in command txt files to shortrange of 12hours (shortrange  -1200:).
#						  The reason for this is that you get partially invalid values (-4) in LU_dat as the collection period 
#						  does not match the avaiable data (max 24 hours) when the collection takes too long.
#  2013-09-20 - v1.4	- Adding longrange -600000 to all command.txt
#  2013-09-25 - v1.5	- Removing longrange -600000 from all command.txt
#  2013-10-03 - v1.6	- Adding error output of EXPORTTOOL_RUN
#  2013-12-05 - v1.7	- Change command.txt settings from "PhyPG" to "PhyPG Short" and "PhyLDEV" to "PhyLDEV Short"
#						  The reason for this is to shorten the runtime of the export tool
#  2014-01-06 - v1.8	- Changing die-function to be logged in file
#  2014-02-07 - v2.0	- raidcom collection added => use HDSPERF_RAIDCOM.pm module
#  2014-02-07 - v2.1	- adding FindBin, adding HORCM_DIR absolute path due to issue with starting the instance
#  2014-02-17 - v2.2	- adding parsable file (semicolon delimiter) from raidcom output
#  2014-02-21 - v2.3	- add raidcom collection of parity groups
#  2014-04-07 - v2.4	- add raidcom collection of pool and dp_pool
#						- Adding "short" to all long range groups in all command.txt that only short range gets collected. 
#						  The reason for this is to shorten the runtime.
#  2014-04-07 - v3.0	- Adding G1000 support
#						- Changing absolute path handling (=> abs_path) + slashes to backslash as path
#						- Adding -m at raidcom collection to get all MUs for raidscan
#						  But only for TC and not for SI as the collection would take too long.
#  2014-06-13 - v3.1	- Adding 7zip support for archiving with -z. This is to avoid high memory usage if the server has not enough memory to archive.
#						  (When you get error message: "out of memory")
#  2014-12-26 - v3.2	- Adding log file output
#  2015-04-02 - v3.3	- Adding -f option to use FC device instead of IP command device
#						  Adding -o option to collect only raidcom data and skip export tool data collection
#						  Adding Sys::Hostname to get hostname of script
#  2015-04-30 - v3.4	- Using export tool JSanRmiApiSx instead of JSanRmiApiEx for G1000
#  2015-05-02 - v4.0	- Change from parameter to config file => pm file config::tiny needed
#						  Changing script name from HDSperfcollect.pl to HCHppcollect.pl to meet swiss script standard
#  2015-05-05 			- Bugfix (missing 3 in in CDM file for G1000 when using FC command device)
#  2015-05-10 - v4.1.0	- Reorganize folder structure for HiCHperfpkg according to Swiss development group 
#						  (tgz only works but only with absolute pahts => do not use it anymore)
#						  Log format changed (LOG_MSG)
#						  Adding new parameter like ppcollect_ccicollectionlevel
#  2015-07-02 - v4.1.1	- Allow G200, G400, G600 and G800 as parameter
#						  Adding ip-regexp from cmd.txt for G800/G600/G400/G200 (ip instead of svpip)
#						  Adding GUM ip regexp from cmd.txt for G800/G600/G400/G200 
#  2015-07-05 - v4.1.2	- Bugfix regexp to get outpath correctly from cmd.txt
#						  Bugfix unquote outpath
#  2015-08-29 - v4.2.0	- Implement G1000 MC v3 export toool changed 
#						 (also for G800/G600/G400/G200 as assumption that it will be the same soon)							
#						  Change variable to hash for export tool
#  2015-09-10 - v4.2.1	- When user or password "n/a" then exit with error message
#  2016-03-25 - v4.3.0	- LOG->autoflush(1), fix DELETE_OLD_SAVES DELOD log message
#						- adding DELETE_OLD_LOGS / log file rotation ppcollect_logkeepdays
#  2016-04-02 - v4.3.1	- Change log output PWD = ***
#  2016-11-19 - v5.0.0	- adding logging module HiCH_LOG, change logging call to have more parameters (e.g. verbose)
#						- adding REST API collection and method of collection levels (1-7)
#						- adding ppcollect_memoryxmx=variable
#						- adding G1500 support
#						- changing DO_ARCHIV
#  2016-11-24 - v5.0.1	- adding restapi user and rest api password due to raidcom/REST api issue with same user on SVP
#  2016-11-26 - v5.0.2	- adding restapi password decryption
#  2018-02-14 - v5.0.3	- Delete only out*** when export tool flag set
#						- adding REST API collection for FMC (REST_GET_POOLS_FMC)
#						- change REST API collection ldev counter from 1000 to 16384
#  2018-04-18 - v5.0.4	- Get parity group -key opt information
#						- deleting all parsing and _p files (not raw files) for raidcom.
#						- REST API collection level 0(full),1(partial = no LDEV),2(minimum = no LDEV, no hostgroups)
#						- add ${flag_config_filename} to path of script logfile

use strict;
use warnings;
use POSIX qw(strftime);
use Time::Local;
use File::Find;
use File::Path;
use File::Spec;
use File::Basename;
use Archive::Tar;
use Getopt::Long qw(:config no_ignore_case bundling); # Get options /  qw(:config bundling no_ignore_case) makes getopt case sensitive 
use Config;
use Cwd 'abs_path';
use Sys::Hostname; # To get hostname
use IO::Handle; # required for autoflush

# Use raidcom lib. FindBin is needed to find the module when running the script from relativ/absolute paths.
use FindBin qw($Bin);
use lib $Bin;
use lib "$Bin\\..\\lib\\modules";
use Config::Tiny;
use HiCHpp_RAIDCOM;
use HiCH_LOG;
use HiCH_RESTSVP;
use Data::Dumper;

### main ### DO NOT CHANGE ANYTHING IF DO NOT KNOW WHAT YOU ARE DOING!
my $VERSION = "5.0.4";

my $vbse;
my $lglvl = "normal";
my $log_fh;

INIT {$SIG{__DIE__}=sub {LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"GENERAL","Script died: $_[0]") and LOG_MSG_CLOSE($log_fh);}}
INIT {$SIG{__WARN__}=sub {LOG_MSG($log_fh,$vbse,"n",$lglvl,4,"GENERAL","Script warning: $_[0]")}}

# Change to directory of the running script 
# abs_path for $0 or __FILE__ , 
my ( $script_name, $script_path, $script_suffix ) = fileparse( abs_path($0), qr{\.[^.]*$} );
my $script_dirname = dirname (abs_path($0));
# Change slashes to backslashes
$script_dirname =~ s/\//\\/g;
chdir ($script_dirname);
my $script_rootdir =  dirname ($script_dirname);

my $time_scriptstart = strftime("%Y-%m-%d_%H-%M-%S", localtime);
my $HOST = hostname;

# Check Flags
my $flag_help;
my $flag_version;
my $flag_config;

GetOptions (
	'h|help' => \$flag_help,
	'V|VER' => \$flag_version,
	'v|verbose' => \$vbse,
	'c|config=s' => \$flag_config,
           ) or die USAGE();

# Check flags and print usage
if ($flag_version)
{
	print "Version: $VERSION\n";
	exit;
}

if ($flag_help)
{
	USAGE();
	exit;
}

if (!$flag_config)
{
	print "\n!!! You must define -c <config file>. Example: ppcollect.pl -c pp_54068.conf !!!\n";
    USAGE();
    exit;
}

unless (-e "${script_rootdir}\\etc\\${flag_config}")
{
	print "\n!!! Your with -c defined config file \"${script_rootdir}\\etc\\${flag_config}\" does not exist. !!!\n";
    USAGE();
    exit;
}
my $flag_config_filename = $flag_config;
$flag_config = "${script_rootdir}\\etc\\${flag_config}";

# Get config
my $conf_file = Config::Tiny->new;
$conf_file = Config::Tiny->read($flag_config);

# Check & set variables for log

if (!$conf_file->{maindir}->{pp_log})
{
	print "\n!!! Variable \"[maindir] pp_log\" is not defined correctly in $flag_config!!!\n";
	exit;
}

if (!$conf_file->{main}->{ppcollect_serial})
{
	print "\n!!! Variable \"[main] ppcollect_serial\" is not defined correctly in $flag_config!!!\n";
	exit;
}

my $flag_system = uc $conf_file->{main}->{ppcollect_system};
if ($flag_system !~ m/^VSP$|^HUSVM$|^USPV$|^G1500$|^G1000$|^G800$|^G600$|^G400$|^G200$/)
{
	print "\n!!! Variable \"[main] ppcollect_system\" is not defined correctly in $flag_config!!!\n";
	print "!!! It must be set to G1500, G1000, VSP, HUSVM, G800, G600, G400, G200or USPV!!!\n";
	exit;
}

my $scriptdir_log = "${script_rootdir}\\$conf_file->{maindir}->{pp_log}";
my $serial = $conf_file->{main}->{ppcollect_serial};
my $SCRIPTLOG_FILE = "${scriptdir_log}\\${script_name}_${flag_system}_${serial}_${flag_config_filename}_${time_scriptstart}.log";
my $SCRIPTLOG_DEL_PATH = "${scriptdir_log}\\${script_name}_${flag_system}_${serial}_*.log";
print $SCRIPTLOG_FILE;

# Create directories
unless (-d $scriptdir_log) { mkpath($scriptdir_log) or die ("Can't create dir $scriptdir_log: $!\n"); }

# Open script logfile
$log_fh = LOG_MSG_OPEN($SCRIPTLOG_FILE);
LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Starting Script $0");
LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Version: ${VERSION}");
LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Logfile: $SCRIPTLOG_FILE");
LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Operating system: \"$Config{osname}\"");
LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Hostname: \"$HOST\"");
LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","This is a ${flag_system} with the serial ${serial}");
LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Config file: \"$flag_config\"");
LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"PARAM","Parameters (not checked yet) ..."); 

foreach my $conf_section (sort keys %{$conf_file}) 
{
   LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"PARAM","[$conf_section]");
    foreach my $conf_parameter (sort keys %{$conf_file->{$conf_section}}) 
	{
		if ($conf_parameter =~ m/pwd/)
		{
			if ($conf_file->{$conf_section}->{$conf_parameter})
			{
				LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"PARAM","\t$conf_parameter = ***");
			}
			else
			{
				LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"PARAM","\t$conf_parameter = $conf_file->{$conf_section}->{$conf_parameter}");
			}
		}	
		else
		{
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"PARAM","\t$conf_parameter = $conf_file->{$conf_section}->{$conf_parameter}");
		}
	}
}

### Check & set variables for the rest

my $EXPORTTOOL_JAVA;
my $flag_commandtxt;
my $flag_lib;
my $flag_keepdays;
my $flag_logkeepdays;
my $flag_memoryxmx;
my $flag_collection;
my %flag_collection_table = (
	"1" => "export tool only",
	"2" => "raidcom only",
	"3" => "REST API only",
	"4" => "export tool + raidcom",
	"5" => "export tool + REST API",
	"6" => "raidcom + REST API",
	"7" => "export tool + raidcom + REST API",
	);
my %raid_amount_ldevs = (
	"G1500" => "65280",
	"G1000" => "65280",
	"G800" => "16384",
	"G600" => "4096",
	"G400" => "4096",
	"G200" => "2048",
	);
my $flag_win;
my $flag_cci_dir;
my $flag_raidcom_level;
my $RAIDCOM_FCDEVICE;
my $rest_protocol;
my $rest_level;
my $rest_dir;
my $rest_usr;
my $rest_pwd;
my $ZIP_PATH;
my $flag_zip;
my $ARCHIVE_SUFFIX;
my $EXPORTTOOL_LOG;
my $ARCHIVE_DIR;

LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"PARAM","Checking parameter now");
LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"PARAM","Logdir: $scriptdir_log");
LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"PARAM","Serial: $serial");
LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"PARAM","System: $flag_system");

if (lc $conf_file->{log}->{ppcollect_loglevel} eq "normal")
{
	$lglvl = $conf_file->{log}->{ppcollect_loglevel};
	LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"PARAM","Loglevel: normal");
}

elsif (lc $conf_file->{log}->{ppcollect_loglevel} eq "debug")
{
	$lglvl = $conf_file->{log}->{ppcollect_loglevel};
	LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"PARAM","Loglevel: debug");
}

else
{
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","\"[log] ppcollect_loglevel\" is not valid."); 
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","See \"$flag_config\" and check if it is set to \"normal\" or \"debug\".");
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"GENERAL","Finishing Script $0");
	LOG_MSG_CLOSE($log_fh);
	exit;
}

if (!$conf_file->{maindir}->{pp_bin})
{
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","\"[maindir] pp_bin\" is not defined correctly.");
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","Please check your config \"$flag_config\".");
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"GENERAL","Finishing Script $0");
	LOG_MSG_CLOSE($log_fh);
    exit;
}
my $scriptdir_bin = "${script_rootdir}\\$conf_file->{maindir}->{pp_bin}";

if (!$conf_file->{maindir}->{pp_data})
{
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","\"[maindir] pp_data\" is not defined correctly.");
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","Please check your config \"$flag_config\".");
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"GENERAL","Finishing Script $0");
	LOG_MSG_CLOSE($log_fh);
    exit;
}
my $scriptdir_data = "${script_rootdir}\\$conf_file->{maindir}->{pp_data}";

if (!$conf_file->{maindir}->{pp_etc})
{
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","\"[maindir] pp_etc\" is not defined correctly.");
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","Please check your config \"$flag_config\".");
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"GENERAL","Finishing Script $0");
	LOG_MSG_CLOSE($log_fh);
    exit;
}
my $scriptdir_etc = "${script_rootdir}\\$conf_file->{maindir}->{pp_etc}";

if (!$conf_file->{maindir}->{pp_doc})
{
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","\"[maindir] pp_doc\" is not defined correctly.");
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","Please check your config \"$flag_config\".");
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"GENERAL","Finishing Script $0");
	LOG_MSG_CLOSE($log_fh);
    exit;
}
my $scriptdir_doc = "${script_rootdir}\\$conf_file->{maindir}->{pp_doc}";

if (!$conf_file->{maindir}->{pp_lib})
{
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","\"[maindir] pp_lib\" is not defined correctly.");
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","Please check your config \"$flag_config\".");
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"GENERAL","Finishing Script $0");
	LOG_MSG_CLOSE($log_fh);
    exit;
}
my $scriptdir_lib = "${script_rootdir}\\$conf_file->{maindir}->{pp_lib}";

if (!$conf_file->{maindir}->{pp_spool})
{
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","\"[maindir] pp_spool\" is not defined correctly.");
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","Please check your config \"$flag_config\".");
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"GENERAL","Finishing Script $0");
	LOG_MSG_CLOSE($log_fh);
    exit;
}
my $scriptdir_spool = "${script_rootdir}\\$conf_file->{maindir}->{pp_spool}";

unless ((-d $scriptdir_bin) and (-d $scriptdir_data) and (-d $scriptdir_etc) and (-d $scriptdir_doc) and (-d $scriptdir_lib) and (-d $scriptdir_spool) and (-d $scriptdir_log))
{
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","Any of the following directories does not exist...");
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","\t$scriptdir_bin");
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","\t$scriptdir_data");
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","\t$scriptdir_etc");
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","\t$scriptdir_doc");
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","\t$scriptdir_lib");
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","\t$scriptdir_log");
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","\t$scriptdir_spool");
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","\tPlease check your directories and/or [maindir] section in config \"$flag_config\".");
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"GENERAL","\tFinishing Script $0");
	LOG_MSG_CLOSE($log_fh);
    exit;
}

if (!$conf_file->{java}->{ppcollect_javaexecutable})
{
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","\"[java] ppcollect_javaexecutable\" is not defined correctly."); 
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","Please check your config \"$flag_config\".");
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"GENERAL","Finishing Script $0");
	LOG_MSG_CLOSE($log_fh);
	exit;
}
$EXPORTTOOL_JAVA = $conf_file->{java}->{ppcollect_javaexecutable};
LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"PARAM","Java executable: $EXPORTTOOL_JAVA");

if (lc $conf_file->{main}->{pp_mode} ne "windows")
{
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","\"[main] pp_mode\" is not supported. Only windows."); 
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"GENERAL","Finishing Script $0");
	LOG_MSG_CLOSE($log_fh);
    exit;
}
else
{
	if ($Config{osname}=~/Win/)
	{
		$flag_win = 1;
		LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"PARAM","Mode: $conf_file->{main}->{pp_mode}");
	}
	else
	{
		LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","It looks like your OS it not Windows \"$Config{osname}\". Only windows is not supported."); 
		LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"GENERAL","Finishing Script $0");
		LOG_MSG_CLOSE($log_fh);
		exit;
	}
}

$flag_commandtxt = "${scriptdir_etc}\\${flag_system}_command_${serial}.txt";
LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"PARAM","Commandtxt: $flag_commandtxt");

unless (-e $flag_commandtxt)
{
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","Your defined command file: \"$flag_commandtxt\" does not exist.");
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","Please check your config \"$flag_config\" and/or your \"$flag_commandtxt\".");
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"GENERAL","Finishing Script $0");
	LOG_MSG_CLOSE($log_fh);
    exit;
}

if (!$conf_file->{main}->{ppcollect_exporttoollib})
{
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","\"[main] ppcollect_exporttoollib\" is not defined correctly.");
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","Please check your config \"$flag_config\".");
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"GENERAL","Finishing Script $0");
	LOG_MSG_CLOSE($log_fh);
    exit;
}
$flag_lib = "${scriptdir_lib}\\$conf_file->{main}->{ppcollect_exporttoollib}";
LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"PARAM","Libdir: $flag_lib");

unless (-d $flag_lib) 
{
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","Your defined lib dir: \"$flag_lib\" does not exist.");
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","Please check your config \"$flag_config\" and/or your \"$flag_lib\".");
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"GENERAL","Finishing Script $0");
	LOG_MSG_CLOSE($log_fh);
    exit;
}

if ($conf_file->{main}->{ppcollect_keepdays} !~ /^\d+$/)
{
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","\"[main] ppcollect_keepdays\" must be set, a whole number and bigger or equal 0"); 
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","Please check your config \"$flag_config\".");
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"GENERAL","Finishing Script $0");
	LOG_MSG_CLOSE($log_fh);
    exit;
}
$flag_keepdays = $conf_file->{main}->{ppcollect_keepdays};
LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"PARAM","Keepdays: $flag_keepdays");

if ($conf_file->{main}->{ppcollect_logkeepdays} !~ /^\d+$/)
{
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","\"[main] ppcollect_logkeepdays\" must be set, a whole number and bigger or equal 0"); 
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","Please check your config \"$flag_config\".");
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"GENERAL","Finishing Script $0");
	LOG_MSG_CLOSE($log_fh);
    exit;
}
$flag_logkeepdays = $conf_file->{main}->{ppcollect_logkeepdays};
LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"PARAM","Logfile keepdays: $flag_logkeepdays");

if ($conf_file->{main}->{ppcollect_memoryxmx} !~ /^\d+$/)
{
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","\"[main] ppcollect_memoryxmx\" must be set, a whole number and bigger or equal 0"); 
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","Please check your config \"$flag_config\".");
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"GENERAL","Finishing Script $0");
	LOG_MSG_CLOSE($log_fh);
    exit;
}
$flag_memoryxmx = $conf_file->{main}->{ppcollect_memoryxmx};
LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"PARAM","Memory for export tool: $flag_memoryxmx");

if ($conf_file->{main}->{ppcollect_collection} !~ /^\d+$/)
{
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","\"[main] ppcollect_collection\" must be set, a whole number and bigger 0"); 
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","Please check your config \"$flag_config\".");
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"GENERAL","Finishing Script $0");
	LOG_MSG_CLOSE($log_fh);
    exit;
}
$flag_collection = $conf_file->{main}->{ppcollect_collection};
LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"PARAM","Collection level: $flag_collection ($flag_collection_table{$flag_collection})");


unless (-d $conf_file->{raidcom}->{ppcollect_ccidir}) 
{
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","\"[raidcom] ppcollect_ccidir\" does not exist."); 
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","Please check your config \"$flag_config\".");
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"GENERAL","Finishing Script $0");
	LOG_MSG_CLOSE($log_fh);
	exit;
}
$flag_cci_dir = $conf_file->{raidcom}->{ppcollect_ccidir};

if ($conf_file->{raidcom}->{ppcollect_ccicollectionlevel} !~ m/^0|1|2$/)
{
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","\"[raidcom] ppcollect_ccicollectionlevel\" does not exist or is not set to 0, 1 or 2."); 
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","Please check your config \"$flag_config\".");
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"GENERAL","Finishing Script $0");
	LOG_MSG_CLOSE($log_fh);
	exit;
	
}
else
{
	LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"PARAM","Raidcom level set to: $conf_file->{raidcom}->{ppcollect_ccicollectionlevel}");
	$flag_raidcom_level = $conf_file->{raidcom}->{ppcollect_ccicollectionlevel};
}

if (uc $conf_file->{raidcom}->{ppcollect_commanddevice} eq "IP")
{
	LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"PARAM","Raidcom: Set IP CMD");
	$RAIDCOM_FCDEVICE = "no";
}

elsif (uc $conf_file->{raidcom}->{ppcollect_commanddevice} eq "FC")
{
	LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"PARAM","raidcom: Set FC CMD");
	$RAIDCOM_FCDEVICE = "yes";
}

else
{
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","\"[raidcom] ppcollect_commanddevice\" is not valid."); 
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","Please check your config \"$flag_config\" and check if it is set to \"IP\" or \"FC\".");
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"GENERAL","Finishing Script $0");
	LOG_MSG_CLOSE($log_fh);
	exit;
}

if (lc $conf_file->{zip}->{ppcollect_zipmethod} eq "zip") 
{
    
	unless (-e "${scriptdir_bin}\\$conf_file->{zip}->{ppcollect_zippath}") 
	{
		LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","\"[zip] ppcollect_zippath\" does not exist."); 
		LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","Please check your config \"$flag_config\".");
		LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"GENERAL","Finishing Script $0");
		LOG_MSG_CLOSE($log_fh);
		exit;
	}
	$flag_zip = 1;
	$ZIP_PATH = "${scriptdir_bin}\\$conf_file->{zip}->{ppcollect_zippath}";
	$ARCHIVE_SUFFIX = ".zip";
	LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"PARAM","Zipsuffix: $ARCHIVE_SUFFIX - path: $ZIP_PATH ");
}

elsif (lc $conf_file->{zip}->{ppcollect_zipmethod} eq "tgz")
{
	$ARCHIVE_SUFFIX = ".tgz";
	LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"PARAM","Zipsuffix: $ARCHIVE_SUFFIX");
}

else
{
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","\"[zip] ppcollect_zipmethod\" is not valid."); 
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","Please check your config \"$flag_config\" and check if it is set to \"zip\" or \"tgz\".");
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"GENERAL","Finishing Script $0");
	LOG_MSG_CLOSE($log_fh);
	exit;
}

if (!$conf_file->{log}->{ppcollect_exporttoollogprefix})
{
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","\"[log] ppcollect_exporttoollogprefix\" is not defined correctly."); 
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","Please check your config \"$flag_config\".");
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"GENERAL","Finishing Script $0");
	LOG_MSG_CLOSE($log_fh);
	exit;
}
$EXPORTTOOL_LOG = "$scriptdir_log\\$conf_file->{log}->{ppcollect_exporttoollogprefix}${serial}";
LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"PARAM","Export tool logfile: $EXPORTTOOL_LOG");

if (!$conf_file->{zip}->{ppcollect_archivedir})
{
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","\"[zip] ppcollect_archivedir\" is not defined correctly."); 
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","Please check your config \"$flag_config\".");
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"GENERAL","Finishing Script $0");
	LOG_MSG_CLOSE($log_fh);
	exit;
}
$ARCHIVE_DIR = "${scriptdir_data}\\$conf_file->{zip}->{ppcollect_archivedir}${serial}";
LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"PARAM","Archive dir: $ARCHIVE_DIR");
# make path absolute
#$ARCHIVE_DIR =  File::Spec->rel2abs( $ARCHIVE_DIR ) ;
my $ARCHIVE_PREFIX = "${ARCHIVE_DIR}\\${flag_system}_${serial}";
LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"PARAM","Archive dir prefix: $ARCHIVE_PREFIX");

if (!$conf_file->{raidcom}->{ppcollect_horcmdirprefix})
{
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","\"[raidcom] ppcollect_horcmdirprefix\" is not defined correctly."); 
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","Please check your config \"$flag_config\".");
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"GENERAL","Finishing Script $0");
	LOG_MSG_CLOSE($log_fh);
	exit;
}
my $HORCM_DIR = "${scriptdir_data}\\$conf_file->{raidcom}->{ppcollect_horcmdirprefix}${serial}";
LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"PARAM","Horcmdir dir: $HORCM_DIR");


if ($conf_file->{restapi}->{ppcollect_restapiusr})
{
	$rest_usr = $conf_file->{restapi}->{ppcollect_restapiusr};
	LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"PARAM","REST API user: $rest_usr");
}	

if ($conf_file->{restapi}->{ppcollect_restapicryptpwd})
{
	$rest_pwd = $conf_file->{restapi}->{ppcollect_restapicryptpwd};
	LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"PARAM","REST API pwd: ***");
	LOG_MSG($log_fh,$vbse,"v",$lglvl,7,"PARAM","REST API pwd: $rest_pwd");
}	

if ($conf_file->{restapi}->{ppcollect_restapicollectionlevel} !~ m/^0|1|2$/)
{
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","\"[restapi] ppcollect_restapicollectionlevel\" does not exist or is not set to 0, 1 or 2."); 
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","Please check your config \"$flag_config\".");
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"GENERAL","Finishing Script $0");
	LOG_MSG_CLOSE($log_fh);
	exit;
}
$rest_level = $conf_file->{restapi}->{ppcollect_restapicollectionlevel};
LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"PARAM","REST API collection level: $rest_level");
	
if (!$conf_file->{restapi}->{ppcollect_restapidirprefix})
{
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","\"[restapi] ppcollect_restapidirprefix\" is not defined correctly."); 
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","Please check your config \"$flag_config\".");
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"GENERAL","Finishing Script $0");
	LOG_MSG_CLOSE($log_fh);
	exit;
}
$rest_dir = "${scriptdir_data}\\$conf_file->{restapi}->{ppcollect_restapidirprefix}${serial}";
LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"PARAM","REST API dir: $rest_dir");

if ($conf_file->{restapi}->{ppcollect_restapiprotocol} !~ m/^http$|^https$/)
{
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","\"[restapi] ppcollect_restapiprotocol\" is not defined correctly."); 
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"PARAM","Please check your config \"$flag_config\".");
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"GENERAL","Finishing Script $0");
	LOG_MSG_CLOSE($log_fh);
	exit;
}
$rest_protocol = $conf_file->{restapi}->{ppcollect_restapiprotocol};
LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"PARAM","REST API protocol: $rest_protocol");

LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"PARAM","Checking parameter finished");

# Set variables

my $script_crypt = "${scriptdir_lib}\\modules\\c.pl";
my $script_crypt_phrase = "rest_pa33word";
my $script_crypt_length = 31;

my $COMMAND_TXT = $flag_commandtxt;
my $EXPORTTOOL_LIB = $flag_lib;
# make path absolute
$EXPORTTOOL_LIB =  File::Spec->rel2abs( $EXPORTTOOL_LIB ) ;
my $EXPORTTOOL_RUN;
my $EXPORTTOOL_DEL = "n/a";

my %EXPORTTOOLv3_LOADER = (
	"G1500v3_WIN" => "${EXPORTTOOL_LIB}/JSanExportLoader.jar",
	"G1000v3_WIN" => "${EXPORTTOOL_LIB}/JSanExportLoader.jar",
	"G800v3_WIN" => "${EXPORTTOOL_LIB}/JSanExportLoader.jar",
	"G600v3_WIN" => "${EXPORTTOOL_LIB}/JSanExportLoader.jar",
	"G400v3_WIN" => "${EXPORTTOOL_LIB}/JSanExportLoader.jar",
	"G200v3_WIN" => "${EXPORTTOOL_LIB}/JSanExportLoader.jar",
	);
	
my %EXPORTTOOL_CMD = (
	"VSP_WIN" => "${EXPORTTOOL_JAVA} -classpath ${EXPORTTOOL_LIB}/JSanExport.jar;${EXPORTTOOL_LIB}/JSanRmiApiEx.jar;${EXPORTTOOL_LIB}/JSanRmiServerUx.jar -Xmx${flag_memoryxmx}" .
	" -Dmd.command=${COMMAND_TXT} -Dmd.logpath=${EXPORTTOOL_LOG} sanproject.getmondat.RJMdMain",

	"HUSVM_WIN" => "${EXPORTTOOL_JAVA} -classpath ${EXPORTTOOL_LIB}/JSanExport.jar;${EXPORTTOOL_LIB}/JSanRmiApiEx.jar;${EXPORTTOOL_LIB}/JSanRmiServerUx.jar -Xmx${flag_memoryxmx}" .
	" -Dmd.command=${COMMAND_TXT} -Dmd.logpath=${EXPORTTOOL_LOG} sanproject.getmondat.RJMdMain",
	
	"G1000_WIN" => "${EXPORTTOOL_JAVA} -classpath ${EXPORTTOOL_LIB}/JSanExport.jar;${EXPORTTOOL_LIB}/JSanRmiApiSx.jar;${EXPORTTOOL_LIB}/JSanRmiServerUx.jar -Xmx${flag_memoryxmx}" . 
	" -Dmd.command=${COMMAND_TXT} -Dmd.logpath=${EXPORTTOOL_LOG} sanproject.getmondat.RJMdMain",

	"G800_WIN" => "${EXPORTTOOL_JAVA} -classpath ${EXPORTTOOL_LIB}/JSanExport.jar;${EXPORTTOOL_LIB}/JSanRmiApiSx.jar;${EXPORTTOOL_LIB}/JSanRmiServerUx.jar;${EXPORTTOOL_LIB}/SanRmiApi.jar -Xmx${flag_memoryxmx}" .
	" -Dmd.command=${COMMAND_TXT} -Dmd.logpath=${EXPORTTOOL_LOG} sanproject.getmondat.RJMdMain",

	"G600_WIN" => "${EXPORTTOOL_JAVA} -classpath ${EXPORTTOOL_LIB}/JSanExport.jar;${EXPORTTOOL_LIB}/JSanRmiApiSx.jar;${EXPORTTOOL_LIB}/JSanRmiServerUx.jar;${EXPORTTOOL_LIB}/SanRmiApi.jar -Xmx${flag_memoryxmx}" .
	" -Dmd.command=${COMMAND_TXT} -Dmd.logpath=${EXPORTTOOL_LOG} sanproject.getmondat.RJMdMain",

	"G400_WIN" => "${EXPORTTOOL_JAVA} -classpath ${EXPORTTOOL_LIB}/JSanExport.jar;${EXPORTTOOL_LIB}/JSanRmiApiSx.jar;${EXPORTTOOL_LIB}/JSanRmiServerUx.jar;${EXPORTTOOL_LIB}/SanRmiApi.jar -Xmx${flag_memoryxmx}" .
	" -Dmd.command=${COMMAND_TXT} -Dmd.logpath=${EXPORTTOOL_LOG} sanproject.getmondat.RJMdMain",
	
	"G200_WIN" => "${EXPORTTOOL_JAVA} -classpath ${EXPORTTOOL_LIB}/JSanExport.jar;${EXPORTTOOL_LIB}/JSanRmiApiSx.jar;${EXPORTTOOL_LIB}/JSanRmiServerUx.jar;${EXPORTTOOL_LIB}/SanRmiApi.jar -Xmx${flag_memoryxmx}" .
	" -Dmd.command=${COMMAND_TXT} -Dmd.logpath=${EXPORTTOOL_LOG} sanproject.getmondat.RJMdMain",
	
	"USPV_WIN" => "${EXPORTTOOL_JAVA} -classpath ${EXPORTTOOL_LIB}/JSanExport.jar;${EXPORTTOOL_LIB}/JSanRmiServerSx.jar; -Xmx${flag_memoryxmx}" .
	" -Dmd.command=${COMMAND_TXT} -Dmd.logpath=${EXPORTTOOL_LOG} sanproject.getmondat.RJMdMain",
		
	"G1500v3_WIN_RUN" => "${EXPORTTOOL_JAVA} -classpath $EXPORTTOOLv3_LOADER{G1500v3_WIN} -Del.tool.Xmx=${flag_memoryxmx} -Dmd.command=${COMMAND_TXT} -Del.logpath=${EXPORTTOOL_LOG}" .
	" -Dmd.rmitimeout=20 sanproject.getexptool.RJElMain",
	
	"G1500v3_WIN_DEL" => "${EXPORTTOOL_JAVA} -classpath $EXPORTTOOLv3_LOADER{G1500v3_WIN} -Dmd.command=${COMMAND_TXT} -Del.logpath=${EXPORTTOOL_LOG} -Del.mode=delete" .
	" -Dmd.rmitimeout=20 sanproject.getexptool.RJElMain",
	
	"G1000v3_WIN_RUN" => "${EXPORTTOOL_JAVA} -classpath $EXPORTTOOLv3_LOADER{G1000v3_WIN} -Del.tool.Xmx=${flag_memoryxmx} -Dmd.command=${COMMAND_TXT} -Del.logpath=${EXPORTTOOL_LOG}" .
	" -Dmd.rmitimeout=20 sanproject.getexptool.RJElMain",
	
	"G1000v3_WIN_DEL" => "${EXPORTTOOL_JAVA} -classpath $EXPORTTOOLv3_LOADER{G1000v3_WIN} -Dmd.command=${COMMAND_TXT} -Del.logpath=${EXPORTTOOL_LOG} -Del.mode=delete" .
	" -Dmd.rmitimeout=20 sanproject.getexptool.RJElMain",
	
	"G800v3_WIN_RUN" => "${EXPORTTOOL_JAVA} -classpath $EXPORTTOOLv3_LOADER{G800v3_WIN} -Del.tool.Xmx=${flag_memoryxmx} -Dmd.command=${COMMAND_TXT} -Del.logpath=${EXPORTTOOL_LOG}" .
	" -Dmd.rmitimeout=20 sanproject.getexptool.RJElMain",
	
	"G800v3_WIN_DEL" => "${EXPORTTOOL_JAVA} -classpath $EXPORTTOOLv3_LOADER{G800v3_WIN} -Dmd.command=${COMMAND_TXT} -Del.logpath=${EXPORTTOOL_LOG} -Del.mode=delete" .
	" -Dmd.rmitimeout=20 sanproject.getexptool.RJElMain",
		
	"G600v3_WIN_RUN" => "${EXPORTTOOL_JAVA} -classpath $EXPORTTOOLv3_LOADER{G600v3_WIN} -Del.tool.Xmx=${flag_memoryxmx} -Dmd.command=${COMMAND_TXT} -Del.logpath=${EXPORTTOOL_LOG}" .
	" -Dmd.rmitimeout=20 sanproject.getexptool.RJElMain",
	
	"G600v3_WIN_DEL" => "${EXPORTTOOL_JAVA} -classpath $EXPORTTOOLv3_LOADER{G600v3_WIN} -Dmd.command=${COMMAND_TXT} -Del.logpath=${EXPORTTOOL_LOG} -Del.mode=delete" .
	" -Dmd.rmitimeout=20 sanproject.getexptool.RJElMain",
	
	"G400v3_WIN_RUN" => "${EXPORTTOOL_JAVA} -classpath $EXPORTTOOLv3_LOADER{G400v3_WIN} -Del.tool.Xmx=${flag_memoryxmx} -Dmd.command=${COMMAND_TXT} -Del.logpath=${EXPORTTOOL_LOG}" .
	" -Dmd.rmitimeout=20 sanproject.getexptool.RJElMain",
	
	"G400v3_WIN_DEL" => "${EXPORTTOOL_JAVA} -classpath $EXPORTTOOLv3_LOADER{G400v3_WIN} -Dmd.command=${COMMAND_TXT} -Del.logpath=${EXPORTTOOL_LOG} -Del.mode=delete" .
	" -Dmd.rmitimeout=20 sanproject.getexptool.RJElMain",
		
	"G200v3_WIN_RUN" => "${EXPORTTOOL_JAVA} -classpath $EXPORTTOOLv3_LOADER{G200v3_WIN} -Del.tool.Xmx=${flag_memoryxmx} -Dmd.command=${COMMAND_TXT} -Del.logpath=${EXPORTTOOL_LOG}" .
	" -Dmd.rmitimeout=20 sanproject.getexptool.RJElMain",
	
	"G200v3_WIN_DEL" => "${EXPORTTOOL_JAVA} -classpath $EXPORTTOOLv3_LOADER{G200v3_WIN} -Dmd.command=${COMMAND_TXT} -Del.logpath=${EXPORTTOOL_LOG} -Del.mode=delete" .
	" -Dmd.rmitimeout=20 sanproject.getexptool.RJElMain",
	
	);
	
my $EXPORTTOOL_RUN_VSP_UNIX = "${EXPORTTOOL_JAVA} -classpath ${EXPORTTOOL_LIB}/JSanExport.jar:${EXPORTTOOL_LIB}/JSanRmiApiEx.jar:${EXPORTTOOL_LIB}/JSanRmiServerUx.jar -Xmx${flag_memoryxmx} -Dmd.command=${COMMAND_TXT} -Dmd.logpath=${EXPORTTOOL_LOG} sanproject.getmondat.RJMdMain";
my $EXPORTTOOL_RUN_HUSVM_UNIX = "${EXPORTTOOL_JAVA} -classpath ${EXPORTTOOL_LIB}/JSanExport.jar:${EXPORTTOOL_LIB}/JSanRmiApiEx.jar:${EXPORTTOOL_LIB}/JSanRmiServerUx.jar -Xmx${flag_memoryxmx} -Dmd.command=${COMMAND_TXT} -Dmd.logpath=${EXPORTTOOL_LOG} sanproject.getmondat.RJMdMain";
my $EXPORTTOOL_RUN_G1000_UNIX = "${EXPORTTOOL_JAVA} -classpath ${EXPORTTOOL_LIB}/JSanExport.jar:${EXPORTTOOL_LIB}/JSanRmiApiSx.jar:${EXPORTTOOL_LIB}/JSanRmiServerUx.jar -Xmx${flag_memoryxmx} -Dmd.command=${COMMAND_TXT} -Dmd.logpath=${EXPORTTOOL_LOG} sanproject.getmondat.RJMdMain";
my $EXPORTTOOL_RUN_G800_UNIX = "${EXPORTTOOL_JAVA} -classpath ${EXPORTTOOL_LIB}/JSanExport.jar:${EXPORTTOOL_LIB}/JSanRmiApiSx.jar:${EXPORTTOOL_LIB}/JSanRmiServerUx.jar;${EXPORTTOOL_LIB}/SanRmiApi.jar -Xmx${flag_memoryxmx} -Dmd.command=${COMMAND_TXT} -Dmd.logpath=${EXPORTTOOL_LOG} sanproject.getmondat.RJMdMain";
my $EXPORTTOOL_RUN_G600_UNIX = "${EXPORTTOOL_JAVA} -classpath ${EXPORTTOOL_LIB}/JSanExport.jar:${EXPORTTOOL_LIB}/JSanRmiApiSx.jar:${EXPORTTOOL_LIB}/JSanRmiServerUx.jar;${EXPORTTOOL_LIB}/SanRmiApi.jar -Xmx${flag_memoryxmx} -Dmd.command=${COMMAND_TXT} -Dmd.logpath=${EXPORTTOOL_LOG} sanproject.getmondat.RJMdMain";
my $EXPORTTOOL_RUN_G400_UNIX = "${EXPORTTOOL_JAVA} -classpath ${EXPORTTOOL_LIB}/JSanExport.jar:${EXPORTTOOL_LIB}/JSanRmiApiSx.jar:${EXPORTTOOL_LIB}/JSanRmiServerUx.jar;${EXPORTTOOL_LIB}/SanRmiApi.jar -Xmx${flag_memoryxmx} -Dmd.command=${COMMAND_TXT} -Dmd.logpath=${EXPORTTOOL_LOG} sanproject.getmondat.RJMdMain";
my $EXPORTTOOL_RUN_G200_UNIX = "${EXPORTTOOL_JAVA} -classpath ${EXPORTTOOL_LIB}/JSanExport.jar:${EXPORTTOOL_LIB}/JSanRmiApiSx.jar:${EXPORTTOOL_LIB}/JSanRmiServerUx.jar;${EXPORTTOOL_LIB}/SanRmiApi.jar -Xmx${flag_memoryxmx} -Dmd.command=${COMMAND_TXT} -Dmd.logpath=${EXPORTTOOL_LOG} sanproject.getmondat.RJMdMain";
my $EXPORTTOOL_RUN_USPV_UNIX = "${EXPORTTOOL_JAVA} -classpath ${EXPORTTOOL_LIB}/JSanExport.jar:${EXPORTTOOL_LIB}/JSanRmiServerSx.jar; -Xmx${flag_memoryxmx} -Dmd.command=${COMMAND_TXT} -Dmd.logpath=${EXPORTTOOL_LOG} sanproject.getmondat.RJMdMain";

# Create directories
unless (-d $ARCHIVE_DIR) { mkpath($ARCHIVE_DIR) or die ("Can't create dir $ARCHIVE_DIR: $!\n"); }


# Check the flags and set $EXPORTTOOL_RUN to environment

if ($flag_win)
{
	if ($flag_system eq "VSP")
	{
		$EXPORTTOOL_RUN = $EXPORTTOOL_CMD{VSP_WIN};
		LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Running in windows mode");
		LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Running in VSP mode");
		LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","EXPORTTOOL_RUN = $EXPORTTOOL_RUN");
	}
	elsif ($flag_system eq "USPV")
	{
		$EXPORTTOOL_RUN = $EXPORTTOOL_CMD{USPV_WIN};
		LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Running in windows mode");
		LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Running in USPV mode");
		LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","EXPORTTOOL_RUN = $EXPORTTOOL_RUN");
	}
	elsif ($flag_system eq "HUSVM")
	{
		$EXPORTTOOL_RUN = $EXPORTTOOL_CMD{HUSVM_WIN};
		LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Running in windows mode");
		LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Running in HUSVM mode");
		LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","EXPORTTOOL_RUN = $EXPORTTOOL_RUN");
	}
	elsif ($flag_system eq "G1500")
	{
		#must be >v3 microcode
		$EXPORTTOOL_RUN = $EXPORTTOOL_CMD{G1500v3_WIN_RUN};
		$EXPORTTOOL_DEL = $EXPORTTOOL_CMD{G1500v3_WIN_DEL};
		LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Running in windows mode");
		LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Running in G1500 mode above v3 MC");
		LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","EXPORTTOOL_RUN = $EXPORTTOOL_RUN");
		LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","EXPORTTOOL_DEL = $EXPORTTOOL_DEL");
	}
	elsif ($flag_system eq "G1000")
	{
		#check if G1000 v3 microcode
		if (-e $EXPORTTOOLv3_LOADER{G1000v3_WIN})
		{
			$EXPORTTOOL_RUN = $EXPORTTOOL_CMD{G1000v3_WIN_RUN};
			$EXPORTTOOL_DEL = $EXPORTTOOL_CMD{G1000v3_WIN_DEL};
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Running in windows mode");
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Running in G1000 mode above v3 MC");
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","EXPORTTOOL_RUN = $EXPORTTOOL_RUN");
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","EXPORTTOOL_DEL = $EXPORTTOOL_DEL");
		}
		else
		{
			$EXPORTTOOL_RUN = $EXPORTTOOL_CMD{G1000_WIN};
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Running in windows mode");
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Running in G1000 mode below v3 MC");
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","EXPORTTOOL_RUN = $EXPORTTOOL_RUN");
		}
	}
	elsif ($flag_system eq "G800")
	{
		#check if G800 v3 microcode
		if (-e $EXPORTTOOLv3_LOADER{G800v3_WIN})
		{
			$EXPORTTOOL_RUN = $EXPORTTOOL_CMD{G800v3_WIN_RUN};
			$EXPORTTOOL_DEL = $EXPORTTOOL_CMD{G800v3_WIN_DEL};
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Running in windows mode");
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Running in G800 mode above v3 MC");
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","EXPORTTOOL_RUN = $EXPORTTOOL_RUN");
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","EXPORTTOOL_DEL = $EXPORTTOOL_DEL");
		}
		else
		{
		$EXPORTTOOL_RUN = $EXPORTTOOL_CMD{G800_WIN};
		LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Running in windows mode");
		LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Running in G800 mode");
		LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","EXPORTTOOL_RUN = $EXPORTTOOL_RUN");
		}
	}
	elsif ($flag_system eq "G600")
	{
		#check if G600 v3 microcode
		if (-e $EXPORTTOOLv3_LOADER{G600v3_WIN})
		{
			$EXPORTTOOL_RUN = $EXPORTTOOL_CMD{G600v3_WIN_RUN};
			$EXPORTTOOL_DEL = $EXPORTTOOL_CMD{G600v3_WIN_DEL};
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Running in windows mode");
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Running in G600 mode above v3 MC");
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","EXPORTTOOL_RUN = $EXPORTTOOL_RUN");
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","EXPORTTOOL_DEL = $EXPORTTOOL_DEL");
		}
		else
		{
			$EXPORTTOOL_RUN = $EXPORTTOOL_CMD{G600_WIN};
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Running in windows mode");
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Running in G600 mode");
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","EXPORTTOOL_RUN = $EXPORTTOOL_RUN");
		}
	}
	elsif ($flag_system eq "G400")
	{
		#check if G400 v3 microcode
		if (-e $EXPORTTOOLv3_LOADER{G400v3_WIN})
		{
			$EXPORTTOOL_RUN = $EXPORTTOOL_CMD{G400v3_WIN_RUN};
			$EXPORTTOOL_DEL = $EXPORTTOOL_CMD{G400v3_WIN_DEL};
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Running in windows mode");
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Running in G400 mode above v3 MC");
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","EXPORTTOOL_RUN = $EXPORTTOOL_RUN");
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","EXPORTTOOL_DEL = $EXPORTTOOL_DEL");
		}
		else
		{
			$EXPORTTOOL_RUN = $EXPORTTOOL_CMD{G400_WIN};
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Running in windows mode");
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Running in G400 mode");
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","EXPORTTOOL_RUN = $EXPORTTOOL_RUN");
		}
	}
	elsif ($flag_system eq "G200")
	{
		#check if G200 v3 microcode
		if (-e $EXPORTTOOLv3_LOADER{G200v3_WIN})
		{
			$EXPORTTOOL_RUN = $EXPORTTOOL_CMD{G200v3_WIN_RUN};
			$EXPORTTOOL_DEL = $EXPORTTOOL_CMD{G200v3_WIN_DEL};
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Running in windows mode");
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Running in G200 mode above v3 MC");
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","EXPORTTOOL_RUN = $EXPORTTOOL_RUN");
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","EXPORTTOOL_DEL = $EXPORTTOOL_DEL");
		}
		else
		{
			$EXPORTTOOL_RUN = $EXPORTTOOL_CMD{G200_WIN};
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Running in windows mode");
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Running in G200 mode");
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","EXPORTTOOL_RUN = $EXPORTTOOL_RUN");
		}
	}
	else
	{
		LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"GENERAL","You should never be here. Check -s $flag_system");
		exit;
	}
}

else 
{
        if ($flag_system eq "VSP")
        {
			$EXPORTTOOL_RUN = $EXPORTTOOL_RUN_VSP_UNIX;
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Running in Unix mode");
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Running in VSP mode");
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","EXPORTTOOL_RUN = $EXPORTTOOL_RUN");
        }
        elsif ($flag_system eq "USPV")
        {
			$EXPORTTOOL_RUN = $EXPORTTOOL_RUN_USPV_UNIX;
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Running in Unix mode");
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Running in USPV mode");
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","EXPORTTOOL_RUN = $EXPORTTOOL_RUN");
        }
        elsif ($flag_system eq "HUSVM")
        {
			$EXPORTTOOL_RUN = $EXPORTTOOL_RUN_HUSVM_UNIX;
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Running in Unix mode");
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Running in HUSVM mode");
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","EXPORTTOOL_RUN = $EXPORTTOOL_RUN");
        }
		elsif ($flag_system eq "G1000")
        {
			$EXPORTTOOL_RUN = $EXPORTTOOL_RUN_G1000_UNIX;
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Running in Unix mode");
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Running in G1000 mode");
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","EXPORTTOOL_RUN = $EXPORTTOOL_RUN");
        }
        else
        {
			LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"GENERAL","You should never be here. Check -s $flag_system");
			LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"GENERAL","Finishing Script $0");
			LOG_MSG_CLOSE($log_fh);
			exit;
        }
}

my ($EXPORTTOOL_OUTPATH,$EXPORTTOOL_IP,$EXPORTTOOL_USER,$EXPORTTOOL_PWD,$GUM_IPs) = GET_OUTPATH_IP_PWD($serial,$COMMAND_TXT,$log_fh,$vbse,$lglvl);

# Delete outpath if existing and export tool gets collected
if (-d $EXPORTTOOL_OUTPATH)
{	
	LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Directory $EXPORTTOOL_OUTPATH exists");
	if (($flag_collection == 1) or ($flag_collection == 4) or ($flag_collection == 5) or ($flag_collection == 7))
	{
		# 1=export tool only, 2=raidcom only, 3=REST API only,
		# 4=export tool + raidcom, 5=export tool + REST API,
		# 6=raidcom + REST API, 7=export tool + raidcom + REST API
		LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Export tool in collection level ($flag_collection)");
		LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Deleting old $EXPORTTOOL_OUTPATH");
		rmtree ($EXPORTTOOL_OUTPATH) or die LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"GENERAL","Can't delete dirs $EXPORTTOOL_OUTPATH: $!\n");
		LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Deleting finished");
	}
	else
	{
		LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Not deleted due to export tool not in collection level ($flag_collection)");
	}

}


# Export tool only
if ($flag_collection == 1)
{
	LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Collection level $flag_collection ($flag_collection_table{$flag_collection}) start");
	
	# export tool	
	my $SHORTRANGE = DO_EXPORTTOOL_STUFF();

	# archive
	my $archive_file = "${ARCHIVE_PREFIX}_${flag_collection}_${SHORTRANGE}${ARCHIVE_SUFFIX}";
	my @backup_dirs = ($COMMAND_TXT,$EXPORTTOOL_OUTPATH,$EXPORTTOOL_LOG);
	my @backup_dirs_del_after_run = ($EXPORTTOOL_OUTPATH,$EXPORTTOOL_LOG);
	DO_ARCHIV(\@backup_dirs,\@backup_dirs_del_after_run,$archive_file,$flag_zip,$ZIP_PATH,$log_fh,$vbse,$lglvl);

	unless ($flag_keepdays == 0)
	{
		DELETE_OLD_SAVES($flag_keepdays,$ARCHIVE_PREFIX,$ARCHIVE_SUFFIX,$log_fh,$vbse,$lglvl);
	}

	unless ($flag_logkeepdays == 0)
	{
		DELETE_OLD_LOGS($flag_logkeepdays,$SCRIPTLOG_DEL_PATH,$log_fh,$vbse,$lglvl);
	}
		
	LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Finishing Script $0");
	LOG_MSG_CLOSE($log_fh);
}
# raidcom only
elsif  ($flag_collection == 2)
{
	LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Collection level $flag_collection ($flag_collection_table{$flag_collection}) start");
	
	LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Setting file name to:\"${time_scriptstart}\"");
	my $SHORTRANGE = "${time_scriptstart}";
	
	# raidcom
	DO_RAIDCOM_STUFF();
	
	# archive
	my $archive_file = "${ARCHIVE_PREFIX}_${flag_collection}_${SHORTRANGE}${ARCHIVE_SUFFIX}";
	my @backup_dirs = ($COMMAND_TXT,$HORCM_DIR);
	my @backup_dirs_del_after_run = ($HORCM_DIR);
	DO_ARCHIV(\@backup_dirs,\@backup_dirs_del_after_run,$archive_file,$flag_zip,$ZIP_PATH,$log_fh,$vbse,$lglvl);

	unless ($flag_keepdays == 0)
	{
		DELETE_OLD_SAVES($flag_keepdays,$ARCHIVE_PREFIX,$ARCHIVE_SUFFIX,$log_fh,$vbse,$lglvl);
	}

	unless ($flag_logkeepdays == 0)
	{
		DELETE_OLD_LOGS($flag_logkeepdays,$SCRIPTLOG_DEL_PATH,$log_fh,$vbse,$lglvl);
	}
		
	LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Finishing Script $0");
	LOG_MSG_CLOSE($log_fh);
}
# REST API only
elsif  ($flag_collection == 3)
{
	LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Collection level $flag_collection ($flag_collection_table{$flag_collection}) start");
	
	LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Setting file name to:\"${time_scriptstart}\"");
	my $SHORTRANGE = "${time_scriptstart}";
	
	# REST API
	DO_RESTAPI_STUFF();
	
	#archive
	my $archive_file = "${ARCHIVE_PREFIX}_${flag_collection}_${SHORTRANGE}${ARCHIVE_SUFFIX}";
	my @backup_dirs = ($COMMAND_TXT,$rest_dir);
	my @backup_dirs_del_after_run = ($rest_dir);
	DO_ARCHIV(\@backup_dirs,\@backup_dirs_del_after_run,$archive_file,$flag_zip,$ZIP_PATH,$log_fh,$vbse,$lglvl);

	unless ($flag_keepdays == 0)
	{
		DELETE_OLD_SAVES($flag_keepdays,$ARCHIVE_PREFIX,$ARCHIVE_SUFFIX,$log_fh,$vbse,$lglvl);
	}

	unless ($flag_logkeepdays == 0)
	{
		DELETE_OLD_LOGS($flag_logkeepdays,$SCRIPTLOG_DEL_PATH,$log_fh,$vbse,$lglvl);
	}
	
	LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Finishing Script $0");
	LOG_MSG_CLOSE($log_fh);
}
# Export tool and raidcom
elsif  ($flag_collection == 4)
{
	
	LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Collection level $flag_collection ($flag_collection_table{$flag_collection}) start");
	
	# export tool	
	my $SHORTRANGE = DO_EXPORTTOOL_STUFF();
	
	# raidcom 
	DO_RAIDCOM_STUFF();
	
	# archive
	my $archive_file = "${ARCHIVE_PREFIX}_${flag_collection}_${SHORTRANGE}${ARCHIVE_SUFFIX}";
	my @backup_dirs = ($COMMAND_TXT,$EXPORTTOOL_OUTPATH,$EXPORTTOOL_LOG,$HORCM_DIR);
	my @backup_dirs_del_after_run = ($EXPORTTOOL_OUTPATH,$EXPORTTOOL_LOG,$HORCM_DIR);
	DO_ARCHIV(\@backup_dirs,\@backup_dirs_del_after_run,$archive_file,$flag_zip,$ZIP_PATH,$log_fh,$vbse,$lglvl);

	unless ($flag_keepdays == 0)
	{
		DELETE_OLD_SAVES($flag_keepdays,$ARCHIVE_PREFIX,$ARCHIVE_SUFFIX,$log_fh,$vbse,$lglvl);
	}

	unless ($flag_logkeepdays == 0)
	{
		DELETE_OLD_LOGS($flag_logkeepdays,$SCRIPTLOG_DEL_PATH,$log_fh,$vbse,$lglvl);
	}
	
	LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Finishing Script $0");
	LOG_MSG_CLOSE($log_fh);
}
# Export tool and REST API
elsif  ($flag_collection == 5)
{
	LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Collection level $flag_collection ($flag_collection_table{$flag_collection}) start");
	
	# export tool	
	my $SHORTRANGE = DO_EXPORTTOOL_STUFF();
	
	# REST API
	DO_RESTAPI_STUFF();
	
	#archive
	my $archive_file = "${ARCHIVE_PREFIX}_${flag_collection}_${SHORTRANGE}${ARCHIVE_SUFFIX}";
	my @backup_dirs = ($COMMAND_TXT,$EXPORTTOOL_OUTPATH,$EXPORTTOOL_LOG,$rest_dir);
	my @backup_dirs_del_after_run = ($EXPORTTOOL_OUTPATH,$EXPORTTOOL_LOG,$rest_dir);
	DO_ARCHIV(\@backup_dirs,\@backup_dirs_del_after_run,$archive_file,$flag_zip,$ZIP_PATH,$log_fh,$vbse,$lglvl);

	unless ($flag_keepdays == 0)
	{
		DELETE_OLD_SAVES($flag_keepdays,$ARCHIVE_PREFIX,$ARCHIVE_SUFFIX,$log_fh,$vbse,$lglvl);
	}

	unless ($flag_logkeepdays == 0)
	{
		DELETE_OLD_LOGS($flag_logkeepdays,$SCRIPTLOG_DEL_PATH,$log_fh,$vbse,$lglvl);
	}
		
	LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Finishing Script $0");
	LOG_MSG_CLOSE($log_fh);
}
# radicom and REST API
elsif  ($flag_collection == 6)
{
	LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Collection level $flag_collection ($flag_collection_table{$flag_collection}) start");
	
	LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Setting file name to:\"${time_scriptstart}\"");
	my $SHORTRANGE = "${time_scriptstart}";
	
	# raidcom
	DO_RAIDCOM_STUFF();
	
	# REST API
	DO_RESTAPI_STUFF();
	
	#archive
	my $archive_file = "${ARCHIVE_PREFIX}_${flag_collection}_${SHORTRANGE}${ARCHIVE_SUFFIX}";
	my @backup_dirs = ($COMMAND_TXT,$rest_dir,$HORCM_DIR);
	my @backup_dirs_del_after_run = ($rest_dir,$HORCM_DIR);
	DO_ARCHIV(\@backup_dirs,\@backup_dirs_del_after_run,$archive_file,$flag_zip,$ZIP_PATH,$log_fh,$vbse,$lglvl);

	unless ($flag_keepdays == 0)
	{
		DELETE_OLD_SAVES($flag_keepdays,$ARCHIVE_PREFIX,$ARCHIVE_SUFFIX,$log_fh,$vbse,$lglvl);
	}

	unless ($flag_logkeepdays == 0)
	{
		DELETE_OLD_LOGS($flag_logkeepdays,$SCRIPTLOG_DEL_PATH,$log_fh,$vbse,$lglvl);
	}
		
	LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Finishing Script $0");
	LOG_MSG_CLOSE($log_fh);
}
# Export tool and raidcom and REST API
elsif  ($flag_collection == 7)
{
	LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Collection level $flag_collection ($flag_collection_table{$flag_collection}) start");
	
	# export tool	
	my $SHORTRANGE = DO_EXPORTTOOL_STUFF();
	
	# raidcom 
	DO_RAIDCOM_STUFF();
	
	# REST API
	DO_RESTAPI_STUFF();
	
	#archive
	my $archive_file = "${ARCHIVE_PREFIX}_${flag_collection}_${SHORTRANGE}${ARCHIVE_SUFFIX}";
	my @backup_dirs = ($COMMAND_TXT,$EXPORTTOOL_OUTPATH,$EXPORTTOOL_LOG,$HORCM_DIR,$rest_dir);
	my @backup_dirs_del_after_run = ($EXPORTTOOL_OUTPATH,$EXPORTTOOL_LOG,$HORCM_DIR,$rest_dir);
	DO_ARCHIV(\@backup_dirs,\@backup_dirs_del_after_run,$archive_file,$flag_zip,$ZIP_PATH,$log_fh,$vbse,$lglvl);

	unless ($flag_keepdays == 0)
	{
		DELETE_OLD_SAVES($flag_keepdays,$ARCHIVE_PREFIX,$ARCHIVE_SUFFIX,$log_fh,$vbse,$lglvl);
	}

	unless ($flag_logkeepdays == 0)
	{
		DELETE_OLD_LOGS($flag_logkeepdays,$SCRIPTLOG_DEL_PATH,$log_fh,$vbse,$lglvl);
	}
	
	LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Finishing Script $0");
	LOG_MSG_CLOSE($log_fh);
}
else
{
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"GENERAL","Collection level $flag_collection (see conf file ppcollect_collection) not valid.");
	LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"GENERAL","Finishing Script $0");
	LOG_MSG_CLOSE($log_fh);
}

### sub routines ###
### subroutine DO_RAIDCOM can be found in module HiCHpp_RAIDCOM.pm

### special sub, no parameters ###
sub DO_EXPORTTOOL_STUFF
{
	LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Export tool start");
	# export tool	
	if ( $EXPORTTOOL_DEL ne "n/a" )
	{
		DEL_EXPORTTOOL($EXPORTTOOL_DEL,$log_fh,$vbse,$lglvl);
	}
	RUN_EXPORTTOOL($EXPORTTOOL_RUN,$log_fh,$vbse,$lglvl);
	my $SHORTRANGE;
	$SHORTRANGE = GET_SHORTRANGE($EXPORTTOOL_LOG,$log_fh,$vbse,$lglvl);
	if (!$SHORTRANGE)
	{
		LOG_MSG($log_fh,$vbse,"n",$lglvl,4,"GENERAL","Did not get Shortrange. Setting it to:\"NO_SHORTRANGE_${time_scriptstart}\"");
		$SHORTRANGE = "NO_SHORTRANGE_${time_scriptstart}";
	}

	LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Export tool finished");
	return $SHORTRANGE;
}

sub DO_RAIDCOM_STUFF
{
	LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Raidcom start");

	if (($flag_win) and (($flag_system eq "HUSVM") or ($flag_system eq "VSP") or ($flag_system eq "G1000") or ($flag_system eq "G1500")
	or ($flag_system eq "G800") or ($flag_system eq "G600") or ($flag_system eq "G400") or ($flag_system eq "G200")))
	{
		LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Creating directory for raidcom: $HORCM_DIR");
		unless (-d $HORCM_DIR) { mkpath($HORCM_DIR) or warn LOG_MSG($log_fh,$vbse,"n",$lglvl,4,"GENERAL","Can't create dir $HORCM_DIR: $!\n"); }
		
		# This check is required as Gx00 uses GUM IP for raidcom instead of SVP IP
		if ($flag_system =~ /^G800$|^G600$|^G400$|^G200$/)
		{
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","GUM detected. Setting GUM IPs.");
			DO_RAIDCOM ($flag_cci_dir,$GUM_IPs,$HORCM_DIR,$EXPORTTOOL_USER,$EXPORTTOOL_PWD,$serial,$RAIDCOM_FCDEVICE,$flag_system,$HOST,$flag_raidcom_level,$log_fh,$vbse,$lglvl);
		}
		else
		{
			DO_RAIDCOM ($flag_cci_dir,$EXPORTTOOL_IP,$HORCM_DIR,$EXPORTTOOL_USER,$EXPORTTOOL_PWD,$serial,$RAIDCOM_FCDEVICE,$flag_system,$HOST,$flag_raidcom_level,$log_fh,$vbse,$lglvl);
		}
	}
	else
	{
		LOG_MSG($log_fh,$vbse,"n",$lglvl,4,"GENERAL","Currently only windows for HUS-VM/VSP/G1500/G1000/G800/G600/G400/G200 is supported for raidcom collection therefore raidcom information will NOT get collected.");
	}
	
	LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Raidcom finished");
}

sub DO_RESTAPI_STUFF
{
	my $restapi_request_usr = $rest_usr;
	my $restapi_request_pwd = $rest_pwd;
	
	LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST start");
	
	if ((!defined $restapi_request_usr) or (!defined $restapi_request_pwd) or ($restapi_request_usr eq "") or ($restapi_request_pwd eq "")) 
	{
		LOG_MSG($log_fh,$vbse,"n",$lglvl,4,"GENERAL","REST user or password not set in conf file. Not collecting REST API!");
		return -1;
	}
	
	# REST API
	if (($flag_win) and (($flag_system eq "G1000") or ($flag_system eq "G1500") or ($flag_system eq "G800") 
	or ($flag_system eq "G600") or ($flag_system eq "G400") or ($flag_system eq "G200")))
	{
		LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Creating directory for REST API: $rest_dir");
		unless (-d $rest_dir) { mkpath($rest_dir) or warn LOG_MSG($log_fh,$vbse,"n",$lglvl,4,"GENERAL","Can't create dir $rest_dir: $!\n"); }
		
		LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Decode REST API password start");
		
		my $length_encrypted_restapi_cryptpwd = length $restapi_request_pwd;
		if ($length_encrypted_restapi_cryptpwd != $script_crypt_length)
		{			
			LOG_MSG($log_fh,$vbse,"n",$lglvl,4,"GENERAL","Something went wrong with the REST API pwd decoding.");
			LOG_MSG($log_fh,$vbse,"n",$lglvl,4,"GENERAL","The default length ($script_crypt_length) does not match with crypt pwd length ($length_encrypted_restapi_cryptpwd). Not collecting REST API!");
			return -1;
		}
		
		my $encrypted_restapi_cryptpwd = "n/a";
		LOG_MSG($log_fh,$vbse,"v",$lglvl,7,"GENERAL","Decode REST command: $script_crypt -d $restapi_request_pwd -p $script_crypt_phrase");
		open (CMD_CRYPT, "$script_crypt -d $restapi_request_pwd -p $script_crypt_phrase 2>&1|") or warn LOG_MSG($log_fh,$vbse,"n",$lglvl,4,"GENERAL","!!! Can't run program: $!\n");
		while (my $line = <CMD_CRYPT>)
		{
			LOG_MSG($log_fh,$vbse,"v",$lglvl,7,"GENERAL","Crypt output: $line");
			if ($line =~ m/^DECODED: ([0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz_-]{6,29})/)
			{
				$encrypted_restapi_cryptpwd = $1;
			}
		}
		close CMD_CRYPT;
		LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","Decode REST API password finished");
		if ($encrypted_restapi_cryptpwd eq "n/a")
		{			
			LOG_MSG($log_fh,$vbse,"n",$lglvl,4,"GENERAL","Something went wrong with the REST API pwd decoding. Please check logfile and/or enable debug mode. Not collecting REST API!");
			return -1;
		}
		
		LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST storage collection ref start");
		my $rest_storage_ref_h = REST_GET_STORAGE($restapi_request_usr,$encrypted_restapi_cryptpwd,$rest_protocol,$EXPORTTOOL_IP,"custom",$log_fh,$vbse,$lglvl);
		LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST storage collection ref finished");
		
		LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST version collection ref start");
		my $rest_version = REST_GET_VERSION($restapi_request_usr,$encrypted_restapi_cryptpwd,$rest_protocol,$EXPORTTOOL_IP,"custom",$log_fh,$vbse,$lglvl);
		LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST version collection ref finished");
		
		# Check if REST_GET_STORAGE was successful, if not you get -1 as value
		if ($rest_storage_ref_h == "-1")
		{
			LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"GENERAL","REST StorageDeviceId: $rest_storage_ref_h");
			LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"GENERAL","REST version: $rest_version");
			LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"GENERAL","The value of StorageDeviceId should not be -1. There is some issue. Skipping REST collection.");
		}
		else
		{
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST StorageDeviceId: $rest_storage_ref_h->{storageDeviceId}");
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST version: $rest_version");
			
			my $rest_sessiontoken = "n/a";
			my $rest_sessionid = "n/a";
			
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST session creation start");
			($rest_sessiontoken,$rest_sessionid) = REST_CREATE_SESSION($restapi_request_usr,$encrypted_restapi_cryptpwd,$rest_protocol,$EXPORTTOOL_IP,$rest_storage_ref_h->{storageDeviceId},"custom",$log_fh,$vbse,$lglvl);
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST session creation finished");
			
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST Session token: $rest_sessiontoken");
			LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST Session ID: $rest_sessionid");
		
			unless (($rest_sessiontoken eq "-1" ) or ($rest_sessiontoken eq "n/a" ))
			{ 
				LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST storage collection start");
				my $filename1 = "${rest_dir}\\${serial}_rest_storage.txt";
				LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST creating file: $filename1");
				open (FH_REST_GET_STORAGE, ">", $filename1) or warn "Cannot create $filename1: $!\n";
				print FH_REST_GET_STORAGE REST_GET_STORAGE($restapi_request_usr,$encrypted_restapi_cryptpwd,$rest_protocol,$EXPORTTOOL_IP,"raw",$log_fh,$vbse,$lglvl);
				close FH_REST_GET_STORAGE;
				LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST close file: $filename1");
				LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST storage collection finished");
				
				LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST version collection start");
				my $filename2 = "${rest_dir}\\${serial}_rest_version.txt";
				LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST creating file: $filename2");
				open (FH_REST_GET_VERSION, ">", $filename2) or warn "Cannot create $filename2: $!\n";
				print FH_REST_GET_VERSION REST_GET_VERSION($restapi_request_usr,$encrypted_restapi_cryptpwd,$rest_protocol,$EXPORTTOOL_IP,"raw",$log_fh,$vbse,$lglvl);
				close FH_REST_GET_VERSION;
				LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST close file: $filename2");
				LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST version collection finished");
				
				LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST port collection ref start");
				my $rest_port_ref_h = REST_GET_PORTS($rest_sessiontoken,$rest_protocol,$EXPORTTOOL_IP,$rest_storage_ref_h->{storageDeviceId},"custom",$log_fh,$vbse,$lglvl);
				LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST port collection ref finished");
				
				LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST port collection start");
				my $filename3 = "${rest_dir}\\${serial}_rest_ports.txt";
				LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST creating file: $filename3");
				open (FH_REST_GET_PORTS, ">", $filename3) or warn "Cannot create $filename3: $!\n";
				print FH_REST_GET_PORTS REST_GET_PORTS($rest_sessiontoken,$rest_protocol,$EXPORTTOOL_IP,$rest_storage_ref_h->{storageDeviceId},"raw",$log_fh,$vbse,$lglvl);
				close FH_REST_GET_PORTS;
				LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST close file: $filename3");
				LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST port collection finished");
			
				LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST paritiy group collection start");
				my $filename4 = "${rest_dir}\\${serial}_rest_parity.txt";
				LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST creating file: $filename4");
				open (FH_REST_GET_PARITYGROUPS, ">", $filename4) or warn "Cannot create $filename4: $!\n";
				print FH_REST_GET_PARITYGROUPS REST_GET_PARITYGROUPS($rest_sessiontoken,$rest_protocol,$EXPORTTOOL_IP,$rest_storage_ref_h->{storageDeviceId},"raw",$log_fh,$vbse,$lglvl);
				close FH_REST_GET_PARITYGROUPS;
				LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST close file: $filename4");
				LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST paritiy group collection finished");
				
				LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST pool collection start");
				my $filename5 = "${rest_dir}\\${serial}_rest_pool.txt";
				LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST creating file: $filename5");
				open (FH_REST_GET_POOLS, ">", $filename5) or warn "Cannot create $filename5: $!\n";
				print FH_REST_GET_POOLS REST_GET_POOLS($rest_sessiontoken,$rest_protocol,$EXPORTTOOL_IP,$rest_storage_ref_h->{storageDeviceId},"raw",$log_fh,$vbse,$lglvl);
				close FH_REST_GET_POOLS;
				LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST close file: $filename5");
				LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST pool collection finished");
				
				LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST pool fmc collection start");
				my $filename6 = "${rest_dir}\\${serial}_rest_pool_fmc.txt";
				LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST creating file: $filename6");
				open (FH_REST_GET_POOLS_FMC, ">", $filename6) or warn "Cannot create $filename6: $!\n";
				print FH_REST_GET_POOLS_FMC REST_GET_POOLS_FMC($rest_sessiontoken,$rest_protocol,$EXPORTTOOL_IP,$rest_storage_ref_h->{storageDeviceId},"raw",$log_fh,$vbse,$lglvl);
				close FH_REST_GET_POOLS_FMC;
				LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST close file: $filename6");
				LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST pool fmc collection finished");
								
				LOG_MSG($log_fh,$vbse,"v",$lglvl,7,"GENERAL","REST checking collection level: $rest_level");	
				
				if (($rest_level eq 0) or ($rest_level eq 1))
				{
				
					my $rest_hostgroup_ref_h;
					my $rest_hostwwns_ref_h;
					for my $rest_portid ( keys %{ $rest_port_ref_h } ) 
					{
						
						LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST host group collection ref start");
						$rest_hostgroup_ref_h = REST_GET_HOSTGROUPS($rest_sessiontoken,$rest_protocol,$EXPORTTOOL_IP,$rest_storage_ref_h->{storageDeviceId},$rest_portid,"custom",$log_fh,$vbse,$lglvl);
						LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST host group collection ref finished");
						
						LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST host group collection start");
						my $filename6 = "${rest_dir}\\${serial}_rest_hostgroups_${rest_portid}.txt";
						LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST creating file: $filename6");
						open (FH_REST_GET_HOSTGROUPS, ">", $filename6) or warn "Cannot create $filename6: $!\n";
						print FH_REST_GET_HOSTGROUPS REST_GET_HOSTGROUPS($rest_sessiontoken,$rest_protocol,$EXPORTTOOL_IP,$rest_storage_ref_h->{storageDeviceId},$rest_portid,"raw",$log_fh,$vbse,$lglvl);
						close FH_REST_GET_HOSTGROUPS;
						LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST close file: $filename6");
						LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST host group collection finished");
					
						LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST host wwns collection start");
						LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST lu path collection start");
						foreach my $hg_key (keys %{ $rest_hostgroup_ref_h } ) 
						{
							my $rest_hostGroupNumber=$rest_hostgroup_ref_h->{$hg_key}{hostGroupNumber};
							
							my $hg_key_file = $hg_key;
							$hg_key_file =~ s/,/-/g;
							
							my $filename7 = "${rest_dir}\\${serial}_rest_hostwwns_${hg_key_file}.txt";
							LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST creating file: $filename7");
							open (FH_REST_GET_HOSTWWNS, ">", $filename7) or warn "Cannot create $filename7: $!\n";
							print FH_REST_GET_HOSTWWNS REST_GET_HOSTWWNS($rest_sessiontoken,$rest_protocol,$EXPORTTOOL_IP,$rest_storage_ref_h->{storageDeviceId},$rest_portid,$rest_hostGroupNumber,"raw",$log_fh,$vbse,$lglvl);
							close FH_REST_GET_HOSTWWNS;
							LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST close file: $filename7");
							
							my $filename8 = "${rest_dir}\\${serial}_rest_lupath_${hg_key_file}.txt";
							LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST creating file: $filename8");
							open (FH_REST_GET_LUPATH, ">", $filename8) or warn "Cannot create $filename8: $!\n";
							print FH_REST_GET_LUPATH REST_GET_LUPATH($rest_sessiontoken,$rest_protocol,$EXPORTTOOL_IP,$rest_storage_ref_h->{storageDeviceId},$rest_portid,$rest_hostGroupNumber,"raw",$log_fh,$vbse,$lglvl);
							close FH_REST_GET_LUPATH;
							LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST close file: $filename8");
						}
						LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST host wwns collection finished");
						LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST lu path collection finished");
					}
					

					if ($rest_level eq 0)
					{
						my $counter = 16384;
						LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST collection level: $rest_level");
						LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST ldev collection start");
						LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST get ldevs counter: $counter");
						LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST get ldevs system: $flag_system");
						LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST get ldevs maximum ldevs: $raid_amount_ldevs{$flag_system}");
						for (my $ldevhead=0; $ldevhead <= $raid_amount_ldevs{$flag_system}; $ldevhead = $ldevhead + $counter)
						{
							my $fromldev = sprintf ("%06d",$ldevhead);
							my $toldev = sprintf ("%06d",$ldevhead + $counter - 1);
							my $filename9 = "${rest_dir}\\${serial}_rest_ldevs_${fromldev}_${toldev}.txt";
							
							LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST get ldev from ${fromldev} to ${toldev}");
							LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST creating file: $filename9");
							open (FH_REST_GET_LDEVS, ">", $filename9) or warn "Cannot create $filename9: $!\n";
							print FH_REST_GET_LDEVS REST_GET_LDEVS($rest_sessiontoken,$rest_protocol,$EXPORTTOOL_IP,$rest_storage_ref_h->{storageDeviceId},$ldevhead,$counter,"raw",$log_fh,$vbse,$lglvl);
							close FH_REST_GET_LDEVS;
							LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST close file: $filename9");
						}
						LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST ldev collection finished");
					}
				
				}
				
				LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST session discard start");
				REST_DISCARD_SESSION($rest_sessiontoken,$rest_sessionid,$rest_protocol,$EXPORTTOOL_IP,$rest_storage_ref_h->{storageDeviceId},"raw",$log_fh,$vbse,$lglvl);
				LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST session discard finished");
			}
		}
	}
	else
	{
		LOG_MSG($log_fh,$vbse,"n",$lglvl,4,"GENERAL","Currently only windows for G1500/G1000/G800/G600/G400/G200 is supported for REST API collection therefore REST API information will NOT get collected.");
	}
	
	LOG_MSG($log_fh,$vbse,"n",$lglvl,5,"GENERAL","REST finished");
}

### normal sub routines ###

sub DELETE_OLD_LOGS
{
	my $par_logkeepdays = $_[0];
	my $par_SCRIPTLOG_DEL_PATH = $_[1];
	my $par_log_fh = $_[2];
	my $par_vbse = $_[3];
	my $par_lglvl = $_[4];

	
	my $now = time();       # get current time
	my $AGE = 60*60*24*$par_logkeepdays;  # convert $par_logkeepdays days into seconds
	
	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"DELOLDLOGS","Start deleting log files older than $par_logkeepdays days");
	
	my @fileliste_logs = <${par_SCRIPTLOG_DEL_PATH}>;
	
	foreach my $filename (@fileliste_logs) 
	{
		# If modified time of the file is older then $AGE = days in seconds
		if ($now-(stat $filename)[9] > $AGE)
		{
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"DELOLDLOGS","Found the following log file to delete: $filename");
			unlink $filename or warn LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"DELOLDLOGS","Could not delete $filename: $!");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"DELOLDLOGS","Deleted $filename");
		}
	}
	
	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"DELOLDLOGS","Deleting log files finished");

}

sub DELETE_OLD_SAVES
{
	my $par_keepdays = $_[0];
	my $par_ARCHIVE_PREFIX = $_[1];
	my $par_ARCHIVE_SUFFIX = $_[2];
	my $par_log_fh = $_[3];
	my $par_vbse = $_[4];
	my $par_lglvl = $_[5];

	my $now = time();       # get current time
	my $AGE = 60*60*24*$par_keepdays;  # convert $par_keepdays days into seconds

	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"DELOLD","Start deleting files older than $par_keepdays days");

	my @fileliste = <${par_ARCHIVE_PREFIX}*${par_ARCHIVE_SUFFIX}>;
	foreach my $filename (@fileliste) 
	{
		# If modified time of the file is older then $AGE = days in seconds
		if ($now-(stat $filename)[9] > $AGE)
		{
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"DELOLD","Found the following file to delete: $filename");
			unlink $filename or warn LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"DELOLD","Could not delete $filename: $!");
		}
	}
	
	#
	# !!! THIS IS RECURSIVE FOR SUBDIRS, but so everything gets deleted  !!! I keep it just in case.
	#
	#my @all_archiv_files;
        #find(\&push_archiv_files, ($par_ARCHIVE_DIR));
        #sub push_archiv_files
        #{
		# Check if $_ is a file. 	
		# Important : Check (-f) $File::Find::name will not work as find() is doing a chdir.
		# $_ and $File::Find::name behave differntly. See http://search.cpan.org/~rjbs/perl-5.18.0/lib/File/Find.pm
		# Therefore check against $_ but save $File::Find::name
	#	if ( -f $_ )
	#	{
        #       		push(@all_archiv_files,$File::Find::name);
	#	}
        #}

	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"DELOLD","Finished deleting files older than $par_keepdays days");
}

sub GET_OUTPATH_IP_PWD
{
	my $par_serial = $_[0];
	my $par_COMMAND_TXT = $_[1];
	my $par_log_fh = $_[2];
	my $par_vbse = $_[3];
	my $par_lglvl = $_[4];
	my $sub_outpath = "n/a";
	my $sub_ip = "n/a";
	my $sub_user = "n/a";
	my $sub_pwd = "n/a";
	my $sub_gum = "n/a";
	
	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"GETCMDTXT","Open file \"$par_COMMAND_TXT\"");
	open(IN,"< $par_COMMAND_TXT") or die LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,3,"GETCMDTXT"," Couldn't open $par_COMMAND_TXT: $!\n");
	while (my $line = <IN>)
	{
		if ($line =~ m/^outpath\s+\"(.*\w+)\"\s*\;/)
		#if ($line =~ m/^outpath\s+(out${par_serial})\s*\;/)
		{
			$sub_outpath = $1;
			# Unquote \ to make C:\\...\\.... to C:\...\  
			$sub_outpath =~ s/\\\\/\\/g;
		}
		
		# Just in case if outpath gets changes manual
		if ($line =~ m/^outpath\s+(.*\w+)\s*\;/)
		{
			$sub_outpath = $1;
		}
		
		if ($line =~ m/^svpip\s+(\d+.\d+.\d+.\d+)\s*\;/)
		{
			$sub_ip = $1;
		}
		
		#  G800/G600/G400/G200
		if ($line =~ m/^ip\s+(\d+.\d+.\d+.\d+)\s*\;/)
		{
			$sub_ip = $1;
		}
		
		#  G800/G600/G400/G200
		if ($line =~ m/gum\s+(\d+.\d+.\d+.\d+)\s+(\d+.\d+.\d+.\d+)\s+/)
		{
			$sub_gum = "$1\;$2";
		}
		
		if ($line =~ m/^login\s+"{0,1}([-\w]+)"{0,1}\s+"{0,1}([-\w]+)"{0,1}\s*\;/)
		{
			$sub_user = $1;
			$sub_pwd = $2;
		}
	}
	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"GETCMDTXT","Outpath = \"$sub_outpath\"");
	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"GETCMDTXT","SVPIP = \"$sub_ip\"");
	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"GETCMDTXT","GUM IPs = \"$sub_gum\"");
	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"GETCMDTXT","User = \"$sub_user\"");	
	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"GETCMDTXT","PWD = ***");
	LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,7,"GETCMDTXT","PWD = \"$sub_pwd\"");	
	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"GETCMDTXT","Close file \"$par_COMMAND_TXT\"");
	close IN;
	
	if (($sub_user eq "n/a") or ($sub_ip eq "n/a"))
	{
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,3,"GETCMDTXT","Could not get user and/or password from $par_COMMAND_TXT");
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,3,"GETCMDTXT","Maybe it does not meet the allowed characters which are a-z,A-Z,0-9,-");
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,3,"GENERAL","Finishing Script $0");
		LOG_MSG_CLOSE($log_fh);
		exit;
	}
	
	return ($sub_outpath,$sub_ip,$sub_user,$sub_pwd,$sub_gum);
}

sub GET_SHORTRANGE
{
	my $par_EXPORTTOOL_LOG = $_[0];
	my $par_log_fh = $_[1];
	my $par_vbse = $_[2];
	my $par_lglvl = $_[3];

	# Get latest modified log file in the directory. Assuming this is correct one ...
	my @logfile_list = <$par_EXPORTTOOL_LOG/export*.log>;
	@logfile_list = sort { (stat ($a))[9] <=> (stat ($b))[9]  } @logfile_list; # Schwartzian transform to sort old to young
	my $latest_logfile = "$logfile_list[-1]";
	unless ( -f $latest_logfile ) { LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,3,"GETSHORT","No logfile found: \"$latest_logfile\""); exit;}
	
	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"GETSHORT","Latest logfile: \"$latest_logfile\"");

	# Get shortrange of export tool
	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"GETSHORT","Open file \"$latest_logfile\" to get shortrange");
	my $shortrange_time;
	open(IN,"< $latest_logfile") or die LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,3,"GETSHORT","Couldn't open $latest_logfile: $!\n");
	while (my $line = <IN>)
	{
		#regex to get shortrange date out of "2013/07/05 11:39:36     shortrange [2013/07/05 12:24:00]:[2013/07/05 12:26:00], count = 2"
		# that it can be tranforded in format VSP_85070_2013-02-11_0259-2013-02-12_0259
		my $regex = '\s+\[(\d{4}/\d{2}/\d{2})\s+(\d{2}:\d{2}):\d{2}\]:\[(\d{4}/\d{2}/\d{2})\s+(\d{2}:\d{2}):';
		#if ($line =~ m#shortrange${regex}#)
		if ($line =~ m#range${regex}#)
		{
			$shortrange_time = "${1}_${2}-${3}_${4}";
			$shortrange_time =~ s/\//-/g;
			$shortrange_time =~ s/://g;
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"GETSHORT","Shortrange: \"$shortrange_time\"");
		}
	}
	close IN;
	return $shortrange_time;
}

sub DEL_EXPORTTOOL
{
	my $par_EXPORTTOOL_DEL = $_[0];
	my $par_log_fh = $_[1];
	my $par_vbse = $_[2];
	my $par_lglvl = $_[3];

	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"EXPORTDEL","Starting DEL Exporttool");
	open (CMD, "$par_EXPORTTOOL_DEL 2>&1|") or die LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,3,"EXPORTDEL","Can't run program: $!\n");
	while (my $line = <CMD>)
	{
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"EXPORTDEL","\t$line");
	}
	close CMD;
	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"EXPORTDEL","DEL Exporttool finished");
}

sub RUN_EXPORTTOOL
{
	my $par_EXPORTTOOL_RUN = $_[0];
	my $par_log_fh = $_[1];
	my $par_vbse = $_[2];
	my $par_lglvl = $_[3];

	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"EXPORT","Starting Exporttool");
	open (CMD, "$par_EXPORTTOOL_RUN 2>&1|") or die LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,3,"EXPORT","Can't run program: $!\n");
	while (my $line = <CMD>)
	{
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"EXPORT","\t$line");
	}
	close CMD;
	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"EXPORT","Exporttool finished");
}

sub DO_ARCHIV
{
	my @par_backupdirs = @{ $_[0] };
	my @par_backupdirs_delete_after_run = @{ $_[1] };
	my $par_archive_file = $_[2];
	my $par_flag_zip  = $_[3];
	my $par_ZIP_PATH = $_[4];
	my $par_log_fh = $_[5];
	my $par_vbse = $_[6];
	my $par_lglvl = $_[7];

	my @all_files;
	
	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"ARCHIVE","Files/dirs to archive: @par_backupdirs");
	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"ARCHIVE","Files/dirs to delete after run: @par_backupdirs_delete_after_run");
	
	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"ARCHIVE","Checking now if files/dirs exist. If not, it gets skipped for archiv and delete");
	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"ARCHIVE","Checking files/dirs to archive");
	my $i=0;
	foreach my $path (@par_backupdirs)
	{
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"ARCHIVE","Checking $path");
		unless (-e $path)
		{
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"ARCHIVE","$path does not exsist. Skipping from list.");
			splice(@par_backupdirs,$i,1)
		}
		$i++;
	}
	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"ARCHIVE","Checking files/dirs to delete after archive");
	
	$i=0;
	foreach my $path (@par_backupdirs_delete_after_run)
	{
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"ARCHIVE","Checking $path");
		unless (-e $path)
		{
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"ARCHIVE","$path does not exsist. Skipping from list.");
			splice(@par_backupdirs_delete_after_run,$i,1)
		}
		$i++;
	}
	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"ARCHIVE","Checking finished");

		
	# Get all export and log files and command file to tar it.
	local *push_files = sub { push(@all_files,$File::Find::name); LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"ARCHIVE","ARCHIVING $File::Find::name\n"); }; # nested sub needed as use warnings complains if not
	# this from module file::find
	find(\&push_files, @par_backupdirs);
		
	#IMPORTANT: ppmonitor is checking this line in the log file!
	my $abs_path_archiv_file = File::Spec->rel2abs($par_archive_file);
	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"ARCHIVE","Start Archiving to \"$abs_path_archiv_file\"");
	
	# Check if 7zip or built-in zip
	if ($par_flag_zip)
	{
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"ARCHIVE","Flag zip found. Using $par_ZIP_PATH");
		foreach my $filetozip (@par_backupdirs)
		{
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"ARCHIVE","Doing $par_ZIP_PATH a -tzip $par_archive_file $filetozip");
			open (CMD_ZIP, "$par_ZIP_PATH a -tzip $par_archive_file $filetozip 2>&1|") or die LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,3,"ARCHIVE"," Can't run program: $!\n");
			while (my $line = <CMD_ZIP>)
			{
				LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"ARCHIVE","$line");
			}
			close CMD_ZIP;
		}
	}

	else 
	{
		my $tar = Archive::Tar->new();
		$tar->add_files(@all_files) or die LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,3,"ARCHIVE","Can't run program: $!\n");
		# This is needed for older Perl version => otherwise "COMPRESS_GZIP" Bareword warning
		no strict;
		$tar->write( "$par_archive_file", COMPRESS_GZIP) or die LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,3,"ARCHIVE","Can't run program: $!\n");
	}
	
	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"ARCHIVE","Archiving finished");

	# Delete archived files
	if ( -f $par_archive_file )
	{
		foreach my $dirtodel (@par_backupdirs_delete_after_run)
		{
				LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"ARCHIVE","Start deleting directory \"$dirtodel\"");
				rmtree ($dirtodel) or die LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,3,"ARCHIVE","Can't delete dirs $dirtodel: $!\n");
				LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"ARCHIVE","Deleting directory finished");
		}
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"ARCHIVE","Deleting directories finished");
	}
	else
	{
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,3,"ARCHIVE","No archived file found: $par_archive_file. NOT deleting files");
	}
}

sub USAGE
{
	my ($message)=<<'MESSAGE';

NAME ppcollect.pl

SYNOPSIS
     ppcollect.pl -c <config file>
     ppcollect.pl [-h] [-v]

DESCRIPTION
     This script runs the export tool of the VSP/USPV/HUS-VM/G1500/G1000/G800/G600/G400/G200, zip or tgz's the files and moves it to a save directory.
	 It can also collect several configuration information with raidcom and raidscan.

REQUIREMENT
     > Install perl (tested with 5.10 and above)
     > Install java version <TBD> (latest version should work)
     > Define java path in configuration file!! Default is just "java". (Take care that there are no blank in the path especially in windows)
     > Copy export tool java files (according to the Microcode of the VSP/USPV/HUS-VM/G1500/G1000/G800/G600/G400/G200) into lib direcory
     > Edit export tool command file
        > IP adress / user / password
        > outpath must be set to: "outpath d:\path\to\scripthome\data\out<Serial>"
        > command file must be named: "<storage system>_command_<Serial>.txt"
        > command file must be located in the same directory as ppcollect.pl
     > Run this script with appropriate privileges! If you are not sure use root or administrator.

OPTIONS

	-c <configuration file>
		Path to configuration file
		Please see the manual (http://loop.hds.com/docs/DOC-45708) for detailed information about configuration settings.

	-h
		Show help message.
			 
	-v
		Displays the version of the script	

FILES
	$SCRIPT_HOME/bin/ppcollect.pl
	$SCRIPT_HOME/lib/modules/* (e.g. HCiHpp_RAIDCOM, Config::Tiny)
	$SCRIPT_HOME/etc/* (e.g. pp_serial.conf, <storage system>_command_<Serial>.txt)
	$SCRIPT_HOME/bin/7-ZipPortable/*

EXAMPLES
	ppcollect.pl -h
	ppcollect.pl -v
	ppcollect.pl -c pp_123456.conf

MESSAGE

	print "Version:${VERSION}\n$message";
}