#!/usr/bin/perl
# Author: Maik Ernst
# History:
# 2015-06-17 - initial script
#
# LICENSE
# Hitachi, Ltd. All rights reserved
# Hitachi Data Systems products and services can be ordered only under the terms and conditions
# of the applicable Hitachi Data Systems agreements.
#
# NAME
# ppsetup.pl
# Hitachi Confoederatio Helvetica performance package transfer tool
# Part of HiCHperfpkg
# 
#
# SYNOPSIS
# see usage function below
#
# DESCRIPTION
#
# OPTIONS 
# see "usage" function below.
#
# NOTES
#
# AUTHOR
#   Maik Ernst (maik.ernst@hds.com) - June 2015
#
# MODIFICATIONS
# 2015-06-17 - v1.0.0	- initial script
# 2015-07-02 - v1.0.1	- Fixing Gx00 bug (ip instead of svpip in cmd.txt and other order cmd.txt)
# 2015-07-02 - v1.0.1	  Don't allow "." and " " in script path. Disable raidcom for Gx00
# 2015-07-03 - v1.1.0	- Check for controller IPs on Gx00 (GUM) and allow raidcom.
# 2015-07-04 - v1.1.1	- Add " to command.txt outpath
# 2015-07-24 - v1.1.2	- Changing default from "ppmonitor_switch = on" to "ppmonitor_switch = off"
# 2016-03-25 - v1.2.0	- LOG->autoflush(1), format change, adding CFGFILE pptransfer_proxyoptions
#						- transfer_https_pxy_user and transfer_https_pxy_pwd check (@ and & not allowed)
#						- adding ppcollect_keepdays, pptransfer_keepdays, ppmonitor_keepdays
#						- adding DO_UPGRADE_CONF (change config files)
# 2016-04-02 - v1.2.1	- Change screenoutput 7.3.0 to 7.3.x
# 2016-11-19 - v1.3.0	- adding logging module HiCH_LOG, change logging call to have more parameters (e.g. verbose)
#						- adding ppcollect_memoryxmx to setup inital and upgrade
#						- fixing bug in upgrade procedure (range instead of shorange)
#						- change shortrange default from -1200 to -1230:
#						- adding G1500 support 
#						- adding conf file updates 
#						- adding cmdtxt updates 
#						- Change screenoutput from 7.3.x to 8.0.x
# 2016-11-24 - v1.3.1   - adding restapi user and rest api password to initial and upgrade procedure
# 2016-11-26 - v1.3.2   - adding restapi password encryption

use strict;
use warnings;
use POSIX qw(strftime); # Get time on LOG_MSG
use Cwd 'abs_path'; # Get absolute path
use File::Basename; # For function file parse & dirname
use Config;
use Sys::Hostname; # To get hostname
use Getopt::Long qw(:config no_ignore_case bundling); # Get options /  qw(:config bundling no_ignore_case) makes getopt case sensitive
use IO::Handle; # required for autoflush

# Use custom/local perl modules lib. FindBin is needed to find the module when running the script from relativ/absolute paths.
use FindBin qw($Bin);
use lib $Bin;
use lib "$Bin\\..\\lib\\modules";
use Config::Tiny;
use HiCH_LOG;

### main ### DO NOT CHANGE ANYTHING IF DO NOT KNOW WHAT YOU ARE DOING!
my $VERSION = "1.3.2";
my $conf_prefix = "pp";
my $conf_suffix = "conf";
my $cmd_middle_name = "command";
my $cmd_suffix = "txt";

my $vbse;
my $lglvl = "normal";
my $log_fh;

# Error handling. Change function that a message gets written to log file before dieing.
INIT {$SIG{__DIE__}=sub {LOG_MSG($log_fh,$vbse,"n",$lglvl,3,"GENERAL","Script died: $_[0]") and LOG_MSG_CLOSE($log_fh);}}
INIT {$SIG{__WARN__}=sub {LOG_MSG($log_fh,$vbse,"n",$lglvl,4,"GENERAL","Script warning: $_[0]")}}

# Check Flags
my $flag_help;
my $flag_version;

GetOptions (
	'h|help' => \$flag_help,
	'v|verbose' => \$vbse,
	'V|VER' => \$flag_version,
           ) or die USAGE();
		   
		   # Change to directory of the running script 
# abs_path for $0 or __FILE__ , 
my ( $script_name, $script_path, $script_suffix ) = fileparse( abs_path($0), qr{\.[^.]*$} );
my $script_dirname = dirname (abs_path($0));
# Change slashes to backslashes
$script_dirname =~ s/\//\\/g;
chdir ($script_dirname);
my $script_rootdir =  dirname ($script_dirname);

# Check flags and print usage
if ($flag_version)
{
	print "Version: $VERSION\n";
	exit;
}

if ($flag_help)
{
	USAGE();
	exit;
}

## main ##
my $scriptdir_etc =  "${script_rootdir}\\etc";
my $scriptdir_bin =  "${script_rootdir}\\bin";
my $scriptdir_data =  "${script_rootdir}\\data";
my $scriptdir_log =  "${script_rootdir}\\log";
my $cmdtxt_outpath = "${scriptdir_data}\\out";
my $script_crypt =  "${script_rootdir}\\lib\\modules\\c.pl";

my $time_scriptstart = strftime("%Y-%m-%d_%H-%M-%S", localtime);
my $HOST = hostname;
my $SCRIPTLOG_FILE = "${scriptdir_log}\\${script_name}_${time_scriptstart}.log";

# Create directories
unless (-d $scriptdir_log) { mkpath($scriptdir_log) or die ("Can't create dir $scriptdir_log: $!\n"); }

# Open script logfile
$log_fh = LOG_MSG_OPEN($SCRIPTLOG_FILE);
LOG_MSG($log_fh,$vbse,"v",$lglvl,5,"GENERAL","Starting Script: \"$0\"");
LOG_MSG($log_fh,$vbse,"v",$lglvl,5,"GENERAL","Version: ${VERSION}");
LOG_MSG($log_fh,$vbse,"v",$lglvl,5,"GENERAL","Logfile: \"$SCRIPTLOG_FILE\"");
LOG_MSG($log_fh,$vbse,"v",$lglvl,5,"GENERAL","Operating system: \"$Config{osname}\"");
LOG_MSG($log_fh,$vbse,"v",$lglvl,5,"GENERAL","Hostname: \"$HOST\"");

print "\n";
LOG_MSG($log_fh,$vbse,"s",$lglvl,5,"GENERAL","## HiCHperfpkg - Performance Package Setup");
LOG_MSG($log_fh,$vbse,"s",$lglvl,5,"GENERAL","   -------------------------------------------------------------------");
LOG_MSG($log_fh,$vbse,"s",$lglvl,5,"GENERAL","   ------------ HiCH - Performance Collection Setup Menu -------------");
LOG_MSG($log_fh,$vbse,"s",$lglvl,5,"GENERAL","   -------------------------------------------------------------------");
print "\n";

# Get user input
my %user_choice = CONFIG_FROM_USER ($log_fh,$vbse,$lglvl);

if (($user_choice{initial} == 0) or ($user_choice{initial} == 1) or ($user_choice{initial} == 2))
{
	# Create config file
	MAKE_CONFIGFILE ($scriptdir_etc,$conf_prefix,$conf_suffix,\%user_choice,$script_crypt,$log_fh,$vbse,$lglvl);
	# Create cmd txt if required
	if (($user_choice{initial} == 0) or ($user_choice{initial} == 1))
	{
		MAKE_COMMANDTXT ($scriptdir_etc,\%user_choice,$cmdtxt_outpath,$cmd_middle_name,$cmd_suffix,$log_fh,$vbse,$lglvl);
	}
	# Create bat files
	MAKE_BATFILE ($scriptdir_etc,$scriptdir_bin,$conf_prefix,$conf_suffix,\%user_choice,$log_fh,$vbse,$lglvl);
}

if ($user_choice{initial} == 10)
{
	DO_UPGRADE_CONF($scriptdir_etc,$conf_prefix,$conf_suffix,$script_crypt,$log_fh,$vbse,$lglvl);
	DO_UPGRADE_COMMANDTXT($scriptdir_etc,$cmd_middle_name,$cmd_suffix,$log_fh,$vbse,$lglvl);
}

print "\n";
LOG_MSG($log_fh,$vbse,"s",$lglvl,5,"GENERAL","   -------------------------------------------------------------------");
LOG_MSG($log_fh,$vbse,"s",$lglvl,5,"GENERAL","   Setup successfully completed.");
SAY_BYEBYE($log_fh,$vbse,$lglvl);

		   
### sub routines ###

sub DO_UPGRADE_COMMANDTXT
{
	my $par_scriptdir_etc = $_[0]; 
	my $par_cmd_middle_name = $_[1];
	my $par_cmd_suffix = $_[2];
	my $par_log_fh = $_[3];
	my $par_vbse = $_[4];
	my $par_lglvl = $_[5];
	
	
	LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CMDTXT","   -------------------------------------------------------------------");
	LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CMDTXT","   Starting the upgrade procedure command txt...");
	print "\n";

	my @filelist_cmdtxt = <${par_scriptdir_etc}\\*_${par_cmd_middle_name}_*.${par_cmd_suffix}>;
	
	if (!@filelist_cmdtxt)
	{
		LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CMDTXT","   No configuration files found. Update procedure not required.");
		return;
	}
	
	foreach my $filename (@filelist_cmdtxt) 
	{
		my @file_array;
		my %bad_value_seen = (); # lookup table
		
		print "   - Config file found: $filename\n";
		
		open(IN,"<", $filename) or die LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,3,"UPGRADE_CMDTXT","Couldn't open $filename: $!");
		chomp (@file_array = (<IN>));
		close(IN);
		
		
		# check if parameter exists in cmd file - if so replace it with new parameter
		my $i=0;
		foreach my $line (@file_array) 
		{
			if ($line =~ /^shortrange\s+-1200:/) 
			{
				$bad_value_seen{"shortrange1200"} = 1;
				splice(@file_array,$i,1,"shortrange  -1230:                   ; Number of monitored CUs is 64 or less");
				LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CMDTXT","     Change line: $file_array[$i]");
				LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CMDTXT","     To line: shortrange  -1230:                   ; Number of monitored CUs is 64 or less");
			}
			elsif ($line =~ /^range\s+-1200:/) 
			{
				$bad_value_seen{"range1200"} = 1;
				splice(@file_array,$i,1,"range  -1230:                   ; Number of monitored CUs is 64 or less");
				LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CMDTXT","     Change line: $file_array[$i]");
				LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CMDTXT","     To line: range  -1230:                   ; Number of monitored CUs is 64 or less");
			}
			$i++;
		}
		
		
		# check if parameters already exist 
		if ( ($bad_value_seen{"shortrange1200"}) or ($bad_value_seen{"range1200"}) )
		{
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CMDTXT","     Deleting $filename");
			unlink $filename or die LOG_MSG($par_lglvl,3,"UPGRADE_CMDTXT","Could not delete $filename: $!");
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CMDTXT","     Deleted $filename");
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CMDTXT","     Recreate $filename");
			open(CMDTXT,">", $filename) or die LOG_MSG ($par_vbse,"s",$par_lglvl,3,"UPGRADE_CMDTXT","     Can't open $filename: $!\n");
			foreach my $line (@file_array) 
			{
				print CMDTXT "$line\n";
			}
			close CMDTXT;
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CMDTXT","     Recreation $filename finished");
		}
		else
		{
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CMDTXT","     Already up to date. Nothing to do.")
		}
		print "\n";
		
	}
}

sub DO_UPGRADE_CONF
{
	my $par_scriptdir_etc = $_[0]; 
	my $par_conf_prefix = $_[1];
	my $par_conf_suffix = $_[2];
	my $par_script_crypt = $_[3];
	my $par_log_fh = $_[4];
	my $par_vbse = $_[5];
	my $par_lglvl = $_[6];
	
	LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CONF","   -------------------------------------------------------------------");
	LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CONF","   Starting the upgrade procedure conf file...");
	print "\n";

	my @filelist_conf = <${par_scriptdir_etc}\\${par_conf_prefix}_*.${par_conf_suffix}>;
	
	if (!@filelist_conf)
	{
		LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CONF","   No configuration files found. Update procedure not required.");
		return;
	}
	
	foreach my $filename (@filelist_conf) 
	{
		my @file_array;
		my %value_seen = (); # lookup table
		my %value_seen_del_chg = (); # lookup table
		
		print "   - Config file found: $filename\n";
		
		open(IN,"<",$filename) or die LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,3,"UPGRADE_CONF","Couldn't open $filename: $!");
		chomp (@file_array = (<IN>));
		close(IN);
		
		# check if parameter exists in conf file
		foreach my $line (@file_array) 
		{
			if ($line =~ /ppcollect_logkeepdays/) 
			{
				$value_seen{"ppcollect_logkeepdays"} = 1;
			}
			elsif ($line =~ /pptransfer_logkeepdays/) 
			{
				$value_seen{"pptransfer_logkeepdays"} = 1;
			}
			elsif ($line =~ /ppmonitor_logkeepdays/) 
			{
				$value_seen{"ppmonitor_logkeepdays"} = 1;
			}
			elsif ($line =~ /pptransfer_proxyoptions/) 
			{
				$value_seen{"pptransfer_proxyoptions"} = 1;
			}
			# added in version 8.0.0
			elsif ($line =~ /ppcollect_memoryxmx/) 
			{
				$value_seen{"ppcollect_memoryxmx"} = 1;
			}
			# added in version 8.0.0
			elsif ($line =~ /\[restapi\]/) 
			{
				$value_seen{"[restapi]"} = 1;
			}
			# added in version 8.0.0
			elsif ($line =~ /ppcollect_collection/) 
			{
				$value_seen{"ppcollect_collection"} = 1;
			}
			elsif ($line =~ /# on, off, raidcomonly/) 
			{
				$value_seen_del_chg{"# on, off, raidcomonly"} = 1;
			}
			# added in version 8.0.0
			elsif ($line =~ /ppcollect_ccicollection\s+/) 
			{
				$value_seen_del_chg{"ppcollect_ccicollection"} = 1;
			}
			# added in version 8.0.0
			elsif ($line =~ /# \[G1000\/G800\/G600\/G400\/G200\/VSP\/HUSVM\/USPV\]/) 
			{
				$value_seen_del_chg{"# [G1000/G800/G600/G400/G200/VSP/HUSVM/USPV]"} = 1;
			}
			# added in version 8.0.0
			elsif ($line =~ /pptransfer_maxentries = 360/)
			{
				$value_seen_del_chg{"pptransfer_maxentries = 360"} = 1;
			}
		}
		
		# check/add ppcollect_logkeepdays
		# if not seen than add.
		unless ($value_seen{"ppcollect_logkeepdays"}) 
		{
			my $i=0;
			foreach my $line (@file_array) 
			{
				$i++;
				if ($line =~ /ppcollect_keepdays/)
				{
					last;
				}
			}
			# add parameter at $i
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CONF","     Add line: ppcollect_logkeepdays = 60");
			splice(@file_array,$i,0,"ppcollect_logkeepdays = 60");
		}
		
		# check/add ppcollect_memoryxmx
		unless ($value_seen{"ppcollect_memoryxmx"}) 
		{
			my $i=0;
			foreach my $line (@file_array) 
			{
				$i++;
				if ($line =~ /ppcollect_logkeepdays/)
				{
					last;
				}
			}
			# add parameter at $i
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CONF","     Add line: ppcollect_memoryxmx = 1073741824");
			splice(@file_array,$i,0,"ppcollect_memoryxmx = 1073741824");
		}
		
		# check/add ppcollect_collection
		unless ($value_seen{"ppcollect_collection"}) 
		{
			my $i=0;
			foreach my $line (@file_array) 
			{
				$i++;
				if ($line =~ /ppcollect_memoryxmx/)
				{
					last;
				}
			}
			# add parameter at $i
			my $string_ppcollect_collection = "# 1=export tool only, 2=raidcom only, 3=REST API only," . "\n"
						 . "# 4=export tool + raidcom, 5=export tool + REST API," . "\n"
						 . "# 6=raidcom + REST API, 7=export tool + raidcom + REST API" . "\n"
						 .  "ppcollect_collection = 7";
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CONF","     Add line: # 1=export tool only, 2=raidcom only,3=REST API only,");
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CONF","     Add line: # 4=export tool + raidcom, 5=export tool + REST API,");
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CONF","     Add line: # 6=raidcom + REST API, 7=export tool + raidcom + REST API");
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CONF","     Add line: ppcollect_collection = 7");
			splice(@file_array,$i,0,$string_ppcollect_collection);
		}

		# check/add [restapi]
		unless ($value_seen{"[restapi]"}) 
		{
			# added in version 8.0.1
			# get rest api user and password
			my $restapi_usr_upgrade = "n/a";
			my $restapi_cryptpwd_upgrade = "n/a";
			my $collect_correct_upgrade = "n/a";
			while (1)
			{
				# get rest api user
				print "\n";
				LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CONF","     IMPORTANT IMPORTANT IMPORTANT"); 
				LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CONF","     Create a NEW USER for REST API collection in the same user group as for export tool & raidcom collection."); 
				LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CONF","     Using the same user for raidcom and REST API will cause errors. Details see install manual (chapter 3.3.2.1)."); 
				print "\n";
				LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CONF","     Enter user name of \"REST API\" that runs data collection"); 
				LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"UPGRADE_CONF","     User name: "); 
				print "     User name: ";
				chomp ($restapi_usr_upgrade = <STDIN>);
				
				# get rest api password
				while (1)
				{
					print "\n";
					LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CONF","     Enter password  of\"REST API\" user"); 
					LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"UPGRADE_CONF","     Password: "); 
					print "     Password: ";
					chomp ($restapi_cryptpwd_upgrade = <STDIN>);
							
					if ($restapi_cryptpwd_upgrade !~ m/^[0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz_-]{6,29}$/)
					{
						LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CONF","     Sorry, \"REST API\" password must have at 6 characters and can only contain these characters \"0-9A-Za-z_-\", try again.");
						print "\n";			
						next;
					}
					else
					{	
						LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"UPGRADE_CONF","     \"$restapi_usr_upgrade\" is valid.");
						last;
					}
				}				
								
				# check settings
				print "\n";
				LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CONF","     Provided settings");
				LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CONF","     - REST API user name  :\"$restapi_usr_upgrade\"");
				LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CONF","     - REST API password   :\"$restapi_cryptpwd_upgrade\"");
				print "\n";
			
				while (1)
				{
					LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"UPGRADE_CONF","     Are these setting correct (y/n): "); 
					print "     Are these setting correct (y/n): ";
					chomp ($collect_correct_upgrade = lc <STDIN>);
						
					if ($collect_correct_upgrade eq "y")
					{
						LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"UPGRADE_CONF","     \"$collect_correct_upgrade\" is valid and approved by user.");
						last;
					}
					if ($collect_correct_upgrade eq "n")
					{
						LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"UPGRADE_CONF","     \"$collect_correct_upgrade\" is valid but not approved by user.");
						LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"UPGRADE_CONF","     Starting again.");
						last;
					}
					else
					{	
						LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"UPGRADE_CONF","     \"$collect_correct_upgrade\" was not a valid choice");
						LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CONF","     ...please answer \"y\" or \"n\"");
						print "\n";			
						next;
					}
				}
			
				if ($collect_correct_upgrade eq "y")
				{
					LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"UPGRADE_CONF","     User choice finished.");				
					last;
				}
				else
				{	
					LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"UPGRADE_CONF","     User choice starting again.");	
					($restapi_usr_upgrade,$restapi_cryptpwd_upgrade,$collect_correct_upgrade) = ("n/a","n/a","n/a","n/a","n/a","n/a") ;		
					next;
				}
			}
				
								
			my $i=0;
			foreach my $line (@file_array) 
			{
				$i++;
				if ($line =~  /ppcollect_horcmdirprefix/)
				{
					last;
				}
			}
			
			
			# Do encoding
			LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"UPGRADE_CONF","     Encode REST API password start");
			my $decrypt_restapi_cryptpwd = "n/a";
			open (CMD_CRYPT, "$par_script_crypt -e $restapi_cryptpwd_upgrade 2>&1|") or warn ("!!! Can't run program: $!\n");
			while (my $line = <CMD_CRYPT>)
			{
				LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,7,"UPGRADE_CONF","     $line");
				if ($line =~ m/^ENCODED: ([0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz_-]{3,50})/)
				{
					$decrypt_restapi_cryptpwd = $1;
				}
			}
			close CMD_CRYPT;
			LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"UPGRADE_CONF","     Encode REST API password finished");
			if ($decrypt_restapi_cryptpwd eq "n/a")
			{			
				LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,4,"UPGRADE_CONF","     Something went wrong with the REST API pwd encryption. Please check ppsetup logfile and/turn on debug log level.");
			}
			
			# add parameter at $i
			my $string_restapi = "\n" . "[restapi]" . "\n"
					 	 . "ppcollect_restapiusr = ${restapi_usr_upgrade}" . "\n"
						 . "# The password must be encrypted. Details see manual.". "\n"
						 . "ppcollect_restapicryptpwd = ${decrypt_restapi_cryptpwd}". "\n"
						 . "# 0=full, 1=partial, 2=minimum" . "\n"
						 . "ppcollect_restapicollectionlevel = 0" . "\n"
						 . "ppcollect_restapidirprefix = rest" . "\n"
						 . "# http, https" . "\n"				 
						 . "ppcollect_restapiprotocol = https";
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CONF","     Add line: [restapi]");
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CONF","     Add line: ppcollect_restapiusr = ${restapi_usr_upgrade}");
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CONF","     Add line: # The password must be encrypted. Details see manual.");
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CONF","     Add line: ppcollect_restapicryptpwd = ${decrypt_restapi_cryptpwd}");
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CONF","     Add line: # 0=full, 1=partial ,2=minimum");
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CONF","     Add line: ppcollect_restapicollectionlevel = 0");
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CONF","     Add line: ppcollect_restapidirprefix = rest");
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CONF","     Add line: # http, https");
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CONF","     Add line: ppcollect_restapiprotocol = https");
			splice(@file_array,$i,0,$string_restapi);
		}
		
		# check/add pptransfer_logkeepdays
		unless ($value_seen{"pptransfer_logkeepdays"}) 
		{
			my $i=0;
			foreach my $line (@file_array) 
			{
				$i++;
				if ($line =~ /pptransfer_sourcedir/)
				{
					last;
				}
			}
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CONF","     Add line: pptransfer_logkeepdays = 60");
			splice(@file_array,$i,0,"pptransfer_logkeepdays = 60");
		}
		
		# check/add ppmonitor_logkeepdays
		unless ($value_seen{"ppmonitor_logkeepdays"}) 
		{
			my $i=0;
			foreach my $line (@file_array) 
			{
				$i++;
				if ($line =~ /ppmonitor_runsperday/)
				{
					last;
				}
			}
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CONF","     Add line: ppmonitor_logkeepdays = 60");
			splice(@file_array,$i,0,"ppmonitor_logkeepdays = 60");
		}
		
		# check/add pptransfer_proxyoptions
		unless ($value_seen{"pptransfer_proxyoptions"}) 
		{
			my $i=0;
			foreach my $line (@file_array) 
			{
				$i++;
				if ($line =~ /pptransfer_proxypwd/)
				{
					last;
				}
			}
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CONF","     Add line: pptransfer_proxyoptions = ");
			splice(@file_array,$i,0,"pptransfer_proxyoptions = ");
		}
		
		# Delete / Change if seen
		if ($value_seen_del_chg{"# on, off, raidcomonly"}) 
		{
			my $i=0;
			foreach my $line (@file_array) 
			{
				if ($line =~ /# on, off, raidcomonly/)
				{
					last;
				}
				# raise index after check, not before like when adding
				$i++;
			}
			# delete parameter at $i
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CONF","     Delete line: $file_array[$i]");
			splice(@file_array,$i,1);
		}
		
		if ($value_seen_del_chg{"ppcollect_ccicollection"}) 
		{
			my $i=0;
			foreach my $line (@file_array) 
			{
				if ($line =~ /ppcollect_ccicollection\s+/)
				{
					last;
				}
				# raise index after check, not before like when adding
				$i++;
			}
			# delete parameter at $i
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CONF","     Delete line: $file_array[$i]");
			splice(@file_array,$i,1);
		}
		
		if ($value_seen_del_chg{"# [G1000/G800/G600/G400/G200/VSP/HUSVM/USPV]"}) 
		{
			my $i=0;
			foreach my $line (@file_array) 
			{
				if ($line =~ /# \[G1000\/G800\/G600\/G400\/G200\/VSP\/HUSVM\/USPV\]/)
				{
					last;
				}
				# raise index after check, not before like when adding
				$i++;
			}
			# delete parameter at $i
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CONF","     Change line: $file_array[$i]");
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CONF","     To line: # [G1500/G1000/G800/G600/G400/G200/VSP/HUSVM/USPV]");
			splice(@file_array,$i,1,"# [G1500/G1000/G800/G600/G400/G200/VSP/HUSVM/USPV]");
		}
		
		if ($value_seen_del_chg{"pptransfer_maxentries = 360"}) 
		{
			my $i=0;
			foreach my $line (@file_array) 
			{
				if ($line =~ /pptransfer_maxentries = 360/)
				{
					last;
				}
				# raise index after check, not before like when adding
				$i++;
			}
			# delete parameter at $i
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CONF","     Change line: $file_array[$i]");
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CONF","     To line: pptransfer_maxentries = 1000");
			splice(@file_array,$i,1,"pptransfer_maxentries = 1000");
		}
	
		# check if parameters already exist 
		if ( ($value_seen{"ppcollect_logkeepdays"}) and ($value_seen{"pptransfer_logkeepdays"}) and ($value_seen{"ppmonitor_logkeepdays"}) 
		and ($value_seen{"pptransfer_proxyoptions"}) and ($value_seen{"ppcollect_memoryxmx"}) and ($value_seen{"ppcollect_collection"})
		and ($value_seen{"[restapi]"}) and (!$value_seen_del_chg{"# on, off, raidcomonly"}) and (!$value_seen_del_chg{"ppcollect_ccicollection"})
		and (!$value_seen_del_chg{"# [G1000/G800/G600/G400/G200/VSP/HUSVM/USPV]"}) and (!$value_seen_del_chg{"pptransfer_maxentries = 360"}) )
		{
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CONF","     Already up to date. Nothing to do.")
		}
		else
		{
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CONF","     Deleting $filename");
			unlink $filename or die LOG_MSG($par_lglvl,3,"UPGRADE_CONF","Could not delete $filename: $!");
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CONF","     Deleted $filename");
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CONF","     Recreate $filename");
			open(CONFFILE,"> $filename") or die LOG_MSG ($par_vbse,"s",$par_lglvl,3,"UPGRADE_CONF","     Can't open $filename: $!\n");
			foreach my $line (@file_array) 
			{
				print CONFFILE "$line\n";
			}
			close CONFFILE;
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"UPGRADE_CONF","     Recreation $filename finished");
		}
		print "\n";
	}
}

sub MAKE_SCHEDTASK
{
	my $par_bat_type = $_[0];
	my $par_scriptdir_etc = $_[1];
	my $par_scriptdir_bin = $_[2];
	my $par_batfile = $_[3];
	my $par_href_user_choice = $_[4];
	my $par_log_fh = $_[5];
	my $par_vbse = $_[6];
	my $par_lglvl = $_[7];
	
	my $xml_file = "${par_scriptdir_etc}\\schtasks_${par_bat_type}.xml";
	my $CMD_schtasks;
	if ($par_bat_type  eq "serial") { $CMD_schtasks = "schtasks /create /XML $xml_file /tn \"HiCHperfpkg $par_href_user_choice->{serial}\""; }
	elsif ($par_bat_type  eq "log") { $CMD_schtasks = "schtasks /create /XML $xml_file /tn \"HiCHperfpkg log\""; }
	elsif ($par_bat_type  eq "vmio") { $CMD_schtasks = "schtasks /create /XML $xml_file /tn \"HiCHperfpkg vmio\""; }
	else { LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,3,"SCHEDTASK","   !! Adding scheduled task for $par_batfile not successfully !! You should never be here."); return;}
	
	LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"SCHEDTASK","   Creating $xml_file.");
	my $minute_range = 59;
	my $random_minute = int(rand($minute_range));
	my $random_minute_long = sprintf("%02d", $random_minute);
	
	LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"SCHEDTASK","   Random minute for schedtask: $random_minute_long");
	
	open(XMLFILE,"> $xml_file") or die LOG_MSG ($par_vbse,"s",$par_lglvl,3,"   SCHEDTASK","Can't open $xml_file: $!\n");
	print XMLFILE "<?xml version=\"1.0\" encoding=\"UTF-16\"?>\n";
	print XMLFILE "<Task version=\"1.2\" xmlns=\"http://schemas.microsoft.com/windows/2004/02/mit/task\">\n";
	print XMLFILE "  <RegistrationInfo>\n";
	print XMLFILE "    <Date>2014-09-03T13:41:51.2597656</Date>\n";
	print XMLFILE "    <Author>Administrator</Author>\n";
	print XMLFILE "  </RegistrationInfo>\n";
	print XMLFILE "  <Triggers>\n";
	if ($par_bat_type  eq "log")
	{
		print XMLFILE "    <CalendarTrigger>\n";
		print XMLFILE "      <StartBoundary>2014-09-03T23:55:00</StartBoundary>\n";
		print XMLFILE "      <ExecutionTimeLimit>PT8H</ExecutionTimeLimit>\n";
		print XMLFILE "      <Enabled>true</Enabled>\n";
		print XMLFILE "      <ScheduleByDay>\n";
		print XMLFILE "        <DaysInterval>1</DaysInterval>\n";
		print XMLFILE "      </ScheduleByDay>\n";
		print XMLFILE "    </CalendarTrigger>\n";
	}
	else
	{
		print XMLFILE "    <CalendarTrigger>\n";
		print XMLFILE "      <StartBoundary>2014-09-03T00:${random_minute_long}:00</StartBoundary>\n";
		print XMLFILE "      <ExecutionTimeLimit>PT8H</ExecutionTimeLimit>\n";
		print XMLFILE "      <Enabled>true</Enabled>\n";
		print XMLFILE "      <ScheduleByDay>\n";
		print XMLFILE "        <DaysInterval>1</DaysInterval>\n";
		print XMLFILE "      </ScheduleByDay>\n";
		print XMLFILE "    </CalendarTrigger>\n";
		print XMLFILE "    <CalendarTrigger>\n";
		print XMLFILE "      <StartBoundary>2014-09-03T12:${random_minute_long}:00</StartBoundary>\n";
		print XMLFILE "      <ExecutionTimeLimit>PT8H</ExecutionTimeLimit>\n";
		print XMLFILE "      <Enabled>true</Enabled>\n";
		print XMLFILE "      <ScheduleByDay>\n";
		print XMLFILE "        <DaysInterval>1</DaysInterval>\n";
		print XMLFILE "      </ScheduleByDay>\n";
		print XMLFILE "    </CalendarTrigger>\n";
	}
	print XMLFILE "  </Triggers>\n";
	print XMLFILE "  <Principals>\n";
	print XMLFILE "    <Principal id=\"Author\">\n";
	print XMLFILE "      <UserId>S-1-5-20</UserId>\n";
	print XMLFILE "      <RunLevel>HighestAvailable</RunLevel>\n";
	print XMLFILE "    </Principal>\n";
	print XMLFILE "  </Principals>\n";
	print XMLFILE "  <Settings>\n";
	print XMLFILE "    <MultipleInstancesPolicy>IgnoreNew</MultipleInstancesPolicy>\n";
	print XMLFILE "    <DisallowStartIfOnBatteries>true</DisallowStartIfOnBatteries>\n";
	print XMLFILE "    <StopIfGoingOnBatteries>true</StopIfGoingOnBatteries>\n";
	print XMLFILE "    <AllowHardTerminate>true</AllowHardTerminate>\n";
	print XMLFILE "    <StartWhenAvailable>false</StartWhenAvailable>\n";
	print XMLFILE "    <RunOnlyIfNetworkAvailable>false</RunOnlyIfNetworkAvailable>\n";
	print XMLFILE "    <IdleSettings>\n";
	print XMLFILE "      <StopOnIdleEnd>true</StopOnIdleEnd>\n";
	print XMLFILE "      <RestartOnIdle>false</RestartOnIdle>\n";
	print XMLFILE "    </IdleSettings>\n";
	print XMLFILE "    <AllowStartOnDemand>true</AllowStartOnDemand>\n";
	print XMLFILE "    <Enabled>true</Enabled>\n";
	print XMLFILE "    <Hidden>false</Hidden>\n";
	print XMLFILE "    <RunOnlyIfIdle>false</RunOnlyIfIdle>\n";
	print XMLFILE "    <WakeToRun>false</WakeToRun>\n";
	print XMLFILE "    <ExecutionTimeLimit>PT8H</ExecutionTimeLimit>\n";
	print XMLFILE "    <Priority>7</Priority>\n";
	print XMLFILE "  </Settings>\n";
	print XMLFILE "  <Actions Context=\"Author\">\n";
	print XMLFILE "    <Exec>\n";
	print XMLFILE "      <Command>$par_batfile</Command>\n";
	print XMLFILE "      <WorkingDirectory>$par_scriptdir_bin</WorkingDirectory>\n";
	print XMLFILE "    </Exec>\n";
	print XMLFILE "  </Actions>\n";
	print XMLFILE "</Task>\n";
	close XMLFILE;
	

	LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"SCHEDTASK","   Creating $xml_file finished.");
	
	#print "\n";
	#LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"SCHEDTASK","   Creating scheduled task for $par_batfile.");
	LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"SCHEDTASK","   Command: $CMD_schtasks.");
	my @CMD_OUTPUT;
	open (CMD, "$CMD_schtasks 2>&1|") or die LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,3,"SCHEDTASK","   Can't run program: $!\n");
	while (my $line = <CMD>)
	{
		push(@CMD_OUTPUT, $line);
		LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"SCHEDTASK","   $line");
	}
 	close CMD;
	
	my $exit_CMD = $?;
	if ($exit_CMD == 0)
	{
		LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"SCHEDTASK","   - scheduled task for $par_batfile successfully created.");
	}
	else 
	{
		LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,3,"SCHEDTASK","   !! Creating scheduled task for $par_batfile not successfully !!");
		foreach (@CMD_OUTPUT)
		{
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,3,"SCHEDTASK","   !! $_");
		}
		LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,3,"SCHEDTASK","   !! Please fix the issue or add manually as decribed in the manual !!");
		unlink $xml_file or warn LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,4,"SCHEDTASK","   Could not delete $xml_file: $!");
		SAY_BYEBYE($par_log_fh,$par_vbse,$par_lglvl);
	}
	
	# Delete temporary xml file
	unlink $xml_file or warn LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,4,"SCHEDTASK","   Could not delete $xml_file: $!");
}             

sub MAKE_CONFIGFILE
{
	my $par_scriptdir_etc = $_[0];
	my $par_conf_prefix = $_[1];
	my $par_conf_suffix = $_[2];
	my $par_href_user_choice = $_[3];
	my $par_script_crypt = $_[4];
	my $par_log_fh = $_[5];
	my $par_vbse = $_[6];
	my $par_lglvl = $_[7];
	
	my ($CONFIG_start)=<<'CONFIG_start';
# Configuration file for HiCHperfpkg (Automatically created by ppsetup.pl)
# Hitachi Confoederatio Helvetica performance package collect tool
# Part of HiCHperfpkg
# Please see manual for detailed information http://loop.hds.com/docs/DOC-45708
# Author: Maik Ernst (maik.ernst\@hds.com), HDS
CONFIG_start
	
	my ($CONFIG_base)=<<'CONFIG_base';	

[monitor]
# on, off, test
ppmonitor_switch = off
ppmonitor_mailrelay = mail.company.com
# Email adresses must be coma seperated like mail@company.com,nextmail@company.com
ppmonitor_emailrecipient = recipient@company.com
ppmonitor_emailsender = sender@company.com
ppmonitor_runsperday = 2
ppmonitor_logkeepdays = 60
# on, off
ppmonitor_transfercheck = on

#### PLEASE DO NOT CHANGE ANYTHING IF YOU ARE NOT 100% SURE ####
[zip]
# zip, tgz
ppcollect_zipmethod = zip
ppcollect_zippath = 7-ZipPortable\7-Zip\7z.exe
ppcollect_archivedir = saves

[java]
ppcollect_javaexecutable = java

[log]
ppcollect_exporttoollogprefix = log
# normal, debug
ppcollect_loglevel = normal
pptransfer_loglevel = normal
ppmonitor_loglevel = normal

[features]
pptransfer_maxentries = 1000
#pptransfer_deletefiles = 0

[spooling]
pptransfer_spoolingprefix = spooling
pptransfer_spoolinglistfile = spooling_list.txt
pptransfer_spoolingsuffix = tgz,zip,log

[maindir]
pp_bin = bin
pp_data = data
pp_etc = etc
pp_doc = doc
pp_lib = lib
pp_log = log
pp_spool = spool
#################################################################
CONFIG_base
	
	my %config_file = (
	"system" => "n/a",
	"log" => "n/a",
	"vmio" => "n/a",
	);
		
	if (($par_href_user_choice->{initial} == 0) or ($par_href_user_choice->{initial} == 1))
	{
		$config_file{system} = "${par_scriptdir_etc}\\${par_conf_prefix}_$par_href_user_choice->{serial}.${par_conf_suffix}";
	}
	
	if ($par_href_user_choice->{initial} == 1)
	{
		$config_file{log} = "${par_scriptdir_etc}\\${par_conf_prefix}_log.${par_conf_suffix}";
	}
		
	if ($par_href_user_choice->{initial} == 2)
	{
		$config_file{vmio} = "${par_scriptdir_etc}\\${par_conf_prefix}_vmio.${par_conf_suffix}";
	}
	
	print "\n";
	LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIGFILE","   Checking for config file ...");

	# go through config file hash
	for my $key_cfg ( keys %config_file ) 
	{
		my $choice_del = "n/a";
		
		# if hash is not "n/a" => when it was set by user
		unless ( $config_file{$key_cfg} eq "n/a" )
		{
			if ( -e  $config_file{$key_cfg} )
			{
				LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIGFILE","   - config file: $config_file{$key_cfg} already exists.");
				LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIGFILE","     Do you want to keep or overwrite it?");
				while (1)
				{
					LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIGFILE","     Choice (keep/overwrite): ");
					print "     Choice (keep/overwrite): ";
					chomp ($choice_del = lc <STDIN>);
						
					if ($choice_del eq "keep")
					{
						LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIGFILE","     Keeping $config_file{$key_cfg}.");
						print "\n";
						last;
					}
					
					elsif ($choice_del eq "overwrite")
					{
						LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIGFILE","     Deleting $config_file{$key_cfg}.");
						unlink $config_file{$key_cfg} or warn LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,4,"CONFIGFILE","     Could not delete $config_file{$key_cfg}: $!");
						LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIGFILE","     Deleting $config_file{$key_cfg} finished.");
						print "\n";
						last;
					}
					
					else 
					{
						LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIGFILE","     Sorry, \"$choice_del\" was not a valid choice, try again.");
						LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIGFILE","     ...please answer \"keep\" or \"overwrite\"");
						next;
					}
				}
			}
			else
			{
				LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIGFILE","     $config_file{$key_cfg} does not exist.");
			}
			
			if (($choice_del eq "overwrite") or ($choice_del eq "n/a"))
			{
					
				LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIGFILE","    Creating $config_file{$key_cfg}.");		

				open(CFGFILE,"> $config_file{$key_cfg}") or die LOG_MSG ($par_vbse,"s",$par_lglvl,3,"CFGFILE","     Can't open $config_file{$key_cfg}: $!\n");
				print CFGFILE $CONFIG_start;
				print CFGFILE "\n";
				print CFGFILE "[main]\n";
				print CFGFILE "# windows\n";
				print CFGFILE "pp_mode = windows\n";
				print CFGFILE "# [G1500/G1000/G800/G600/G400/G200/VSP/HUSVM/USPV]\n";
				if ($key_cfg eq "system") { print CFGFILE "ppcollect_system = $par_href_user_choice->{system_type}\n"; }
				else { print CFGFILE "ppcollect_system = \n";}
				if ($key_cfg eq "system") { print CFGFILE "ppcollect_serial = $par_href_user_choice->{serial}\n"; }
				else { print CFGFILE "ppcollect_serial = \n";}
				if ($key_cfg eq "system") { print CFGFILE "ppcollect_exporttoollib = lib_$par_href_user_choice->{serial}\n"; }
				else { print CFGFILE "ppcollect_exporttoollib = \n";}
				print CFGFILE "ppcollect_keepdays = 60\n";
				print CFGFILE "ppcollect_logkeepdays = 60\n";
				print CFGFILE "ppcollect_memoryxmx = 1073741824\n";
				print CFGFILE "# 1=export tool only,2=raidcom only, 3=REST API only,\n";
				print CFGFILE "# 4=export tool + raidcom, 5=export tool + REST API,\n";
				print CFGFILE "# 6=raidcom + REST API, 7=export tool + raidcom + REST API\n";
				print CFGFILE "ppcollect_collection = 7\n";
				print CFGFILE "# off, https, ssh\n";
				if (($par_href_user_choice->{initial} == 1) or ($par_href_user_choice->{initial} == 2))
				{
					print CFGFILE "pptransfer_method = $par_href_user_choice->{transfer_type}\n";
					print CFGFILE "# defaultraid, defaultlog, defaultvmio, d:\\path\\to\\dir\n";
					if ($key_cfg eq "system") { print CFGFILE "pptransfer_sourcedir = defaultraid\n"; }
					if ($key_cfg eq "log") { print CFGFILE "pptransfer_sourcedir = defaultlog\n"; }
					if ($key_cfg eq "vmio") { print CFGFILE "pptransfer_sourcedir = defaultvmio\n"; }
					print CFGFILE "pptransfer_logkeepdays = 60\n";
					print CFGFILE "\n";
					print CFGFILE "[raidcom]\n";
					print CFGFILE "# 0=full, 1=partial, 2=minimum\n";
					print CFGFILE "ppcollect_ccicollectionlevel = 0\n";
					print CFGFILE "# IP, FC\n";
					print CFGFILE "ppcollect_commanddevice = IP\n";
					print CFGFILE "ppcollect_ccidir = C:\\HORCM\n";
					print CFGFILE "ppcollect_horcmdirprefix = horcmdir\n";
					print CFGFILE "\n";
					print CFGFILE "[restapi]\n";
					if ($par_href_user_choice->{restapi_user} eq "n/a") { print CFGFILE "ppcollect_restapiusr = \n"; }
					else { print CFGFILE "ppcollect_restapiusr = $par_href_user_choice->{restapi_user}\n"; }
					print CFGFILE "# The password must be encrypted. Details see manual\n";
					if ($par_href_user_choice->{restapi_cryptpwd} eq "n/a") 
					{ 
						print CFGFILE "ppcollect_restapicryptpwd = \n"; 
					}
					else 
					{ 
						LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIGFILE","    Encode REST API password start");
						my $decrypt_restapi_cryptpwd = "n/a";
						open (CMD_CRYPT, "$par_script_crypt -e $par_href_user_choice->{restapi_cryptpwd} 2>&1|") or warn ("!!! Can't run program: $!\n");
						while (my $line = <CMD_CRYPT>)
						{
							LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,7,"CONFIGFILE","   $line");
							if ($line =~ m/^ENCODED: ([0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz_-]{3,50})/)
							{
								$decrypt_restapi_cryptpwd = $1;
							}
						}
						close CMD_CRYPT;
						LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIGFILE","    Encode REST API password finished");
						if ($decrypt_restapi_cryptpwd eq "n/a")
						{			
							LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,4,"CONFIGFILE","   Something went wrong with the REST API pwd encryption. Please check ppsetup logfile and/turn on debug log level.");
						}
						print CFGFILE "ppcollect_restapicryptpwd = $decrypt_restapi_cryptpwd\n"; 
					}
					print CFGFILE "# 0=full, 1=partial, 2=minimum\n";
					print CFGFILE "ppcollect_restapicollectionlevel = 0\n";
					print CFGFILE "ppcollect_restapidirprefix = rest\n";
					print CFGFILE "# http, https\n";
					print CFGFILE "ppcollect_restapiprotocol = https\n";
					print CFGFILE "\n";
					print CFGFILE "[ssh]\n";
					if ($par_href_user_choice->{transfer_ssh_user} eq "n/a") { print CFGFILE "pptransfer_username = \n"; }
					else { print CFGFILE "pptransfer_username = $par_href_user_choice->{transfer_ssh_user}\n"; }
					if ($par_href_user_choice->{transfer_ssh_key} eq "n/a") { print CFGFILE "pptransfer_keyfile = \n"; }
					else { print CFGFILE "pptransfer_keyfile = key\\$par_href_user_choice->{transfer_ssh_key}\n"; }
					print CFGFILE "pptransfer_remotehost = ftp.hdsioportal.com\n";
					print CFGFILE "pptransfer_remotedir = incoming\n";
					print CFGFILE "pptransfer_pscppath = putty\\pscp.exe\n";
					print CFGFILE "pptransfer_plinkpath = putty\\plink.exe\n";
					print CFGFILE "\n";
					print CFGFILE "[https]\n";
					if ($par_href_user_choice->{transfer_https_token} eq "n/a") { print CFGFILE "pptransfer_token = \n"; }
					else {	print CFGFILE "pptransfer_token = $par_href_user_choice->{transfer_https_token}\n"; }
					if ($par_href_user_choice->{transfer_https_pxy_server} eq "n/a") { print CFGFILE "pptransfer_proxyserver = \n"; }
					else {	print CFGFILE "pptransfer_proxyserver = $par_href_user_choice->{transfer_https_pxy_server}\n"; }
					if ($par_href_user_choice->{transfer_https_pxy_protocol} eq "n/a") { print CFGFILE "pptransfer_proxyprotocol = \n"; }
					else {	print CFGFILE "pptransfer_proxyprotocol = $par_href_user_choice->{transfer_https_pxy_protocol}\n"; }
					if ($par_href_user_choice->{transfer_https_pxy_port} eq "n/a") { print CFGFILE "pptransfer_proxyport = \n"; }
					else {	print CFGFILE "pptransfer_proxyport = $par_href_user_choice->{transfer_https_pxy_port}\n"; }
					if ($par_href_user_choice->{transfer_https_pxy_user} eq "n/a") { print CFGFILE "pptransfer_proxyuser = \n"; }
					else {	print CFGFILE "pptransfer_proxyuser = $par_href_user_choice->{transfer_https_pxy_user}\n"; }
					if ($par_href_user_choice->{transfer_https_pxy_pwd} eq "n/a") { print CFGFILE "pptransfer_proxypwd = \n"; }
					else {	print CFGFILE "pptransfer_proxypwd = $par_href_user_choice->{transfer_https_pxy_pwd}\n"; }
					if ($par_href_user_choice->{transfer_https_pxy_options} eq "n/a") { print CFGFILE "pptransfer_proxyoptions = \n"; }
					else {	print CFGFILE "pptransfer_proxyoptions = $par_href_user_choice->{transfer_https_pxy_options}\n"; }
					print CFGFILE "pptransfer_remoteserver = www.hdsioportal.com\n";
					print CFGFILE "pptransfer_remotepath = file_upload/store_file\n";
					print CFGFILE "pptransfer_curlpath = curl\\curl.exe\n";
				}
				else
				{
					print CFGFILE "pptransfer_method = off\n";
					print CFGFILE "# defaultraid, defaultlog, defaultvmio, d:\\path\\to\\dir\n";
					print CFGFILE "pptransfer_sourcedir = defaultraid\n";
					print CFGFILE "pptransfer_logkeepdays = 60\n";
					print CFGFILE "\n";
					print CFGFILE "[raidcom]\n";
					print CFGFILE "# 0=full, 1=partial, 2=minimum\n";
					print CFGFILE "ppcollect_ccicollectionlevel = 0\n";
					print CFGFILE "# IP, FC\n";
					print CFGFILE "ppcollect_commanddevice = IP\n";
					print CFGFILE "ppcollect_ccidir = C:\\HORCM\n";
					print CFGFILE "ppcollect_horcmdirprefix = horcmdir\n";
					print CFGFILE "\n";
					print CFGFILE "[restapi]\n";
					if ($par_href_user_choice->{restapi_user} eq "n/a") { print CFGFILE "ppcollect_restapiusr = \n"; }
					else { print CFGFILE "ppcollect_restapiusr = $par_href_user_choice->{restapi_user}\n"; }
					print CFGFILE "# The password must be encrypted. Details see manual\n";
					if ($par_href_user_choice->{restapi_cryptpwd} eq "n/a") 
					{ 
						print CFGFILE "ppcollect_restapicryptpwd = \n"; 
					}
					else 
					{ 
						LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIGFILE","    Encode REST API password start");
						my $decrypt_restapi_cryptpwd = "n/a";
						open (CMD_CRYPT, "$par_script_crypt -e $par_href_user_choice->{restapi_cryptpwd} 2>&1|") or warn ("!!! Can't run program: $!\n");
						while (my $line = <CMD_CRYPT>)
						{
							LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,7,"CONFIGFILE","   $line");
							if ($line =~ m/^ENCODED: ([0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz_-]{3,50})/)
							{
								$decrypt_restapi_cryptpwd = $1;
							}
						}
						close CMD_CRYPT;
						LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIGFILE","    Encode REST API password finished");
						if ($decrypt_restapi_cryptpwd eq "n/a")
						{			
							LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,4,"CONFIGFILE","   Something went wrong with the REST API pwd encryption. Please check ppsetup logfile.");
						}
						print CFGFILE "ppcollect_restapicryptpwd = $decrypt_restapi_cryptpwd\n"; 
					}
					print CFGFILE "# 0=full, 1=partial, 2=minimum\n";
					print CFGFILE "ppcollect_restapicollectionlevel = 0\n";
					print CFGFILE "ppcollect_restapidirprefix = rest\n";
					print CFGFILE "# http, https\n";
					print CFGFILE "ppcollect_restapiprotocol = https\n";
					print CFGFILE "\n";
					print CFGFILE "[ssh]\n";
					print CFGFILE "pptransfer_username = \n";
					print CFGFILE "pptransfer_keyfile = \n";
					print CFGFILE "pptransfer_remotehost = ftp.hdsioportal.com\n";
					print CFGFILE "pptransfer_remotedir = incoming\n";
					print CFGFILE "pptransfer_pscppath = putty\\pscp.exe\n";
					print CFGFILE "pptransfer_plinkpath = putty\\plink.exe\n";
					print CFGFILE "\n";
					print CFGFILE "[https]\n";
					print CFGFILE "pptransfer_token = \n";
					print CFGFILE "pptransfer_proxyserver = \n";
					print CFGFILE "pptransfer_proxyprotocol = \n";
					print CFGFILE "pptransfer_proxyport = \n";
					print CFGFILE "pptransfer_proxyuser = \n";
					print CFGFILE "pptransfer_proxypwd = \n";
					print CFGFILE "pptransfer_proxyoptions = \n";
					print CFGFILE "pptransfer_remoteserver = www.hdsioportal.com\n";
					print CFGFILE "pptransfer_remotepath = file_upload/store_file\n";
					print CFGFILE "pptransfer_curlpath = curl\\curl.exe\n";
					print CFGFILE "\n";
				}

				print CFGFILE $CONFIG_base;
				close CFGFILE;
				LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIGFILE","   - config file: $config_file{$key_cfg} successfully created.");
			}
		}
	}
	
}

sub MAKE_BATFILE
{
	my $par_scriptdir_etc = $_[0];
	my $par_scriptdir_bin = $_[1];
	my $par_conf_prefix = $_[2];
	my $par_conf_suffix = $_[3];
	my $par_href_user_choice = $_[4];
	my $par_log_fh = $_[5];
	my $par_vbse = $_[6];
	my $par_lglvl = $_[7];
	
	my $bat_prefix = "ppwrapper";
	my $bat_suffix = "bat";
	my $collect_script = "ppcollect.pl";
	my $transfer_script = "pptransfer.pl";
	my $bat_file;
	my $choice_del = "n/a";
	
	# Create Serial bat file
	if (($par_href_user_choice->{initial} == 0) or ($par_href_user_choice->{initial} == 1))
	{
		$bat_file = "${par_scriptdir_bin}\\${bat_prefix}_$par_href_user_choice->{serial}.${bat_suffix}";
		print "\n";
		LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"BATFILE","   Checking for batch file ...");
		
		$choice_del = "n/a";
		# check if bat file exists
		if ( -e  $bat_file )
		{
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"BATFILE","   - batch file: ${bat_file} already exists.");
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"BATFILE","     Do you want to keep or overwrite it?");
			while (1)
			{
				LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"BATFILE","     Choice (keep/overwrite): ");
				print "     Choice (keep/overwrite): ";
				chomp ($choice_del = lc <STDIN>);
					
				if ($choice_del eq "keep")
				{
					LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"BATFILE","     Keeping ${bat_file}.");
					print "\n";
					last;
				}
				
				elsif ($choice_del eq "overwrite")
				{
					LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"BATFILE","     Deleting  ${bat_file}.");
					unlink ${bat_file} or warn LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,4,"BATFILE","     Could not delete ${bat_file}: $!");
					LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"BATFILE","     Deleting ${bat_file} finished.");
					print "\n";
					last;
				}
				
				else 
				{
					LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"BATFILE","     Sorry, \"$choice_del\" was not a valid choice, try again.");
					LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"BATFILE","     ...please answer \"keep\" or \"overwrite\"");
					next;
				}
			}
		}
		else
		{
			LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"BATFILE","     ${bat_file} does not exist.");
		}
			
		# if bat file delete or not exist then create for serial
		if (($choice_del eq "overwrite") or ($choice_del eq "n/a"))
		{
			LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"BATFILE","    Creating ${bat_file}.");
			open(BATFILE,"> ${bat_file}") or die LOG_MSG ($par_vbse,"s",$par_lglvl,3,"BATFILE","    Can't open ${bat_file}: $!\n");
			print BATFILE "\@echo off\n";
			print BATFILE "rem ===================================\n";
			print BATFILE "rem\n";
			print BATFILE "rem Script: ${bat_prefix}_$par_href_user_choice->{serial}.bat\n";
			print BATFILE "rem Automatically created by ppsetup.pl\n";
			print BATFILE "rem Author: Maik Ernst (maik.ernst\@hds.com), HDS\n";
			print BATFILE "rem\n";
			print BATFILE "rem ===================================\n";
			print BATFILE "set STRING=$par_href_user_choice->{serial}\n";
			print BATFILE "rem === PERFORMANCE AND/OR RAIDCOM COLLECTION ===\n";
			print BATFILE "${par_scriptdir_bin}\\${collect_script} -c ${conf_prefix}_%STRING%.${conf_suffix}\n";
			if ($par_href_user_choice->{initial} == 1)
			{
				print BATFILE "rem === TRANSFER DATA TO HDSIOPORTAL ===\n";
				print BATFILE "${par_scriptdir_bin}\\${transfer_script} -c ${conf_prefix}_%STRING%.${conf_suffix}\n";
			}
			close BATFILE;
			
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"BATFILE","   - batch file: $bat_file successfully created.");
			
			#Creating task
			MAKE_SCHEDTASK ("serial",$par_scriptdir_etc,$par_scriptdir_bin,$bat_file,$par_href_user_choice,$par_log_fh,$par_vbse,$par_lglvl);
		}
		
		# Create log bat file
		if ($par_href_user_choice->{initial} == 1)
		{
			$bat_file = "${par_scriptdir_bin}\\${bat_prefix}_log.${bat_suffix}";
			
			$choice_del = "n/a";
			if ( -e  $bat_file )
			{
				LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"BATFILE","   - batch file: ${bat_file} already exists.");
				LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"BATFILE","     Do you want to keep or overwrite it?");
				while (1)
				{
					LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"BATFILE","     Choice (keep/overwrite): ");
					print "     Choice (keep/overwrite): ";
					chomp ($choice_del = lc <STDIN>);
						
					if ($choice_del eq "keep")
					{
						LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"BATFILE","     Keeping ${bat_file}.");
						print "\n";
						last;
					}
					
					elsif ($choice_del eq "overwrite")
					{
						LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"BATFILE","     Deleting  ${bat_file}.");
						unlink ${bat_file} or warn LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,4,"CONFIGFILE","     Could not delete ${bat_file}: $!");
						LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"BATFILE","     Deleting ${bat_file} finished.");
						print "\n";
						last;
					}
					
					else 
					{
						LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"BATFILE","     Sorry, \"$choice_del\" was not a valid choice, try again.");
						LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"BATFILE","     ...please answer \"keep\" or \"overwrite\"");
						next;
					}
				}
			}
			else
			{
				print "\n";
				LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"BATFILE","     ${bat_file} does not exist.");
			}
				
				
			if (($choice_del eq "overwrite") or ($choice_del eq "n/a"))
			{
				LOG_MSG($par_log_fh,$par_vbse,"c",$par_lglvl,5,"BATFILE","     Creating ${bat_file}.");
				open(BATFILE3,"> ${bat_file}") or die LOG_MSG ($par_vbse,"s",$par_lglvl,3,"BATFILE","     Can't open ${bat_file}: $!\n");
				print BATFILE3 "\@echo off\n";
				print BATFILE3 "rem ===================================\n";
				print BATFILE3 "rem\n";
				print BATFILE3 "rem Script: ${bat_prefix}_log.bat\n";
				print BATFILE3 "rem Automatically created by ppsetup.pl\n";
				print BATFILE3 "rem Author: Maik Ernst (maik.ernst\@hds.com), HDS\n";
				print BATFILE3 "rem\n";
				print BATFILE3 "rem ===================================\n";
				print BATFILE3 "set STRING=log\n";
				print BATFILE3 "rem === TRANSFER LOG TO HDSIOPORTAL ===\n";
				print BATFILE3 "${par_scriptdir_bin}\\${transfer_script} -c ${conf_prefix}_%STRING%.${conf_suffix}\n";
				close BATFILE3;
				
				LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"BATFILE","   - batch file: $bat_file successfully created.");
				
				#Creating task
				MAKE_SCHEDTASK ("log",$par_scriptdir_etc,$par_scriptdir_bin,$bat_file,$par_href_user_choice,$par_log_fh,$par_vbse,$par_lglvl);
			}
		}	
		
	}
	
	# Create vmio bat file
	if ($par_href_user_choice->{initial} == 2)
	{
		$bat_file = "${par_scriptdir_bin}\\${bat_prefix}_vmio.${bat_suffix}";
		
		$choice_del = "n/a";
		if ( -e  $bat_file )
		{
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"BATFILE","   - batch file: ${bat_file} already exists.");
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"BATFILE","     Do you want to keep or overwrite it?");
			while (1)
			{
				LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"BATFILE","     Choice (keep/overwrite): ");
				print "     Choice (keep/overwrite): ";
				chomp ($choice_del = lc <STDIN>);
					
				if ($choice_del eq "keep")
				{
					LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"BATFILE","     Keeping ${bat_file}.");
					print "\n";
					last;
				}
				
				elsif ($choice_del eq "overwrite")
				{
					LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"BATFILE","     Deleting  ${bat_file}.");
					unlink ${bat_file} or warn LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,4,"CONFIGFILE","     Could not delete ${bat_file}: $!");
					LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"BATFILE","     Deleting ${bat_file} finished.");
					print "\n";
					last;
				}
				
				else 
				{
					LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"BATFILE","     Sorry, \"$choice_del\" was not a valid choice, try again.");
					LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"BATFILE","     ...please answer \"keep\" or \"overwrite\"");
					next;
				}
			}
		}
		else
		{
			LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"BATFILE","     ${bat_file} does not exist.");
		}
		
		if (($choice_del eq "overwrite") or ($choice_del eq "n/a"))
		{
			LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"BATFILE","     Creating ${bat_file}.");
			open(BATFILE3,"> ${bat_file}") or die LOG_MSG ($par_vbse,"s",$par_lglvl,3,"BATFILE","Can't open ${bat_file}: $!\n");
			print BATFILE3 "\@echo off\n";
			print BATFILE3 "rem ===================================\n";
			print BATFILE3 "rem\n";
			print BATFILE3 "rem Script: ${bat_prefix}_vmio.bat\n";
			print BATFILE3 "rem Automatically created by ppsetup.pl\n";
			print BATFILE3 "rem Author: Maik Ernst (maik.ernst\@hds.com), HDS\n";
			print BATFILE3 "rem\n";
			print BATFILE3 "rem ===================================\n";
			print BATFILE3 "set STRING=vmio\n";
			print BATFILE3 "rem === COLLECT & TRANSFER VMIO DATA TO HDSIOPORTAL ===\n";
			print BATFILE3 "powershell ${par_scriptdir_bin}\\vmio\\extractVMIOstats.ps1 -interval 12\n";
			print BATFILE3 "${par_scriptdir_bin}\\${transfer_script} -c ${conf_prefix}_%STRING%.${conf_suffix}\n";
			close BATFILE3;
			
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"BATFILE","   - config file: $bat_file successfully created.");
			
			#Creating task
			MAKE_SCHEDTASK ("vmio",$par_scriptdir_etc,$par_scriptdir_bin,$bat_file,$par_href_user_choice,$par_log_fh,$par_vbse,$par_lglvl);
		}
	}
}

sub MAKE_COMMANDTXT
{
	my $par_scriptdir_etc = $_[0];
	my $par_href_user_choice = $_[1];
	my $par_flag_svpout = $_[2];
	my $par_cmd_middle_name = $_[3];
	my $par_cmd_suffix = $_[4];
	my $par_log_fh = $_[5];
	my $par_vbse = $_[6];
	my $par_lglvl = $_[7];

	$par_flag_svpout = $par_flag_svpout . $par_href_user_choice->{serial};
	my $choice_del = "n/a";

	print "\n";
	LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CMDTXT","   Checking for command.txt file ...");
	
	my $command_file = "${par_scriptdir_etc}\\$par_href_user_choice->{system_type}_${par_cmd_middle_name}_$par_href_user_choice->{serial}.${par_cmd_suffix}";
	if ( -e  $command_file )
	{
		LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CMDTXT","   - config file: $command_file already exists.");
		LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CMDTXT","     Do you want to keep or overwrite it?");
		while (1)
		{
			LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CMDTXT","     Choice (keep/overwrite): ");
			print "     Choice (keep/overwrite): ";
			chomp ($choice_del = lc <STDIN>);
				
			if ($choice_del eq "keep")
			{
				LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CMDTXT","     Keeping $command_file.");
				print "\n";
				last;
			}
			
			elsif ($choice_del eq "overwrite")
			{
				LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CMDTXT","     Deleting $command_file.");
				unlink $command_file or warn LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,4,"CMDTXT","     Could not delete $command_file: $!");
				LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CMDTXT","     Deleting $command_file finished.");
				print "\n";
				last;
			}
			
			else 
			{
				LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CMDTXT","     ...please answer \"keep\" or \"overwrite\"");
				LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CMDTXT","     Sorry, \"$choice_del\" was not a valid choice, try again.");
				next;
			}
		}
	}
	else
	{
		LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CMDTXT","     $command_file does not exist.");
	}
	
	if (($choice_del eq "overwrite") or ($choice_del eq "n/a"))
	{
		LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CMDTXT","     Creating $command_file.");
		my $cmd_txt_output = CMD_GROUPS($par_href_user_choice->{system_type},$par_log_fh,$par_vbse,$par_lglvl);
		
		open(COMMANDTXT,"> $command_file") or die LOG_MSG ($par_vbse,"s",$par_lglvl,3,"CMDTXT","     Can't open $command_file: $!\n");
		print COMMANDTXT "\;  +------------------------ Automatically created by ppsetup.pl ---------------------------------+\n";
		if ($par_href_user_choice->{system_type} =~ /^G800$|^G600$|^G400$|^G200$/)
		{
			print COMMANDTXT "\;gum $par_href_user_choice->{gumip0} $par_href_user_choice->{gumip1} \; DO NOT DELETE/EDIT THIS LINE and LEAVE ALL AS COMMENT (otherwise perf/raidcom collection will fail)\n";
			print COMMANDTXT "ip $par_href_user_choice->{svpip}               \; Specifies IP address of SVP\n";
			print COMMANDTXT "dkcsn $par_href_user_choice->{serial}    \;  Specifies Serial Number of DKC\n";
			print COMMANDTXT "login $par_href_user_choice->{collect_user} \"$par_href_user_choice->{collect_pwd}\"    \; Logs user into SVP\n";
		}
		else
		{
			print COMMANDTXT "svpip $par_href_user_choice->{svpip}               \; Specifies IP address of SVP\n";
			print COMMANDTXT "login $par_href_user_choice->{collect_user} \"$par_href_user_choice->{collect_pwd}\"    \; Logs user into SVP\n";
		}
		print COMMANDTXT "show                            \; Outputs storing period & gethering interval to standard output\n";
		print COMMANDTXT $cmd_txt_output;
		
		# Quote \ to make C:\...\ to C:\\...\\....
		$par_flag_svpout =~ s/\\/\\\\/g;
		print COMMANDTXT "outpath \"$par_flag_svpout\"      \; Specifies the sub-directory in which files will be saved\n";
		print COMMANDTXT "option compress                 \; Specifies whether to compress files\n";
		print COMMANDTXT "apply                           \; Executes processing for saving monitoring data in files\n";
		close COMMANDTXT;
		
		LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CMDTXT","   - command.txt file: $command_file successfully created.");
	}

}

sub CONFIG_FROM_USER
{
	my $par_log_fh = $_[0];
	my $par_vbse = $_[1];
	my $par_lglvl = $_[2];
	
	my %choice = (
	"initial" => "n/a",
	"system_type" => "n/a",
	"serial" => "n/a",
	"svpip" => "n/a",
	"collect_user" => "n/a",
	"collect_pwd" => "n/a",
	"restapi_user" => "n/a",
	"restapi_cryptpwd" => "n/a",
	"collect_correct" => "n/a",
	"transfer_type" => "n/a",
	"transfer_ssh_user" => "n/a",
	"transfer_ssh_key" => "n/a",
	"transfer_https_token" => "n/a",
	"transfer_https_pxy_protocol" => "n/a",
	"transfer_https_pxy_server" => "n/a",
	"transfer_https_pxy_port" => "n/a",
	"transfer_https_pxy_user" => "n/a",
	"transfer_https_pxy_pwd" => "n/a",
	"transfer_https_pxy_options" => "n/a",
	"transfer_correct" => "n/a",
	);
	
	# get initial choice 0,1,2
	while (1)
	{	
		LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   0) Collect \"Performance & Config Info\" only"); 
		LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   1) Collect \"Performance & Config Info\" & upload to hdsioportal.com"); 
		LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   2) Collect \"VMIO Info\" & upload to hdsioportal.com");
		LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","  10) Update configuration files to HiCHperfpkg v8.0.x");
		LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   Choice (0/1/2/10): ");
		print "   Choice (0/1/2/10): ";
		chomp ($choice{initial} = <STDIN>);
		
		if ($choice{initial} !~ /^0$|^1$|^2$|^10$/)
		{
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   Sorry, \"$choice{initial}\" was not a valid choice, try again.");
			print "\n";			
			next;
		}
		else
		{	
			LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   \"$choice{initial}\" is valid.");
			print "\n";	
			last;
		}
	}

	if (($choice{initial} == 0) or ($choice{initial} == 1))
	{
		# Repeat user choice when "n" at the end.
		while (1)
		{	
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   -------------------------------------------------------------------");
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   Please provide storage subsystem related information first...");
			print "\n";			
			# get system type
			while (1)
			{
				LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   Enter storage type (G1500/G1000/G800/G600/G400/G200/VSP/HUSVM/USPV)"); 
				LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   Type: "); 
				print "   Type: ";
				chomp ($choice{system_type} = <STDIN>);
						
				if ($choice{system_type} !~ /^G1500$|^G1000$|^G800$|^G600$|^G400$|^G200$|^VSP$|^HUSVM$|^USPV$/)
				{
					LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   Sorry, \"$choice{system_type}\" was not a valid choice, try again.");
					print "\n";			
					next;
				}
				else
				{	
					LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   \"$choice{system_type}\" is valid.");
					last;
				}
			}
			
			# get serial
			while (1)
			{
				print "\n";
				LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   Enter serial number of storage subsystem"); 
				LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   Serial#: "); 
				print "   Serial#: ";
				chomp ($choice{serial} = <STDIN>);
						
				if ($choice{serial} !~ /^\d+$/)
				{
					LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   Sorry, \"$choice{serial}\" was not a valid choice, try again.");
					print "\n";			
					next;
				}
				else
				{	
					LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   \"$choice{serial}\" is valid.");
					last;
				}
			}
			
			# get svpip
			while (1)
			{
				print "\n";
				LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   Enter IP address of storage subsystem (SVP)"); 
				LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   IP adress: "); 
				print "   IP adress: ";
				chomp ($choice{svpip} = <STDIN>);
						
				if ($choice{svpip} !~ m/^\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}$/)
				{
					LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   Sorry, \"$choice{svpip}\" was not a valid choice, try again.");
					print "\n";			
					next;
				}
				else
				{	
					LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   \"$choice{svpip}\" is valid.");
					last;
				}
			}
			
			# get GUM ip
			if ($choice{system_type} =~ /^G800$|^G600$|^G400$|^G200$/)
			{
				while (1)
				{
					print "\n";
					LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   Enter IP address of Controler 0 (GUM)"); 
					LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   IP adress: "); 
					print "   IP adress: ";
					chomp ($choice{gumip0} = <STDIN>);
							
					if ($choice{gumip0} !~ m/^\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}$/)
					{
						LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   Sorry, \"$choice{gumip0}\" was not a valid choice, try again.");
						print "\n";			
						next;
					}
					else
					{	
						LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   \"$choice{gumip0}\" is valid.");
						last;
					}
				}
				
				while (1)
				{
					print "\n";
					LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   Enter IP address of Controler 1 (GUM)"); 
					LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   IP adress: "); 
					print "   IP adress: ";
					chomp ($choice{gumip1} = <STDIN>);
							
					if ($choice{gumip0} !~ m/^\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}$/)
					{
						LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   Sorry, \"$choice{gumip1}\" was not a valid choice, try again.");
						print "\n";			
						next;
					}
					else
					{	
						LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   \"$choice{gumip1}\" is valid.");
						last;
					}
				}
			}
			
			# get export tool user
			print "\n";
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   Enter user name of \"export tool & raidcom\" that runs data collection"); 
			LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   User name: "); 
			print "   User name: ";
			chomp ($choice{collect_user} = <STDIN>);
			
			# get export tool password
			print "\n";
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   Enter password  of \"export tool & raidcom\" user"); 
			LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   Password: "); 
			print "   Password: ";
			chomp ($choice{collect_pwd} = <STDIN>);
			
			# get rest api user
			while (1)
			{
				print "\n";
				LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   Enter user name of \"REST API\" that runs data collection"); 
				LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   User name: "); 
				print "   User name: ";
				chomp ($choice{restapi_user} = <STDIN>);
						
				if ($choice{restapi_user} eq $choice{collect_user})
				{
					LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   Sorry, \"REST API\" user ($choice{restapi_user}) cannot be identical with \"export tool & raidcom\" user ($choice{collect_user}) , try again.");
					print "\n";			
					next;
				}
				else
				{	
					LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   \"$choice{restapi_user}\" is valid.");
					last;
				}
			}
			
			# get rest api password
			while (1)
			{
				print "\n";
				LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   Enter password  of\"REST API\" user"); 
				LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   Password: "); 
				print "   Password: ";
				chomp ($choice{restapi_cryptpwd} = <STDIN>);
						
				if ($choice{restapi_cryptpwd} !~ m/^[0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz_-]{6,29}$/)
				{
					LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   Sorry, \"REST API\" password  must have at 6 characters and can only contain these characters \"0-9A-Za-z_-\", try again.");
					print "\n";			
					next;
				}
				else
				{	
					LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   \"$choice{restapi_cryptpwd}\" is valid.");
					last;
				}
			}
					
			# 6-29
			#0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz_-
			
			# check settings
			print "\n";
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   Provided settings");
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   - Type:      :\"$choice{system_type}\"");
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   - Serial#    :\"$choice{serial}\"");
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   - SVP IP address :\"$choice{svpip}\"");
			if ($choice{system_type} =~ /^G800$|^G600$|^G400$|^G200$/)
			{
				LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   - Contoller 0 (GUM) IP address :\"$choice{gumip0}\"");
				LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   - Contoller 1 (GUM) IP address :\"$choice{gumip1}\"");
			}
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   - Export tool user name  :\"$choice{collect_user}\"");
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   - Export tool password   :\"$choice{collect_pwd}\"");
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   - REST API user name  :\"$choice{restapi_user}\"");
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   - REST API password   :\"$choice{restapi_cryptpwd}\"");
			print "\n";
			
			while (1)
			{
				LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   Are these setting correct (y/n): "); 
				print "   Are these setting correct (y/n): ";
				chomp ($choice{collect_correct} = lc <STDIN>);
						
				if ($choice{collect_correct} eq "y")
				{
					LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   \"$choice{collect_correct}\" is valid and approved by user.");
					last;
				}
				if ($choice{collect_correct} eq "n")
				{
					LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   \"$choice{collect_correct}\" is valid but not approved by user.");
					LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   Starting again.");
					last;
				}
				else
				{	
					LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   \"$choice{collect_correct}\" was not a valid choice");
					LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   ...please answer \"y\" or \"n\"");
					print "\n";			
					next;
				}
			}
			
			if ($choice{collect_correct} eq "y")
			{
				LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   User choice finished.");	
				last;
			}
			else
			{	
				LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   User choice starting again.");	
				($choice{system_type},$choice{serial},$choice{svpip},$choice{collect_user},$choice{collect_pwd},$choice{collect_correct}) = ("n/a","n/a","n/a","n/a","n/a","n/a") ;	
				($choice{restapi_user},$choice{restapi_cryptpwd}) = ("n/a","n/a") ;
				($choice{gumip0},$choice{gumip1}) = ("n/a","n/a") ;				
				next;
			}
		}
	}	
	
	# get transfer information
	
	if (($choice{initial} == 1) or ($choice{initial} == 2))
	{
		# Repeat user choice when "n" at the end.
		while (1)
		{	
			print "\n";	
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   -------------------------------------------------------------------");
			LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   Please provide upload related information ...");
			# get transfer type
			while (1)
			{
				print "\n";
				LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   Enter transfer mode (ssh/https) to hdsioportal.com"); 
				LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   Transfer mode: "); 
				print "   Transfer mode: ";
				chomp ($choice{transfer_type} = lc <STDIN>);
						
				if ($choice{transfer_type} !~ /^ssh$|^https$/)
				{
					LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   ...please answer \"ssh\" or \"https\"");
					print "\n";			
					next;
				}
				else
				{	
					LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   \"$choice{transfer_type}\" is valid.");
					last;
				}
			}	
			
			if ($choice{transfer_type} eq "ssh")
			{
				# get ssh user
				print "\n";
				LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   Enter SSH user name: ");
				print "   Enter SSH user name: ";
				chomp ($choice{transfer_ssh_user} = <STDIN>);
				
				# get ssh keyfile
				LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   Enter SSH keyfile: ");
				print "   Enter SSH keyfile: ";
				chomp ($choice{transfer_ssh_key} = <STDIN>);
		
				# check settings
				print "\n";
				LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   Provided settings");
				LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   - Transfer mode : \"$choice{transfer_type}\"");
				LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   - User name     : \"$choice{transfer_ssh_user}\"");
				LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   - Keyfile       : \"$choice{transfer_ssh_key}\"");
				print "\n";
				
			}
			elsif ($choice{transfer_type} eq "https")
			{
				# get https token
				while (1)
				{
					print "\n";
					LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   Enter HTTPS token");
					LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   HTTPS token: "); 
					print "   HTTPS token: ";
					chomp ($choice{transfer_https_token} = <STDIN>);
						
					if ($choice{transfer_https_token} !~ /^\w+$/)
					{
						LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   Sorry, \"$choice{transfer_https_token}\" was not a valid choice, try again.");
						print "\n";			
						next;
					}
					else
					{	
						LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   \"$choice{transfer_https_token}\" is valid.");
						last;
					}
				}

				# proxy check
				my $proxy_chk;
				while (1)
				{
					print "\n";
					LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   Do you use a HTTPS proxy (y/n): "); 
					print "   Do you use a HTTPS proxy (y/n): ";
					chomp ($proxy_chk = lc <STDIN>);
						
					if ($proxy_chk eq "y")
					{
						LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   \"$proxy_chk\" is valid and proxy seettings will be done.");
						
						# get https proxy server
						print "\n";
						LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   Enter HTTPS proxy server: "); 
						print "   Enter HTTPS proxy server: ";
						chomp ($choice{transfer_https_pxy_server} = <STDIN>);
		
						# get https proxy port
						LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   Enter HTTPS proxy port: "); 
						print "   Enter HTTPS proxy port: ";
						chomp ($choice{transfer_https_pxy_port} = <STDIN>);
						
						# get https proxy protocol
						LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   Enter HTTPS proxy protocol: "); 
						print "   Enter HTTPS proxy protocol: ";
						chomp ($choice{transfer_https_pxy_protocol} = <STDIN>);
						
						# get https proxy user
						while (1)
						{
							LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   Enter HTTPS proxy user name: "); 
							print "   Enter HTTPS proxy user name: ";
							chomp ($choice{transfer_https_pxy_user} = <STDIN>);
							
							if ($choice{transfer_https_pxy_user} =~ m/@|&/)
							{
								LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,4,"CONFIG_USER","   Sorry, \"$choice{transfer_https_pxy_user}\" was not a valid choice. Characters \"@\" and \"&\" are not allowed, try again.");
								print "\n";			
								next;
							}
							else
							{	
								LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   \"$choice{transfer_https_pxy_user}\" is valid.");
								last;
							}
						}
						
						# get https proxy pwd
						while (1)
						{
							LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   Enter HTTPS proxy password: "); 
							print "   Enter HTTPS proxy password: ";
							chomp ($choice{transfer_https_pxy_pwd} = <STDIN>);
							
							if ($choice{transfer_https_pxy_pwd} =~ m/@|&/)
							{
								LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,4,"CONFIG_USER","   Sorry, \"$choice{transfer_https_pxy_pwd}\" was not a valid choice. Characters \"@\" and \"&\" are not allowed, try again.");
								print "\n";			
								next;
							}
							else
							{	
								LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   \"$choice{transfer_https_pxy_pwd}\" is valid.");
								last;
							}
						}

						# get https proxy options
						LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   Enter HTTPS proxy options: "); 
						print "   Enter HTTPS proxy options: ";
						chomp ($choice{transfer_https_pxy_options} = <STDIN>);
						
						last;
					}
					if ($proxy_chk eq "n")
					{
						LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   \"$proxy_chk\" is valid and proxy seettings will not be done.");
						last;
					}
					else
					{	
						LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   Sorry, \"$proxy_chk\" was not a valid choice, try again.");
						LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   ...please answer \"y\" or \"n\"");
						print "\n";			
						next;
					}
				
				}		
				
				# check settings
				print "\n";
				LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   Provided settings");
				LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   - Transfer mode   : \"$choice{transfer_type}\"");
				LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   - HTTPS token   : \"$choice{transfer_https_token}\"");
				if ($proxy_chk eq "y")
				{
					LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   - HTTPS proxy server   : \"$choice{transfer_https_pxy_server}\"");
					LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   - HTTPS proxy port     : \"$choice{transfer_https_pxy_port}\"");
					LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   - HTTPS proxy protocol : \"$choice{transfer_https_pxy_protocol}\"");
					LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   - HTTPS proxy user     : \"$choice{transfer_https_pxy_user}\"");
					LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   - HTTPS proxy password : \"$choice{transfer_https_pxy_pwd}\"");
					LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   - HTTPS proxy options : \"$choice{transfer_https_pxy_options}\"");
				}
				print "\n"; 
				
			}
			else
			{
				LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,3,"CONFIG_USER","   You should never ever be here. Exit.");
				SAY_BYEBYE($par_log_fh,$par_vbse,$par_lglvl);
			}
			
			while (1)
			{
				LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   Are these setting correct (y/n): "); 
				print "   Are these setting correct (y/n): ";
				chomp ($choice{transfer_correct} = lc <STDIN>);
						
				if ($choice{transfer_correct} eq "y")
				{
					LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   \"$choice{transfer_correct}\" is valid and approved by user.");
					last;
				}
				if ($choice{transfer_correct} eq "n")
				{
					LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   \"$choice{transfer_correct}\" is valid but not approved by user.");
					LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   Starting again.");
					last;
				}
				else
				{	
					LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   Sorry, \"$choice{transfer_correct}\" was not a valid choice, try again.");
					LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,5,"CONFIG_USER","   ...please answer \"y\" or \"n\"");
					print "\n";			
					next;
				}
			}
			
			if ($choice{transfer_correct} eq "y")
			{
				LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   User choice finished.");				
				last;
			}
			else
			{	
				LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,5,"CONFIG_USER","   User choice starting again.");	

				($choice{transfer_type},$choice{transfer_ssh_user},$choice{transfer_https_token},$choice{transfer_https_pxy_protocol}) = ("n/a","n/a","n/a","n/a");
				($choice{transfer_https_pxy_server},$choice{transfer_https_pxy_port},$choice{transfer_https_pxy_user},$choice{transfer_https_pxy_pwd}) = ("n/a","n/a","n/a","n/a");
				$choice{transfer_https_pxy_options} = "n/a";
				$choice{transfer_correct} = "n/a";		
				next;
			}
		}
	}
	
	return %choice;
}

sub SAY_BYEBYE
{
	my $par_log_fh = $_[0];
	my $par_vbse = $_[1];
	my $par_lglvl = $_[2];
	LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,3,"GENERAL","   Please press any key to exit ...");
	my $byebye = <STDIN>;
	LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,3,"GENERAL","   Thank you, goodbye");
	LOG_MSG_CLOSE($log_fh);
	exit;
}

sub CMD_GROUPS
{
	my $par_system = $_[0];
	my $par_log_fh = $_[1];
	my $par_vbse = $_[2];
	my $par_lglvl = $_[3];

	my ($G1500_groups)=<<'G1500_GROUPS';
;  +---------------------------------------------------------------------------------------------------+
group PhyPG Short               ; Parity Groups
group PhyLDEV Short             ; Logical Volumes
group PhyProc Short             ; Micro-Processor usage
group PhyExG                    ; External Volume Group usage
group PhyExLDEV                 ; External Volume usage
group PhyCMPK Short             ; Access Paths and Write Pending
group PhyMPPK                   ; MPPK Performance Information
group PG                        ; Parity Group Statistics
group LDEV                      ; LDEV usage in PGs, External Volume Groups or V-VOL Groups
group Port                      ; Port usage
group MFPort                    ; MFPort usage
group PortWWN                   ; Stats for HBAs connected to ports.
group LU                        ; LDEV usage Summarised by LU Path
;group PPCGWWN                  ; Stats about HBAs
group RemoteCopy                ; Remote Copy Usage Summarized by Subsystem
group RCLU                      ; Remote Copy Usage Summarized by LU path
group RCLDEV                    ; Remote Copy Usage Summarized by LDEV
group UniversalReplicator       ; Remote Copy Usage by UR Summarized by Subsystem
group URJNL                     ; Remote Copy Usage by UR Summarized by Journal Group
group URLU                      ; Remote Copy Usage by UR Summarized by LU Path
group URLDEV                    ; Remote Copy Usage by UR Summarized by LDEV
group LDEVEachOfCU              ; LDEV usage in CUs - Recommended
;  +---------------------------------------------------------------------------------------------------+
shortrange  -1230:              ; Number of monitored CUs is 64 or less
;longrange -600000:
;  +---------------------------------------------------------------------------------------------------+
G1500_GROUPS

# Take care ... range insread of shortrange is required!

	my ($Gx00_groups)=<<'Gx00_GROUPS';
;  +---------------------------------------------------------------------------------------------------+
group PhyPG                     ; Parity Groups
group PhyLDEV                   ; Logical Volumes
group PhyProc                   ; Micro-Processor usage
group PhyExG                    ; External Volume Group usage
group PhyExLDEV                 ; External Volume usage
group PhyMPU                    ; Access Paths and Write Pending
group PG                        ; Parity Group Statistics
group LDEV                      ; LDEV usage in PGs, External Volume Groups or V-VOL Groups
group Port                      ; Port usage
group PortWWN                   ; Stats for HBAs connected to ports.
group LU                        ; LDEV usage Summarised by LU Path
group PPCGWWN                   ; Stats about HBAs
group RemoteCopy                ; Remote Copy Usage Summarized by Subsystem
group RCLU                      ; Remote Copy Usage Summarized by LU path
group RCLDEV                    ; Remote Copy Usage Summarized by LDEV
group UniversalReplicator       ; Remote Copy Usage by UR Summarized by Subsystem
group URJNL                     ; Remote Copy Usage by UR Summarized by Journal Group
group URLU                      ; Remote Copy Usage by UR Summarized by LU Path
group URLDEV                    ; Remote Copy Usage by UR Summarized by LDEV
group LDEVEachOfCU              ; LDEV usage in CUs - Recommended
;  +---------------------------------------------------------------------------------------------------+
range  -1230:                   ; Number of monitored CUs is 64 or less
;  +---------------------------------------------------------------------------------------------------+
Gx00_GROUPS
	
	my ($G1000_groups)=<<'G1000_GROUPS';
;  +---------------------------------------------------------------------------------------------------+
group PhyPG Short               ; Parity Groups
group PhyLDEV Short             ; Logical Volumes
group PhyProc Short             ; Micro-Processor usage
group PhyExG                    ; External Volume Group usage
group PhyExLDEV                 ; External Volume usage
group PhyCMPK Short             ; Access Paths and Write Pending
group PhyMPPK                   ; MPPK Performance Information
group PG                        ; Parity Group Statistics
group LDEV                      ; LDEV usage in PGs, External Volume Groups or V-VOL Groups
group Port                      ; Port usage
group MFPort                    ; MFPort usage
group PortWWN                   ; Stats for HBAs connected to ports.
group LU                        ; LDEV usage Summarised by LU Path
;group PPCGWWN                  ; Stats about HBAs
group RemoteCopy                ; Remote Copy Usage Summarized by Subsystem
group RCLU                      ; Remote Copy Usage Summarized by LU path
group RCLDEV                    ; Remote Copy Usage Summarized by LDEV
group UniversalReplicator       ; Remote Copy Usage by UR Summarized by Subsystem
group URJNL                     ; Remote Copy Usage by UR Summarized by Journal Group
group URLU                      ; Remote Copy Usage by UR Summarized by LU Path
group URLDEV                    ; Remote Copy Usage by UR Summarized by LDEV
group LDEVEachOfCU              ; LDEV usage in CUs - Recommended
;  +---------------------------------------------------------------------------------------------------+
shortrange  -1230:              ; Number of monitored CUs is 64 or less
;longrange -600000:
;  +---------------------------------------------------------------------------------------------------+
G1000_GROUPS

	my ($HUSVM_groups)=<<'HUSVM_GROUPS';
;  +---------------------------------------------------------------------------------------------------+
group PhyPG Short               ; Parity Groups
group PhyLDEV Short             ; Logical Volumes
group PhyProc Short             ; Micro-Processor usage
group PhyExG                    ; External Volume Group usage
group PhyExLDEV                 ; External Volume usage
group PhyMPPK                   ; MPPK Performance Information
group PhyMainPK Short           ; Access Paths and Write Pending
group PG                        ; Parity Group Statistics
group LDEV                      ; LDEV usage in PGs, External Volume Groups or V-VOL Groups
group Port                      ; Port usage
group PortWWN                   ; Stats for HBAs connected to ports.
group LU                        ; LDEV usage Summarised by LU Path
;group PPCGWWN                  ; Stats about HBAs
group RemoteCopy                ; Remote Copy Usage Summarized by Subsystem
group RCLU                      ; Remote Copy Usage Summarized by LU path
group RCLDEV                    ; Remote Copy Usage Summarized by LDEV
group UniversalReplicator       ; Remote Copy Usage by UR Summarized by Subsystem
group URJNL                     ; Remote Copy Usage by UR Summarized by Journal Group
group URLU                      ; Remote Copy Usage by UR Summarized by LU Path
group URLDEV                    ; Remote Copy Usage by UR Summarized by LDEV
group LDEVEachOfCU              ; LDEV usage in CUs - Recommended
;  +---------------------------------------------------------------------------------------------------+
shortrange  -1230:              ; Number of monitored CUs is 64 or less
;longrange -600000:
;  +---------------------------------------------------------------------------------------------------+
HUSVM_GROUPS

	my ($VSP_groups)=<<'VSP_GROUPS';
;  +---------------------------------------------------------------------------------------------------+
group PhyPG Short               ; Parity Groups
group PhyLDEV Short             ; Logical Volumes
group PhyProc Short             ; Micro-Processor usage
group PhyExG                    ; External Volume Group usage
group PhyExLDEV                 ; External Volume usage
group PhyESW Short              ; Access Paths and Write Pending
group PhyMPPK                   ; MPPK Performance Information
group PG                        ; Parity Group Statistics
group LDEV                      ; LDEV usage in PGs, External Volume Groups or V-VOL Groups
group Port                      ; Port usage
group PortWWN                   ; Stats for HBAs connected to ports.
group LU                        ; LDEV usage Summarised by LU Path
;group PPCGWWN                  ; Stats about HBAs
group RemoteCopy                ; Remote Copy Usage Summarized by Subsystem
group RCLU                      ; Remote Copy Usage Summarized by LU path
group RCLDEV                    ; Remote Copy Usage Summarized by LDEV
group UniversalReplicator       ; Remote Copy Usage by UR Summarized by Subsystem
group URJNL                     ; Remote Copy Usage by UR Summarized by Journal Group
group URLU                      ; Remote Copy Usage by UR Summarized by LU Path
group URLDEV                    ; Remote Copy Usage by UR Summarized by LDEV
group LDEVEachOfCU              ; LDEV usage in CUs - Recommended
;  +---------------------------------------------------------------------------------------------------+
shortrange  -1230:              ; Number of monitored CUs is 64 or less
;longrange -600000:
;  +---------------------------------------------------------------------------------------------------+
VSP_GROUPS

	my ($USPV_groups)=<<'USPV_GROUPS';
;  +---------------------------------------------------------------------------------------------------+
group PhyPG Short               ; Parity Groups
group PhyLDEV Short             ; Logical Volumes
group PhyProc                   ; Micro-Processor usage
group PhyExG                    ; External Volume Group usage
group PhyExLDEV                 ; External Volume usage
group PhyCSW                    ; Access Paths and Write Pending
group PG                        ; Parity Group Statistics
group LDEV internal             ; LDEV usage in PGs, External Volume Groups or V-VOL Groups
group Port                      ; Port usage
group PortWWN                   ; Stats for HBAs connected to ports.
group LU                        ; LDEV usage Summarised by LU Path
;group PPCG                     ; SPM/PFC Group usage stats
;group PPCGWWN                  ; Stats about HBAs belonging to SPM/PFC Groups
group RemoteCopy                ; Remote Copy Usage Summarized by Subsystem
group RCLU                      ; Remote Copy Usage Summarized by LU path
group RCLDEV                    ; Remote Copy Usage Summarized by LDEV
group RCCLPR                    ; Remote Copy Usage Summarized by CLPR
group UniversalReplicator       ; Remote Copy Usage by UR Summarized by Subsystem
group URJNL                     ; Remote Copy Usage by UR Summarized by Journal Group
group URLU                      ; Remote Copy Usage by UR Summarized by LU Path
group URLDEV                    ; Remote Copy Usage by UR Summarized by LDEV
group LDEVEachOfCU virtual      ; LDEV usage in CUs
;  +---------------------------------------------------------------------------------------------------+
shortrange  -1200:              ; Number of monitored CUs is 64 or less
;longrange -600000:
;  +---------------------------------------------------------------------------------------------------+
USPV_GROUPS


	if ($par_system eq "G1000") { return $G1000_groups; } 
	elsif ($par_system eq "HUSVM") { return $HUSVM_groups; }
	elsif ($par_system eq "VSP")   { return $VSP_groups; }
	elsif ($par_system eq "USPV")  { return $USPV_groups; }
	elsif ($par_system eq "G800")  { return $Gx00_groups; }
	elsif ($par_system eq "G600")  { return $Gx00_groups; }
	elsif ($par_system eq "G400")  { return $Gx00_groups; }
	elsif ($par_system eq "G200")  { return $Gx00_groups; }
	elsif ($par_system eq "G1500")  { return $G1500_groups; }
	else 
	{
		LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,3,"CMDGROUPS","   !!! Sorry, you should never end up here !!!");
		LOG_MSG($par_log_fh,$par_vbse,"s",$par_lglvl,3,"CMDGROUPS","   !!! Pleaese start the script again !!!");
		SAY_BYEBYE($par_log_fh,$par_vbse,$par_lglvl);
	}
}

sub USAGE
{
	my ($message)=<<'MESSAGE';

NAME ppsetup.pl (part of HiCHperfpkg)

SYNOPSIS
    ppsetup.pl
    ppsetup.pl [-h] [-v]

DESCRIPTION
    This script is doing the setup interactively on the SVP for HiCHperfpkg. 
    It creates the config files, command txt files, bat files and schedules the task.
    What must be done manually?
    - Create performance collection user on the storage system.
    - Copy lib file into correct direcory. ($SCRIPT_HOME/lib/lib_<serial>

REQUIREMENT
     > Install perl (tested with 5.10 and above)
     > Run this script with appropriate privileges => administrator.

OPTIONS

	-h
		Show help message.
			 
	-v
		Displays the version of the script	

FILES
	$SCRIPT_HOME/bin/*
	$SCRIPT_HOME/data/*
	$SCRIPT_HOME/doc/*
	$SCRIPT_HOME/etc/*
	$SCRIPT_HOME/lib/*
	$SCRIPT_HOME/log/*
	$SCRIPT_HOME/spool/*

EXAMPLES
	ppsetup.pl
	ppsetup.pl -h
	ppsetup.pl -v

MESSAGE

	print "Version:${VERSION}\n$message";
}