package HiCH_RESTSVP; # Name of the module and namespace
#!/usr/bin/perl
# Author: Maik Ernst
# History:
# 2016-11-19 - initial module
#
# LICENSE
# Hitachi, Ltd. All rights reserved
# Hitachi Data Systems products and services can be ordered only under the terms and conditions
# of the applicable Hitachi Data Systems agreements.
#
# NAME
# HiCH_RESTSVP
#
# DESCRIPTION
# This module is doing all the RESTAPI stuff against the SVP which is needed
#
# AUTHOR
#   Maik Ernst (maik.ernst@hds.com)
#
# MODIFICATIONS
#   2016-11-19 - v1.0.0		- initial module
#							  Functions: REST_GET_LDEVS, REST_GET_LUPATH, REST_GET_HOSTWWNS, REST_GET_HOSTGROUPS, REST_GET_PARITYGROUPS, 
#										 REST_GET_POOLS, REST_GET_PORTS, REST_GET_POOLS, REST_GET_PORTS, REST_GET_STORAGE, REST_GET_VERSION,
#										 REST_DISCARD_SESSION, REST_CREATE_SESSION, REST_REQUEST, REST_REQUEST_SESSION
#	2018-02-14 - v1.1.0		- REST_GET_POOLS_FMC

use strict;
use warnings;
require Exporter;
use vars qw(@ISA @EXPORT);

#use Config;
use LWP::UserAgent;
use JSON;
use Data::Dumper;
use HiCH_LOG; 
@ISA = qw(Exporter);

# All functions and variables which need to be exported and available in the importing scripts.
@EXPORT = qw(
REST_GET_POOLS_FMC
REST_GET_LDEVS
REST_GET_LUPATH
REST_GET_HOSTWWNS
REST_GET_HOSTGROUPS
REST_GET_PARITYGROUPS
REST_GET_POOLS
REST_GET_PORTS
REST_GET_STORAGE
REST_GET_VERSION
REST_DISCARD_SESSION
REST_CREATE_SESSION
REST_REQUEST
REST_REQUEST_SESSION
);

# REST_GET_POOLS_FMC
# Parameter needed: Token, protocol, SVP IP, StorageID, return format (raw | custom), log file handle (return from LOG_MSG_OPEN)*, verbose mode (true|false)*, log level (normal|debug)*
# *see module HiCH_LOG
# Example: REST_GET_POOLS_FMC("da2fc96e-a9bb-4c4f-80e0-ebbfa8ecf44e","https","127.0.0.1","800000058068","custom",<FH from LOG_MSG_OPEN>,<boolean false>,"normal");
#
# Returns: 
# raw = http|s request raw ($http_decoded)
# custom =	Reference of HASH of HASH (of Array of Hash)
#			REST_pools_fmc{poolId}{tierNumber [{tierUsedCapacity, performanceRate,etc.}],virutalVolumeCapacityRater,tiers,etc.}
#			keys - pool id (0)
#			keys of keys = tierOperationStatus, fmcPoolVolumesCapacityExpansionRate, etc.
#			example
#			'22' => {
#			          'tierOperationStatus' => 'MON',
#			          'fmcPoolVolumesCapacityExpansionRate' => 205,
#			          'usedPhysicalFMCPoolVolumesCapacity' => 85129,
#			          'reservedVolumeCount' => 0,
#			          'availableVolumeCapacity' => 12175212,
#			          'usedPhysicalCapacity' => 85134,
#			          'monitoringMode' => 'CM',
#			          'depletionThreshold' => 99,
#			          'compressionRate' => 77,
#			          'dataReductionRate' => 0,
#			          'locatedVolumeCount' => 7,
#			          'tiers' => [
#			                       {
#			                         'bufferRate' => 2,
#			                         'tierNumber' => 1,
#			                         'tierUsedCapacity' => 385308,
#			                         'performanceRate' => 0,
#			                         'tablespaceRate' => 0,
#			                         'progressOfReplacing' => 100,
#			                         'tierLevelRange' => '00000000',
#			                         'tierDeltaRange' => '00000000',
#			                         'tierTotalCapacity' => 10066224
#			                       },
#			                       {
#			                         'tierNumber' => 2,
#			                         'bufferRate' => 2,
#			                         'tierDeltaRange' => '00000000',
#			                         'tierLevelRange' => '00000000',
#			                         'tierTotalCapacity' => 2494296,
#			                         'progressOfReplacing' => 100,
#			                         'tablespaceRate' => 8,
#			                         'tierUsedCapacity' => 0,
#			                         'performanceRate' => 0
#			                       }
#			                     ],
#			          'usedFMCPoolVolumesCapacity' => 384894,
#			          'duplicationRate' => 0,
#			          'availablePhysicalFMCPoolVolumesCapacity' => 4910262,
#			          'fmcPoolVolumesCapacitySavingRate' => 77,
#			          'numOfLdevs' => 10,
#			          'warningThreshold' => 99,
#			          'duplicationNumber' => 0,
#			          'blockingMode' => 'NB',
#			          'totalPhysicalCapacity' => 7404558,
#			          'dat' => 'VAL',
#			          'poolType' => 'RT',
#			          'dataReductionAccelerateCompRate' => 77,
#			          'dataReductionAccelerateCompCapacity' => 613918692,
#			          'availableFMCPoolVolumesCapacity' => 10066224,
#			          'isShrinking' => $VAR1->{'0'}{'isMainframe'},
#			          'totalReservedCapacity' => 0,
#			          'usedCapacityRate' => 3,
#			          'firstLdevId' => 1792,
#			          'fmcPoolVolumesCapacitySaving' => 299764,
#			          'availablePhysicalVolumeCapacity' => 7319424,
#			          'poolActionMode' => 'AUT',
#			          'poolStatus' => 'POLN',
#			          'poolName' => 'FMC_HDT',
#			          'totalPoolCapacity' => 12560520,
#			          'isMainframe' => $VAR1->{'0'}{'isMainframe'},
#			          'poolId' => 22,
#			          'dataReductionBeforeCapacity' => 0,
#			          'totalLocatedCapacity' => 7341474,
#			          'usedPhysicalCapacityRate' => 1,
#			          'dataReductionCapacity' => 0,
#			          'virtualVolumeCapacityRate' => -1

# REST_GET_LDEVS
# Parameter needed: Token, protocol, SVP IP, StorageID,headLdevId*,count*,return format (raw | custom), log file handle (return from LOG_MSG_OPEN)*, verbose mode (true|false)*, log level (normal|debug)*
# *see module HiCH_LOG
# Example: REST_GET_POOLS("da2fc96e-a9bb-4c4f-80e0-ebbfa8ecf44e","https","127.0.0.1","800000058068","0","1000","custom",<FH from LOG_MSG_OPEN>,<boolean false>,"normal");
# *max count = 1000
# *max ldevs .. (11/2016)
# G200 = 2048
# G400 = 4096
# G600 = 4096
# G800 = 16384
# G1000 = 65280
# G1500 = 65280
# Example for G1000: 
# my $count = 1000;
# for (my $i=0; $i <= 65280; $i = $i + $count)
# {
#	my $ldev_ref_h = REST_GET_LDEVS($Sessiontoken,$protocol,$ip,$storageDeviceId,$i,$count,"custom",$log_fh,$vbse,$lglvl);
#	print Dumper  $ldev_ref_h;
# }
# 
# Returns: 
# raw = http|s request raw ($http_decoded)
# custom =	Reference of HASH of HASH (of Array of Hash)
#			REST_ldevs{ldevId}{blockCapacity,status,ports[{lun,portId,hostGroupName,etc.}],attributes[],emulationType,etc.}
#			keys - ldevId (5122)
#			keys of keys = virtualVolumeCapacityRate,usedCapacityRate,poolId,poolStatus,tierNumber [ {tierUsedCapacity, performanceRate, tablespaceRate, tierDeltaRange, tierTotalCapacity ,progressOfReplacing,tierLevelRange} ],numOfLdevs,
#			monitoringMode,firstLdevId,availableVolumeCapacity,blockingMode,dat,tierOperationStatus,poolActionMode,warningThreshold,isMainframe,poolName,totalLocatedCapacity,
#			depletionThreshold,locatedVolumeCount,isShrinking,totalPoolCapacity
#			keys of keys = ldevId,resourceGroupId,status,emulationType,clprId,blockCapacity,mpBladeId,ports [{lun,portId,hostGroupName,hostGroupNumber}],byteFormatCapacity,label,
#			attributes [],numOfPorts,ssid
#			example '5122' : 'ldevId' => 5122, 'resourceGroupId' => 0,'status' => 'NML','emulationType' => 'OPEN-V-CVS','clprId' => 0,'blockCapacity' => 2147483648,'mpBladeId' => 4,
#			'ports' => [{'lun' => 15,'portId' => 'CL1-B','hostGroupName' => 'peseville_petole','hostGroupNumber' => 6},{'lun' => 2,'portId' => 'CL1-B','hostGroupName' => 'IS_PRE_SNAP_HOST','hostGroupNumber' => 12}],
#			'byteFormatCapacity' => '1.00 T','label' => 'IS_SVOL_SP_2_31521_15382','attributes' => ['CVS','VVOL'],'numOfPorts' => 2,'ssid' => '000F'
 
# REST_GET_LUPATH
# Parameter needed: Token, protocol, SVP IP, StorageID, PortID, hostGroupNumber, return format (raw | custom), log file handle (return from LOG_MSG_OPEN)*, verbose mode (true|false)*, log level (normal|debug)*
# *see module HiCH_LOG
# Example: REST_GET_LUPATH("da2fc96e-a9bb-4c4f-80e0-ebbfa8ecf44e","https","127.0.0.1","800000058068","CL6-D","1","custom",<FH from LOG_MSG_OPEN>,<boolean false>,"normal");
#
# Returns: 
# raw = http|s request raw ($http_decoded)
# custom =	Reference of HASH of HASH (of Array/Hash)
#			$REST_luns{lunId}{hostMode,ldevId,luHostReserve,etc}
#			keys - lunId (CL2-D,1,1)
#			key of key = lunId,portId,hostGroupNumber,hostMode,lun,ldevId,isCommandDevice,luHostReserve{openSystem,persistent,pgrKey,mainframe,acaReserve},hostModeOptions[40,73,etc]
# 			example CL2-D,1,1' : 'hostMode' => 'VMWARE_EX', 'luHostReserve' => {'acaReserve' => 'false','pgrKey' => 'false','openSystem' => 'false','persistent' => 'false','mainframe' => 'false' },
#			'ldevId' => 12289,'isCommandDevice' => 'false' ,'portId' => 'CL2-D','hostGroupNumber' => 1,'lun' => 1,'lunId' => 'CL2-D,1,1','hostModeOptions' => [54,63]

# REST_GET_HOSTWWNS
# Parameter needed: Token, protocol, SVP IP, StorageID, PortID, hostGroupNumber, return format (raw | custom), log file handle (return from LOG_MSG_OPEN)*, verbose mode (true|false)*, log level (normal|debug)*
# *see module HiCH_LOG
# Example: REST_GET_HOSTWWNS("da2fc96e-a9bb-4c4f-80e0-ebbfa8ecf44e","https","127.0.0.1","800000058068","CL6-D","1","custom",<FH from LOG_MSG_OPEN>,<boolean false>,"normal");
#
# Returns: 
# raw = http|s request raw ($http_decoded)
# custom =	Reference of HASH of HASH
#			$REST_hostwwns{hostWwnId}{hostWwn,wwnNickname,hostGroupName,etc]
#			keys - hostWwnId (CL3-B,30,10000090fa19b806)
#			key of key = hostWwn,wwnNickname,portId,hostGroupName,hostGroupNumber,hostWwnId
#			example: CL3-B,30,10000090fa19b806: 'hostWwn' => '10000090fa19b806', 'wwnNickname' => 'pwaruba01','portId' => 'CL3-B', 'hostGroupName' => 'pwaruba', 'hostGroupNumber' => 30, 'hostWwnId' => 'CL3-B,30,10000090fa19b806'
	 
# REST_GET_HOSTGROUPS
# Parameter needed: Token, protocol, SVP IP, StorageID, PortID, return format (raw | custom), log file handle (return from LOG_MSG_OPEN)*, verbose mode (true|false)*, log level (normal|debug)*
# *see module HiCH_LOG
# Example: REST_GET_HOSTGROUPS("da2fc96e-a9bb-4c4f-80e0-ebbfa8ecf44e","https","127.0.0.1","800000058068","CL6-D","custom",<FH from LOG_MSG_OPEN>,<boolean false>,"normal");
#
# Returns: 
# raw = http|s request raw ($http_decoded)
# custom =	Reference of HASH of HASH (of Array)
#			$REST_hostgroups{hostGroupId}{hostGroupId,portId,hostGroupNumber,etc][0-x hostModeOptions]
#			keys - hostGroupId (CL1-A,1)
#			key of key = hostGroupId,portId,hostGroupNumber,hostGroupName,hostMode,hostModeOptions[ 40, 73, etc ]
#			example: CL6-D,1: hostGroupId=CL6-D,1 , portId=CL6-d, hostGroupNumber=1,hostGroupName=hitachiblade6,hostMode=WIN_EX,hostModeOptions=[ 40, 73 ]

# REST_GET_PARITYGROUPS
# Parameter needed: Token, protocol, SVP IP, StorageID, return format (raw | custom), log file handle (return from LOG_MSG_OPEN)*, verbose mode (true|false)*, log level (normal|debug)*
# *see module HiCH_LOG
# Example: REST_GET_PARITYGROUPS("da2fc96e-a9bb-4c4f-80e0-ebbfa8ecf44e","https","127.0.0.1","800000058068",,"custom",<FH from LOG_MSG_OPEN>,<boolean false>,"normal");
#
# Returns: 
# raw = http|s request raw ($http_decoded)
# custom =	Reference of HASH of HASH parity groups
#			$REST_paritygroups{parityGroupId}{usedCapacityRate,raidType,etc = value
#			keys - parityGroupId (1-5)
#			key of key = usedCapacityRate,raidType,parityGroupId,driveType,clprId,numOfLdevs,availableVolumeCapacity,raidLevel
#			example: 1-5: usedCapacityRate=91 raidType=3D+1P parityGroupId=1-5 driveType=SLB5A-M400SS clprId=0 numOfLdevs=4 availableVolumeCapacity=96 raidLevel=RAID5

# REST_GET_POOLS
# Parameter needed: Token, protocol, SVP IP, StorageID, return format (raw | custom), log file handle (return from LOG_MSG_OPEN)*, verbose mode (true|false)*, log level (normal|debug)*
# *see module HiCH_LOG
# Example: REST_GET_POOLS("da2fc96e-a9bb-4c4f-80e0-ebbfa8ecf44e","https","127.0.0.1","800000058068","custom",<FH from LOG_MSG_OPEN>,<boolean false>,"normal");
#
# Returns: 
# raw = http|s request raw ($http_decoded)
# custom =	Reference of HASH of HASH (of Array of Hash)
#			REST_pools{poolId}{tierNumber [{tierUsedCapacity, performanceRate,etc.}],virutalVolumeCapacityRater,tiers,etc.}
#			keys - pool id (0)
#			keys of keys = virtualVolumeCapacityRate,usedCapacityRate,poolId,poolStatus,tierNumber [ {tierUsedCapacity, performanceRate, tablespaceRate, tierDeltaRange, tierTotalCapacity ,progressOfReplacing,tierLevelRange} ],numOfLdevs,
#			monitoringMode,firstLdevId,availableVolumeCapacity,blockingMode,dat,tierOperationStatus,poolActionMode,warningThreshold,isMainframe,poolName,totalLocatedCapacity,
#			depletionThreshold,locatedVolumeCount,isShrinking,totalPoolCapacity
#			example '0' : 'usedCapacityRate' => 48,'dataReductionAccelerateCompCapacity' => 0,'compressionRate' => 0,'dataReductionCapacity' => 0,'monitoringMode' => 'CM','firstLdevId' => 61445,'totalReservedCapacity' => 0,
#			'availableVolumeCapacity' => 3101448,'tierOperationStatus' => 'MON','poolType' => 'RT', 'poolActionMode' => 'AUT', 'warningThreshold' => 70, 'isMainframe' => $VAR1->{'127'}{'isMainframe'}, 'poolName' => 'HDT',
#			'reservedVolumeCount' => 0, 'depletionThreshold' => 85, 'locatedVolumeCount' => 186, 'duplicationNumber' => 0, 'virtualVolumeCapacityRate' => -1, 'poolId' => 0, 'poolStatus' => 'POLN', 'numOfLdevs' => 3, 
#			'tiers' => [ { 'tierUsedCapacity' => 1104012, 'tierNumber' => 1, 'performanceRate' => 7, 'bufferRate' => 2, 'tablespaceRate' => 0, 'tierDeltaRange' => '00000005','tierTotalCapacity' => 1126776,'tierLevelRange' => '00000000', 'progressOfReplacing' => 100},
#			{ 'tierUsedCapacity' => 1864632, 'tierNumber' => 2,'performanceRate' => 0,'bufferRate' => 2,'tablespaceRate' => 8,'tierDeltaRange' => '00000000','tierTotalCapacity' => 4943316, 'tierLevelRange' => '00000000', 'progressOfReplacing' => 100} ],
#			'blockingMode' => 'FB', 'dat' => 'VAL', 'dataReductionBeforeCapacity' => 0, 'duplicationRate' => 0, 'totalLocatedCapacity' => 27013998, 'isShrinking' => bless( do{\(my $o = 1)}, 'JSON::PP::Boolean' ), 'dataReductionAccelerateCompRate' => 0,
#			'dataReductionRate' => 0, 'totalPoolCapacity' => 6070092 },

# REST_GET_PORTS
# Parameter needed: Token, protocol, SVP IP, StorageID, return format (raw | custom), log file handle (return from LOG_MSG_OPEN)*, verbose mode (true|false)*, log level (normal|debug)*
# *see module HiCH_LOG
# Example: REST_GET_PORTS("da2fc96e-a9bb-4c4f-80e0-ebbfa8ecf44e","https",127.0.0.1","800000058068","custom",<FH from LOG_MSG_OPEN>,<boolean false>,"normal");
#
# Returns: 
# raw = http|s request raw ($http_decoded)
# custom =	Reference of HASH of HASH (of Array) Ports
#			$REST_port{portId}{PortType,portAttributes,etc}[0-x portAttributes]
#			keys - port id (CL1-A)
#			key of key = portId, PortType,portAttributes [TAR, MCU, etc.] ,portSpeed,loopId,fabricMode,portConnection,lunSecuritySetting,wwn
#			example: CL1-A: portId=CL1-A portSpeed=8G wwn=50060e8007c5f700 loopId=EF portType=FIBRE portConnection=PtoP lunSecuritySetting=true fabricMode=true portAttributes=[TAR,etc.]

# REST_GET_STORAGE
# Parameter needed: User, password,  protocol (http|https), SVP IP, return format (raw | custom), log file handle (return from LOG_MSG_OPEN)*, verbose mode (true|false)*, log level (normal|debug)*
# *see module HiCH_LOG
# Example: REST_GET_STORAGE("<user RAID>","<pw RAID>","https",127.0.0.1","custom",<FH from LOG_MSG_OPEN>,<boolean false>,"normal");
#
# Returns: 
# raw = http|s request raw ($http_decoded)
# custom =	Reference of HASH
#			keys - svpIp,model,storageDeviceId,serialNumber
#			example values: 10.71.5.112, VSP G1000, 800000050679,50679

# REST_GET_VERSION
# Parameter needed: User, password,  protocol (http|https), SVP IP, return format (raw | custom), log file handle (return from LOG_MSG_OPEN)*, verbose mode (true|false)*, log level (normal|debug)*
# *see module HiCH_LOG
# Example: REST_GET_VERSION(<user RAID>","<pw RAID>","https",127.0.0.1","raw",<FH from LOG_MSG_OPEN>,<boolean false>,"normal");
#
# Returns: 
# raw = http|s request raw ($http_decoded)
# custom =	Scalar
#			Value 1 = Version
#			example: version = 1.3.0

# REST_DISCARD_SESSION
# Parameter needed: Token, SessionID, protocol (http|https), SVP IP, StorageID, return format (raw), log file handle (return from LOG_MSG_OPEN)*, verbose mode (true|false)*, log level (normal|debug)*
# *see module HiCH_LOG
# Example: REST_DISCARD_SESSION("da2fc96e-a9bb-4c4f-80e0-ebbfa8ecf44e","49""https",127.0.0.1","800000058068","raw",<FH from LOG_MSG_OPEN>,<boolean false>,"normal");
#
# Returns: 
# raw = http|s request raw ($http_decoded)

# REST_CREATE_SESSION
# Parameter needed: User, password, protocol (http|https), SVP IP, StorageID, return format (raw | custom), log file handle (return from LOG_MSG_OPEN)*, verbose mode (true|false)*, log level (normal|debug)*
# *see module HiCH_LOG
# Example: REST_CREATE_SESSION(<user RAID>","<pw RAID>","https",127.0.0.1","800000058068","raw",<FH from LOG_MSG_OPEN>,<boolean false>,"normal");
#
# Returns: 
# raw = http|s request raw ($http_decoded)
# custom = Scalars
#		   Value 1 = Session token, Value 2 = SessionID
#          example:  token = 170decb4-838e-480e-888e-ab17f4ab63fa, sessionId =49

# REST_REQUEST
# Parameter needed: User, password, protocol (http|https), SVP IP, method (GET|POST|DELETE..), URL, log file handle (return from LOG_MSG_OPEN)*, verbose mode (true|false)*, log level (normal|debug)*
# *see module HiCH_LOG
# Example: REST_REQUEST("<user RAID>","<pw RAID>","https",127.0.0.1","GET","ConfigurationManager/v1/objects/storages/",<FH from LOG_MSG_OPEN>,<boolean false>,"normal");
#
# Returns: $http_status_code, $http_decoded

# REST_REQUEST_SESSION
# Parameter needed: Session token, protocol (http|https), SVP IP, method (GET|POST|DELETE) , URL, log file handle (return from LOG_MSG_OPEN)*, verbose mode (true|false)*, log level (normal|debug)*
# *see module HiCH_LOG
# Example: REST_REQUEST_SESSION("da2fc96e-a9bb-4c4f-80e0-ebbfa8ecf44e","https",127.0.0.1","GET","ConfigurationManager/v1/objects/storages/800000058068/parity-groups",<FH from LOG_MSG_OPEN>,<boolean false>,"normal");
#
# Returns: $http_status_code, $http_decoded


################# Start #####################
sub REST_GET_POOLS_FMC
{
	my $par_token = $_[0];
	my $par_protocol = $_[1];
	my $par_ip = $_[2];
	my $par_storageid = $_[3];
	my $par_return = $_[4];
	my $par_log_fh = $_[5];
	my $par_vbse = $_[6];
	my $par_lglvl = $_[7];
	
	my $httpmethod = "GET";
	my $url_path = "ConfigurationManager/v1/objects/storages/${par_storageid}/pools?detailInfoType=FMC";
	
	my %REST_pools_fmc;
	
	my $http_status_code;
	my $http_decoded;
	
	my $json_obj = JSON->new->utf8;
	my $json_decoded = "n/a";
	
	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPOOLSFMC","Get pools start");
	
	($http_status_code,$http_decoded) = REST_REQUEST_SESSION($par_token,$par_protocol,$par_ip,$httpmethod,$url_path,$par_log_fh,$par_vbse,$par_lglvl);
	
	if ($http_status_code eq "200")
	{

		if ($par_return eq "raw")
		{
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPOOLSFMC","Return parameter: $par_return");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPOOLSFMC","Get pools successful");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPOOLSFMC","Get pools finished");
			return $http_decoded;
		}
		elsif ($par_return eq "custom")
		{
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPOOLSFMC","Return parameter: $par_return");
			
			$json_decoded = $json_obj->decode($http_decoded);
	
			# Parse data example
			#
			#"poolId" : 22,
			#
			#"poolStatus" : "POLN",
			#
			#"usedCapacityRate" : 3,
			#
			#"usedPhysicalCapacityRate" : 1,
			#
			#"poolName" : "FMC_HDT",
			#
			#"availableVolumeCapacity" : 12175212,
			#
			#"availablePhysicalVolumeCapacity" : 7319424,
			#
			#"usedPhysicalCapacity" : 85134,
			#
			#"totalPoolCapacity" : 12560520,
			#
			#"totalPhysicalCapacity" : 7404558,
			#
			#"numOfLdevs" : 10,
			#
			#"firstLdevId" : 1792,
			#
			#"warningThreshold" : 99,
			#
			#"depletionThreshold" : 99,
			#
			#"virtualVolumeCapacityRate" : -1,
			#
			#"isMainframe" : false,
			#
			#"isShrinking" : false,
			#
			#"locatedVolumeCount" : 7,
			#
			#"totalLocatedCapacity" : 7341474,
			#
			#"blockingMode" : "NB",
			#
			#"totalReservedCapacity" : 0,
			#
			#"reservedVolumeCount" : 0,
			#
			#"poolActionMode" : "AUT",
			#
			#"tierOperationStatus" : "MON",
			#
			#"dat" : "VAL",
			#
			#"poolType" : "RT",
			#
			#"monitoringMode" : "CM",
			#
			#"tiers" : [ {
			#
			#  "tierNumber" : 1,
			#
			#  "tierLevelRange" : "00000000",
			#
			#  "tierDeltaRange" : "00000000",
			#
			#  "tierUsedCapacity" : 385308,
			#
			#  "tierTotalCapacity" : 10066224,
			#
			#  "tablespaceRate" : 0,
			#
			#  "performanceRate" : 0,
			#
			#  "progressOfReplacing" : 100,
			#
			#  "bufferRate" : 2
			#
			#}, {
			#
			#  "tierNumber" : 2,
			#
			#  "tierLevelRange" : "00000000",
			#
			#  "tierDeltaRange" : "00000000",
			#
			#  "tierUsedCapacity" : 0,
			#
			#  "tierTotalCapacity" : 2494296,
			#
			#  "tablespaceRate" : 8,
			#
			#  "performanceRate" : 0,
			#
			#  "progressOfReplacing" : 100,
			#
			#  "bufferRate" : 2
			#
			#} ],
			#
			#"duplicationNumber" : 0,
			#
			#"dataReductionAccelerateCompCapacity" : 613918692,
			#
			#"dataReductionCapacity" : 0,
			#
			#"dataReductionBeforeCapacity" : 0,
			#
			#"dataReductionAccelerateCompRate" : 77,
			#
			#"duplicationRate" : 0,
			#
			#"compressionRate" : 77,
			#
			#"dataReductionRate" : 0,
			#
			#"availablePhysicalFMCPoolVolumesCapacity" : 4910262,
			#
			#"usedPhysicalFMCPoolVolumesCapacity" : 85129,
			#
			#"availableFMCPoolVolumesCapacity" : 10066224,
			#
			#"usedFMCPoolVolumesCapacity" : 384894,
			#
			#"fmcPoolVolumesCapacitySaving" : 299764,
			#
			#"fmcPoolVolumesCapacitySavingRate" : 77,
			#
			#"fmcPoolVolumesCapacityExpansionRate" : 205
			
			
			for my $key1 (keys %{ $json_decoded })
			{
				# key1 = data
				# print "k1 $key1\n";
				for my $key2 (0..$#{ $json_decoded->{$key1} }) 
				{
					# key2 = 0 .. x
					# print "k2 $key2\n";
					for my $key3 (keys %{ $json_decoded->{$key1}[$key2] }) 
					{
						# key3 = poolId,poolStatus,etc
						# print "k3 $key3\n";
						if ($key3 eq "tiers")
						{
							for my $key4 (0..$#{ $json_decoded->{$key1}[$key2]{$key3} })
							{
								# key4 = 0 .. x 
								# print "k4 $key4 .. $json_decoded->{$key1}[$key2]{$key3}[$key4]\n";
								# REST_pools_fmc{poolId}{tierNumber [{tierUsedCapacity, performanceRate,etc.}],virutalVolumeCapacityRater,tiers,etc.}
								push(@{$REST_pools_fmc{$json_decoded->{$key1}[$key2]{poolId}}{$key3}}, $json_decoded->{$key1}[$key2]{$key3}[$key4]);
							}
						}										
						else
						{
							$REST_pools_fmc{$json_decoded->{$key1}[$key2]{poolId}}{$key3} = $json_decoded->{$key1}[$key2]{$key3};
						}
        
					}
				}
			}
			
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPOOLSFMC","Get pools successful");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPOOLSFMC","Get pools finished");
			return	\%REST_pools_fmc;
		}
		else
		{
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTGETPOOLSFMC","Return parameter: $par_return");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTGETPOOLSFMC","return parameter unknown. Returning raw http data");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTGETPOOLSFMC","Get pools successful");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTGETPOOLSFMC","Get pools finished");
			return $http_decoded;
		}
		
	}
	else
	{
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,3,"RESTGETPOOLSFMC","Get pools NOT successful");
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,3,"RESTGETPOOLSFMC","Get pools finished");
		return -1;
	}
	
}

sub REST_GET_LDEVS
{
	my $par_token = $_[0];
	my $par_protocol = $_[1];
	my $par_ip = $_[2];
	my $par_storageid = $_[3];
	my $par_headldevid = $_[4];
	my $par_count = $_[5];
	my $par_return = $_[6];
	my $par_log_fh = $_[7];
	my $par_vbse = $_[8];
	my $par_lglvl = $_[9];
	
	my $httpmethod = "GET";
	my $url_path = "ConfigurationManager/v1/objects/storages/${par_storageid}/ldevs?headLdevId=${par_headldevid}&count=${par_count}";
	
	my %REST_ldevs;
	
	my $http_status_code;
	my $http_decoded;
	
	my $json_obj = JSON->new->utf8;
	my $json_decoded = "n/a";
	
	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPGROUPS","Get ldevs start");

	($http_status_code,$http_decoded) = REST_REQUEST_SESSION($par_token,$par_protocol,$par_ip,$httpmethod,$url_path,$par_log_fh,$par_vbse,$par_lglvl);
	
	if ($http_status_code eq "200")
	{
		if ($par_return eq "raw")
		{
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPGROUPS","Return parameter: $par_return");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPGROUPS","Get ldevs successful");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPGROUPS","Get ldevs finished");
			return $http_decoded;
		}
		elsif ($par_return eq "custom")
		{
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPGROUPS","Return parameter: $par_return");
			
			$json_decoded = $json_obj->decode($http_decoded);
			#
			#			
			#			 "data" : [ {
			# "ldevId" : 1000,
			# "emulationType" : "NOT DEFINED"
			# "resourceGroupId" : 0
			# }, {
			#  "ldevId" : 3,
			#  "clprId" : 0,
			#  "emulationType" : "OPEN-V-CVS-CM",
			#  "byteFormatCapacity" : "50.00 M",
			#  "blockCapacity" : 102400,
			#  "numOfPorts" : 2,
			#  "ports" : [ {
			#    "portId" : "CL1-B",
			#    "hostGroupNumber" : 6,
			#    "hostGroupName" : "peseville_petole",
			#    "lun" : 30
			#  }, {
			#    "portId" : "CL2-B",
			#    "hostGroupNumber" : 6,
			#    "hostGroupName" : "peseville_petole",
			#    "lun" : 4
			#  } ],
			#  "attributes" : [ "CMD", "CVS", "HDT" ],
			#  "label" : "CMDDev_4_Probeserver",
			#  "status" : "NML",
			#  "operationType" : "RBL",
			#  "preparingOperationProgressRate" : 100,
			#  "mpBladeId" : 4,
			#  "ssid" : "000B",
			#  "poolId" : 0,
			#  "numOfUsedBlock" : 0,
			#  "isRelocationEnabled" : true,
			#  "tierLevel" : "all",
			#  "usedCapacityPerTierLevel1" : 0,
			#  "usedCapacityPerTierLevel2" : 0,
			#  "tierLevelForNewPageAllocation" : "M",
			#  "isFullAllocationEnabled" : false,
			#  "resourceGroupId" : 0,
			#  "dataReductionStatus" : "DISABLED",
			#  "dataReductionMode" : "disabled"
			#}, {
			#				
		
			for my $key1 (keys %{ $json_decoded })
			{
				# key1 = data
				#print "k1 $key1\n";
				# Important!! (0..$#{ $json_decoded->{$key1} }) must be used as (keys @{ $json_decoded->{$key1} }) does not work on old and new perl versions
				for my $key2 (0..$#{ $json_decoded->{$key1} }) 
				{
					# key2 = 0 .. x
					#print "k2 $key2\n";
					for my $key3 (keys %{ $json_decoded->{$key1}[$key2] }) 
					{
						# key3 = ldevId,attributes,ports,blockCapacity,etc
						#print "k3 $key3\n";
						if (($key3 eq "ports") or ($key3 eq "attributes"))
						{
							for my $key4 (0..$#{ $json_decoded->{$key1}[$key2]{$key3} })
							{
								# key4 = 0 .. x 
								#print "k4 $key4 .. $json_decoded->{$key1}[$key2]{$key3}[$key4]\n";
								#REST_ldevs{ldevId}{blockCapacity,status,ports[{lun,portId,hostGroupName,etc.}],attributes[],emulationType,etc.}
								push(@{$REST_ldevs{$json_decoded->{$key1}[$key2]{ldevId}}{$key3}}, $json_decoded->{$key1}[$key2]{$key3}[$key4]);
							}
						}	
						else
						{
							$REST_ldevs{$json_decoded->{$key1}[$key2]{ldevId}}{$key3} = $json_decoded->{$key1}[$key2]{$key3};
						}

					}
				}
			}
			
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPGROUPS","Get ldevs successful");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPGROUPS","Get ldevs finished");
			return	\%REST_ldevs;
		}
		else
		{
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTGETPGROUPS","Return parameter: $par_return");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTGETPGROUPS","Return parameter unknown. Returning raw http data");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTGETPGROUPS","Get ldevs successful");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTGETPGROUPS","Get ldevs finished");
			return $http_decoded;
		}
		
	}
	else
	{
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,3,"RESTGETPGROUPS","Get ldevs NOT successful");
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,3,"RESTGETPGROUPS","Get ldevs finished");
		return -1;
	}
	
}

sub REST_GET_LUPATH
{
	my $par_token = $_[0];
	my $par_protocol = $_[1];
	my $par_ip = $_[2];
	my $par_storageid = $_[3];
	my $par_portid = $_[4];
	my $par_hostGroupNumber = $_[5];
	my $par_return = $_[6];
	my $par_log_fh = $_[7];
	my $par_vbse = $_[8];
	my $par_lglvl = $_[9];
	
	my $httpmethod = "GET";
	my $url_path = "ConfigurationManager/v1/objects/storages/${par_storageid}/luns?portId=${par_portid}&hostGroupNumber=${par_hostGroupNumber}";
	
	my %REST_luns;
	
	my $http_status_code;
	my $http_decoded;
	
	my $json_obj = JSON->new->utf8;
	my $json_decoded = "n/a";
	
	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPGROUPS","Get luns - port ${par_portid} / hostGroupNumber ${par_hostGroupNumber} - start");

	($http_status_code,$http_decoded) = REST_REQUEST_SESSION($par_token,$par_protocol,$par_ip,$httpmethod,$url_path,$par_log_fh,$par_vbse,$par_lglvl);
	
	if ($http_status_code eq "200")
	{
		if ($par_return eq "raw")
		{
			#print $http_decoded;
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPGROUPS","Return parameter: $par_return");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPGROUPS","Get luns - port ${par_portid} / hostGroupNumber ${par_hostGroupNumber} - successful");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPGROUPS","Get luns - port ${par_portid} / hostGroupNumber ${par_hostGroupNumber} - finished");
			return $http_decoded;
		}
		elsif ($par_return eq "custom")
		{
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPGROUPS","Return parameter: $par_return");
			
			$json_decoded = $json_obj->decode($http_decoded);
			
			# Parse data example
			#{
			#"data" : [ {
			#  "lunId" : "CL1-C,1,0",
			#  "portId" : "CL1-C",
			#  "hostGroupNumber" : 1,
			#  "hostMode" : "VMWARE_EX",
			#  "lun" : 0,
			#  "ldevId" : 65184,
			#  "isCommandDevice" : false,
			#  "luHostReserve" : {
			#    "openSystem" : false,
			#    "persistent" : false,
			#    "pgrKey" : false,
			#    "mainframe" : false,
			#    "acaReserve" : false
			#  },
			#  "hostModeOptions" : [ 54, 63 ]
			#}, {
			#  "lunId" : "CL1-C,1,1",
			#  "portId" : "CL1-C",
			#  "hostGroupNumber" : 1,
			#  "hostMode" : "VMWARE_EX",
			#  "lun" : 1,
			#  "ldevId" : 65185,
			#  "isCommandDevice" : false,
			#  "luHostReserve" : {
			#    "openSystem" : false,
			#    "persistent" : false,
			#    "pgrKey" : false,
			#    "mainframe" : false,
			#    "acaReserve" : false
			#  },
			#  "hostModeOptions" : [ 54, 63 ]
			
			
									
			for my $key1 (keys %{ $json_decoded })
			{
				# key1 = data
				#print "k1 $key1\n";
				for my $key2 (0..$#{ $json_decoded->{$key1} }) 
				{
					#print "k2 $key2\n";
					# key2 = 0 .. x
					for my $key3 (keys %{ $json_decoded->{$key1}[$key2] }) 
					{
						#print "k3 $key3\n";
						# key3 = lunId,portId,hostGroupNumber,hostMode,ldevId,luHostReserve,etx
						
						## Check if hostModeOptions, if so make an array an add values
						if ($key3 eq "hostModeOptions")
						{
							for my $key4 (0..$#{ $json_decoded->{$key1}[$key2]{$key3} })
							{
								# key4 = 0 .. x which are "hostModeOptions" : >>>> [ 40, 73 ] <<<<
								push(@{$REST_luns{$json_decoded->{$key1}[$key2]{lunId}}{$key3}}, $json_decoded->{$key1}[$key2]{$key3}[$key4]);
							}
						}
						elsif ($key3 eq "luHostReserve")
						{
							for my $key4 (keys %{ $json_decoded->{$key1}[$key2]{$key3} })
							{
								# key4 = openSystem,persistent,pgrKey,mainframe,acaReserve
								# $REST_luns{lunId}{hostMode,ldevId,luHostReserve,etc}
								$REST_luns{$json_decoded->{$key1}[$key2]{lunId}}{$key3}{$key4} = $json_decoded->{$key1}[$key2]{$key3}{$key4};
							}
							
						}
						else
						{
							$REST_luns{$json_decoded->{$key1}[$key2]{lunId}}{$key3} = $json_decoded->{$key1}[$key2]{$key3};
						}
					}
				}
			}
									
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPGROUPS","Get luns successful");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPGROUPS","Get luns finished");
			return	\%REST_luns;
			
		}
		else
		{
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTGETPGROUPS","Return parameter: $par_return");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTGETPGROUPS","Return parameter unknown. Returning raw http data");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTGETPGROUPS","Get luns - port ${par_portid} / hostGroupNumber ${par_hostGroupNumber} - successful");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTGETPGROUPS","Get luns - port ${par_portid} / hostGroupNumber ${par_hostGroupNumber} - finished");
			return $http_decoded;
		}
		
	}
	else
	{
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,3,"RESTGETPGROUPS","Get luns - port ${par_portid} / hostGroupNumber ${par_hostGroupNumber} - NOT successful");
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,3,"RESTGETPGROUPS","Get luns- port ${par_portid} / hostGroupNumber ${par_hostGroupNumber} - finished");
		return -1;
	}
	
}

sub REST_GET_HOSTWWNS
{
	my $par_token = $_[0];
	my $par_protocol = $_[1];
	my $par_ip = $_[2];
	my $par_storageid = $_[3];
	my $par_portid = $_[4];
	my $par_hostGroupNumber = $_[5];
	my $par_return = $_[6];
	my $par_log_fh = $_[7];
	my $par_vbse = $_[8];
	my $par_lglvl = $_[9];
	
	my $httpmethod = "GET";
	my $url_path = "ConfigurationManager/v1/objects/storages/${par_storageid}/host-wwns?portId=${par_portid}&hostGroupNumber=${par_hostGroupNumber}";
	
	my %REST_hostwwns;
	
	my $http_status_code;
	my $http_decoded;
	
	my $json_obj = JSON->new->utf8;
	my $json_decoded = "n/a";
	
	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPGROUPS","Get host-wwns - port ${par_portid} / hostGroupNumber ${par_hostGroupNumber} - start");

	($http_status_code,$http_decoded) = REST_REQUEST_SESSION($par_token,$par_protocol,$par_ip,$httpmethod,$url_path,$par_log_fh,$par_vbse,$par_lglvl);
	
	if ($http_status_code eq "200")
	{
		if ($par_return eq "raw")
		{
			#print $http_decoded;
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPGROUPS","Return parameter: $par_return");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPGROUPS","Get host-wwns - port ${par_portid} / hostGroupNumber ${par_hostGroupNumber} - successful");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPGROUPS","Get host-wwns - port ${par_portid} / hostGroupNumber ${par_hostGroupNumber} - finished");
			return $http_decoded;
		}
		elsif ($par_return eq "custom")
		{
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPGROUPS","Return parameter: $par_return");
			
			$json_decoded = $json_obj->decode($http_decoded);
			
			# Parse data example
			#{
			#"data" : [ {
			#"hostWwnId" : "CL3-B,7,24000000878058d8",
			#"portId" : "CL3-B",
			#"hostGroupNumber" : 7,
			#"hostGroupName" : "peseville_petoledo_pegranada",
			#"hostWwn" : "24000000878058d8",
			#"wwnNickname" : "petoledo_hba1"
			#}, {
			#"hostWwnId" : "CL3-B,7,24000000878058c8",
			#"portId" : "CL3-B",
			#"hostGroupNumber" : 7,
			#"hostGroupName" : "peseville_petoledo_pegranada",
			#"hostWwn" : "24000000878058c8",
			#"wwnNickname" : "peseville_hba1"
			#}, 
			
						
			for my $key1 (keys %{ $json_decoded })
			{
				# key1 = data
				#print "k1 $key1\n";
				for my $key2 (0..$#{ $json_decoded->{$key1} }) 
				{
					#print "k2 $key2\n";
					# key2 = 0 .. x
					for my $key3 (keys %{ $json_decoded->{$key1}[$key2] }) 
					{
						#print "k3 $key3\n";
						# key3 = hostWwnId,portId,hostGroupNumber,etc
						# $REST_hostwwns{hostWwnId}{hostWwn,wwnNickname,hostGroupName,etc]
						$REST_hostwwns{$json_decoded->{$key1}[$key2]{hostWwnId}}{$key3} = $json_decoded->{$key1}[$key2]{$key3};
					}
				}
			}
									
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPGROUPS","Get host-wwns successful");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPGROUPS","Get host-wwns finished");
			return	\%REST_hostwwns;
			
		}
		else
		{
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTGETPGROUPS","Return parameter: $par_return");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTGETPGROUPS","Return parameter unknown. Returning raw http data");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTGETPGROUPS","Get host-wwns - port ${par_portid} / hostGroupNumber ${par_hostGroupNumber} - successful");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTGETPGROUPS","Get host-wwns - port ${par_portid} / hostGroupNumber ${par_hostGroupNumber} - finished");
			return $http_decoded;
		}
		
	}
	else
	{
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,3,"RESTGETPGROUPS","Get host-wwns - port ${par_portid} / hostGroupNumber ${par_hostGroupNumber} - NOT successful");
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,3,"RESTGETPGROUPS","Get host-wwns - port ${par_portid} / hostGroupNumber ${par_hostGroupNumber} - finished");
		return -1;
	}
	
}

sub REST_GET_HOSTGROUPS
{
	my $par_token = $_[0];
	my $par_protocol = $_[1];
	my $par_ip = $_[2];
	my $par_storageid = $_[3];
	my $par_portid = $_[4];
	my $par_return = $_[5];
	my $par_log_fh = $_[6];
	my $par_vbse = $_[7];
	my $par_lglvl = $_[8];
	
	my $httpmethod = "GET";
	my $url_path = "ConfigurationManager/v1/objects/storages/${par_storageid}/host-groups?portId=${par_portid}";
	
	my %REST_hostgroups;
	
	my $http_status_code;
	my $http_decoded;
	
	my $json_obj = JSON->new->utf8;
	my $json_decoded = "n/a";
	
	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPGROUPS","Get host-groups - port ${par_portid} - start");

	($http_status_code,$http_decoded) = REST_REQUEST_SESSION($par_token,$par_protocol,$par_ip,$httpmethod,$url_path,$par_log_fh,$par_vbse,$par_lglvl);
	
	if ($http_status_code eq "200")
	{
		if ($par_return eq "raw")
		{
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPGROUPS","Return parameter: $par_return");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPGROUPS","Get host-groups - port ${par_portid} - successful");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPGROUPS","Get host-groups - port ${par_portid} - finished");
			return $http_decoded;
		}
		elsif ($par_return eq "custom")
		{
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPGROUPS","Return parameter: $par_return");
			
			$json_decoded = $json_obj->decode($http_decoded);

			# Parse data example
			#{
			#  "data" : [ {
			#    "hostGroupId" : "CL6-D,0",
			#    "portId" : "CL6-D",
			#    "hostGroupNumber" : 0,
			#    "hostGroupName" : "6D-G00",
			#    "hostMode" : "LINUX/IRIX"
			#  }, {
			#    "hostGroupId" : "CL6-D,1",
			#    "portId" : "CL6-D",
			#    "hostGroupNumber" : 1,
			#    "hostGroupName" : "hitachiblade6",
			#    "hostMode" : "WIN_EX",
			#    "hostModeOptions" : [ 40, 73 ]
			#  } ]
			#}
			#
			#			
			for my $key1 (keys %{ $json_decoded })
			{
				# key1 = data
				for my $key2 (0..$#{ $json_decoded->{$key1} }) 
				{
					# key2 = 0 .. x
					for my $key3 (keys %{ $json_decoded->{$key1}[$key2] }) 
					{
					
					## Check if hostModeOptions, if so make an array an add values
						if ($key3 eq "hostModeOptions")
						{
							# key3 = hostGroupId,portId,hostGroupNumberetc
							for my $key4 (0..$#{ $json_decoded->{$key1}[$key2]{$key3} })
							{
								# key4 = 0 .. x which are "hostModeOptions" : >>>> [ 40, 73 ] <<<<
								# $REST_hostgroups{hostGroupId}{hostGroupId,portId,hostGroupNumber,etc][0-x hostModeOptions]
								push(@{$REST_hostgroups{$json_decoded->{$key1}[$key2]{hostGroupId}}{$key3}}, $json_decoded->{$key1}[$key2]{$key3}[$key4]);
							}
						}
						else
						{
							$REST_hostgroups{$json_decoded->{$key1}[$key2]{hostGroupId}}{$key3} = $json_decoded->{$key1}[$key2]{$key3};
						}
					}
				}
			}
						
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPGROUPS","Get host-groups - port ${par_portid} - successful");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPGROUPS","Get host-groups - port ${par_portid} - finished");
			return	\%REST_hostgroups;
		}
		else
		{
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTGETPGROUPS","Return parameter: $par_return");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTGETPGROUPS","Return parameter unknown. Returning raw http data");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTGETPGROUPS","Get host-groups - port ${par_portid} - successful");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTGETPGROUPS","Get host-groups - port ${par_portid} - finished");
			return $http_decoded;
		}
		
	}
	else
	{
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,3,"RESTGETPGROUPS","Get host-groups - port ${par_portid} - NOT successful");
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,3,"RESTGETPGROUPS","Get host-groups - port ${par_portid} - finished");
		return -1;
	}
	
}

sub REST_GET_PARITYGROUPS
{
	my $par_token = $_[0];
	my $par_protocol = $_[1];
	my $par_ip = $_[2];
	my $par_storageid = $_[3];
	my $par_return = $_[4];
	my $par_log_fh = $_[5];
	my $par_vbse = $_[6];
	my $par_lglvl = $_[7];
	
	my $httpmethod = "GET";
	my $url_path = "ConfigurationManager/v1/objects/storages/${par_storageid}/parity-groups";
	
	my %REST_paritygroups;
	
	my $http_status_code;
	my $http_decoded;
	
	my $json_obj = JSON->new->utf8;
	my $json_decoded = "n/a";
	
	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPGROUPS","Get parity groups start");

	($http_status_code,$http_decoded) = REST_REQUEST_SESSION($par_token,$par_protocol,$par_ip,$httpmethod,$url_path,$par_log_fh,$par_vbse,$par_lglvl);
	
	if ($http_status_code eq "200")
	{
		if ($par_return eq "raw")
		{
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPGROUPS","Return parameter: $par_return");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPGROUPS","Get parity groups successful");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPGROUPS","Get parity groups finished");
			return $http_decoded;
		}
		elsif ($par_return eq "custom")
		{
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPGROUPS","Return parameter: $par_return");
			
			$json_decoded = $json_obj->decode($http_decoded);

			
			# Parse data example
			#			{
			#
			#"data" : [ {
			#
			#	"parityGroupId" : "1-1",
			#
			#	"numOfLdevs" : 2,
			#
			#	"usedCapacityRate" : 97,
			#
			#	"availableVolumeCapacity" : 131,
			#
			#	"raidLevel" : "RAID6",
			#
			#	"raidType" : "6D+2P",
			#
			#	"clprId" : 0,
			#
			#	"driveType" : "DKR5D-J900SS"
			#
			#}, {
			#
			#	"parityGroupId" : "1-2",
			
			for my $key1 (keys %{ $json_decoded })
			{
				# key1 = data
				#print "k1 $key1\n";
				for my $key2 (0..$#{ $json_decoded->{$key1} }) 
				{
					#print "k2 $key2\n";
					# key2 = 0 .. x
					for my $key3 (keys %{ $json_decoded->{$key1}[$key2] }) 
					{
						#print "k3 $key3\n";
						# key3 = parityGroupId,numOfLdevs,etc
						# $REST_paritygroups{parityGroupId}{usedCapacityRate,raidType,etc = value
						$REST_paritygroups{$json_decoded->{$key1}[$key2]{parityGroupId}}{$key3} = $json_decoded->{$key1}[$key2]{$key3};
					}
				}
			}
						
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPGROUPS","Get parity groups successful");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPGROUPS","Get parity groups finished");
			return	\%REST_paritygroups;
		}
		else
		{
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTGETPGROUPS","Return parameter: $par_return");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTGETPGROUPS","Return parameter unknown. Returning raw http data");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTGETPGROUPS","Get parity groups successful");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTGETPGROUPS","Get parity groups finished");
			return $http_decoded;
		}
		
	}
	else
	{
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,3,"RESTGETPGROUPS","Get parity groups NOT successful");
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,3,"RESTGETPGROUPS","Get parity groups finished");
		return -1;
	}
	
}

sub REST_GET_POOLS
{
	my $par_token = $_[0];
	my $par_protocol = $_[1];
	my $par_ip = $_[2];
	my $par_storageid = $_[3];
	my $par_return = $_[4];
	my $par_log_fh = $_[5];
	my $par_vbse = $_[6];
	my $par_lglvl = $_[7];
	
	my $httpmethod = "GET";
	my $url_path = "ConfigurationManager/v1/objects/storages/${par_storageid}/pools";
	
	my %REST_pools;
	
	my $http_status_code;
	my $http_decoded;
	
	my $json_obj = JSON->new->utf8;
	my $json_decoded = "n/a";
	
	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPOOLS","Get pools start");
	
	($http_status_code,$http_decoded) = REST_REQUEST_SESSION($par_token,$par_protocol,$par_ip,$httpmethod,$url_path,$par_log_fh,$par_vbse,$par_lglvl);
	
	if ($http_status_code eq "200")
	{

		if ($par_return eq "raw")
		{
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPOOLS","Return parameter: $par_return");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPOOLS","Get pools successful");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPOOLS","Get pools finished");
			return $http_decoded;
		}
		elsif ($par_return eq "custom")
		{
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPOOLS","Return parameter: $par_return");
			
			$json_decoded = $json_obj->decode($http_decoded);
			
			# Parse data example
			#			{
			#
			#"data" : [ {
			#
			#	"poolId" : 0,
			#
			#	"poolStatus" : "POLN",
			#
			#	"usedCapacityRate" : 31,
			#
			#	"poolName" : "HDT",
			#
			#	"availableVolumeCapacity" : 5136936,
			#
			#	"totalPoolCapacity" : 7471968,
			#
			#	"numOfLdevs" : 4,
			#
			#	"firstLdevId" : 65024,
			#
			#	"warningThreshold" : 70,
			#
			#	"depletionThreshold" : 80,
			#
			#	"virtualVolumeCapacityRate" : -1,
			#
			#	"isMainframe" : false,
			#
			#	"isShrinking" : false,
			#
			#	"locatedVolumeCount" : 149,
			#
			#	"totalLocatedCapacity" : 12643994,
			#
			#	"blockingMode" : "NB",
			#
			#	"poolActionMode" : "AUT",
			#
			#	"tierOperationStatus" : "MON",
			#
			#	"dat" : "VAL",
			#
			#	"monitoringMode" : "CM",
			#
			#	"tiers" : [ {
			#
			#	"tierNumber" : 1,
			#
			#	"tierLevelRange" : "00000001",
			#
			#	"tierDeltaRange" : "00000005",
			#
			#	"tierUsedCapacity" : 251874,
			#
			#	"tierTotalCapacity" : 256998,
			#
			#	"tablespaceRate" : 0,
			#
			#	"performanceRate" : 0,
			#
			#	"progressOfReplacing" : 100,
			#
			#	"bufferRate" : 2
			#
			#	}, {
			#
			#	"tierNumber" : 2,
			#
			#	"tierLevelRange" : "00000000",
			#
			#	"tierDeltaRange" : "00000000",
			#
			#	"tierUsedCapacity" : 2083158,
			#
			#	"tierTotalCapacity" : 7214970,
			#
			#	"tablespaceRate" : 8,
			#
			#	"performanceRate" : 0,
			#
			#	"progressOfReplacing" : 100,
			#
			#	"bufferRate" : 2
			#
			#	} ]
			#
			#}, {
			#
			#	"poolId" : 1,
			
			
			for my $key1 (keys %{ $json_decoded })
			{
				# key1 = data
				# print "k1 $key1\n";
				for my $key2 (0..$#{ $json_decoded->{$key1} }) 
				{
					# key2 = 0 .. x
					# print "k2 $key2\n";
					for my $key3 (keys %{ $json_decoded->{$key1}[$key2] }) 
					{
						# key3 = poolId,poolStatus,etc
						# print "k3 $key3\n";
						if ($key3 eq "tiers")
						{
							for my $key4 (0..$#{ $json_decoded->{$key1}[$key2]{$key3} })
							{
								# key4 = 0 .. x 
								# print "k4 $key4 .. $json_decoded->{$key1}[$key2]{$key3}[$key4]\n";
								# REST_pools{poolId}{tierNumber [{tierUsedCapacity, performanceRate,etc.}],virutalVolumeCapacityRater,tiers,etc.}
								push(@{$REST_pools{$json_decoded->{$key1}[$key2]{poolId}}{$key3}}, $json_decoded->{$key1}[$key2]{$key3}[$key4]);
							}
						}										
						else
						{
							$REST_pools{$json_decoded->{$key1}[$key2]{poolId}}{$key3} = $json_decoded->{$key1}[$key2]{$key3};
						}

					}
				}
			}
			
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPOOLS","Get pools successful");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPOOLS","Get pools finished");
			return	\%REST_pools;
		}
		else
		{
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTGETPOOLS","Return parameter: $par_return");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTGETPOOLS","return parameter unknown. Returning raw http data");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTGETPOOLS","Get pools successful");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTGETPOOLS","Get pools finished");
			return $http_decoded;
		}
		
	}
	else
	{
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,3,"RESTGETPOOLS","Get pools NOT successful");
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,3,"RESTGETPOOLS","Get pools finished");
		return -1;
	}
	
}

sub REST_GET_PORTS
{
	my $par_token = $_[0];
	my $par_protocol = $_[1];
	my $par_ip = $_[2];
	my $par_storageid = $_[3];
	my $par_return = $_[4];
	my $par_log_fh = $_[5];
	my $par_vbse = $_[6];
	my $par_lglvl = $_[7];
	
	my $httpmethod = "GET";
	my $url_path = "ConfigurationManager/v1/objects/storages/${par_storageid}/ports";
	
	my %REST_port;
	
	my $http_status_code;
	my $http_decoded;
	
	my $json_obj = JSON->new->utf8;
	my $json_decoded = "n/a";
	
	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPORTS","Get ports start");
		
	($http_status_code,$http_decoded) = REST_REQUEST_SESSION($par_token,$par_protocol,$par_ip,$httpmethod,$url_path,$par_log_fh,$par_vbse,$par_lglvl);
	
	if ($http_status_code eq "200")
	{

		if ($par_return eq "raw")
		{
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPORTS","Return parameter: $par_return");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPORTS","Get ports successful");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPORTS","Get ports finished");
			return $http_decoded;
		}
		elsif ($par_return eq "custom")
		{
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPORTS","Return parameter: $par_return");
			
			$json_decoded = $json_obj->decode($http_decoded);
			
			# Parse data example
			#			{
			#
			#  "data" : [ {
			#
			#    "portId" : "CL1-A",
			#
			#    "portType" : "FIBRE",
			#
			#    "portAttributes" : [ "TAR" ],
			#
			#    "portSpeed" : "8G",
			#
			#    "loopId" : "EF",
			#
			#    "fabricMode" : true,
			#
			#    "portConnection" : "PtoP",
			#
			#    "lunSecuritySetting" : true,
			#
			#    "wwn" : "50060e8007c5f700"
			#
			#  }, {
			#
			#    "portId" : "CL1-B",
			
							
			for my $key1 (keys %{ $json_decoded })
			{
				# key1 = data
				
				for my $key2 (0..$#{ $json_decoded->{$key1} }) 
				{
					# key2 = 0 .. x
					for my $key3 (keys %{ $json_decoded->{$key1}[$key2] }) 
					{
						# Check if there is more than 1 value in array for portAttributes, if so make string of it and store inro HoH
						if ($key3 eq "portAttributes")
						{
							# key3 = portId,portType,etc
							for my $key4 (0..$#{ $json_decoded->{$key1}[$key2]{$key3} })
							{
									# key4 = 0 .. x which are "portAttributes" : >>>> [ "TAR" ] <<<<
									# $REST_port{portId}{PortType,portAttributes,etc}[0-x portAttributes]
									push(@{$REST_port{$json_decoded->{$key1}[$key2]{portId}}{$key3}}, $json_decoded->{$key1}[$key2]{$key3}[$key4]);
							}
						}
						
						else
						{
							$REST_port{$json_decoded->{$key1}[$key2]{portId}}{$key3} = $json_decoded->{$key1}[$key2]{$key3};
						}

					}
				}
			}

			
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPORTS","Get ports successful");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETPORTS","Get ports finished");
			return	\%REST_port;
		}
		else
		{
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTGETPORTS","Return parameter: $par_return");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTGETPORTS","Return parameter unknown. Returning raw http data");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTGETPORTS","Get ports successful");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTGETPORTS","Get ports finished");
			return $http_decoded;
		}
		
	}
	else
	{
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,3,"RESTGETPORTS","Get ports NOT successful");
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,3,"RESTGETPORTS","Get ports finished");
		return -1;
	}
	
	
}
	

	
sub REST_GET_STORAGE
{
	my $par_user = $_[0];
	my $par_pass = $_[1];
	my $par_protocol = $_[2];
	my $par_ip = $_[3];
	my $par_return = $_[4];
	my $par_log_fh = $_[5];
	my $par_vbse = $_[6];
	my $par_lglvl = $_[7];
	
	my $httpmethod = "GET";
	my $url_path = "ConfigurationManager/v1/objects/storages/";
	
	my %REST_storage;
	
	my $http_status_code;
	my $http_decoded;
	
	my $json_obj = JSON->new->utf8;
	my $json_decoded = "n/a";
	
	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETSTORAGE","Get storage start");
	
	($http_status_code,$http_decoded) = REST_REQUEST($par_user,$par_pass,$par_protocol,$par_ip,$httpmethod,$url_path,$par_log_fh,$par_vbse,$par_lglvl);
	
	if ($http_status_code eq "200")
	{
		if ($par_return eq "raw")
		{
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETSTORAGE","Return parameter: $par_return");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETSTORAGE","Get storage successful");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETSTORAGE","Get storage finished");
			return $http_decoded;
		}
		elsif ($par_return eq "custom")
		{
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETSTORAGE","Return parameter: $par_return");
			
			$json_decoded = $json_obj->decode($http_decoded);
			
			# Parse data example
			#		{
			#
			#  "data" : [ {
			#
			#    "storageDeviceId" : "800000050679",
			#
			#    "model" : "VSP G1000",
			#
			#    "serialNumber" : 50679,
			#
			#    "svpIp" : "10.70.5.145"
			#
			#  } ]
			#
			#}
			for my $key1 (keys %{ $json_decoded })
			{
				# key1 = data
				for my $key2 (0..$#{ $json_decoded->{$key1} }) 
				{
					# key2 = 0 .. x
					for my $key3 (keys %{ $json_decoded->{$key1}[$key2] }) 
					{
						# key3 = storageDeviceId,model,etc
						$REST_storage{$key3} = $json_decoded->{$key1}[$key2]{$key3};
					}
				}
			}

			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETSTORAGE","Get storage successful");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETSTORAGE","Get storage finished");
			return \%REST_storage;
		}
		else
		{
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTGETSTORAGE","Return parameter: $par_return");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTGETSTORAGE","Return parameter unknown. Returning raw http data");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTGETSTORAGE","Get storage successful");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTGETSTORAGE","Get storage finished");
			return $http_decoded;
		}
	}
	else
	{
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,3,"RESTGETSTORAGE","Get storage NOT successful");
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,3,"RESTGETSTORAGE","Get storage finished");
		return -1;
	}
}
	
sub REST_GET_VERSION
{
	my $par_user = $_[0];
	my $par_pass = $_[1];
	my $par_protocol = $_[2];
	my $par_ip = $_[3];
	my $par_return = $_[4];
	my $par_log_fh = $_[5];
	my $par_vbse = $_[6];
	my $par_lglvl = $_[7];
	
	my $httpmethod = "GET";
	my $url_path = "ConfigurationManager/configuration/version";
	
	my $REST_version;

	my $http_status_code;
	my $http_decoded;
	
	my $json_obj = JSON->new->utf8;
	my $json_decoded = "n/a";
	
	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETVERSION","Get version start");
	
	($http_status_code,$http_decoded) = REST_REQUEST($par_user,$par_pass,$par_protocol,$par_ip,$httpmethod,$url_path,$par_log_fh,$par_vbse,$par_lglvl);
	
	if ($http_status_code eq "200")
	{
		
		if ($par_return eq "raw")
		{
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETVERSION","Return parameter: $par_return");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETVERSION","Get version successful");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETVERSION","Get version finished");
			return $http_decoded;
		}
		elsif ($par_return eq "custom")
		{
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETVERSION","Return parameter: $par_return");
			$json_decoded = $json_obj->decode($http_decoded);
			
			# Parse data example
			#{
			#
			#  "productName" : "Configuration Manager REST API",
			#
			#  "apiVersion" : "1.3.0"
			#
			#}
			
			$REST_version = $json_decoded->{apiVersion};
			
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETVERSION","Get version successful");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTGETVERSION","Get version finished");
			return	$REST_version;
		}
		else
		{
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTGETVERSION","Return parameter: $par_return");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTGETVERSION","return parameter unknown. Returning raw http data");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTGETVERSION","Get version successful");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTGETVERSION","Get version finished");
			return $http_decoded;
		}
		
	}
	else
	{
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,3,"RESTGETVERSION","Get version NOT successful");
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,3,"RESTGETVERSION","Get version finished");
		return -1;
	}
	
}


sub REST_DISCARD_SESSION
{
	my $par_token = $_[0];
	my $par_sessionid = $_[1];
	my $par_protocol = $_[2];
	my $par_ip = $_[3];
	my $par_storageid = $_[4];
	my $par_return = $_[5];
	my $par_log_fh = $_[6];
	my $par_vbse = $_[7];
	my $par_lglvl = $_[8];
	
	my $httpmethod = "DELETE";
	my $url_path = "ConfigurationManager/v1/objects/storages/${par_storageid}/sessions/${par_sessionid}";	
	
	my $http_status_code;
	my $http_decoded;
	
	my $json_obj = JSON->new->utf8;
	my $json_decoded = "n/a";
	
	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTCREATESESSION","Discard session start");
	
	($http_status_code,$http_decoded) = REST_REQUEST_SESSION($par_token,$par_protocol,$par_ip,$httpmethod,$url_path,$par_log_fh,$par_vbse,$par_lglvl);
	
	if ($http_status_code eq "200")
	{
		
		if ($par_return eq "raw")
		{
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTCREATESESSION","Return parameter: $par_return");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTCREATESESSION","Discard session token: $par_token");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTCREATESESSION","Discard session id: $par_sessionid");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTCREATESESSION","Discard session successful");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTCREATESESSION","Discard session finished");
			return $http_decoded;
		}
		else
		{
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTCREATESESSION","Return parameter: $par_return");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTCREATESESSION","Return parameter unknown. Returning raw http data");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTCREATESESSION","Discard session token: $par_token");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTCREATESESSION","Discard session id: $par_sessionid");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTCREATESESSION","Discard session successful");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTCREATESESSION","Discard session finished");
			return $http_decoded;
		}
		
	}
	else
	{
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTCREATESESSION","Discard session token: $par_token");
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTCREATESESSION","Discard session id: $par_sessionid");
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,3,"RESTCREATESESSION","Discard session NOT successful");
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,3,"RESTCREATESESSION","Discard session finished");
		return -1;
	}
	
}

sub REST_CREATE_SESSION
{
	my $par_user = $_[0];
	my $par_pass = $_[1];
	my $par_protocol = $_[2];
	my $par_ip = $_[3];
	my $par_storageid = $_[4];
	my $par_return = $_[5];
	my $par_log_fh = $_[6];
	my $par_vbse = $_[7];
	my $par_lglvl = $_[8];
	
	my $httpmethod = "POST";
	my $url_path = "ConfigurationManager/v1/objects/storages/${par_storageid}/sessions";	
	
	my $http_status_code;
	my $http_decoded;
	
	my $json_obj = JSON->new->utf8;
	my $json_decoded = "n/a";
	
	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTCREATESESSION","Create session start");
	
	($http_status_code,$http_decoded) = REST_REQUEST($par_user,$par_pass,$par_protocol,$par_ip,$httpmethod,$url_path,$par_log_fh,$par_vbse,$par_lglvl);
	
	if ($http_status_code eq "200")
	{
		
		if ($par_return eq "raw")
		{
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTCREATESESSION","Return parameter: $par_return");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTCREATESESSION","Create session successful");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTCREATESESSION","Create session finished");
			return ($http_decoded,$http_decoded);
		}
		elsif ($par_return eq "custom")
		{
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTCREATESESSION","Return parameter: $par_return");
			$json_decoded = $json_obj->decode($http_decoded);
			
			# Parse data example
			# {
			# "token" : "da2fc96e-a9bb-4c4f-80e0-ebbfa8ecf44e",
			# "sessionId" : 10
			# }
			
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTCREATESESSION","Token: $json_decoded->{token}");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTCREATESESSION","Session id:$json_decoded->{sessionId}");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTCREATESESSION","Create session successful");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTCREATESESSION","Create session finished");
			return ($json_decoded->{token},$json_decoded->{sessionId});
			
		}
		else
		{
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTCREATESESSION","Return parameter: $par_return");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTCREATESESSION","Return parameter unknown. Returning raw http data");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTCREATESESSION","Create session successful");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RESTCREATESESSION","Create session finished");
			return ($http_decoded,$http_decoded);
		}
		
	}
	else
	{
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,3,"RESTCREATESESSION","Create session NOT successful");
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,3,"RESTCREATESESSION","Create session finished");
		return ("-1","-1");
	}
	
}

sub REST_REQUEST
{
	my $par_user = $_[0];
	my $par_pass = $_[1];
	my $par_protocol = $_[2];
	my $par_ip = $_[3];
	my $par_httpmethod = $_[4];
	my $par_url_path = $_[5];
	my $par_log_fh = $_[6];
	my $par_vbse = $_[7];
	my $par_lglvl = $_[8];
	
	my $url_string = "${par_protocol}://${par_user}:${par_pass}\@${par_ip}/${par_url_path}";

	my $http_status_code = "n/a";
	my $http_decoded = "n/a";

	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTREQUESTUSER","Request start");
	
	#my $ua = LWP::UserAgent->new;
	no strict;
	my $ua = LWP::UserAgent->new ( ssl_opts => { verify_hostname => 0, SSL_verify_mode => SSL_VERIFY_NONE } );
	use strict;
	$ua->agent("Mozilla/8.0"); # pretend we are very capable browser
	$ua->timeout(60);
	$ua->env_proxy;

	
	my $req = HTTP::Request->new;
	$req->header('content-type' => 'application/json');
	$req->header('accept' => 'application/json');
	$req->method($par_httpmethod);
	$req->uri($url_string);
	
	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTREQUESTUSER","Protocol: $par_protocol");
	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTREQUESTUSER","Method: $par_httpmethod");
	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTREQUESTUSER","Authorisation (hide pwd): $par_user,***");
	LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,7,"RESTREQUESTUSER","Authorisation (show pwd): $par_user,$par_pass");
	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTREQUESTUSER","URL (hide pwd): ${par_protocol}://${par_user}:***\@${par_ip}/${par_url_path}");
	LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,7,"RESTREQUESTUSER","URL (show pwd): $url_string");
	my $response = $ua->request($req);
 
	if ($response->is_success)
	{
		$http_status_code = $response->code;
		$http_decoded = $response->decoded_content;
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTREQUESTUSER","Request successful");
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTREQUESTUSER","Status code: $http_status_code");
		LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,7,"RESTREQUESTUSER","Decoded: $http_decoded");
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTREQUESTUSER","Request finished");
		return ($http_status_code,$http_decoded);
	}
	else 
	{
		$http_status_code = $response->code;
		$http_decoded = $response->decoded_content;
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,3,"RESTREQUESTUSER","Request NOT successful");
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,3,"RESTREQUESTUSER","Status code: $http_status_code");
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,3,"RESTREQUESTUSER","Decoded: $http_decoded");
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,3,"RESTREQUESTUSER","Request finished");
		return ($http_status_code,$http_decoded);
	}
	
	
}

sub REST_REQUEST_SESSION
{
	my $par_session = $_[0];
	my $par_protocol = $_[1];
	my $par_ip = $_[2];
	my $par_httpmethod = $_[3];
	my $par_url_path = $_[4];
	my $par_log_fh = $_[5];
	my $par_vbse = $_[6];
	my $par_lglvl = $_[7];

	my $url_string = "${par_protocol}://${par_ip}/${par_url_path}";

	my $http_status_code = "n/a";
	my $http_decoded = "n/a";

	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTREQUESTSESSION","Request start");
	
	#my $ua = LWP::UserAgent->new;
	no strict;
	my $ua = LWP::UserAgent->new ( ssl_opts => { verify_hostname => 0, SSL_verify_mode => SSL_VERIFY_NONE } );
	use strict;
	$ua->agent("Mozilla/8.0"); # pretend we are very capable browser
	$ua->timeout(600); # 10 minutes as the LDEV request with 16'384 can take very long.
	$ua->env_proxy;

	
	my $req = HTTP::Request->new;
	$req->header('content-type' => 'application/json');
	$req->header('accept' => 'application/json');
	$req->header('Authorization' => "Session $par_session");
	$req->method($par_httpmethod);
	$req->uri($url_string);
	
	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTREQUESTSESSION","Protocol: $par_protocol");
	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTREQUESTSESSION","Method: $par_httpmethod");
	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTREQUESTSESSION","Authorisation token: $par_session");
	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTREQUESTSESSION","URL (session): $url_string");
	my $response = $ua->request($req);
 
	if ($response->is_success)
	{
		$http_status_code = $response->code;
		$http_decoded = $response->decoded_content;
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTREQUESTSESSION","Request successful");
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTREQUESTSESSION","Status code: $http_status_code");
		LOG_MSG($par_log_fh,$par_vbse,"v",$par_lglvl,7,"RESTREQUESTSESSION","Decoded: $http_decoded");
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RESTREQUESTSESSION","Request finished");
		return ($http_status_code,$http_decoded);
	}
	else 
	{
		$http_status_code = $response->code;
		$http_decoded = $response->decoded_content;
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,3,"RESTREQUESTSESSION","Request NOT successful");
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,3,"RESTREQUESTSESSION","Status code: $http_status_code");
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,3,"RESTREQUESTSESSION","Decoded: $http_decoded");
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,3,"RESTREQUESTSESSION","Request finished");
		return ($http_status_code,$http_decoded);
	}
	
	
}
1; # End of the module with a true value