package HiCHpp_RAIDCOM; # Name of the module and namespace
#!/usr/bin/perl
# Author: Maik Ernst
# History:
# 2014-02-07 - initial module
#
# LICENSE
# Hitachi, Ltd. All rights reserved
# Hitachi Data Systems products and services can be ordered only under the terms and conditions
# of the applicable Hitachi Data Systems agreements.
#
# NAME
# HiCHpp_RAIDCOM
#
# DESCRIPTION
# This module is doing all the horcm & raidcom stuff for HCHppcollect.pl
#
# AUTHOR
#   Maik Ernst (maik.ernst@hds.com)
#
# MODIFICATIONS
#   2014-02-07 - v1.0	- initial module
#   2014-02-11 - v1.1	- adding -CLI to raidscan
#   2014-02-14 - v1.2	- adding CHECK_OLD_INSTANCE
#   2014-02-16 - v1.3	- add an output file of raidcom which is parsable with semicolon as delimiter
#						This is quite difficult as the current delimiter is a space and the values can have spaces also.
#						Not every output of a values is known therefore there are a lot of guesses if strings,digits,etc. 
#						The following values has been testes and they can have one or more spaces in their output:
#						meta_resource, pool name, host group name, HMO
#						For now it looks it is working but troubleshooting within the regex might be needed. 
#						If you experience issue please let me know.
#   2014-02-21 - v1.4	- add raidcom collection of parity groups
#   2014-04-07 - v1.5	- adding -fx to every collection
#						- add raidcom get pool, raidcom get dp_pool -key opt
#   2014-05-22 - v1.6	- Adding -m at raidcom collection to get all MUs for raidscan.
#						But only for TC and not for SI as the collection would take too long.
#   2015-04-02 - v1.7	- Adding -f at HDSperfcollect to use FC device instead of IP command device
#	2015-05-02 - v1.8	- Changing module name from HDSPERF_RAIDCOM.pm to HCHpp_RAIDCOM.pm to meet swiss script standard
#   2015-05-05			- Bugfix (missing 3 in in CDM file for G1000 when using FC command device)
#	2015-05-05 - v1.9	- add raidcom level to disable some collections.
#						  0 = collect all, 1 = disable raidscan, 2 = disable raidscan and host_grp
#						  change log function
#	2015-07-03 - v2.0.0	- adding raidcom support for G800,G600,G400,G200$ (GUM IPs instead of SVP IP for these systems)
#   2016-03-25 - v2.1.0	- adding raidcom get dp_pool, fixing CMD_RAIDCOM_GET_POOL_OPT, changing randcom from 200 to 500 for horcm instance
#						- delete old horcm log dir of horcm instance
#						- adding 2 / 3 options for CMD_RAIDCOM_GET_POOL_OPT and CMD_RAIDCOM_GET_DPPOOL
#	2016-11-19 - v2.2.0	- adding logging module HiCH_LOG, change logging call
#						- adding G1500 support
#						- adding logging module HiCH_LOG, change logging call
#						- adding raidcom -h (raw only)
#						- adding raidcom get pool -key saving (raw only)
#						- adding raidcom get pool -key fmc (raw only)
#						- adding raidcom get resource -key opt  (raw only)
#						- making get ports unique
#						- fix parser initilaze value for get partiy grp, get port on Gx00, get dppool, get hostgroup on Gx00
#	2016-11-24 - v2.2.1	- adding fucntion CHECK_HORCM_LOGIN to check if still logged in. But not in use yet.
#	2018-04-18 - v2.3.0	- adding raidcom get parity -key opt (raw only)
#						- deleting all parsing and _p files (not raw files). These files are not needed and need a lot of maintenance due to parsing changes.



use strict;
use warnings;
use File::Path;

use FindBin qw($Bin);
use lib $Bin;
use lib "$Bin\\..\\lib\\modules";
use HiCH_LOG;

require Exporter;
use vars qw(@ISA @EXPORT);

@ISA = qw(Exporter);

# All functions and variables which need to be exported and available in the importing scripts.
@EXPORT = qw(
DO_RAIDCOM
);

sub DO_RAIDCOM
{
	my $par_flag_cci_dir = $_[0];
	my $par_EXPORTTOOL_IP = $_[1];
	my $par_horcm_dirname = $_[2];
	my $par_EXPORTTOOL_USER = $_[3];
	my $par_EXPORTTOOL_PWD = $_[4];
	my $par_serial = $_[5];
	my $par_RAIDCOM_FCDEVICE = $_[6];
	my $par_system = $_[7];
	my $par_HOST = $_[8];
	my $par_raidcom_level = $_[9];
	my $par_log_fh = $_[10];
	my $par_vbse = $_[11];
	my $par_lglvl = $_[12];


	my $CCI_horcmstart = "${par_flag_cci_dir}\\etc\\horcmstart.exe";
	my $CCI_horcmshutdown = "${par_flag_cci_dir}\\etc\\horcmshutdown.exe";
	my $CCI_raidcom = "${par_flag_cci_dir}\\etc\\raidcom.exe";
	my $CCI_raidscan = "${par_flag_cci_dir}\\etc\\raidscan.exe";
	
	my $horcmconf = "${par_horcm_dirname}\\horcm";
	my $random_number;
	my $skipraidcom = "no";
	
	
	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Starting raidcom stuff");
	
	unless (-e $CCI_horcmstart and -e $CCI_horcmshutdown and -e $CCI_raidcom and -e $CCI_raidscan)
	{
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RAIDCOM","$CCI_horcmstart or $CCI_horcmshutdown or $CCI_raidcom or $CCI_raidscan does not exist.");
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RAIDCOM","raidcom information will NOT get collected.");
		$skipraidcom = "yes";
	}
	
	# Cleanup old instances if the script has been aborted before
	CHECK_OLD_INSTANCE($par_flag_cci_dir,$par_horcm_dirname,$par_serial,$par_log_fh,$par_vbse,$par_lglvl);
	
	
	if ($skipraidcom eq "no")
	{
	
		# Make a random horcm instance number. Check if horcm instance exists. If yes try another number for max 10 times.
		my $i=0;
		until ($i==10)
		{
			$i++;
			# random number between 20 and 520
			$random_number = int(rand(500) +20 );
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","$i attempt ... horcm random instance number is: $random_number");
			unless (-e "${horcmconf}${random_number}.conf" )
			{
				LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","${horcmconf}${random_number}.conf does not exist yet");
				$i=10;
				next;
			}	
				LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","${horcmconf}${random_number}.conf already exists");
				
			if ($i==10)
			{
				LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RAIDCOM","This was the $i attempt ... giving up");
				LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RAIDCOM","raidcom information will NOT get collected.");
				$skipraidcom = "yes";
			}
			
		}
			
		if ($skipraidcom eq "no")
		{
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOMDELHORCDIR","Checking if horcm log dir exists: ${par_flag_cci_dir}\\log${random_number}");
			if (-d "${par_flag_cci_dir}\\log${random_number}")
			{
				LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOMDELHORCDIR","Horcm log dir exists: ${par_flag_cci_dir}\\log${random_number}");
				LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOMDELHORCDIR","Deleting directory: ${par_flag_cci_dir}\\log${random_number}");
				rmtree ("${par_flag_cci_dir}\\log${random_number}") or warn LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RAIDCOMDELHORCDIR","Can't delete dir ${par_flag_cci_dir}\\log${random_number}: $!\n");
				LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOMDELHORCDIR","Directory deleted: ${par_flag_cci_dir}\\log${random_number}");
			}
			else
			{
				LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOMDELHORCDIR","Horcm log dir does not exist: ${par_flag_cci_dir}\\log${random_number}");
			}
		}
	
		if ($skipraidcom eq "no")
		{
			# Set windows environment variables
			$ENV{HORCM_CONF} = "${horcmconf}${random_number}.conf";
			$ENV{HORCMINST} = $random_number;
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Set windows environment vars");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","\$ENV{HORCM_CONF} = $ENV{HORCM_CONF}");
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","\$ENV{HORCMINST} = $ENV{HORCMINST} ");

			# Generate horcm file
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Creating ${horcmconf}${random_number}.conf");
			open(HORCM_FH, ">${horcmconf}${random_number}.conf") or warn "Cannot create ${horcmconf}${random_number}.conf: $!\n";
			print HORCM_FH ("HORCM_CMD\n");
			
			# Check if FC device flag is set. If "yes" use FC-CMD, otherwise use IP-CMD.
			if ($par_RAIDCOM_FCDEVICE eq "yes")
			{
				if (($par_system eq "G1000") or ($par_system eq "G1500"))
				{
					LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","G1000/G1500 flag found");
					print HORCM_FH ("\\\\.\\CMD-3${par_serial}\n");	
					LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Commande Device: FC CMD = CMD-3${par_serial}");
					print HORCM_FH ("\\\\.\\CMD-3${par_serial}\n");	
				}
				else
				{
					LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Commande Device: FC CMD = CMD-${par_serial}");
					print HORCM_FH ("\\\\.\\CMD-${par_serial}\n");	
				}
			}
			else
			{
					if ($par_system =~ /^G800$|^G600$|^G400$|^G200$/)
					{
						my $IP_string;
						my @GUM_IP = split(';', $par_EXPORTTOOL_IP);
						foreach my $ip (@GUM_IP) 
						{
							LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Commande Device (Gx00): IP CMD = IPCMD-${ip}-31001 and IPCMD-${ip}-31002");
							$IP_string = $IP_string . "\\\\.\\IPCMD-${ip}-31001 \\\\.\\IPCMD-${ip}-31002 ";
						}
						print HORCM_FH ($IP_string);
					}
					else
					{
						LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Commande Device: IP CMD = IPCMD-${par_EXPORTTOOL_IP}-31001");
						print HORCM_FH ("\\\\.\\IPCMD-${par_EXPORTTOOL_IP}-31001\n");
					}
			}
			
			close (HORCM_FH);
			
			# Start horcm instance
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Starting horcm instance");
			open (CMD_HORCMSTART, "$CCI_horcmstart 2>&1|") or warn ("!!! Can't run program: $!\n");
			while (my $line = <CMD_HORCMSTART>)
			{
				LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","$line");
			}
			close CMD_HORCMSTART;
			my $exit_CMD_HORCMSTART = $?;
			
			if($exit_CMD_HORCMSTART != 0)
			{
				LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RAIDCOM","Failed to start HORCM instance.");
				LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RAIDCOM","raidcom information will NOT get collected.");
				LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Deleting ${horcmconf}${random_number}.conf");
				unlink "${horcmconf}${random_number}.conf" or warn ("Could not delete ${horcmconf}${random_number}.conf: $!");
				$skipraidcom = "yes";
			}
			
			if ($skipraidcom eq "no")
			{

				# raidcom login
				LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Run $CCI_raidcom -login -I${random_number} $par_EXPORTTOOL_USER ***** ");
				open (CMD_RAIDCOM_LOGIN, "$CCI_raidcom -login \"$par_EXPORTTOOL_USER\" \"$par_EXPORTTOOL_PWD\" -I${random_number} 2>&1|") or warn ("!!! Can't run program: $!\n");
				while (my $line = <CMD_RAIDCOM_LOGIN>)
				{
					LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","$line");
				}
				close CMD_RAIDCOM_LOGIN;
				my $exit_CMD_RAIDCOM_LOGIN = $?;
				
				if($exit_CMD_RAIDCOM_LOGIN != 0)
				{
					LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RAIDCOM","Failed to raidcom login");
					LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RAIDCOM","raidcom information will NOT get collected.");
					$skipraidcom = "yes";
				}

				if ($skipraidcom eq "no")
				{						
					# raidcom -h
					LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Run $CCI_raidcom -h -I${random_number}");
					open (CMD_RAIDCOM_HELP, "$CCI_raidcom -h -I${random_number} 2>&1|") or warn ("!!! Can't run program: $!\n");
					
					open (FH_RAIDCOM_HELP, ">${par_horcm_dirname}\\${par_serial}_raidcom_help.txt") 
					or warn "Cannot create ${par_horcm_dirname}\\${par_serial}_raidcom_help.txt: $!\n";
					
					while (my $line = <CMD_RAIDCOM_HELP>)
					{
						print FH_RAIDCOM_HELP $line;
					}
					close CMD_RAIDCOM_HELP;
					close FH_RAIDCOM_HELP;
					LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Finished $CCI_raidcom -h -I${random_number}");
					
					# raidcom get resource -key opt
					LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Run $CCI_raidcom get resource -key opt -fx -I${random_number}");
					open (CMD_RAIDCOM_GET_RES_OPT, "$CCI_raidcom get resource -key opt -fx -I${random_number} 2>&1|") or warn ("!!! Can't run program: $!\n");
							
					open (FH_RAIDCOM_GET_RES_OPT, ">${par_horcm_dirname}\\${par_serial}_raidcom_get_res_opt.txt") 
					or warn "Cannot create ${par_horcm_dirname}\\${par_serial}_raidcom_get_res_opt.txt: $!\n";					
					
					while (my $line = <CMD_RAIDCOM_GET_RES_OPT>)
					{
						print FH_RAIDCOM_GET_RES_OPT $line;
							
					}
					close CMD_RAIDCOM_GET_RES_OPT;
					close FH_RAIDCOM_GET_RES_OPT;
					LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Finished $CCI_raidcom get resource -key opt -fx -I${random_number}");
				
					# raidcom get resource
					LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Run $CCI_raidcom get resource -fx -I${random_number}");
					open (CMD_RAIDCOM_GET_RES, "$CCI_raidcom get resource -fx -I${random_number} 2>&1|") or warn ("!!! Can't run program: $!\n");
					
					open (FH_RAIDCOM_GET_RES, ">${par_horcm_dirname}\\${par_serial}_raidcom_get_res.txt") 
					or warn "Cannot create ${par_horcm_dirname}\\${par_serial}_raidcom_get_res.txt: $!\n";
						
					while (my $line = <CMD_RAIDCOM_GET_RES>)
					{
						print FH_RAIDCOM_GET_RES $line;

					}
					close CMD_RAIDCOM_GET_RES;
					close FH_RAIDCOM_GET_RES;
					LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Finished $CCI_raidcom get resource -fx -I${random_number}");
					
					
					# raidcom get pool
					LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Run $CCI_raidcom get pool -fx -I${random_number}");
					open (CMD_RAIDCOM_GET_POOL, "$CCI_raidcom get pool -fx -I${random_number} 2>&1|") or warn ("!!! Can't run program: $!\n");
					
					open (FH_RAIDCOM_GET_POOL, ">${par_horcm_dirname}\\${par_serial}_raidcom_get_pool.txt") 
					or warn "Cannot create ${par_horcm_dirname}\\${par_serial}_raidcom_get_pool.txt: $!\n";
					
					while (my $line = <CMD_RAIDCOM_GET_POOL>)
					{
						print FH_RAIDCOM_GET_POOL $line;				
					}
					close CMD_RAIDCOM_GET_POOL;
					close FH_RAIDCOM_GET_POOL;
					LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Finished $CCI_raidcom get pool -fx -I${random_number}");
					
					# raidcom get pool -key opt
					LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Run $CCI_raidcom get pool -key opt -fx -I${random_number}");
					open (CMD_RAIDCOM_GET_POOL_OPT, "$CCI_raidcom get pool -key opt -fx -I${random_number} 2>&1|") or warn ("!!! Can't run program: $!\n");
					
					open (FH_RAIDCOM_GET_POOL_OPT, ">${par_horcm_dirname}\\${par_serial}_raidcom_get_pool_opt.txt") 
					or warn "Cannot create ${par_horcm_dirname}\\${par_serial}_raidcom_get_pool_opt.txt: $!\n";
					
					my $fields_getpoolopt_size;
					while (my $line = <CMD_RAIDCOM_GET_POOL_OPT>)
					{
						print FH_RAIDCOM_GET_POOL_OPT $line;
					}
					close CMD_RAIDCOM_GET_POOL_OPT;
					close FH_RAIDCOM_GET_POOL_OPT;
					LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Finished $CCI_raidcom get pool -key opt -fx -I${random_number}");
				
					# raidcom get dp_pool
					LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Run $CCI_raidcom get dp_pool -fx -I${random_number}");
					open (CMD_RAIDCOM_GET_DPPOOL, "$CCI_raidcom get dp_pool -fx -I${random_number} 2>&1|") or warn ("!!! Can't run program: $!\n");
					
					open (FH_RAIDCOM_GET_DPPOOL, ">${par_horcm_dirname}\\${par_serial}_raidcom_get_dppool.txt") 
					or warn "Cannot create ${par_horcm_dirname}\\${par_serial}_raidcom_get_dppool.txt: $!\n";
					
					my $fields_getdppool_size;
					while (my $line = <CMD_RAIDCOM_GET_DPPOOL>)
					{
						print FH_RAIDCOM_GET_DPPOOL $line;
					}
					
					close CMD_RAIDCOM_GET_DPPOOL;
					close FH_RAIDCOM_GET_DPPOOL;
					LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Finished $CCI_raidcom get dppool -fx -I${random_number}");
					
				 # raidcom get dp_pool -key opt
					LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Run $CCI_raidcom get dp_pool -key opt -fx -I${random_number}");
					open (CMD_RAIDCOM_GET_DPPOOL_OPT, "$CCI_raidcom get dp_pool -key opt -fx -I${random_number} 2>&1|") or warn ("!!! Can't run program: $!\n");
					
					open (FH_RAIDCOM_GET_DPPOOL_OPT, ">${par_horcm_dirname}\\${par_serial}_raidcom_get_dppool_opt.txt") 
					or warn "Cannot create ${par_horcm_dirname}\\${par_serial}_raidcom_get_dppool_opt.txt: $!\n";
					
					while (my $line = <CMD_RAIDCOM_GET_DPPOOL_OPT>)
					{
						print FH_RAIDCOM_GET_DPPOOL_OPT $line;
					}
					close CMD_RAIDCOM_GET_DPPOOL_OPT;
					close FH_RAIDCOM_GET_DPPOOL_OPT;
					LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Finished $CCI_raidcom get pool -key opt -fx -I${random_number}");

					# raidcom get pool -key saving
					LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Run $CCI_raidcom get pool -key saving -fx -I${random_number}");
					open (CMD_RAIDCOM_GET_POOL_OPT_SAVING, "$CCI_raidcom get pool -key saving -fx -I${random_number} 2>&1|") or warn ("!!! Can't run program: $!\n");
					
					open (FH_RAIDCOM_GET_POOL_OPT_SAVING, ">${par_horcm_dirname}\\${par_serial}_raidcom_get_pool_saving.txt") 
					or warn "Cannot create ${par_horcm_dirname}\\${par_serial}_raidcom_get_pool_saving.txt: $!\n";

					while (my $line = <CMD_RAIDCOM_GET_POOL_OPT_SAVING>)
					{
						print FH_RAIDCOM_GET_POOL_OPT_SAVING $line;
					}
					close CMD_RAIDCOM_GET_POOL_OPT_SAVING;
					close FH_RAIDCOM_GET_POOL_OPT_SAVING;
					LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Finished $CCI_raidcom get pool -key saving -fx -I${random_number}");
					
					# raidcom get pool -key fmc
					LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Run $CCI_raidcom get pool -key fmc -fx -I${random_number}");
					open (CMD_RAIDCOM_GET_POOL_OPT_FMC, "$CCI_raidcom get pool -key fmc -fx -I${random_number} 2>&1|") or warn ("!!! Can't run program: $!\n");
					
					open (FH_RAIDCOM_GET_POOL_OPT_FMC, ">${par_horcm_dirname}\\${par_serial}_raidcom_get_pool_fmc.txt") 
					or warn "Cannot create ${par_horcm_dirname}\\${par_serial}_raidcom_get_pool_fmc.txt: $!\n";

					while (my $line = <CMD_RAIDCOM_GET_POOL_OPT_FMC>)
					{
						print FH_RAIDCOM_GET_POOL_OPT_FMC $line;
					}
					close CMD_RAIDCOM_GET_POOL_OPT_FMC;
					close FH_RAIDCOM_GET_POOL_OPT_FMC;
					LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Finished $CCI_raidcom get pool -key fmc -fx -I${random_number}");
					
					# -	raidcom get snapshot
					LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Run $CCI_raidcom get snapshot -fx -I${random_number}");
					open (CMD_RAIDCOM_GET_SNAPSHOT, "$CCI_raidcom get snapshot -fx -I${random_number} 2>&1|") or warn ("!!! Can't run program: $!\n");
					
					open (FH_RAIDCOM_GET_SNAPSHOT, ">${par_horcm_dirname}\\${par_serial}_raidcom_get_snapshot.txt") 
					or warn "Cannot create ${par_horcm_dirname}\\${par_serial}_raidcom_get_snapshot.txt: $!\n";
					
					while (my $line = <CMD_RAIDCOM_GET_SNAPSHOT>)
					{
						print FH_RAIDCOM_GET_SNAPSHOT $line;
										}
					close CMD_RAIDCOM_GET_SNAPSHOT;
					close FH_RAIDCOM_GET_SNAPSHOT;
					LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Finished $CCI_raidcom get snapshot -fx -I${random_number}");
					
					# -	raidcom get ldev -ldev_list defined -fx
					LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Run $CCI_raidcom get ldev -ldev_list defined -fx -I${random_number}");
					open (CMD_RAIDCOM_GET_LDEVLIST_DEF, "$CCI_raidcom get ldev -ldev_list defined -fx -I${random_number} 2>&1|") or warn ("!!! Can't run program: $!\n");
					
					open (FH_RAIDCOM_GET_LDEVLIST_DEF, ">${par_horcm_dirname}\\${par_serial}_raidcom_get_ldevlist_def.txt") 
					or warn "Cannot create ${par_horcm_dirname}\\${par_serial}_raidcom_ldevlist_def.txt: $!\n";
					
					while (my $line = <CMD_RAIDCOM_GET_LDEVLIST_DEF>)
					{
						print FH_RAIDCOM_GET_LDEVLIST_DEF $line;
					}
					close CMD_RAIDCOM_GET_LDEVLIST_DEF;
					close FH_RAIDCOM_GET_LDEVLIST_DEF;

					LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Finished $CCI_raidcom get ldev -ldev_list defined -fx -I${random_number}");
					
					# -	raidcom get port
					my @all_ports;
					LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Run $CCI_raidcom get port -fx -I${random_number}");
					open (CMD_RAIDCOM_GET_PORT, "$CCI_raidcom get port -fx -I${random_number} 2>&1|") or warn ("!!! Can't run program: $!\n");
					
					open (FH_RAIDCOM_GET_PORT, ">${par_horcm_dirname}\\${par_serial}_raidcom_get_port.txt") 
					or warn "Cannot create ${par_horcm_dirname}\\${par_serial}_raidcom_get_port.txt: $!\n";
										
					while (my $line = <CMD_RAIDCOM_GET_PORT>)
					{
						print FH_RAIDCOM_GET_PORT $line;
						if ($line =~ m/^(CL[-\w]+)\s+/)
						{
							my $port = $1;
							push(@all_ports,$port);
							LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Port found: \"$port\"");
						}

					}
					close CMD_RAIDCOM_GET_PORT;
					close FH_RAIDCOM_GET_PORT;
					LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Finished $CCI_raidcom get port -fx -I${random_number}");
					
					LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Making ports unique");
					# Making ports unique as Gx00 is not unique with get ports
					my %port_seen = ();
					my @ports_uniq = ();
					foreach my $item (@all_ports) 
					{
						unless ($port_seen{$item})
						{
							# if we get here, we have not seen it before
							$port_seen{$item} = 1;
							LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Port unique: \"$item\"");
							push(@ports_uniq, $item);
						}
					}
					
				
					# -	raidcom get detailed information of every port							
					if (@ports_uniq) 
					{
						LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Ports founds therefore detailed port information gets collected");
						
						foreach my $fed_port (@ports_uniq) 
						{
							if ($par_raidcom_level < 2)
							{							
								# -	raidcom get host_grp -port [port name] -key host_grp
								LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Run $CCI_raidcom get host_grp -port $fed_port -key host_grp -fx -I${random_number}");
								open (CMD_RAIDCOM_GET_HOSTGRP_PORT, "$CCI_raidcom get host_grp -port $fed_port -key host_grp -fx -I${random_number} 2>&1|") or warn ("!!! Can't run program: $!\n");
								
								open (FH_RAIDCOM_GET_HOSTGRP_PORT, ">${par_horcm_dirname}\\${par_serial}_raidcom_get_hostgrp_port_${fed_port}.txt") 
								or warn "Cannot create ${par_horcm_dirname}\\${par_serial}_raidcom_get_hostgrp_port_${fed_port}.txt: $!\n";
															
								my $fields_get_hostgrpport_size;
								while (my $line = <CMD_RAIDCOM_GET_HOSTGRP_PORT>)
								{
									print FH_RAIDCOM_GET_HOSTGRP_PORT $line;
							
								}
								close CMD_RAIDCOM_GET_HOSTGRP_PORT;
								close FH_RAIDCOM_GET_HOSTGRP_PORT;
								LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Finished $CCI_raidcom get host_grp -port $fed_port -key host_grp -fx -I${random_number}");
							}
							
							if ($par_raidcom_level < 1)
							{
								# -	raidscan -p [port name] -fx -CLI -ISI=> Shadow Image
								LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Run $CCI_raidscan -CLI -p $fed_port -fx -ISI${random_number}");
								open (CMD_RAIDSCAN_PORT_SI, "$CCI_raidscan -CLI -p $fed_port -fx -ISI${random_number} 2>&1|") or warn ("!!! Can't run program: $!\n");
								
								open (FH_RAIDSCAN_PORT_SI, ">${par_horcm_dirname}\\${par_serial}_raidscan_port_SI_${fed_port}.txt")
								or warn "Cannot create ${par_horcm_dirname}\\${par_serial}_raidscan_port_SI_${fed_port}.txt: $!\n";
								
								
								while (my $line = <CMD_RAIDSCAN_PORT_SI>)
								{
									print FH_RAIDSCAN_PORT_SI $line;
																
								}
								close CMD_RAIDSCAN_PORT_SI;
								close FH_RAIDSCAN_PORT_SI;
								LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Finished $CCI_raidscan -p $fed_port -fx -ISI${random_number}");
								
								# -	raidscan -p [port name] -fx -m all -CLI -ITC => TrueCopy
								LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Run $CCI_raidscan -CLI -p $fed_port -fx -m all -ITC${random_number}");
								open (CMD_RAIDSCAN_PORT_TC, "$CCI_raidscan -CLI -p $fed_port -fx -m all -ITC${random_number} 2>&1|") or warn ("!!! Can't run program: $!\n");
								
								open (FH_RAIDSCAN_PORT_TC, ">${par_horcm_dirname}\\${par_serial}_raidscan_port_TC_${fed_port}.txt")
								or warn "Cannot create ${par_horcm_dirname}\\${par_serial}_raidscan_port_TC_${fed_port}.txt: $!\n";
														
								while (my $line = <CMD_RAIDSCAN_PORT_TC>)
								{
									print FH_RAIDSCAN_PORT_TC $line;													
								}
								close CMD_RAIDSCAN_PORT_TC;
								close FH_RAIDSCAN_PORT_TC;
								LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Finished $CCI_raidscan -p $fed_port -fx -m all -ITC${random_number}");	
							}
						}
						
					}
					else 
					{ 
						LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RAIDCOM","No ports founds therefore no detailed port information gets collected");
					}
					
					#  raidcom get parity_grp -I135 -fx
					LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Run $CCI_raidcom get parity_grp -fx -I${random_number}");
					open (CMD_RAIDCOM_GET_PARITY, "$CCI_raidcom get parity_grp -fx -I${random_number} 2>&1|") or warn ("!!! Can't run program: $!\n");
					
					open (FH_RAIDCOM_GET_PARITY, ">${par_horcm_dirname}\\${par_serial}_raidcom_get_parity.txt") 
					or warn "Cannot create ${par_horcm_dirname}\\${par_serial}_raidcom_get_parity.txt: $!\n";
					
					while (my $line = <CMD_RAIDCOM_GET_PARITY>)
					{
						print FH_RAIDCOM_GET_PARITY $line;
												
					}
					close CMD_RAIDCOM_GET_PARITY;
					close FH_RAIDCOM_GET_PARITY;
					LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Finished $CCI_raidcom get parity_grp -fx -I${random_number}");
					
										
					LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Run $CCI_raidcom get parity_grp -key opt -fx -I${random_number}");
					open (CMD_RAIDCOM_GET_PARITY_OPT, "$CCI_raidcom get parity_grp -key opt -fx -I${random_number} 2>&1|") or warn ("!!! Can't run program: $!\n");
					
					open (FH_RAIDCOM_GET_PARITY_OPT, ">${par_horcm_dirname}\\${par_serial}_raidcom_get_parity_key_opt.txt") 
					or warn "Cannot create ${par_horcm_dirname}\\${par_serial}_raidcom_get_parity_key_opt.txt: $!\n";
					
					while (my $line = <CMD_RAIDCOM_GET_PARITY_OPT>)
					{
						print FH_RAIDCOM_GET_PARITY_OPT $line;
							
					}
					close CMD_RAIDCOM_GET_PARITY_OPT;
					close FH_RAIDCOM_GET_PARITY_OPT;
				
					# raidcom logout
					LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Run CCI_raidcom logout -I${random_number}");
					open (CMD_RAIDCOM_LOGOUT, "$CCI_raidcom -logout -I${random_number} 2>&1|") or warn ("!!! Can't run program: $!\n");
					while (my $line = <CMD_RAIDCOM_LOGOUT>)
					{
						LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","$line");
					}
					close CMD_RAIDCOM_LOGOUT;
									
					# Stop horcm instance
					LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Stopping horcm instance");
					open (CMD_HORCMSHUTDOWN, "$CCI_horcmshutdown 2>&1|") or warn ("!!! Can't run program: $!\n");
					while (my $line = <CMD_HORCMSHUTDOWN>)
					{
						LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","$line");
					}
					close CMD_HORCMSHUTDOWN;
					my $exit_CMD_HORCMSHUTDOWN = $?;
					if ($exit_CMD_HORCMSHUTDOWN != 0)
					{
						LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RAIDCOM","Failed to stop HORCM instance.");
					}
					LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Deleting ${horcmconf}${random_number}.conf");
					unlink "${horcmconf}${random_number}.conf" or warn ("Could not delete ${horcmconf}${random_number}.conf: $!");
				}
				
				else
				{
					LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RAIDCOM","Leaving raidcom subroutine.");
					# Stop horcm instance
					LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Stopping horcm instance");
					open (CMD_HORCMSHUTDOWN, "$CCI_horcmshutdown 2>&1|") or warn ("!!! Can't run program: $!\n");
					while (my $line = <CMD_HORCMSHUTDOWN>)
					{
						LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","$line");
					}
					close CMD_HORCMSHUTDOWN;
					my $exit_CMD_HORCMSHUTDOWN = $?;
					if ($exit_CMD_HORCMSHUTDOWN != 0)
					{
						LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RAIDCOM","Failed to stop HORCM instance.");
					}
					LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOM","Deleting ${horcmconf}${random_number}.conf");
					unlink "${horcmconf}${random_number}.conf" or warn ("Could not delete ${horcmconf}${random_number}.conf: $!");
			
				}
			}
			
			else
			{
				LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RAIDCOM","Leaving raidcom subroutine.");		
			}
			
		}
		
		else
		{
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RAIDCOM","Leaving raidcom subroutine.");
		}
	
	}

	else
	{
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RAIDCOM","Leaving raidcom subroutine.");
	}
		
}


sub CHECK_HORCM_LOGIN
{
	my $par_HOST = $_[0];
	my $par_serial = $_[1];
	my $par_flag_cci_dir = $_[2];
	my $par_log_fh = $_[3];
	my $par_vbse = $_[4];
	my $par_lglvl = $_[5];
	
	my $horcm_usr_var = "${par_flag_cci_dir}\\usr\\var\\${par_HOST}*${par_serial}";

	#my $SCRIPTLOG_DEL_PATH = "${scriptdir_log}\\${script_name}_${flag_system}_${serial}_*.log";
	 
	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOMCHECK","Raidcom login check start");
	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOMCHECK","Looking for files: $horcm_usr_var ");
	my @login_files = <${horcm_usr_var}>;
	my $check_flag = "bad";
	if (@login_files)
	{
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOMCHECK","Files found");
		foreach my $file (@login_files) 
		{
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOMCHECK","File: $file");
			
			if ($file =~ m/^${par_HOST}_SYSTEM_P/)
			{
				LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOMCHECK","REST API file, not relevant for raidcom");
			}
			else
			{
				LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOMCHECK","Raidcom login file, good");
				$check_flag = "good";
			}
		}
		
		if  ($check_flag eq "bad")
		{
			LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RAIDCOMCHECK","No raidcom login file found. Re-login required. (bad-flag)");
		}
	}
	else
	{
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RAIDCOMCHECK","No raidcom login file found. Re-login required.");
	}
	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOMCHECK","Raidcom login check finished");
	
	return $check_flag;
	
	# raidcom login check would look like ...
	# my $login_check_flag = "bad";
	# $login_check_flag = CHECK_HORCM_LOGIN($par_HOST,$par_serial,$par_flag_cci_dir,$par_log_fh,$par_vbse,$par_lglvl);
	# if ($login_check_flag eq "bad") {LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RAIDCOM","Raidcom login issue. Stop raidcom collection."); return -1;}

}

sub CHECK_OLD_INSTANCE
{

	my $par_flag_cci_dir = $_[0];
	my $par_horcm_dirname = $_[1];
	my $par_serial = $_[2];
	my $par_log_fh = $_[3];
	my $par_vbse = $_[4];
	my $par_lglvl = $_[5];

	my $horcmconf = "${par_horcm_dirname}\\horcm";

	my $CCI_horcmstart = "${par_flag_cci_dir}\\etc\\horcmstart.exe";
	my $CCI_horcmshutdown = "${par_flag_cci_dir}\\etc\\horcmshutdown.exe";
	my $CCI_horcctl = "${par_flag_cci_dir}\\etc\\horcctl.exe";
	
	my $skipraidcom = "no";
	
	
	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOMOLD","Starting horcm check of old instances");
	
	unless (-e $CCI_horcmstart and -e $CCI_horcmshutdown and -e $CCI_horcctl)
	{
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RAIDCOMOLD","$CCI_horcmstart or $CCI_horcmshutdown or $CCI_horcctl does not exist.");
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RAIDCOMOLD","raidcom horcm check of old intstance will NOT be done collected.");
		$skipraidcom = "yes";
	}
	
		
	if ($skipraidcom eq "no")
	{
	
		my @horcmfiles = <${horcmconf}*.conf>;
		if (@horcmfiles)
		{
		
		LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOMOLD","Old horcm file found");
						
			foreach my $file (@horcmfiles) 
			{
				LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOMOLD","File: ${file}");
				if ($file =~ /horcm(\d+).conf$/)
				{
					# Set windows environment variables
					my $instance = $1;
					$ENV{HORCMINST} = $instance;
					$ENV{HORCM_CONF} = $file;
					LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOMOLD","Set windows environment vars");
					LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOMOLD","\$ENV{HORCM_CONF} = $ENV{HORCM_CONF}");
					LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOMOLD","\$ENV{HORCMINST} = $ENV{HORCMINST} ");
				
					LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOMOLD","Checking HORCCTL");
					open (CMD_HORCCTL, "$CCI_horcctl -D -I${instance} 2>&1|") or warn LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RAIDCOMOLD","!!! Can't run program: $!\n");
					while (my $line = <CMD_HORCCTL>)
					{
						LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOMOLD","$line");
					}
					close CMD_HORCCTL;
					my $exit_CMD_HORCCTL = $?;
				
					# If the command device is not valid delete the file
					if($exit_CMD_HORCCTL != 0)
					{
						LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOMOLD","HORCM instance ${instance} already stopped");
						LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOMOLD","Deleting ${file}");
						unlink "${file}" or warn LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RAIDCOMOLD","!!! Could not delete ${file}: $!");
					}
					# If the command device is valid shutdown the instance and delete the file
					else
					{
						# Stop horcm instance
						LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOMOLD","Stopping horcm instance ${instance}");
						open (CMD_HORCMSHUTDOWN, "$CCI_horcmshutdown 2>&1|") or warn LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RAIDCOMOLD","!!! Can't run program: $!\n");
						while (my $line = <CMD_HORCMSHUTDOWN>)
						{
							LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOMOLD","$line");
						}
						close CMD_HORCMSHUTDOWN;
						my $exit_CMD_HORCMSHUTDOWN = $?;
					
						if ($exit_CMD_HORCMSHUTDOWN != 0)
						{
							LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RAIDCOMOLD","Failed to stop HORCM instance.");
						}
						
						LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOMOLD","Deleting ${file}");
						unlink "${file}" or warn LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,4,"RAIDCOMOLD","!!! Could not delete ${file}: $!");						
					}
					
				}
				
			}
		}
	
	}
	LOG_MSG($par_log_fh,$par_vbse,"n",$par_lglvl,5,"RAIDCOMOLD","Finished horcm check of old instances");
}

1; # End of the module with a true value