# coding:utf-8

"""
auditlog_syslog_server setting

This program requires API version 1.4.x or newer.
"""

import requests
import json
import sys
import http.client
import time
import traceback
import rest_server_param
import storage_param

from block_storage_api import BlockStorageAPI

# #################Initialize parameters################# #
# Change the following parameters to fit your environment

# A path of root certificate
ROOT_CERT_FILE_PATH = "D:\cert\\"

# A root certificate name
ROOT_CERT_FILE_NAME = "root.crt"

# A path of client certificate
CLIENT_CERT_FILE_PATH = "D:\cert\\"

# A client certificate name
CLIENT_CERT_FILE_NAME = "client.pfx"

# A transfer protocol
TRANSFER_PROTOCOL = "TLS"

# A location name
LOCATION_NAME = "STORAGE_SYSTEM_1"

# A retry interval
RETRY_INTERVAL = 5

# A primary syslog server IP address
PRIMARY_SYSLOG_SERVER_IP_ADDR = "192.0.1.101"

# A primary syslog server port number
PRIMARY_SYSLOG_SERVER_PORT = "12345"

# A password of the client certificate
CLIENT_CERT_FILE_PASSWORD = "certFilePass"

# This parameter defines the first interval to access
# an asynchronous job. (Unit: Second)
FIRST_WAIT_TIME = 1

# This parameter defines the maximum retry time
# to confirm job status.
MAX_RETRY_COUNT = 6

# An user id and password of the target storage
USER_CREDENTIAL = ("user1", "pass1")

###########################################################

# ###You don't have to change the following parameters### #
block_storage_api = BlockStorageAPI(
    rest_server_param.REST_SERVER_IP_ADDR,
    rest_server_param.REST_SERVER_PORT,
    storage_param.STORAGE_MODEL,
    storage_param.SERIAL_NUMBER)

headers = {"content-type": "application/json",
           "accept": "application/json"}
file_upload_headers = {"accept": "application/json"}

REQUIRED_MAJOR_VERSION = 1
REQUIRED_MINOR_VERSION = 4

session_id = 0

###########################################################

"""
Check whether the asynchronous command was finished.
@param job_id the job ID to identify
       the asynchronous command
@return r the response data
"""


def check_update(job_id):
    url = block_storage_api.job(str(job_id))
    r = requests.get(url, headers=headers, verify=False)
    return r


"""
Execute the HTTP request (POST or PUT)
@param method_type HTTP request method (POST or PUT)
@param url URL to execute HTTP method
@param body The information of a resource
@return job_result.json()["affectedResources"][0]
         URL of an affected resource
"""


def invoke_async_command(method_type, url, body):
    if method_type == "put":
        if body is None:
            r = requests.put(url, headers=headers, verify=False)
        else:
            r = requests.put(url, headers=headers,
                             data=json.dumps(body), verify=False)
    elif method_type == "post":
        r = requests.post(
            url,
            headers=headers,
            data=json.dumps(body),
            verify=False)
    if r.status_code != http.client.ACCEPTED:
        raise requests.HTTPError(r)
    print("Request was accepted. JOB URL : " +
          r.json()["self"])
    status = "Initializing"
    job_result = None
    retry_count = 1
    wait_time = FIRST_WAIT_TIME
    while status != "Completed":
        if retry_count > MAX_RETRY_COUNT:
            raise Exception("Timeout Error! "
                            "Operation was not completed.")
        time.sleep(wait_time)
        job_result = check_update(r.json()["jobId"])
        status = job_result.json()["status"]
        double_time = wait_time * 2
        if double_time < 120:
            wait_time = double_time
        else:
            wait_time = 120
        retry_count += 1
    if job_result.json()["state"] == "Failed":
        error_obj = job_result.json()["error"]
        if "errorCode" in error_obj:
            if "SSB1" in error_obj["errorCode"]:
                print("Error! SSB code : ",
                      error_obj["errorCode"]["SSB1"],
                      ", ", error_obj["errorCode"]["SSB2"])
            elif "errorCode" in error_obj["errorCode"]:
                print("Error! error code : ",
                      error_obj["errorCode"]["errorCode"])
        raise Exception("Job Error!", job_result.text)
    if "affectedResources" in job_result.json():
        print("Async job was succeeded. affected resource : " +
              job_result.json()["affectedResources"][0])
        return job_result.json()["affectedResources"][0]
    else:
        print("Async job was succeeded.")
        return None


"""
Check whether this API version allows the REST
 Server to execute this program

@param api_version api version of this REST Server
@param required_major_version the lowest number of
       the major version that this program requires
@param required_minor_version the lowest number of
       the minor version that this program requires

"""


def check_api_version(api_version, required_major_version,
                      required_minor_version):
    version = api_version.split(".")
    major_version = int(version[0])
    minor_version = int(version[1])
    if not ((major_version == required_major_version and
             minor_version >= required_minor_version) or
            major_version >= required_major_version + 1):
        sys.exit("This program requires API Version " +
                 str(required_major_version) + "." +
                 str(required_minor_version) +
                 "." + "x or newer.\n")

try:
    # step1 Check the API version #
    print("Check the API version")
    url = block_storage_api.api_version()
    r = requests.get(url, headers=headers, verify=False)
    if r.status_code != http.client.OK:
        raise requests.HTTPError(r)
    check_api_version(
        r.json()["apiVersion"],
        REQUIRED_MAJOR_VERSION,
        REQUIRED_MINOR_VERSION)

    # step2 Generate a session #
    print("Generate a session")
    url = block_storage_api.generate_session()
    r = requests.post(url, headers=headers, auth=USER_CREDENTIAL,
                      verify=False)
    if r.status_code != http.client.OK:
        raise requests.HTTPError(r)
    token = r.json()["token"]
    auth = "Session " + token
    session_id = r.json()["sessionId"]
    headers["Authorization"] = auth
    file_upload_headers["Authorization"] = auth

    # step3 Upload a root certificate #
    print("Upload a root certificate")
    url = block_storage_api.file_upload()
    files = {"file": (ROOT_CERT_FILE_NAME, open(ROOT_CERT_FILE_PATH + ROOT_CERT_FILE_NAME, "rb"), "")}
    r = requests.post(url, headers=file_upload_headers, data={"fileType": "AuditSyslogPrimaryRootCertFile"},
                      files=files, verify=False)
    if r.status_code != http.client.OK:
        raise requests.HTTPError(r)

    # step4 Upload a client certificate #
    print("Upload a client certificate")
    files = {"file": (CLIENT_CERT_FILE_NAME, open(CLIENT_CERT_FILE_PATH + CLIENT_CERT_FILE_NAME, "rb"), "")}
    r = requests.post(url, headers=file_upload_headers, data={"fileType": "AuditSyslogPrimaryClientCertFile"},
                      files=files, verify=False)
    if r.status_code != http.client.OK:
        raise requests.HTTPError(r)

    # step5 Modify the syslog server #
    print("Modify the syslog server")
    url = block_storage_api.auditlog_syslog()
    body = {
        "transferProtocol": TRANSFER_PROTOCOL,
        "locationName": LOCATION_NAME,
        "retries": True,
        "retryInterval": RETRY_INTERVAL,
        "primarySyslogServer": {
            "isEnabled": True,
            "ipAddress": PRIMARY_SYSLOG_SERVER_IP_ADDR,
            "port": PRIMARY_SYSLOG_SERVER_PORT,
            "clientCertFileName": CLIENT_CERT_FILE_NAME,
            "clientCertFilePassword": CLIENT_CERT_FILE_PASSWORD,
            "rootCertFileName": ROOT_CERT_FILE_NAME
        },
        "secondarySyslogServer": {
            "isEnabled": False
        }
    }
    invoke_async_command("put", url, body)

    # step6 Send a test message to the syslog server #
    print("Send a test message to the syslog server")
    url = block_storage_api.auditlog_syslog_send_test()
    invoke_async_command("put", url, None)

    # step7 Get the syslog server #
    print("get the syslog server")
    url = block_storage_api.auditlog_syslog()
    r = requests.get(url, headers=headers, verify=False)
    if r.status_code != http.client.OK:
        raise requests.HTTPError(r)
    print("TRANSFER PROTOCOL : " + str(r.json()["transferProtocol"]))
    print("LOCATION NAME : " + str(r.json()["locationName"]))
    print("RETRIES : " + str(r.json()["retries"]))
    print("RETRY INTERVAL : " + str(r.json()["retryInterval"]))
    print("PRIMARY SYSLOG SERVER")
    print("IP ADDRESS : " + str(r.json()["primarySyslogServer"]["ipAddress"]))
    print("PORT : " + str(r.json()["primarySyslogServer"]["port"]))

except requests.ConnectionError:
    sys.stderr.write("Connection Error!\n")
    sys.stderr.write(traceback.format_exc())
except requests.HTTPError as he:
    sys.stderr.write("HTTP Error! status code : ")
    sys.stderr.write(str(he.args[0].status_code) + "\n")
    sys.stderr.write(he.args[0].text + "\n")
except Exception as e:
    sys.stderr.write(traceback.format_exc())
    for msg in e.args:
        sys.stderr.write(str(msg) + "\n")
finally:
    # step8 Discard the session #
    print("Discard the session")
    url = block_storage_api.discard_session(session_id)
    r = requests.delete(url, headers=headers, verify=False)
    try:
        if r.status_code != http.client.OK:
            raise requests.HTTPError(r)
    except requests.HTTPError as he:
        sys.stderr.write("HTTP Error! status code : ")
        sys.stderr.write(str(he.args[0].status_code) + "\n")
        sys.stderr.write(he.args[0].text + "\n")

    print("Operation was completed.")
    sys.exit()
