# coding:utf-8
"""
This class creates a URL for the REST API
to manage block storage
"""

STORAGE_MODEL_DICT = {"VSP G800": "836000",
                      "VSP G600": "834000",
                      "VSP G400": "834000",
                      "VSP G200": "832000",
                      "VSP F800": "836000",
                      "VSP F600": "834000",
                      "VSP F400": "834000"}

class BlockStorageAPI():
    # SVP IP address
    svp_ip_addr = None
    # port number
    port = None
    # storage URL
    base_url = None
    # object URL
    object_url = None
    # service URL
    service_url = None
    # storage device ID
    storage_id = None

    def __init__(self, svp_ip_addr, port, storage_model,
                 serial_number):
        self.svp_ip_addr = svp_ip_addr
        self.port = port
        self.storage_id = STORAGE_MODEL_DICT[storage_model]\
            + serial_number
        self.base_url = "https://" + \
            self.svp_ip_addr + ":" + self.port + \
            "/ConfigurationManager/v1"
        self.object_url = "/objects/storages/" + self.storage_id
        self.service_url = "/" + self.storage_id + "/services"

    def get_storage_id(self):
        return self.storage_id

    def ldevs(self):
        url = self.base_url + self.object_url + "/ldevs"
        return url

    def undefined_ldev(self):
        url = self.ldevs() + \
            "?ldevOption=undefined&count=1"
        return url

    def ldev(self, object_id):
        url = self.ldevs() + "/" + str(object_id)
        return url

    def views_ldevs(self):
        url = self.base_url + "/views/ldevs?$query=ldev.storageDeviceId eq '" + \
              self.storage_id + "'"
        return url

    def views_undefined_ldev(self):
        url = self.views_ldevs() + \
            "&$query=ldev.isDefined eq false&$count=1"
        return url

    def host_groups(self):
        url = self.base_url + self.object_url + "/host-groups"
        return url

    def host_wwns(self):
        url = self.base_url + self.object_url + "/host-wwns"
        return url

    def luns(self):
        url = self.base_url + self.object_url + "/luns"
        return url

    def local_copy_pairs(self):
        url = self.base_url + self.object_url + "/local-clone-copypairs"
        return url

    def split_local_copy_pair_template(self, pair_url):
        url = pair_url + "/actions/split"
        return url

    def split_local_copy_pair(self, pair_url):
        url = pair_url + "/actions/split/invoke"
        return url

    def generate_session(self):
        url = self.base_url + self.object_url + "/sessions"
        return url

    def discard_session(self, object_id):
        url = self.base_url + self.object_url + "/sessions/" + \
            str(object_id)
        return url

    def lock(self):
        url = self.base_url + self.service_url + \
            "/resource-group-service/" + \
            "actions/lock/invoke"
        return url

    def unlock(self):
        url = self.base_url + self.service_url + \
            "/resource-group-service/" + \
            "actions/unlock/invoke"
        return url

    def remote_storage(self):
        url = self.base_url + self.object_url + "/remote-storages"
        return url

    def remote_copy_pairs(self):
        url = self.base_url + self.object_url +\
            "/remote-mirror-copypairs"
        return url
        
    def job(self, object_id):
        url = self.base_url + self.object_url + "/jobs/" + \
            str(object_id)
        return url

    def affected_resource(self, affected_resource):
        url = "https://" + self.svp_ip_addr + ":" + \
              self.port \
              + affected_resource
        return url

    def api_version(self):
        url = "https://" + self.svp_ip_addr + ":" + \
              self.port \
            + "/ConfigurationManager/configuration/version"
        return url
    
    def local_storages(self):
        url = self.base_url + "/objects/storages"
        return url
    
    def local_storage(self, object_id):
        url = self.local_storages() + "/" + str(object_id)
        return url

    def file_upload(self):
        url = self.base_url + self.object_url + "/actions/file-upload/invoke"
        return url

    def auditlog_syslog(self):
        url = self.base_url + self.object_url + "/auditlog-syslog-servers" + \
                "/instance"
        return url

    def auditlog_syslog_send_test(self):
        url = self.auditlog_syslog() + "/actions/send-test/invoke"
        return url
