# coding:utf-8

"""
local_copy

This program requires API version 1.2.x or newer.
"""

import traceback
import requests
import json
import sys
import http.client
import time
import rest_server_param
import storage_param

from block_storage_api import BlockStorageAPI

# #################Initialize parameters################# #
# Change the following parameters to fit your environment

# A copy group name
COPY_GROUP_NAME = "SI_347"

# A copy pair name
COPY_PAIR_NAME = "p_347-348"

# A primary volume ID
# Specify already created and allocated volume ID by decimal
PVOL_LDEV_ID = 347

# A secondary volume ID which has the exactly same size
# as the primary volume
# Specify already created and allocated volume ID by decimal
SVOL_LDEV_ID = 348


# This parameter defines the first interval to access
# an asynchronous job. (Unit: Second)
FIRST_WAIT_TIME = 1

# This parameter defines the maximum retry time
# to confirm job status.
MAX_RETRY_COUNT = 10

# An user id and password of the target storage
USER_CREDENTIAL = ("user1", "pass1")

###########################################################

# ###You don't have to change the following parameters### #
block_storage_api = BlockStorageAPI(
    rest_server_param.REST_SERVER_IP_ADDR,
    rest_server_param.REST_SERVER_PORT,
    storage_param.STORAGE_MODEL,
    storage_param.SERIAL_NUMBER)

headers = {"content-type": "application/json",
           "accept": "application/json",
           "Response-Job-Status": "Completed"}

REQUIRED_MAJOR_VERSION = 1
REQUIRED_MINOR_VERSION = 2

session_id = 0

###########################################################


"""
Check whether the asynchronous command was finished.
@param job_id The job ID to identify
              the asynchronous command
@return r.json() The JSON data that contains response data

"""


def check_update(job_id):
    url = block_storage_api.job(str(job_id))
    r = requests.get(url, headers=headers, verify=False)
    return r


"""
Execute the HTTP request (POST or PUT)
@param method_type HTTP request method (POST or PUT)
@param url URL to execute HTTP method
@param body The information of a resource
@return job_result.json()["affectedResources"][0]
         URL of an affected resource
"""


def invoke_async_command(method_type, url, body):
    if method_type == "put":
        r = requests.put(url, headers=headers,
                         data=json.dumps(body), verify=False)
    elif method_type == "post":
        r = requests.post(
            url,
            headers=headers,
            data=json.dumps(body),
            verify=False)
    if r.status_code != http.client.ACCEPTED:
        raise requests.HTTPError(r)
    print("Request was accepted. JOB URL : " +
          r.json()["self"])
    status = "Initializing"
    job_result = None
    retry_count = 1
    wait_time = FIRST_WAIT_TIME
    while status != "Completed":
        if retry_count > MAX_RETRY_COUNT:
            raise Exception("Timeout Error! "
                            "Operation was not completed.")
        time.sleep(wait_time)
        job_result = check_update(r.json()["jobId"])
        status = job_result.json()["status"]
        double_time = wait_time * 2
        if double_time < 120:
            wait_time = double_time
        else:
            wait_time = 120
        retry_count += 1
    if job_result.json()["state"] == "Failed":
        error_obj = job_result.json()["error"]
        if "errorCode" in error_obj:
            if "SSB1" in error_obj["errorCode"]:
                print("Error! SSB code : ",
                      error_obj["errorCode"]["SSB1"],
                      ", ", error_obj["errorCode"]["SSB2"])
            elif "errorCode" in error_obj["errorCode"]:
                print("Error! error code : ",
                      error_obj["errorCode"]["errorCode"])
        raise Exception("Job Error!", job_result.text)
    print("Async job was succeeded. affected resource : " +
          job_result.json()["affectedResources"][0])
    return job_result.json()["affectedResources"][0]

"""
Check whether this API version allows the REST
 Server to execute this program

@param api_version api version of this REST Server
@param required_major_version the lowest number of
       the major version that this program requires
@param required_minor_version the lowest number of
       the minor version that this program requires

"""


def check_api_version(api_version, required_major_version,
                      required_minor_version):
    version = api_version.split(".")
    major_version = int(version[0])
    minor_version = int(version[1])
    if not ((major_version == required_major_version and
             minor_version >= required_minor_version) or
            major_version >= required_major_version + 1):
        sys.exit(
            "This program requires API Version " +
            str(required_major_version) +
            "." +
            str(required_minor_version) +
            "." +
            "x or newer.\n")

try:
    # step1 Check the API version #
    print("Check the API version")
    url = block_storage_api.api_version()
    r = requests.get(url, headers=headers, verify=False)
    if r.status_code != http.client.OK:
        raise requests.HTTPError(r)
    check_api_version(
        r.json()["apiVersion"],
        REQUIRED_MAJOR_VERSION,
        REQUIRED_MINOR_VERSION)

    # step2 Generate a session #
    print("Generate a session")
    url = block_storage_api.generate_session()
    r = requests.post(url, headers=headers,
                      auth=USER_CREDENTIAL, verify=False)
    if r.status_code != http.client.OK:
        raise requests.HTTPError(r)
    token = r.json()["token"]
    auth = "Session " + token
    session_id = r.json()["sessionId"]

    # step3 Create a local copy pair #
    print("Create a local copy pair")
    url = block_storage_api.local_copy_pairs()
    body = {
        "copyGroupName": COPY_GROUP_NAME,
        "copyPairName": COPY_PAIR_NAME,
        "replicationType": "SI",
        "pvolLdevId": PVOL_LDEV_ID,
        "pvolMuNumber": 0,
        "svolLdevId": SVOL_LDEV_ID,
        "isNewGroupCreation": True,
    }
    headers["Authorization"] = auth
    affected_resource = invoke_async_command("post",
                                             url, body)
    pair_url = block_storage_api.affected_resource(
        affected_resource)

    # step4 Split the local copy pair #
    print("Split the local copy pair")
    url = block_storage_api.split_local_copy_pair_template(
        pair_url)
    r = requests.get(url, headers=headers, verify=False)
    if r.status_code != http.client.OK:
        raise requests.HTTPError(r)
    print("Action template(split):")
    print(r.text)
    body = r.json()
    body["parameters"]["copyPace"] = 3
    split_url = block_storage_api.split_local_copy_pair(
        pair_url)
    invoke_async_command("put", split_url, body)

    # step5 Print the pair status #
    print("Print the pair status")
    r = requests.get(pair_url,
                     headers=headers, verify=False)
    if r.status_code != http.client.OK:
        raise requests.HTTPError(r)

    print("COPY GROUP NAME : " + r.json()["copyGroupName"])
    print("COPY PAIR NAME : " + r.json()["copyPairName"])
    print("P-VOL LDEV ID : " + str(r.json()["pvolLdevId"]))
    print("S-VOL LDEV ID : " + str(r.json()["svolLdevId"]))
    print("P-VOL STATUS : " + r.json()["pvolStatus"])
    print("S-VOL STATUS : " + r.json()["svolStatus"])
    print("LOCAL CLONE COPY PAIR ID : " +
          r.json()["localCloneCopypairId"])
    print()

except requests.ConnectionError:
    sys.stderr.write("Connection Error!\n")
    sys.stderr.write(traceback.format_exc())
except requests.HTTPError as he:
    sys.stderr.write("HTTP Error! status code : ")
    sys.stderr.write(str(he.args[0].status_code) + "\n")
    sys.stderr.write(he.args[0].text + "\n")
except Exception as e:
    sys.stderr.write(traceback.format_exc())
    for msg in e.args:
        sys.stderr.write(str(msg) + "\n")
finally:
    # step6 Discard the session #
    print("Discard the session")
    url = block_storage_api.discard_session(session_id)
    r = requests.delete(url, headers=headers, verify=False)
    try:
        if r.status_code != http.client.OK:
            raise requests.HTTPError(r)
    except requests.HTTPError as he:
        sys.stderr.write("HTTP Error! status code : ")
        sys.stderr.write(str(he.args[0].status_code) + "\n")
        sys.stderr.write(he.args[0].text + "\n")

    print("Operation was completed.")
    sys.exit()
