﻿# coding:utf-8

"""
provisioning

This program requires API version 1.3.x or newer.
"""

import requests
import json
import sys
import http.client
import time
import traceback
import rest_server_param
import storage_param


from block_storage_api import BlockStorageAPI

# #################Initialize parameters################# #
# Change the following parameters to fit your environment

# A POOL ID for creating a volume
POOL_ID = 8

# The DP volume capacity to create
BYTE_CAPACITY = "1T"

# A port name to add a LUN path
PORT_ID = "CL1-A"

# A host group name to create
# You can assign any host group name
HOST_GRP_NAME = "WindowsHost"

# A Host mode for the created host group
# Please refer to the manual and set an appropriate mode
HOST_MODE = "WIN"

# A World Wide Name of the host (HBA) to allocate the volume
HOST_WWN = "aaaabbbbcccc0123"


# This parameter defines the first interval to access
# an asynchronous job. (Unit: Second)
FIRST_WAIT_TIME = 1

# This parameter defines the maximum retry time
# to confirm job status.
MAX_RETRY_COUNT = 6

# An user id and password of the target storage
USER_CREDENTIAL = ("user1", "pass1")

###########################################################

# ###You don't have to change the following parameters### #
block_storage_api = BlockStorageAPI(
    rest_server_param.REST_SERVER_IP_ADDR,
    rest_server_param.REST_SERVER_PORT,
    storage_param.STORAGE_MODEL,
    storage_param.SERIAL_NUMBER)

headers = {"content-type": "application/json",
           "accept": "application/json"}

REQUIRED_MAJOR_VERSION = 1
REQUIRED_MINOR_VERSION = 3

session_id = 0

###########################################################


"""
Check whether the asynchronous command was finished.
@param job_id the job ID to identify
       the asynchronous command
@return r the response data
"""


def check_update(job_id):
    url = block_storage_api.job(str(job_id))
    r = requests.get(url, headers=headers, verify=False)
    return r


"""
Execute the HTTP request (POST or PUT)
@param method_type HTTP request method (POST or PUT)
@param url URL to execute HTTP method
@param body The information of a resource
@return job_result.json()["affectedResources"][0]
         URL of an affected resource
"""


def invoke_async_command(method_type, url, body):
    if method_type == "put":
        r = requests.put(url, headers=headers,
                         data=json.dumps(body), verify=False)
    elif method_type == "post":
        r = requests.post(
            url,
            headers=headers,
            data=json.dumps(body),
            verify=False)
    if r.status_code != http.client.ACCEPTED:
        raise requests.HTTPError(r)
    print("Request was accepted. JOB URL : " +
          r.json()["self"])
    status = "Initializing"
    job_result = None
    retry_count = 1
    wait_time = FIRST_WAIT_TIME
    while status != "Completed":
        if retry_count > MAX_RETRY_COUNT:
            raise Exception("Timeout Error! "
                            "Operation was not completed.")
        time.sleep(wait_time)
        job_result = check_update(r.json()["jobId"])
        status = job_result.json()["status"]
        double_time = wait_time * 2
        if double_time < 120:
            wait_time = double_time
        else:
            wait_time = 120
        retry_count += 1
    if job_result.json()["state"] == "Failed":
        error_obj = job_result.json()["error"]
        if "errorCode" in error_obj:
            if "SSB1" in error_obj["errorCode"]:
                print("Error! SSB code : ",
                      error_obj["errorCode"]["SSB1"],
                      ", ", error_obj["errorCode"]["SSB2"])
            elif "errorCode" in error_obj["errorCode"]:
                print("Error! error code : ",
                      error_obj["errorCode"]["errorCode"])
        raise Exception("Job Error!", job_result.text)
    print("Async job was succeeded. affected resource : " +
          job_result.json()["affectedResources"][0])
    return job_result.json()["affectedResources"][0]

"""
Check whether this API version allows the REST
 Server to execute this program

@param api_version api version of this REST Server
@param required_major_version the lowest number of
       the major version that this program requires
@param required_minor_version the lowest number of
       the minor version that this program requires

"""


def check_api_version(api_version, required_major_version,
                      required_minor_version):
    version = api_version.split(".")
    major_version = int(version[0])
    minor_version = int(version[1])
    if not ((major_version == required_major_version and
             minor_version >= required_minor_version) or
            major_version >= required_major_version + 1):
        sys.exit("This program requires API Version " +
                 str(required_major_version) + "." +
                 str(required_minor_version) +
                 "." + "x or newer.\n")

try:
    # step1 Check the API version #
    print("Check the API version")
    url = block_storage_api.api_version()
    r = requests.get(url, headers=headers, verify=False)
    if r.status_code != http.client.OK:
        raise requests.HTTPError(r)
    check_api_version(
        r.json()["apiVersion"],
        REQUIRED_MAJOR_VERSION,
        REQUIRED_MINOR_VERSION)

    # step2 Generate a session #
    print("Generate a session")
    url = block_storage_api.generate_session()
    r = requests.post(url, headers=headers, auth=USER_CREDENTIAL,
                      verify=False)
    if r.status_code != http.client.OK:
        raise requests.HTTPError(r)
    token = r.json()["token"]
    auth = "Session " + token
    session_id = r.json()["sessionId"]

    # step3 Search for an unused LDEV ID #
    print("Search for an unused LDEV ID")

    url = block_storage_api.views_undefined_ldev()
    headers["Authorization"] = auth
    r = requests.get(url, headers=headers, verify=False)
    if r.status_code != http.client.OK:
        raise requests.HTTPError(r)
    # Get an unused LDEV ID (smallest value)
    if len(r.json()["data"]) is 0:
        raise Exception("Free LDEV ID does not exist")
    undefined_ldev = r.json()["data"][0]["ldev"]
    ldev_id = undefined_ldev["ldevId"]

    print("Free LDEV ID:", ldev_id)

    # step4 Add an LDEV #
    print("Add an LDEV")
    url = block_storage_api.ldevs()
    body = {
        "ldevId": ldev_id,
        "poolId": POOL_ID,
        "byteFormatCapacity": BYTE_CAPACITY
    }
    invoke_async_command("post", url, body)

    # step5 Add a host group #
    print("Add a host group")
    url = block_storage_api.host_groups()
    body = {
        "portId": PORT_ID,
        "hostGroupName": HOST_GRP_NAME
    }
    affected_resource_path = invoke_async_command("post",
                                                  url, body)
    url = block_storage_api.affected_resource(
        affected_resource_path)

    r = requests.get(url, headers=headers, verify=False)
    if r.status_code != http.client.OK:
        raise requests.HTTPError(r)

    host_group_number = r.json()["hostGroupNumber"]

    # step6 Modify the host group #
    print("Modify the host group")
    body = {
        "hostMode": HOST_MODE
    }
    invoke_async_command("put", url, body)

    # step7 Add an HBA WWN #
    print("Add an HBA WWN")
    url = block_storage_api.host_wwns()
    body = {
        "hostWwn": HOST_WWN,
        "portId": PORT_ID,
        "hostGroupNumber": host_group_number
    }
    invoke_async_command("post", url, body)

    # step8 Add a LUN path #
    print("Add a LUN path")
    url = block_storage_api.luns()
    body = {
        "ldevId": ldev_id,
        "portId": PORT_ID,
        "hostGroupNumber": host_group_number
    }
    invoke_async_command("post", url, body)

    # step9 Print the LDEV #
    print("Print the LDEV")
    url = block_storage_api.ldev(ldev_id)
    r = requests.get(url, headers=headers, verify=False)
    if r.status_code != http.client.OK:
        raise requests.HTTPError(r)

    print("LDEV ID : " + str(r.json()["ldevId"]))
    print("POOL ID : " + str(r.json()["poolId"]))
    print("CAPACITY : " +
          str(r.json()["byteFormatCapacity"]))
    print("PORT : " + str(r.json()["ports"]))
    print()

except requests.ConnectionError:
    sys.stderr.write("Connection Error!\n")
    sys.stderr.write(traceback.format_exc())
except requests.HTTPError as he:
    sys.stderr.write("HTTP Error! status code : ")
    sys.stderr.write(str(he.args[0].status_code) + "\n")
    sys.stderr.write(he.args[0].text + "\n")
except Exception as e:
    sys.stderr.write(traceback.format_exc())
    for msg in e.args:
        sys.stderr.write(str(msg) + "\n")
finally:
    # ----step10 Discard the session----#
    print("Discard the session")
    url = block_storage_api.discard_session(session_id)
    r = requests.delete(url, headers=headers, verify=False)
    try:
        if r.status_code != http.client.OK:
            raise requests.HTTPError(r)
    except requests.HTTPError as he:
        sys.stderr.write("HTTP Error! status code : ")
        sys.stderr.write(str(he.args[0].status_code) + "\n")
        sys.stderr.write(he.args[0].text + "\n")

    print("Operation was completed.")
    sys.exit()
