# coding:utf-8

"""
register_remote_storage

This program requires API version 1.3.x or newer.
"""

import traceback
import requests
import json
import sys
import http.client
import time
import remote_copy_param

from block_storage_api import BlockStorageAPI

# #################Initialize parameters################# #
# Change the following parameters to fit your environment

# This parameter defines the first interval to access
# an asynchronous job. (Unit: Second)
FIRST_WAIT_TIME = 1

# This parameter defines the maximum retry time
# to confirm job status.
MAX_RETRY_COUNT = 10

# An user id and password of the local storage
LOCAL_USER_CREDENTIAL = ("local_user", "local_pass")

# An user id and password of the remote storage
REMOTE_USER_CREDENTIAL = ("remote_user", "remote_pass")

###########################################################

# ###You don't have to change the following parameters### #
local_storage_api = BlockStorageAPI(
    remote_copy_param.LOCAL_REST_SERVER_IP_ADDR,
    remote_copy_param.LOCAL_PORT,
    remote_copy_param.LOCAL_STORAGE_MODEL,
    remote_copy_param.LOCAL_SERIAL_NUMBER)

remote_storage_api = BlockStorageAPI(
    remote_copy_param.REMOTE_REST_SERVER_IP_ADDR,
    remote_copy_param.REMOTE_PORT,
    remote_copy_param.REMOTE_STORAGE_MODEL,
    remote_copy_param.REMOTE_SERIAL_NUMBER)

local_headers = {"content-type": "application/json",
                 "accept": "application/json",
                 "Response-Job-Status": "Completed"}

remote_headers = {"content-type": "application/json",
                  "accept": "application/json",
                  "Response-Job-Status": "Completed"}

REQUIRED_MAJOR_VERSION = 1
REQUIRED_MINOR_VERSION = 3

local_session_id = 0
remote_session_id = 0

###########################################################


"""
Check whether the asynchronous command was finished.

@param storage_api storage_api
@param job_id the job ID to identify
the asynchronous command
@param headers the array of the http headers
@return r the response data
"""


def check_update(storage_api, job_id, headers):
    url = storage_api.job(str(job_id))
    r = requests.get(url, headers=headers, verify=False)
    return r

"""
Execute the HTTP request (POST or PUT)

@param storage_api storage_api
@param method_type HTTP request method (POST or PUT)
@param headers the array of the http headers
@param url URL to execute HTTP method
@param body The information of a resource
@return job_result.json()["affectedResources"][0]
         URL of an affected resource
"""


def invoke_async_command(storage_api, method_type, headers,
                         url, body):
    if method_type == "put":
        r = requests.put(url, headers=headers,
                         data=json.dumps(body), verify=False)
    elif method_type == "post":
        r = requests.post(
            url,
            headers=headers,
            data=json.dumps(body),
            verify=False)
    if r.status_code != http.client.ACCEPTED:
        raise requests.HTTPError(r)
    print("Request was accepted. JOB URL : " +
          r.json()["self"])
    status = "Initializing"
    job_result = None
    retry_count = 1
    wait_time = FIRST_WAIT_TIME
    while status != "Completed":
        if retry_count > MAX_RETRY_COUNT:
            raise Exception("Timeout Error! "
                            "Operation was not completed.")
        time.sleep(wait_time)
        job_result = check_update(storage_api,
                                  r.json()["jobId"], headers)
        status = job_result.json()["status"]
        double_time = wait_time * 2
        if double_time < 120:
            wait_time = double_time
        else:
            wait_time = 120
        retry_count += 1
    if job_result.json()["state"] == "Failed":
        error_obj = job_result.json()["error"]
        if "errorCode" in error_obj:
            if "SSB1" in error_obj["errorCode"]:
                print("Error! SSB code : ",
                      error_obj["errorCode"]["SSB1"],
                      ", ", error_obj["errorCode"]["SSB2"])
            elif "errorCode" in error_obj["errorCode"]:
                print("Error! error code : ",
                      error_obj["errorCode"]["errorCode"])
        raise Exception("Job Error!", job_result.text)
    print("Async job was succeeded. affected resource : " +
          job_result.json()["affectedResources"][0])
    return job_result.json()["affectedResources"][0]

"""
Check whether this API version allows the REST
 Server to execute this program

@param api version api_version of this REST Server
@param required_major_version the lowest number of
the major version that this program requires
@param required_minor_version the lowest number of
the minor version that this program requires

"""


def check_api_version(api_version, required_major_version,
                      required_minor_version):
    version = api_version.split(".")
    major_version = int(version[0])
    minor_version = int(version[1])
    if not ((major_version == required_major_version and
             minor_version >= required_minor_version) or
            major_version >= required_major_version + 1):
        sys.exit("This program requires API Version " +
                 str(required_major_version) + "." +
                 str(required_minor_version) +
                 "." + "x or newer.\n")

try:
    # step1 Check the API version of the local REST API #
    print("Check the API version of the local REST API")
    url = local_storage_api.api_version()
    r = requests.get(url, headers=local_headers,
                     verify=False)
    if r.status_code != http.client.OK:
        raise requests.HTTPError(r)
    check_api_version(r.json()["apiVersion"],
                      REQUIRED_MAJOR_VERSION,
                      REQUIRED_MINOR_VERSION)

    # step1 Check the API version of the remote REST API #
    print("Check the API version of the remote REST API")
    url = remote_storage_api.api_version()
    r = requests.get(url, headers=remote_headers,
                     verify=False)
    if r.status_code != http.client.OK:
        raise requests.HTTPError(r)
    check_api_version(r.json()["apiVersion"],
                      REQUIRED_MAJOR_VERSION,
                      REQUIRED_MINOR_VERSION)

    # step2 Generate a local session #
    print("Generate a local session")
    url = local_storage_api.generate_session()
    r = requests.post(
        url,
        headers=local_headers,
        auth=LOCAL_USER_CREDENTIAL,
        verify=False)
    if r.status_code != http.client.OK:
        raise requests.HTTPError(r)
    local_token = r.json()["token"]
    local_auth = "Session " + local_token
    local_session_id = r.json()["sessionId"]

    # step2 Generate a remote session #
    print("Generate a remote session")
    url = remote_storage_api.generate_session()
    r = requests.post(url, headers=remote_headers,
                      auth=REMOTE_USER_CREDENTIAL,
                      verify=False)
    if r.status_code != http.client.OK:
        raise requests.HTTPError(r)
    remote_token = r.json()["token"]
    remote_auth = "Session " + remote_token
    remote_session_id = r.json()["sessionId"]
    remote_headers["Authorization"] = remote_auth

    # step3 Register a remote storage device #
    print("Register a remote storage device")
    url = local_storage_api.remote_storage()
    body = {
        "storageDeviceId": remote_storage_api.
        get_storage_id(),
        "restServerIp": remote_copy_param.REMOTE_REST_SERVER_IP_ADDR,
        "restServerPort": remote_copy_param.REMOTE_PORT
    }
    local_headers["Authorization"] = local_auth
    local_headers["Remote-Authorization"] = remote_auth
    affected_resource_path = invoke_async_command(
        local_storage_api, "post",
        local_headers, url, body)

    # step4 Print the remote storage device #
    print("Print the remote storage device")
    url = local_storage_api.affected_resource(
        affected_resource_path)
    r = requests.get(url, headers=local_headers,
                     verify=False)
    if r.status_code != http.client.OK:
        raise requests.HTTPError(r)
    print("STORAGE DEVICE ID : " +
          str(r.json()["storageDeviceId"]))
    print("DKC TYPE : " + str(r.json()["dkcType"]))
    print("REST SERVER IP : " + str(r.json()["restServerIp"]))
    print("REST SERVER PORT : " + str(r.json()["restServerPort"]))
    print("MODEL : " + str(r.json()["model"]))
    print("SERIAL NUMBER : " +
          str(r.json()["serialNumber"]))

except requests.ConnectionError:
    sys.stderr.write("Connection Error!\n")
    sys.stderr.write(traceback.format_exc())
except requests.HTTPError as he:
    sys.stderr.write("HTTP Error! status code : ")
    sys.stderr.write(str(he.args[0].status_code) + "\n")
    sys.stderr.write(he.args[0].text + "\n")
except Exception as e:
    sys.stderr.write(traceback.format_exc())
    for msg in e.args:
        sys.stderr.write(str(msg) + "\n")
finally:
    # step5 Discard the local session #
    print("Discard the local session")
    url = local_storage_api.discard_session(
        local_session_id)
    r = requests.delete(url, headers=local_headers,
                        verify=False)
    if r.status_code != http.client.OK:
        raise requests.HTTPError(r)

    # step5 Discard the remote session #
    print("Discard the remote session")
    url = remote_storage_api.discard_session(
        remote_session_id)
    r = requests.delete(url, headers=remote_headers,
                        verify=False)
    if r.status_code != http.client.OK:
        raise requests.HTTPError(r)

    print("Operation was completed.")
    sys.exit()
