# coding:utf-8

"""
register_storage

This program requires API version 1.9.1 or newer.
"""

import traceback
import requests
import json
import sys
import http.client
import storage_param
import rest_server_param

from block_storage_api import BlockStorageAPI

# #################Initialize parameters################# #
# Change the following parameters to fit your environment

# An user id and password of the target storage
USER_CREDENTIAL = ("user1", "pass1")

# Enable cipher mode
IS_SECURE = "true"

###########################################################

# ###You don't have to change the following parameters### #
block_storage_api = BlockStorageAPI(
    rest_server_param.REST_SERVER_IP_ADDR,
    rest_server_param.REST_SERVER_PORT,
    storage_param.STORAGE_MODEL,
    storage_param.SERIAL_NUMBER)

headers = {"content-type": "application/json",
           "accept": "application/json"}

REQUIRED_MAJOR_VERSION = 1
REQUIRED_MINOR_VERSION = 9
REQUIRED_REVISION_VERSION = 1

###########################################################

"""
Check whether this API version allows the REST
 Server to execute this program

@param api_version api version of this REST Server
@param required_major_version the lowest number of
       the major version that this program requires
@param required_minor_version the lowest number of
       the minor version that this program requires

"""


def check_api_version(api_version, required_major_version,
        required_minor_version, required_revision_version):
    version = api_version.split(".")
    major_version = int(version[0])
    minor_version = int(version[1])
    revision_version = int(version[2])
    version_error = False
    if major_version == required_major_version:
        if minor_version == required_minor_version:
            if revision_version < required_revision_version:
                version_error = True
        elif minor_version < required_minor_version: 
            version_error = True
    elif major_version < required_major_version:    
        version_error = True
     
    if version_error == True:
        sys.exit(
            "This program requires API Version " +
            str(required_major_version) +
            "." +
            str(required_minor_version) +
            "." +
            str(required_revision_version) +
            " or newer.\n")
    
try:
    # step1 Check the API version #
    print("Check the API version")
    url = block_storage_api.api_version()
    r = requests.get(url, headers=headers, verify=False)
    if r.status_code != http.client.OK:
        raise requests.HTTPError(r)
    check_api_version(
        r.json()["apiVersion"],
        REQUIRED_MAJOR_VERSION,
        REQUIRED_MINOR_VERSION,
        REQUIRED_REVISION_VERSION)

    # step2 Register a storage
    print("Register a storage")
    url = block_storage_api.local_storages()
    if storage_param.SVP_IP_ADDR is not None:
        body = {
            "svpIp": storage_param.SVP_IP_ADDR,
            "serialNumber": storage_param.SERIAL_NUMBER,
            "model": storage_param.STORAGE_MODEL,
            "isSecure" : IS_SECURE
        }
    else:
        body = {
            "ctl1Ip": storage_param.GUM1_IP_ADDR,
            "ctl2Ip": storage_param.GUM2_IP_ADDR,
            "serialNumber": storage_param.SERIAL_NUMBER,
            "model": storage_param.STORAGE_MODEL
        }        
    r = requests.post(url, headers=headers,
                      data=json.dumps(body),
                      auth=USER_CREDENTIAL, verify=False)
    if r.status_code != http.client.OK:
        raise requests.HTTPError(r)
    
    # step3 Get the storage
    print("Get the storage")
    url = block_storage_api.local_storage(
                        str(r.json()["storageDeviceId"]))
    r = requests.get(url, headers=headers,
                     auth=USER_CREDENTIAL, verify=False)
    if r.status_code != http.client.OK:
        raise requests.HTTPError(r)
    
    if storage_param.SVP_IP_ADDR is not None:
        print("STORAGE DEVICE ID : " + 
              str(r.json()["storageDeviceId"]))
        print("MODEL : " + r.json()["model"])
        print("SERIAL NUMBER : " + str(r.json()["serialNumber"]))
        print("SVP IP ADDRESS : " + r.json()["svpIp"])
        print("RMI PORT : " + str(r.json()["rmiPort"]))
        print("DKC MICRO VERSION : " +
               r.json()["dkcMicroVersion"])
        print()
    else:
        print("STORAGE DEVICE ID : " +
               str(r.json()["storageDeviceId"]))
        print("MODEL : " + r.json()["model"])
        print("SERIAL NUMBER : " + str(r.json()["serialNumber"]))
        print("CTL1 IP ADDRESS : " + r.json()["ctl1Ip"])
        print("CTL2 IP ADDRESS : " + r.json()["ctl2Ip"])
        print("TARGET CONTROLLER : " + str(r.json()["targetCtl"]))
        print("DKC MICRO VERSION : " + r.json()["dkcMicroVersion"])
        print()  
    
    
except requests.ConnectionError:
    sys.stderr.write("Connection Error!\n")
    sys.stderr.write(traceback.format_exc())
except requests.HTTPError as he:
    sys.stderr.write("HTTP Error! status code : ")
    sys.stderr.write(str(he.args[0].status_code) + "\n")
    sys.stderr.write(he.args[0].text + "\n")
except Exception as e:
    sys.stderr.write(traceback.format_exc())
    for msg in e.args:
        sys.stderr.write(str(msg) + "\n")
finally:
    print("Operation was completed.")
    sys.exit()
