#! /usr/bin/perl

#***********************************************************************
#  Copyright 2015 Newisys, Inc. as an unpublished work                  
#  All Rights Reserved                                                  
#                                                                       
#  The information contained herein is confidential property of         
#  Newisys, Inc. The use, copying, transfer, or disclosure of such      
#  information is prohibited except by express written agreement with   
#  Newisys, Inc.                                                        
# ---------------------------------------------------------------------
#  Filename: xeeprom.pl
#  Abstract: VPD EEPROM Programming Tool
#
#  Date      Version  Description
#  --------  -------  -----------
#  08/04/15  00.00    Initial version.
#  09/10/15  00.01    Support 'mixed' format (ASCII + Hex digits).
#                     Add -modify command option.
#  02/24/17  00.02    Fixed checksum computation bug (read/write/print)
#  09/25/17  00.03    Fixed compare errors in read algorithm
#  11/28/17  00.04    Updated modify to just update the area where parameter changed
#                     Added failure and exit of read after 4 attempts at comparision
#  11/30/17  00.05    total pagelength correction in sendWriteAddressAndData
#
#***********************************************************************
use strict;
use warnings;
use Getopt::Long;
use IO::File;
use File::Spec;
use Cwd;


#***********************************************************************
#  globals
#***********************************************************************
my $debug  = 0;
my $dryRun = 0;
my $saveXmlVPDdata   = '';
my $saveReadVPDdata  = '';
my $saveWriteVPDdata = '';

my @read_memory    = ();
my @read_memory_cmp = ();
my @xml_memory     = ();
my $xml_memorysize = 0;
my @write_memory   = ();

my @xml_regionList = ();
my %xml            = ();
my @xml_offsetList = ();
my %xml_offset     = ();


#***********************************************************************
#  printUsage() - 
#***********************************************************************
sub printUsage()
{
    print "
Usage:
  xeeprom.pl -read | -update | -replace | -fix | -modify=<R/F/D>
             -device=<SG_DEVICE> -id=<VPD_EEPROM_ID> [-dryrun]
             <XML_FILENAME>
where:
  -read      Reads and displays the EEPROM's VPD data.
  -update    Updates the EEPROM's non-volatile VPD fields with data
             from the XML file.
  -replace   Replaces the EEPROM's VPD data from the XML file.
  -fix       Fixes the EEPROM's VPD checksums.
  -modify    Modifies a field in EEPROM VPD as specified by text
             identifying (Region/Field/Data).  E.g.
                -modify=ChassisInfo/SerialNumber/USSCI00002PRB003
                -modify=ProductInfo/Name/\"text with spaces\"
                -modify=IPConfig/MacAddress/00093D00046F
  -dryrun    Does all the same things, EXCEPT that it doesn't
             actually write the VPD, plus it displays what it
             would have written, in the same format as the output
             from the -read option

";
}

#***********************************************************************
#  print_hidden_options() - 
#***********************************************************************
sub print_hidden_options()
{
    print "
  -debug=hex string ( example: -debug=cfc

      0x0001 - enable debug output while parsing XML details
      0x0002 - print summary of XML data
      0x0004 - dump raw data read from VPD
      0x0008 - print formatted data read from VPD
      0x0010 - dump raw data from the XML file
      0x0020 - print formatted data to be written to VPD
      0x0100 - print all sent sg_utils commands
      0x0200 - print all sg_util command responses
      0x0400 - enable debug output while trying to read existing VPD
      0x1000 - for 'update' print the names of the fields KEPT
      0x2000 - for 'update' print the names of the fields REPLACED

  -savexml=filename
  -saveread=filename
  -savewrite=filename

";
}

#***********************************************************************
#  usageExit() -
#***********************************************************************
sub usageExit()
{
    printUsage();
    exit(1);
}

#***********************************************************************
#  appendAscii() -
#***********************************************************************
sub appendAscii($)
{
   my $inStr = "";
   my $outStr = "";
   my $i;
   my $c;

   $inStr = shift(@_);

   for($i = 0; $i < length ($inStr); $i++ )
   {
      $c = ord(substr($inStr, $i, 1));
      $outStr .= sprintf "%02X,", $c;
   }

   chop( $outStr );
   return( $outStr );
}

#***********************************************************************
#  () - 
#***********************************************************************
sub sendSGCommand($)
{
   my $cmd = shift(@_);
   my $result = "";
   my $sgdir = "";

   # if NDSSG_DIR set use it
   $sgdir = $ENV{'NDSSG_DIR'};
   if ( defined $sgdir )
   {
      if ( $sgdir ne "" )
      {
         $cmd = File::Spec->catfile( $sgdir, $cmd );
      }
   }

   print "sendSGCommand: sending cmd '$cmd'\n" if ( $debug & 0x100 );
   $result = `$cmd`;
   print "sendSGCommand: result '$result'\n" if ( $debug & 0x200 );

   return( $result );
}

#***********************************************************************
#* issueTestUnitReady() -
#***********************************************************************
sub issueTestUnitReady($)
{
   my $device = shift(@_);
   my $cmd    = "sg_turs $device";
   my $result;

   $result = sendSGCommand( $cmd );

   print "TUR Returned '$result'\n" if ( $debug & 0x200 );

   return ( $result );
}

#***********************************************************************
#* sendWriteAddressAndData() -
#***********************************************************************
sub sendWriteAddressAndData($$$$)
{
    my $device = shift(@_);
    my $id     = shift(@_);
    my $addr   = shift(@_);
	my $length = shift(@_);

	my $total_length = $length + 5;
    my $cmd    = sprintf("sg_senddiag $device -p --raw=90,00,%02X,%02X,%02X,%02X,%02X,%02X,%02X",($total_length>>8),($total_length&0xFF),$id,($addr>>8),($addr&0xFF),($length>>8),($length&0xFF) );
    my $result = "";

    my $ii;
    for ($ii=0;$ii<$length;$ii++)
    {
        $cmd .= sprintf(",%02X",$write_memory[ $addr + $ii ]);
    }

    $result = sendSGCommand($cmd);
    if ('' ne $result)
    {
        printf "Unexpected response from SES while trying to write VPD address %d./0x%0X\n"
             ."sendWriteAddressAndData received: <<$result>>\n", $addr,$addr;
        exit(0);
    }
    print "sendWriteAddressAndData received: '$result'\n" if ( $debug & 0x200 );

    return ( $result );
}

#***********************************************************************
#* sendReadAddress() -
#***********************************************************************
sub sendReadAddress($$$)
{
    my $device = shift(@_);
    my $id     = shift(@_);
    my $addr   = shift(@_);
    my $cmd    = sprintf("sg_senddiag $device -p --raw=90,00,00,05,%02X,%02X,%02X,0,0",$id,($addr>>8),($addr&0xFF) );
    my $result = "";

    $result = sendSGCommand($cmd);
    if ('' ne $result)
    {
        print "unexpected response from SES while trying to set the VPD address to read\n"
             ."sendReadAddress received: <<$result>>\n";
        exit(0);
    }
}

#***********************************************************************
#* sendReadData() -
#
#   This routine has gotten to 'twisted' in the parsing (which has some GOOD features)...
#   It really should just read ALL the hex data into a temporary array, and then
#   copy out what the data portion, rather than special casing everything along
#   the way to compensate for 5 unneeded bytes...   but it is working now... jfk
#
#0         1         2         3         4         5         6
#0123456789012345678901234567890123456789012345678901234567890123456789
# 00     90 00 01 01 02 01 12 01  07 18 00 23 aa 01 30 17    ...........#..0.
# ...
# f0     20 20 20 c4 30 30 30 31  d0 55 4e 44 45 46 49 4e       .0001.UNDEFIN
# 100    45 44 20 20 20                                      ED
#***********************************************************************
sub sendReadData($$)
{
    my $device = shift(@_);
    my $cmd    = "sg_ses $device -p 0x90";
    my $addr   = shift(@_);
	my $max_retry = 4;

  for (my $retry = 0; $retry < $max_retry; $retry++)
  {
    my $result = "";
    my $result_cmp = "";
   

    $result = sendSGCommand($cmd);
    $result_cmp = sendSGCommand($cmd);
    print "sendReadData address $addr, received:\n$result\n" if ($debug & 0x400);

    my @lines = split('\n',$result);
    my @lines_cmp = split('\n',$result_cmp);
    print "number of lines returned ".scalar(@lines)."\n" if ($debug & 0x400);
    my $addrcheck = 0;
    my $expectedAddr = 0;
    my $addrcheck_cmp = 0;
    my $expectedAddr_cmp = 0;
    foreach my $line (@lines)
    {
        next if (length($line) < 61);
        my $data = substr($line,0,56);
        next unless ( $data =~ /^[\s0-9a-fA-F]+$/ );
        print "...$data\n" if ($debug & 0x400);
        my @tokens = split(' +',$data);
        shift @tokens; #discard the empty token
        print "...".scalar(@tokens)."\n" if ($debug & 0x400);
        my $lineaddr = hex(shift @tokens);
        if ($expectedAddr != 0)
        {
            $lineaddr -= 5;
        }
        if ($lineaddr != $expectedAddr)
        {
            print "Confused, lineaddr $lineaddr not expected $expectedAddr\nline: $line\ndata: $data\n";
        }
        $expectedAddr += 16;
        if (0 == $lineaddr) # discard the first 5 bytes of the packet (the header)
        {
            shift @tokens;
            shift @tokens;
            shift @tokens;
            shift @tokens;
            shift @tokens;
            $expectedAddr -= 5;
        }
        foreach my $token (@tokens)
        {
            $read_memory[ $addr + $lineaddr++ ] = hex($token);
            $addrcheck++;
        }
    }
    foreach my $line_cmp (@lines_cmp)
    {
        next if (length($line_cmp) < 61);
        my $data_cmp = substr($line_cmp,0,56);
        next unless ( $data_cmp =~ /^[\s0-9a-fA-F]+$/ );
        print "...$data_cmp\n" if ($debug & 0x400);
        my @tokens_cmp = split(' +',$data_cmp);
        shift @tokens_cmp; #discard the empty token
        print "...".scalar(@tokens_cmp)."\n" if ($debug & 0x400);
        my $lineaddr_cmp = hex(shift @tokens_cmp);
        if ($expectedAddr_cmp != 0)
        {
            $lineaddr_cmp -= 5;
        }
        if ($lineaddr_cmp != $expectedAddr_cmp)
        {
            print "Confused, lineaddr $lineaddr_cmp not expected $expectedAddr_cmp\nline: $line_cmp\ndata: $data_cmp\n";
        }
        $expectedAddr_cmp += 16;
        if (0 == $lineaddr_cmp) # discard the first 5 bytes of the packet (the header)
        {
            shift @tokens_cmp;
            shift @tokens_cmp;
            shift @tokens_cmp;
            shift @tokens_cmp;
            shift @tokens_cmp;
            $expectedAddr_cmp -= 5;
        }
        foreach my $token_cmp (@tokens_cmp)
        {
            $read_memory_cmp[ $addr + $lineaddr_cmp++ ] = hex($token_cmp);
            $addrcheck_cmp++;
        }
    }
    if ($addrcheck != 256)
    {
        print "Suspicious data length $addrcheck, expecting 256\n@lines\n";
    }
    if ($addrcheck_cmp != 256)
    {
        print "Suspicious data length $addrcheck_cmp, expecting 256\n@lines_cmp\n";
    }
    my $b_cmp = 0;

    if ($addrcheck == $addrcheck_cmp && $addrcheck != 0)
    {
        for ( my $index = $addr; $index < $addr + $addrcheck; $index++ )
    	{
            if ( $read_memory_cmp[ $index ] != $read_memory[ $index ] )
            {
                $b_cmp = 1;
                $index = $addr + $addrcheck;
            }          
        }
     }
     else
     {
        $b_cmp = 2;
        print "Size incorrect on retry $retry\n";
     }

     if ($b_cmp == 0)
     {
         $retry = $max_retry;
     }
     if ($b_cmp == 1)
     {
         print "Compare failed on retry $retry\n";
     }

	 if ($retry == $max_retry-1) 
	 {
		if ($b_cmp == 2 || $b_cmp == 1) 
		{
		   printf "\nREAD failed at address $addr\n";
		   exit(0);
		}
	 }
   }  
}

#***********************************************************************
#* saveStructureToFile() -
#***********************************************************************
sub saveStructureToFile($$)
{
    my $filename    = shift(@_);
    my $memArrayRef = shift(@_);

    $filename .= '.struct';

    unless ( open(XXX,'>',$filename) )
    {
        print "Failed to open output file '$filename' for write, ignoring\n";
        return;
    }


    my $goodChecksumCount = 0;
    my $badChecksumCount  = 0;

    my $checksum = 0;
    my $oldchecksum = 0;
    foreach my $offset (@xml_offsetList)
    {
        printf XXX "%3d./%04X %-28.28s %-28.28s %-5.5s %2d.  = \""
            ,$offset,$offset
            ,$xml_offset{$offset}{region}
            ,$xml_offset{$offset}{name}
            ,$xml_offset{$offset}{Format}
            ,$xml_offset{$offset}{Length}
            ;
        my $asciiErrors = 0;
        my $last = $offset + $xml_offset{$offset}{Length};
        for (my $ii=$offset; $ii<$last; $ii++)
        {
            if (0 != $oldchecksum)
            {
                $checksum = 0;
                $oldchecksum = 0;
            }
            if ('checksum' eq $xml_offset{$offset}{name})
            {
                $oldchecksum = $checksum & 0xFF;
            }
            $checksum += ${$memArrayRef}[$ii];

            if ( 'hex' eq $xml_offset{$offset}{Format} )
            {
                printf XXX "%02X",${$memArrayRef}[$ii];
                next;
            }

            if ('ascii' eq $xml_offset{$offset}{Format})
            {
                my $char = chr(${$memArrayRef}[$ii]);
                if ( (${$memArrayRef}[$ii] < 0x20)||(${$memArrayRef}[$ii] > 0x7E) )
                {
                    $asciiErrors++;
                    $char = '?';
                }
                print XXX $char;
                next;
            }

            # mixed ascii/hex format for PSU ASCII fields
            if ('mixed' eq $xml_offset{$offset}{Format})
            {
                my $char = chr(${$memArrayRef}[$ii]);
                if ( (${$memArrayRef}[$ii] >= 0) && (${$memArrayRef}[$ii] < 0x10) )
                {
                    $char = chr(${$memArrayRef}[$ii] + 0x30);
                }
                elsif ( (${$memArrayRef}[$ii] < 0x20)||(${$memArrayRef}[$ii] > 0x7E) )
                {
                    $asciiErrors++;
                    $char = '?';
                }
                print XXX $char;
                next;
            }
        }

        print XXX "\"";
        if ('checksum' eq $xml_offset{$offset}{name})
        {
            $oldchecksum = (0 - $oldchecksum) & 0xFF;
            printf XXX "  [%02X]",$oldchecksum;
            if ($oldchecksum == ${$memArrayRef}[$offset])
            {
                print XXX " Good!";
                $goodChecksumCount++;
            } else
            {
                print XXX " BAD ********************";
                $badChecksumCount++;
            }

            $checksum = 0;        # Added! 2/24/2017
            $oldchecksum = 0;     # Added! 2/24/2017
        }
        print XXX "\n";

        if (0 != $asciiErrors)
        {
            print XXX "*** the above ASCII string contains $asciiErrors invalid characters\n"
                 ."*** which are displayed above as '?'\n";
        }
    }
    print XXX "\n\nGood checksums $goodChecksumCount, bad checksums $badChecksumCount\n\n";

    unless ( close(XXX) )
    {
        print "Failed to close output file '$filename' for write, ignoring\n";
    } else
    {
        print "successfully saved to file'$filename'\n";
    }
}

#***********************************************************************
#* saveDataToFile() -
#***********************************************************************
sub saveDataToFile($$)
{
    my $filename    = shift(@_);
    my $memArrayRef = shift(@_);

    unless ( open(XXX,'>',$filename) )
    {
        print "Failed to open output file '$filename' for write, ignoring\n";
        return;
    }

    for (my $ii=0;$ii<$xml_memorysize;$ii+=16)
    {
        printf XXX "%04X: %02X %02X %02X %02X %02X %02X %02X %02X  %02X %02X %02X %02X %02X %02X %02X %02X   ",$ii
        ,${$memArrayRef}[$ii   ],${$memArrayRef}[$ii+ 1],${$memArrayRef}[$ii+ 2],${$memArrayRef}[$ii+ 3]
        ,${$memArrayRef}[$ii+ 4],${$memArrayRef}[$ii+ 5],${$memArrayRef}[$ii+ 6],${$memArrayRef}[$ii+ 7]
        ,${$memArrayRef}[$ii+ 8],${$memArrayRef}[$ii+ 9],${$memArrayRef}[$ii+10],${$memArrayRef}[$ii+11]
        ,${$memArrayRef}[$ii+12],${$memArrayRef}[$ii+13],${$memArrayRef}[$ii+14],${$memArrayRef}[$ii+15];
        for (my $jj=$ii;$jj<($ii+16);$jj++)
        {
            my $val = ${$memArrayRef}[$jj];
            my $char = chr($val);
               $char = '.' if (($val < 0x20)||($val > 0x7E));
            print XXX $char;
        }
        print XXX "\n";
    }

    unless ( close(XXX) )
    {
        print "Failed to close output file '$filename' for write, ignoring\n";
    } else
    {
        print "successfully saved to file'$filename'\n";
    }
}

#***********************************************************************
#* defineXMLoffset() -
#***********************************************************************
sub defineXMLoffset($$$$$$$$)
{
    my $xml_memoryaddr = shift(@_);
    my $xml_regionname = shift(@_);
    my $xml_fieldname  = shift(@_);
    my $xml_length     = shift(@_);
    my $xml_format     = shift(@_);
    my $xml_retain     = shift(@_);
    my $xml_volatile   = shift(@_);
    my $xml_fill       = shift(@_);

    push(@xml_offsetList,$xml_memoryaddr);
    $xml_offset{$xml_memoryaddr}{name}     = $xml_fieldname;
    $xml_offset{$xml_memoryaddr}{region}   = $xml_regionname;
    $xml_offset{$xml_memoryaddr}{Length}   = $xml_length;
    $xml_offset{$xml_memoryaddr}{Format}   = $xml_format;
    $xml_offset{$xml_memoryaddr}{retain}   = $xml_retain;
    $xml_offset{$xml_memoryaddr}{volatile} = $xml_volatile;
    $xml_offset{$xml_memoryaddr}{Fill}     = $xml_fill;
}

#***********************************************************************
#* ConfusedExit() -
#***********************************************************************
sub ConfusedExit($$$)
{
    my $text    = shift(@_);
    my $linenum = shift(@_);
    my $line    = shift(@_);
    print "Confused by $text line #$linenum, line '$line'\n";
    exit(0);
}

#***********************************************************************
#* readXML() -
#***********************************************************************
sub readXML($)
{
    my $filename = shift(@_);
    print "Reading XML file '$filename'\n" ;#if ($debug & 0x400);

    my $xml_eepromname = '';
    my $xml_eepromfill = 0;
    my $xml_groupname  = '';
    my $xml_regionname = '';
    my $xml_regionfill;
    my $xml_fieldname ;
    my $xml_id        ;
    my $xml_retain    ;
    my $xml_volatile  ;
    my $xml_length    ;
    my $xml_format    ;
    my $xml_fill      ;
    my $xml_region    ;
    my $xml_type      ;
    my $xml_checksum  ;
    my $xml_data      ;
    my $xml_inCommentBlock = 0;
    my $xml_memoryaddr     = 0;

    my $line; my $linenum;
    open(FILE, "$filename");
    $linenum=-1;
    while ($line = <FILE>)
    {
        chomp $line;
        $line =~ s/^\s+//;
        $line =~ s/\s+$//;
        $linenum++;
        next if (('' eq $line)||(0 == $linenum));
        if (0 != $xml_inCommentBlock)
        {
            if ( $line eq '-->')
            {
                $xml_inCommentBlock = 0;
            }
            next;
        }
        if ( ($line =~ '^<!--') && ($line =~ '-->') )
        {
            next;
        }
        if ( $line eq '<!--')
        {
            $xml_inCommentBlock = 1;
            next;
        }
        if ( $debug & 1 )
        {
            print "\nworking on line $linenum <<$line>>"
                  ."\nEEPROM '$xml_eepromname'"
                  ."\nRegion '$xml_regionname'"
                  ."\nOffset $xml_memoryaddr\n"
                  ;
        }

        $xml_fieldname = '';
        $xml_id        = '';
        $xml_retain    = '';
        $xml_volatile  = '';
        $xml_length    = '';
        $xml_format    = '';
        $xml_fill      = '';
        $xml_region    = '';
        $xml_type      = '';
        $xml_checksum  = '';
        $xml_data      = '';
        #
        #   line formats
        #   <yyy a="ww" b="uu" ...>
        #   <xxx a="ww" b="uu" ...>data</xxx>
        #   </yyy>
        #
        my @fields = split('>',$line);
        my $num1 = @fields;
        my @control = split(' ',$fields[0]);
        foreach my $pair (@control)
        {
            print "\tpair <<$pair>>\n" if ( $debug & 1 );
            my @halves = split('=',$pair);
            if (1<@halves)  # remove quotes from RHSide
            {
                $halves[1] =~ s/"//g;
            }
            if ('<' eq substr($halves[0],0,1))
            {
                 $xml_fieldname = substr($halves[0],1);
            }
            elsif ('id' eq $halves[0])
            {
                $xml_id        = $halves[1];
            }
            elsif ('retain' eq $halves[0])
            {
                $xml_retain    = $halves[1];
            }
            elsif ('volatile' eq $halves[0])
            {
                $xml_volatile  = $halves[1];
            }
            elsif ('length' eq $halves[0])
            {
                $xml_length    = $halves[1];
            }
            elsif ('format' eq $halves[0])
            {
                $xml_format    = $halves[1];
                $xml_format    = 'hex' if ( '' eq $xml_format);
            }
            elsif ('fill' eq $halves[0])
            {
                $xml_fill      = $halves[1];
            }
            elsif ('region' eq $halves[0])
            {
                $xml_region    = $halves[1];
            }
            elsif ('type' eq $halves[0])
            {
                $xml_type      = $halves[1];
            }
            elsif ('checksum' eq $halves[0])
            {
                $xml_checksum  = $halves[1];
                if ('last_byte' ne $xml_checksum)
                {
                    ConfusedExit("pair '$pair'",$linenum,$line);
                }
            }
            elsif ('offset' eq $halves[0] )
            {
                if ('0' ne $halves[1])
                {
                    ConfusedExit("this",$linenum,$line);
                }
                next;
            }
            else
            {
                ConfusedExit("pair '$pair'",$linenum,$line);
            }
        }
        if (2 == $num1)
        {
            my @data = split('<',$fields[1]);
            $xml_data = $data[0];
        }
        ### process the parsed data
        if (1 == $num1 )
        {
            if ( $xml_fieldname eq "/$xml_eepromname" )
            {
                # end of main block reached - do we NEED to parse more?
                $xml_eepromname = "<<ClOsEd>>";
                next;
            }
            if ($xml_fieldname eq "/$xml_regionname")
            {
                if ( 'last_byte' eq $xml{$xml_regionname}{checksum})
                {
                    defineXMLoffset($xml_memoryaddr,$xml_regionname,'checksum',1,'hex','','', 0);
                    $xml_memoryaddr++;
                }
                $xml{$xml_regionname}{lastAddr} = $xml_memoryaddr - 1;
                $xml_regionname = '';
                next;
            }
            if ($xml_fieldname eq "/$xml_groupname")
            {
                $xml_groupname = '';
                next;
            }
            if ('region' eq $xml_type)
            {
                if ('' ne $xml_regionname)
                {
                    print "Found region '$xml_fieldname' before the end of region '$xml_regionname\n";
                    exit(0);
                }
                $xml_regionname = $xml_fieldname;
                $xml_regionfill = $xml_fill;
                push(@xml_regionList,$xml_regionname);
                $xml{$xml_regionname}{checksum}  = $xml_checksum;
                $xml{$xml_regionname}{firstAddr} = $xml_memoryaddr;
                next;
            }
            if ('group' eq $xml_type)
            {
                if ('' ne $xml_groupname)
                {
                    print "Found group '$xml_fieldname' before the end of group '$xml_groupname\n";
                    exit(0);
                }
                $xml_groupname = $xml_fieldname;
                next;
            }
            if ('eeprom' eq $xml_type)
            {
                if ('' eq $xml_eepromname) # start of real data
                {
                    $xml_eepromname = $xml_fieldname;
                    $xml_eepromfill = $xml_fill if '' ne $xml_fill;
                    printf "eeprom size is %d./0x%X\n",$xml_length,$xml_length if ( $debug & 1 );
                    $xml_memorysize = $xml_length;
                    @xml_memory = (hex($xml_eepromfill)) x $xml_memorysize;
                    $xml_memoryaddr = 0;
                    next;
                }
            }
            ConfusedExit("this",$linenum,$line);
        }
        if ('' ne $xml_type)
        {
            print "Unknown type '$xml_type' in line '$line'\n";
            exit(0);
        }
        my $fill = $xml_fill;
        $fill = $xml_regionfill if $fill eq '';
        $fill = $xml_eepromfill if $fill eq '';
        my $fillval = hex($fill);
        if ( ('ascii' eq $xml_format) || ('mixed' eq $xml_format) )
        {
            for (my $ii=0;$ii<$xml_length;$ii++) # fill the full area
            {
                $xml_memory[$xml_memoryaddr+$ii] = $fillval;
            }
            for (my $jj=0;$jj<length($xml_data);$jj++) # copy the ASCII data, left justified
            {
                $xml_memory[$xml_memoryaddr+$jj] = ord(substr($xml_data,$jj,1));
            }
            defineXMLoffset($xml_memoryaddr,$xml_regionname,$xml_fieldname,$xml_length,$xml_format,$xml_retain,$xml_volatile,$fillval);
            $xml_memoryaddr += $xml_length;
            next;
        }

        # hex data - fully specified, i.e. no fill necessary
        if (('hex' eq $xml_format) && ((2 * $xml_length) == length($xml_data)))
        {
            defineXMLoffset($xml_memoryaddr,$xml_regionname,$xml_fieldname,$xml_length,$xml_format,$xml_retain,$xml_volatile,$fillval);
            for (my $jj=0;$jj<$xml_length;$jj++)
            {
                my $val = hex(substr($xml_data,$jj<<1,2));
                $xml_memory[$xml_memoryaddr++] = $val;
            }
            next;
        }
        # hex data - fill necessary
        if ('hex' eq $xml_format)
        {
            if (0 != (length($xml_data) % 2))
            {
                print "hex data length is NOT a multiple of 2, line #$linenum, line '$line'\n";
                exit(0);
            }
            defineXMLoffset($xml_memoryaddr,$xml_regionname,$xml_fieldname,$xml_length,$xml_format,$xml_retain,$xml_volatile,$fillval);
            # first fill memory with the fill value
            for (my $jj=0;$jj<$xml_length;$jj++)
            {
                $xml_memory[$xml_memoryaddr+$jj] = $fillval;
            }
            my $addr = $xml_memoryaddr - (length($xml_data)>>1); # position to RIGHT JUSTIFY the hex value
            for (my $jj=0;$jj<length($xml_data);$jj+=2)
            {
                my $val = hex(substr($xml_data,$jj,2));
                $xml_memory[$xml_memoryaddr+$jj] = $val;
            }
            $xml_memoryaddr += $xml_length;
            next;
        }
        print "Unhandled XML data, line #$linenum '$line'\n";
    }
    close(FILE);

    if ( $debug & 2 )
    {
        print "\n Offset XML map of the data\n";
        foreach my $offset (@xml_offsetList)
        {
            printf "%5d./%04X %-28.28s %-28.28s %-5.5s %2d.\n"
                ,$offset,$offset
                ,$xml_offset{$offset}{region}
                ,$xml_offset{$offset}{name}
                ,$xml_offset{$offset}{Format}
                ,$xml_offset{$offset}{Length}
                ;
        }
        print "\n";
    }

    if ( $debug & 0x10 )
    {

        printf "memory size %d./0x%X\n",$xml_memorysize,$xml_memorysize;
        for (my $ii=0;$ii<$xml_memorysize;$ii+=16)
        {
            printf "%03X: %02X %02X %02X %02X %02X %02X %02X %02X  %02X %02X %02X %02X %02X %02X %02X %02X\n",$ii
            ,$xml_memory[$ii   ],$xml_memory[$ii+ 1],$xml_memory[$ii+ 2],$xml_memory[$ii+ 3]
            ,$xml_memory[$ii+ 4],$xml_memory[$ii+ 5],$xml_memory[$ii+ 6],$xml_memory[$ii+ 7]
            ,$xml_memory[$ii+ 8],$xml_memory[$ii+ 9],$xml_memory[$ii+10],$xml_memory[$ii+11]
            ,$xml_memory[$ii+12],$xml_memory[$ii+13],$xml_memory[$ii+14],$xml_memory[$ii+15];
        }
    }

    if ('' ne $saveXmlVPDdata)
    {
        saveDataToFile(      $saveXmlVPDdata,\@xml_memory);
        saveStructureToFile( $saveXmlVPDdata,\@xml_memory);
    }
}

#***********************************************************************
#* read_vpd_data() -
#***********************************************************************
sub read_vpd_data($$)
{
    my $device   = shift(@_);
    my $eepromId = shift(@_);

    issueTestUnitReady( $device );
    issueTestUnitReady( $device );

    @read_memory = (0xFF) x $xml_memorysize;

    print "Reading existing VPD EEROM ID #$eepromId, size $xml_memorysize\n" ;#if ($debug & 0x400);
    for ( my $offset = 0; $offset < $xml_memorysize; $offset += 256 )
    {
        print "offset $offset of memorysize $xml_memorysize\n" if ($debug & 0x400);
        sendReadAddress( $device, $eepromId, $offset );
        sendReadData(    $device, $offset );
    }

    if ($debug & 4)
    {
        printf "\nCurrent VPD contents(raw), size %d./0x%X:\n",$xml_memorysize,$xml_memorysize;
        for (my $ii=0;$ii<$xml_memorysize;$ii+=16)
        {
            printf "%04X: %02X %02X %02X %02X %02X %02X %02X %02X  %02X %02X %02X %02X %02X %02X %02X %02X   ",$ii
            ,$read_memory[$ii   ],$read_memory[$ii+ 1],$read_memory[$ii+ 2],$read_memory[$ii+ 3]
            ,$read_memory[$ii+ 4],$read_memory[$ii+ 5],$read_memory[$ii+ 6],$read_memory[$ii+ 7]
            ,$read_memory[$ii+ 8],$read_memory[$ii+ 9],$read_memory[$ii+10],$read_memory[$ii+11]
            ,$read_memory[$ii+12],$read_memory[$ii+13],$read_memory[$ii+14],$read_memory[$ii+15];
            for (my $jj=$ii;$jj<($ii+16);$jj++)
            {
                my $val = $read_memory[$jj];
                my $char = chr($val);
                   $char = '.' if (($val < 0x20)||($val > 0x7E));
                print $char;
            }
            print "\n";
        }
    }

    if ( $debug & 8 )
    {
        print "\n  Offset Data read from VPD\n";

        my $goodChecksumCount = 0;
        my $badChecksumCount  = 0;

        my $checksum = 0;
        my $oldchecksum = 0;
        foreach my $offset (@xml_offsetList)
        {
            printf "%3d./%04X %-28.28s %-28.28s %-5.5s %2d.  = \""
                ,$offset,$offset
                ,$xml_offset{$offset}{region}
                ,$xml_offset{$offset}{name}
                ,$xml_offset{$offset}{Format}
                ,$xml_offset{$offset}{Length}
                ;
            my $asciiErrors = 0;
            my $last = $offset + $xml_offset{$offset}{Length};
            for (my $ii=$offset; $ii<$last; $ii++)
            {
                if (0 != $oldchecksum)
                {
                    $checksum = 0;
                    $oldchecksum = 0;
                }
                if ('checksum' eq $xml_offset{$offset}{name})
                {
                    $oldchecksum = $checksum & 0xFF;
                }
                $checksum += $read_memory[$ii];

                if ( 'hex' eq $xml_offset{$offset}{Format} )
                {
                    printf "%02X",$read_memory[$ii];
                    next;
                }

                if ('ascii' eq $xml_offset{$offset}{Format})
                {
                    my $char = chr($read_memory[$ii]);
                    if ( ($read_memory[$ii] < 0x20)||($read_memory[$ii] > 0x7E) )
                    {
                        $asciiErrors++;
                        $char = '?';
                    }
                    print $char;
                    next;
                }

                # mixed ascii/hex format for PSU ASCII fields
                if ('mixed' eq $xml_offset{$offset}{Format})
                {
                    my $char = chr($read_memory[$ii]);
                    if ( ($read_memory[$ii] >= 0) && ($read_memory[$ii] < 0x10) )
                    {
                        $char = chr($read_memory[$ii] + 0x30);
                    }
                    elsif ( ($read_memory[$ii] < 0x20)||($read_memory[$ii] > 0x7E) )
                    {
                        $asciiErrors++;
                        $char = '?';
                    }
                    print $char;
                    next;
                }

                print "Unknown format '$xml_offset{$offset}{Format}'";
                exit(0);
            }
            print "\"";
            if ('checksum' eq $xml_offset{$offset}{name})
            {
                $oldchecksum = (0 - $oldchecksum) & 0xFF;
                printf "  [%02X]",$oldchecksum;
                if ($oldchecksum == $read_memory[$offset])
                {
                    print " Good!";
                    $goodChecksumCount++;
                }
                else
                {
                    print " BAD ********************";
                    $badChecksumCount++;
                }

                $checksum = 0;        # Added! 2/24/2017
                $oldchecksum = 0;     # Added! 2/24/2017
            }
            print "\n";

            if (0 != $asciiErrors)
            {
                print "*** the above ASCII string contains $asciiErrors invalid characters\n"
                     ."*** which are displayed above as '?'\n";
            }
        }
        print "\n\nGood checksums $goodChecksumCount, bad checksums $badChecksumCount\n\n";
    }

    if ('' ne $saveReadVPDdata)
    {
        saveDataToFile(      $saveReadVPDdata,\@read_memory);
        saveStructureToFile( $saveReadVPDdata,\@read_memory);
    }
}

#***********************************************************************
#* print_memory_structures() - 
#***********************************************************************
sub print_memory_structures($)
{
    my $memArrayRef = shift(@_);

    my $goodChecksumCount = 0;
    my $badChecksumCount  = 0;

    my $checksum = 0;
    my $oldchecksum = 0;
    foreach my $offset (@xml_offsetList)
    {
        printf "%3d./%04X %-28.28s %-28.28s %-5.5s %2d.  = \""
            ,$offset,$offset
            ,$xml_offset{$offset}{region}
            ,$xml_offset{$offset}{name}
            ,$xml_offset{$offset}{Format}
            ,$xml_offset{$offset}{Length}
            ;
        my $asciiErrors = 0;
        my $last = $offset + $xml_offset{$offset}{Length};
        for (my $ii=$offset; $ii<$last; $ii++)
        {
            if (0 != $oldchecksum)
            {
                $checksum = 0;
                $oldchecksum = 0;
            }
            if ('checksum' eq $xml_offset{$offset}{name})
            {
                $oldchecksum = $checksum & 0xFF;
            }
            $checksum += ${$memArrayRef}[$ii];

            if ( 'hex' eq $xml_offset{$offset}{Format} )
            {
                printf "%02X",${$memArrayRef}[$ii];
                next;
            }

            if ('ascii' eq $xml_offset{$offset}{Format})
            {
                my $char = chr(${$memArrayRef}[$ii]);
                if ( (${$memArrayRef}[$ii] < 0x20)||(${$memArrayRef}[$ii] > 0x7E) )
                {
                    $asciiErrors++;
                    $char = '?';
                }
                print $char;
                next;
            }

            # mixed ascii/hex format for PSU ASCII fields
            if ('mixed' eq $xml_offset{$offset}{Format})
            {
                my $char = chr(${$memArrayRef}[$ii]);
                if ( (${$memArrayRef}[$ii] >= 0) && (${$memArrayRef}[$ii] < 0x10) )
                {
                    $char = chr(${$memArrayRef}[$ii] + 0x30);
                }
                elsif ( (${$memArrayRef}[$ii] < 0x20)||(${$memArrayRef}[$ii] > 0x7E) )
                {
                    $asciiErrors++;
                    $char = '?';
                }
                print XXX $char;
                next;
            }


            print "Unknown format '$xml_offset{$offset}{Format}'";
            exit(0);
        }
        print "\"";
        if ('checksum' eq $xml_offset{$offset}{name})
        {
            $oldchecksum = (0 - $oldchecksum) & 0xFF;
            printf "  [%02X]",$oldchecksum;
            if ($oldchecksum == ${$memArrayRef}[$offset])
            {
                print " Good!";
                $goodChecksumCount++;
            } else
            {
                print " BAD ********************";
                $badChecksumCount++;
            }

            $checksum = 0;        # Added! 2/24/2017
            $oldchecksum = 0;     # Added! 2/24/2017
        }
        print "\n";

        if (0 != $asciiErrors)
        {
            print "*** the above ASCII string contains $asciiErrors invalid characters\n"
                 ."*** which are displayed above as '?'\n";
        }
    }
    print "\n\nGood checksums $goodChecksumCount, bad checksums $badChecksumCount\n\n";

}

#***********************************************************************
#* performOperation() - 
#***********************************************************************
sub performOperation($$$$$$$)
{
    my $filename      = shift(@_);
    my $device        = shift(@_);
    my $eepromId      = shift(@_);
    my $operation     = shift(@_);
    my $modifyRegion  = shift(@_);
    my $modifyField   = shift(@_);
    my $modifyData    = shift(@_);

    # read in the XML file
    readXML($filename);

    # read EEPROM data from enclosure
    read_vpd_data($device,$eepromId);
    return if ( 'READ' eq $operation );

    # create the data to be output - the @write_memory
    if ('FIX' eq $operation)
    {
        @write_memory = @read_memory;
    }
    elsif ('REPLACE' eq $operation)
    {
        @write_memory = @xml_memory;

        # pad write_memory to the full size
        my $eepromSize = scalar(@read_memory);
        while (scalar(@write_memory) < $eepromSize)
        {
            push(@write_memory,0);
        }
    }
    elsif ('UPDATE' eq $operation)
    {
        @write_memory = @read_memory;

        foreach my $offset (@xml_offsetList)
        {
            if ( ($xml_offset{$offset}{volatile} ne '') || ($xml_offset{$offset}{retain} ne '') )
            {
                print "keeping   $xml_offset{$offset}{region} $xml_offset{$offset}{name}\n" if ($debug & 0x1000);
                next;
            }
            print "REPLACING $xml_offset{$offset}{region} $xml_offset{$offset}{name}\n" if ($debug & 0x2000);

            my $last = $offset + $xml_offset{$offset}{Length};
            for (my $ii=$offset; $ii<$last; $ii++)
            {
                $write_memory[$ii] = $xml_memory[$ii];
            }
        }
    }
    elsif ('MODIFY' eq $operation)
    {
        @write_memory = @read_memory;
        my $offset = -1;

        # find the field to modify
        foreach my $search_offset (@xml_offsetList)
        {
            if ( ($xml_offset{$search_offset}{region} eq $modifyRegion) && ($xml_offset{$search_offset}{name} eq $modifyField) )
            {
                $offset = $search_offset;
                last;
            }
        }
        if ( $offset == -1 )
        {
            print "NOT FOUND: VPD Data Field $modifyRegion $modifyField\n";
            return;
        }

        my $field_format = $xml_offset{$offset}{Format};
        my $field_length = $xml_offset{$offset}{Length};
        my $fill_value   = $xml_offset{$offset}{Fill};
        my $data_length  = length($modifyData);
        print "Writing VPD field: name=$modifyField, region=$modifyRegion, offset=$offset, format=$field_format\n";

        if ( ('ascii' eq $field_format) || ('mixed' eq $field_format) )
        {
            if ( $data_length > $field_length )
            {
                printf "DATA TRUNCATION: reduced to $field_length characters\n";
            }
            for ( my $ii = 0; $ii < $field_length; $ii++ ) 
            {
                if ( $ii < $data_length )
                {
                    $write_memory[$offset+$ii] = ord(substr($modifyData,$ii,1));
                }
                else
                {
                    $write_memory[$offset+$ii] = $fill_value;
                }
            }
        }

        # hex data - fully specified, i.e. no fill necessary
        elsif (('hex' eq $field_format) && ((2 * $field_length) == $data_length))
        {
            for ( my $ii=0; $ii<$field_length; $ii++ )
            {
                $write_memory[$offset+$ii] = hex(substr($modifyData, $ii<<1, 2));
            }
        }

        # hex data - fill necessary
        elsif ( 'hex' eq $field_format )
        {
            if (0 != ($data_length % 2))
            {
                print "DATA ERROR: hex field data length is NOT a multiple of 2\n";
                exit(0);
            }
            if ( $data_length > (2 *$field_length) )
            {
                printf "DATA TRUNCATION: reduced to $field_length bytes\n";
            }

            for ( my $ii = 0; $ii < $field_length; $ii++ ) 
            {
                if ( ($ii * 2) < $data_length )
                {
                    $write_memory[$offset+$ii] = hex(substr($modifyData,($ii * 2), 2));
                }
                else
                {
                    $write_memory[$offset+$ii] = $fill_value;
                }
            }
        }
        print "MODIFYING $xml_offset{$offset}{region} $xml_offset{$offset}{name}\n" if ($debug & 0x2000);

		my $checksum_mod = 0;
		my $last_mod = 0;
		my $region_found = 0;
		foreach my $offset_mod (@xml_offsetList)
		{
		   if ($region_found == 1) 
		   {
			  last;
		   }
		   my $length_mod = $xml_offset{$offset_mod}{Length};

		   $last_mod = $offset_mod + $length_mod;
		   for (my $ii=$offset_mod; $ii<$last_mod; $ii++)
		   {
			  if ($xml_offset{$offset_mod}{region} eq $xml_offset{$offset}{region}) 
			  { 
				 if ($xml_offset{$offset_mod}{name} eq 'checksum')
				 {
					$write_memory[$ii] = (0 - $checksum_mod) & 0xFF;
					$region_found = 1;
					$checksum_mod = 0;
					last;
				 }
				 else
				 {
					$checksum_mod += $write_memory[$ii];
				 }
			  }
		   }
	    }

		my $status_mod = 0;
		my $write_length = $last_mod - $offset;
		
        $status_mod = sendWriteAddressAndData( $device, $eepromId, $offset, $write_length );

		return;
    }

    # update checksums
    my $checksum = 0;
    foreach my $offset (@xml_offsetList)
    {
        my $last = $offset + $xml_offset{$offset}{Length};
        for (my $ii=$offset; $ii<$last; $ii++)
        {
            if ($xml_offset{$offset}{name} eq 'checksum')
            {
                $write_memory[$ii] = (0 - $checksum) & 0xFF;
                $checksum = 0;
                next;
            }
            $checksum += $write_memory[$ii];
        }
    }
    if ( $debug & 0x20)
    {
        print "\nData to be written to the VPD\n";
        print_memory_structures(\@write_memory);
    }

    if ('' ne $saveWriteVPDdata)
    {
        saveDataToFile(      $saveWriteVPDdata,\@write_memory);
        saveStructureToFile( $saveWriteVPDdata,\@write_memory);
    }


    #print "\nsizes:\nXML\t".scalar(@xml_memory)." & $xml_memorysize\nRead\t".scalar(@read_memory)."\nWrite\t".scalar(@write_memory)."\n";

    if ( $dryRun )
    {
        print "\n\nThis is a dry run - no data was WRITTEN TO any VPD!\n\n";
        print "\nData that would have been written to the VPD\n";
        print_memory_structures(\@write_memory);
        return;
    }

    print "Writing to VPD EEROM ID #$eepromId, size $xml_memorysize\n" ;#if ($debug & 0x400);
    #$debug |= 0x300;
    for ( my $offset = 0; $offset < $xml_memorysize; $offset += 256 )
    {
        my $status = 0;

        # print "offset $offset of memorysize $xml_memorysize\n";
        $status = sendWriteAddressAndData( $device, $eepromId, $offset, 256 );
    }

    print "\n\nVPD has been updated!\n\n";
}

#***********************************************************************
#* main() - 
#***********************************************************************
sub main()
{
    my $device       = "";
    my $eepromId     = "";
    my $readOnly     = 0;
    my $updateOnly   = 0;
    my $replaceOnly  = 0;
    my $fixOnly      = 0;
    my $devo         = 0;
    my $modifyOnly   = 0;
    my $modifyStr    = "";
    my $modifyRegion = "";
    my $modifyField  = "";
    my $modifyData   = "";
    my $help         = 0;
    my $sum;
    my $operation;

    # Find OS
    my $os = $^O;
    print "Running in: $os\n";
#    if ('linux' ne $os)
#    {
#        print "This program is currently NOT supported on this operating system!\n";
#        exit(0);
#    }

    # get options from command line
    my $debugstring = '';
    my $result = GetOptions(
                             "device=s"     => \$device,
                             "id=s"         => \$eepromId,
                             "read"         => \$readOnly,
                             "update"       => \$updateOnly,
                             "replace"      => \$replaceOnly,
                             "modify=s"     => \$modifyStr,
                             "fix"          => \$fixOnly,
                             "dryrun"       => \$dryRun,
                             "help"         => \$help,
                   #hidden
                             "devo"         => \$devo,
                             "debug=s"      => \$debugstring,
                             "savexml=s"    => \$saveXmlVPDdata,
                             "saveread=s"   => \$saveReadVPDdata,
                             "savewrite=s"  => \$saveWriteVPDdata,
                           );

    # can't run with no options
    if (0 == $result)
    {
        usageExit();
    }

    # usage menu options: -help, -devo
    if ( $help != 0 )
    {
        printUsage();
        exit(0);
    }
    if ( $devo != 0 )
    {
        print_hidden_options();
        exit(0);
    }


    # parse -modify string if available
    if ( $modifyStr ne "" )
    {
        ($modifyRegion, $modifyField, $modifyData) = split( '/', $modifyStr );

        if ( ($modifyRegion eq "") || ($modifyField eq "") || ($modifyData eq "") )
        {
            print "Invalid Region/Field/Data string for -modify.\n";
            usageExit();
        }
        $modifyOnly = 1;
    }

    # only one operation allowed
    $sum = $readOnly + $updateOnly + $replaceOnly + $fixOnly + $modifyOnly;
    if ( 0 == $sum )
    {
        print "Defaulting to READ existing VPD ONLY, no writes will be performed\n";
        $readOnly = 1;
    }
    else
    {
        if ( 1 != $sum )
        {
            print "You can only specify ONE of these options at a time: -read -replace -update -modify\n";
            usageExit();
        }
    }

    # verify that -device, -id, and xml filename are there
    if ( $debugstring ne '' )
    {
        $debug = hex $debugstring;
        printf "Running in DEBUG mode %d./0x%X\n",$debug,$debug;
    }
    if ( $device eq "" )
    {
        print "ERROR: Device not specified\n";
        usageExit();
    }
    if ( $eepromId eq "" )
    {
        print "ERROR: EEPROM ID not specified\n";
        usageExit();
    }
    if ( @ARGV == 0 )
    {
        print "No file was specified\n";
        exit(1);
    }
    my $filename = $ARGV[0];

    unless (-e $filename)
    {
       printf("File \'$filename\' does not exist.\n");
       usageExit();
    }

    # all set, now perform the operation
    $operation = 'REPLACE'  if $replaceOnly;
    $operation = 'UPDATE'   if $updateOnly ;
    $operation = 'FIX'      if $fixOnly    ;
    $operation = 'READ'     if $readOnly   ;
    $operation = 'MODIFY'   if $modifyOnly ;
    printf( "Using: File $filename, Device $device, EEPROM ID $eepromId, Operation $operation\n" );

    if ($operation eq 'READ')
    {
        $debug |= 8;
    }
    performOperation( $filename, $device, $eepromId, $operation, $modifyRegion, $modifyField, $modifyData );
}

#***********************************************************************
#* it all starts here 
#***********************************************************************
main();
print "Done.\n";
exit(0);

