#!/bin/bash
#
# This Script will be invoked from diagnoseProbe command for EMC VMAX Probe.
#
# usage:
#
# ./diagnoseProbe_vmax.sh arg1 arg2 arg3(Non-Interactive Mode)
#
# Please see diagnoseProbe.xml in conf folder for the argument mapping.
#
# Argument Mapping
# args1 : <parameter name="Unisphere IP Address" property="unisphereIP" />
# args2 : <parameter name="Username" property="username" prompt="true"/>
# args3 : <parameter name="Password" property="password" prompt="true"/>


PASS="Success"
FAIL="Failure"
checkpoint_run=0
checkpoint_failed=0
symmetric_ids=""

summary()
{
  echo
  echo "Summary Result"
  echo "Checkpoints Run: " $checkpoint_run
  echo "Checkpoints Failed: " $checkpoint_failed
  echo "Checkpoints Passed: " $((checkpoint_run - checkpoint_failed))
  echo
  echo "Done!!!"
}

help_method()
{
  echo "Please provide the mandatory details to access the probe diagnosis report!!!"
  echo "Try again by using below mentioned options..."
  echo "1. Interactive mode - './diagnoseProbe_vmax.sh -i'"
  echo "2. Non interative mode - './diagnoseProbe_vmax.sh <IP> <Username> <Password>' "
}

function valid_ip()
{
    local  ip=$1
    local  stat=1

    if [[ $ip =~ ^[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}$ ]]; then
        OIFS=$IFS
        IFS='.'
        ip=($ip)
        IFS=$OIFS
        [[ ${ip[0]} -le 255 && ${ip[1]} -le 255 \
            && ${ip[2]} -le 255 && ${ip[3]} -le 255 ]]
        stat=$?
    fi
    return $stat
}


general_vmax_diagnose()
{
#Checkpoint 1 : Checking  Unisphere connectivity

    result_checkpoint_1=$FAIL
    checkpoint_run=$((checkpoint_run + 1))
    echo
    echo "Verifying target Connection on ($args1:'8443')..."
	if ! [ -x "$(command -v nc)" ]; then
		echo "Cannot verify the connection, install the package nmap-ncat."
		result_checkpoint_1=$FAIL
		checkpoint_failed=$((checkpoint_failed + 1))
	else
		ncStatus=$(nc -zv $args1 8443 2>&1 | grep Connected)
		if [ "$?" -eq 0 ]; then
			result_checkpoint_1=$PASS
		else
			echo "EMC Unisphere is not accessible from HDCA Probe"
			result_checkpoint_1=$FAIL
			checkpoint_failed=$((checkpoint_failed + 1))
			echo "Result: " $result_checkpoint_1
			summary
			exit 1
		fi
	fi
    echo "Result: " $result_checkpoint_1


#Checkpoint 2 : Check if EMC Solutions Enabler Symmetrix CLI is installed on Probe

    result_checkpoint_2=$FAIL
    checkpoint_run=$((checkpoint_run + 1))
    echo
    echo "Verifying SYMCLI version..."
    version_check=$(/usr/symcli/bin/symcli)    
    if [ "$?" -eq 0 ]; then
        echo "$version_check" 
        result_checkpoint_2=$PASS
    else
        echo "Install EMC Solutions Enabler Symmetrix CLI on HDCA probe."
        checkpoint_failed=$((checkpoint_failed + 1))
        result_checkpoint_2=$FAIL
        echo "Result: " $result_checkpoint_2
        summary
        exit 1
    fi
    echo "Result: " $result_checkpoint_2

#Checkpoint 3 : Checking EMC provider details in /var/symapi/config/netcnfg file

    result_checkpoint_3=$FAIL
    checkpoint_run=$((checkpoint_run + 1))
    echo
    echo "Verifying EMC provider details..."
    provider_details=$(cat /var/symapi/config/netcnfg | tail | grep SYMCLI)
    if [ "$?" -eq 0 ]; then
        echo "$provider_details"
        result_checkpoint_3=$PASS
    else
        echo "Provide entry for Unisphere IP in /var/symapi/config/netcnfg file"
        checkpoint_failed=$((checkpoint_failed + 1))
        result_checkpoint_3=$FAIL
        echo "Result: " $result_checkpoint_3
        summary
        exit 1
    fi
    echo "Result: " $result_checkpoint_3

#Checkpoint 4 : Verifying SYMCLI certificate handshake with the target machine
	
	pushd 2> /dev/null
    echo
    echo "Verifying SYMCLI certificate handshake..."
    if [ -f "/usr/storapi/bin/manage_server_cert" ]; then
	if [ "$EUID" == "0" ]; then
		cd /usr/emc/API/symapi/config/cert/
		hostname=$(/usr/storapi/bin/manage_server_cert list)
	else
		echo " "
		echo "Warning: Please login as root user to execute checkpoints for verifing SYMCLI certificate and data collection. Exiting!"
		summary
		exit 1
	fi
    elif [ -f "/usr/storapi/bin/manage_server_cert.sh" ]; then
		cd /usr/emc/API/symapi/config/cert/
		hostname=$(/usr/storapi/bin/manage_server_cert.sh list 2>/dev/null )
    else
		echo ""
		echo "Error:  File manage_server_cert.sh doesn't exists, so unable to execute checkpoints for verifing SYMCLI certificate and data collection. Exiting!"
		summary
		exit 1
	fi
 
	result_checkpoint_4=$FAIL
	checkpoint_run=$((checkpoint_run + 1))

	ip=$(echo "$hostname"|awk '{print $NF}'|cut -d ":" -f2 )
	IPValidation_result=$(if valid_ip $ip; then echo valid; else echo invalid; fi)
	if [ $IPValidation_result == "invalid" ];then
		lookup_result=$(nslookup $ip | tr '\n' ' ' | awk '{print $NF}')
		lookup_validation=$(if valid_ip $lookup_result; then echo valid; else echo invalid; fi)
		if [ $lookup_validation == "valid" ];then
			cd /opt/emc/SYMCLI/bin
			export SYMCLI_CONNECT=SYMCLI_$args1
			result=$(./symcfg list)
			echo "$result"
			if [ -n "$(echo $result | grep 'SymmID')" ]; then
				symmetric_ids=$(echo "$result" | grep '[0-9*]'| awk '{print $1}')
				result_checkpoint_4=$PASS
			else
				#echo "Remote client/server handshake failed."
				checkpoint_failed=$((checkpoint_failed + 1))
				result_checkpoint_4=$FAIL
				echo "Result: " $result_checkpoint_4
				summary
				exit 1
			fi
		else
			echo "Hostname not Found. Please import SYMAPI certificate using IP address of HDCA probe!!"
			checkpoint_failed=$((checkpoint_failed + 1))
			result_checkpoint_4=$FAIL
			echo "Result: " $result_checkpoint_4
			summary
			exit 1
		fi
	elif [ $IPValidation_result == "valid" ];then
		cd /opt/emc/SYMCLI/bin
		export SYMCLI_CONNECT=SYMCLI_$args1
		result=$(./symcfg list)
		echo "$result"
		if [ -n "$(echo $result | grep 'SymmID')" ]; then
			symmetric_ids=$(echo "$result" | grep '[0-9*]'| awk '{print $1}')
			result_checkpoint_4=$PASS
	        else
			#echo "Remote client/server handshake failed."
			checkpoint_failed=$((checkpoint_failed + 1))
			result_checkpoint_4=$FAIL
			echo "Result: " $result_checkpoint_4
			summary
			exit 1
        fi
    fi
	popd 2> /dev/null
    echo "Result: " $result_checkpoint_4
    dataCollection_vmax_diagnose

}

dataCollection_vmax_diagnose()
{
#Checkpoint 5 : Checking Data collection from VMAX storage syatem

    result_checkpoint_5=$FAIL
    checkpoint_run=$((checkpoint_run + 1))
    echo
    echo "Verifying data collection..."
    array_list_not_monitored=""
    storage_system_array=$(curl -s -u $args2:$args3 -k1 https://$args1:8443/univmax/restapi/performance/Array/keys)
    if [ -n "$(echo $storage_system_array | grep 'Error')" ]; then
       echo "Invalid Credentials!!"
       checkpoint_failed=$((checkpoint_failed + 1))
       result_checkpoint_5=$FAIL
    else
       array_symm_ids=$(echo $storage_system_array| grep -P '(?<=\"symmetrixId\":).*' | grep -oP ':"\K[^"]+')
       for uni_value in $symmetric_ids; do
       flag=false
       for arr_symId in $array_symm_ids; do
         if [ $uni_value = $arr_symId ]; then
             lastAvailableDate=$(echo $storage_system_array | grep -oP '"lastAvailableDate":\K[0-9]+')
             currentdate=$(date +%s%3N)
             time_gap=$(($currentdate - $lastAvailableDate))
             if [ $time_gap -lt 1800000 ]; then
		echo $uni_value "has passed Data Collection Checkpoint"
                 result_checkpoint_5=$PASS
             fi
             flag=true
             break
         fi
         done 
             $flag ||  array_list_not_monitored+="$uni_value "       
        done
          if [ -n "$(echo $array_list_not_monitored)" ]; then
            echo "VMAX storage" $array_list_not_monitored "is not monitored by this Unisphere."
            checkpoint_failed=$((checkpoint_failed + 1))
            result_checkpoint_5=$FAIL
         else
            result_checkpoint_5=$PASS
         fi
      fi   

 echo "Result: " $result_checkpoint_5

summary

}


# interactive_mode or non-interactive_mode
while getopts ":i" opt; do
  case $opt in
    i)
      interactive_mode="true"
      ;;
   \?)
      echo "Invalid option: -$OPTARG"
      exit 1
      ;;
  esac
done

# If mode is interactive_mode then ask from user
if [ "$interactive_mode" == "true" ]; then
    echo "Provide the following details: "
    read -p "Enter Unisphere IP Address : " args1
    if [ -z "$args1" ]; then
        echo "IP address is Mandatory !!!"
        exit 1
    fi

    read -p "Enter Username : " args2
    if [ -z "$args2" ]; then
        echo "Username is Mandatory !!!"
        exit 1
    fi

    read -s -p "Enter Password : " args3
    echo
    if [ -z "$args3" ]; then
        echo "Password is Mandatory !!!"
        exit 1
    fi
    general_vmax_diagnose
else
    #Arguments
    if [ "$#" -eq "3" ]; then
        args1=$1
        args2=$2
        args3=$3
        general_vmax_diagnose
    else
        help_method
        exit 1
    fi
fi
