# Copyright 2008-2014 Cumulus Systems Incorporated.
# All Rights Reserved.

# This file contains the code to collect the Configuration data for a XIV Storage whose property file name is passed as argument to this script.
#
# @param : Property File Name - Name of the probe instance specific property file.
#
# @return : In case of success, returns 0. In case of failure, returns 1.
#
# Sample command: perl ibmxivConfDataGenerator.pl <Property File Name>

#!/usr/bin/perl -w

# It is used for strict compilation.
use strict;

# It is used to get base file name from the complete file name.
use File::Basename;

# It is used to get the current date and time.
use Time::HiRes qw(gettimeofday);
use POSIX "strftime";

# This is used to create folder recursively.
use File::Path qw(mkpath);

# This is use to move folder.
use File::Copy;

# It is used to indicate the path where common module is placed.
use lib "/usr/local/megha/lib/common/perl";

# It is required for logging information.
use commonLogModule;

# It is required for getting Key from probe specific property file.
use commonPerlUtil;

# It is required for getting runtime location of the xivGetConfData.sh file.
use commonPropFileModule;

# It is required for parsing LUN resource data.
use commonDataParsingUtil;

# It is used for using common conf util..
use commonProbeConfUtil;

# ----------------------------------------------------------------------------------------------------------------------------------------------------
# GLOBAL Variables
# ----------------------------------------------------------------------------------------------------------------------------------------------------

# This variable is used to store the name of .properties file which corresponds to the input XIV IP address.
my $propFileName = "";

# This variable is used to store the return status of function calls.
my $status = 0;

# This variable is used to store the XIV Storage User-Id which is used to get the Storage Configuration data.
my $xivUserId = "";

# This variable is used to store the XIV Storage Key corresponding to $xivUserId.
my $xivKey = "";

# This variable is used to store the XIV Storage IP Address.
my $xivIpAddress = "";

# This variable is used to store the MARS Site name.
my $marsSiteName = "";

# This variable is used to store the base folder path on the MARS probe.
my $baseFolder = "/usr/local/megha";

# This variable holds the path where raw data corresponding to each resource will get dumped.
my $tempConfDataDir = "";

# This variable holds the path where final conf data will be dumped.
my $xivConfFolder = "";

# This variable holds the timestamp of the folder containing raw conf dump files.
my $currentTime = "";

# This variable stores the probe type.
my $probeType = "IbmXiv";

# This variable stores the run time location of the xivGetConfData.sh file.
my $xivRuntimelocation = "";

# This variable stores commonLogModule instance.
my $logObj = "";

# Get commonProbeConfUtil instance.
my $probeConfUtilObj = "";

# ----------------------------------------------------------------------------------------------------------------------------------------------------
# Main function
# ----------------------------------------------------------------------------------------------------------------------------------------------------
# Initialize the environment.
$status = init();
if (0 != $status)
{
    $logObj->error("Call to init() failed.");
    goto EXIT;
}

# Dump meta-data info
$status = dumpMetaData();
if (0 != $status) {
    $logObj->error("Call to dumpMetaData() failed.");
    goto EXIT;
}

# Get the raw Configuration data.
$status = getRawConfData();
if (0 != $status)
{
    $logObj->error("Call to getRawConfData() failed.");
    goto EXIT;
}

EXIT:

exit $status;

# ----------------------------------------------------------------------------------------------------------------------------------------------------
# Sub-routines
# ----------------------------------------------------------------------------------------------------------------------------------------------------

# This function is used to initialize the environment to run the Perl script correctly.
#
# @affected global variables :
#   $baseFolder
#   $propFileName
#   $tempConfDataDir
#   $currentTime
#   $xivConfFolder
#
# @return :
#   0 if Success
#   1 if Error
sub init
{
    # This variable is used to store the return value of function calls.
    my $retVal = 0;

    # This variable stores Log folder path.
    my $logFolder = $baseFolder."/logs";

    # If log folder does not exist then create it.
    if (! (-e $logFolder))
    {
        mkpath($logFolder);
    }

    # Get commonLogModule instance.
    $logObj = commonLogModule->getInstance($logFolder, $probeType, $xivIpAddress);

    # If number of command line arguments are not correct then quit.
    if (1 != ($#ARGV + 1))
    {
        $logObj->error("Number of arguments to ibmxivConfDataGenerator.pl are not correct. Input argument count = ".($#ARGV + 1));
        $retVal = 1;

        goto EXIT;
    }
	
	# Get the name of the property file from command line argument.
    $propFileName = $ARGV[0];

    # Read the .properties file corresponding to XIV Storage.
    $status = readPropertyFile();
    if (0 != $status)
    {
        $logObj->error("Call to readPropertyFile() failed.");
        $retVal = 1;

        goto EXIT;
    }

    # Update the final log message information with populated XIV probe id.
    $logObj->setProbeId($xivIpAddress);

	# Get commonProbeConfUtil instance.
	$probeConfUtilObj = commonProbeConfUtil->getInstance($baseFolder, $probeType, $xivIpAddress, $logObj);

    # Get commonPerlUtil instance.
    my $perlUtilObj = commonPerlUtil->getInstance($baseFolder, $probeType, $xivIpAddress, $logObj);

    # This variable stores maximum retry count for decrypting XIV key.
    my $retryCount = 3;

    # Decrypt XIV Key.
    while (0 < $retryCount)
    {
        $status = $perlUtilObj->getDataValue($xivKey, $xivKey);

        # In case of successful execution, exit from while loop.
        if (0 == $status)
        {
            last;
        }
        # If call invocation fails after all retries then it is an error. So exit from the script.
        elsif (1 == $retryCount)
        {
            $logObj->error("Call to getDataValue() failed.");
            goto EXIT;
        }

        # Try calling getDataValue() after some delay.
        $logObj->warn("Call to getDataValue() failed. Will retry after sixty seconds."); 
        sleep(60);

        $retryCount--;
    }

    # Get commonPropFileModule instance.
    my $propFileObj = commonPropFileModule->getInstance($baseFolder, $probeType, $xivIpAddress, $logObj);

    # Get the runtime location.
    $status = $propFileObj->getProbeProperty("runtime.location", $xivRuntimelocation);
    if (0 != $status)
    {
        $logObj->error("Call to getProbeProperty() failed.");
        $retVal = 1;

        goto EXIT;
    }

    # This variable is used to store the path of Probe folder in db/probe.
    my $xivProbeFolder = $baseFolder."/db/probe/".$marsSiteName."/IbmXiv_".$xivIpAddress;
    $logObj->info("Creating folder: [$xivProbeFolder].");

    # Create the folder for this XIV Probe in the probe folder.
    unless (defined eval {mkpath($xivProbeFolder)})
    {
        $logObj->error("Not able to create folder: [$xivProbeFolder].");
        $retVal = 1;

        goto EXIT;
    }

    # This variable hold temporary conf data collection path.
    $tempConfDataDir = $xivProbeFolder."/temp_conf";

    # Create the folder for temporary conf data.
    $logObj->info("Creating temporary conf data path: [$tempConfDataDir].");

    unless (defined eval {mkpath($tempConfDataDir)})
    {
        $logObj->error("Not able to create folder: [$tempConfDataDir].");
        $retVal = 1;

        goto EXIT;
    }

    # It is used to store the current date and time in "YYYYMMDD_HHMMSS" format.
    $currentTime = strftime("%Y%m%d_%H%M%S", gmtime(time));

    # This variable is used to store the path of folder where final MARS Conf files will be created.
    $xivConfFolder = $xivProbeFolder."/conf";
    $logObj->info("Creating folder: [$xivConfFolder].");

    # Create the folder for the final MARS Conf files.
    unless (defined eval {mkpath($xivConfFolder)})
    {
        $logObj->error("Not able to create folder: [$xivConfFolder].");
        $retVal = 1;

        goto EXIT;
    }

EXIT:

    return $retVal;
}

# This function is used to read the .properties file of IBM XIV. It reads the XIV IP, User-Id and XIV Key values.
#
# @affected global variables :
#    $xivIpAddress
#    $xivUserId
#    $xivKey
#    $marsSiteName
#
# @return :
#   0 if Success
#   1 if Error
sub readPropertyFile
{
    # This variable is used to store the return code for this function.
    my $retVal = 0;

    $logObj->info("Reading file: $propFileName");

    open(PROP_FILE, $propFileName) or die $logObj->error("Not able to open the file: [$propFileName].");

    # Loop through the content of .properties file.
    while (<PROP_FILE>)
    {
        # Remove the new line character.
        $_ =~ s/\r\n//g;
        $_ =~ s/\n//g;

        # Look for XIV IP Address.
        if ($_ =~ /^StorageIP=(\S+)(\s*)$/)
        {
            $xivIpAddress = $1;
        }

        # Look for XIV UserId.
        elsif ($_ =~ /^UserName=(.+)(\s*)$/)
        {
            $xivUserId = $1;
        }

        # Look for XIV Key.
        elsif ($_ =~ /^Password=(.+)(\s*)$/)
        {
            $xivKey = $1;
        }

        # Look for Site name.
        elsif ($_ =~ /^SiteName=(.+)(\s*)$/)
        {
            $marsSiteName = $1;
        }
        else
        {
            # Current line is of no use. So ignore it.
        }
    }

    if ("" eq $xivIpAddress)
    {
        $logObj->error("XIV IP not present in the properties file.");
        $retVal = 1;

        goto EXIT;

    }

    if ("" eq $xivUserId)
    {
        $logObj->error("XIV User-Id not present in the properties file.");
        $retVal = 1;

        goto EXIT;
    }

    if ("" eq $xivKey)
    {
        $logObj->error("XIV Key not present in the properties file.");
        $retVal = 1;

        goto EXIT;
    }
    if ("" eq $marsSiteName)
    {
        $logObj->error("MARS Site name is not present in the properties file.");
        $retVal = 1;

        goto EXIT;
    }

EXIT:

    close(PROP_FILE);

    return $retVal;
}

# This function is used to get the storage raw configuration data.
#
# @affected global variables :
#    None
#
# @return :
#   0 if Success
#   1 if Error
sub getRawConfData
{
    # This variable is used to store the return code for this function.
    my $status = 0;

    #Dump custom resource
	$status = dumpCustomResource();

	if (0 != $status) {
        $logObj->error("Call to dumpCustomResource() failed.");
        goto EXIT;
    }

	# Dump storage data
	my $rawConfFile = $tempConfDataDir."/xivStorageConf.dump";

    # This variable is used to store the command to get the XIV Storage Conf data.
    my $cmd = $xivRuntimelocation."/ibmxivGetConfData.sh Storage $xivIpAddress $xivUserId $xivKey > ".$rawConfFile;
    my $cmdWithoutKey = $xivRuntimelocation."/ibmxivGetConfData.sh Storage $xivIpAddress $xivUserId <Key> ".$rawConfFile;

    # This variable stores maximum retry count for checking storage data.
    my $retryCount = 3;

    # Check storage resource data.
    while (0 < $retryCount)
    {
        $logObj->info("Running command: [$cmdWithoutKey].");

        # Run the command to get the XIV Storage Conf data.
        system($cmd);

        $status = checkStorageData();

        # In case of successful execution, exit from while loop.
        if (0 == $status)
        {
            last;
        }
        # If call invocation fails after all retries then it is an error. So exit from the script.
        elsif (1 == $retryCount)
        {
            $logObj->error("Call to checkStorageData() failed.");
            goto EXIT;
        }

        # Try calling checkStorageData() after some delay.
        $logObj->warn("Call to checkStorageData() failed. Will retry after ten minutes."); 
        sleep(600);

        $retryCount--;
    }

    $rawConfFile = $tempConfDataDir."/xivHostConf.dump";

    # This variable is used to store the command to get the XIV Host Conf data.
    my $cmd = $xivRuntimelocation."/ibmxivGetConfData.sh Host $xivIpAddress $xivUserId $xivKey > ".$rawConfFile;
    $cmdWithoutKey = $xivRuntimelocation."/ibmxivGetConfData.sh Host $xivIpAddress $xivUserId <Key> ".$rawConfFile;

    $logObj->info("Running command: [$cmdWithoutKey].");

    # Run the command to get the XIV Conf data.
    system($cmd);

    $rawConfFile = $tempConfDataDir."/xivDiskConf.dump";

    # This variable is used to store the command to get the XIV Disk Conf data.
    my $cmd = $xivRuntimelocation."/ibmxivGetConfData.sh Disk $xivIpAddress $xivUserId $xivKey > ".$rawConfFile;
    $cmdWithoutKey = $xivRuntimelocation."/ibmxivGetConfData.sh Disk $xivIpAddress $xivUserId <Key> > ".$rawConfFile;

    $logObj->info("Running command: [$cmdWithoutKey].");

    # Run the command to get the XIV Conf data.
    system($cmd);

    $rawConfFile = $tempConfDataDir."/xivVolumeConf.dump";

    # This variable is used to store the command to get the XIV Volume Conf data.
    my $cmd = $xivRuntimelocation."/ibmxivGetConfData.sh Volume $xivIpAddress $xivUserId $xivKey > ".$rawConfFile;
    $cmdWithoutKey = $xivRuntimelocation."/ibmxivGetConfData.sh Volume $xivIpAddress $xivUserId <Key> > ".$rawConfFile;

    $logObj->info("Running command: [$cmdWithoutKey].");

    # Run the command to get the XIV Conf data.
    system($cmd);

    $rawConfFile = $tempConfDataDir."/xivPoolConf.dump";

    # This variable is used to store the command to get the XIV Pool Conf data.
    my $cmd = $xivRuntimelocation."/ibmxivGetConfData.sh Pool $xivIpAddress $xivUserId $xivKey > ".$rawConfFile;
    $cmdWithoutKey = $xivRuntimelocation."/ibmxivGetConfData.sh Pool $xivIpAddress $xivUserId <Key> > ".$rawConfFile;

    $logObj->info("Running command: [$cmdWithoutKey].");

    # Run the command to get the XIV Conf data.
    system($cmd);

    $rawConfFile = $tempConfDataDir."/xivLocalFCPortConf.dump";

    # This variable is used to store the command to get the XIV Local FC Port Conf data.
    my $cmd = $xivRuntimelocation."/ibmxivGetConfData.sh LocalFCPort $xivIpAddress $xivUserId $xivKey > ".$rawConfFile;
    $cmdWithoutKey = $xivRuntimelocation."/ibmxivGetConfData.sh LocalFCPort $xivIpAddress $xivUserId <Key> > ".$rawConfFile;

    $logObj->info("Running command: [$cmdWithoutKey].");

    # Run the command to get the XIV Conf data.
    system($cmd);

	$rawConfFile = $tempConfDataDir."/xivHostConnectivityConf.dump";

    # This variable is used to store the command to get the XIV Local FC Port Conf data.
    my $cmd = $xivRuntimelocation."/ibmxivGetConfData.sh HostConnect $xivIpAddress $xivUserId $xivKey > ".$rawConfFile;
    $cmdWithoutKey = $xivRuntimelocation."/ibmxivGetConfData.sh HostConnect $xivIpAddress $xivUserId <Key> > ".$rawConfFile;

    $logObj->info("Running command: [$cmdWithoutKey].");

    # Run the command to get the XIV Conf data.
    system($cmd);
	
    # Get the raw data for LUN resource.
    getLunRawConfData();

    # Once all data collection done, move conf data from tempConfDataDir to ConfDataDir and delete temp files.
    my $newDir = $xivConfFolder."/Conf_".$currentTime;

    move($tempConfDataDir, $newDir ) or die $logObj->error("Not able to move folder [$tempConfDataDir] to [$newDir].");

    rename($newDir, "$newDir.DONE") || die $logObj->error("Not able to rename [$newDir] to [$newDir.DONE]");

    $logObj->info("Configuration data collection done. Next Data will be collected After 4 hours.");

EXIT:

    return $status;
}

# This function is used to dump custom attribute resource in xivProbeConf.dump file.
#
# @affected global variables :
#    None
#
# @return :
#   None
sub dumpCustomResource {
	
	# This variable is used to store the return code for this function.
    my $retVal = 0;

	# File to dump custom attribute data
	my $rawConfFile = $tempConfDataDir."/xivProbeConf.dump";
	
	$logObj->info("[ibmxivConfDataGenerator] Dumping custom attributes...");

	# It stores custom attribute information.
	my %customAttrHash = $probeConfUtilObj->getCustomAttrValues($marsSiteName);
	
	open(my $NAME_FILE_HANDLE, ">", $rawConfFile) or die $logObj->error("Not able to open the file: [$rawConfFile].");

	#Dump custom resource
	print($NAME_FILE_HANDLE "R:__probe#" . $marsSiteName . "_" . $xivIpAddress . "\n");
	print($NAME_FILE_HANDLE "a:__name=" . $marsSiteName . "_" . $xivIpAddress . "\n");
	
	foreach my $key (keys(%customAttrHash)) {
		print($NAME_FILE_HANDLE "a:" . $key . "=" . $customAttrHash{$key} . "\n");
	}

	print($NAME_FILE_HANDLE "\n");
	$logObj->info("[ibmxivConfDataGenerator] Finished dumping custom attributes.");
	close($NAME_FILE_HANDLE);
	
	return $retVal;
}

# This function is used to dump meta data info into xivMetaData.dump file.
#
# @affected global variables :
#    None
#
# @return :
#   0 if Success
#   1 if Error
sub dumpMetaData {
	
	# This variable is used to store the return code for this function.
    my $retVal = 0;

	# Get commonPropFileModule instance.
    my $propFileObj = commonPropFileModule->getInstance($baseFolder, $probeType, $xivIpAddress, $logObj);

	# It stores "probe_host" meta-data information.
	my $probeHost = "";

    # Get the probe host details.
    $status = $propFileObj->getAppProperty("probe.host", $probeHost);
    if (0 != $status) {
        $logObj->error("[ibmxivConfDataGenerator][dumpMetaData] Call to getAppProperty() failed.");
        $retVal = 1;

        goto EXIT;
    }

	# File to dump meta data info
	my $metaDataFile = $tempConfDataDir."/xivMetaData.dump";

	$logObj->info("[ibmxivConfDataGenerator][dumpMetaData] Dumping meta-data info...");

	open(my $NAME_FILE_HANDLE, ">", $metaDataFile) or die $logObj->error("Not able to open the file: [$metaDataFile].");

	# Dump meta-data information.
	print($NAME_FILE_HANDLE "probe_host=" . $probeHost);
	print($NAME_FILE_HANDLE "\n");

	$logObj->info("[ibmxivConfDataGenerator][dumpMetaData] Finished dumping meta-data information.");

	# Closing file handle.
	close($NAME_FILE_HANDLE);

	return $retVal;
}

# This function first fetches the list of all hosts attached to the storage. It read the same dump file where all information about Hosts are
# present and store the host name in a new dump file. This host name file is then used for getting LUN related raw data.
#
# @affected global variables :
#    None
#
# @return :
#   None
sub getLunRawConfData
{
    # This file stores LUN resource raw conf data.
    my $rawConfFile = $tempConfDataDir."/xivLunConf.dump";

    # This file stores host related raw conf data. We will parse this file to get host names and dump it in $hostNameFile.
    my $hostDataFile = $tempConfDataDir."/xivHostConf.dump";

    # This file stores name of all hosts present in the storage separated by new line.
    my $hostNameFile = $tempConfDataDir."/xivHostNameList.dump";

    # This variable keeps track of the line containing column header whose data we need to extract.
    my $isHeaderInfo = 0;

    # This variable keeps track of line of which we need to do column related parsing.
    my $isDataStart = 0;

    # This array is used to store the required column numbers of a resource.
    my @columnNumberList = (0);

    # This array is used to store the column value of each column of a resource.
    my @columnValueList = ();

    # This hash is used to store the name of the resource and the column header name corresponding to each record.
    my %columnHeaderPositionHash = ();

    # Get commonDataParsingUtil instance.
    my $dataParsingObj = commonDataParsingUtil->getInstance($baseFolder, $probeType, $xivIpAddress, $logObj);

    $logObj->info("Reading raw configuration data for LUN resource.");

    open(my $NAME_FILE_HANDLE, ">", $hostNameFile) or die $logObj->error("Not able to open the file: [$hostNameFile].");

    open(HOST_DUMP, $hostDataFile) or die $logObj->error("Not able to open the file: [$hostDataFile].");

    while (<HOST_DUMP>)
    {
        # Remove the new line character.
        $_ =~ s/\r\n//g;
        $_ =~ s/\n//g;

        if (/^(\s*)(.+)>>host_list(\s*)$/)
        {
            $isHeaderInfo = 1;
        }
        elsif (/^(\s*)(.+)>>(\s*)$/)
        {
            $isDataStart = 0;
        }
        elsif (1 == $isHeaderInfo)
        {
            $dataParsingObj->getColumnDataPosition($_, \%columnHeaderPositionHash);

            $isHeaderInfo = 0;
            $isDataStart = 1;
        }
        elsif (1 == $isDataStart)
        {
            $dataParsingObj->getColumnDataValues($_, \%columnHeaderPositionHash, \@columnNumberList, \@columnValueList);

            print $NAME_FILE_HANDLE @columnValueList[0]."\n";
        }
        else
        {
            # Current line is of no use. So ignore it.
        }
    }

    close (HOST_DUMP);
    close($NAME_FILE_HANDLE);

    # This variable is used to store the command to get the XIV LUN Conf data.
    my $cmd = $xivRuntimelocation."/ibmxivGetConfData.sh Lun $xivIpAddress $xivUserId $xivKey $hostNameFile > ".$rawConfFile;
    my $cmdWithoutKey = $xivRuntimelocation."/ibmxivGetConfData.sh Pool $xivIpAddress $xivUserId <Key> > ".$rawConfFile;

    $logObj->info("Running command: [$cmdWithoutKey].");

    # Run the command to get the XIV Conf data.
    system($cmd);
}

# This function is used to check the storage raw configuration data. It tries to fetch the storage name from the raw dump file, the call is success
# if storage name fetched correctly. 
#
# @affected global variables :
#    None
#
# @return :
#   0 if Success
#   1 if Error
sub checkStorageData
{
    # This variable is used to store the return code for this function.
    my $retVal = 0;

    # This variable stores the storage name.
    my $storageName = "";

    my $rawConfFile = $tempConfDataDir."/xivStorageConf.dump";

    $logObj->info("Reading raw configuration data for Storage resource.");

    open(CONF_DUMP, $rawConfFile) or die $logObj->error("Not able to open the file: [$rawConfFile].");

    while (<CONF_DUMP>)
    {
        # Remove the new line character.
        $_ =~ s/\r\n//g;
        $_ =~ s/\n//g;

        # Get System Name.
        if (/^(\s*)system_name(\s*)(.+)(\s*)$/)
        {
            $storageName = $3;

            # Remove trailing spaces.
            $storageName =~ s/\s+$//;

            # Replace space by underscore in storage name.
            $storageName =~ s/ /_/g;

            # Replace "^" by underscore in storage name.
            $storageName =~ s/\^/_/g;

            $logObj->info("Storage name: [$storageName].");
        }
    }

    # Check if storage name populated correctly.
    if ($storageName eq "")
    {
        $retVal = 1;
    }

    return $retVal;
}