#!/usr/bin/expect -f

# Copyright 2008-2014 Cumulus Systems Incorporated.
# All Rights Reserved.

# Following are the error codes. Always keep it in sync with error codes in ibmxivStorageVerifier.pl file.
set ERROR_SUCCESS            0
set ERROR_INVALID_CRED       1
set ERROR_CONNECTION_FAILED  2
set ERROR_EXCEPTION_FOUND    3

set cmdType [lindex $argv 0]

# The script waits for 240 seconds and then only exits.
set timeout 240

# This function is used to handle the process for connecting to the Storage.
#
# @param xivIpAddress : XIV IP Address.
# @param userName     : User name for XIV Storage.
# @param password     : Password for XIV Storage.
proc sendConnectionDetails { xivIpAddress userName password } {
    global ERROR_INVALID_CRED
    global ERROR_CONNECTION_FAILED
    global ERROR_EXCEPTION_FOUND

    # Disable the logging of username and password in the dump files.
    log_user 0

    expect {
        "*User *ame:*" {
            send "$userName\r"
            exp_continue
        }

        "*Password:*" {
            send "$password\r"
            exp_continue
        }

        "*Machine IP/Hostname:*" {
            send "$xivIpAddress\r"
            exp_continue
        }

        "*>>" {
            send "\r";
        }

        "*Error:*LOGIN_FAILURE_USER_FAILED_TO_LOGIN*" {
            exit $ERROR_INVALID_CRED
        }

        "*Error:*NO_CONNECTION*" {
            exit $ERROR_CONNECTION_FAILED
        }

        "*Exception*" {
            exit $ERROR_EXCEPTION_FOUND
        }
    }

    log_user 1
}

if {"Version" == $cmdType} {
    set xivIpAddress [lindex $argv 1]
    set userName [lindex $argv 2]
    set password [lindex $argv 3]

    cd /opt/XIVGUI
    spawn -nottycopy -nottyinit ./xcli

    sendConnectionDetails $xivIpAddress $userName $password

    expect "*>>"
    send "version_get\r"

    expect "*>>"
    send "exit\r"

    exit $ERROR_SUCCESS
} elseif {"Storage" == $cmdType} {
    set xivIpAddress [lindex $argv 1]
    set userName [lindex $argv 2]
    set password [lindex $argv 3]

    cd /opt/XIVGUI
    spawn -nottycopy -nottyinit ./xcli

    sendConnectionDetails $xivIpAddress $userName $password

    expect "*>>"
    send "cod_list\r"

    expect "*>>"
    send "version_get\r"

    expect "*>>"
    send "system_capacity_list\r"

    expect "*>>"
    send "exit\r"

    exit $ERROR_SUCCESS
} elseif {"Host" == $cmdType} {
    set xivIpAddress [lindex $argv 1]
    set userName [lindex $argv 2]
    set password [lindex $argv 3]

    cd /opt/XIVGUI
    spawn -nottycopy -nottyinit ./xcli

    sendConnectionDetails $xivIpAddress $userName $password

    expect "*>>"
    send "host_list\r"

    expect "*>>"
    send "exit\r"

    exit $ERROR_SUCCESS
} elseif {"Pool" == $cmdType} {
    set xivIpAddress [lindex $argv 1]
    set userName [lindex $argv 2]
    set password [lindex $argv 3]

    cd /opt/XIVGUI
    spawn -nottycopy -nottyinit ./xcli

    sendConnectionDetails $xivIpAddress $userName $password

    expect "*>>"
    send "pool_list\r"

    expect "*>>"
    send "exit\r"

    exit $ERROR_SUCCESS
} elseif {"Disk" == $cmdType} {
    set xivIpAddress [lindex $argv 1]
    set userName [lindex $argv 2]
    set password [lindex $argv 3]

    cd /opt/XIVGUI
    spawn -nottycopy -nottyinit ./xcli

    sendConnectionDetails $xivIpAddress $userName $password

    expect "*>>"
    send "disk_list\r"

    expect "*>>"
    send "exit\r"

    exit $ERROR_SUCCESS
} elseif {"Volume" == $cmdType} {
    set xivIpAddress [lindex $argv 1]
    set userName [lindex $argv 2]
    set password [lindex $argv 3]

    cd /opt/XIVGUI
    spawn -nottycopy -nottyinit ./xcli

    sendConnectionDetails $xivIpAddress $userName $password

    expect "*>>"
    send "vol_list\r"

    expect "*>>"
    send "exit\r"

    exit $ERROR_SUCCESS
} elseif {"LocalFCPort" == $cmdType} {
    set xivIpAddress [lindex $argv 1]
    set userName [lindex $argv 2]
    set password [lindex $argv 3]

    cd /opt/XIVGUI
    spawn -nottycopy -nottyinit ./xcli

    sendConnectionDetails $xivIpAddress $userName $password

    expect "*>>"
    send "fc_port_list\r"

    expect "*>>"
    send "exit\r"

    exit $ERROR_SUCCESS
} elseif {"Lun" == $cmdType} {
    set xivIpAddress [lindex $argv 1]
    set userName [lindex $argv 2]
    set password [lindex $argv 3]
    set objListFile [lindex $argv 4]

    set fid [open $objListFile r]
    set content [read $fid]
    close $fid

    # Split into records on newlines.
    set hosts [split $content "\n"]

    cd /opt/XIVGUI
    spawn -nottycopy -nottyinit ./xcli

    sendConnectionDetails $xivIpAddress $userName $password

    foreach host $hosts {
        expect "*>>"
        send "mapping_list host=\"$host\"\r"
    }

    send "exit\r"

    exit $ERROR_SUCCESS
} elseif {"HostConnect" == $cmdType} {
    set xivIpAddress [lindex $argv 1]
    set userName [lindex $argv 2]
    set password [lindex $argv 3]

    cd /opt/XIVGUI
    spawn -nottycopy -nottyinit ./xcli

    sendConnectionDetails $xivIpAddress $userName $password

    expect "*>>"
    send "host_connectivity_list\r"

    expect "*>>"
    send "exit\r"

    exit $ERROR_SUCCESS
} elseif {"HostPerf" == $cmdType} {
    set xivIpAddress [lindex $argv 1]
    set userName [lindex $argv 2]
    set password [lindex $argv 3]
    set timeStamp [lindex $argv 4]
    set storageName [lindex $argv 5]
    set objListFile [lindex $argv 6]

    set fid [open $objListFile r]
    set content [read $fid]
    close $fid

    # Split into records on newlines
    set hosts [split $content "\n"]

    cd /opt/XIVGUI
    spawn -nottycopy -nottyinit ./xcli

    sendConnectionDetails $xivIpAddress $userName $password

    puts "\nPROBE_META-ixHost,$storageName"

    foreach host $hosts {
        expect "*>>"
        puts "STOP"
        send "statistics_get start=$timeStamp count=15 interval=1 resolution_unit=minute host=\"$host\"\r"
        puts "** START-$host **";
    }

    puts "\nSTOP"

    send "exit\r"

    exit $ERROR_SUCCESS
} elseif {"VolumePerf" == $cmdType} {
    set xivIpAddress [lindex $argv 1]
    set userName [lindex $argv 2]
    set password [lindex $argv 3]
    set timeStamp [lindex $argv 4]
    set storageName [lindex $argv 5]
    set objListFile [lindex $argv 6]

    set fid [open $objListFile r]
    set content [read $fid]
    close $fid

    # Split into records on newlines
    set vols [split $content "\n"]

    cd /opt/XIVGUI
    spawn -nottycopy -nottyinit ./xcli

    sendConnectionDetails $xivIpAddress $userName $password

    puts "\nPROBE_META-ixVolume,$storageName"

    foreach vol $vols {
        expect "*>>"
        puts "STOP"
        send "statistics_get start=$timeStamp count=15 interval=1 resolution_unit=minute vol=\"$vol\"\r"
        puts "** START-$vol **";
    }

    puts "\nSTOP"

    send "exit\r"

    exit $ERROR_SUCCESS
} elseif {"LocalFcPortPerf" == $cmdType} {
    set xivIpAddress [lindex $argv 1]
    set userName [lindex $argv 2]
    set password [lindex $argv 3]
    set timeStamp [lindex $argv 4]
    set storageName [lindex $argv 5]
    set objListFile [lindex $argv 6]

    set fid [open $objListFile r]
    set content [read $fid]
    close $fid

    # Split into records on newlines
    set localFcPorts [split $content "\n"]

    cd /opt/XIVGUI
    spawn -nottycopy -nottyinit ./xcli

    sendConnectionDetails $xivIpAddress $userName $password

    puts "\nPROBE_META-ixLocalFcPort,$storageName"

    foreach localFcPort $localFcPorts {
        expect "*>>"
        puts "STOP"
        send "statistics_get start=$timeStamp count=15 interval=1 resolution_unit=minute local_fc_port=\"$localFcPort\"\r"
        puts "** START-$localFcPort **";
    }

    puts "\nSTOP"

    send "exit\r"

    exit $ERROR_SUCCESS
} elseif {"HostFcPortPerf" == $cmdType} {
    set xivIpAddress [lindex $argv 1]
    set userName [lindex $argv 2]
    set password [lindex $argv 3]
    set timeStamp [lindex $argv 4]
    set storageName [lindex $argv 5]
    set objListFile [lindex $argv 6]

    set fid [open $objListFile r]
    set content [read $fid]
    close $fid

    ## Split into records on newlines
    set hostFcPorts [split $content "\n"]

    cd /opt/XIVGUI
    spawn -nottycopy -nottyinit ./xcli

    sendConnectionDetails $xivIpAddress $userName $password

    puts "\nPROBE_META-ixHostFcPort,$storageName"

    foreach hostFcPort $hostFcPorts {
        expect "*>>"
        puts "STOP"
        send "statistics_get start=$timeStamp count=15 interval=1 resolution_unit=minute host_fc_port=\"$hostFcPort\"\r"
        puts "** START-$hostFcPort **";
    }

    puts "\nSTOP"

    send "exit\r"

    exit $ERROR_SUCCESS
} elseif {"PerfObjList" == $cmdType} {
    set xivIpAddress [lindex $argv 1]
    set userName [lindex $argv 2]
    set password [lindex $argv 3]

    cd /opt/XIVGUI
    spawn -nottycopy -nottyinit ./xcli

    sendConnectionDetails $xivIpAddress $userName $password

    expect "*>>"
    puts "\n** START-Host **"
    send "host_list\r"

    expect "*>>"
    puts "STOP\n** START-Volume **"
    send "vol_list\r"

    expect "*>>"
    puts "STOP\n** START-LocalFCPort **"
    send "fc_port_list\r"

    expect "*>>"
    puts "STOP\n** START-HostFCPort **"
    send "host_connectivity_list\r"

    expect "*>>"
    puts "STOP\n** START-Storage **"
    send "cod_list\r"

    expect "*>>"
    puts "STOP"

    send "exit\r"

    exit $ERROR_SUCCESS
}
