# Copyright 2008-2014 Cumulus Systems Incorporated.
# All Rights Reserved.

# This file contains the code to collect the Performance data for the IBM XIV Storage whose Property File name is passed as argument to this script.
#
# @param : Property File Name - Name of the probe specific property file.
#
# @return : In case of Success, returns 0. In case of Failure, returns 1.
#
# Sample command: perl ibmxivPerfDataGenerator.pl <Property File Name>

#!/usr/bin/perl -w

# It is used for strict compilation.
use strict;

# It is used to subtract or add time from a given date.
use Date::Calc qw(Add_Delta_DHMS);

# It is used to get base file name from the complete file name.
use File::Basename;

# It is use to get current date and time.
use Time::HiRes qw(gettimeofday);

# Convert date and time information to string.
use POSIX "strftime";

# This is used to create folder recursively.
use File::Path qw(mkpath);

# This is use to move folder.
use File::Copy;

# It is used to indicate the path where common module is placed.
use lib "/usr/local/megha/lib/common/perl";

# It is required for logging information.
use commonLogModule;

# It is required for getting Key from probe specific property file.
use commonPerlUtil;

# It is required for getting runtime location of the xivGetConfData.sh file.
use commonPropFileModule;

# It is required for getting resources name.
use commonDataParsingUtil;

# ----------------------------------------------------------------------------------------------------------------------------------------------------
# GLOBAL Variables
# ----------------------------------------------------------------------------------------------------------------------------------------------------

# This variable is used to store the name of .properties file which corresponds to the input XIV IP address.
my $propFileName = "";

# This variable is used to store the return status of function calls.
my $status = 0;

# This variable is used to store the XIV Storage User-Id which is used to get the Storage Performance data.
my $xivUserId = "";

# This variable is used to store the XIV Storage Key corresponding to $xivUserId.
my $xivKey = "";

# This variable is used to store the XIV Storage IP Address.
my $xivIpAddress = "";

# This variable is used to store the MARS Site name.
my $marsSiteName = "";

# This variable is used to store the base folder path on the MARS probe.
my $baseFolder = "/usr/local/megha";

# This variable holds the path where raw data corresponding to each resource will get dumped.
my $tempPerfDataDir = "";

# This variable holds the path where final perf data will be dumped.
my $xivPerfFolder = "";

# This variable holds the path where temporary configuration data will be dump from which the resources names will be extracted to be used to generate
# performance data.
my $rawTempFile = "";

# This variable holds the timestamp of the folder containing raw perf dump files.
my $currentTime = "";

# This variable stores the probe type.
my $probeType = "IbmXiv";

# This variable stores Storage Name.
my $storageName = "";

# This variable stores System time in "HH:MM:SSS" format.
my $systemTime = "";

# This variable stores System date in "YYYY-MM-DD" format.
my $systemDate = "";

# This variable stores the run time location of the xivGetConfData.sh file.
my $xivRuntimelocation = "";

# This variable is used to store the absolute path of the file which contain list of host names
my $hostNamesFile = "";

# This variable is used to store the absolute path of the file which contain list of volume names.
my $volNamesFile = "";

# This variable is used to store the absolute path of the file which contain list of local FC port names.
my $localFcPortNamesFile = "";

# This variable is used to store the absolute path of the files which contain list of host FC port names.
my $hostFcPortNamesFile = "";

# This variable is used to store the line number of the file temp.txt to be skip while extracting the resources data from it.
my $lineToSkip = undef;

# This variable stores commonLogModule instance.
my $logObj = "";

# ----------------------------------------------------------------------------------------------------------------------------------------------------
# Main function
# ----------------------------------------------------------------------------------------------------------------------------------------------------

# Initialize the environment.
$status = init();
if (0 != $status)
{
    $logObj->error("Call to init() failed.");
    goto EXIT;
}

# Dump meta-data info
$status = dumpMetaData();
if (0 != $status) {
    $logObj->error("Call to dumpMetaData() failed.");
    goto EXIT;
}

# Read the configuration data of host, volume and FC port resources into the temp file.
$status = dumpConfDataIntoFile();
if (0 != $status)
{
    $logObj->error("Call to dumpConfDataIntoFile() failed.");
    goto EXIT;
}

# Parse the temp file to get the names of Host, Volume, and FC port (local/host) into separate files.
$status = createFilesWithResourceNames();
if (0 != $status)
{
    $logObj->error("Call to createFilesWithResourceNames() failed.");
    goto EXIT;
}

# Get the raw Performance data.
$status = getRawPerfData();
if (0 != $status)
{
    $logObj->error("Call to getRawPerfData() failed.");
    goto EXIT;
}

EXIT:

exit $status;

# ----------------------------------------------------------------------------------------------------------------------------------------------------
# Sub-routines
# ----------------------------------------------------------------------------------------------------------------------------------------------------
# This function is used to initialize the environment to run the Perl script correctly.
#
# @affected global variables :
#   $propFileName
#   $xivKey
#   $xivRuntimelocation
#   $tempPerfDataDir
#   $rawTempFile
#   $hostNamesFile
#   $volNamesFile
#   $localFcPortNamesFile
#   $hostFcPortNamesFile
#   $xivPerfFolder
#
# @return :
#   0 if Success
#   1 if Error
sub init
{
    # This variable is used to store the return value of function calls.
    my $retVal = 0;

    # This variable stores Log folder path.
    my $logFolder = $baseFolder."/logs";

    # If log folder does not exist then create it.
    if (! (-e $logFolder))
    {
        mkpath($logFolder);
    }

    # Get commonLogModule instance.
    $logObj = commonLogModule->getInstance($logFolder, $probeType, $xivIpAddress);

    # If number of command line arguments are not correct then quit.
    if (2 != ($#ARGV + 1))
    {
        $logObj->error("Number of arguments to ibmxivPerfDataGenerator.pl are not correct. Input arguments count = ".($#ARGV + 1));
        $retVal = 1;

        goto EXIT;
    }

    # Get the name of the property file from command line argument.
    $propFileName = $ARGV[0];

    # Number of seconds for which thread to sleep to achieve staggered data collection
    my $mycurrentTime = strftime("%Y%m%d_%H%M%S", gmtime(time));
    
    $logObj->info("ibmxivPerfDataGenerator Staggered Collection Time - $ARGV[1]");

    if (-1 != $ARGV[1])
    {
        $logObj->info("ibmxivPerfDataGenerator current time - $mycurrentTime");
        $logObj->info("ibmxivPerfDataGenerator thread sleeping for -[ $ARGV[1] ]");
    
        sleep($ARGV[1]);

        $mycurrentTime = strftime("%Y%m%d_%H%M%S", gmtime(time));
        $logObj->info("ibmxivPerfDataGenerator current time - $mycurrentTime");
    }

    # Read the .properties file corresponding to XIV Storage.
    $status = readPropertyFile();
    if (0 != $status)
    {
        $logObj->error("Call to readPropertyFile() failed.");
        $retVal = 1;

        goto EXIT;
    }

    # Update the final log message information with populated XIV probe id.
    $logObj->setProbeId($xivIpAddress);

    # Get commonPerlUtil instance.
    my $perlUtilObj = commonPerlUtil->getInstance($baseFolder, $probeType, $xivIpAddress, $logObj);

    # Decrypt the Key.
    $status = $perlUtilObj->getDataValue($xivKey, $xivKey);
    if (0 != $status)
    {
        $logObj->error("Call to getDataValue() failed.");
        $retVal = 1;

        goto EXIT;
    }

    # Get commonPropFileModule instance.
    my $propFileObj = commonPropFileModule->getInstance($baseFolder, $probeType, $xivIpAddress, $logObj);

    # Get the runtime location.
    $status = $propFileObj->getProbeProperty("runtime.location", $xivRuntimelocation);
    if (0 != $status)
    {
        $logObj->error("Call to getProbeProperty() failed.");
        $retVal = 1;

        goto EXIT;
    }

    # This variable is used to store the path of Probe folder in db/probe.
    my $xivProbeFolder = $baseFolder."/db/probe/".$marsSiteName."/IbmXiv_".$xivIpAddress;
    $logObj->info("Creating folder: [$xivProbeFolder].");

    # Create the folder for this XIV Probe in the probe folder.
    unless (defined eval {mkpath($xivProbeFolder)})
    {
        $logObj->error("Not able to create folder: [$xivProbeFolder].");
        $retVal = 1;

        goto EXIT;
    }

    # This variable hold perf data directory path.
    $tempPerfDataDir = $xivProbeFolder."/temp_perf";

    # Create the folder for temporary perf data.
    $logObj->info("Creating temp Perf data path: [$tempPerfDataDir].");

    unless (defined eval {mkpath($tempPerfDataDir)})
    {
        $logObj->error("Not able to create folder: [$tempPerfDataDir].");
        $retVal = 1;

        goto EXIT;
    }

    # Set the file path which will contain resource names.
    $rawTempFile = $tempPerfDataDir."/temp.txt";
    $hostNamesFile = $tempPerfDataDir."/hostNames.txt";
    $volNamesFile = $tempPerfDataDir."/volNames.txt";
    $localFcPortNamesFile = $tempPerfDataDir."/localFcPortNames.txt";
    $hostFcPortNamesFile = $tempPerfDataDir."/hostFcPortNames.txt";

    # This variable is used to store the path of folder where final MARS Perf files will be created.
    $xivPerfFolder = $xivProbeFolder."/perf";
    $logObj->info("Creating folder: [$xivPerfFolder].");

    # Create the folder for the final MARS Perf files.
    unless (defined eval {mkpath($xivPerfFolder)})
    {
        $logObj->error("Not able to create folder: [$xivPerfFolder].");
        $retVal = 1;

        goto EXIT;
    }

EXIT:

    return $retVal;
}

# This function is used to read the .properties file of IBM XIV. It reads the XIV IP Address, User-Id and Key values.
#
# @affected global variables :
#   $xivIpAddress
#   $xivUserId
#   $xivKey
#   $marsSiteName
#
# @return :
#   0 if Success
#   1 if Error
sub readPropertyFile
{
    # This variable is used to store the return code for this function.
    my $retVal = 0;

    $logObj->info("Reading file: $propFileName");

    open(PROP_FILE, $propFileName) or die $logObj->error("Not able to open the file: [$propFileName].");

    # Loop through the content of .properties file.
    while (<PROP_FILE>)
    {
        # Remove the new line character.
        $_ =~ s/\r\n//g;
        $_ =~ s/\n//g;

        # Look for XIV IP Address.
        if ($_ =~ /^StorageIP=(\S+)(\s*)$/)
        {
            $xivIpAddress = $1;
        }
        # Look for XIV UserId.
        elsif ($_ =~ /^UserName=(.+)(\s*)$/)
        {
            $xivUserId = $1;
        }
        # Look for XIV Key.
        elsif ($_ =~ /^Password=(.+)(\s*)$/)
        {
            $xivKey = $1;
        }
        # Look for Site name.
        elsif ($_ =~ /^SiteName=(.+)(\s*)$/)
        {
             $marsSiteName = $1;
        }
        else
        {
            # Current line is of no use. So ignore it.
        }
    }

    if ("" eq $xivIpAddress)
    {
        $logObj->error("XIV IP not present in the properties file.");
        $retVal = 1;

        goto EXIT;

    }

    if ("" eq $xivUserId)
    {
        $logObj->error("XIV User-Id not present in the properties file.");
        $retVal = 1;

        goto EXIT;
    }

    if ("" eq $xivKey)
    {
        $logObj->error("XIV Key not present in the properties file.");
        $retVal = 1;

        goto EXIT;
    }

    if ("" eq $marsSiteName)
    {
        $logObj->error("Mars Site name is not present in the properties file.");
        $retVal = 1;

        goto EXIT;
    }

EXIT:

    close(PROP_FILE);

    return $retVal;
}

# This function is used to read the configuration data of host, volume, and FC port resources into the temp file.
#
# @affected global variables :
#   $currentTime
#
# @return :
#   0 if Success
#   1 if Error
sub dumpConfDataIntoFile
{
    # This variable is used to store the return code for this function.
    my $retVal = 0;

    # This variable is used to store the command to get the XIV Storage Host, Volume, and FC ports configuration data.
    my $cmd = $xivRuntimelocation."/ibmxivGetConfData.sh PerfObjList $xivIpAddress $xivUserId $xivKey > ".$rawTempFile;
    my $cmdWithoutKey = $xivRuntimelocation."/ibmxivGetConfData.sh PerfObjList $xivIpAddress $xivUserId <Key> > ".$rawTempFile;

    $logObj->info("Running command: [$cmdWithoutKey].");

    # Run the command to populate the temp file with configuration data.
    system($cmd);

    # It is used to store the current time (UTC) minus 15 minute in "YYYYMMDD_HHMMSS" format.
    $currentTime = strftime("%Y%m%d_%H%M%S", gmtime(time - (15 * 60)));

EXIT:

    return $retVal;
}

# This function is used to dump meta data info into xivMetaData.txt file.
#
# @affected global variables :
#    None
#
# @return :
#   0 if Success
#   1 if Error
sub dumpMetaData {
	
	# This variable is used to store the return code for this function.
    my $retVal = 0;

	# Get commonPropFileModule instance.
    my $propFileObj = commonPropFileModule->getInstance($baseFolder, $probeType, $xivIpAddress, $logObj);

	# It stores "probe_host" meta-data information.
	my $probeHost = "";

    # Get the probe host details.
    $status = $propFileObj->getAppProperty("probe.host", $probeHost);
    if (0 != $status) {
        $logObj->error("[ibmxivPerfDataGenerator][dumpMetaData] Call to getAppProperty() failed.");
        $retVal = 1;

        goto EXIT;
    }

	# File to dump meta data info
	my $metaDataFile = $tempPerfDataDir."/xivMetaData.txt";

	$logObj->info("[ibmxivPerfDataGenerator][dumpMetaData] Dumping meta-data info...");

	open(my $NAME_FILE_HANDLE, ">", $metaDataFile) or die $logObj->error("Not able to open the file: [$metaDataFile].");

	# Dump meta-data information.
	print($NAME_FILE_HANDLE "probe_host=" . $probeHost);
	print($NAME_FILE_HANDLE "\n");

	$logObj->info("[ibmxivPerfDataGenerator][dumpMetaData] Finished dumping meta-data information.");

	# Closing file handle.
	close($NAME_FILE_HANDLE);

	return $retVal;
}

# This function is used to parse the temp file to get the names of Host, Volume, and FC ports into separate files.
#
# @affected global variables :
#    $storageName
#    $systemDate
#    $systemTime
#
# @return :
#   0 if Success
#   1 if Error
sub createFilesWithResourceNames
{
    # This variable is used to store the return code for this function.
    my $retVal = 0;

    # This array is used to store the required column numbers of a resource.
    my @columnNumberList = ();

    # This array is used to store the column value of each column of a resource.
    my @columnValueList = ();

    # This hash is used to store the name of the resource and the column header name corresponding to each record.
    my %columnHeaderPositionHash = ();

    # This variable keeps track of the line containing column header whose data we need to extract.
    my $isHeaderInfo = 0;

    # This variable keeps track of line of which we need to do column related parsing.
    my $isDataStart = 0;

    # This variable keeps track of the line containing column header whose data we need to extract.
    my $isCodHeaderInfo = 0;

    # This variable keeps track of line of which we need to do column related parsing.
    my $isCodDataStart = 0;

    # This variable stores resource name.
    my $resName = "";

    # This variable is used to store the file handle of files containing name of resources.
    my $RES_FILE_HANDLE;

    # Get commonDataParsingUtil instance.
    my $dataParsingObj = commonDataParsingUtil->getInstance($baseFolder, $probeType, $xivIpAddress, $logObj);

    open(PERF_TEMP, $rawTempFile) or die $logObj->error("Not able to open the file: [$rawTempFile].");

    while (<PERF_TEMP>)
    {
        # Skip line containing the command.
        if (defined $lineToSkip)
        {
            next if $. == $lineToSkip;
        }

        # Remove the new line character.
        $_ =~ s/\r\n//g;
        $_ =~ s/\n//g;

        # Indicate starting of Host configuration data.
        if (/^\*\* START-Host \*\*$/)
        {
            $isHeaderInfo = 1;

            # Extract the first column from Host conf data containing Host names.
            @columnNumberList = (0);

            # Skip line containing the command "host_list".
            $lineToSkip = ($. + 1);

            open($RES_FILE_HANDLE, ">", $hostNamesFile) or die $logObj->error("Not able to open the file: [$hostNamesFile].");
        }
        # Indicate starting of Volume configuration data.
        elsif (/^\*\* START-Volume \*\*$/)
        {
            $isHeaderInfo = 1;

            # Extract the first column from Volume conf data containing Volume names.
            @columnNumberList = (0);

            # Skip line containing the command "vol_list".
            $lineToSkip = ($. + 1);

            open($RES_FILE_HANDLE, ">", $volNamesFile) or die $logObj->error("Not able to open the file: [$volNamesFile].");
        }
        # Indicate starting of local FC port configuration data.
        elsif (/^\*\* START-LocalFCPort \*\*$/)
        {
            $isHeaderInfo = 1;

            # Extract the first column from local FC port conf data containing FC port component Id.
            @columnNumberList = (0);

            # Skip line containing the command "fc_port_list".
            $lineToSkip = ($. + 1);

            open($RES_FILE_HANDLE, ">", $localFcPortNamesFile) or die $logObj->error("Not able to open the file: [$localFcPortNamesFile].");
        }
        # Indicate starting of host FC port configuration data.
        elsif (/^\*\* START-HostFCPort \*\*$/)
        {
            $isHeaderInfo = 1;

            # Extract the second column from Host FC port conf data containing FC port WWPNs.
            @columnNumberList = (1);

            # Skip line containing the command "host_connectivity_list".
            $lineToSkip = ($. + 1);

            open($RES_FILE_HANDLE, ">", $hostFcPortNamesFile) or die $logObj->error("Not able to open the file: [$hostFcPortNamesFile].");
        }
        # Indicate starting of COD data.
        elsif (/^\*\* START-Storage \*\*$/)
        {
            $isCodHeaderInfo = 1;

            # Skip line containing the command "cod_list".
            $lineToSkip = ($. + 1);
        }
        # Indicate ending of configuration data.
        elsif (/^(\s*)(.+)>>STOP$/)
        {
            $isDataStart = 0;

            # Empty the content of list containing the number of column to be extracted from data.
            undef @columnNumberList;

            close($RES_FILE_HANDLE);
        }
        # Indicate starting of resource configuration data header.
        elsif (1 == $isHeaderInfo)
        {
            $dataParsingObj->getColumnDataPosition($_, \%columnHeaderPositionHash);

            $isHeaderInfo = 0;
            $isDataStart = 1;
        }
        # Indicate starting of COD configuration data header.
        elsif (1 == $isCodHeaderInfo)
        {
            $dataParsingObj->getColumnDataPosition($_, \%columnHeaderPositionHash);

            $isCodHeaderInfo = 0;
            $isCodDataStart = 1;
        }
        # Indicate starting of actual resource configuration data.
        elsif (1 == $isDataStart)
        {
            $dataParsingObj->getColumnDataValues($_, \%columnHeaderPositionHash, \@columnNumberList, \@columnValueList);

            $resName = @columnValueList[0];

            # Replace space by underscore in host name.
            $resName =~ s/ /_/g;

            print $RES_FILE_HANDLE "$resName\n";
        }
        # Indicate starting of COD actual configuration data.
        elsif (1 == $isCodDataStart)
        {
            # Get System Name.
            if ($_ =~ /^(\s*)system_name(\s*)(.+)(\s*)$/)
            {
                $storageName = $3;

                # Remove trailing spaces.
                $storageName =~ s/\s+$//;

                # Replace space by underscore in storage name.
                $storageName =~ s/ /_/g;

                # Replace character "^" by underscore in storage name.
                $storageName =~ s/\^/_/g;
            }

            # Get the system date.
            if ($_ =~ /^(\s*)date(\s*)(.+)(\s*)$/)
            {
                $systemDate = $3;

                # Remove trailing spaces.
                $systemDate =~ s/\s+$//;
            }

            # Get the system time.
            if ($_ =~ /^(\s*)time(\s+)(.+)(\s*)$/)
            {
                $systemTime = $3;

                # Remove trailing spaces.
                $systemTime =~ s/\s+$//;

                $logObj->info("Storage Time: [$systemTime].");
            }
        }
        else
        {
            # Current line is of no use. So ignore it.
        }
    }

    close (PERF_TEMP);

    # Delete the temp file.
    unlink($rawTempFile);
    $logObj->info("Deleting temporary file: [$rawTempFile].");

EXIT:

    return $retVal;
}

# This function is used to get the storage raw performance data.
#
# @affected global variables :
#    None
#
# @return :
#   0 if Success
#   1 if Error
sub getRawPerfData
{
    # This variable is used to store the return code for this function.
    my $retVal = 0;

    # This variable holds the path of the dump file containing the raw performance data.
    my $rawPerfFile = "";

    # This variable holds the raw perf collection start time stamp.
    my $timeStamp = "";

    # Split the system Date and time into array.
    my @date = split('-', $systemDate);
    my @time = split(':', $systemTime);

    # Subtract 15 minute from the time.
    (my $year, my $month, my $day, my $hh, my $mm, my $ss) = Add_Delta_DHMS($date[0], $date[1], $date[2], $time[0], $time[1], $time[2], 0, 0, -15, 0);

    # Set the time in XIV storage format.
    $timeStamp = sprintf('%d-%02d-%02d.%02d:%02d:%02d', $year, $month, $day, $hh, $mm, $ss);

    # Get Host Perf Data
    $rawPerfFile = $tempPerfDataDir."/ibmxivHostPerf.dump";

    # This variable is used to store the command to get the XIV Host Perf data.
    my $cmd = $xivRuntimelocation."/ibmxivGetConfData.sh HostPerf $xivIpAddress $xivUserId $xivKey $timeStamp $storageName $hostNamesFile > ".$rawPerfFile;
    my $cmdWithoutKey = $xivRuntimelocation."/ibmxivGetConfData.sh HostPerf $xivIpAddress $xivUserId <Key> $timeStamp $storageName $hostNamesFile > ".$rawPerfFile;

    $logObj->info("Running command: [$cmdWithoutKey].");

    # Run the command to get the XIV Perf data.
    system($cmd);

    # Get Volume Perf Data
    $rawPerfFile = $tempPerfDataDir."/ibmxivVolumePerf.dump";

    # This variable is used to store the command to get the XIV Volume Perf data.
    $cmd = $xivRuntimelocation."/ibmxivGetConfData.sh VolumePerf $xivIpAddress $xivUserId $xivKey $timeStamp $storageName $volNamesFile > ".$rawPerfFile;
    $cmdWithoutKey = $xivRuntimelocation."/ibmxivGetConfData.sh VolumePerf $xivIpAddress $xivUserId <Key> $timeStamp $storageName $volNamesFile > ".$rawPerfFile;

    $logObj->info("Running command: [$cmdWithoutKey].");

    # Run the command to get the XIV Perf data.
    system($cmd);

    # Get Local FC Port Perf Data
    $rawPerfFile = $tempPerfDataDir."/ibmxivLocalFCPortPerf.dump";

    # This variable is used to store the command to get the XIV Local FC Port Perf data.
    $cmd = $xivRuntimelocation."/ibmxivGetConfData.sh LocalFcPortPerf $xivIpAddress $xivUserId $xivKey $timeStamp $storageName $localFcPortNamesFile > ".$rawPerfFile;
    $cmdWithoutKey = $xivRuntimelocation."/ibmxivGetConfData.sh LocalFcPortPerf $xivIpAddress $xivUserId <Key> $timeStamp $storageName $localFcPortNamesFile > ".$rawPerfFile;

    $logObj->info("Running command: [$cmdWithoutKey].");

    # Run the command to get the XIV Perf data.
    system($cmd);

    # Get Host FC Port Perf Data
    $rawPerfFile = $tempPerfDataDir."/ibmxivHostFCPortPerf.dump";

    # This variable is used to store the command to get the XIV Host FC Port Perf data.
    $cmd = $xivRuntimelocation."/ibmxivGetConfData.sh HostFcPortPerf $xivIpAddress $xivUserId $xivKey $timeStamp $storageName $hostFcPortNamesFile > ".$rawPerfFile;
    $cmdWithoutKey = $xivRuntimelocation."/ibmxivGetConfData.sh HostFcPortPerf $xivIpAddress $xivUserId <Key> $timeStamp $storageName $hostFcPortNamesFile > ".$rawPerfFile;

    $logObj->info("Running command: [$cmdWithoutKey].");

    # Run the command to get the XIV Perf data.
    system($cmd);

    # Once all data collection done, move perf data from tempPerfDataDir to perfDataDir and delete temp files.
    my $timeStampDir = $xivPerfFolder."/Perf_".$currentTime;

    move($tempPerfDataDir, $timeStampDir ) or die $logObj->error("Not able to move folder [$tempPerfDataDir] to [$timeStampDir].");

    rename($timeStampDir, "$timeStampDir.DONE") || die $logObj->error("Not able to rename [$timeStampDir] to [$timeStampDir.DONE]");

    $logObj->info("Performance data collection done. Next Data will be collected After 15 minutes.");

EXIT:

    return $retVal;
}
