# Copyright 2008-2013 Cumulus Systems Incorporated.
# All Rights Reserved.

# This file contains the code to collect the Raw Configuration data for a NetApp Storage whose IP Address is passed as argument to this script.
#
# @param : IP Address     - NetApp Storage IP Address
# @param : Operation type - Specify the opertion type which could be either All_Raw_Data or Cifs_Data
#
# @return : In case of success, returns 0. In case of failure, returns 1.
#
# Sample command: perl netappCliDataGenerator.pl <IP Address> <Op type>

#!/usr/bin/perl -w

# It is used for strict compilation.
use strict;

# It is used to get base file name from the complete file name.
use File::Basename;

# It is used to get the current date and time.
use Time::HiRes qw(gettimeofday);
use POSIX "strftime";

# This is used to create folder recursively.
use File::Path qw(mkpath);

# This module provide functions that returns newly created HTTP::Request objects.
use HTTP::Request::Common qw(POST);

# It is used to dispatch web requests.
use LWP::UserAgent;

# It is used to decode the base64 encoded string.
use MIME::Base64;

# ----------------------------------------------------------------------------------------------------------------------------------------------------
# Error codes corresponding to netappGetCliData.sh. Always keep it in sync with error codes in netappGetCliData.sh.
# ----------------------------------------------------------------------------------------------------------------------------------------------------
my $ERROR_SUCCESS               = 0;
my $ERROR_SSH_INVALID_CRED      = 1;
my $ERROR_SSH_CONNECTION_FAILED = 2;
my $ERROR_INVALID_OPTION_FOUND  = 3;

# ----------------------------------------------------------------------------------------------------------------------------------------------------
# GLOBAL Variables
# ----------------------------------------------------------------------------------------------------------------------------------------------------

# Contain Probe Version number.
my $probeVersion = "10";

# This variable is used to store the path which contains properties files for MARS probe. It is needed to read the MARS Site name.
my $propFilePath = "";

# This variable is used to store the name of NetApp Probe properties file corresponding to input NetApp IP address.
my $propFileName = "";

# This variable is used to store the file handle for NetApp CLI output data.
my $CLI_DATA_HANDLE;

# This variable is used to store the return status of a function call.
my $status = 0;

# This variable is used to store NetApp Storage User name which is used to connect to the NetApp storage.
my $netappUserName = "";

# This variable is used to store NetApp Storage Password which is used to connect to the NetApp storage.
my $netappPassword = "";

# This variable is used to store NetApp Storage IP Address.
my $netappIpAddress = "";

# This variable is used to store location of NetApp runtime files.
my $netappRunTimeLocation = "";

# This variable is used to store the Server Mode.
my $netappServerMode = "";

# This variable is used to store the complete path of MARS app.properties file.
my $appPropertiesFilePath = "";

# This variable is used to store MARS Site name.
my $marsSiteName = "";

# This variable is used to store the path which NetApp CLI output file will get created.
my $netappCliOutputPath = "";

# This variable is used to store the name of file which is used to dump NetApp CLI output.
my $netappCliOutputFile = "";

# This variable is used to store the complete name of log file.
my $logFile = "";

# This variable is used to store the file handle of log file.
my $LOG_HANDLE;

# This variable is used to store the base folder path on the MARS probe.
my $baseFolder = "";

# This variable is used to store the complete path of final NetApp CLI output file.
my $finalCliOutputFile = "";

# This array holds all the properties file present in "conf/probe" folder.
my @files = ();

# This variable is used to store the name of the file which is used to dump the Raw NetApp CLI output.
my $netappCliRawOutputFileName = "netappRawCliOutput.dump";

# This variable is used to store the name of NetApp Expect script which is used to get the NetApp CLI data.
my $netappExpectScript = "netappGetCliData.sh";

# This variable is used to store the operation type which this script needs to perform.
my $operationType = "";

# This variable is used to store the total number of arguments expected by this script.
my $ARG_COUNT = 2;

# This variable is used to store the complete path of probe specific folder in db/probe folder.
my $netappLogFolder = "";
# ----------------------------------------------------------------------------------------------------------------------------------------------------
# Main function
# ----------------------------------------------------------------------------------------------------------------------------------------------------

# Initialize the environment.
$status = init();
if (0 != $status)
{
    logMsg("Error: Call to init() failed.");
    goto EXIT;
}

if ("All_Raw_Data" eq $operationType)
{
    # Prepare the Probe Metadata.
    $status = prepareProbeMetadata();
    if (0 != $status)
    {
        logMsg("Error: Call to prepareProbeMetadata() failed.");
        goto EXIT;
    }
    
    # Get NetApp CLI data.
    $status = getNetAppCliData();
    if (0 != $status)
    {
        logMsg("Error: Call to getNetAppCliData() failed.");
        goto EXIT;
    }
    
    # Rename the final CLI output file to make it ready to be processed by java conf thread.
    $status = processConfFile();
    if (0 != $status)
    {
        logMsg("Error: Call to processConfFile() failed.");
        goto EXIT;
    }
}
elsif ("Cifs_Data" eq $operationType)
{
    # Dump CIFS raw data in a file which Java thread will read.
    $status = dumpCifsData();
    if (0 != $status)
    {
        logMsg("Error: Call to dumpCifsData() failed.");
        goto EXIT;
    }
}
elsif ("Port" eq $operationType)
{
    # Dump Port raw data in a file which Java thread will read.
    $status = dumpPortData();
    if (0 != $status)
    {
        logMsg("Error: Call to dumpPortData() failed.");
        goto EXIT;
    }
}
else
{
    logMsg("Error: Invalid operation type passed. Operation type = [$operationType].");
}

EXIT:

# De-init the environment.
deInit();

logMsg("Info: Return status: [$status].");

exit $status;

# ----------------------------------------------------------------------------------------------------------------------------------------------------
# Sub-routines
# ----------------------------------------------------------------------------------------------------------------------------------------------------

# This function is used to initialize the environment to run the Perl script correctly.
#
# @affected global variables :
#   $baseFolder
#   $CONF_HANDLE
#   $netappIpAddress
#   $netappUserName
#   $netappPassword
#   $propFilePath
#   $appPropertiesFilePath
#   $netappCliOutputPath
#   $netappRunTimeLocation
#
# @return :
#   0 if Success
#   1 if Error
sub init
{
    # This variable is used to store the return value of function calls.
    my $retVal = 0;

    # Set MARS base folder.
    #$baseFolder = "/cygdrive/e/Megha_NetApp/megha/server/deploy";
    $baseFolder = "/usr/local/megha";

    # Set NetApp log folder path.
    $netappLogFolder = $baseFolder."/logs/netappProbe";

    # If log folder does not exist then create it.
    if (!(-e $netappLogFolder))
    {
         # Create the folder for log folder.
        unless (defined eval {mkpath($netappLogFolder)})
        {
            print("Error: Not able to create folder: [$netappLogFolder].");
            return;
        }
    }

    # If number of command line arguments are not correct then quit.
    if ($ARG_COUNT != ($#ARGV + 1))
    {
        logMsg("Error: Number of arguments to netappCliOutputGenerator.pl are not correct. Input argument count = [".($#ARGV + $ARG_COUNT)."]");
        $retVal = 1;

        goto EXIT;
    }

    # Get NetApp Storage IP Addrss from command line argument.
    $netappIpAddress = $ARGV[0];
    logMsg("Info: Input IP Address = [$netappIpAddress]");

    # Get Operation type from command line argument.
    $operationType = $ARGV[1];
    logMsg("Info: Input Operation type = [$operationType]");

    # Set MARS Probe properties file path.
    $propFilePath = $baseFolder."/conf/probe";

    # Get the name of .properties file corresponding to NetApp Storage.
    $retVal = getPropertyFileName();
    if (0 != $retVal)
    {
        logMsg("Error: Call to getPropertyFileName() failed.");
        $retVal = 1;

        goto EXIT;
    }

    # Read the .properties file corresponding to NetApp Storage.
    $retVal = readPropertyFile();
    if (0 != $retVal)
    {
        logMsg("Error: Call to readPropertyFile() failed.");
        $retVal = 1;

        goto EXIT;
    }

    # This variable is used to store the path of NetApp Probe folder in "db/probe" folder.
    my $netAppProbeFolder = $baseFolder."/db/probe/".$marsSiteName."/NetApp_".$netappIpAddress;
    logMsg("Info: Creating folder: [$netAppProbeFolder].");

    # Create the folder for this NetApp Probe in the probe folder.
    unless (defined eval {mkpath($netAppProbeFolder)})
    {
        logMsg("Error: Not able to create folder: [$netAppProbeFolder].");
        $retVal = 1;

        goto EXIT;
    }

    # Set the path where NetApp CLI output will get dumped.
    $netappCliOutputPath = $netAppProbeFolder."/cli_data";
    logMsg("Info: Creating NetApp CLI output folder: [$netappCliOutputPath].");

    # Create the folder for temporary conf data.
    unless (defined eval {mkpath($netappCliOutputPath)})
    {
        logMsg("Error: Not able to create folder: [$netappCliOutputPath].");
        $retVal = 1;

        goto EXIT;
    }

    # Set the complete path of MARS app.properties file.
    $appPropertiesFilePath = $baseFolder."/conf/app.properties";
    logMsg("Info: app.properties file path: [$appPropertiesFilePath].");

    # Read the content of app.properties file to get NetApp runtime file location.
    $retVal = readAppPropertyFile();
    if (0 != $retVal)
    {
        logMsg("Error: Call to readAppPropertyFile() failed.");
        $retVal = 1;

        goto EXIT;
    }

    # Get the decrypted password
    $retVal = getDecryptedPassword();
    if (0 != $retVal)
    {
        logMsg("Error: Call to getDecryptedPassword() failed.");
        $retVal = 1;

        goto EXIT;
    }

EXIT:

    return $retVal;
}

# This function is used to collect NetApp CLI output data and dump it in a file.
#
# @affected global variables :
#   None
#
# @return :
#   0 if Success
#   1 if Error
sub getNetAppCliData
{
    # This variable is used to store the return code for this function.
    my $retVal = 0;

    # This variable is used to store the exit status of "system" command.
    my $exitCode = 0;

    # This variable is used to indicate whether we have found NetApp CLI name in the raw CLI output file.
    my $commandFound = 0;

    # This variable is used to indicate the beginning of NetApp CLI output in the raw CLI output file.
    my $cliOutputStart = 0;
    
    # This variable is used to indicate the beginning of license related output in the raw CLI output file.
    my $licenseFound = 0;
    
    # This variable is used to store the content of line which got passed in the previous iteration. We are storing it as we don't want to append
    # "\n \" in the last line of CLI output.
    my $previousLine = "";

    # This array is used to store the name of filers present on the NetApp Storage.
    my @filerList = ();

    # Construct the complete name of NetApp CLI raw output file.
    my $netappCliRawOutputFile = $netappCliOutputPath."/".$netappCliRawOutputFileName;

    # This variable is used to store the command to get the NetApp CLI data specific to NetApp Storage.
    my $cmd = $netappRunTimeLocation."/".$netappExpectScript." NetApp $netappIpAddress $netappUserName $netappPassword > ".$netappCliRawOutputFile;
    logMsg("Info: Running command: [$netappRunTimeLocation/$netappExpectScript NetApp $netappIpAddress $netappUserName <password> > $netappCliRawOutputFile].");

    # Run the command to get the NetApp CLI data.
    $exitCode = executeSystemCmd($cmd);
    if ($ERROR_SUCCESS != $exitCode)
    {
        # If there is some problem is collecting NetApp CLI data then it is an error.
        logMsg("Error: Failed to collect CLI output for NetApp Storage.");

        $retVal = $exitCode;
        goto EXIT;
    }

    # Get the list of filers present on the NetApp Storage.
    my $retValue = getFilerList(\@filerList);
    if (0 != $retValue)
    {
        logMsg("Error: Call to getFilerList() failed.");
    }

    # Loop through each filer name.
    foreach my $vfilerName (@filerList)
    {
        logMsg("Info: vFiler name: [$vfilerName].");

        # Prepare the command to get the CLI output specific to a Filer.
        $cmd = $netappRunTimeLocation."/".$netappExpectScript." Filer $netappIpAddress $netappUserName $netappPassword $vfilerName >> ".$netappCliRawOutputFile;
        logMsg("Info: Running command: [$netappRunTimeLocation/$netappExpectScript Filer $netappIpAddress $netappUserName <password> $vfilerName >> $netappCliRawOutputFile].");

        # Run the command to get the Filer information.
        $exitCode = executeSystemCmd($cmd);
        if ($ERROR_SUCCESS != $exitCode)
        {
            # If there is some problem is collecting NetApp CLI data then it is an error.
            logMsg("Error: Failed to collect CLI output for NetApp Storage.");
    
            $retVal = $exitCode;
            goto EXIT;
        }
    }

    logMsg("Info: Reading NetApp raw CLI data.");

    open(CONF_DUMP, $netappCliRawOutputFile) or die logMsg("Error: Not able to open the file: [$netappCliRawOutputFile].");

    # Loop through the content of NetApp raw CLI output file.
    while (<CONF_DUMP>)
    {
        # Remove the new line character.
        $_ =~ s/\r\n//g;
        $_ =~ s/\n//g;

        # Look for "NetApp CLI" string in the file. This string is the start indicator of NetApp CLI execution.
        if ($_ =~ /^NetApp CLI: (.+)$/)
        {
            # Replace " " (space) in the command name with "_" as Java property file doesn't except spaces in the name part of name-value pair.
            my $command = $1;
            $command =~ s/ /_/g;

            if($command =~ /^license$/)
            {
               $licenseFound = 1;
            }

            # We need to append "\n \" so that Java code can get the value part of name-value pair which is spread over multiple lines.
            print $CLI_DATA_HANDLE "$command=\\n \\\n";

            $commandFound = 1;

            next;
        }

        # If we found "Output start" then it means that we will find the CLI output from the next line.
        if ((1 == $commandFound) && ($_ =~ /^Output start$/))
        {
            $cliOutputStart = 1;

            next;
        }

        # If we found "Output end" then it means that we reached at the end of CLI output.
        if ((1 == $cliOutputStart) && ($_ =~ /^Output end$/))
        {
            # Dump the last line of the CLI output without appending "\n \" at the end.
            print $CLI_DATA_HANDLE ":$previousLine\n";

            # Reset different variables.
            $cliOutputStart = 0;
            $commandFound = 0;
            $previousLine = "";
            $licenseFound = 0;

            next;
        }

        # We will come here when we have found NetApp CLI data in the file.
        if (1 == $cliOutputStart)
        {
            # If previous line is not empty then dump it along with appending "\n \" at the end.
            if ("" ne $previousLine)
            {
                my $stringToReplace = "=====";
                my $stringToReplaceWith = "*****";

                # Replace the string beginning with "=====" with "*****"
                $previousLine =~ s/$stringToReplace/$stringToReplaceWith/;

                # Remove the license key from the license output.
                if(($licenseFound == 1) && ($previousLine =~ /^(\s+)(.+)(\s+)([A-Z]{3,})(\s*)(.*)$/) && (index($previousLine, 'ENABLED') == -1))
                {
                    $previousLine =~ s/$4/<LICENSE>/;
                }

                print $CLI_DATA_HANDLE ":$previousLine\\n \\\n";
            }

            # Save the current line.
            $previousLine = $_;

            next;
        }
    }

EXIT:

    close (CONF_DUMP);

    return $retVal;
}

# This function is used to append ".DONE" to the final NetApp CLI output file name. This is done so that Java configuration thread use it only when
# all the resources data has been dumped in it.
#
# @affected global variables :
#  None
#
# @return :
#   0 if Success
#   1 if Failure
sub processConfFile
{
    # This variable is used to store the return code for this function.
    my $retVal = 0;

    # Generate the new file.
    my $newFileName = $finalCliOutputFile.".DONE";

    # Rename the file.
    my $status = rename($finalCliOutputFile, $newFileName);
    if (1 == $status)
    {
        logMsg("Info: Renamed [".basename($finalCliOutputFile)."] to [".basename($newFileName)."].");
    }
    else
    {
        logMsg("Error: Rename from [".basename($finalCliOutputFile)."] to [".basename($newFileName)."] failed.");
        $retVal = 1;
    }

    return $retVal;
}

# This function is used to log a message in the log file. It logs the message in the following format:
# [YYYY-MM-DD HH:MM:ss,SSS]  [NetApp-<IP Address>] <Log message>
#
# @param :
#   $_[0] - Log message.
#
# @affected global variables :
#   None
#
# @return :
#   Nothing
sub logMsg
{
    # Get the current date and time.
    my ($seconds, $microseconds) = gettimeofday;
    my $currentTime = strftime("%Y-%m-%d %H:%M:%S", localtime($seconds));

    # Get the date from the given date and time .
    my($date, $time) = split(' ', $currentTime, 2);

    # Set the name of log file.
    $logFile = $netappLogFolder."/netappProbe.".$date.".log";

    # Open the log file.
    open($LOG_HANDLE, ">>", $logFile) or die "Not able to open the file: [$logFile].";

    # Read input parameters.
    my $logMsg = $_[0];

    # For getting time in a given format.
    $currentTime = "[".$currentTime.",".$microseconds."]";

    # Construct the final log message.
    my $finalMsg = $currentTime." [NetApp-".$netappIpAddress."] ".$logMsg."\n";

    # Dump the final log message in log file.
    print $LOG_HANDLE $finalMsg;
}

# This function is used to update the NetApp CLI output file with the Metadata information about the Probe.
#
# @affected global variables :
#   None
#
# @return :
#   0 if Success
#   1 if Error
sub prepareProbeMetadata
{
    # It is used to store the current date and time (GMT) in "YYYYMMDD_HHMMSS" format.
    (my $second, my $minute, my $hour, my $dayOfMonth, my $month, my $yearOffset, my $dayOfWeek, my $dayOfYear, my $daylightSavings) = gmtime();
    my $year = 1900 + $yearOffset;
    my $currentTime = sprintf("%04d%02d%02d_%02d%02d%02d", $year, $month + 1, $dayOfMonth, $hour, $minute, $second);

    # Set the final NetApp CLI output file name.
    $finalCliOutputFile = $netappCliOutputPath."/".$marsSiteName."_".$netappIpAddress."_customdata_".$currentTime."_".$probeVersion.".txt";
    logMsg("Info: NetApp CLI output file name: [$finalCliOutputFile].");

    # Open the final NetApp CLI output file.
    open($CLI_DATA_HANDLE, ">", $finalCliOutputFile) or die logMsg("Error: Not able to open the file: [$finalCliOutputFile].");

    print $CLI_DATA_HANDLE "#%meta\n";
    print $CLI_DATA_HANDLE "#probe_id:$marsSiteName"."_"."$netappIpAddress\n";
    print $CLI_DATA_HANDLE "#probe_type:NetApp\n";
    print $CLI_DATA_HANDLE "#probe_site:$marsSiteName\n";
    print $CLI_DATA_HANDLE "#probe_server:$netappIpAddress\n";
    print $CLI_DATA_HANDLE "#probe_version:$probeVersion\n";
    print $CLI_DATA_HANDLE "#%meta\n";
    print $CLI_DATA_HANDLE "#t:$currentTime\n";

    return 0;
}

# This function is used to read the .properties file which corresponds to the input NetApp IP address. It reads the MARS Site name, NetApp Storage
# Password, and User Name from the .properties file.
#
# @affected global variables :
#   $marsSiteName
#   $netappPassword
#   $netappUserName
#
# @return :
#   0 if Success
#   1 if Error
sub readPropertyFile
{
    # This variable is used to store the return code for this function.
    my $retVal = 0;

    logMsg("Info: Reading file: $propFileName");

    open(my $PROP_FILE, $propFileName) or die logMsg("Error: Not able to open the file: [$propFileName].");

    # Loop through the content of .properties file.
    while (<$PROP_FILE>)
    {
        # Remove the new line character.
        chomp $_;

        # Look for MARS Site name.
        if ($_ =~ /^SiteName=(\S+)(\s*)$/)
        {
            $marsSiteName = $1;
            logMsg("Info: MARS Site name: [$marsSiteName]");
        }
        # Look for encrypted NetApp Storage Password.
        elsif ($_ =~ /^Password=(\S+)(\s*)$/)
        {
            $netappPassword = $1;
        }
        elsif ($_ =~ /^Username=(\S+)(\s*)$/)
        {
            $netappUserName = $1;
            logMsg("Info: Input User Name = [$netappUserName]");
        }
        else
        {
            # Current line is of no use. So ignore it.
        }
    }

    if ("" eq $marsSiteName)
    {
        logMsg("Error: MARS Site name is not present in the properties file.");
        $retVal = 1;

        goto EXIT;
    }

EXIT:

    close($PROP_FILE);

    return $retVal;
}

# This function is used to get the decrypted password corresponding to the password in the properties file.
#
# @affected global variables :
#   $netappPassword
#
# @return :
#   0 if Success
#   1 if Error
sub getDecryptedPassword
{
    # This variable is used to store the return code for this function.
    my $retVal = 0;

    # This variable is used to store the url to decrypt NetApp Probe password.
    my $netappURL = "";

    # This variable stores the data stored in probe instance property file.
    my $data = $netappPassword;

    # This variable stores appId.
    my $appId = "_netapphnasmgr!23_NetApp";

    # Retry count
    my $retryCount = 2;

    # Output from jar file 
    my $output = "unchanged";
    
    # Execute the Java Jar file here
    my $jar_file = "/usr/local/megha/lib/common/java/commonUtil.jar";
    my $action = "--action";
    my $actionValue = "DecryptData";
    my $appsId = "--appID";
    my $arg1 = $appId;
    my $Ddata = "--encData";
    my $arg2 = $data;


    # Collect output, if "error" --> then we failed to get it decrypted
    #                    "some base64 string" ---> process further
    #                    "null, empty" cases will be handled at jar level, only two outputs will be allowed            
    
    $output = `java -jar $jar_file $action $actionValue $appsId $arg1 $Ddata $arg2`;

    my $request_succeeded = 0;

while (!$request_succeeded && $retryCount > 0) {
    
    $output = `java -jar $jar_file $action $actionValue $appsId $arg1 $Ddata $arg2`;

    # check if request succeeded
    if (index($output, "error") != -1 || (!defined $output || $output eq "") || $output eq "unchanged") {
        # Error found in output
        logMsg("Error :Retry after 60 seconds.");
        $retryCount--;
        sleep(60);
        
    } else {
        # The string "error" was not found in $myOutput, so decode its contents
        $request_succeeded = 1;
        logMsg("Info :Value found in output");
        $netappPassword = "'".decode_base64($output)."'";
        last;
    }
}

if ($request_succeeded) {
    return 0;
} else {
    return 1;
}  
    
}

# This function is used to search all files present in $propFilePath folder and find the correct .properties file corresponding to input NetApp IP
# address.
#
# @param :
#   None
#
# @affected global variables :
#   None
#
# @return :
#   0 if found correct .properties file
#   1 if not found correct .properties file
sub getPropertyFileName
{
    # It is used to check whether we have found the correct .properties file for the NetApp Storage.
    my $foundFile = 0;

    # Get the list of all .properties files.
    getPropertyFileArray();

    # Check each .properties file in the folder.
    foreach my $propFileNameInList (@files)
    {
        logMsg("Info: Found properties file: [$propFileNameInList].");

        # Extract the file name from the complete file path.
        my $fileName = basename($propFileNameInList);

        # Check whether we are looking at correct .properties file.
        if ($fileName =~ /^NetApp_(\S+)_(\S+).properties$/)
        {
            # Match the given IP Address to find the correct .properties file because there could be multiple NetApp properties files present in the
            # folder.
            if ($netappIpAddress =~ $2)
            {
                logMsg("Info: Found correct properties file: [$fileName].");
                $propFileName = $propFileNameInList;

                $foundFile = 1;
                last;
            }
        }
    }

    # If we found the correct .properties file then return with Success.
    if (1 == $foundFile)
    {
        return 0;
    }
    else
    {
        return 1;
    }
}

# This function is used to select all properties files present in "$propFilePath" folder corresponding to input NetApp IP address.
#
# @param :
#   None
#
# @affected global variables :
#   @files
#
# @return :
#   None
sub getPropertyFileArray
{
    # Opening the directory from where we need to select the properties file.
    opendir(DIR_HANDLE, $propFilePath) or die logMsg("Error: Not able to open the folder: [$propFilePath].");

    while (defined (my $file = readdir DIR_HANDLE))
    {
        # Select only properties file from the given folder and push it to the array of files.
        if ($file =~ m!(.properties$)!)
        {
            push(@files, $propFilePath."/".$file);
        }
    }
}

# This function is used to read "NetApp runtime location" and "server web protcol" variable. It read the properties files in following sequence:
# app.properties
# netapp_default.properties
#
# @param :
#   None
#
# @affected global variables :
#   None
#
# @return :
#   0 if found correct information.
#   1 if not found correct information.
sub readAppPropertyFile
{
    # This variable is used to store the return value of function call.
    my $retVal = 0;

    # This array stores the list of property files.
    my @propertyFileArray = ();

    my $defaultPropertyFile = $propFilePath."/netapp_default.properties";

    # Put app.properties file at beginning of array since we need to search the property name in the given order.
    push(@propertyFileArray, $appPropertiesFilePath);

    # Put netapp_default.properties file in the array.
    push(@propertyFileArray, $defaultPropertyFile);

    foreach my $propertyFileName (@propertyFileArray)
    {
        # Opening current file.
        open(my $PROP_FILE, $propertyFileName) or die logMsg("Error: Not able to open the file: [$propertyFileName].");

        # Loop through the content of file.
        while (<$PROP_FILE>)
        {
            # Remove the new line character.
            $_ =~ s/\r\n//g;
            $_ =~ s/\n//g;

            # Look for "NetApp.runtime.location" property.
            if ($_ =~ /^(\s*)NetApp.runtime.location=(\s*)(.+)(\s*)$/)
            {
                $netappRunTimeLocation = $3;
            }
            # Look for "SERVER.WEB.PROTOCOL" property.
            elsif ($_ =~ /^(\s*)SERVER.WEB.PROTOCOL=(\s*)(.+)(\s*)$/)
            {
                $netappServerMode = $3;
            }
            else
            {
                # Current line is of no use. So ignore it.
            }
        }

        close($PROP_FILE);
    }
 
    # If we are not able to find needed property then it is an error.
    if ("" eq $netappRunTimeLocation)
    {
        logMsg("Error: NetApp.runtime.location property not found in netapp_default.properties file.");
        $retVal = 1;

        goto EXIT;
    }

    # If we are not able to find needed property then it is an error.
    if ("" eq $netappServerMode)
    {
        logMsg("Error: SERVER.WEB.PROTOCOL property not found in app.properties file.");
        $retVal = 1;

        goto EXIT;
    }

EXIT:

    return $retVal;
}

# This function is used to de-initialize the environment.
#
# @affected global variables :
#   $CONF_HANDLE
#
# @return :
#   None
sub deInit
{
    if (0 != $CLI_DATA_HANDLE)
    {
        close($CLI_DATA_HANDLE);
    }

    if (0 != $LOG_HANDLE)
    {
        close($LOG_HANDLE);
    }
}

# This function is used to find the list of filers present on the NetApp Storage.
#
# @param :
#   $_[0] - Array of filer name.
#
# @return :
#   0 if found at least one filer.
#   1 if not able to find any filer.
sub getFilerList
{
    # This variable is used to store the return value of function call.
    my $retVal = 0;

    # This variable is used to store the exit status of "system" command.
    my $exitCode = 0;

    # This variable is used to check whether we are able to find at least one filer on the NetApp Storage.
    my $foundFiler = 0;

    # This variable is used to store the name of file where name of NetApp vfiler name will get dumped.
    my $vfilerListFile = "vfilerListOutput.dump";

    my @inputFilerList = $_[0];

    # Construct the complete name of NetApp CLI raw output file where name of NetApp vfiler name will get dumped.
    my $netappTempCliOutputFile = $netappCliOutputPath."/".$vfilerListFile;
    
    # This variable is used to store the command to get the NetApp raw CLI data.
    my $cmd = $netappRunTimeLocation."/".$netappExpectScript." FilerList $netappIpAddress $netappUserName $netappPassword > ".$netappTempCliOutputFile;
    logMsg("Info: Running command: [$netappRunTimeLocation/$netappExpectScript FilerList $netappIpAddress $netappUserName <password> > $netappTempCliOutputFile].");

    # Run the command to get NetApp vFiler list.
    $exitCode = executeSystemCmd($cmd);
    if ($ERROR_SUCCESS != $exitCode)
    {
        # If there is some problem is collecting NetApp CLI data then it is an error.
        logMsg("Error: Failed to collect CLI output for NetApp Storage.");

        $retVal = $exitCode;
        goto EXIT;
    }

    open(CONF_DUMP, $netappTempCliOutputFile) or die logMsg("Error: Not able to open the file: [$netappTempCliOutputFile].");

    # This variable is used to indicate that we have reached at the output line from where vFiler information start.
    my $dataStart = 0;

    # Loop through the content of "vfilerListOutput.dump" file.
    while (<CONF_DUMP>)
    {
        # Remove the new line character.
        $_ =~ s/\r\n//g;
        $_ =~ s/\n//g;
        logMsg("Info: Filer line: [$_].");

        # If following condition is true then filer data will start from the next line.
        if ($_ =~ /^Output start$/)
        {
            $dataStart = 1;
            next;
        }

        # If following condition is true then it is end of the filer data.
        if ((1 == $dataStart) && ($_ =~ /^Output end$/))
        {
            last;
        }

        # If we haven't reached at the beginning of filer data then skip the current line.
        if (1 != $dataStart)
        {
            next;
        }

        # Look for string which contain filer name on NetApp storage.
        if ($_ =~ /^(\S+)(\s+)(\S+)$/)
        {
            logMsg("Info: Found Filer: [$1].");
            push(@{$_[0]}, $1);

            $foundFiler = 1;
        }
    }

    # If we are not able to find any filer then it is an error.
    if (0 == $foundFiler)
    {
        logMsg("Error: No vFiler found on the NetApp server.");
        $retVal = 1;
    }

EXIT:

    close (CONF_DUMP);

    return $retVal;
}

# This function is used to dump CIFS raw data in a file which Java code will read to find the CIFS configuration data.
#
# @param :
#   None
#
# @return :
#   0 if found at least one filer.
#   1 if not able to find any filer.
sub dumpCifsData
{
    # This variable is used to store the return value of function call.
    my $retVal = 0;

    # This variable is used to store the exit status of "system" command.
    my $exitCode = 0;

    # This variable is used to store the name of file which is used to dump the raw CIFS output data.
    my $cifsRawDataFile = "cifsRawData.dump";

    # This array is used to store the name of filers present on the NetApp Storage.
    my @filerList = ();

    # Construct the complete name of NetApp CLI raw output file.
    my $netappCifsRawOutputFile = $netappCliOutputPath."/".$cifsRawDataFile;

    # Delete the file (if exist) which used to capture the output of "Store" command.
    if (-e $netappCifsRawOutputFile)
    {
        if (0 == unlink($netappCifsRawOutputFile))
        {
            logMsg("Error: Not able to delete file: [$netappCifsRawOutputFile]. Error: [$!].");
        }
        else
        {
            logMsg("Info: File deleted: [$netappCifsRawOutputFile].");
        }
    }

    # Get the list of filers present on the NetApp Storage.
    $retVal = getFilerList(\@filerList);
    if (0 != $retVal)
    {
        logMsg("Warning: Call to getFilerList() failed.");

        # This variable is used to store the command to get the CIFS Share data specific to NetApp Storage.
        my $cmd = $netappRunTimeLocation."/".$netappExpectScript." CIFSDEFAULT $netappIpAddress $netappUserName $netappPassword >> ".$netappCifsRawOutputFile;
        logMsg("Info: Running command: [$netappRunTimeLocation/$netappExpectScript CIFSDEFAULT $netappIpAddress $netappUserName <password> >> $netappCifsRawOutputFile].");

        # Run the command to get the NetApp CIFS raw data.
        $exitCode = executeSystemCmd($cmd);
        if ($ERROR_SUCCESS != $exitCode)
        {
            # If there is some problem is collecting NetApp CLI data then it is an error.
            logMsg("Error: Failed to collect CLI output for NetApp Storage.");
        }

        $retVal = $exitCode;
    }
    else
    {
        # Loop through each filer name.
        foreach my $vfilerName (@filerList)
        {
            logMsg("Info: vFiler name: [$vfilerName].");
    
            # This variable is used to store the command to get the CIFS Share data specific to NetApp Storage.
            my $cmd = $netappRunTimeLocation."/".$netappExpectScript." CIFS $netappIpAddress $netappUserName $netappPassword $vfilerName >> ".$netappCifsRawOutputFile;
            logMsg("Info: Running command: [$netappRunTimeLocation/$netappExpectScript CIFS $netappIpAddress $netappUserName <password> $vfilerName >> $netappCifsRawOutputFile].");
    
            # Run the command to get the NetApp CIFS raw data.
            $exitCode = executeSystemCmd($cmd);
            if ($ERROR_SUCCESS != $exitCode)
            {
                # If there is some problem is collecting NetApp CLI data then it is an error.
                logMsg("Error: Failed to collect CLI output for NetApp Storage.");
    
                $retVal = $exitCode;
            }
        }
    }

EXIT:

    return $retVal;
}

# This function is used to dump Port raw data in a file which Java code will read to find the Port configuration data.
#
# @param :
#   None
#
# @return :
#   0 if found port data.
#   1 if not able to find port data.
sub dumpPortData
{
    # This variable is used to store the return value of function call.
    my $retVal = 0;

    # This variable is used to store the name of file which is used to dump the raw Port output data.
    my $portRawDataFile = "portRawData.dump";

    # Construct the complete name of NetApp CLI raw output file.
    my $netappPortRawOutputFile = $netappCliOutputPath."/".$portRawDataFile;

    # Delete the file (if exist) which used to capture the output of "Store" command.
    if (-e $netappPortRawOutputFile)
    {
        if (0 == unlink($netappPortRawOutputFile))
        {
            logMsg("Error: Not able to delete file: [$netappPortRawOutputFile]. Error: [$!].");
        }
        else
        {
            logMsg("Info: File deleted: [$netappPortRawOutputFile].");
        }
    }

    # This variable is used to store the command to get the Port data specific to NetApp Storage.
    my $cmd = $netappRunTimeLocation."/".$netappExpectScript." Port $netappIpAddress $netappUserName $netappPassword >> ".$netappPortRawOutputFile;
    logMsg("Info: Running command: [$netappRunTimeLocation/$netappExpectScript Port $netappIpAddress $netappUserName <password> >> $netappPortRawOutputFile].");

    # Run the command to get the NetApp Port raw data.
    $retVal = executeSystemCmd($cmd);
    if ($ERROR_SUCCESS != $retVal)
    {
        # If there is some problem is collecting NetApp CLI data then it is an error.
        logMsg("Error: Failed to collect CLI output for NetApp Storage.");
    }

    return $retVal;
}

# This function is used to run a command through system() call. system() call somehow returns error code differently by shifting the exit code to 8
# bits left side. So this function also shifted back the exit code to 8 bits right side and return it.
#
# @affected global variables :
#   None
#
# @return :
#   Exit code shifted to 8 bits right.
sub executeSystemCmd
{
    my $tempExitCode = system($_[0]);

    my $exitCode = $tempExitCode >> 8;

    return $exitCode;
}
# ----------------------------------------------------------------------------------------------------------------------------------------------------