# Copyright 2008-2016 Cumulus Systems Incorporated.
# All Rights Reserved.

# This file contains the code to collect the performance data for the VNX Block Storage whose instance property file path passed as argument to the
# script.
#
# @param : Property File Path - Complete path of the probe instance specific property file.
#
# @return : In case of Success, returns 0. In case of Failure, returns 1.
#
# Sample command: perl vnxBlockPerfDataGenerator.pl <Property File Path> <Time Stamp>

#!/usr/bin/perl -w

# It is used for strict compilation.
use strict;

# It is used to get base file name from the complete file name.
use File::Basename;

# It is used to get the current date and time.
use Time::HiRes qw(gettimeofday);
use POSIX "strftime";

# This is used to create folder recursively.
use File::Path qw(mkpath);

# This is used to move folders.
use File::Copy;

# It is used to indicate the path where common module is placed.
use lib "/usr/local/megha/lib/common/perl";

# It is required for logging information.
use commonLogModule;

# It is used for reading property files.
use commonPropFileModule;

# It is required for getting decrypted password, and for executing commands.
use commonPerlUtil;

# ----------------------------------------------------------------------------------------------------------------------------------------------------
# GLOBAL Variables
# ----------------------------------------------------------------------------------------------------------------------------------------------------

# Application home directory
use constant APP_HOME => "/usr/local/megha";

# This variable stores the probe type.
my $probeType = "VNXBlock";

# This variable stores naviseccli path.
my $naviseccliPath = "";

# This variable stores time out value for 1 command. As per Navicli document default will be 600 seconds.
my $vnxCmdTimeout = 600;

# This variable stores user scope.
my $scope = "0";

# This variable is used to store the path of probe properties files.
my $propFilePath = "";

# This variable stores commonPropFileModule instance.
my $propFileUtilObj = "";

# This variable stores commonPerlUtil instance.
my $perlUtilObj = "";

# This variable stores commonLogModule instance.
my $logObj = "";

# This variable is used to store the return status of function calls.
my $status = 0;

# This variable is used to store the probe name of the VNX Block Storage Probe.
my $serialNumber = "";

# This variable is used to store the user name of the VNX Block Storage.
my $username = "";

# This variable is used to store password corresponding to $username.
my $password = "";

# This variable is used to store the Site name.
my $siteName = "";

# This variable is used to store IP Address of VNX Block Storage's processor A.
my $processorA_IP = "";

# This variable is used to store IP Address of VNX Block Storage's processor B.
my $processorB_IP = "";

# This variable is used to store the location of VNX Block Probe's runtime scripts.
my $vnxBlockRunTimeLocation = "";

# This variable is used to store the path of particular VNX Block Probe's folder.
my $vnxBlockProbeFolder = "";

# This variable is used to store the path of raw/perf directory inside particular VNX Block Probe folder.
my $vnxRawPerfFolder = "";

# This variable is used to store the path of folder where temporary perf data is dumped for a particular VNX Block Probe.
my $tempPerfDumpPath = "";

# This variable stores the current time in %Y%m%d_%H%M%S format.
my $currentTime = "";

# This variable stores processor's IP's.
my @processors = ();

# This variable stores the map of sub command to its output file name.
my %commandToOutputFileMap = ('getcontrol' => 'getcontrol.xml',
                              'cache -fast -info -perfdata' => 'cache.xml',
                              'lun -list -perfData' => 'lun.xml',
                              'getlun -all' => 'getlun.xml',
                              'getdisk -all' => 'getdisk.xml',
                              'port -list -sp -all' => 'port.xml');

# ----------------------------------------------------------------------------------------------------------------------------------------------------
# Main function
# ----------------------------------------------------------------------------------------------------------------------------------------------------

# Initialize the environment.
$status = init();
if (0 != $status) {
    $logObj->error("Call to init() failed.");
    goto EXIT;
}

# Get All Raw performance data.
$status = getRawData();
if (0 != $status) {
    $logObj->error("Call to getRawData() failed.");
    goto EXIT;
}

$logObj->info("Performance data collection done. [$currentTime]");

EXIT:

exit $status;

# ----------------------------------------------------------------------------------------------------------------------------------------------------
# Sub-routines
# ----------------------------------------------------------------------------------------------------------------------------------------------------

# This function is used to initialize the environment to run the Perl script correctly.
#
# @return :
#   0 if Success
#   1 if Error
sub init {
    # This variable is used to store the return value of function calls.
    my $retVal = 0;

    # This variable is used to store the base folder path on the probe.
    my $baseFolder = APP_HOME;

    # This variable stores Log folder path.
    my $logFolder = $baseFolder."/logs";

    # If log folder does not exist then create it.
    if (! (-e $logFolder)) {
        mkpath($logFolder);
    }

    # Get commonLogModule instance.
    $logObj = commonLogModule->getInstance($logFolder, $probeType, $serialNumber);

    # If number of command line arguments are not correct then quit.
    if (2 != ($#ARGV + 1)) {
       $logObj->error("Number of arguments to vnxBlockPerfDataGenerator.pl are not correct. Input argument count = ".($#ARGV + 1));
       $retVal = 1;

       goto EXIT;
    }

    # Get the VNX Block probe property file path from command line argument.
    $propFilePath = $ARGV[0];
    $logObj->info("Properties file path: [$propFilePath].");

    $currentTime = $ARGV[1];
    $logObj->info("Collecting data for time : [$currentTime].");

    # Read the instance properties file corresponding to VNX Block Probe.
    # NOTE :- We cannot use common prop module for this because we don't have probeId for probe yet.
    # We have to read probe Id(IP address) from probe
    # instance property file first.
    $status = readPropertyFile();
    if (0 != $status) {
        $logObj->error("Call to readPropertyFile() failed.");
        $retVal = 1;

        goto EXIT;
    }

    # Load the commonPropFileModule object for this probe
    $propFileUtilObj = commonPropFileModule->getInstance($baseFolder, $probeType, $serialNumber, $logObj);

    # Update the final log message information with populated VNXBlock Probe id.
    $logObj->setProbeId($serialNumber);

    # Get commonPerlUtil instance.
    $perlUtilObj = commonPerlUtil->getInstance($baseFolder, $probeType, $serialNumber, $logObj);
    
    # Decrypt password
    $status = decryptPassword();
    if (0 != $status) {
        $logObj->error("Call to decryptPassword() failed.");
        $retVal = 1;
    
        goto EXIT;
    }   

    # Get the runtime location.
    $status = $propFileUtilObj->getProbeProperty("runtime.location", $vnxBlockRunTimeLocation);
    if (0 != $status) {
        $logObj->error("Call to getProbeProperty() failed for runtime.location property.");
        $retVal = 1;

        goto EXIT;
    }
    $logObj->info("Probe runtime location : [$vnxBlockRunTimeLocation].");

    # Get the runtime location.
    $status = $propFileUtilObj->getProbeProperty("naviseccli.location", $naviseccliPath);
    if (0 != $status) {
        $logObj->error("Call to getProbeProperty() failed for naviseccli.location property.");
        $retVal = 1;

        goto EXIT;
    }
    $logObj->info("naviseccli path location : [$naviseccliPath].");
	
	# Get command time out for perf command.
    $status = $propFileUtilObj->getProbeProperty("perf.vnx.cmd.timeout.sec", $vnxCmdTimeout);
    if (0 != $status) {
        $logObj->warn("Call to getProbeProperty() failed for perf.vnx.cmd.timeout.sec property.");
		$status = 0;
    }
    $logObj->info("Performance command time out : [$vnxCmdTimeout] Sec.");

    # This variable is used to store the path of Probe folder in db/probe.
    $vnxBlockProbeFolder = $baseFolder."/db/probe/".$siteName."/".$probeType."_".$serialNumber;

    $logObj->info("Creating folder: [$vnxBlockProbeFolder].");

    # Create the folder for this VNX Probe in the probe folder.
    unless (defined eval {mkpath($vnxBlockProbeFolder)}) {
        $logObj->error("Not able to create folder: [$vnxBlockProbeFolder].");
        $retVal = 1;

        goto EXIT;
    }

    # This variable is used to store the path of folder where final Raw Perf files will be moved.
    $vnxRawPerfFolder = $vnxBlockProbeFolder."/raw/perf";

    $logObj->info("Creating folder: [$vnxRawPerfFolder]");

    # Create the folder for the final Perf files.
    unless (defined eval {mkpath($vnxRawPerfFolder)}) {
        $logObj->error("Not able to create folder: [$vnxRawPerfFolder]");
        $retVal = 1;

        goto EXIT;
    }
    
    # Set the path where raw perf data gets dumped temporarily. When all raw perf data is collected,
    # we will append ".DONE" at the end of directory name.
    $tempPerfDumpPath = $vnxRawPerfFolder."/Perf_".$currentTime;

    $logObj->info("Creating temp Perf data path: [$tempPerfDumpPath]");

    # Create the folder for temporary perf data.
    unless (defined eval {mkpath($tempPerfDumpPath)}) {
        $logObj->error("Not able to create folder: [$tempPerfDumpPath]");
        $retVal = 1;

        goto EXIT;
    }

EXIT:

    return $retVal;
}

# This function is used to decrypt the password.
#
# @return :
#   0 if Success
#   1 if Error
sub decryptPassword {
    # This variable stores maximum retry count for decrypting VNX password.
    my $retryCount = 3;
    
    $status = 0;

    # Decrypt VNX password.
    while (0 < $retryCount) {
        $status = $perlUtilObj->getDataValue($password, $password);

        if (0 == $status) {
            # In case of successful execution, exit from while loop.

            last;
        } elsif (1 == $retryCount) {
            # If call invocation fails after all retries then it is an error. So exit from the script.

            $logObj->error("Call to getDataValue() failed.");
            goto EXIT;
        }

        # Try calling getDataValue() after some delay.
        $logObj->warn("Call to getDataValue() failed. Will retry after 60 seconds.");
        sleep(60);

        $retryCount--;
    }
    
EXIT :

    return $status;
}

# This function is used to collect all VNX Block Raw perf data from VNX Block Storage.
#
# @return :
#   0 if Success
#   1 if Error
sub getRawData {

    # This variable stores return status of this function.
    my $retVal = 0;
	
	# This variable stores if data is collected for other than controller resources e.g disk,lun,...
    my $dataCollected = "false";

    # Get raw data from each processor.
    foreach my $processor (@processors) {

        # This variable stores common part of command.
        # /opt/Navisphere/bin/naviseccli -h 172.17.248.234 -User sysadmin -Password sysadmin -Scope 0 getall -storagepool
        my $baseCmd = "$naviseccliPath -h $processor -User $username -Password $password -Scope $scope -xml";

        # This variable stores common part of command to log.
        my $cmdToLog = "$naviseccliPath -h $processor -User $username -Password <Password> -Scope $scope -xml";

        # This variable stores final command.
        my $command = "";

        # This variable stores final command to log.
        my $commandToLog = "";
			
		# This variable stores if controller is working or not.
		my $isWorkingController = "false";
		
		#Execute processor command first to collect data. While collecting we will decide working controller.
		foreach my $switch (sort keys %commandToOutputFileMap) {
		
			if ( $switch ne "getcontrol" ) {
				next;
			}
			
            my $outputFileName = $commandToOutputFileMap{$switch};

            $command = $baseCmd ." -t " .$vnxCmdTimeout. " ". $switch . " > " . $tempPerfDumpPath . "/" . $processor . "_" .$outputFileName;
            $commandToLog = $cmdToLog ." -t " .$vnxCmdTimeout. " ". $switch . " > " . $tempPerfDumpPath . "/" . $processor . "_" . $outputFileName;

            $logObj->info("Executing command: [$commandToLog].");

            $retVal = $perlUtilObj->executeSystemCmd($command);

            if (0 != $retVal) {
                $logObj->warn("Failed to execute command: [$commandToLog].");

                my $outputFilePath = $tempPerfDumpPath . "/" . $processor . "_" .$outputFileName;
                my $errMsg;
				open(my $FH, '<', $outputFilePath) or die $!;
                {
					local $/;
				    $errMsg = <$FH>;
				}
				close($FH);
				$logObj->error("[$errMsg]");
				$logObj->info("Removing file: [$outputFilePath]");
				$command = "rm -f $outputFilePath";
				$retVal = $perlUtilObj->executeSystemCmd($command);
                if(0 != $retVal) {
					$logObj->info("Failed to remove file $outputFilePath");
                }

                $retVal = 0;
            } else {
				$isWorkingController = "true";
			}
        }
		
		#Skip failed processor.
		if (($isWorkingController eq "false") || ($dataCollected eq "true")) {
			next;
		}

		#Execute remaining data collection commands with working controller.
        foreach my $switch (sort keys %commandToOutputFileMap) {
		
			if ($switch eq "getcontrol") {
				next;
			}
			
            my $outputFileName = $commandToOutputFileMap{$switch};

            $command = $baseCmd ." ". $switch . " > " . $tempPerfDumpPath . "/" . $processor . "_" .$outputFileName;
            $commandToLog = $cmdToLog ." ". $switch . " > " . $tempPerfDumpPath . "/" . $processor . "_" . $outputFileName;

            $logObj->info("Executing command: [$commandToLog].");

            $retVal = $perlUtilObj->executeSystemCmd($command);

            if (0 != $retVal) {
                $logObj->warn("Failed to execute command: [$commandToLog].");
                $retVal = 0;
            } else {
				$dataCollected = "true";
			}
        }
    }

    # Rename raw data folder to *.DONE
    my $command = "mv '$tempPerfDumpPath' '$tempPerfDumpPath'.DONE";
    $logObj->info("Executing command: [$command]");

    $retVal = $perlUtilObj->executeSystemCmd($command);

    if (0 != $retVal) {
        $logObj->error("Unable to move folder [$tempPerfDumpPath] to [$tempPerfDumpPath.DONE]");
        $retVal = 1;

        goto EXIT;
    }

EXIT :

    return $retVal;
}

# This function is used to read the input probe instance properties file and set the global variables corresponding to 
# the different properties.
#
# @return :
#   0 if Success
#   1 if Error
sub readPropertyFile {
    # This variable is used to store the return code for this function.
    my $retVal = 0;

    $logObj->info("Reading file: $propFilePath");

    open(PROP_FILE, $propFilePath) or die $logObj->error("Not able to open the file: [$propFilePath].");

    # Loop through the content of .properties file.
    while (<PROP_FILE>) {
        # Remove the new line character.
        chomp;

        if (/^\s*SerialNumber\s*=\s*(\S+)\s*$/i) {
            # Look for SerialNumber.

            $serialNumber = $1;
            $serialNumber =~s/[\r\n]+//g;
        }

        if (/^\s*UserName\s*=\s*(\S+)\s*$/i) {
            # Look for Username.

            $username = $1;
            $username =~s/[\r\n]+//g;

            # Some special characters [like $] will break the flow.
            $username = "'$username'";
        } elsif (/^\s*Password\s*=\s*(\S+)\s*$/i) {
            # Look for Password.

            $password = $1;
            $password =~s/[\r\n]+//g;

            # Some special characters [like $] will break the flow.
            $password = "'$password'";
        } elsif (/^\s*ProcessorA_IP\s*=\s*(\S+)\s*$/i) {
            # Look for ProcessorA_IP.

            $processorA_IP = $1;
            $processorA_IP =~s/[\r\n]+//g;
        } elsif (/^\s*ProcessorB_IP\s*=\s*(\S+)\s*$/i) {
            # Look for ProcessorB_IP.

            $processorB_IP = $1;
            $processorB_IP =~s/[\r\n]+//g;
        } elsif (/^\s*SiteName\s*=\s*(\S+)\s*$/i) {
            # Look for sitename.

            $siteName = $1;
            $siteName =~s/[\r\n]+//g;
        } elsif (/^\s*scope\s*=\s*(\S+)\s*$/i) {
            # Look for scope.
            $scope = $1;
            $scope =~s/[\r\n]+//g;
         }
	 else {
            # Current line is of no use. So ignore it.
        }
    }

    if ("" eq $serialNumber) {
        $logObj->error("Probe name not present in the properties file.");
        $retVal = 1;

        goto EXIT;
    }
    $logObj->info("Found User-Id: [$serialNumber] in the properties file.");

    if ("" eq $username) {
        $logObj->error("User-Id not present in the properties file.");
        $retVal = 1;

        goto EXIT;
    }
    $logObj->info("Found User-Id: [$username] in the properties file.");

    if ("" eq $password) {
        $logObj->error("Password not present in the properties file.");
        $retVal = 1;

        goto EXIT;
    }

    if ("" eq $processorA_IP) {
        $logObj->error("IP address of Processor A not present in the properties file.");
        $retVal = 1;

        goto EXIT;
    }
    $logObj->info("Found VNX Processor A IP Address: [$processorA_IP] in the properties file.");

    if ("" eq $processorB_IP) {
        $logObj->error("IP address of Processor B not present in the properties file.");
        $retVal = 1;

        goto EXIT;
    }
    $logObj->info("Found VNX Processor B IP Address: [$processorB_IP] in the properties file.");

    # Add processor ip's to list
    push(@processors, $processorA_IP);
    push(@processors, $processorB_IP);

    if ("" eq $siteName) {
        $logObj->error("Sitename is not present in the properties file.");
        $retVal = 1;

        goto EXIT;
    }
    $logObj->info("Found Site name: [$siteName] in the properties file.");

EXIT:

    close(PROP_FILE);

    return $retVal;
}
