# Copyright 2008-2016 Cumulus Systems Incorporated.
# All Rights Reserved.

# This file contains the code to process the raw Configuration data for the VNX Block Storage and dump it in final  format.
#
# @param : <Instance property file path>  - Instance property file path.
#
# @return : In case of Success, returns 0. In case of Failure, returns 1.
#
# Sample command: perl vnxBlockRawConfDataProcessor.pl <Property File Path>

#!/usr/bin/perl -w

# It is used for strict compilation.
use strict;

# It is used to get base file name from the complete file name.
use File::Basename;

# This is used to create folder recursively.
use File::Path qw(mkpath);

# This is used to move folder.
use File::Copy;

# It is used to indicate the path where common module is placed.
use lib "/usr/local/megha/lib/common/perl";

# It is required for logging information.
use commonLogModule;

# It is used for parsing raw configuration data.
use commonDataParsingUtil;

# It is used for getting probe Configuration meta data.
use commonProbeConfUtil;

# It is required for executing commands.
use commonPerlUtil;

# ----------------------------------------------------------------------------------------------------------------------------------------------------
# GLOBAL Variables
# ----------------------------------------------------------------------------------------------------------------------------------------------------
# This variable stores the probe type.
my $probeType = "VNXBlock";

# This variable is used to store the return status of function calls.
my $status = 0;

# This variable is used to store the base folder path on the  probe.
my $baseFolder = "/usr/local/megha";

# This variable stores VNX Block Storage Name.
my $vnxBlockSystemName = "";

# This variable is used to store the VNX Block Storage Probe Name.
my $vnxBlockSerialNumber = "";

# This variable is used to store the  Site name.
my $siteName = "";

# This variable is used to store the Configuration data start time.
my $startTime = "";


# This variable is used to store the Configuration data local start time.
my $localStartTime = "";

# This variable is used to store the storage block size in bytes.
my $blockSizeInBytes = 512;

# This variable is used to store the path of probe properties files.
my $propFilePath = "";

# This variable is used to store the name of Raw configuration data file.
my $rawConfFile = "";

# This variable is used to store the name of the final file in  format.
my $finalConfFileName = "";

# This variable holds the directory path of probe instance home directory.
my $probeInstanceDirectoryPath = "";

# This variable holds the directory path of raw directory.
my $rawDirectoryPath = "";

# This variable holds the directory path of raw conf directory.
my $rawConfDirectoryPath = "";

# This variable holds the directory path of current raw data directory.
my $rawDataDirectoryPath = "";

# This variable holds the directory path of raw failed data directory.
my $rawFailedDirectoryPath = "";

# This variable holds the directory path of raw backup data directory.
my $rawBackupDirectoryPath = "";

# This variable holds path of MARS db/backup folder.
my $dbBackupDirectoryPath = "";

# This variable is used to store the file handle of  configuration file.
my $CONF_HANDLE;

# This variable stores commonLogModule instance.
my $logObj = "";

# This variable stores commonDataParsingUtil instance.
my $dataParsingObj = "";

# This variable stores commonPerlUtil instance.
my $perlUtilObj = "";

# This variable stores real time enabled status value.
my $realTimeEnabled = "";

# This variable stores hostname of probe appliance.
my $probeHost = "";

# This variable stores version of probe.
my $probeVersion = "";

# This variable stores Storage Processor IP address to Storage processor Id Map.
my %ipToIdMap = ();

# This variable stores physical port to SP port mapping
my %portToSpPortMap = ();

# This variable stores lun to storage group mapping
my %lunToSgId = ();

# This variable stores host wwn to storage group mapping
my %hostWwnToSg = ();

# Used to store host wwn info until all Sg entries are read and can be dumped in file.
# Key is "Sg Id" and Value is various attribute of that Sg.
my %hostWwnInfoHash = ();

# ----------------------------------------------------------------------------------------------------------------------------------------------------
# Main function
# ----------------------------------------------------------------------------------------------------------------------------------------------------
# Initialize the environment.
$status = init();
if (0 != $status) {
    $logObj->error("Call to init() failed.");
    goto EXIT;
}

# List raw conf data Folders and process them.
$status = getRawConfFolders();
if (0 != $status){
    $logObj->error("Call to getRawConfFolders() failed.");
    goto EXIT;
}

# De-init the environment.
deInit();

EXIT:

exit $status;

# ----------------------------------------------------------------------------------------------------------------------------------------------------
# Sub-routines
# ----------------------------------------------------------------------------------------------------------------------------------------------------

# This function is used to initialize the environment to run the Perl script correctly.
#
# @return :
#   0 if Success
#   1 if Error
sub init {
    # This variable is used to store the return value of function calls.
    my $retVal = 0;

    # This variable stores Log folder path.
    my $logFolder = $baseFolder."/logs";

    # If log folder does not exist then create it.
    if (! (-e $logFolder)) {
        mkpath($logFolder);
    }

    # Get commonLogModule instance.
    $logObj = commonLogModule->getInstance($logFolder, $probeType, $vnxBlockSerialNumber);

    # If number of command line arguments are not correct then quit.
    if (1 != ($#ARGV + 1)) {
       $logObj->error("Number of arguments to vnxBlockConfDataGenerator.pl are not correct. Input argument count = ".($#ARGV + 1));
       $retVal = 1;

       goto EXIT;
    }

    # Get the VNX Block probe property file path from command line argument.
    $propFilePath = $ARGV[0];
    $logObj->info("Properties file path: [$propFilePath].");

    # Read the instance properties file corresponding to VNX Block Probe.
    $status = readPropertyFile();
    if (0 != $status) {
        $logObj->error("Call to readPropertyFile() failed.");
        $retVal = 1;

        goto EXIT;
    }

    # Update the final log message information with populated VNXBlock probe id.
    $logObj->setProbeId($vnxBlockSerialNumber);


    # Get commonDataParsingUtil instance.
    $dataParsingObj = commonDataParsingUtil->getInstance($baseFolder, $probeType, $vnxBlockSerialNumber, $logObj);

    # This variable holds path of top directory where all the files/directory related work performed for this probe instance.
    $probeInstanceDirectoryPath = $baseFolder."/db/probe/$siteName/VNXBlock_".$vnxBlockSerialNumber;

    $logObj->info("Raw directory path: [$probeInstanceDirectoryPath].");
    # This variable is used to store the path of Probe folder in db/probe.

    my $confFolder = $probeInstanceDirectoryPath . "/conf";

    $logObj->info("Creating conf folder: [$confFolder].");

    # Create the conf folder for this VNX Probe in the probe folder.
    unless (defined eval {mkpath($confFolder)}) {
        $logObj->error("Not able to create conf folder: [$confFolder].");
        $retVal = 1;

        goto EXIT;
    }

    # This variable holds path of directory where raw data parsing, backup and failure related task performed.
    $rawDirectoryPath = $probeInstanceDirectoryPath."/raw";

    $logObj->info("Raw directory path: [$rawDirectoryPath].");

    # This variable holds path of directory where failed raw data will be stored.
    $rawFailedDirectoryPath = $rawDirectoryPath . "/failed";

    $logObj->info("Raw failed directory path: [$rawFailedDirectoryPath].");

    # Create the failed folder for this VNX Probe in the probe folder.
    unless (defined eval {mkpath($rawFailedDirectoryPath)}) {
        $logObj->error("Not able to create raw failed data folder: [$rawFailedDirectoryPath].");
        $retVal = 1;

        goto EXIT;
    }

    # This variable holds path of directory where backup raw data will be stored.
    $rawBackupDirectoryPath = $rawDirectoryPath . "/backup";

    $logObj->info("Raw failed directory path: [$rawBackupDirectoryPath].");

    # Create the backup folder for this VNX Probe in the probe folder.
    unless (defined eval {mkpath($rawBackupDirectoryPath)}) {
        $logObj->error("Not able to create raw failed data folder: [$rawBackupDirectoryPath].");
        $retVal = 1;

        goto EXIT;
    }
    
    # This variable holds path of MARS db/backup folder.
    $dbBackupDirectoryPath = $baseFolder . "/db/backup";
    
    # Create DB backup folder for MARS
    unless (defined eval {mkpath($dbBackupDirectoryPath)}) {
        $logObj->error("Not able to create db backup folder: [$dbBackupDirectoryPath].");
        $retVal = 1;

        goto EXIT;
    }

    # This variable holds path of directory where conf raw data will be stored.
    $rawConfDirectoryPath = $rawDirectoryPath . "/conf";

    $logObj->info("Raw failed directory path: [$rawConfDirectoryPath].");

    # Create the conf folder for this VNX Probe in the probe folder.
    unless (defined eval {mkpath($rawConfDirectoryPath)}) {
        $logObj->error("Not able to create raw conf data folder: [$rawConfDirectoryPath].");
        $retVal = 1;

        goto EXIT;
    }

    # Get commonPerlUtil instance.
    $perlUtilObj = commonPerlUtil->getInstance($baseFolder, $probeType, $vnxBlockSerialNumber, $logObj);

    # Get commonPropFileModule instance.
    my $propFileObj = commonPropFileModule->getInstance($baseFolder, $probeType, $vnxBlockSerialNumber, $logObj);

    # Get the collection interval.
    $status = $propFileObj->getProbeProperty("block.size.bytes", $blockSizeInBytes);
    if (0 != $status) {
        $logObj->error("Call to getProbeProperty() failed for [block.size.bytes].");
        $retVal = 1;

        goto EXIT;
    }

    $logObj->info("Block size in bytes: [$blockSizeInBytes]");

	# Get probe host value.
	$status = $propFileObj->getAppProperty("probe.host", $probeHost);

	if (0 != $status) {
		$probeHost = "";
    }
	
	# Get probe version value.
	$status = $propFileObj->getAppProperty("app.version", $probeVersion);

	if (0 != $status) {
		$probeVersion = "10";
    }
	
	# Get real time collection enabled value.
	$status = $propFileObj->getProbeProperty("RealtimeCollection", $realTimeEnabled);

	if (0 != $status) {
		$realTimeEnabled = "false";
    }

	$logObj->info("Real time collection enabled: [$realTimeEnabled]");

EXIT:

    return $retVal;
}

# This function parse raw data directories and call dumpProcessedData function on each directory.
#
# @return :
#         0 : In case of Success.
#         1 : In case of Failure.
sub getRawConfFolders {
    # This variable is used to store the return value of function calls.
    my $retVal = 0;

    my $folderNamePattern = "Conf_*.DONE";

    # This variable stores raw conf data folder path pattern.
    my $folderPathPattern = $rawConfDirectoryPath . "/" . $folderNamePattern;

    # This variable stores list of raw data files to process.
    my @rawDataFileList = glob($folderPathPattern);

    if (-1 == $#rawDataFileList) {
        $logObj->warn("No raw folders to process : [$folderPathPattern].");
        
        goto EXIT;
    }

    foreach my $folder (@rawDataFileList) {
        $logObj->info("Processing raw data directory : [$folder].");

        if (!(-d $folder)) {
            $logObj->error("Raw data directory not exists: [$folder].");
            $retVal = 1;

            goto EXIT;
        }

        # Set current raw data directory.
        $rawDataDirectoryPath = $folder;

        my $folderBasename = basename($folder);

        $folderBasename =~ s/Conf_|\.DONE//g;

		($startTime = $folderBasename) =~ s/\.\d*_\d*//g;

		($localStartTime = $folderBasename) =~ s/\d*_\d*\.//g;

        $logObj->info("Time stamp: [$startTime] and local Timestamp : [$localStartTime].");

        # Set the final Configuration file name.
        $finalConfFileName = $siteName."_".$vnxBlockSerialNumber."_snapshot_".$startTime.".txt";
        $logObj->info("VNXBlock Conf File name: [$finalConfFileName].");

        my $tempConfFilePath = $rawDirectoryPath."/conf/".$finalConfFileName;
        $logObj->info("VNXBlock temporary Conf File path: [$tempConfFilePath].");

        # Open the final  configuration file.
        open($CONF_HANDLE, ">", $tempConfFilePath) or die $logObj->error("Not able to open the file: [$tempConfFilePath].");

        # Process raw conf data and dump it in final  format.
        $retVal = dumpProcessedData();
        if (0 != $retVal){
            $logObj->error("Call to dumpProcessedData() failed.");
            goto EXIT;
        }
    }

EXIT:

    return $retVal;
}

# This function is used to process VNXBlock Storage raw conf data and dump it in final  format.
#
# @return :
#   0 if Success
#   1 if Error
sub dumpProcessedData {
    # This variable is used to store the return value of function calls.
    my $retVal = 0;

    # This variable stores VNXBlock configuration metadata.
    my $metaData = "";
	
	# It stores optional meta data information.
	my %optionalMetaDataHash = ();

	$optionalMetaDataHash{'realTime'} = $realTimeEnabled;
	$optionalMetaDataHash{'probe_host'} = $probeHost;

    # Get commonProbeConfUtil instance.
    my $probeConfUtilObj = commonProbeConfUtil->getInstance($baseFolder, $probeType, $vnxBlockSerialNumber, $logObj);

    # This function dumps the probe metadata in final mars configuration file.
    $probeConfUtilObj->getFormattedProbeConfMetadata($siteName, $startTime, $probeVersion, $metaData, \%optionalMetaDataHash);

    # Dump the Metadata for the VNXBlock probe in the configuration file.
    print $CONF_HANDLE $metaData;
	
	$logObj->info("[vnxBlockRawConfDataProcessor] Dumping custom attributes...");

	# It stores custom attribute information.
	my %customAttrHash = $probeConfUtilObj->getCustomAttrValues($siteName);


	#Dump custom resource
	print($CONF_HANDLE "R:__probe#" . $siteName . "_" . $vnxBlockSerialNumber . "\n");
	print($CONF_HANDLE "a:__name=" . $siteName . "_" . $vnxBlockSerialNumber . "\n");
	
	foreach my $key (keys(%customAttrHash)) {
		print($CONF_HANDLE "a:" . $key . "=" . $customAttrHash{$key} . "\n");
	}

	print($CONF_HANDLE "\n");
	$logObj->info("[vnxBlockRawConfDataProcessor] Finished dumping custom attributes.");

    # This function dumps the processed System resource conf data in final mars format.
    $retVal = processSystemData();
    if (0 != $retVal){
        $logObj->error("Call to processSystemData() failed.");
        $retVal = 1;

        goto EXIT;
    }

    print $CONF_HANDLE "\n";

    # This function dumps the processed Storage Processor resource conf data in final mars format.
    $retVal = processStorageProcessorData();
    if (0 != $retVal){
        $logObj->error("Call to processStorageProcessorData() failed.");
        $retVal = 1;

        goto EXIT;
    }

    print $CONF_HANDLE "\n";

    # This function dumps the processed Pool resource conf data in final mars format.
    $retVal = processPoolData();

    # If processPoolData function call fails there may be no RAID Groups configured hence continue.
    if (0 != $retVal){
        $logObj->warn("Call to processPoolData() failed.");
    }

    print $CONF_HANDLE "\n";

    # This function dumps the processed RAID Group resource conf data in final mars format.
    $retVal = processRaidGroupData();

    # If processRaidGroupData function call fails there may be no RAID Groups configured hence continue.
    if (0 != $retVal){
        $logObj->warn("Call to processRaidGroupData() failed.");
    }

    print $CONF_HANDLE "\n";

    # This function dumps the processed port Wwn resource conf data in final mars format.
    $retVal = processPortWwnData();
    
    # If processPortWwnData function call fails there may be no port wwn configured hence continue.
    if (0 != $retVal){
        $logObj->warn("Call to processPortWwnData() failed.");
    }

    print $CONF_HANDLE "\n";

    # This function dumps the processed Storage Group resource conf data in final mars format.
    $retVal = processSgData();

    # If processSgData function call fails there may be no Storage Group configured hence continue.
    if (0 != $retVal){
        $logObj->warn("Call to processSgData() failed.");
    }
    
    print $CONF_HANDLE "\n";

    # This function dumps the processed host port resource conf data in final mars format.
    $retVal = processHbaPortData();
    
    # If processPortWwnData function call fails there may be no port wwn configured hence continue.
    if (0 != $retVal){
        $logObj->warn("Call to processHbaPortData() failed.");
    }

    print $CONF_HANDLE "\n";

    # This function dumps the processed Port resource conf data in final mars format.
    $retVal = processPortData();
    if (0 != $retVal){
        $logObj->error("Call to processPortData() failed.");
        $retVal = 1;

        goto EXIT;
    }

    print $CONF_HANDLE "\n";

    # This function dumps the processed Lun resource conf data in final mars format.
    $retVal = processLunData();

    # If processLunData function call fails there may be no luns configured hence continue.
    if (0 != $retVal){
        $logObj->warn("Call to processLunData() failed.");
    }

    print $CONF_HANDLE "\n";

    # This function dumps the processed Disk resource conf data in final mars format.
    $retVal = processDiskData();
    if (0 != $retVal) {
        $logObj->warn("Call to processDiskData() failed.");
        $retVal = 1;

        goto EXIT;
    }

    # This variable is used to store the command to Zip the final Mars Conf file.
    my $command = "mv '$rawDirectoryPath/conf/$finalConfFileName' '$probeInstanceDirectoryPath/conf/$finalConfFileName'.DONE";
    $logObj->info("Running command: [$command]");

    # Run the command to Zip the final Mars Conf file.
    my $exitCode = $perlUtilObj->executeSystemCmd($command);
    if (0 != $exitCode) {
        $logObj->error("Failed to move the final Mars Conf file.");

        # Move raw data into failed directory.
        $command = "mv $rawDataDirectoryPath $rawFailedDirectoryPath";
        $logObj->info("Running command: [$command]");

        # Run the command to Zip the final Mars Conf file.
        $exitCode = $perlUtilObj->executeSystemCmd($command);
        if (0 != $exitCode) {
            $logObj->error("Failed to move the raw data into failed directory.");
        }

        $retVal = 1;

        goto EXIT;
    }

EXIT:

    if (0 != $retVal) {
        $logObj->error("Failed to parse raw file, moving raw file to failed directory.");

        # Move raw data into failed directory.
        $command = "mv $rawDataDirectoryPath $rawFailedDirectoryPath";
        $logObj->info("Running command: [$command]");

        # Run the command to Zip the final Mars Conf file.
        $exitCode = $perlUtilObj->executeSystemCmd($command);
        if (0 != $exitCode) {
            $logObj->error("Failed to move the raw data into failed directory.");
        }

        return $retVal;
    }

    # Move raw data into backup directory.
    $logObj->info("Moving raw file to backup directory.");
    # Zip the files.
    my $zipFileName = $dbBackupDirectoryPath. "/Conf" .$startTime. "_" . $siteName."_".$vnxBlockSerialNumber."_VNXBlock_Raw.zip" ;
    $logObj->info("Creating zip  $zipFileName");
    
   $exitCode = $perlUtilObj->executeSystemCmd("zip -9 -r $zipFileName $rawDataDirectoryPath"); 
    # Check zipping operation status.
   if ($exitCode) {
     $logObj->error("Zipping raw conf file present in folder [$rawDataDirectoryPath] to [$zipFileName]  failed.");
    }
    $logObj->info("Created zip  $zipFileName");
    
    $logObj->info("Removing  $rawDataDirectoryPath");
    $command = "rm -r $rawDataDirectoryPath";
    $logObj->info("Running command: [$command]");

    $exitCode = $perlUtilObj->executeSystemCmd($command);
    if (0 != $exitCode) {
         $logObj->error("Failed to remove directory $rawDataDirectoryPath ");
    }
   
    return $retVal;
}

# This function dumps the processed Storage System resource conf data in final mars format.
#
# @return :
#   0 if Success
#   1 if Error
sub processSystemData {
    # This variable is used to store the return value of function calls.
    my $retVal = 0;

    # This variable tracks start of Storage System Info data in raw conf file.
    my $isSystemtDataStarted = 0;

    # Used to store Storage System info until all Storage System entries are read and can be dumped in file.
    # Key is "Storage System Serial number" and Value is various attribute of that Storage.
    my %systemIdInfoHash = ();

    # This variable is used to store list of storage processors raw files.
    my %fileToIdMap = ();

    # Following variables are used to store Storage system attributes.
    my $agentRevision = "";
    my $revision = "";
    my $model = "";
    my $modelType = "";
    my $promRevision = "";
    my $cabinet = "";

    if (parseRawFiles('host', \%fileToIdMap)) {

        foreach my $rawConfFile ( keys %fileToIdMap ) {
            $logObj->info("Reading raw configuration data for Storage System resource.");

            open(CONF_DUMP, $rawConfFile) or die $logObj->error("Not able to open the file: [$rawConfFile].");

            while (<CONF_DUMP>) {
                # Remove the new line character.
                chomp;

                # If this line found then it means Pool data starts from next line
                if (/^\s*Agent\s+Rev\s*:\s*(.+)\s*/i) {
                    $logObj->debug("Found VNX Block Storage System data start.");
                    $isSystemtDataStarted = 1;
                    $agentRevision = $1;

                    next;
                }

                # Storage System data not started yet. Skip this line
                if (!$isSystemtDataStarted) {
                    next;
                }

                # Read Actual data.
                if (/^\s*Revision\s*:\s*(.*)/i) {
                    # This should not be the case.
                    if (($revision ne "") && ($revision ne $1)) {
                        $logObj->warn("revision attribute of Storage System resource is not matching");
                    }
                    $revision = $1;
                } elsif (/^\s*Model\s*:\s*(.*)/i) {
                    # This should not be the case.
                    if (($model ne "") && ($model ne $1)) {
                        $logObj->warn("model attribute of Storage System resource is not matching");
                    }
                    $model = $1;
                } elsif (/^\s*Model\s+Type\s*:\s*(.*)/i) {
                    # This should not be the case.
                    if (($modelType ne "") && ($modelType ne $1)) {
                        $logObj->warn("modelType attribute of Storage System resource is not matching");
                    }
                    $modelType = $1;
                } elsif (/^\s*Prom\s+Rev\s*:\s*(.*)/i) {
                    # This should not be the case.
                    if (($promRevision ne "") && ($promRevision ne $1)) {
                        $logObj->warn("promRevision attribute of Storage System resource is not matching");
                    }
                    $promRevision = $1;
                } elsif (/^\s*Serial\s+No\s*:\s*(.*)/i) {
                    # This should not be the case.
                    if (($vnxBlockSystemName ne "") && ($vnxBlockSystemName ne $1)) {
                        $logObj->warn("vnxBlockSystemName attribute of Storage System resource is not matching");
                    }
                    $vnxBlockSystemName = $1;
                } elsif (/^\s*Cabinet\s*:\s*(.*)/i) {
                    # This should not be the case.
                    if (($cabinet ne "") && ($cabinet ne $1)) {
                        $logObj->warn("cabinet attribute of Storage System resource is not matching");
                    }
                    $cabinet = $1;
                }
            }

            close (CONF_DUMP);
        }

        if ($vnxBlockSystemName eq "") {
            $logObj->error("System name not found");

            $retVal = 1;
            goto EXIT;
        }

        print $CONF_HANDLE "\nR:vnxBlockSystem#$vnxBlockSystemName\n";

        print $CONF_HANDLE "a:name=$vnxBlockSystemName\n";
        print $CONF_HANDLE "a:agentRevision=$agentRevision\n";
        print $CONF_HANDLE "a:revision=$revision\n";
        print $CONF_HANDLE "a:model=$model\n";
        print $CONF_HANDLE "a:modelType=$modelType\n";
        print $CONF_HANDLE "a:promRevision=$promRevision\n";
        print $CONF_HANDLE "a:cabinet=$cabinet\n";
    } else {
        $logObj->warn("No raw files found for Storage System resource.");

        $retVal = 1;
        goto EXIT;
    }

EXIT:

    return $retVal;
}


# This function dumps the processed Storage Processor resource conf data in final mars format.
#
# @return :
#   0 if Success
#   1 if Error
sub processStorageProcessorData {
    # This variable is used to store the return value of function calls.
    my $retVal = 0;

    # This variable tracks start of Storage Processor Info data in raw conf file.
    my $isSpDataStarted = 0;

    # Temp variable for storing matched value from regex.
    my $propValue = "";

    # This variable stores Storage Processor'd Id.
    my $spId = "";

    # Used to store Storage Processor info until all Storage processor entries are read and can be dumped in file.
    # Key is "Processor Id" and Value is various attribute of that Processor.
    my %spIdInfoHash = ();

    # This variable is used to store list of storage processors raw files.
    my %fileToIpMap = ();

    # Get Storage Processor information from <SP_IP>_sp.txt file.
    if (parseRawFiles('sp', \%fileToIpMap)) {

        foreach my $rawConfFile ( keys %fileToIpMap ) {

            $logObj->info("Reading raw configuration data for Storage Processor resource.");

            open(CONF_DUMP, $rawConfFile) or die $logObj->error("Not able to open the file: [$rawConfFile].");

            while (<CONF_DUMP>) {
                # Remove the new line character.
                chomp;

                # If this line found then it means Storage Processor data starts from next line
                if (/^\s*Storage\s+Processor\s*:\s*(.+)\s*/i) {
                    $logObj->debug("Found VNX Block Storage Processor data start.");
                    $isSpDataStarted = 1;

                    $spId = $1;

                    # Replace spaces within empty.
                    $spId =~ s/\s+//g;

                    # Upper case all letters.
                    $spId = uc $spId;

                    # Create entry for this pool id.
                    $spIdInfoHash{$spId} = {name => "",
                                            ipAddress => "",
                                            subnetMask => "",
                                            gatewayAddress => "",
                                            linkStatus => "",
                                            capableSpeed => "",
                                            readCacheState => "",
                                            writeCacheState => "",
                                            systemBuffer => "",
                                            physicalMemSize => "",
                                            readCacheSize => "",
                                            writeCacheSizeInMB => ""};

                    $spIdInfoHash{$spId}{name} = $spId;

                    # Populate Storage Processor IP address to Storage Processor Id map.
                    my $processorIp = $fileToIpMap{$rawConfFile};
                    $ipToIdMap{$processorIp} = $spId;

                    next;
                }

                # Storage Processor data not started yet. Skip this line
                if (!$isSpDataStarted) {
                    next;
                }

                # Read Actual data.
                if (/^\s*Storage\s+Processor\s+IP\s+Address\s*:\s*(.*)/i) {
                    $spIdInfoHash{$spId}{ipAddress} = $1;
                } elsif (/^\s*Storage\s+Processor\s+Subnet\s+Mask\s*:\s*(.*)/i) {
                    $spIdInfoHash{$spId}{subnetMask} = $1;
                } elsif (/^\s*Storage\s+Processor\s+Gateway\s+Address\s*:\s*(.*)/i) {
                    $spIdInfoHash{$spId}{gatewayAddress} = $1;
                } elsif (/^\s*Link\s+Status\s*:\s*(.*)/i) {
                    $spIdInfoHash{$spId}{linkStatus} = $1;
                } elsif (/^\s*Current\s+Speed\s*:\s*(.*)/i) {
                    $spIdInfoHash{$spId}{capableSpeed} = $1;
                } elsif (/^\s*SP\s+Read\s+Cache\s+State\s*(.*)/i) {
                    $spIdInfoHash{$spId}{readCacheState} = $1;
                } elsif (/^\s*SP\s+Write\s+Cache\s+State\s*(.*)/i) {
                    $spIdInfoHash{$spId}{writeCacheState} = $1;
                }
            }

            close (CONF_DUMP);
        }

        # Check whether storage processor map contains the data.
        if (!%spIdInfoHash) {
            $logObj->error("Storage processor map is empty.");
            $retVal = 1;

            goto EXIT;
        }
    } else {
        $logObj->warn("No raw files found for Storage Processors resource.");
        $retVal = 1;

        goto EXIT;
    }

    # Reset
    $isSpDataStarted = 0;

    # Get Storage Processor information from <SP_IP>_cache.txt file.
    if (parseRawFiles('cache', \%fileToIpMap)) {

        foreach my $rawConfFile ( keys %fileToIpMap ) {

            $logObj->info("Reading raw configuration data for Storage Pool resource.");

            $spId = "";

            open(CONF_DUMP, $rawConfFile) or die $logObj->error("Not able to open the file: [$rawConfFile].");

            while (<CONF_DUMP>) {
                # Remove the new line character.
                chomp;

                # Read Actual data.
                if (/\s*System\s+Buffer\s*\((.+)\):\s*(\d*)/i) {
                    $spId = uc $1;

                    if (exists $spIdInfoHash{$spId}) {
                        $spIdInfoHash{$spId}{systemBuffer} = $2;
                    }
                } elsif (/\s*(.+)\s+Physical\s+Memory\s+Size\s*=\s*(\d*)/i) {
                    $spId = uc $1;

                    if (exists $spIdInfoHash{$spId}) {
                        $spIdInfoHash{$spId}{physicalMemSize} = $2;
                    }
                } elsif (/\s*(.+)\s+Physical\s+Memory\s+Size\s*\(MB\)\s*:\s*(\d*)/i) {
                    $spId = uc $1;

                    if (exists $spIdInfoHash{$spId}) {
                        $spIdInfoHash{$spId}{physicalMemSize} = $2;
                    }
                } elsif (/\s*(.+)\s+Read\s+Cache\s+Size\s*=\s*(\d*)/i) {
                    $spId = uc $1;

                    if (exists $spIdInfoHash{$spId}) {                    
                        $spIdInfoHash{$spId}{readCacheSize} = $2;
                    }
                } elsif (/\s*(.+)\s+Write\s+Cache\s+Size\s*=\s*(\d*)/i) {
                    $spId = uc $1;

                    if (exists $spIdInfoHash{$spId}) {
                        $spIdInfoHash{$spId}{writeCacheSizeInMB} = $2;
                    }
                }
            }

            close (CONF_DUMP);
        }
    } else {
        $logObj->warn("No raw files found for Storage Processors cache resource.");
        $retVal = 1;

        goto EXIT;
    }

    # Now iterate through hash and dump in  format.
    my $name = "";
    my $ipAddress = "";
    my $subnetMask = "";
    my $gatewayAddress = "";
    my $linkStatus = "";
    my $capableSpeed = "";
    my $readCacheState = "";
    my $writeCacheState = "";
    my $systemBuffer = "";
    my $physicalMemSize = "";
    my $readCacheSize = "";
    my $writeCacheSizeInMB = "";

    for my $id ( keys %spIdInfoHash ) {
        $name = $spIdInfoHash{$id}{name};
        $ipAddress = $spIdInfoHash{$id}{ipAddress};
        $subnetMask = $spIdInfoHash{$id}{subnetMask};
        $gatewayAddress = $spIdInfoHash{$id}{gatewayAddress};
        $linkStatus = $spIdInfoHash{$id}{linkStatus};
        $capableSpeed = $spIdInfoHash{$id}{capableSpeed};
        $readCacheState = $spIdInfoHash{$id}{readCacheState};
        $writeCacheState = $spIdInfoHash{$id}{writeCacheState};
        $systemBuffer = $spIdInfoHash{$id}{systemBuffer};
        $physicalMemSize = $spIdInfoHash{$id}{physicalMemSize};
        $readCacheSize = $spIdInfoHash{$id}{readCacheSize};
        $writeCacheSizeInMB = $spIdInfoHash{$id}{writeCacheSizeInMB};

        print $CONF_HANDLE "\nR:vnxBlockProcessor#$vnxBlockSystemName^$name\n";
        print $CONF_HANDLE "a:name=$vnxBlockSystemName^$name\n";
        print $CONF_HANDLE "a:ipAddress=$ipAddress\n";
        print $CONF_HANDLE "a:subnetMask=$subnetMask\n";
        print $CONF_HANDLE "a:gatewayAddress=$gatewayAddress\n";
        print $CONF_HANDLE "a:linkStatus=$linkStatus\n";
        print $CONF_HANDLE "a:capableSpeed=$capableSpeed\n";
        print $CONF_HANDLE "a:readCacheState=$readCacheState\n";
        print $CONF_HANDLE "a:writeCacheState=$writeCacheState\n";
        print $CONF_HANDLE "a:systemBuffer=$systemBuffer\n";
        print $CONF_HANDLE "a:physicalMemSize=$physicalMemSize\n";
        print $CONF_HANDLE "a:readCacheSize=$readCacheSize\n";
        print $CONF_HANDLE "a:writeCacheSizeInMB=$writeCacheSizeInMB\n";

        print $CONF_HANDLE "r:vnxBlockSystem#$vnxBlockSystemName\n";
    }

EXIT:

    return $retVal;
}

# This function dumps the processed Port resource conf data in final mars format.
#
# @return :
#   0 if Success
#   1 if Error
sub processPortData {
    # This variable is used to store the return value of function calls.
    my $retVal = 0;

    # This variable tracks start of Port Info data in raw conf file.
    my $isPortDataStarted = 0;

    # This variable stores port Id.
    my $portId = "";

    # This variable stores processor Id.
    my $spId = "";

    # This variable stores I/O module slot Id.
    my $moduleId = "";

    # Used to store Port info until all Port entries are read and can be dumped in file.
    # Key is "Port Id" and Value is various attribute of that Port.
    my %portIdInfoHash = ();

    # This variable is used to store list of storage processors raw files.
    my %fileToIpMap = ();

    # Get Storage Processor information from <SP_IP>_sp.txt file.
    if (parseRawFiles('port', \%fileToIpMap)) {

        foreach my $rawConfFile ( keys %fileToIpMap ) {

            $logObj->info("Reading raw configuration data for Port resource.");

            open(CONF_DUMP, $rawConfFile) or die $logObj->error("Not able to open the file: [$rawConfFile].");

            while (<CONF_DUMP>) {
                # Remove the new line character.
                chomp;

                # If this line found then it means Storage Processor data starts from next line
                if (/\s*Information\s*about\s*each\s*I\/O\s*module\(s\)\s*on\s*(.+)\s*:\s*/i) {
                    $spId = uc $1;
                    $logObj->debug("Found VNX Block Port data start.");
                    $isPortDataStarted = 1;

                    # Remove Spaces.
                    next;
                }

                # Port data not started yet. Skip this line
                if (!$isPortDataStarted) {
                    next;
                }

                # Read Actual data.
                if (/^\s*I\/O\s*Module\s*Slot\s*:\s*(\d+)\s*/i) {
                    $moduleId = $1;
                } elsif (/\s*Physical\s*Port\s*ID\s*:\s*(\d+)/i) {
                    $portId = $spId . "^" . $moduleId . "-" . $1;
                    $portIdInfoHash{$portId}{name} = $portId;
                    $portIdInfoHash{$portId}{id} = $1;
                    $portIdInfoHash{$portId}{moduleId} = $moduleId;
                    $portIdInfoHash{$portId}{spId} = $spId;
                } elsif (/^\s*Port\s*Role\s*:\s*(.*)/i) {
                    $portIdInfoHash{$portId}{role} = $1;
                } elsif (/^\s*Port\s*Usage\s*:\s*(.*)/i) {
                    $portIdInfoHash{$portId}{usage} = $1;
                } elsif (/^\s*Port\s*Type\s*:\s*(.*)/i) {
                    $portIdInfoHash{$portId}{type} = $1;
                } elsif (/^\s*Port\s*State\s*:\s*(.*)/i) {
                    $portIdInfoHash{$portId}{state} = $1;
                } elsif (/^\s*Port\s*Substate\s*:\s*(.*)/i) {
                    $portIdInfoHash{$portId}{subState} = $1;
                } elsif (/^\s*Is\s*Persisted\s*:\s*(.*)/i) {
                    $portIdInfoHash{$portId}{isPersisted} = $1;
                }
            }

            close (CONF_DUMP);
        }

        # Check whether storage processor map contains the data.
        if (!%portIdInfoHash) {
            $logObj->error("Storage port map is empty.");
            $retVal = 1;

            goto EXIT;
        }
    } else {
        $logObj->warn("No raw files found for Port resource.");
        $retVal = 1;

        goto EXIT;
    }

    # Now iterate through hash and dump in  format.
    my $name = "";
    my $id = "";
    my $moduleId = "";
    my $role = "";
    my $usage = "";
    my $type = "";
    my $state = "";
    my $subState = "";
    my $isPersisted = "";

    for my $id ( keys %portIdInfoHash ) {
        $name = $portIdInfoHash{$id}{name};
        $portId = $portIdInfoHash{$id}{id};
        $moduleId = $portIdInfoHash{$id}{moduleId};
        $role = $portIdInfoHash{$id}{role};
        $usage = $portIdInfoHash{$id}{usage};
        $type = $portIdInfoHash{$id}{type};
        $state = $portIdInfoHash{$id}{state};
        $subState = $portIdInfoHash{$id}{subState};
        $isPersisted = $portIdInfoHash{$id}{isPersisted};

        print $CONF_HANDLE "\nR:vnxBlockPort#$vnxBlockSystemName"."^"."$name\n";
        print $CONF_HANDLE "a:name=$name\n";
        print $CONF_HANDLE "a:id=$portId\n";
        print $CONF_HANDLE "a:moduleId=$moduleId\n";
        print $CONF_HANDLE "a:role=$role\n";
        print $CONF_HANDLE "a:usage=$usage\n";
        print $CONF_HANDLE "a:type=$type\n";
        print $CONF_HANDLE "a:state=$state\n";
        print $CONF_HANDLE "a:subState=$subState\n";
        print $CONF_HANDLE "a:isPersisted=$isPersisted\n";

        my $spId = $portIdInfoHash{$id}{spId};

        print $CONF_HANDLE "r:vnxBlockSystem#$vnxBlockSystemName,vnxBlockProcessor#$vnxBlockSystemName^$spId";
		$logObj->debug("TestChecking Relation with sp for name=$name");
		my $wwn = $portToSpPortMap{$name}{portWwn};
		$logObj->debug("wwn=$wwn");
		if ($wwn !~ /^IQN/i && $wwn ne ""){
			my $formatedWwn =  substr $wwn,-23;
			print $CONF_HANDLE ",sp#$formatedWwn";
		}
		print $CONF_HANDLE "\n";

    }

EXIT:

    return $retVal;
}

# This function dumps the processed Disk resource conf data in final mars format.
#
# @return :
#   0 if Success
#   1 if Error
sub processDiskData {
    # This variable is used to store the return value of function calls.
    my $retVal = 0;

    # This variable tracks start of Disk Info data in raw conf file.
    my $isDiskDataStarted = 0;

    # This variable stores disk Id.
    my $diskId = "";

    # Used to store Disk info until all Disk entries are read and can be dumped in file.
    # Key is "Disk Id" and Value is various attribute of that Disk.
    my %diskIdInfoHash = ();

    # This variable is used to store list of storage processors raw files.
    my %fileToIpMap = ();

    # Get Storage Processor information from <SP_IP>_disk.txt file.
    if (parseRawFiles('disk', \%fileToIpMap)) {

        foreach my $rawConfFile ( keys %fileToIpMap ) {

            $logObj->info("Reading raw configuration data for Disk resource.");

            open(CONF_DUMP, $rawConfFile) or die $logObj->error("Not able to open the file: [$rawConfFile].");

            while (<CONF_DUMP>) {
                # Remove the new line character.
                chomp;

                # If this line found then it means Disk data starts from next line
                if (/\s*Bus\s*(\S+)\s*Enclosure\s*(\S+)\s*Disk\s*(\S+)\s*/i) {
                    $logObj->debug("Found VNX Block Disk data start.");
                    $isDiskDataStarted = 1;

                    $diskId = "$1-$2-$3";

                    # Create entry for this Disk id.
                    $diskIdInfoHash{$diskId} = {name => "",
                                                vendorId => "",
                                                productId => "",
                                                type => "",
                                                state => "",
                                                hotSpare => "",
                                                serialNumber => "",
                                                capacityInGB => "",
                                                driveType => "",
                                                currentSpeed => ""};

                    $diskIdInfoHash{$diskId}{name} = $vnxBlockSystemName . "^" . $diskId;

                    next;
                }

                # Disk data not started yet. Skip this line
                if (!$isDiskDataStarted) {
                    next;
                }

                # Read Actual data.
                if (/^\s*Vendor\s*Id\s*:\s*(.*)/i) {
                    $diskIdInfoHash{$diskId}{vendorId} = $1;
                } elsif (/^\s*Product\s*Id\s*:\s*(.*)/i) {
                    $diskIdInfoHash{$diskId}{productId} = $1;
                } elsif (/^\s*Drive\s*Type\s*:\s*(.*)/i) {
                    $diskIdInfoHash{$diskId}{driveType} = $1;
                } elsif (/^\s*Type\s*:\s*(.*)/i) {
                    $diskIdInfoHash{$diskId}{type} = $1;
                } elsif (/^\s*State\s*:\s*(.*)/i) {
                    $diskIdInfoHash{$diskId}{state} = $1;
                } elsif (/^\s*Hot\s*Spare\s*:\s*(.*)/i) {
                    $diskIdInfoHash{$diskId}{hotSpare} = $1;
                } elsif (/^\s*Serial\s*Number\s*:\s*(.*)/i) {
                    $diskIdInfoHash{$diskId}{serialNumber} = $1;
                } elsif (/^\s*Capacity\s*:\s*(.*)/i) {
                    $diskIdInfoHash{$diskId}{capacityInGB} = ($1/1024);
                } elsif (/^\s*Current\s*Speed\s*:\s*(\d+)/i) {
                    $diskIdInfoHash{$diskId}{currentSpeed} = $1;
                }
            }

            close (CONF_DUMP);
        }

        # Check whether storage processor map contains the data.
        if (!%diskIdInfoHash) {
            $logObj->error("Storage disk map is empty.");
            $retVal = 1;

            goto EXIT;
        }
    } else {
        $logObj->warn("No raw files found for Disk resource.");
    }

    # Now iterate through hash and dump in  format.
    my $name = "";
    my $vendorId = "";
    my $productId = "";
    my $type = "";
    my $state = "";
    my $hotSpare = "";
    my $serialNumber = "";
    my $capacityInGB = "";
    my $driveType = "";
    my $currentSpeed = "";

    for my $id ( keys %diskIdInfoHash ) {
        $name = $diskIdInfoHash{$id}{name};
        $vendorId = $diskIdInfoHash{$id}{vendorId};
        $productId = $diskIdInfoHash{$id}{productId};
        $type = $diskIdInfoHash{$id}{type};
        $state = $diskIdInfoHash{$id}{state};
        $hotSpare = $diskIdInfoHash{$id}{hotSpare};
        $serialNumber = $diskIdInfoHash{$id}{serialNumber};
        $capacityInGB = $diskIdInfoHash{$id}{capacityInGB};
        $driveType = $diskIdInfoHash{$id}{driveType};
        $currentSpeed = $diskIdInfoHash{$id}{currentSpeed};

        print $CONF_HANDLE "\nR:vnxBlockDisk#$name\n";
        print $CONF_HANDLE "a:name=$name\n";
        print $CONF_HANDLE "a:vendorId=$vendorId\n";
        print $CONF_HANDLE "a:productId=$productId\n";
        print $CONF_HANDLE "a:type=$type\n";
        print $CONF_HANDLE "a:state=$state\n";
        print $CONF_HANDLE "a:hotSpare=$hotSpare\n";
        print $CONF_HANDLE "a:serialNumber=$serialNumber\n";
        print $CONF_HANDLE "a:capacityInGB=$capacityInGB\n";
        print $CONF_HANDLE "a:driveType=$driveType\n";
        print $CONF_HANDLE "a:currentSpeed=$currentSpeed\n";

        print $CONF_HANDLE "r:vnxBlockSystem#$vnxBlockSystemName\n";
    }

EXIT:

    return $retVal;
}

# This function dumps the processed Lun resource conf data in final mars format.
#
# @return :
#   0 if Success
#   1 if Error
sub processLunData {
    # This variable is used to store the return value of function calls.
    my $retVal = 0;

    # This variable tracks start of Lun Info data in raw conf file.
    my $isLunDataStarted = 0;

    # This variable stores Lun Id.
    my $lunId = "";

    # Used to store Lun info until all Lun entries are read and can be dumped in file.
    # Key is "Lun Id" and Value is various attribute of that Lun.
    my %lunIdInfoHash = ();

    # This variable is used to store list of storage processors raw files.
    my %fileToIpMap = ();

    # Get Storage Processor information from <SP_IP>_lun.txt file.
    if (parseRawFiles('lun', \%fileToIpMap)) {

        foreach my $rawConfFile ( keys %fileToIpMap ) {

            $logObj->info("Reading raw configuration data for Lun resource.");

            open(CONF_DUMP, $rawConfFile) or die $logObj->error("Not able to open the file: [$rawConfFile].");

            while (<CONF_DUMP>) {
                # Remove the new line character.
                chomp;

                # If this line found then it means Storage Processor data starts from next line
                if (/\s*LOGICAL\s*UNIT\s*NUMBER\s*(\d+)\s*/i) {
                    $logObj->debug("Found VNX Block Lun data start.");
                    $lunId = $1;
                    $lunIdInfoHash{$lunId}{name} = $vnxBlockSystemName ."^". $lunId;
                    $isLunDataStarted = 1;

                    # Remove Spaces.
                    next;
                }

                # Lun data not started yet. Skip this line
                if (!$isLunDataStarted) {
                    next;
                }

                # Read Actual data.
                if (/^\s*RAID\s*Type\s*:\s*(.+)\s*/i) {
                    $lunIdInfoHash{$lunId}{raidType} = $1;
                } elsif (/^\s*State\s*:\s*(.+)/i) {
                    $lunIdInfoHash{$lunId}{state} = $1;
                } elsif (/^\s*LUN\s*Capacity\(Megabytes\)\s*:\s*(\d+)/i) {
                    my $inGb = ($1/1024);
                    $lunIdInfoHash{$lunId}{capacityInGB} = $inGb;
                } elsif (/^\s*Current\s*owner\s*:\s*(.*)/i) {
                    $lunIdInfoHash{$lunId}{owner} = $1;
                } elsif (/^\s*Is\s*Thin\s*LUN\s*:\s*(.*)/i) {
                    $lunIdInfoHash{$lunId}{isThinLUN} = $1;
                } elsif (/^\s*Name\s*(.*)/i) {
                    $lunIdInfoHash{$lunId}{lunName} = $1;
                }
            }

            close (CONF_DUMP);
        }

        # Check whether Storage Lun map contains the data.
        if (!%lunIdInfoHash) {
            $logObj->error("Storage lun map is empty.");
            $retVal = 1;

            goto EXIT;
        }
    } else {
        $logObj->warn("No raw files found for lun resource.");
        $retVal = 1;

        goto EXIT;
    }

    $isLunDataStarted = 0;
    $lunId = "";

    # Get Storage Processor information from <SP_IP>_lunlist.txt file.
    if (parseRawFiles('lunlist', \%fileToIpMap)) {

        foreach my $rawConfFile ( keys %fileToIpMap ) {

            $logObj->info("Reading raw configuration data for Lun resource.");

            open(CONF_DUMP, $rawConfFile) or die $logObj->error("Not able to open the file: [$rawConfFile].");

            while (<CONF_DUMP>) {
                # Remove the new line character.
                chomp;

                # If this line found then it means Storage Processor data starts from next line
                if (/\s*LOGICAL\s*UNIT\s*NUMBER\s*(\d+)\s*/i) {
                    $logObj->debug("Found VNX Block Lun data start.");
                    $lunId = $1;
                    $isLunDataStarted = 1;
                    # Remove Spaces.
                    next;
                }

                # Lun data not started yet. Skip this line
                if (!$isLunDataStarted) {
                    next;
                }

                # Read Actual data.
                if (/^\s*RAID\s*Type\s*:\s*(.+)\s*/i) {
                    $lunIdInfoHash{$lunId}{raidType} = uc $1;
                } elsif (/^\s*Consumed\s*Capacity\s*\(GBs\)\s*:\s*([0-9\.]*)/i) {
                    $lunIdInfoHash{$lunId}{usedCapacityInGB} = $1;
                }
            }

            close (CONF_DUMP);
        }

        # Check whether Storage Lun map contains the data.
        if (!%lunIdInfoHash) {
            $logObj->error("Storage lun map is empty.");
            $retVal = 1;

            goto EXIT;
        }
    } else {
        $logObj->warn("No raw files found for lun resource.");
        $retVal = 1;

        goto EXIT;
    }

    # Now iterate through hash and dump in  format.
    my $name = "";
    my $raidType = "";
    my $state = "";
    my $capacityInGB = "";
    my $owner = "";
    my $isThinLUN = "";
    my $usableCapacityInGB = 0;
	my $lunUsedCapacityInGB = "";

    for my $id ( keys %lunIdInfoHash ) {
        $name = $lunIdInfoHash{$id}{name};
        $raidType = $lunIdInfoHash{$id}{raidType};
        $state = $lunIdInfoHash{$id}{state};
        $capacityInGB = $lunIdInfoHash{$id}{capacityInGB};
        $owner = $lunIdInfoHash{$id}{owner};
        $isThinLUN = $lunIdInfoHash{$id}{isThinLUN};

		if ($isThinLUN =~ m/YES/) {
			$lunUsedCapacityInGB = $lunIdInfoHash{$id}{usedCapacityInGB};
		}

        print $CONF_HANDLE "\nR:vnxBlockLun#$name\n";
        print $CONF_HANDLE "a:name=".$lunIdInfoHash{$id}{lunName}."\n";
        print $CONF_HANDLE "a:raidType=$raidType\n";
        print $CONF_HANDLE "a:state=$state\n";
        print $CONF_HANDLE "a:capacityInGB=$capacityInGB\n";
        print $CONF_HANDLE "a:owner=$owner\n";
        print $CONF_HANDLE "a:isThinLUN=$isThinLUN\n";
		print $CONF_HANDLE "a:usedCapacityInGB=$lunUsedCapacityInGB\n";

        print $CONF_HANDLE "r:vnxBlockSystem#$vnxBlockSystemName\n";

        if (defined($capacityInGB)) {
            $usableCapacityInGB = $usableCapacityInGB + $capacityInGB;
        }
    }

    print $CONF_HANDLE "\nR:vnxBlockSystem#$vnxBlockSystemName\n";
    print $CONF_HANDLE "a:usableCapacityInGB=$usableCapacityInGB\n";

    my %sgCapacityMap = ();

    # Calculating total capacity of Storage group by sum of capacities of LUNs in it.
    for my $sgId ( keys %lunToSgId ) {
        for my $lunId ( keys %{$lunToSgId{$sgId}}) {
            if (defined($lunIdInfoHash{$lunId}{capacityInGB}) && ($lunIdInfoHash{$lunId}{capacityInGB} ne "")) {
                $capacityInGB = $lunIdInfoHash{$lunId}{capacityInGB};
                if (defined($sgCapacityMap{$sgId})) {
                    $sgCapacityMap{$sgId} = $sgCapacityMap{$sgId} + $capacityInGB;
                } else {
                    $sgCapacityMap{$sgId} = $capacityInGB;
                }
            }
        }
    }

    foreach my $sgId (keys %sgCapacityMap) {
        print $CONF_HANDLE "\nR:vnxBlockSg#$vnxBlockSystemName^$sgId\n";
        print $CONF_HANDLE "a:capacityInGB=".sprintf("%.2f", $sgCapacityMap{$sgId})."\n";
    }

EXIT:

    return $retVal;
}

# This function dumps the processed Pool resource conf data in final mars format.
#
# @return :
#   0 if Success
#   1 if Error
sub processPoolData {
    # This variable is used to store the return value of function calls.
    my $retVal = 0;

    # This variable tracks start of RAID Group Info data in raw conf file.
    my $isPoolDataStarted = 0;

    # Used to store Pool info until all Pool entries are read and can be dumped in file.
    # Key is "Pool Id" and Value is various attribute of that Pool.
    my %poolIdInfoHash = ();

    # This variable stores Pool Id.
    my $poolName = "";

    # This variable is used to store list of storage processors raw files.
    my %fileToIpMap = ();

    # Get Storage Processor information from <SP_IP>_storagepool.txt file.
    if (parseRawFiles('storagepool', \%fileToIpMap)) {

        foreach my $rawConfFile ( keys %fileToIpMap ) {

            $logObj->info("Reading raw configuration data for Pool resource.");

            open(CONF_DUMP, $rawConfFile) or die $logObj->error("Not able to open the file: [$rawConfFile].");

            while (<CONF_DUMP>) {
                # Remove the new line character.
                chomp;

                # If this line found then it means Disk data starts from next line
                if (/\s*Pool\s*Name\s*:\s*(.+)\s*/i) {
                    $logObj->debug("Found VNX Block Pool data start.");
                    $isPoolDataStarted = 1;

                    $poolName = $1;

                    # Create entry for this Disk id.
                    $poolIdInfoHash{$poolName} = {name => "",
                                                    id => "",
                                                    raidType => "",
                                                    fullThreshold => "",
                                                    diskType => "",
                                                    status => "",
                                                    state => "",
                                                    totalRawCapacityInGB => "",
                                                    totalUserCapacity => "",
                                                    totalUsedCapacity => "",
                                                    totalFreeCapacity => "",
                                                    totalSubscribedCapacity => ""};

                    $poolIdInfoHash{$poolName}{name} = $vnxBlockSystemName . "^" . $poolName;

                    next;
                }

                # Disk data not started yet. Skip this line
                if (!$isPoolDataStarted) {
                    next;
                }

                # Read Actual data.
                if (/^\s*Pool\s*ID\s*:\s*(.*)/i) {
                    $poolIdInfoHash{$poolName}{id} = $1;
                } elsif (/^\s*Raid\s*Type\s*:\s*(.*)\s*/i) {
                    $poolIdInfoHash{$poolName}{raidType} = $1;
                } elsif (/\s*Bus\s*(\S+)\s*Enclosure\s*(\S+)\s*Disk\s*(\S+)\s*/i) {
                    my $propValue = $poolIdInfoHash{$poolName}{disks};
                    if (!defined($propValue)) {
                        $poolIdInfoHash{$poolName}{disks} = "$1-$2-$3";
                    } else {
                        $poolIdInfoHash{$poolName}{disks} = $propValue . ",$1-$2-$3";
                    }
                } elsif (/^\s*LUNs\s*:\s*([\s*\d+,]+)/i) {
                    $poolIdInfoHash{$poolName}{luns} = $1;
                } elsif (/\s*Percent\s*Full\s*Threshold\s*:\s*(.*)/i) {
                    $poolIdInfoHash{$poolName}{fullThreshold} = $1;
                } elsif (/\s*Disk\s*Type\s*:\s*(.*)/i) {
                    $poolIdInfoHash{$poolName}{diskType} = $1;
                } elsif (/^\s*Status\s*:\s*(\w+)/i) {
                    $poolIdInfoHash{$poolName}{status} = $1;
                } elsif (/^\s*State\s*:\s*(.*)/i) {
                    $poolIdInfoHash{$poolName}{state} = $1;
                } elsif (/^\s*Raw\s*Capacity\s*\(GBs\)\s*:\s*(.*)/i) {
                    $poolIdInfoHash{$poolName}{totalRawCapacityInGB} = $1;
                } elsif (/^\s*User\s*Capacity\s*\(GBs\)\s*:\s*(.*)/i) {
                    $poolIdInfoHash{$poolName}{totalUserCapacity} = $1;
                } elsif (/^\s*Consumed\s*Capacity\s*\(GBs\)\s*:\s*(.*)/i) {
                    $poolIdInfoHash{$poolName}{totalUsedCapacity} = $1;
                } elsif (/^\s*Available\s*Capacity\s*\(GBs\)\s*:\s*(.*)/i) {
                    $poolIdInfoHash{$poolName}{totalFreeCapacity} = $1;
                } elsif (/^\s*Total\s*Subscribed\s*Capacity\s*\(GBs\)\s*:\s*(.*)/i) {
                    $poolIdInfoHash{$poolName}{totalSubscribedCapacity} = $1;
                }
            }

            close (CONF_DUMP);
        }

        # Check whether Storage Pool map contains the data.
        if (!%poolIdInfoHash) {
            $logObj->error("Storage Pool map is empty.");
            $retVal = 1;

            goto EXIT;
        }
    } else {
        $logObj->warn("No raw files found for Pool resource.");
        $retVal = 1;

        goto EXIT;
    }

    # Now iterate through hash and dump in  format.
    my $name = "";
    my $poolId = "";
    my $raidType = "";
    my $fullThreshold = "";
    my $diskType = "";
    my $status = "";
    my $state = "";
    my $totalRawCapacityInGB = "";
    my $totalUserCapacity = "";
    my $totalUsedCapacity = "";
    my $totalFreeCapacity = "";
    my $totalSubscribedCapacity = "";

    my $luns = "";
    my $disks = "";

    for my $id ( keys %poolIdInfoHash ) {
        $name = $poolIdInfoHash{$id}{name};
        $poolId = $poolIdInfoHash{$id}{id};
        $raidType = $poolIdInfoHash{$id}{raidType};
        $fullThreshold = $poolIdInfoHash{$id}{fullThreshold};
        $diskType = $poolIdInfoHash{$id}{diskType};
        $status = $poolIdInfoHash{$id}{status};
        $state = $poolIdInfoHash{$id}{state};
        $totalRawCapacityInGB = $poolIdInfoHash{$id}{totalRawCapacityInGB};
        $totalUserCapacity = $poolIdInfoHash{$id}{totalUserCapacity};
        $totalUsedCapacity = $poolIdInfoHash{$id}{totalUsedCapacity};
        $totalFreeCapacity = $poolIdInfoHash{$id}{totalFreeCapacity};
        $totalSubscribedCapacity = $poolIdInfoHash{$id}{totalSubscribedCapacity};

        print $CONF_HANDLE "\nR:vnxBlockPool#$name\n";
        print $CONF_HANDLE "a:name=$name\n";
        print $CONF_HANDLE "a:id=$poolId\n";
        print $CONF_HANDLE "a:raidType=$raidType\n";
        print $CONF_HANDLE "a:fullThreshold=$fullThreshold\n";
        print $CONF_HANDLE "a:diskType=$diskType\n";
        print $CONF_HANDLE "a:status=$status\n";
        print $CONF_HANDLE "a:state=$state\n";
        print $CONF_HANDLE "a:totalRawCapacityInGB=$totalRawCapacityInGB\n";
        print $CONF_HANDLE "a:totalUserCapacity=$totalUserCapacity\n";
        print $CONF_HANDLE "a:totalUsedCapacity=$totalUsedCapacity\n";
        print $CONF_HANDLE "a:totalFreeCapacity=$totalFreeCapacity\n";
        print $CONF_HANDLE "a:totalSubscribedCapacity=$totalSubscribedCapacity\n";

        $disks = $poolIdInfoHash{$id}{disks};

        my @diskList = split /,/, $disks;
        my $relations = "r:vnxBlockSystem#$vnxBlockSystemName";

        foreach my $disk (@diskList) {
            $relations = $relations . ",vnxBlockDisk#$vnxBlockSystemName^$disk";
        }

        $luns = $poolIdInfoHash{$id}{luns};
        $luns =~ s/,/ /g;

        my @lunList = split /\s+/, $luns;

        foreach my $lun (@lunList) {
            $relations = $relations . ",vnxBlockLun#$vnxBlockSystemName^$lun";
        }

        print $CONF_HANDLE "$relations\n";
    }

EXIT:

    return $retVal;
}

# This function dumps the processed RAID Group resource conf data in final mars format.
#
# @return :
#   0 if Success
#   1 if Error
sub processRaidGroupData {
    # This variable is used to store the return value of function calls.
    my $retVal = 0;

    # This variable tracks start of RAID Group Info data in raw conf file.
    my $isRaidGroupDataStarted = 0;

    # Used to store RAID Group info until all RAID Group entries are read and can be dumped in file.
    # Key is "RAID Group Id" and Value is various attribute of that RAID Group.
    my %raidGroupIdInfoHash = ();

    # This variable stores RAID Group Id.
    my $raidGroupId = "";

    # This variable stores RAID Group state scan is on or off.
    my $activateStateScan = 0;

    # This variable is used to store list of storage processors raw files.
    my %fileToIpMap = ();

    # Get Storage Processor information from <SP_IP>_rg.txt file.
    if (parseRawFiles('rg', \%fileToIpMap)) {

        foreach my $rawConfFile ( keys %fileToIpMap ) {

            $logObj->info("Reading raw configuration data for RAID Group resource.");

            open(CONF_DUMP, $rawConfFile) or die $logObj->error("Not able to open the file: [$rawConfFile].");

            while (<CONF_DUMP>) {
                # Remove the new line character.
                chomp;

                # If this line found then it means Disk data starts from next line
                if (/^\s*RaidGroup\s*ID\s*:\s*(\d+)\s*/i) {
                    $logObj->debug("Found VNX Block RAID Group data start.");
                    $isRaidGroupDataStarted = 1;

                    $raidGroupId = $1;

                    # Create entry for this Disk id.
                    $raidGroupIdInfoHash{$raidGroupId} = {name => "",
                                                          type => "",
                                                          state => "",
                                                          maxDiskCount => "",
                                                          maxLunCount => "",
                                                          totalRawCapacityInGB => "",
                                                          totalLogicalCapacity => "",
                                                          totalFreeCapacity => ""};

                    $raidGroupIdInfoHash{$raidGroupId}{name} = $vnxBlockSystemName . "^" . $raidGroupId;

                    # Do not scan state until first property for the RAID state found.
                    $activateStateScan = 0;

                    next;
                }

                # Disk data not started yet. Skip this line
                if (!$isRaidGroupDataStarted) {
                    next;
                }

                # Read Actual data.
                if (/^\s*RaidGroup\s*Type\s*:\s*(.*)/i) {
                    $raidGroupIdInfoHash{$raidGroupId}{type} = $1;
                } elsif (/^\s*List\s*of\s*luns\s*:\s*([\s*\d+]+)\s*/i) {
                    $raidGroupIdInfoHash{$raidGroupId}{luns} = $1;
                } elsif (/\s*Bus\s*(\S+)\s*Enclosure\s*(\S+)\s*Disk\s*(\S+)\s*/i) {
                    $activateStateScan = 0;
                    my $propValue = $raidGroupIdInfoHash{$raidGroupId}{disks};
                    if (!defined($propValue)) {
                        $raidGroupIdInfoHash{$raidGroupId}{disks} = "$1-$2-$3";
                    } else {
                        $raidGroupIdInfoHash{$raidGroupId}{disks} = $propValue . ",$1-$2-$3";
                    }
                } elsif (/^\s*RaidGroup\s*State\s*:\s*(.*)/i) {
                    $raidGroupIdInfoHash{$raidGroupId}{state} = $1;

                    # Activate state scan as values are on multiple lines.
                    $activateStateScan = 1;
                } elsif (/^\s*Max\s*Number\s*of\s*disks\s*:\s*(.*)/i) {
                    $raidGroupIdInfoHash{$raidGroupId}{maxDiskCount} = $1;
                } elsif (/^\s*Max\s*Number\s*of\s*luns\s*:\s*(.*)/i) {
                    $raidGroupIdInfoHash{$raidGroupId}{maxLunCount} = $1;
                } elsif (/^\s*Raw\s*Capacity\s*\(Blocks\)\s*:\s*(\d*)/i) {
                    $raidGroupIdInfoHash{$raidGroupId}{totalRawCapacityInGB} = ($1 * $blockSizeInBytes) / (1024 * 1024 * 1024);
                } elsif (/^\s*Logical\s*Capacity\s*\(Blocks\)\s*:\s*(.*)/i) {
                    $raidGroupIdInfoHash{$raidGroupId}{totalLogicalCapacity} = ($1 * $blockSizeInBytes) / (1024 * 1024 * 1024);;
                } elsif (/^\s*Free\s*Capacity\s*\(Blocks,non\-contiguous\)\s*:\s*(.*)/i) {
                    $raidGroupIdInfoHash{$raidGroupId}{totalFreeCapacity} = ($1 * $blockSizeInBytes) / (1024 * 1024 * 1024);;
                } elsif ($activateStateScan) {
                    # Scan until next attribute that is 'List of disks' found.
                    # Trim the current line;
                    s/^\s+|\s+$//g;
                    my $line = $_;
                    $raidGroupIdInfoHash{$raidGroupId}{state} = $raidGroupIdInfoHash{$raidGroupId}{state} . "," . $line;
                }
            }

            close (CONF_DUMP);
        }

        # Check whether Storage RAID Group map contains the data.
        if (!%raidGroupIdInfoHash) {
            $logObj->error("Storage RAID Group map is empty.");
            $retVal = 1;

            goto EXIT;
        }
    } else {
        $logObj->warn("No raw files found for RAID Group resource.");
        $retVal = 1;

        goto EXIT;
    }

    # Now iterate through hash and dump in  format.
    my $name = "";
    my $type = "";
    my $state = "";
    my $maxDiskCount = "";
    my $maxLunCount = "";
    my $totalRawCapacityInGB = "";
    my $totalLogicalCapacity = "";
    my $totalFreeCapacity = "";

    my $luns = "";
    my $disks = "";

    for my $id ( keys %raidGroupIdInfoHash ) {
        $name = $raidGroupIdInfoHash{$id}{name};
        $type = $raidGroupIdInfoHash{$id}{type};
        $state = $raidGroupIdInfoHash{$id}{state};
        $maxDiskCount = $raidGroupIdInfoHash{$id}{maxDiskCount};
        $maxLunCount = $raidGroupIdInfoHash{$id}{maxLunCount};
        $totalRawCapacityInGB = $raidGroupIdInfoHash{$id}{totalRawCapacityInGB};
        $totalLogicalCapacity = $raidGroupIdInfoHash{$id}{totalLogicalCapacity};
        $totalFreeCapacity = $raidGroupIdInfoHash{$id}{totalFreeCapacity};

        print $CONF_HANDLE "\nR:vnxBlockRaidGroup#$name\n";
        print $CONF_HANDLE "a:name=$name\n";
        print $CONF_HANDLE "a:type=$type\n";
        print $CONF_HANDLE "a:state=$state\n";
        print $CONF_HANDLE "a:maxDiskCount=$maxDiskCount\n";
        print $CONF_HANDLE "a:maxLunCount=$maxLunCount\n";
        print $CONF_HANDLE "a:totalRawCapacityInGB=$totalRawCapacityInGB\n";
        print $CONF_HANDLE "a:totalLogicalCapacity=$totalLogicalCapacity\n";
        print $CONF_HANDLE "a:totalFreeCapacity=$totalFreeCapacity\n";

        $disks = $raidGroupIdInfoHash{$id}{disks};

        my @diskList = split /,/, $disks;
        my $relations = "r:vnxBlockSystem#$vnxBlockSystemName";

        foreach my $disk (@diskList) {
            $relations = $relations . ",vnxBlockDisk#$vnxBlockSystemName^$disk";
        }

        $luns = $raidGroupIdInfoHash{$id}{luns};

        my @lunList = split /\s+/, $luns;

        foreach my $lun (@lunList) {
            $relations = $relations . ",vnxBlockLun#$vnxBlockSystemName^$lun";
        }

        print $CONF_HANDLE "$relations\n";
    }

EXIT:

    return $retVal;
}


# This function dumps the processed Storage Group resource and Host Wwn resource conf data in final mars format.
#
# @return :
#   0 if Success
#   1 if Error
sub processSgData {
    # This variable is used to store the return value of function calls.
    my $retVal = 0;

    # This variable tracks start of Sg Info data in raw conf file.
    my $isSgDataStarted = 0;

    # This variable stores Sg Id.
    my $sgId = "";

    # This variable stores Sg Name.
    my $sgName = "";

    # Used to store Sg info until all Sg entries are read and can be dumped in file.
    # Key is "Sg Id" and Value is various attribute of that Sg.
    my %sgIdInfoHash = ();

    # This variable is used to store host wwwn in current storage group.
    my %currentWwnHostMap = ();

    # This variable is used to store list of storage processors raw files.
    my %fileToIpMap = ();

    # Get Storage Processor information from <SP_IP>_disk.txt file.
    if (parseRawFiles('sg', \%fileToIpMap)) {

        foreach my $rawConfFile ( keys %fileToIpMap ) {

            $logObj->info("Reading raw configuration data for Sg resource.");

            open(CONF_DUMP, $rawConfFile) or die $logObj->error("Not able to open the file: [$rawConfFile].");

            while (<CONF_DUMP>) {
                # Remove the new line character.
                chomp;

                # Skip empty lines.
                if (/^\s*$/) {
                    next;
                }

                if (/^\s*Storage\s*Group\s*Name\s*:\s*(.*)\s*/i) {
                    $sgName = $1;
                    next;
                }

                # If this line found then it means Sg data starts from next line
                if (/^\s*Storage\s*Group\s*UID\s*:\s*(.*)\s*/i) {
                    $logObj->debug("Found VNX Block storage group data start.");
                    $isSgDataStarted = 1;

                    $sgId = $1;

                    $sgIdInfoHash{$sgId}{name} = $vnxBlockSystemName . "^" . $sgName;

                    # Clear host wwns associated with last storage group.
                    %currentWwnHostMap = ();

                    next;
                }

                # Disk data not started yet. Skip this line
                if (!$isSgDataStarted) {
                    next;
                }

                # Read Actual data.
                if (/^\s*([a-zA-Z0-9:]+)\s+([SP].+)\s+(\d+)\s*/i) {
                    my $wwn = $1;
                    my $sp = $2;
                    my $port = $3;

                    # Replace spaces if any and upper case all.
                    $sp =~ s/\s+//g;
                    $sp = uc $sp;

                    $sgIdInfoHash{$sgId}{wwns}{$wwn} = "";
                    $sgIdInfoHash{$sgId}{ports}{$sp . "^" . $port} = "";


                    # Check if this host wwn is present in map

                    if (exists $hostWwnInfoHash{$wwn}) {

                        if (index($hostWwnInfoHash{$wwn}{ports}, $sp . "^". $port) == -1) {
                            $hostWwnInfoHash{$wwn}{ports} = $hostWwnInfoHash{$wwn}{ports} . "," . $sp . "^". $port;
                        }
                    } else {
                        # Create entry for this host wwn.
                        $hostWwnInfoHash{$wwn} = {ports => ""};
                        $hostWwnInfoHash{$wwn}{ports} = $sp . "^". $port;
                    }

                    $currentWwnHostMap{$wwn} = "";
                } elsif (/^\s*(\d+)\s*(\d+)\s*$/i) {
                    my $lun = $2;
                    $sgIdInfoHash{$sgId}{luns}{$lun} = "";

                    foreach my $item (keys %currentWwnHostMap) {
                        $hostWwnInfoHash{$item}{relation}{"vnxBlockLun#$vnxBlockSystemName^$lun"} = "";
                    }
                }
            }

            close (CONF_DUMP);
        }

        # Check whether storage processor map contains the data.
        if (!%sgIdInfoHash) {
            $logObj->error("Storage Sg map is empty.");
            $retVal = 1;

            goto EXIT;
        }
    } else {
        $logObj->warn("No raw files found for Sg resource.");
    }
    
    for my $id ( keys %sgIdInfoHash ) {
        my $spPortID = "";
        my $wwnCount = 0;
        my $lunCount = 0;
        my $relations = "r:vnxBlockSystem#$vnxBlockSystemName";
        my $name = $sgIdInfoHash{$id}{name};
        
        foreach my $port (keys %{$sgIdInfoHash{$id}{ports}}) {
            for my $tempId ( keys %portToSpPortMap ) {
                if($portToSpPortMap{$tempId}{spPort} eq $port) {
                        $logObj->info("Mapping Found port:spPort:sg $tempId:$port:$name");
                        $relations = $relations . ",vnxBlockPort#$vnxBlockSystemName^$tempId";
                        $spPortID = $port;
                }
            }
        }

        #Currently not dumping wwn information
        foreach my $wwn (keys %{$sgIdInfoHash{$id}{wwns}}) {
            my $sigWwn = "vnxBlockHostWwn#$vnxBlockSystemName^$wwn";
            $relations = $relations . ",". $sigWwn;
            $wwnCount++;

            $hostWwnToSg{$sigWwn}{"vnxBlockSg#$vnxBlockSystemName^$id"} = "";
        }

        foreach my $lun (keys %{$sgIdInfoHash{$id}{luns}}) {
            $relations = $relations . ",vnxBlockLun#$vnxBlockSystemName^$lun";
            $lunCount++;

            # Following map is used to calculate total capacity of Storage group from LUNs in it.
            $lunToSgId{$id}{$lun} = "1";
        }

        print $CONF_HANDLE "\nR:vnxBlockSg#$vnxBlockSystemName^$id\n";
        print $CONF_HANDLE "a:name=$name\n";
        print $CONF_HANDLE "a:id=$id\n";
        print $CONF_HANDLE "a:spPort=$spPortID\n";
        print $CONF_HANDLE "a:wwnHostCount=$wwnCount\n";
        print $CONF_HANDLE "a:lunCount=$lunCount\n";
        print $CONF_HANDLE "$relations\n";
    }

    #Dump host wwn information
    for my $id ( keys %hostWwnInfoHash ) {
        my $ports = $hostWwnInfoHash{$id}{ports};

        $logObj->info("commaPorts $ports:$id");

        my @portList = split /,/, $ports;

        my $relations = "r:vnxBlockSystem#$vnxBlockSystemName";

        if (defined($hostWwnInfoHash{$id}{relation})) {
            foreach my $lunSig (keys %{$hostWwnInfoHash{$id}{relation}}) {
                $relations = $relations.",".$lunSig;
            }
        }

        foreach my $port (@portList) {
            for my $tempId ( keys %portToSpPortMap ) {
                $logObj->info("Checking for port:spPort:wwn $tempId:$port:$id");
                if($portToSpPortMap{$tempId}{spPort} eq $port) {
                    $relations = $relations . ",vnxBlockPort#$vnxBlockSystemName^$tempId";
                }
            }
        }

		# add hbaPort in relation
		if(length($id) >= 23) {
			my $tempWwn = substr $id, -23;
			$relations = $relations . ",hbaPort#$tempWwn";
		}

		$logObj->info("dumping vnxBlockHostWwn\n");
        print $CONF_HANDLE "\nR:vnxBlockHostWwn#$vnxBlockSystemName^$id\n";
        print $CONF_HANDLE "a:name=$vnxBlockSystemName^$id\n";
        print $CONF_HANDLE "$relations\n";
    }

	for my $tempId ( keys %portToSpPortMap ) {
			my $relations = "";

			for my $id ( keys %hostWwnInfoHash ) {

				my $ports = $hostWwnInfoHash{$id}{ports};
				
				$logObj->info("commaPorts $ports:$id");
				my @portList = split /,/, $ports;

				foreach my $port (@portList) {

					if(($portToSpPortMap{$tempId}{spPort} eq $port) && (length($id) >= 23)) {
						my $tempWwn = substr $id, -23;
						$relations = $relations . ",hbaPort#$tempWwn";
					}
				
				}
			}

			my $wwn = $portToSpPortMap{$tempId}{portWwn};
			if ($wwn !~ /^IQN/i && $wwn ne ""){
				$wwn =  substr $wwn,-23;

				print $CONF_HANDLE "\nR:sp#$wwn\n";
				print $CONF_HANDLE "a:name=$wwn\n";
				print $CONF_HANDLE "a:wwn=$wwn\n";
				if ($relations ne ""){
					print $CONF_HANDLE "r:$relations\n";
				}
			}
	}
	

EXIT:

    return $retVal;
}


# This function dumps the processed PortWwn resource conf data in final mars format.
#
# @return :
#   0 if Success
#   1 if Error
sub processPortWwnData {
    # This variable is used to store the return value of function calls.
    my $retVal = 0;

    # This variable tracks start of PortWwn Info data in raw conf file.
    my $isPortWwnDataStarted = 0;

    # This variable stores PortWwn Id.
    my $PortWwnId = "";

    # This variable stores processor Id.
    my $spId = "";

    # This variable stores I/O module slot Id.
    my $moduleId = "";
    
    # This variable stores I/O module slot Id.
    my $portId = "";
    
    # This variable stores I/O module slot Id.
    my $spPortId = "";

    # Used to store PortWwn info until all PortWwn entries are read and can be dumped in file.
    # Key is "PortWwn Id" and Value is various attribute of that PortWwn.
    my %portWwnIdInfoHash = ();

    # This variable is used to store list of storage processors raw files.
    my %fileToIpMap = ();

    # Get Storage Processor information from <SP_IP>_sp.txt file.
    if (parseRawFiles('portWwn', \%fileToIpMap)) {

        foreach my $rawConfFile ( keys %fileToIpMap ) {

            $logObj->info("Reading raw configuration data for PortWwn resource.");

            open(CONF_DUMP, $rawConfFile) or die $logObj->error("Not able to open the file: [$rawConfFile].");

            while (<CONF_DUMP>) {
                # Remove the new line character.
                chomp;
                if (/^\s*SP\s*Name\s*:\s*(.*)\s*/i) {
                
                    $spId = $1;

                    # Replace spaces if any and upper case all.
                    $spId =~ s/\s+//g;
                    $spId = uc $spId;
                    next;
                } elsif (/^\s*SP\s*Port\s*ID\s*:\s*(\d+)/i) {
                    $spPortId = $1;
                }
    
                # If this line found then it means Storage Processor data starts from next line
                if (/^\s*SP\s*UID\s*:\s*([a-fA-F0-9:]+|iqn.*)/i) {
                    $PortWwnId = uc $1;
                    $logObj->debug("Found VNX Block PortWwn data start. $PortWwnId");
                    $isPortWwnDataStarted = 1;
                    
                    # Create entry for this Disk id.
                    $portWwnIdInfoHash{$PortWwnId} = {name => "",
                                                      port => ""};

                    $portWwnIdInfoHash{$PortWwnId}{name} =  $PortWwnId;
                    next;
                }

                # PortWwn data not started yet. Skip this line
                if (!$isPortWwnDataStarted) {
                    next;
                }

                if (/^\s*I\/O\s*Module\s*Slot\s*:\s*(\d+)/i) {
                    $moduleId = $1;
                    next;
                } elsif (/^\s*Physical\s*Port\s*ID\s*:\s*(\d+)/i) {
                    $portId = $1;
                    $portWwnIdInfoHash{$PortWwnId}{port} = $vnxBlockSystemName . "^" . $spId . "^" . $moduleId . "-" . $portId;
                    # Fill physical port to SP port map
                    $portToSpPortMap{$spId . "^" . $moduleId . "-" . $portId} = {spPort => "",
                                                                                 portWwn => ""};
                    $portToSpPortMap{$spId . "^" . $moduleId . "-" . $portId}{spPort} = $spId . "^" . $spPortId;
                    $portToSpPortMap{$spId . "^" . $moduleId . "-" . $portId}{portWwn} = $PortWwnId;
                    $logObj->info("adding port:spPort:portWwn: $spId^$moduleId-$portId:$spId^$spPortId:$spPortId:$PortWwnId");
                    next;
                }
            }

            close (CONF_DUMP);
        }

        # Check whether storage processor map contains the data.
        if (!%portWwnIdInfoHash) {
            $logObj->error("Storage PortWwn map is empty.");
            $retVal = 1;

            goto EXIT;
        }
    } else {
    
        $logObj->warn("No raw files found for PortWwn resource.");
        $retVal = 1;

        goto EXIT;
    }

    # Now iterate through hash and dump in  format.
    my $name = "";
    my $id = "";
    my $port = "";
 
    for my $id ( keys %portWwnIdInfoHash ) {
        $name = $portWwnIdInfoHash{$id}{name};
        $port = $portWwnIdInfoHash{$id}{port};
 
        print $CONF_HANDLE "\nR:vnxBlockPortWwn#$vnxBlockSystemName^$name\n";
        print $CONF_HANDLE "a:name=$vnxBlockSystemName^$name\n";
 
        print $CONF_HANDLE "r:vnxBlockSystem#$vnxBlockSystemName,vnxBlockPort#$port\n";        
    }

EXIT:

    return $retVal;
}

# This function dumps the processed HBA Ports resource conf data in final mars format.
#
# @return :
#   0 if Success
#   1 if Error
sub processHbaPortData {
    # This variable is used to store the return value of function calls.
    my $retVal = 0;

    # This variable tracks start of HBA Port Info data in raw conf file.
    my $isHostPortDataStarted = 0;

    # This variable stores HBA Port Id.
    my $hbaPortId;

    # This variable stores HBA Id.
    my $hbaId;

    # This variable stores Server Name.
    my $serverName;

    # This variable stores Server IP Address.
    my $serverIpAddr;

    # This variable stores Server Operating system name.
    my $os;

    # This variable stores Server Operating system name.
    my $os;

    # This variable stores Processor name.
    my $spName;

    # This variable stores Processor port ID.
    my $spPortId;

    # This variable stores host port protocol type.
    my $protocol;

    # This variable stores host port logged-in state.
    my $loggedIn;

    # This variable stores host port logged-in state.
    my $initiatorType;

    # This map store HBA wwn information.
    my %hostWwnMap = ();

    # This map store HBA Port information.
    my %hbaPortInfo = ();

    # This variable is used to store list of storage processors raw files.
    my %fileToIpMap = ();

    # This variable is used to store list of host and its hostWwn
    my %hostInfoMap = ();

    # Get Storage Processor information from <SP_IP>_hbaPort.txt file.
    if (parseRawFiles('hbaPort', \%fileToIpMap)) {

        foreach my $rawConfFile ( keys %fileToIpMap ) {

            $logObj->info("Reading raw configuration data for HBA Port resource.");

            open(CONF_DUMP, $rawConfFile) or die $logObj->error("Not able to open the file: [$rawConfFile].");

            while (<CONF_DUMP>) {

                chomp;

                if (/^\s*HBA\s*UID\s*:\s*(.+)\s*/i) {
                    $hbaId = uc $1;
                    $hbaId =~ s/^\s+|\s+$//g;
                    $serverName = "";
                    $serverIpAddr = "";
                    $os = "";
                    next;
                } elsif (/^\s*Server\s*Name\s*:\s*(.+)\s*/i) {
                    $serverName = $1;
                    $serverName =~ s/^\s+|\s+$//g;
                    next;
                } elsif (/^\s*Server\s*IP\s*Address\s*:\s*(.+)\s*/i) {
                    $serverIpAddr = $1;
                    $serverIpAddr =~ s/^\s+|\s+$//g;
                    next;
                } elsif (/^\s*HBA\s*Vendor\s*Description\s*:\s*(.*)\s*/i) {
                    $os = $1;
                    $os =~ s/^\s+|\s+$//g;

                    my $sig = "vnxBlockHostWwn#$vnxBlockSystemName^$hbaId";
                    $hostWwnMap{$sig}{name} = "$vnxBlockSystemName^$hbaId";

                    if (!defined($serverName) || $serverName eq "") {
                        $serverName = $serverIpAddr;
                    }

                    $hostWwnMap{$sig}{serverName} = $serverName;
                    $hostWwnMap{$sig}{serverIpAddr} = $serverIpAddr;
                    $hostWwnMap{$sig}{os} = $os;

                    # Store host related information.
                    $hostInfoMap{$serverIpAddr}{hbas}{$sig} = "";
                    $hostInfoMap{$serverIpAddr}{serverName} = $serverName;
                    $hostInfoMap{$serverIpAddr}{os} = $os;

                    foreach my $key (keys %{$hostWwnToSg{$sig}}) {
                        $hostInfoMap{$serverIpAddr}{sgs}{$key} = "";
                    }
                    
                    if ($hbaId =~ /^\s*([A-Z0-9:]+)\s*$/) {
                        $protocol = "Fibre";
                        $hostWwnMap{$sig}{protocol} = $protocol;
                        $hostInfoMap{$serverIpAddr}{Fibre} = "Fibre";
                    } else {
                        $protocol = "iSCSI";
                        $hostWwnMap{$sig}{protocol} = $protocol;
                        $hostInfoMap{$serverIpAddr}{iSCSI} = "iSCSI";
                    }

                    next;
                } elsif (/^\s*SP\s*Name\s*:\s*SP\s*(A|B)\s*/i) {
                    $spName = $1;
                    $spName =~ s/^\s+|\s+$//g;
                    next;
                } elsif (/^\s*SP\s*Port\s*ID\s*:\s*(.*)\s*/i) {
                    $spPortId = $1;
                    $spPortId =~ s/^\s+|\s+$//g;
                    next;
                } elsif (/^\s*Logged\s*In\s*:\s*(.*)\s*/i) {
                    $loggedIn = $1;
                    $loggedIn =~ s/^\s+|\s+$//g;
                    next;
                } elsif (/^\s*Initiator\s*Type\s*:\s*(.*)\s*/i) {
                    $initiatorType = $1;
                    $initiatorType =~ s/^\s+|\s+$//g;

                    my $sig = "vnxBlockHbaPort#$vnxBlockSystemName^$spName^$spPortId^$hbaId";
                    $hbaPortInfo{$sig}{name} = "$vnxBlockSystemName^$spName^$spPortId^$hbaId";
                    $hbaPortInfo{$sig}{wwn} = $hbaId;
                    $hbaPortInfo{$sig}{spPort} = $spName."-".$spPortId;
                    $hbaPortInfo{$sig}{protocol} = $protocol;
                    $hbaPortInfo{$sig}{loggedIn} = $loggedIn;
                    $hbaPortInfo{$sig}{relation} = "vnxBlockHostWwn#$vnxBlockSystemName^$hbaId";
                    $spName = "";
                    $spPortId = "";
                    $loggedIn = "";
                    next;
                }
            }
            close (CONF_DUMP);
        }

    } else {
    
        $logObj->warn("No raw files found for PortWwn resource.");
        $retVal = 1;

        goto EXIT;
    }
 
    for my $id ( keys %hostWwnMap ) {
 
        print $CONF_HANDLE "\nR:$id\n";
        print $CONF_HANDLE "a:name=".$hostWwnMap{$id}{name}."\n";
        print $CONF_HANDLE "a:serverName=".$hostWwnMap{$id}{serverName}."\n";
        print $CONF_HANDLE "a:serverIpAddr=".$hostWwnMap{$id}{serverIpAddr}."\n";
        print $CONF_HANDLE "a:os=".$hostWwnMap{$id}{os}."\n";
        print $CONF_HANDLE "a:protocol=".$hostWwnMap{$id}{protocol}."\n";
    }

    for my $id ( keys %hostInfoMap ) {
        print $CONF_HANDLE "\nR:vnxBlockHost#$vnxBlockSystemName^$id\n";
        print $CONF_HANDLE "a:name=$hostInfoMap{$id}{serverName}\n";
        print $CONF_HANDLE "a:serverIpAddr=$id\n";
        print $CONF_HANDLE "a:os=$hostInfoMap{$id}{os}\n";

        my $protocol = "";

        if (defined($hostInfoMap{$id}{Fibre})) {
            $protocol = "Fibre";
        }

        if (defined($hostInfoMap{$id}{iSCSI})) {
            if ($protocol eq "") {
                $protocol = "iSCSI";
            } else {
                $protocol = "Fibre,iSCSI";
            }
        }

        print $CONF_HANDLE "a:protocol=$protocol\n";

        my $relation = "vnxBlockSystem#$vnxBlockSystemName";
        my %lunMap = ();

        foreach my $key (keys %{$hostInfoMap{$id}{hbas}}) {
            $relation = $relation .",".$key;

            # Get related luns to this host
            my @wwnTokens = split /\^/, $key;
            if (defined($wwnTokens[1]) && defined($hostWwnInfoHash{$wwnTokens[1]}{relation})) {
                foreach my $lunSig (keys %{$hostWwnInfoHash{$wwnTokens[1]}{relation}}) {
                    $lunMap{$lunSig} = "";
                }
            }
        }

        foreach my $key (keys %lunMap) {
            $relation = $relation .",".$key;
        }

        foreach my $key (keys %{$hostInfoMap{$id}{sgs}}) {
            $relation = $relation .",".$key;
        }

        print $CONF_HANDLE "r:$relation\n";
    }

    for my $id ( keys %hbaPortInfo ) {
 
        print $CONF_HANDLE "\nR:$id\n";
        print $CONF_HANDLE "a:name=".$hbaPortInfo{$id}{name}."\n";
        print $CONF_HANDLE "a:wwn=".$hbaPortInfo{$id}{wwn}."\n";
        print $CONF_HANDLE "a:spPort=".$hbaPortInfo{$id}{spPort}."\n";
        print $CONF_HANDLE "a:protocol=".$hbaPortInfo{$id}{protocol}."\n";
        print $CONF_HANDLE "a:loggedIn=".$hbaPortInfo{$id}{loggedIn}."\n";
        print $CONF_HANDLE "r:vnxBlockSystem#$vnxBlockSystemName,".$hbaPortInfo{$id}{relation}."\n";
    }

EXIT:

    return $retVal;
}

# This function is used to read the input probe instance properties file and set the global variables corresponding to 
# the different properties.
#
# @return :
#   0 if Success
#   1 if Error
sub readPropertyFile {
    # This variable is used to store the return code for this function.
    my $retVal = 0;

    $logObj->info("Reading file: $propFilePath");

    open(PROP_FILE, $propFilePath) or die $logObj->error("Not able to open the file: [$propFilePath].");

    # Loop through the content of .properties file.
    while (<PROP_FILE>) {
        # Remove the new line character.
        chomp;

        if (/^\s*SerialNumber\s*=\s*(\S+)\s*$/i) {
            # Look for SerialNumber.

            $vnxBlockSerialNumber = $1;
            $vnxBlockSerialNumber =~s/[\r\n]+//g;
        } elsif (/^\s*SiteName\s*=\s*(\S+)\s*$/i) {
            # Look for sitename.

            $siteName = $1;
            $siteName =~s/[\r\n]+//g;
        } else {
            # Current line is of no use. So ignore it.
        }
    }

    if ("" eq $vnxBlockSerialNumber) {
        $logObj->error("Probe name not present in the properties file.");
        $retVal = 1;

        goto EXIT;
    }
    $logObj->info("Found User-Id: [$vnxBlockSerialNumber] in the properties file.");

    if ("" eq $siteName) {
        $logObj->error("Sitename is not present in the properties file.");
        $retVal = 1;

        goto EXIT;
    }
    $logObj->info("Found Site name: [$siteName] in the properties file.");

EXIT:

    close(PROP_FILE);

    return $retVal;
}

# This function parse raw data directory to find specific list of file path and processor ids of those files.
#
# @param :
#   $_[0]   -   [In] File token.
#   $_[1]   -   [Out] File to Processor id Map.
#
# @return :
#         0 : In case if files not found.
#         1 : In case if files found.
sub parseRawFiles {

    my $fileToken = shift;
    my $fileToIdMap = shift;

    my $filePattern = "_". $fileToken . ".txt";
    my $filePathPattern = $rawDataDirectoryPath . "/*" . $filePattern;

    # Get processor ids.
    foreach my $file (glob($filePathPattern)) {
        my $basename = basename($file);
        $basename =~ s/$filePattern//g;

        $$fileToIdMap{$file} = $basename;
    }

    # Check if fileList is empty or not.
    if (!%$fileToIdMap) {
        return 0;
    }

    return 1;
}

# This function is used to format the wwn in colon seperated format.
#
# @params
#   wwn : 20000025B520000F
#
# @return :
#   formattedWWN (for e.g. 20:00:00:25:B5:20:00:0F)
sub formatWwn
{
  my $wwn = $_[0];
  my $l = length $wwn;
  if ($l != 16) {
    print $wwn;
    print "Invalid wwn.\n";
    return "";
  }

  $wwn = uc $wwn;
  my @parts = $wwn =~ /(.{1,2})/g;
  my $formattedWwn = "";
  foreach (@parts)
  {
    if ($formattedWwn eq ""){
      $formattedWwn = $_;
    } else {
      $formattedWwn = $formattedWwn.":".$_;
    }
  }
  return $formattedWwn;
}


# This function is used to de-initialize the environment.
#
# @return :
#   None
sub deInit {
    if ($CONF_HANDLE) {
        close($CONF_HANDLE);
    }
}
