# Copyright 2008-2015 Cumulus Systems Incorporated.
# All Rights Reserved.

# This file contains the code to collect the configuration data for the VNX File Storage whose instance property file path passed as argument to the
# script.
#
# @param : Property File Path - Complete path of the probe instance specific property file.
#
# @return : In case of Success, returns 0. In case of Failure, returns 1.
#
# Sample command: perl vnxFileConfDataGenerator.pl <Property File Path>

#!/usr/bin/perl -w

# It is used for strict compilation.
use strict;

# It is used to get base file name from the complete file name.
use File::Basename;

# It is used to get the current date and time.
use Time::HiRes qw(gettimeofday);
use POSIX "strftime";

# This is used to create folder recursively.
use File::Path qw(mkpath);

# This is used to move folders.
use File::Copy;

# This is used to mantain ssh session with VNX File storage.
use Net::OpenSSH;

# It is used to indicate the path where common module is placed.
use lib "/usr/local/megha/lib/common/perl";

# It is required for logging information.
use commonLogModule;

# It is used for reading property files.
use commonPropFileModule;

# It is required for getting decrypted password, and for executing commands.
use commonPerlUtil;

# It is used for using common conf util..
use commonProbeConfUtil;

# ----------------------------------------------------------------------------------------------------------------------------------------------------
# GLOBAL Variables
# ----------------------------------------------------------------------------------------------------------------------------------------------------

# Application home directory
use constant APP_HOME => "/usr/local/megha";

# This variable stores the probe type.
my $probeType = "VNXFile";

# Stores default command prompt pattern on target VNX File Storage.
my $cmdPrompt = "";

# This variable is used to store the path of probe properties files.
my $propFilePath = "";

# This variable stores commonLogModule instance.
my $logObj = "";

# This variable stores commonPropFileModule instance.
my $propFileUtilObj = "";

# Get commonProbeConfUtil instance.
my $probeConfUtilObj = "";

# This variable stores commonPerlUtil instance.
my $perlUtilObj = "";

# This variable is used to store the return status of function calls.
my $status = 0;

# This variable is used to store the user name of the VNX File storage control station.
my $username = "";

# This variable is used to store password corresponding to $username.
my $password = "";

# This variable is used to store IP Address of VNX File Storage control station.
my $ipAddress = "";

# This variable is used to store the Site name.
my $siteName = "";

# This variable is used to store the location of VNX runtime scripts.
my $vnxFileRunTimeLocation = "";

# This variable is used to store the path of particular VNX Probe folder.
my $vnxFileProbeFolder = "";

# This variable is used to store the path of conf directory inside particular VNX Probe folder.
my $vnxConfFolder = "";

# This variable is used to store the path of folder where temporary conf data is dumped for a particular VNX Probe.
my $tempConfDumpPath = "";

# This variable stores the current time in %Y%m%d_%H%M%S format.
my $currentTime = "";

# This variable is used to store the base folder path on the probe.
my $baseFolder = APP_HOME;

# Stores probe specific statistic groups.
my $probeStatsGroups = "";

# Stores default specific statistic groups.
my $defaultStatsGroups = "";

# Store name of online data mover.
my $onlineDM = "";

# store default property file path
my $defaultPropPath = "";

# This variable is used to store OpenSSH session.
my $ssh;

# This variable holds list of data movers.
my @dmNames = ();

# This variable is used to store name of raw data mover conf file name.
my $dataMoverFilename = "vnx_data_mover.txt";

my @tasks = ({
                command => 'nas_xml -info:ALL',
                file    => 'vnx_system.xml',
                mode    => '>'
            },{
                command => 'nas_pool -info -all',
                file    => 'vnx_pool.txt',
                mode    => '>'
            },{
                command => 'nas_pool -size -all',
                file    => 'vnx_pool.txt',
                mode    => '>>'
            },{
                command => 'nas_server -info -all',
                file    => $dataMoverFilename,
                mode    => '>'
            },{
                command => 'nas_volume -info -size -all',
                file    => 'vnx_volume.txt',
                mode    => '>'
            },{
                command => 'nas_fs -info -size -all',
                file    => 'vnx_fs.txt',
                mode    => '>'
            },{
                command => 'nas_storage -list',
                file    => 'vnx_storage_list.txt',
                mode    => '>'
            },{
                async_command => 'nas_storage -info -all -o sync=no',
                command => 'nas_storage -info -all',
                file    => 'vnx_storage_info.txt',
                mode    => '>'
            },{
                command => 'nas_disk -list',
                file    => 'vnx_disk.txt',
                mode    => '>'
            },{
                command => 'server_ifconfig ALL -all',
                file    => 'vnx_network.txt',
                mode    => '>'
            },{
                command => 'nas_replicate -info -all',
                file    => 'vnx_replicate.txt',
                mode    => '>'
            },{
                command => 'server_viruschk ALL',
                file    => 'vnx_viruschk.txt',
                mode    => '>'
            },{
                command => 'server_uptime ALL',
                file    => 'vnx_serverUptime.txt',
                mode    => '>'
            },{
                command => 'nas_server -info -vdm -all',
                file    => 'vnx_vdm.txt',
                mode    => '>'
            },{
                command => 'fs_dedupe -info -all',
                file    => 'vnx_dedupe.txt',
                mode    => '>'
            },{
                command => "nas_fs -query:\* -fields:ID,DefaultBlockGracePeriod,DefaultInodeGracePeriod,UserQuotas,GroupQuotas -format:'ID=%s,DefaultBlockGracePeriod=%s,DefaultInodeGracePeriod=%s\\n%q\\n%q\\n' -query:\* -fields:FileSystem,FileSystemID,Name,Path,BlockHardLimit,BlockSoftLimit,BlockUsage,BlockGracePeriod,InodeHardLimit,InodeSoftLimit,InodeUsage,InodeGracePeriod -format:'QuotaType=User,FileSystem=%s,FileSystemID=%s,Name=%s,Path=%s,BlockHardLimit=%s,BlockSoftLimit=%s,BlockUsage=%s,BlockGracePeriod=%s,InodeHardLimit=%s,InodeSoftLimit=%s,InodeUsage=%s,InodeGracePeriod=%s\\n' -query:\* -fields:FileSystem,FileSystemID,Name,Path,BlockHardLimit,BlockSoftLimit,BlockUsage,BlockGracePeriod,InodeHardLimit,InodeSoftLimit,InodeUsage,InodeGracePeriod -format:'QuotaType=Group,FileSystem=%s,FileSystemID=%s,Name=%s,Path=%s,BlockHardLimit=%s,BlockSoftLimit=%s,BlockUsage=%s,BlockGracePeriod=%s,InodeHardLimit=%s,InodeSoftLimit=%s,InodeUsage=%s,InodeGracePeriod=%s\\n'",
                file    => 'vnx_quotas.txt',
                mode    => '>'
            } );

my @tasks2 = ({
                command => 'server_stats DM_NAME -i 60 -c 1 -m nfs.client',
                file    => 'vnx_nfs_client.txt',
            },{
                command => 'server_stats DM_NAME -i 60 -c 1 -m cifs.user',
                file    => 'vnx_cifs_user.txt',
            },{
                command => 'server_stats DM_NAME -i 60 -c 1 -m cifs.client',
                file    => 'vnx_cifs_client.txt',
            });
# ----------------------------------------------------------------------------------------------------------------------------------------------------
# Main function
# ----------------------------------------------------------------------------------------------------------------------------------------------------

# Initialize the environment.
$status = init();
if (0 != $status) {
    $logObj->error("Call to init() failed.");
    goto EXIT;
}

# Dump meta-data info
$status = dumpMetaData();
if (0 != $status) {
    $logObj->error("Call to dumpMetaData() failed.");
    goto EXIT;
}

# Get All Raw configuration data.
$status = getRawConfData();
if (0 != $status) {
    $logObj->error("Call to getRawConfData() failed.");
    goto EXIT;
}

#Dump custom resource
$status = dumpCustomResource();
if (0 != $status) {
    $logObj->error("Call to dumpCustomResource() failed.");
    goto EXIT;
}

# parse data movers data.
$status = parseDataMover();
if (0 != $status) {
    $logObj->error("Call to parseDataMover() failed.");
    goto EXIT;
}

# Get All Raw configuration data.
$status = getRawNfsAndCifsConfData();
if (0 != $status) {
    $logObj->error("Call to getRawNfsAndCifsConfData() failed.");
    goto EXIT;
}

# Update stats group
validateAndUpdateStatPaths();

$logObj->info("Configuration data collection done. Next Data will be collected After 4 hours.");

EXIT:

exit $status;

# ----------------------------------------------------------------------------------------------------------------------------------------------------
# Sub-routines
# ----------------------------------------------------------------------------------------------------------------------------------------------------

# This function is used to initialize the environment to run the Perl script correctly.
#
# @affected global variables :
#   $propFilePath
#   $currentTime
#   $ipAddress
#   $tempConfDumpPath
#   $vnxFileRunTimeLocation
#   $vnxFileProbeFolder
#   $vnxConfFolder
#   $logObj
#   $status
#   $propFileUtilObj
#   $perlUtilObj
#
# @return :
#   0 if Success
#   1 if Error
sub init {
    # This variable is used to store the return value of function calls.
    my $retVal = 0;

    # Directory where the SSH master control socket will be created.
    # This directory and its parents must be writable only by the current effective user or root, otherwise the connection will be aborted to avoid insecure operation.
    # By default ~/.libnet-openssh-perl is used.
    my $ctlDirectory = $baseFolder."/libnet-openssh-perl";

    # Get current timestamp
    my ($seconds, $microseconds) = gettimeofday;
    $currentTime = strftime("%Y%m%d_%H%M%S", gmtime($seconds));

    # This variable stores Log folder path.
    my $logFolder = $baseFolder."/logs";

	$defaultPropPath = $baseFolder."/conf/probe/vnxfile_default.properties";

    # If log folder does not exist then create it.
    if (! (-e $logFolder)) {
        mkpath($logFolder);
    }

    # Get commonLogModule instance.
    $logObj = commonLogModule->getInstance($logFolder, $probeType, $ipAddress);

    # If number of command line arguments are not correct then quit.
    if (1 != ($#ARGV + 1)) {
       $logObj->error("Number of arguments to vnxFileConfDataGenerator.pl are not correct. Input argument count = ".($#ARGV + 1));
       $retVal = 1;

       goto EXIT;
    }
	
	# Get the VNX File probe property file path from command line argument.
    $propFilePath = $ARGV[0];

    $logObj->info("Properties file path: [$propFilePath].");

    # Read the instance properties file corresponding to VNX probe.
    # NOTE :- We cannot use common prop module for this because we don't have probeId for probe yet. We have to read probe Id(IP address) from probe
    # instance property file first.
    $status = readPropertyFile();
    if (0 != $status) {
        $logObj->error("Call to readPropertyFile() failed.");
        $retVal = 1;

        goto EXIT;
    }

    # Load the commonPropFileModule object for this probe
    $propFileUtilObj = commonPropFileModule->getInstance($baseFolder, $probeType, $ipAddress, $logObj);

    # Update the final log message information with populated VNXFile probe id.
    $logObj->setProbeId($ipAddress);
	
	# Get commonProbeConfUtil instance.
	$probeConfUtilObj = commonProbeConfUtil->getInstance($baseFolder, $probeType, $ipAddress, $logObj);

    # Get commonPerlUtil instance.
    $perlUtilObj = commonPerlUtil->getInstance($baseFolder, $probeType, $ipAddress, $logObj);
    
    # Decrypt password
    $status = decryptPassword();
    if (0 != $status) {
        $logObj->error("Call to decryptPassword() failed.");
        $retVal = 1;

        goto EXIT;
    }

    $password =~ s/^'//g;
    $password =~ s/'$//g;

	# redirect stderror of OpenShh 
	open (my $error_log, ">", "$baseFolder/temp/vnxFile-Conf-$ipAddress-errorlog");

    # Create OpenSSH session
    $ssh = Net::OpenSSH->new($ipAddress, master_stderr_fh => $error_log, ctl_dir => $ctlDirectory, (user => $username, passwd => $password, master_opts => [-o => "StrictHostKeyChecking=no"]));
    if ($ssh->error) {

		close($error_log);

		# Check error to see if SSH key changed. 
		open (my $errorFile, "<", "$baseFolder/temp/vnxFile-Conf-$ipAddress-errorlog");

		my @err = <$errorFile>;

		close($errorFile);

		my $errorMsg = "@err";

        $logObj->error("Failed to connect to the storage system, error:[". $ssh->error ."] detail error [". $errorMsg ."]");

        $retVal = 1;
        goto EXIT;
    }

    # Get the runtime location.
    $status = $propFileUtilObj->getProbeProperty("runtime.location", $vnxFileRunTimeLocation);
    if (0 != $status) {
        $logObj->error("Call to getProbeProperty() failed for runtime.location property.");
        $retVal = 1;

        goto EXIT;
    }

    $logObj->info("Runtime location is [$vnxFileRunTimeLocation].");

    # Get the default command prompt.
    $status = $propFileUtilObj->getProbeProperty("command.prompt.regex", $cmdPrompt);
    if (0 != $status) {
        $logObj->error("Call to getProbeProperty() failed for [command.prompt.regex].");
        $retVal = 1;

        goto EXIT;
    }

	# Get counter groups information.
    $status = $propFileUtilObj->getProbeProperty("perf.counter.groups", $probeStatsGroups);
    if (0 != $status) {
        $logObj->error("Call to getProbeProperty() to get [perf.counter.groups] failed.");
        $retVal = 1;
        goto EXIT;
    }

    $cmdPrompt = "'". $cmdPrompt . "'";

    $logObj->info("Default command prompt is [$cmdPrompt].");

    # This variable is used to store the path of Probe folder in db/probe.
    $vnxFileProbeFolder = $baseFolder."/db/probe/".$siteName."/".$probeType."_".$ipAddress;

    $logObj->info("Creating folder: [$vnxFileProbeFolder].");

    # Create the folder for this VNX Probe in the probe folder.
    unless (defined eval {mkpath($vnxFileProbeFolder)}) {
        $logObj->error("Not able to create folder: [$vnxFileProbeFolder].");
        $retVal = 1;

        goto EXIT;
    }   

    # This variable is used to store the path of folder where final Raw Conf files will be moved.
    $vnxConfFolder = $vnxFileProbeFolder."/conf";

    $logObj->info("Creating folder: [$vnxConfFolder]");

    # Create the folder for the final Conf files.
    unless (defined eval {mkpath($vnxConfFolder)}) {
        $logObj->error("Not able to create folder: [$vnxConfFolder]");
        $retVal = 1;

        goto EXIT;
    }
    
    # Set the path where raw conf data gets dumped temporarily. When all raw conf data is collected, we will append ".DONE" at the end of 
    # directory name.
    $tempConfDumpPath = $vnxConfFolder."/Conf_".$currentTime;

    $logObj->info("Creating temp Conf data path: [$tempConfDumpPath]");

    # Create the folder for temporary conf data.
    unless (defined eval {mkpath($tempConfDumpPath)}) {
        $logObj->error("Not able to create folder: [$tempConfDumpPath]");
        $retVal = 1;

        goto EXIT;
    }

	readDefaultPropertyFile();

EXIT:

    return $retVal;
}

# This function is used to dump meta data info into vnxFileMetaData.txt file.
#
# @affected global variables :
#    None
#
# @return :
#   0 if Success
#   1 if Error
sub dumpMetaData {
	
	# This variable is used to store the return code for this function.
    my $retVal = 0;

    # Load the commonPropFileModule object for this probe
    $propFileUtilObj = commonPropFileModule->getInstance($baseFolder, $probeType, $ipAddress, $logObj);

	# It stores "probe_host" meta-data information.
	my $probeHost = "";

    # Get the probe host details.
    $status = $propFileUtilObj->getAppProperty("probe.host", $probeHost);
    if (0 != $status) {
        $logObj->error("[vnxFileConfDataGenerator][dumpMetaData] Call to getAppProperty() failed.");
        $retVal = 1;

        goto EXIT;
    }

	# File to dump meta data info
	my $metaDataFile = $tempConfDumpPath."/vnxFileMetaData.txt";

	$logObj->info("[vnxFileConfDataGenerator][dumpMetaData] Dumping meta-data info...");

	open(my $META_DATA_HANDLE, ">", $metaDataFile) or die $logObj->error("Not able to open the file: [$metaDataFile].");

	# Dump meta-data information.
	print($META_DATA_HANDLE "probe_host=" . $probeHost);
	print($META_DATA_HANDLE "\n");

	$logObj->info("[vnxFileConfDataGenerator][dumpMetaData] Finished dumping meta-data information.");

	# Closing file handle.
	close($META_DATA_HANDLE);

	return $retVal;
}

# This function is used to decrypt the password.
#
# @return :
#   0 if Success
#   1 if Error
sub decryptPassword {
    # This variable stores maximum retry count for decrypting VNX password.
    my $retryCount = 3;
    
    $status = 0;

    # Decrypt VNX password.
    while (0 < $retryCount) {
        $status = $perlUtilObj->getDataValue($password, $password);

        if (0 == $status) {
            # In case of successful execution, exit from while loop.

            last;
        } elsif (1 == $retryCount) {
            # If call invocation fails after all retries then it is an error. So exit from the script.

            $logObj->error("Call to getDataValue() failed.");
            goto EXIT;
        }

        # Try calling getDataValue() after some delay.
        $logObj->warn("Call to getDataValue() failed. Will retry after 60 seconds.");
        sleep(60);

        $retryCount--;
    }
    
EXIT :

    return $status;
}

# This function is used to dump custom attribute resource in vnxFile_probe_conf.txt file.
#
# @affected global variables :
#    None
#
# @return :
#   None
sub dumpCustomResource {
	
	# This variable is used to store the return code for this function.
    my $retVal = 0;

	# File to dump custom attribute data
	my $rawConfFile = $tempConfDumpPath."/vnx_probe_conf.txt";
	
	$logObj->info("[vnxFileConfDataGenerator] Dumping custom attributes...");

	# It stores custom attribute information.
	my %customAttrHash = $probeConfUtilObj->getCustomAttrValues($siteName);
	
	open(my $NAME_FILE_HANDLE, ">", $rawConfFile) or die $logObj->error("Not able to open the file: [$rawConfFile].");

	#Dump custom resource
	print($NAME_FILE_HANDLE "R:__probe#" . $siteName . "_" . $ipAddress . "\n");
	print($NAME_FILE_HANDLE "a:__name=" . $siteName . "_" . $ipAddress . "\n");
	
	foreach my $key (keys(%customAttrHash)) {
		print($NAME_FILE_HANDLE "a:" . $key . "=" . $customAttrHash{$key} . "\n");
	}

	print($NAME_FILE_HANDLE "\n");
	$logObj->info("[vnxFileConfDataGenerator] Finished dumping custom attributes.");
	close($NAME_FILE_HANDLE);
	
	return $retVal;
}

# This function is used to construct commands to collect configuration from VNX Storage.
#
# @return :
#   0 if Success
#   1 if Error
sub getRawConfData {

    # Execute commands sequentially.
    foreach my $task (@tasks) {
        my $command = $task->{command};
        my $file = $task->{file};
        my $mode = $task->{mode};

        $status = executeTask($command, $file, $mode);

        if (0 != $status) {
            $logObj->error("Failed to execute command: [$command]");

            $command = $task->{async_command};
            if (defined($command)) {
                $logObj->warn("Trying asyncronous command: [$command]");

                $status = executeTask($command, $file, $mode);

                if (0 != $status) {
                    $logObj->error("Failed to execute command: [$command]");
                    return $status;
                }
            } else {
                $logObj->info("Continue with next command");
                next;
            }
        }
    }

    # All Conf data collected. Add .DONE at the end of folder so that framework can pick this folder.
    #rename ($tempConfDumpPath, "$tempConfDumpPath.DONE") || die $logObj->error("Unable to rename folder [$tempConfDumpPath] to [$tempConfDumpPath.DONE]");
    #$logObj->info("Renamed:[$tempConfDumpPath] to [$tempConfDumpPath.DONE]");

    return 0;
}

# This function is used to parse the raw data file to get data mover names.
#
# @affected global variables :
#   $dataMovers
#
# @return :
#   0 if Success
#   1 if Error
sub parseDataMover {
    # This variable is used to store the return code for this function.
    my $retVal = 0;
    my $dataMoverName = "";
    my $dmRawFile = $tempConfDumpPath."/".$dataMoverFilename;
    $logObj->info("Reading file: $dmRawFile");

    open(R_FILE, $dmRawFile) or die $logObj->error("Not able to open the file: [$dmRawFile].");

    # Loop through the content of .properties file.
    while (<R_FILE>) {
        # Remove the new line character.
        chomp;

        if ($_ =~ /^name\s*=\s*(.+)\s*$/) {
            $dataMoverName = $1;
            $dataMoverName =~ s/[\r\n]+//g;
        } elsif ($_ =~ /^\s*actual\s*=\s*.*active.*\s*$/i) {
            push @dmNames, $dataMoverName;
            $dataMoverName = "";
        }elsif (($dataMoverName ne "") && ($_ =~ /^\s*type\s*=\s*nas\s*/i)) {
            # data mover name found and type matching.
			$onlineDM = $dataMoverName;

            $logObj->info("Found online data mover: [$onlineDM].");
        }

    }

    $logObj->info("List of data mover names is :[@dmNames]");

EXIT:

    close(R_FILE);

    return $retVal;
}

# This function is used to construct commands to collect configuration from VNX Storage.
#
# @return :
#   0 if Success
#   1 if Error
sub getRawNfsAndCifsConfData {

    my $isFirst = 0;

    # Execute commands sequentially.
    foreach my $task (@tasks2) {
        my $command = $task->{command};
        my $file = $task->{file};
        my $mode = ">";

        foreach my $dm (@dmNames) {

            if ($isFirst) {
                $mode = ">>";
            }

            $command =~ s/DM_NAME/$dm/g;
            $status = executeTask($command, $file, $mode);

            if (0 != $status) {
                $logObj->warn("Failed to execute command: [$command]");
                #return 1;
            }

            $isFirst = 1;
        }
    }

    # All Conf data collected. Add .DONE at the end of folder so that framework can pick this folder.
    rename ($tempConfDumpPath, "$tempConfDumpPath.DONE") || die $logObj->error("Unable to rename folder [$tempConfDumpPath] to [$tempConfDumpPath.DONE]");
    $logObj->info("Renamed:[$tempConfDumpPath] to [$tempConfDumpPath.DONE]");

    return 0;
}

# This function is used to collect all VNX Raw conf data from VNX Storage.
#
# @return :
#   0 if Success
#   1 if Error
sub executeTask {
    my $command = shift;
    my $rawConfFile = shift;
    my $mode = shift;
    my $outputFile = $tempConfDumpPath."/".$rawConfFile;

    $command = "export NAS_DB=/nas;/nas/bin/".$command;

    $logObj->info("Input to function executeTask- command: [$command], outputFile: [$outputFile], mode: [$mode]");

    my @output = $ssh->capture($command);

    if ($ssh->error) {
        $logObj->error("Failed to execute command :[$command], error [". $ssh->error ."]");
        return 1;
    }

    open (CONF_DUMP, $mode, $outputFile) or die $logObj->error("Not able to open the file for writing : [$outputFile]");

    foreach my $line (@output) {
        print CONF_DUMP $line;
    }

    close (CONF_DUMP);

    return 0;
}

# Verify data retrieved from VNX File Storage and update to propbespecific property file.
#
# @return :
#	0 : success
# 
sub validateAndUpdateStatPaths {
    # Store the return value of the function.
    my $retVal = 0;

    # Stores invalid stat paths.
    my $invalidStatPaths = "";

    # Stores stat paths supported by target system.
    my $targetStatPaths = "";

	my @defaultStatPathArray = ();

	my @probeStatPathArray = ();
	my @supportedStatPathArray = ();

    # Split stat paths based on ','.
    @defaultStatPathArray = split(',', $defaultStatsGroups);
    @probeStatPathArray = split(',', $probeStatsGroups);

    # Check whether non-empty stat path list is provided in vnxfile_default.properties file.
    if ($#defaultStatPathArray == -1 || $#probeStatPathArray == -1) {
        $logObj->error("No stat paths found.");
        $retVal = 1;

        goto EXIT;
    }
	
	# check for online data mover
	if ($onlineDM eq "") {
		$logObj->warn("Unable to validate stat path as no online Data mover found");
		$retVal = 1;
        goto EXIT;
	}

	# check if both array are identical, If yes then no need to do any verification.
	my $status = isIdenticalArray(\@defaultStatPathArray, \@probeStatPathArray);
	if ($status == -1) {
		$logObj->info("default stats [$defaultStatsGroups] and probe specific stats [$probeStatsGroups] are not identical, verify stats path again..");
	} else {
		$retVal = 0;

        goto EXIT;
	}

    $logObj->info("Collecting the samples to verify whether stat paths are present on target.");

    foreach my $currentStatPath (@defaultStatPathArray) {
        my $command = "export NAS_DB=/nas;/nas/bin/server_stats $onlineDM -monitor $currentStatPath -type diff -terminationsummary no -format csv -i 1 -c 1";

        my @output = $ssh->capture($command);

        if ($ssh->error) {
            $logObj->warn("Failed to execute command: [$command], error:[". $ssh->error ."]");

            if ($invalidStatPaths eq "") {
                $invalidStatPaths = $currentStatPath;
            } else {
                $invalidStatPaths = $invalidStatPaths.",".$currentStatPath;
            }
        } else {
			# Create comma separated string out of supported stat path.
			if ($targetStatPaths eq "") {
                $targetStatPaths = $currentStatPath;
            } else {
                $targetStatPaths = $targetStatPaths .",". $currentStatPath;
            }

			 push(@supportedStatPathArray, $currentStatPath);
		}

        $logObj->info("Executed command: [$command] successfully");
    }

    $logObj->info("Valid stat path are : [$targetStatPaths] and Invalid stat path are: [$invalidStatPaths].");

	my $status = isIdenticalArray(\@supportedStatPathArray, \@probeStatPathArray);
	if ($status == -1) {
		$logObj->info("update stat paths [$targetStatPaths] in probe specific file [$propFilePath]");
		
		if (open(prop, "<", $propFilePath)) {
			my @propFileContent = <prop>;

			close prop;
			
			foreach my $line (@propFileContent)
			{
				if ($line =~ m/perf.counter.groups/)
				{
					$line =~ s/=.+/=$targetStatPaths/g;
				}  
			}

			if (open(updateProp, ">", $propFilePath)) {

				print updateProp @propFileContent;   
				
				$logObj->info("updated property file successfully");
				close(updateProp);
			}
		} else {
			$logObj->warn("Not able to open the file: [$propFilePath].");
		}
	} else {
		$logObj->info("no new stat path to update.");
	}

EXIT:

    return $retVal;
}

#arguments are two array references
sub isIdenticalArray {

    my $array_ref_1 = shift;
    my $array_ref_2 = shift;
    my @arr1 = @$array_ref_1;
    my @arr2 = @$array_ref_2;

    #Check if they are not the same length.
    if( scalar(@arr1) != scalar(@arr2) ) {

       return -1;
    }

    @arr1 = sort { $a <=> $b } @arr1;
    @arr2 = sort { $a <=> $b } @arr2;

    foreach my $i( 0 .. $#arr1 ) {

        if ( $arr1[$i] != $arr2[$i] ) {
            return -1;
        }
    }

    return 0;
}


# This function is used to read the input probe instance properties file and set the global variables corresponding to the different properties.
#
# @affected global variables :
#   $username
#   $ipAddress
#   $password
#   $siteName
#
# @return :
#   0 if Success
#   1 if Error
sub readPropertyFile {
    # This variable is used to store the return code for this function.
    my $retVal = 0;

    $logObj->info("Reading file: $propFilePath");

    open(PROP_FILE, $propFilePath) or die $logObj->error("Not able to open the file: [$propFilePath].");

    # Loop through the content of .properties file.
    while (<PROP_FILE>) {
        # Remove the new line character.
        chomp;

        if ($_ =~ /^UserName=(\S+)\s*$/) {
            # Look for Username.

            $username = $1;
            $username =~s/[\r\n]+//g;
        } elsif ($_ =~ /^Password=(\S+)\s*$/) {
            # Look for Password.

            $password = $1;
            $password =~s/[\r\n]+//g;
        } elsif ($_ =~ /^IPAddress=(\S+)\s*$/) {
            # Look for IPAddress.

            $ipAddress = $1;
            $ipAddress =~s/[\r\n]+//g;
        } elsif ($_ =~ /^SiteName=(\S+)\s*$/) {
            # Look for sitename.

            $siteName = $1;
            $siteName =~s/[\r\n]+//g;
        } else {
            # Current line is of no use. So ignore it.
        }
    }

    if ("" eq $username) {
        $logObj->error("User-Id not present in the properties file.");
        $retVal = 1;

        goto EXIT;
    } else {
        $logObj->info("Found User-Id: [".$username."] in the properties file.");
    }

    if ("" eq $password) {
        $logObj->error("Password not present in the properties file.");
        $retVal = 1;

        goto EXIT;
    }

    if ("" eq $ipAddress) {
        $logObj->error("IP address not present in the properties file.");
        $retVal = 1;

        goto EXIT;
    } else {
        $logObj->info("Found VNX IP Address: [".$ipAddress."] in the properties file.");
    }

    if ("" eq $siteName) {
        $logObj->error("Sitename is not present in the properties file.");
        $retVal = 1;

        goto EXIT;
    } else {
        $logObj->info("Found Site name: [".$siteName."] in the properties file.");
    }

EXIT:

    close(PROP_FILE);

    return $retVal;
}

sub readDefaultPropertyFile {
    # This variable is used to store the return code for this function.
    my $retVal = 0;

    $logObj->info("Reading file: $defaultPropPath");

    if (open(PROP_FILE, $defaultPropPath) ) {

		# Loop through the content of .properties file.
		while (<PROP_FILE>) {
			# Remove the new line character.
			chomp;

			if ($_ =~ /^perf.counter.groups=(.+)\s*$/) {
				# Look for Username.

				$defaultStatsGroups = $1;
				$defaultStatsGroups =~s/[\r\n]+//g;
				
				 $logObj->info("default stats group is [$defaultStatsGroups]");
			}
		}

		close(PROP_FILE);
	} else {
		$retVal = 1;
	}

    return $retVal;
}