# Copyright 2008-2015 Cumulus Systems Incorporated.
# All Rights Reserved.

# This file contains the code to verify the connection to VNX File Storage.
#
# @param : MARS Site name - MARS Site name
# @param : Storage IP     - VNX File Storage IP Address
# @param : User Name      - VNX File Storage User Name
# @param : Key            - VNX File Storage Key
#
# @return : In case of Success, returns 0.
#           In case of Warning, returns 1.
#           In case of Failure, returns 2.
#
# Sample command: perl vnxFileConnectionVerifier.pl <MARS Site name> <Storage IP> <User Name> <Key>

#!/usr/bin/perl -w

# It is used for strict compilation.
use strict;

# It is used to get base file name from the complete file name.
use File::Basename;

# These are used to get the current date and time.
use Time::HiRes qw(gettimeofday);
use POSIX "strftime";

# This is used to create folder recursively.
use File::Path qw(mkpath);
use IO::File;

# This is used to fire command on the VNX File storage.
use Net::OpenSSH;

# It is used to indicate the path where common perl module is placed.
use lib "/usr/local/megha/lib/common/perl";

# It is required for logging information.
use commonLogModule;

# It is required for getting runtime location of vnxFileGetConnection.sh file.
use commonPropFileModule;

# It is required for getting Key from probe specific property file.
use commonPerlUtil;

# ----------------------------------------------------------------------------------------------------------------------------------------------------
# GLOBAL Variables
# ----------------------------------------------------------------------------------------------------------------------------------------------------

# Store the location of MARS base folder.
my $baseFolder = "/usr/local/megha";

# Store the complete name of file where output from VNX Storage will be dumped.
my $dumpFile = "";

# Stores the probe type.
my $probeType = "VNXFile";

# Store the return status of function calls.
my $status = 0;

# Stores commonLogModule instance.
my $logObj = "";

# Store the VNX File Storage Username.
my $username = "";

# This variable is used to store the VNX File Storage Key.
my $vnxFileKey = "";

# Store the VNX File Storage IP Address.
my $storageIP = "";

# Store the location of VNXFile probe runtime files.
my $vnxFileRunTimeLocation = "";

# Store name of online data mover.
my $onlineDM = "";

# Stores default list of stat paths on target VNX File Storage.
my $defaultStatPaths = "";

# Stores default command prompt pattern on target VNX File Storage.
my $cmdPrompt = "";

# Stores default list of stat paths on target VNX File Storage.
my @defaultStatPathArray = ();

# Stores list of un-supported stat paths on target VNX File Storage.
my @unsupportedStatPathArray = ();

# Store the MARS Site name.
my $marsSiteName = "";

# Stores the commonPerlUtil instance.
my $perlUtilObj = "";

# Stores handle of status file.
my $VD_FILE;

# Used to store openSsh session.
my $ssh;

# ----------------------------------------------------------------------------------------------------------------------------------------------------
# Main function
# ----------------------------------------------------------------------------------------------------------------------------------------------------

# Initialize the environment.
$status = init();
if (0 != $status) {
    $logObj->error("Call to init() failed.");
    goto EXIT;
}

# Verify VNX File Storage credentials.
$status = verifyCredential();
if (0 != $status) {
    $logObj->error("Call to verifyCredential() failed.");
    goto EXIT;
}

# Verify VNX File Storage stat paths.
$status = validateStatPaths();
if (2 == $status) {
    $logObj->error("Call to validateStatPaths() failed.");
    goto EXIT;
}

EXIT:

deinit();

exit $status;

# ----------------------------------------------------------------------------------------------------------------------------------------------------
# Sub-routines
# ----------------------------------------------------------------------------------------------------------------------------------------------------

# Initialize the environment to run the Perl script correctly.
#
# @return :
#   0 if Success
#   1 if Warning
#   2 if Error
sub init {
    # Store the function return value.
    my $retVal = 0;

    # Store the complete path of probe specific folder in db/probe folder.
    my $probeFolder = "";

    # Stores encrypted Key.
    my $encryptedKey = "";

    # Stores Log folder path.
    my $logFolder = $baseFolder."/logs";

    # If log folder does not exist then create it.
    if (! (-e $logFolder)) {
        mkpath($logFolder);
    }

    # Get commonLogModule instance.
    $logObj = commonLogModule->getInstance($logFolder, $probeType, $storageIP);

    # If number of command line arguments are not correct then quit.
    if (4 != ($#ARGV + 1)) {
        $logObj->error("Number of arguments to vnxFileConnectionVerifier.pl are not correct. Input arguments count = ".($#ARGV + 1));
        $retVal = 2;

        goto EXIT;
    }

    # Get the MARS Site name from command line argument.
    $marsSiteName = $ARGV[0];
    $logObj->info("Input Site name = [$marsSiteName]");

    # Get the VNX File Storage IP Address from command line argument.
    $storageIP = $ARGV[1];
    $logObj->info("Input IP Address = [$storageIP]");

    # Update the final log message information with populated VNX File probe id.
    $logObj->setProbeId($storageIP);

    # Get the VNX File User name from command line argument.
    $username = $ARGV[2];
    $logObj->info("Input User = [$username]");

    # Get the VNX File Key from command line argument.
    $encryptedKey = $ARGV[3];

    # Get commonPerlUtil instance.
    $perlUtilObj = commonPerlUtil->getInstance($baseFolder, $probeType, $storageIP, $logObj);

    # Decrypt the Key.
    $status = $perlUtilObj->getDataValue($encryptedKey, $vnxFileKey);
    if (0 != $status) {
        $logObj->error("Call to getDataValue() failed.");
        $retVal = 2;

        goto EXIT;
    }

    # Get commonPropFileModule instance.
    my $propFileObj = commonPropFileModule->getInstance($baseFolder, $probeType, $storageIP, $logObj);

    # Reading script runtime location from vnxfile_default.properties file.
    $status = $propFileObj->getProbeProperty("runtime.location", $vnxFileRunTimeLocation);
    if (0 != $status) {
        $logObj->error("Call to getProbeProperty() failed while reading [runtime.location] property.");
        $retVal = 2;

        goto EXIT;
    }

    # Reading statpath from vnxfile_default.properties file.
    $status = $propFileObj->getProbeProperty("perf.counter.groups", $defaultStatPaths);
    if (0 != $status) {
        $logObj->error("Call to getProbeProperty() failed while reading [perf.counter.groups] property.");
        $retVal = 2;

        goto EXIT;
    }

    # Split stat paths based on ','.
    @defaultStatPathArray = split(',', $defaultStatPaths);

    # Check whether non-empty stat path list is provided in vnxfile_default.properties file.
    if ($#defaultStatPathArray == -1) {
        $logObj->error("No stat paths found.");
        $retVal = 2;

        goto EXIT;
    }

    # Set MARS probe folder path.
    $probeFolder = $baseFolder."/db/probe/".$marsSiteName."/VNXFile_".$storageIP;
    $logObj->info("Creating folder: [$probeFolder].");

    # Create the VNX Probe folder in the db/probe folder.
    unless (defined eval {mkpath($probeFolder)}) {
        $logObj->error("Not able to create folder: [$probeFolder].");
        $retVal = 2;

        goto EXIT;
    }

    # Set the file name where the output of VNX CLI will be dumped.
    $dumpFile = $probeFolder."/vnxFileConnectionStatus.dump";
    $logObj->info("Dump file name: [$dumpFile].");

    # Open verification result file in Creates, Writes, and Truncates mode.
    open(VD_FILE, ">", $dumpFile) or die $logObj->error("Not able to open the file: [$dumpFile].");

EXIT:

    return $retVal;
}

# Verify Storage access information.
#
# @return :
#   0 if Success
#   1 if Warning
#   2 if Error
sub verifyCredential {
    # This variable is used to store the function return value.
    my $retVal = 0;

    # Directory where the SSH master control socket will be created.
    # This directory and its parents must be writable only by the current effective user or root, otherwise the connection will be aborted to avoid insecure operation.
    # By default ~/.libnet-openssh-perl is used.
    my $ctlDirectory = $baseFolder."/libnet-openssh-perl";

    # Stores current data mover name.
    my $currentDataMover = 0;

    $logObj->info("Verification of access information started.");

    $vnxFileKey =~ s/^'//g;
    $vnxFileKey =~ s/'$//g;

	# redirect stderror of OpenShh 
	open (my $error_log, ">", "$baseFolder/temp/vnxFile-$storageIP-errorlog");
	
    # Create openSSH session
    $ssh = Net::OpenSSH->new($storageIP, master_stderr_fh => $error_log, ctl_dir => $ctlDirectory, (user => $username, passwd => $vnxFileKey, master_opts => [-o => "StrictHostKeyChecking=no"]));
    if ($ssh->error) {

		close($error_log);

		# Check error to see if SSH key changed. 
		open (my $errorFile, "<", "$baseFolder/temp/vnxFile-$storageIP-errorlog");

		my @err = <$errorFile>;

		close($errorFile);
		
		my $errorMsg = "@err";
		
        $logObj->error("Failed to connect to the storage system, error:[". $ssh->error ."] detail error [". $errorMsg ."]");
		
        return 2;
    }

    $logObj->info("Executing command: [export NAS_DB=/nas;/nas/bin/nas_server -info -all]");

    my @output = $ssh->capture("export NAS_DB=/nas;/nas/bin/nas_server -info -all");

    if ($ssh->error) {
        $logObj->error("Failed to connect to execute command: [export NAS_DB=/nas;/nas/bin/nas_server -info -all], error:[". $ssh->error ."]");
        return 2;
    }

    $logObj->info("Executed command: [export NAS_DB=/nas;/nas/bin/nas_server -info -all] successfully");

    # Find active online data mover.
    foreach my $line (@output) {

        # Ignore empty lines.
        if ($line =~ /^$/) {
            next;
        }

        # matching name = then its data mover name.
        if ($line =~ /name\s*=\s*(.+)\s*/i) {
            $currentDataMover = $1;
            $logObj->debug("Found data mover: [$currentDataMover].");

            next;
        }

        # data mover name found and type matching.
        if (($currentDataMover ne "") && ($line =~ /type\s*=\s*nas\s*/i)) {
            $onlineDM = $currentDataMover;

            $logObj->info("Found online data mover: [$onlineDM].");

            last;
        }
    }

    if ($onlineDM eq "") {
        $logObj->error("No active data mover found.");
        $retVal = 2;

        print VD_FILE "ERROR:VNXE_ACTIVE_DATA_MOVER_NOT_FOUND\n";

        goto EXIT;
    }

    $logObj->info("Verification of access information completed.");

    return 0;
}

# Verify data retrieved from VNX File Storage.
#
# @return :
#   0 if Storage support atleast one stat path.
#   1 if Storage support no single stat path.
sub validateStatPaths {
    # Store the return value of the function.
    my $retVal = 0;

    # Stores the current stat path.
    my $currentStatPath = "";

    # Stores invalid stat paths.
    my $invalidStatPaths = "";

    # Stores stat paths supported by target system.
    my $targetStatPaths = "";

    $logObj->info("Collecting the samples to verify whether stat paths are present on target.");

    foreach my $currentStatPath (@defaultStatPathArray) {
        my $command = "export NAS_DB=/nas;/nas/bin/server_stats $onlineDM -monitor $currentStatPath -type diff -terminationsummary no -format csv -i 1 -c 1";

        my @output = $ssh->capture($command);

        if ($ssh->error) {
            $logObj->warn("Failed to execute command: [$command], error:[". $ssh->error ."]");
            push(@unsupportedStatPathArray, $currentStatPath);

            if ($invalidStatPaths eq "") {
                $invalidStatPaths = $currentStatPath;
            } else {
                $invalidStatPaths = $invalidStatPaths.",".$currentStatPath;
            }
        }

        $logObj->info("Executed command: [$command] successfully");
    }

    # If all stat groups supported then dump success status in status file.
    if ($#unsupportedStatPathArray == -1) {
        print VD_FILE "SUCCESS";
        print VD_FILE "\n";

        $retVal = 0;

        goto EXIT;
    }

    $logObj->warn("Invalid stat path are: [$invalidStatPaths].");

    # Probe can not be added in this case if all stat path are not supported.
    if ($#defaultStatPathArray == $#unsupportedStatPathArray) {
        $logObj->error("No stat paths supported by the target system.");
        $retVal = 2;

        print VD_FILE "ERROR:VNXE_ATLEAST_ONE_STAT_PATH_NEEDED_TO_COLLECT_STATISTIC_[%S]:$invalidStatPaths\n";

        goto EXIT;
    }

    # Create comma separated string out of supported stat path.
    foreach my $outer (@defaultStatPathArray) {
        my $found = 0;

        foreach my $inner (@unsupportedStatPathArray) {
            if ($inner eq $outer) {
                $found = 1;
                last;
            }
        }

        # If current stat path is supported.
        if (!$found) {
            if ($targetStatPaths eq "") {
                $targetStatPaths = $outer;
            } else {
                $targetStatPaths = $targetStatPaths .",". $outer;
            }
        }
    }

    print VD_FILE "WARN:VNXE_WARNING_LIST_OF_STAT_PATHS_NOT_SUPPORTED_[%S]:$invalidStatPaths:$targetStatPaths\n";

    $retVal = 1;

EXIT:

    return $retVal;
}

# Deinit the environment.
sub deinit() {
    close(VD_FILE);
}
