# Copyright 2008-2015 Cumulus Systems Incorporated.
# All Rights Reserved.
#
# This file contains the code to collect the performance data for the VNX File Storage whose properties file is passed as argument to this script.
#
# @param : Property File Path - Complete path of the probe instance property file.
#
# @return : In case of Success, returns 0. In case of Failure, returns 1.
#
# Sample command: perl vnxFilePerfDataGenerator.pl <Property File Path>
#
#!/usr/bin/perl -w

# Used for strict compilation.
use strict;

# Used to spawn threads.
use threads;

# Used to get base file name from the complete file name.
use File::Basename;

# Used to get the current date and time.
use Time::HiRes qw(gettimeofday);
use POSIX "strftime";

# Used to create folder recursively.
use File::Path qw(mkpath);

# It is used to copy the file from one location to other.
use File::Copy;

# This is used for XML parsing.
use XML::Simple;

# This is used to fire command on the VNX File storage.
use Net::OpenSSH;

# Used to indicate the path where common module is placed.
use lib "/usr/local/megha/lib/common/perl";

# It is required for logging information.
use commonLogModule;

# It is required for getting runtime location of the VNXFileGetPerfData.sh file.
use commonPropFileModule;

# It is required for get key.
use commonPerlUtil;

# ----------------------------------------------------------------------------------------------------------------------------------------------------
# GLOBAL Variables
# ----------------------------------------------------------------------------------------------------------------------------------------------------

# Used to hold the location of base folder of MARS.
my $baseFolder = "/usr/local/megha";

# Stores the probe type.
my $probeType = "VNXFile";

# Stores default command prompt pattern on target VNX File Storage.
my $cmdPrompt = "";

# Used to store the return status of function calls.
my $status = 0;

# Stores commonLogModule instance.
my $logObj = "";

# Used to store the Storage Username.
my $username = "";

# Used to store the Storage key corresponding to $username.
my $vnxFileKey = "";

# Used to store the encrypted Storage key corresponding to $username.
my $vnxFileKeyEnc = "";

# Used to store the IP Address of Storage.
my $storageIP = "";

# Used to store the Site name.
my $siteName = "";

# Stores storage serial number.
my $storageSerialNo = "";

# Stores active data mover ids.
my @dataMoverIds = ();

# Stores runtime location of scripts.
my $runTimeLocation = "";

# Stores all the non-conflicting statistic groups.
my $statsGroups = "";

# Used to store the collection interval (in secs) of performance raw data.
my $collectionInterval = 0;

# Used to store the time interval after which performance data will be exported.
my $exportInterval = 0;

# Used to store number of samples to be sent by Probe Appliance in one cycle.
my $noOfSamples = 0;

# Used to store the web server protocol type like https or http.
my $serverWebProtocol = "";

# Stores probe instance home directory path.
my $probeFolder = "";

# Stores probe instance property file path.
my $propFilePath = "";

# Used to store the final perf .DONE file path.
my $probePerfPath = "";

# Used to store the path where perf data gets dumped.
my $perfDumpPath = "";

# Used to store the path where raw perf data gets dumped.
my $tempPerfDumpPath = "";

# Used to store the data mover conf file path.
my $dmFile = "";

# Used to store the commonPerlUtil instance.
my $perlUtilObj = "";

# Used to store openSsh session.
my $ssh;
# ----------------------------------------------------------------------------------------------------------------------------------------------------
# Main function
# ----------------------------------------------------------------------------------------------------------------------------------------------------

# Initialize the environment.
$status = init();
if (0 != $status) {
    $logObj->error("Call to init() failed.");
    goto EXIT;
}

# Dump meta-data info
$status = dumpMetaData();
if (0 != $status) {
    $logObj->error("Call to dumpMetaData() failed.");
    goto EXIT;
}

# Find online data movers name.
$status = getOnlineDataMover();
if (0 != $status) {
    $logObj->error("Call to getOnlineDataMover() failed.");
    goto EXIT;
}

# Get VNX File serial number.
$status = getVNXFileSerialNumber();
if (0 != $status) {
    $logObj->error("Call to getVNXFileSerialNumber() failed.");
    goto EXIT;
}

# Get raw Perf data.
$status = generatePerfData();
if (0 != $status) {
    $logObj->error("Call to generatePerfData() failed.");
    goto EXIT;
}

EXIT:

exit $status;

# ----------------------------------------------------------------------------------------------------------------------------------------------------
# Sub-routines
# ----------------------------------------------------------------------------------------------------------------------------------------------------

# This function is used to initialize the environment to run the Perl script correctly.
#
# @return :
#   0 if Success
#   1 if Error
sub init {
    # Store the return value of function calls.
    my $retVal = 0;

    # Directory where the SSH master control socket will be created.
    # This directory and its parents must be writable only by the current effective user or root, otherwise the connection will be aborted to avoid insecure operation.
    # By default ~/.libnet-openssh-perl is used.
    my $ctlDirectory = $baseFolder."/libnet-openssh-perl";

    # This variable stores Log folder path.
    my $logFolder = $baseFolder."/logs";

    # If log folder does not exist then create it.
    if (! (-e $logFolder)) {
        mkpath($logFolder);
    }

    # Get commonLogModule instance.
    $logObj = commonLogModule->getInstance($logFolder, $probeType, $storageIP);

    # If number of command line arguments are not correct then quit.
    if (2 != ($#ARGV + 1)) {
        $logObj->error("Number of arguments to vnxFilePerfDataGenerator.pl are not correct. Input arguments count = ".($#ARGV + 1));
        $logObj->error("Usage: perl vnxFilePerfDataGenerator.pl <Properties File> <Perf Staggered Collection Seconds>");

        $retVal = 1;
        goto EXIT;
    }

    # VNX Properties file full path.
    $propFilePath = $ARGV[0];

    $logObj->info(" Properties file path: [$propFilePath].");

    # Number of seconds for which thread to sleep to achieve staggered data collection
    my $mycurrentTime = strftime("%Y%m%d_%H%M%S", gmtime(time));
    
    $logObj->info("processRawData Staggered Collection Time - $ARGV[1]");

    if (-1 != $ARGV[1])
    {
        $logObj->info("processRawData current time - $mycurrentTime");
        $logObj->info("processRawData thread sleeping for -[ $ARGV[1] ]");
    
        sleep($ARGV[1]);

        $mycurrentTime = strftime("%Y%m%d_%H%M%S", gmtime(time));
        $logObj->info("processRawData current time - $mycurrentTime");
    }
    
    # Read properties file corresponding to this VNX File probe.
    $status = readProbePropertyFile();
    if (0 != $status) {
        $logObj->error("Call to readProbePropertyFile() failed.");
        $retVal = 1;

        goto EXIT;
    }

    # Update the final log message information with populated VNXFile probe id.
    $logObj->setProbeId($storageIP);

    # Get commonPropFileModule instance.
    my $propFileObj = commonPropFileModule->getInstance($baseFolder, $probeType, $storageIP, $logObj);

    # Get the collection interval.
    $status = $propFileObj->getProbeProperty("runtime.location", $runTimeLocation);
    if (0 != $status) {
        $logObj->error("Call to getProbeProperty() to get [runtime.location] failed.");
        $retVal = 1;

        goto EXIT;
    }

    $logObj->info("Perf runtime location: [$runTimeLocation].");

    # Get counter groups for which we need to collect performance data.
    $status = $propFileObj->getProbeProperty("perf.counter.api.groups", $statsGroups);
    if (0 != $status) {
        $logObj->error("Call to getProbeProperty() to get [perf.counter.groups] failed.");
        $retVal = 1;

        goto EXIT;
    }
    

    $logObj->info("Stat path list used to collect performance counters: [$statsGroups].");

    # Get the collection interval.
    $status = $propFileObj->getProbeProperty("probe.perf.collection.interval.secs", $collectionInterval);
    if (0 != $status) {
        $logObj->error("Call to getProbeProperty() failed for [probe.perf.collection.interval.secs].");
        $retVal = 1;

        goto EXIT;
    }

    $logObj->info("Perf collection interval is: [$collectionInterval].");

    # Get the export interval which indicate the time interval for performance data export.
    $status = $propFileObj->getProbeProperty("probe.perf.export.interval.secs", $exportInterval);
    if (0 != $status) {
        $logObj->error("Call to getProbeProperty() failed for [probe.perf.export.interval.secs].");
        $retVal = 1;

        goto EXIT;
    }

    $logObj->info("Perf export interval is [$exportInterval].");

    # Get the default command prompt.
    $status = $propFileObj->getProbeProperty("command.prompt.regex", $cmdPrompt);
    if (0 != $status) {
        $logObj->error("Call to getProbeProperty() failed for [command.prompt.regex].");
        $retVal = 1;

        goto EXIT;
    }

    $cmdPrompt = "'" . $cmdPrompt . "'";
    $logObj->info("Default command prompt is [$cmdPrompt].");

    # Get the number of performance counters samples to be collected in one batch.
    $noOfSamples = $exportInterval/$collectionInterval;

    $logObj->info("Number of performance sample is [$noOfSamples].");

    $status = $propFileObj->getAppProperty("SERVER.WEB.PROTOCOL", $serverWebProtocol);
    if (0 != $status) {
        # If we are not able to get web protocol then continue with using "https" protocol.
        $logObj->warn("Call to getAppProperty() failed for property [SERVER.WEB.PROTOCOL]");

        $serverWebProtocol = "https";
    }

    # Get commonPerlUtil instance.
    $perlUtilObj = commonPerlUtil->getInstance($baseFolder, $probeType, $storageIP, $logObj);

    # Decrypt the Key.
    $status = $perlUtilObj->getDataValue($vnxFileKeyEnc, $vnxFileKey);
    if (0 != $status) {
        $logObj->error("Call to getDataValue() failed.");
        $retVal = 1;

        goto EXIT;
    }

    $vnxFileKey =~ s/^'//g;
    $vnxFileKey =~ s/'$//g;

	# redirect stderror of OpenShh 
	open (my $error_log, ">", "$baseFolder/temp/vnxFile-Perf-$storageIP-errorlog");
    
    # Create openSSH session
    $ssh = Net::OpenSSH->new($storageIP, master_stderr_fh => $error_log, ctl_dir => $ctlDirectory, (user => $username, passwd => $vnxFileKey, master_opts => [-o => "StrictHostKeyChecking=no"]));
    if ($ssh->error) {
        close($error_log);

		# Check error to see if SSH key changed. 
		open (my $errorFile, "<", "$baseFolder/temp/vnxFile-Perf-$storageIP-errorlog");

		my @err = <$errorFile>;

		close($errorFile);

		my $errorMsg = "@err";
        $logObj->error("Failed to connect to the storage system, error:[". $ssh->error ."] detail error [". $errorMsg ."]");

        $retVal = 1;
        goto EXIT;
    }

    # This variable is used to store the path of Probe folder in db/probe.
    my $probeFolder = $baseFolder."/db/probe/".$siteName."/VNXFile_".$storageIP;
    $logObj->info("Creating folder: [$probeFolder]");

    # Create the folder for this probe in the probe folder.
    unless (defined eval {mkpath($probeFolder)}) {
        $logObj->error("Not able to create folder: [$probeFolder]");
        $retVal = 1;

        goto EXIT;
    }

    # Set the path where perf data gets dumped.
    $perfDumpPath = "$probeFolder/perf";

    # Create the folder for perf data.
    unless (defined eval {mkpath($perfDumpPath)}) {
        $logObj->error("Not able to create folder: [$perfDumpPath].");
        $retVal = 1;

        goto EXIT;
    }

    # Set conf data mover file path.
    $dmFile = $perfDumpPath."/nas_server_info_all.dump";

    # It is used to store the current date and time in "YYYYMMDD_HHMMSS" format.
    my $currentTime = strftime("%Y%m%d_%H%M%S", gmtime(time));

    # Set the path where temporary perf data gets dumped.
    $tempPerfDumpPath = $perfDumpPath."/Perf_".$currentTime;
    $logObj->info("Creating temp Perf data path: [$tempPerfDumpPath].");

    # Create the folder for temporary perf data.
    unless (defined eval {mkpath($tempPerfDumpPath)}) {
        $logObj->error("Not able to create folder: [$tempPerfDumpPath].");
        $retVal = 1;

        goto EXIT;
    }

EXIT:

    return $retVal;
}

# This function is used to dump meta data info into vnxFileMetaData.txt file.
#
# @affected global variables :
#    None
#
# @return :
#   0 if Success
#   1 if Error
sub dumpMetaData {
	
	# This variable is used to store the return code for this function.
    my $retVal = 0;

    # Get commonPropFileModule instance.
    my $propFileObj = commonPropFileModule->getInstance($baseFolder, $probeType, $storageIP, $logObj);

	# It stores "probe_host" meta-data information.
	my $probeHost = "";

    # Get the probe host details.
    $status = $propFileObj->getAppProperty("probe.host", $probeHost);
    if (0 != $status) {
        $logObj->error("[vnxFilePerfDataGenerator][dumpMetaData] Call to getAppProperty() failed.");
        $retVal = 1;

        goto EXIT;
    }

	# File to dump meta data info
	my $metaDataFile = $tempPerfDumpPath."/vnxFileMetaData.txt";

	$logObj->info("[vnxFilePerfDataGenerator][dumpMetaData] Dumping meta-data info...");

	open(my $META_DATA_HANDLE, ">", $metaDataFile) or die $logObj->error("Not able to open the file: [$metaDataFile].");

	# Dump meta-data information.
	print($META_DATA_HANDLE "probe_host=" . $probeHost);
	print($META_DATA_HANDLE "\n");

	$logObj->info("[vnxFilePerfDataGenerator][dumpMetaData] Finished dumping meta-data information.");

	# Closing file handle.
	close($META_DATA_HANDLE);

	return $retVal;
}

# This function is used to to find online data mover(s) name. 
#
# @return :
#   0 if Success
#   1 if Error
sub getOnlineDataMover {
    # Used to check whether nas_server_info_all.dump contain output of the command or not. If 1 -> yes If 0 -> No.
    my $foundDmCmd = 0;

    # Used to check whether nas_storage-info-all.dump contain output of the command 'nas_storage -l' or not. If 1 -> yes If 0 -> No.
    my $foundSerialCmd = 0;

    # Store the exit status of "system" command.
    my $exitCode = 0;

    # This variable stores current data mover's name.
    my $currentDM = "";

    # This veriable set to 1 if current data mover's type is NAS else 0.
    my $isTypeNAS = 0;
    
    # This veriable set to 1 if current data mover is online else 0.
    my $isOnline = 0;

    # This variable is used to store the command to get the data.
    my $cmd = "export NAS_DB=/nas;/nas/bin/nas_server -info -all";

    $logObj->info("Running command: [$cmd].");

    my @output = $ssh->capture($cmd);

    if ($ssh->error) {
        $logObj->error("Failed to execute command: [$cmd] error:[" . $ssh->error . "]");
        return 1;
    }

    foreach my $line (@output) {
        chomp;

        # If empty line then next line.
        if ($line =~ /^\s*$/) {
            next;
        }

        if ($line =~ /^\s*name\s*=\s*(\S+)\s*$/i) {
            $currentDM = $1;
        } elsif ($line =~ /^\s*type\s*=\s*nas\s*$/i) {
            $isTypeNAS = 1;
        } elsif ($line =~ /^\s*actual\s*=.+online.+$/i) {
            $logObj->info("Online data mover name : [$currentDM]");

            $isOnline = 1;
        }

        # Add data mover to list if it is of NAS and online.
        if (($isTypeNAS == 1) && ($isOnline == 1)) {
            push(@dataMoverIds, $currentDM);

            $logObj->info("Added to global array online data mover name : [$currentDM]");

            # Reset states to search next active and online data mover.
            $currentDM = "";
            $isTypeNAS = 0;
            $isOnline  = 0;
        }
    }

    if (@dataMoverIds) {
        $logObj->info("Total active online data movers are [" . ($#dataMoverIds + 1) ."].");
        $logObj->info("Active online data movers are [@dataMoverIds].");
    } else {
        $logObj->error("Active online data mover(s) not found.");
        return 1;
    }

    return 0;
}

# This function reads XML raw dump file and load into $xmlInfo record and sets VNX File serial number.
#
# @return:
#   None
sub getVNXFileSerialNumber {

    # This variable keeps track of whether root tag found or not.
    my $foundXmlRootTag = 0;

    # This is Xml format file that will be created after filtering out Non-XML text from $rawConfFile
    my $rawXmlFile = $perfDumpPath."/vnx_system.xml";

    # Command to get VNX File Configuration data.
    my $command = "export NAS_DB=/nas;/nas/bin/nas_xml -info:ALL";
    
    $logObj->info("Running command: [$command].");

    my @output = $ssh->capture($command);

    if ($ssh->error) {
        $logObj->error("Failed to execute command: [$command] error:[" . $ssh->error . "]");
        return 1;
    }

    open (CONF_XML_DUMP, ">", $rawXmlFile) or die $logObj->error("Not able to open the file for writing : $rawXmlFile");

    foreach my $line (@output) {
        # Remove the new line character.
        chomp;

        if ($line =~ /^\s*<CELERRA\s+SRC/) {
            $foundXmlRootTag = 1;
            $logObj->debug("Found XML root tag.");

            print CONF_XML_DUMP "".$line."\n";

            next;
        }
        
        # Skip lines if xml root tag is not yet found
        if (0 == $foundXmlRootTag) {
            next;
        }

        # Filter out cmd prompt.
        if ($line =~ /\$\s*$/) {
            next;
        }

        print CONF_XML_DUMP "".$line."\n";
    }

    close (CONF_XML_DUMP);

    # Now load xml data from xml file
    my $xmlInfo = XMLin($rawXmlFile);

    # Get serial number from $xmlInfo perl record.
    $storageSerialNo = $xmlInfo->{CELERRA_CABINET}->{NAME};
    
    if ($storageSerialNo eq "") {
        $logObj->error("Failed to get storage serial number.");
        return 1;
    }

    $logObj->info("Storage serial number is: [$storageSerialNo].");
    
    return 0;
}

# Used to fetch the Raw Performance data of VNX File Storage. 
#
# @return :
#   0 if Success
#   1 if Error
sub generatePerfData {
    # Used to check whether temp_<Data Mover Id>.dump contain perf data or not. If 1 -> yes If 0 -> No.
    my $found = 0;

    # Used to hold loop indexs.
    my $index = 0;

    # Used to store the command used to collect the performance data from data mover.
    my $command = "";

    # Used to store the file path of raw data.
    my $timestampFilePath = "";

    # Used to store the commands to collect performance data from each data mover.
    my @commandArray = ();

    # Used to store the thread object of each thread.
    my @threadArray = ();

    # Used to store the return status of each thread.
    my @statusArray = ();

    # Used to store file paths where we are dumping raw data for each data mover.
    my @rawTempPerfFilePaths = ();

    # Used to store the return status of each thread.
    my @perfFilePaths = ();

    # Used to store the file path in which final data of the command ouput will be dumped.
    my $rawPerfFile = "";

    # Command server_stats will be always in the output file if everything goes fine.
    my $pattern = 'server_stats';

    my @commandGroups = split('##',$statsGroups);


    $logObj->info("Populating array of commands to execute with apis. [@commandGroups]");

    # Populate commandArray by adding all the commands which need to be fired for each data mover.
    foreach my $dataMoverId (@dataMoverIds) {
        my $index = 0;
        foreach my $commandGroup (@commandGroups) {
	       $rawPerfFile = $tempPerfDumpPath."/perf##" . $storageSerialNo . "##" .$dataMoverId ."#".$index. ".dump";

           push(@perfFilePaths, $rawPerfFile);
           $command = "export NAS_DB=/nas;/nas/bin/server_stats $dataMoverId -monitor $commandGroup -type diff -terminationsummary no -format csv -i $collectionInterval -c $noOfSamples";
           $logObj->info("Adding command: [$command]");
           push(@commandArray, $command);       
           $index = $index + 1 ;
        }
    }

    $logObj->info("Executing commands asynchronously");
    my $index = 0;

    # Spawn threads asynchronously and populate threadArray by adding all the thread object into it.
    foreach $command (@commandArray) {
        my $source = $perfFilePaths[$index];
        my $target = $perfFilePaths[$index].".DONE";

        # Once command returns successfully, append .DONE in the file name so that we can track that all threads got completed.
        push @threadArray, async {
            my $finalCommand = $command;
            my $sourceFile = $source;
            my $targetFile = $target;

            $logObj->info("Executing command asynchronously : [$finalCommand]");

            my @output = $ssh->capture($finalCommand);

            $logObj->info("Command executed successfully : [$finalCommand]");

            open (PERF_DUMP, ">", $sourceFile) or die $logObj->error("Not able to open the file for writing : [$sourceFile]");

            foreach my $line (@output) {
                print PERF_DUMP $line;
            }

            close (PERF_DUMP);

            if (0 == move($sourceFile, $targetFile)) {
                $logObj->error("File move failed. Source: [$sourceFile]. Destination: [$targetFile]");
            } else {
                $logObj->info("File moved successfully. Source: [$sourceFile]. Destination: [$targetFile]");
            }
        };

        $index++;
    }

    $logObj->info("Waiting for all thread to complete.");

    # This variable stores count of total files completed successfully.
    my $fileCount = 0;

    # This variable stores total time in seconds since command fired.
    my $timeCount = 0;

    # This variable stores whether time out happened or not.
    my $isTimeout = 0;

    # We are waiting for 5 extra minutes  to handle any delay in the command execution.
    my $maxWaitTime = $exportInterval + (5 * 60);

    # This variable stores file pattern to check.
    my $filePattern = $tempPerfDumpPath . "/perf*dump.DONE";

    # Break loop if all the commands are completed.
    while (($#perfFilePaths + 1) != $fileCount) {
        sleep(2);

        $timeCount = $timeCount + 2;

        # Break loop if time out.
        if ($timeCount > $maxWaitTime) {
            $isTimeout = 1;
            last;
        }

        my @files = glob($filePattern);
        $fileCount = $#files + 1;
    }

    if ($isTimeout) {
        $logObj->error("Timeout while checking final perf raw data.");
        return 1;
    }

    foreach my $filePath (@perfFilePaths) {
        # Renaming files which are having complete data.    
        $logObj->info("Renaming [$filePath.DONE] to [$filePath]");

        rename($filePath . ".DONE" , $filePath) || $logObj->error("While moving renaming file: [$filePath.DONE] to [$filePath].");
        $timestampFilePath = $filePath;
    }

    $logObj->info("Finding actual timestamp,file: [$timestampFilePath]");

    my $timestamp = readDumpFile($timestampFilePath);

    $logObj->info("Actual timestamp: [$timestamp]");

    if (!defined($timestamp)) {
        $logObj->error("Actual timestamp not found in raw data file: [$timestampFilePath]");
        return 1;
    }

    # Set the path where temporary perf data gets dumped.
    my $finalPerfDumpPath = $perfDumpPath."/Perf_".$timestamp.".DONE";

    $logObj->info("Creating final Perf data path: [$finalPerfDumpPath].");

    # Create the final perf data.
    unless (defined eval {mkpath($finalPerfDumpPath)}) {
        $logObj->error("Not able to create folder: [$finalPerfDumpPath].");
        return 1;
    }

    # Moving directory to perf folder for final processing.
    $logObj->info("Renaming [$tempPerfDumpPath] to [$finalPerfDumpPath]");

    rename($tempPerfDumpPath , $finalPerfDumpPath) || $logObj->error("While moving final performance folder [$tempPerfDumpPath] to [$finalPerfDumpPath].");

    return 0;
}

# This function is used find our data start time.
#
# @param:
#    $_[0]: [In] Dump file name.
#
# @return :
#   timestamp
#
sub readDumpFile {
    my $filename = shift;

    # Store the return value of function calls.
    my $timestamp = undef;

    # Used to check whether header row is found or not. If 0 means not found, 1 means found.
    my $hfound = 0;

    # Used to store current data time index.
    my $timeIndex = 0;

    # Open the file handle to the dump file.
    open(PERF_DUMP, $filename) or die $logObj->error("Not able to open the file: [$filename].");

    # Reading raw performance file.
    while(<PERF_DUMP>) {

        # Remove the new line character.
        $_ =~ s/\r\n//g;
        $_ =~ s/\n//g;

        my $line = $_;

        # Check if headers found. Headers may come one or more times in the data according to change in resources.
        if (/Unix\s*Seconds/i) {

            # Splitting header line by ',' (comma).
            my @tempArray = split(',', $line);

            # Triming and removing "" (Double quote).
            foreach my $item (@tempArray) {
                $item =~ s/^\s*"|\s*"$//g;
                if ($item =~ /Unix\s*Seconds/i) {
                    # Set hfound to 1.
                    $hfound = 1;
                    last;
                }
                $timeIndex++;
            }
            next;
        }

        # If command or header row not found then continue with next line.
        if ($hfound == 0) {
            next;
        }

        # Splitting header line by ',' (comma).
        my @tempArray = split(',', $line);
        my $unixSeconds = $tempArray[$timeIndex];
        $unixSeconds =~ s/^\s*"|\s*"$//g;
        $timestamp = strftime("%Y%m%d_%H%M%S", gmtime($unixSeconds));
        last;
    }

    close (PERF_DUMP);

EXIT:

    return $timestamp;
}

# Used to read the .properties file which corresponds to the VNX probe for which this script is running.
#
# @return :
#   0 if Success
#   1 if Error
sub readProbePropertyFile {
    # Used to store the return code for this function.
    my $retVal = 0;

    $logObj->info("Reading probe instance properties file: $propFilePath");

    open(PROP_FILE, $propFilePath) or die $logObj->error("Not able to open the file: [$propFilePath].");

    # Loop through the content of .properties file.
    while (<PROP_FILE>) {
        # Remove the new line character.
        chomp;

        if ($_ =~ /^\s*UserName=(\S+)\s*$/) {
            # Look for Username.

            $username = $1;
        } elsif ($_ =~ /^\s*Password=(\S+)\s*$/) {
            # Look for Password.

            $vnxFileKeyEnc = $1;
        } elsif ($_ =~ /^\s*SiteName=(.+)\s*$/) {
            # Look for Site name.

            $siteName = $1;
        } elsif ($_ =~ /^IPAddress=(.+)\s*$/) {
            # Look for IP Address.

            $storageIP = $1;
        } else {
            # Current line is of no use. So ignore it.
        }
    }

    if ("" eq $username) {
        $logObj->error("Username not present in the properties file.");
        $retVal = 1;

        goto EXIT;
    }

    if ("" eq $vnxFileKeyEnc) {
        $logObj->error("vnxFileKeyEnc not present in the properties file.");
        $retVal = 1;

        goto EXIT;
    }

    if ("" eq $siteName) {
        $logObj->error("Site name is not present in the properties file.");
        $retVal = 1;

        goto EXIT;
    }
    
    if ("" eq $storageIP) {
        $logObj->error("Storage IP is not present in the properties file.");
        $retVal = 1;

        goto EXIT;
    }

EXIT:

    close(PROP_FILE);

    return $retVal;
}
