#!/bin/bash
#********************************************************************
#  FileName		: install.sh
#  Version		: 1.0.0.1     
#  Date			: 
#  Function List: bootloader and firmware Version load
#  History		:         
#  time         :  2023/10/7
#  version      :  1.0.0.1
#********************************************************************

uname -a | grep -i aarch64 1>/dev/null
if [ $? == 0 ]
then
 	expander_tool="./expanderToolForArm"
	\cp sg_tool_arm/libsgutils2-1.45.so.2 /usr/lib64/libsgutils2-1.45.so.2
	\cp sg_tool_arm/sg_ses /usr/bin/sg_ses
	chmod 777 /usr/bin/sg_ses    
	storcli="./storcli64_arm"
else
 	expander_tool="./expanderTool"
	\cp sg_tool_x86/libsgutils2.so.2 /usr/lib64/libsgutils2.so.2
	\cp sg_tool_x86/sg_ses /usr/bin/sg_ses
	chmod 777 /usr/bin/sg_ses
	storcli="./storcli64"
fi

#get work path
path=$(dirname "$0")
path="${path/\./$(pwd)}"
pwd="${path}"

cd "${pwd}"
#init 
rm "${pwd}/result.xml" 2>/dev/null
rm "${pwd}/work.log" 2>/dev/null
tool_type=""
dev_num=0
flag_fail=0

#column for result.xml
result=""
errorCode=""
description=""

#create result.xml for esight
create_resultXml()
{	
	echo "<?xml version=\"1.0\" encoding=\"utf-8\"?>" >"${pwd}/result.xml"
	echo "<ConfigureResult>" >>"${pwd}/result.xml"
	echo "    <FwUpgrade>" >>"${pwd}/result.xml"
	echo "        <Result>${result}</Result>" >>"${pwd}/result.xml"
	echo "        <ErrorCode>${errorCode}</ErrorCode>" >>"${pwd}/result.xml"
	echo "        <Description>${description}</Description>" >>"${pwd}/result.xml"
	echo "    </FwUpgrade>" >>"${pwd}/result.xml"
	echo "</ConfigureResult>" >>"${pwd}/result.xml"
}

# ***** 查询是否为expander环境以及对应expander版本 *****
get_fw_version()
{
	sgnem=`lsscsi -g | grep encl | grep -i Exp | awk '{print $8}'`
	if [[ $sgnem == '' ]];then
		echo "Can't find Expander device." | tee $SaveFile
		result="Fail"
		errorCode="8"
		description="Expander device not found"
		create_resultXml
		exit 8
	fi
	$expander_tool query $sgnem > getversion.txt
}

# ***** 获取expander芯片类型 *****
get_expander_type()
{
	exp=`lsscsi -g | grep encl | grep -i Exp | awk '{print $5}'`
	Ver=`lsscsi -g | grep encl | grep -i Exp | awk '{print $6}'`
	if [ $exp == '12Gx16' ] || [[ $exp == 12G16* ]];then
		echo "This is PM8053."
		exptype="8053"
	elif [ $exp == '12Gx28' ] || [[ $exp == 12G28* ]] && [[ $Ver == 1* ]];then
	    echo "This is PM8054."
		exptype="8054"
    elif [ $exp == '12Gx28' ] || [[ $exp == 12G28* ]] && [[ $Ver == 3* || $Ver == 4* ]];then
	    echo "This is PM8054+12."
		exptype="8054+12"
	elif [ $exp == '12Gx40' ] || [[ $exp == 12G40* ]];then
	    echo "This is PM8055."
		exptype="8055"
	else
	    echo "unknown expander type; please check it first." | tee $SaveFile
		result="Fail"
		errorCode="7"
		description="unknown expander type"
		create_resultXml
		exit 7
	fi
	$expander_tool query $sgnem > getexpandertype.txt
}

# ***** 校验升级文件的数据完整性 *****
package_verify()
{
	sha256sum boot_mips.bin >> $LocalCheckSum
    sha256sum istr_flash_pm$exptype.bin >> $LocalCheckSum
    sha256sum sxp_evbd_rom.bin >> $LocalCheckSum
    diff $ExpectCheckSum $LocalCheckSum > /dev/null
    if [ $? == 0 ]; then
        echo "File verification succeeded." | tee $SaveFile
    else
	    echo "verify update file failed." | tee $SaveFile
		result="Fail"
		errorCode="6"
		description="verify update file failed"
		create_resultXml
		exit 6
    fi
}

# ***** 升级expander固件 *****
fw_update()
{
	sgnem=`lsscsi -g | grep encl | grep -i Exp | awk '{print $8}'`
	echo "Update boot mips..."
	$expander_tool download=boot_mips.bin $sgnem > /dev/null
	if [[ $? != 0 ]];then
		echo "boot mips update failed!" | tee $SaveFile
		result="Fail"
		errorCode="5"
		description="boot mips update failed"
		create_resultXml
		exit 5
	fi
	echo "Update init string..."
	$expander_tool download=istr_flash_pm$exptype.bin $sgnem > /dev/null
	if [[ $? != 0 ]];then
		echo "istr update failed!" | tee $SaveFile
		result="Fail"
		errorCode="4"
		description="istr update failed"
		create_resultXml
		exit 4
	fi
	echo "Update firmware..."
	$expander_tool download=sxp_evbd_rom.bin $sgnem	> /dev/null
	if [[ $? != 0 ]];then
		echo "sxp evbd rom update failed!" | tee $SaveFile
		result="Fail"
		errorCode="3"
		description="sxp evbd rom update failed"
		create_resultXml
		exit 3
	fi
	echo "Update finished! Manually power off and then power on for the upgrade to take effect..."
	reboot_flag=1
}

# ***** 确保下次启动从升级分区进行启动 *****
boot_verify()
{
	current_boot_query=`$expander_tool query $sgnem | grep -i "Active Image" | awk '{print $3}'`
	current_boot_image=${current_boot_query: -1}
	echo "The current_boot_image is $current_boot_image"
	next_boot_query=`sg_ses -p 0x91 $sgnem  | grep -i 00 | awk '{print $6}'`
	next_boot_image=${next_boot_query: -1}
	echo "The next_boot_image is $next_boot_image"
	if [[ $current_boot_image == $next_boot_image ]];then
		echo "The next startup partition does not meet the expectation. A switchover is required." | tee $SaveFile
		$expander_tool download=sxp_evbd_rom.bin $sgnem > /dev/null
		if [[ $? != 0 ]];then
		    echo "Failed to switch partitions! Do not restart the system until the upgrade is successful." | tee $SaveFile
		    exit 2
	    fi
		echo "Partition switchover succeeded." | tee $SaveFile
	fi
}

# ***** 检查raid卡版本，不为LSI卡 5.13-5.06之间的版本 *****
check_raid_version()
{
	raid_version=`$storcli /c0 show |grep -i "fw version"|awk '{print $4}'| cut -d '.' -f 2`
	raid_version=$(expr "$raid_version" + 0 2> /dev/null)
	if [[ $raid_version == '' ]];then
		echo "The machine does not have an LSI RAID."
	elif [[ $raid_version -le 60 || $raid_version -ge 130 ]];then
		echo "This fw version is normal"
	elif [[ $raid_version -gt 60 && $raid_version -lt 130 ]];then
		echo "The expander backplane fw cannot be installed on the RAID controller card firmware version. Upgrade the RAID controller card fw version to 5.130.00-3059 or later." | tee $SaveFile
		result="Fail"
		errorCode="9"
		description="Failed to installed the expander backplane fw on the RAID controller card firmware version. Upgrade the RAID controller card fw version to 5.130.00-3059 or later."
		create_resultXml
		exit 9	
	fi
}

#######################################################
#main
#######################################################

LocalCheckSum="expcheck.txt"
ExpectCheckSum="ExpFilesha256sum.txt"
SaveFile="save.txt"
reboot_flag=0

rm -rf $SaveFile
rm -rf $LocalCheckSum
chmod +x *

case $1 in
	upgrade)
		# ***** check raid card version *****
		check_raid_version
		
		# ***** Get version from board *****
        get_fw_version
		
		# ***** Get expander type from board *****
        get_expander_type
		
		# ***** 校验升级文件和expander芯片是否匹配 ***** 
        ls | grep -i $exptype
        if [ $? != 0 ];then
 	        echo "upgrade file not match the expander. Download the correct package from http://support.huawei.com and perform the upgrade." | tee $SaveFile
		    result="Fail"
		    errorCode="1"
		    description="upgrade file not match the expander"
		    create_resultXml
		exit 1
        fi

		# ***** 对比MD5值，用于确认升级文件的数据完整性 *****
        package_verify
		
		# ***** 升级expander固件 ***** 
        fw_update
		
		# ***** 启动分区校验 *****
        boot_verify
		
		# ***** 提示重启后生效 *****
        if [[ $reboot_flag == 1 ]];then
	        sync
            echo "Manually power off and then power on for the upgrade to take effect."
		    result="OK"
		    errorCode="0"
		    description="upgrade sussess"
		    create_resultXml
		    exit 0;
        fi		
	;;
	query)
		get_fw_version
		exp=`lsscsi -g | grep encl | grep -i Exp | awk '{print $5}'`
		Ver=`lsscsi -g | grep encl | grep -i Exp | awk '{print $6}'`
		if [ $exp == '12Gx16' ] || [[ $exp == 12G16* ]];then
			exptype="8053"
		elif [ $exp == '12Gx28' ] || [[ $exp == 12G28* ]] && [[ $Ver == 1* ]];then
			exptype="8054"
		elif [ $exp == '12Gx28' ] || [[ $exp == 12G28* ]] && [[ $Ver == 3* || $Ver == 4* ]];then
			exptype="8054+12"
		elif [ $exp == '12Gx40' ] || [[ $exp == 12G40* ]];then
			exptype="8055"
		else
			echo "unknown expander type; please check it first." | tee $SaveFile
			result="Fail"
			errorCode="7"
			description="unknown expander type"
			create_resultXml
			exit 7
		fi
		echo PM$exptype $Ver $sgnem
	;;
	*)
	;;
esac

exit 0













