# -*- coding: UTF-8 -*-

from java.lang import Exception as JException

from cbb.frame.adapter.resourceService import ResrouceFactory
from cbb.frame.base import baseUtil
from cbb.frame.context import contextUtil

CTRL_A_PLAN = ['A', 'C']  # a平面控制器ID
CTRL_B_PLAN = ['B', 'D']  # b平面控制器ID


# 前端链路冗余检查
def execute(dataDict):
    return HostPathCheck(dataDict).execute()


def HandleException(fn):
    def inner(self):
        retrytimes = 3
        while retrytimes > 0:
            try:
                fn(self)
                return True
            except Exception as e:
                self.logger.error("script exception:%s" % unicode(e))
                return False
            except JException, e:
                self.logger.error("java excption:%s" % unicode(e))
                retrytimes = retrytimes - 1
        return False

    return inner


# 主机兼容性检查类
class HostPathCheck():
    def __init__(self, dataDict):
        self.errorMsg = ""
        self.dataDict = dataDict
        self.results = []
        self.allCliRet = ""
        self.hosts = []
        self.errHosts = []
        self.allHBAInfo = {}

    @HandleException
    def queryHosts(self):
        # /rest/[SN]/host
        # filter  "ISADD2HOSTGROUP"为true,"INITIATORNUM">0
        resrouceFactory = ResrouceFactory(self.dataDict)
        allHosts = resrouceFactory.executeInterface("getHost")
        self.allCliRet += ShareData.formatRestData("getHost", allHosts)
        self.logger.info("host num=%s" % len(allHosts))
        if allHosts:
            for host in allHosts:
                # 没有“INITIATORNUM”字段时，默认值置为大于0的值，继续后面的检查
                initiatorNum = int(host.get("INITIATORNUM", 1))
                if host["ISADD2HOSTGROUP"] == "true" and initiatorNum > 0:
                    self.hosts.append(host)

    # 检查执行入口
    def execute(self):
        self.logger = contextUtil.getLogger(self.dataDict)
        self.lang = contextUtil.getLang(self.dataDict)
        self.resource = self.dataDict.get("resource")
        try:
            # 主机查询失败，未检查
            ok = self.queryHosts()
            if not ok:
                suggesstion = baseUtil.getMsg(self.lang, "query.result.abnormal")
                return (4, suggesstion)

            # 没有映射的主机，检查通过
            if len(self.hosts) == 0:
                self.logger.info("no mapped host,the check is passed.")
                suggesstion = baseUtil.getMsg(self.lang, "frontlink.result.pass")
                return (3, suggesstion)

            for host in self.hosts:
                instance = SingleHostPathCheck(self.dataDict, host)
                result = instance.execute()
                ret = dict(hostId=host["ID"], status=result.status, message=result.message,
                           cliRet="\n".join(instance.allCliRet))
                self.results.append(ret)
            state, suggesstion = self.mergeResult()
            return (state, suggesstion)

        except Exception as exception:
            self.logger.error(str(exception))
            suggesstion = baseUtil.getMsg(self.lang, "query.result.abnormal")
            return (4, suggesstion)

    def mergeResult(self):
        allHostCliRet = map(lambda x: x["cliRet"], self.results)
        allHostCliRet = filter(lambda x: x != None, allHostCliRet)
        self.allCliRet += "\n".join(allHostCliRet)

        # 将结果按状态分类
        resultDict = {}
        hostIdList = []
        for status in CheckStatus.getAllItems():
            resultDict[status] = []
        for result in self.results:
            resultDict[result["status"]].append(result)
            if result["status"] == CheckStatus.NOPASS:
                hostIdList.append(result['hostId'])

        # 设置状态
        cannotEvaluResultNum = map(lambda status: len(resultDict[status]), CheckStatus.getCannotEvaluItems())
        if len(resultDict[CheckStatus.NOPASS]) > 0:
            status = 4
        elif len(resultDict[CheckStatus.NO_ONLINE_INITIATOR]) > 0:
            status = 4
        elif reduce(lambda x, y: x + y, cannotEvaluResultNum) > 0:
            status = 4
        else:
            status = 3

        # 设置提示信息
        message = ""
        if status != 3:
            for key in CheckStatus.getAllNotPassItems():
                if len(resultDict[key]) > 0:
                    hostIds = map(lambda x: x['hostId'], resultDict[key])
                    hostIdStr = self.formatOutput(hostIds)
                    message += baseUtil.getMsg(self.lang, "frontlink.result.%s" % key, hostIdStr)
        else:
            message = baseUtil.getMsg(self.lang, "frontlink.result.pass")
        return status, message

    # 将主机按每5个换行显示
    def formatOutput(self, results):
        ret = ""
        for i in range(len(results)):
            if i % 5 == 0 and i / 5 != 0:
                ret += "\n    "
            ret += results[i]
            ret += ", "
        ret = ret[:-2] if ret.endswith(", ") else ret
        return ret


# 单个主机兼容性检查类
class SingleHostPathCheck():
    def __init__(self, dataDict, host):
        self.dataDict = dataDict
        self.hostId = host["ID"]
        self.host = host
        self.lang = contextUtil.getLang(dataDict)
        self.logger = contextUtil.getLogger(self.dataDict)
        self.allCliRet = ["\n=========the host(%s) origin info start=========" % self.hostId]

    def execute(self):
        try:
            context = dict(host=self.host,
                           dataDict=self.dataDict,
                           allCliRet=self.allCliRet)
            flow = InitiatorCheck(self.hostId)
            while not isinstance(flow, FlowEnd):
                self.logger.info("flow start, step=%s" % flow.__class__)
                flow.setContext(context)
                ok = flow.queryInfo()
                if not ok:
                    self.logger.info("quey info failed.")
                    flow = FlowEnd(CheckStatus.QUERY_INFO_ERROR)
                    break
                flow = flow.next()
                self.logger.info("flow next, step=%s" % flow.__class__)
            return flow
        except Exception as e:
            self.logger.error("single host compatibility:%s" % unicode(e))
            return FlowEnd(CheckStatus.QUERY_INFO_ERROR)
        finally:
            self.allCliRet.append("\n=========the host(%s) origin info end=========" % self.hostId)


class FlowEnd():
    def __init__(self, status, message="", data=None):
        self.status = status
        self.message = message
        self.data = data


class CheckStatus():
    PASS = "pass"
    NOPASS = "nopass"
    QUERY_INFO_ERROR = "queryInfoError"
    NO_ONLINE_INITIATOR = "noOnlineInitiator"

    @staticmethod
    def getAllItems():
        return ["pass", "nopass", "queryInfoError", "noOnlineInitiator"]

    @staticmethod
    def getAllNotPassItems():
        return ["nopass", "queryInfoError", "noOnlineInitiator"]

    @staticmethod
    def getCannotEvaluItems():
        return ["queryInfoError"]


class ShareData():
    def __init__(self):
        self.context = {}
        self.dataDict = {}

    def setContext(self, context):
        self.context = context
        self.host = context["host"]
        self.dataDict = context["dataDict"]
        self.logger = contextUtil.getLogger(self.dataDict)
        self.lang = contextUtil.getLang(self.dataDict)
        self.resrouceFactory = ResrouceFactory(self.dataDict)

    def appendCliRet(self, cliRet):
        self.context["allCliRet"].append(cliRet)

    def appendRestRet(self, uri, datas):
        dataStr = self.formatRestData(uri, datas)
        self.context["allCliRet"].append(dataStr)

    @staticmethod
    def formatRestData(uri, datas):
        dataStr = ""
        if datas:
            if not isinstance(datas, list):
                return "\nRest Uri:%s\nResponse:[%s]" % (uri, str(datas))
            for data in datas:
                dataStr += str(data)
                dataStr += "\n"
        return "\nRest Uri:%s\nResponse:[%s]" % (uri, dataStr)


class InitiatorCheck(ShareData):
    def __init__(self, hostId):
        ShareData.__init__(self)
        self.hostId = hostId
        self.initators = []
        self.noRunningInitators = []

    @HandleException
    def queryInfo(self):
        '''查询在线状态的启动器'''
        fcInitiators = self.resrouceFactory.executeInterface("getFcInitiator", self.hostId)
        self.initators.extend(fcInitiators)
        self.appendRestRet("getFcInitiator", fcInitiators)

        iscsiInitiatorsRet = self.resrouceFactory.executeInterface("getIscsiInitiator", self.hostId)
        self.initators.extend(iscsiInitiatorsRet)
        self.appendRestRet("getIscsiInitiator", iscsiInitiatorsRet)

        ibInitiatorsRet = self.resrouceFactory.executeInterface("getIbInitiator", self.hostId)
        self.initators.extend(ibInitiatorsRet)
        self.appendRestRet("getIbInitiator", ibInitiatorsRet)
        return True

    def next(self):
        # 无启动器，检查通过
        if self.initators is None or len(self.initators) == 0:
            return FlowEnd(CheckStatus.PASS)
        else:
            self._filterRunningInitators()
            if len(self.noRunningInitators) > 0:
                return FlowEnd(CheckStatus.NO_ONLINE_INITIATOR)
            return LunNumCheck(self.hostId)

    def _filterRunningInitators(self):
        for initator in self.initators:
            if initator.get("RUNNINGSTATUS") != '27':
                self.noRunningInitators.append(initator)
        return


class LunNumCheck(ShareData):
    def __init__(self, hostId):
        ShareData.__init__(self)
        self.hostId = hostId
        self.lunNum = 0

    @HandleException
    def queryInfo(self):
        lunNumRet = self.resrouceFactory.executeInterface("getHostLunCount", self.hostId)
        self.lunNum = lunNumRet.get("COUNT", 0)
        self.appendRestRet("getHostLunCount", lunNumRet)
        return True

    def next(self):
        if self.lunNum == 0:
            return FlowEnd(CheckStatus.PASS)
        else:
            return HostLinkCheck(self.hostId)


class HostLinkCheck(ShareData):
    def __init__(self, hostId):
        ShareData.__init__(self)
        self.hostId = hostId
        self.hostLinks = None
        self.hostLinksDict = {}

    @HandleException
    def queryInfo(self):
        # /rest/[SN]/host_link?TYPE=255&PARENTID=[host_id]
        datas = self.resrouceFactory.executeInterface("getHostLink", self.hostId)
        self.appendRestRet("getHostLink", datas)
        if datas and len(datas) != 0:
            self.hostLinks = datas
        return True

    def _checkRules(self, ctrlIdDict):
        for engine, ctrlList in ctrlIdDict.items():
            alist = list(set(ctrlList).intersection(set(CTRL_A_PLAN)))
            blist = list(set(ctrlList).intersection(set(CTRL_B_PLAN)))
            if len(alist) == 0 or len(blist) == 0:
                return False
        return True

    def next(self):
        '''
        获取每条信息的CTRL_ID字段，获取到的信息格式为，XA、XB、XC、XD（X为引擎编号，可为0~7）。
        所有信息都获取完后，以引擎号进行分组，分别判断是否同时包含A、B，或者A、D，或者C、B，
        或者C、D，如果所有引擎链路均满足上述规则，说明主机到阵列路径冗余，否则不冗余。
        '''
        if self.hostLinks is None or len(self.hostLinks) == 0:
            return FlowEnd(CheckStatus.NOPASS)
        for hostLink in self.hostLinks:
            # 只判断在线的链路
            if hostLink["RUNNINGSTATUS"] != "27":
                continue
            initiatorType = hostLink.get("INITIATOR_TYPE")
            ctrlId = hostLink.get("CTRL_ID")
            parentId = hostLink.get("PARENTID")
            hostLinkList = self.hostLinksDict.get(initiatorType, [])
            if hostLinkList == []:
                self.hostLinksDict[initiatorType] = hostLinkList
            hostLinkList.append(ctrlId)
        self.logger.info("HostLinkCheck parse host link,hostLinksDict:%s" % str(self.hostLinksDict))

        ctrlIdDict = {}
        for typeKey, ctrlIdList in self.hostLinksDict.items():
            ctrlIdList = list(set(ctrlIdList))
            for ctrlId in ctrlIdList:
                engine = ctrlId[0]
                ctrl = ctrlId[1]
                idList = ctrlIdDict.get(engine, [])
                if len(idList) == 0:
                    ctrlIdDict[engine] = idList
                idList.append(ctrl)
            ok = self._checkRules(ctrlIdDict)
            if ok == False:
                self.logger.info("Host(%s) have not redundant link." % str(self.hostId))
                return FlowEnd(CheckStatus.NOPASS)
        self.logger.info("Host(%s) have redundant link." % str(self.hostId))
        return FlowEnd(CheckStatus.PASS)
