# -*- coding: UTF-8 -*-
from cbb.frame.cli import cliUtil
from cbb.frame.base.exception import UnCheckException


RISK_SOFTWARE_VERSION = ("6.0.1",)

BLACK_HOT_PATCH_VERSION = (
    "SPH0",
    "SPH1",
    "SPH2",
    "SPH3",
    "SPH4",
    "",
    "--",
)

HOT_SPARE_STRATEGY = {"Low": 1, "High": 2, "None": 0}


class CliDomainConfigCheck(object):
    def __init__(self, cli, lang, logger, dest_version="", dest_patch=""):
        self.cli = cli
        self.lang = lang
        self.logger = logger
        self.dest_patch = dest_patch
        self.dest_version = dest_version[0:5]
        self.all_cli_ret = list()
        self.err_msg_notice = list()
        self.err_msg_notice_no_smart = list()

    def execute_check(self):
        if not self.is_risk_version():
            return True

        storage_pool_disk_domain_id = self.get_storage_pool_disk_domain_id()

        (
            multi_engine_disk_domain_id,
            hot_pre_disk_num_dict,
        ) = self.get_multi_engine_disk_domain_id_and_hot_disk_num(
            storage_pool_disk_domain_id
        )

        # 存在硬盘域归属于多个引擎，则继续检查，否则检查结果为通过；
        if len(multi_engine_disk_domain_id) < 1:
            return True
        system_logic_engine_num = self.get_logic_engine_num()

        # 系统仅存在一个逻辑引擎，则检查结果为通过，否则继续检查；
        if system_logic_engine_num < 2:
            return True

        smart_disk_enclosure_id_list = self.get_smart_enclosure_id()
        disk_domain_disk_info = self.get_smart_enclosure_and_member_disk_num(
            multi_engine_disk_domain_id, smart_disk_enclosure_id_list
        )

        self.check_disk_info(disk_domain_disk_info, hot_pre_disk_num_dict)
        return all(
            [
                len(self.err_msg_notice) == 0,
                len(self.err_msg_notice_no_smart) == 0,
            ]
        )

    def check_disk_info(self, disk_domain_disk_info, hot_pre_disk_num_dict):
        """
        未使用智能框，当硬盘域总盘数小于等于（25+热备盘数）或大于（25+热备盘数）*2，通过
        使用智能框，当硬盘域总盘数大于（25+热备盘数）*2时则检查结果为通过，否则检查结果为不通过。
        :param disk_domain_disk_info: 是否使用智能框，硬盘域ID，成员盘数
        :param hot_pre_disk_num_dict: 热备盘数字典
        :return:
        """
        for (use_smart_flag, domain_id, disk_num,) in disk_domain_disk_info:
            if all(
                [
                    use_smart_flag is True,
                    (disk_num <= (25 + hot_pre_disk_num_dict[domain_id]) * 2),
                ]
            ):
                self.err_msg_notice.append(
                    (
                        domain_id,
                        disk_num,
                        (25 + hot_pre_disk_num_dict[domain_id]) * 2,
                    )
                )

            if all(
                [
                    use_smart_flag is False,
                    not (
                        disk_num <= (25 + hot_pre_disk_num_dict[domain_id])
                        or disk_num
                        > ((25 + hot_pre_disk_num_dict[domain_id]) * 2)
                    ),
                ]
            ):
                # 收集每个多引擎硬盘域的错误提示
                self.err_msg_notice_no_smart.append(
                    (
                        domain_id,
                        disk_num,
                        (25 + hot_pre_disk_num_dict[domain_id]),
                        (25 + hot_pre_disk_num_dict[domain_id]) * 2,
                    )
                )

    def is_risk_version(self):
        """
        巡检工具、备件更换评估工具：
        1 若软件版本不为6.0.1，则检查结果为通过，否则继续检查；
        2 若软件版本是6.0.1且热补丁版本为SPH5或之后的版本，则检查结果为通过，否则继续检查；

        升级评估工具：
        1 若当前软件版本为6.0.1，且当前热补丁版本为SPH5之前的版本，则继续检查，否则检查结果为通过；
        2 若目标软件版本为6.0.1.SPH5之前的版本，则继续检查，否则检查结果为通过；
        :return: False 非风险， True 风险版本
        """
        (
            __,
            software_version,
            hot_patch_version,
            cli_ret,
        ) = cliUtil.get_system_version_with_ret(self.cli, self.lang)
        self.all_cli_ret.append(cli_ret)
        if any(
            [
                # 若软件版本不为6.0.1，则检查结果为通过，否则继续检查/非评估的目标版本
                all(
                    [
                        software_version not in RISK_SOFTWARE_VERSION,
                        not self.dest_version,
                    ]
                ),
                # 升级评估 当前版本非风险，目标版本非风险。
                all(
                    [
                        software_version not in RISK_SOFTWARE_VERSION,
                        self.dest_version,
                        self.dest_version not in RISK_SOFTWARE_VERSION,
                    ]
                ),
                # 若当前软件版本为6.0.1，且当前热补丁版本为SPH5之前的版本，则继续检查，否则检查结果为通过；
                all(
                    [
                        software_version in RISK_SOFTWARE_VERSION,
                        hot_patch_version not in BLACK_HOT_PATCH_VERSION,
                    ]
                ),
                # 升级评估第二点
                all(
                    [
                        software_version not in RISK_SOFTWARE_VERSION,
                        self.dest_version,
                        self.dest_version in RISK_SOFTWARE_VERSION,
                        self.dest_patch not in BLACK_HOT_PATCH_VERSION,
                    ]
                ),
            ]
        ):
            return False
        return True

    def get_storage_pool_disk_domain_id(self):
        """
        步骤3 执行命令：show storage_pool general，查询存储池所在硬盘域ID信息
        :return: disk_domain_id_list 硬盘域ID列表
        """
        cmd = "show storage_pool general"
        flag, ret, err_msg = cliUtil.excuteCmdInCliMode(
            self.cli, cmd, True, self.lang
        )

        self.all_cli_ret.append(ret)

        disk_domain_id_list = list()

        if flag is not True:
            raise UnCheckException(err_msg)

        ret_info_list = cliUtil.getHorizontalCliRet(ret)
        for line in ret_info_list:
            poolID = line.get("Disk Domain ID")
            disk_domain_id_list.append(poolID)

        return list(set(disk_domain_id_list))

    def get_multi_engine_disk_domain_id_and_hot_disk_num(self, disk_domain_id):
        """
        步骤4 执行命令：show disk_domain general disk_domain_id=?，查
        询并记录归属多个引擎（Controller enclosure)的
        硬盘域ID列表及其配置的热备盘数Hot Spare Strategy，Low转换为1,
        High转换为2，None转换为0，其它数字直接使用）
        :param disk_domain_id: 步骤3中得到的硬盘域ID
        :return: multi_engine_disk_domain_id, 多引擎硬盘域ID列表
                    hot_pre_disk_num_dict 硬盘域：热备盘数字典
        """
        multi_engine_disk_domain_id = list()
        hot_pre_disk_num_dict = dict()
        for domain_id in disk_domain_id:
            cmd = "show disk_domain general disk_domain_id={}".format(
                domain_id
            )
            flag, ret, err_msg = cliUtil.excuteCmdInCliMode(
                self.cli, cmd, True, self.lang
            )

            self.all_cli_ret.append(ret)

            if flag is not True:
                raise UnCheckException(err_msg)

            ret_info_list = cliUtil.getVerticalCliRet(ret)

            for line in ret_info_list:

                # 获取硬盘域的引擎列表
                ctrl_enclosure = line.get("Controller enclosure", "").split(
                    ","
                )

                if len(ctrl_enclosure) < 2:
                    continue

                # 如果引擎列表中引擎数多余一个，则记录下这个硬盘域ID，及热备盘数
                multi_engine_disk_domain_id.append(domain_id)
                # 获取热备盘字段
                hot_spare_strategy_str = line.get("Hot Spare Strategy")
                # Low表示1，High表示2，None表示0，其他字符串数字转换成数字
                if hot_spare_strategy_str in HOT_SPARE_STRATEGY:
                    hot_pre_disk_num = HOT_SPARE_STRATEGY.get(
                        hot_spare_strategy_str, 0
                    )
                elif hot_spare_strategy_str.isdigit():
                    hot_pre_disk_num = int(hot_spare_strategy_str)
                else:
                    hot_pre_disk_num = 0
                hot_pre_disk_num_dict[domain_id] = hot_pre_disk_num

        # 返回多引擎硬盘域ID和对应热备盘数
        return multi_engine_disk_domain_id, hot_pre_disk_num_dict

    def get_logic_engine_num(self):
        """
        第5、6、7、8步，进入developer模式，进入debug模式，执行sys showcls命令，查询系统逻辑引擎个数
        退出debug模式
        :return: True, len(list(set(logic_engine))), "" 是否查询成功，逻辑引擎数，返回信息
        """
        logic_engine = []
        #  进入developer，在进入debug，执行sys showcls命令，退出debug，返回cli
        flag, ret, err_msg = cliUtil.executeCmdInDebugMode(
            self.cli, "sys showcls", True, self.lang
        )
        self.all_cli_ret.append(ret)

        if flag is not True:
            raise UnCheckException(err_msg)

        ret_dict_list = cliUtil.getHorizontalCliRet(ret)
        for ret_dict in ret_dict_list:
            logic_engine.append(ret_dict.get("group"))
        return len(list(set(logic_engine)))

    def get_smart_enclosure_id(self):
        """
        步骤9 执行命令：show enclosure，查询系统框信息，获取智能框（type项中包含smart字段）ID列表
        :return: smartDiskEnclosureIdList 智能框ID列表
        """
        cmd = "show enclosure"
        flag, ret, err_msg = cliUtil.excuteCmdInCliMode(
            self.cli, cmd, True, self.lang
        )

        self.all_cli_ret.append(ret)
        if flag is not True:
            raise UnCheckException(err_msg)

        smart_disk_enclosure_id_list = list()
        ret_lines_list = cliUtil.getHorizontalCliRet(ret)

        for line in ret_lines_list:
            enclosure_type = line["Type"]
            enclosure_id = line["ID"]
            if "smart" in enclosure_type:
                smart_disk_enclosure_id_list.append(enclosure_id)

        return smart_disk_enclosure_id_list

    def get_smart_enclosure_and_member_disk_num(
        self, disk_domain_id_list, smart_disk_enclosure_id_list
    ):
        """
        步骤10 执行命令：show disk in_domain disk_domain_id=?，（步骤4中查询到的硬盘域ID列表）
        查询是否使用智能框，并统计硬盘域的成员盘个数。

        :param disk_domain_id_list:  步骤4中查询到的多引擎硬盘域ID
        :param smart_disk_enclosure_id_list:  步骤9中查询到的智能框ID列表
        :return: smart_enclosure_and_member_disk_num 是否使用智能框，硬盘域ID，成员盘数
        """
        smart_enclosure_and_member_disk_num = list()
        for disk_domain_id in disk_domain_id_list:
            cmd = "show disk in_domain disk_domain_id=%s" % disk_domain_id
            flag, ret, msg = cliUtil.excuteCmdInCliMode(
                self.cli, cmd, True, self.lang
            )

            self.all_cli_ret.append(ret)
            if flag is not True:
                raise UnCheckException(msg)

            mem_disk_id_list = []
            diskList = cliUtil.getHorizontalCliRet(ret)
            for disk in diskList:
                mem_disk_id_list.append(disk.get("ID").split(".")[0])

            smart_enclosure_and_member_disk_num.append(
                self.check_smart_enclosure_and_member_disk_num(
                    smart_disk_enclosure_id_list,
                    mem_disk_id_list,
                    disk_domain_id,
                )
            )

        return smart_enclosure_and_member_disk_num

    def check_smart_enclosure_and_member_disk_num(
        self, smart_disk_enclosure_id_list, mem_disk_id_list, disk_domain_id
    ):
        """
        当前硬盘域是否使用了智能硬盘框的盘
        :param smart_disk_enclosure_id_list: 智能硬盘框ID
        :param mem_disk_id_list: 成员盘
        :param disk_domain_id: 硬盘域ID
        :return: flag True: 使用了智能框 False 未使用
                    disk_domain_id: 硬盘域ID
                    成员盘数量
        """
        for smart_disk_enclosure_id in smart_disk_enclosure_id_list:
            for mem_disk_id in mem_disk_id_list:
                if str(smart_disk_enclosure_id) == str(mem_disk_id):
                    return True, disk_domain_id, len(mem_disk_id_list)

        return False, disk_domain_id, len(mem_disk_id_list)
