# -*- coding: UTF-8 -*-
"""
@desc: 内存资源检查
@date: 2021.2.4
"""


from cbb.frame.checkitem.base_dsl_check import BaseCheckItem, CheckStatus
from cbb.frame.adapter import replace_adapter
from cbb.frame.ruleConfig.common import getMsg
from cbb.frame.dsl.adapter import NEED_NEW_CTRL_CONN
from cbb.frame.cli import cliUtil
from cbb.common.query.software.cliUtil import get_patch_value

MEMORY_USAGE_THRESHOLD = 97


class MemoryLeakageCheck(BaseCheckItem):
    def __init__(self, tool_context):
        super(MemoryLeakageCheck, self).__init__(tool_context)
        self.all_ret_list = []
        self.product_mode = tool_context["product_mode"]
        self.p_patch = ""
        self.p_version = tool_context["product_version"]
        self.actual_version = ''
        self.error_usage_list = []
        self.tool_type = tool_context.get("tool_type")
        self.context[NEED_NEW_CTRL_CONN] = False
        self.risk_version_all = {
            "V500R007C70SPC100": 999,
            "V500R007C70SPC200": 999,
            "6.1.0": 3,
        }
        self.risk_version_v_six = {
            "6.1.0": "SPH3",
        }
        self.risk_product_mode_v_six = [
            "OceanStor Dorado 8000 V6",
            "OceanStor Dorado 18000 V6",
            "OceanStor Dorado 6800 V6",
            "OceanStor Dorado 18500 V6",
            "OceanStor Dorado 18800 V6",
            "OceanStor Dorado 18800K V6",
        ]
        self.risk_product_mode_ocean_store = [
            "6810 V5",
            "18510 V5",
            "18810 V5",
            "6810F V5",
            "18510F V5",
            "18810F V5",
            "6800 V5",
            "18500 V5",
            "18800 V5",
        ]

    def execute(self):
        self.logger.info("tool type: {}".format(self.tool_type))
        self.dsl("exec_cli 'show system general'")
        (
            flag,
            self.actual_version,
            self.p_patch,
            ret,
        ) = replace_adapter.get_system_version_with_ret(self.context)
        self.add_origin_info(ret)
        if flag is not True:
            self.logger.info("query version error. nocheck")
            return CheckStatus.NOCHECK, self.get_msg("query.result.abnormal")

        if self.tool_type == "inspect":
            return self.check_in_inspect_tool()

        return self.check_in_patch_tool()

    def check_in_patch_tool(self):
        """
        检查补丁工具
        :return:
        """
        self.logger.info("actual_version: {}".format(self.actual_version))
        if not self.is_risk_product_mode_patch_tool():
            return CheckStatus.PASS, ""
        if not self.is_high_share_sas_interface():
            return CheckStatus.PASS, ""
        if self.actual_version not in self.risk_version_v_six:
            return CheckStatus.PASS, ""
        self.check_memory_too_high()
        if self.error_usage_list:
            return CheckStatus.NOTPASS, getMsg(
                self.lang,
                "memory.leakage.not.pass.patch",
                ", ".join(self.error_usage_list)
            )
        return CheckStatus.PASS, ""

    def check_in_inspect_tool(self):
        """
        检查巡检工具
        :return:
        """
        self.logger.info("actual_version: {}".format(self.actual_version))

        if not self.is_risk_product_mode_inspect_tool():
            return CheckStatus.PASS, ""

        if not self.is_high_share_sas_interface():
            return CheckStatus.PASS, ""

        if self.actual_version not in self.risk_version_all:
            return CheckStatus.PASS, ""

        target_patch = self.risk_version_all.get(self.actual_version)

        if target_patch > get_patch_value(self.p_patch):
            self.dsl("exec_cli 'show controller general'", )

            return CheckStatus.NOTPASS, getMsg(
                self.lang, "memory.leakage.not.pass.inspect"
            )

        return CheckStatus.PASS, ""

    def check_memory_too_high(self):
        """
        检查内存占用是否过高
        :return: True: 内存过高，False: 内存不高
        """
        controller_info_ret = self.dsl("exec_cli 'show controller general'",)
        ret_list = cliUtil.getVerticalCliRet(controller_info_ret)

        error_msg = "{}: {}"
        for ret_info in ret_list:
            memory_usage = int(ret_info.get("Memory Usage(%)", "0"))
            controller_id = ret_info.get("Controller")
            if memory_usage > MEMORY_USAGE_THRESHOLD:
                self.error_usage_list.append(
                    error_msg.format(controller_id, memory_usage)
                )

    def is_risk_product_mode_patch_tool(self):
        return self.product_mode in self.risk_product_mode_v_six

    def is_risk_product_mode_inspect_tool(self):
        return (
            self.product_mode
            in self.risk_product_mode_ocean_store
            + self.risk_product_mode_v_six
        )

    def is_high_share_sas_interface(self):
        """
        是否是sas 高端共享卡
        :return:
        """
        interface_info_list = self.dsl(
            "exec_cli 'show interface_module' | horizontal_parser",
        )
        for item in interface_info_list:
            if "SAS" in item.get("Model", ""):
                return True
        return False
