# -*- coding: UTF-8 -*-

from cbb.frame.cli import cliUtil
from cbb.frame.context import contextUtil
from cbb.frame.base import baseUtil
from cbb.frame.ruleConfig import config
from cbb.business.operate.fru.common import common
from cbb.business.operate.fru.common import BaseFactory


class CheckMultiBondItem:

    def __init__(self, context, select_id):
        self.context = context
        self.logger = contextUtil.getLogger(context)
        self.cli = contextUtil.getCli(context)
        self.lang = contextUtil.getLang(context)
        self.logger = contextUtil.getLogger(context)
        self.sug_patch = ""
        self.select_id = select_id

    def get_product_model(self):
        return str(contextUtil.getDevObj(self.context).get("type"))

    def is_checked_product_model(self):
        """
        检查是否为涉及问题的型号
        :return: 返回检查结果:True 需要检查；False不检查
        """
        product_model = self.get_product_model()
        if not baseUtil.isDoradoDev(product_model):
            return True
        return False

    def check_version(self):
        """ 检查是否为涉及问题版本
        :return: 返回检查结果，True 需要检查；
                False 存在错误信息则异常，否则无需检查
        """
        flag, sys_version, hotpatch_version = \
            cliUtil.getSystemVersion(self.cli, self.lang)
        if not flag:
            raise Exception("get version info failed.")

        # Dorado NAS V3系列，避免后续V6影响
        if baseUtil.isDoradoNAS(self.get_product_model()) \
                and sys_version.startswith("V3"):
            return True

        patch_info = config.BOND_VERSION_MAP.get(sys_version, "")
        if patch_info:
            start_version = patch_info.get("start_version")
            self.sug_patch = patch_info.get("sugguest_patch")  # 用于错误消息
            if hotpatch_version == "--" or (start_version and
                                            hotpatch_version < start_version):
                return True

            white_list = patch_info.get("white_list")
            if white_list and hotpatch_version not in white_list:
                return True

            return False

        if sys_version.startswith(
                "V3") and "V300R003C10" <= sys_version <= "V300R006C60":
            return True

        if sys_version.startswith(
                "V5") and "V500R007C00" <= sys_version <= "V500R007C60SPC200":
            return True

        return False

    def is_same_intf_port(self, port_list):
        """ 判断所有端口是否是同一接口模块的端口
        :param port_list: 端口位置列表
        :return: 返回结果
        """

        intf_set = set()
        for port in port_list:
            intf = port[:port.rfind(".")]
            intf_set.add(intf)
        if len(intf_set) > 1:
            return False
        return True

    def get_bond_port(self):
        """ 获取绑定端口信息
        :return:
        """
        checked_bond = {}
        cmd = "show bond_port"
        flag, cli_ret, err_msg = \
            cliUtil.excuteCmdInCliMode(self.cli, cmd, True, self.lang)
        if flag is not True:
            return False, checked_bond, cli_ret, err_msg

        if cliUtil.queryResultWithNoRecord(cli_ret):
            return True, checked_bond, cli_ret, ""

        checked_bond = {}
        bond_port_list = cliUtil.getHorizontalNostandardCliRet(cli_ret)
        for bond in bond_port_list:
            p_list = bond.get("Port ID List", "").split(",")
            if not self.is_same_intf_port(p_list):
                p_name = bond.get("Name")
                checked_bond[p_name] = p_list
        return True, checked_bond, cli_ret, ""

    def get_vlan_port(self):
        """ 获取vlan端口信息
        :return:
        """

        vlan_port_list = set()
        cmd = "show vlan general"
        flag, cli_ret, err_msg = \
            cliUtil.excuteCmdInCliMode(self.cli, cmd, True, self.lang)
        if flag is not True:
            return False, vlan_port_list, cli_ret, err_msg

        if cliUtil.queryResultWithNoRecord(cli_ret):
            return True, vlan_port_list, cli_ret, ""

        vlan_list = cliUtil.getHorizontalNostandardCliRet(cli_ret)
        for vlan in vlan_list:
            vlan_port_list.add(vlan.get("Port ID"))

        return True, vlan_port_list, cli_ret, ""

    def get_multi_bond_port(self):
        """ 检查绑定端口是否配置vlan
        :return:
        """

        multi_bond_posts = {}
        # 获取绑定端口信息
        flag, checked_bond, cli_ret, err_msg = self.get_bond_port()
        if not flag:
            return False, multi_bond_posts

        if not checked_bond:
            return True, multi_bond_posts

        # 获取vlan端口信息
        flag, vlan_port_list, cli_ret, err_msg = self.get_vlan_port()
        if not flag:
            return False, multi_bond_posts

        if not vlan_port_list:
            return True, multi_bond_posts

        # 检查绑定端口是否配置vlan
        for bond in checked_bond:
            if bond in vlan_port_list:
                multi_bond_posts[bond] = checked_bond.get(bond)

        return True, multi_bond_posts

    def get_multi_bond_4_intf(self, intf_id):
        """ 获取当前接口卡上存在跨卡绑定的bond
        :param intf_id: 接口卡ID，如：CTE0.A3, CTE0.A4
        :return:
        """
        err_bonds = []
        flag, multi_bond_posts = self.get_multi_bond_port()
        if not flag:
            return False, err_bonds

        for bond in multi_bond_posts:
            ports = multi_bond_posts.get(bond)
            for port in ports:
                if intf_id in port:
                    err_bonds.append(bond)

        return True, err_bonds

    def check_model_version(self):
        """
        检查设备型号和版本号
        :return: True 需求检查；False 无需继续
        """

        if not self.is_checked_product_model():
            return True

        return self.check_version()

    def is_fault_intf(self):
        sel_row = BaseFactory.json.toDict(
            self.context.get(self.select_id))
        intf_id = sel_row.get("location")
        cmd = "show interface_module interface_module_id=%s" % intf_id
        flag, cli_ret, err_msg = \
            cliUtil.excuteCmdInCliMode(self.cli, cmd, True, self.lang)
        intfs = cliUtil.getVerticalCliRet(cli_ret)
        for intf in intfs:
            loc = intf.get("ID")
            health_status = intf.get("Health Status").lower()
            running_status = intf.get("Running Status").lower()
            if loc == intf_id and (health_status != "normal"
                                   or running_status != "running"):
                return True
        return False

    def check_multi_bond_4_intf(self):

        try:
            if self.is_fault_intf():
                self.logger.info("The interface is fault.")
                return True, "", ""

            sel_row = BaseFactory.json.toDict(
                self.context.get(self.select_id))
            intf_id = sel_row.get("location")
            result = self.check_model_version()
            if not result:
                # 不涉及问题版本通过
                return True, "", ""

            flag, err_bonds = self.get_multi_bond_4_intf(intf_id)
            if not flag:
                err_msg, sug_msg = common.getMsg(self.lang, "get.bond.fail")
                return False, err_msg, sug_msg

            self.logger.info("The cross-card bonds result:%s" % str(err_bonds))
            if err_bonds:
                if self.sug_patch:
                    err_msg, sug_msg = \
                        common.getMsg(self.lang,
                                      "check.bond.patch.fail",
                                      suggestionArgs=self.sug_patch)
                else:
                    err_msg, sug_msg = \
                        common.getMsg(self.lang,
                                      "check.bond.no.patch.fail")
                return False, err_msg, sug_msg
            return True, "", ""
        except Exception as e:
            self.logger.error("check cross-card bonds error:%s" % str(e))
            err_msg, sug_msg = common.getMsg(self.lang, "common.exception")
            return False, err_msg, sug_msg
