# -*- coding: UTF-8 -*-
import re

REGEX_PATTERN = {
    "model_for_CE8850-SAN": re.compile(r'CE8850-SAN', re.I),
    "model": re.compile(r'CE\d{4}-|CE\d{4}E-', re.I),
    "version": re.compile(r'V\d{3}R\d{3}C\d{2}(SPC\d{3})*', re.I),
    "patch": re.compile(r'V\d{3}R\d{3}(C\d{2})*SPH\d{3}', re.I),
}


class Switch:

    def __init__(self, cli, logger):
        self.cli = cli
        self.logger = logger
        self.cmd_ret_cache = dict()
        self.model = ""
        self.version = ""
        self.patch = ""
        self.query_switch_info()

    def query_switch_info(self):
        """
        初始化交换机信息
        :return:
        """
        try:
            cmd = "display version"
            cli_ret = self.get_cmd_cli_ret(cmd)
            self.model = self.get_regex_info(
                cli_ret, "model", REGEX_PATTERN.get("model_for_CE8850-SAN"))
            if self.model == '--':
                model = self.get_regex_info(
                    cli_ret, "model", REGEX_PATTERN.get("model"))
                self.model = model[0:-1]
            self.logger.logInfo("switch model: {}".format(self.model))

            self.version = self.get_regex_info(
                cli_ret, "version", REGEX_PATTERN.get("version"))
            self.patch = self.get_regex_info(
                cli_ret, "patch", REGEX_PATTERN.get("patch"))
            self.logger.logInfo("switch version: {}".format(self.version))
            self.logger.logInfo("switch patch: {}".format(self.patch))

        except Exception as ex:
            self.logger.logException(ex)

    def get_regex_info(self, cli_ret, info_type, pattern):
        """从cli回显中获取交换机版本 V999R999C99SPC999

        :param cli_ret: cli回显
        :param info_type: 信息类型
        :param pattern: 正则搜索pattern
        :return:
        """
        ret = pattern.search(cli_ret)
        if ret:
            return ret.group(0)
        else:
            return "--"

    def get_cmd_cli_ret(self, cmd, refresh=False):
        """执行cmd获取回显

        :param cli: cli连接
        :param cmd: 命令
        :param refresh: 是否再次执行命令刷新缓存
        :return:
        """
        cli_ret = self.cmd_ret_cache.get(cmd, None)
        self.logger.logInfo("cli_ret from cache: %s" % cli_ret)
        self.logger.logInfo("cache: %s" % str(self.cmd_ret_cache))
        # 缓存中有命令的回显或者明确要求刷新时才实际执行命令
        if not refresh and cli_ret:
            return cli_ret

        res = self.cli.execCmd(cmd)
        cli_ret = res
        cli_ret += self._get_more_cli_ret(res)

        self.cmd_ret_cache[cmd] = cli_ret
        return cli_ret

    def _get_more_cli_ret(self, res):
        """有更多信息提示时 获取更多信息

        :return:
        """
        i = 0
        more_cli_ret = ""
        while "---- More ----" in res:
            # 防呆 获取超过512次后强制退出
            if i > 512:
                self.cli.execCmd("#")
                self.cli.execCmd("#")
                break
            res = self.cli.execCmd("\n")
            more_cli_ret += res
            i += 1
        return more_cli_ret

    def check_model(self, requirement):
        """检查交换机的版本型号是否符合要求

        :param requirement: 要求的型号、版本、补丁
        :return:
        """
        try:
            self.logger.logInfo("switch model: {}".format(self.model))
            required_model = requirement.keys()

            if required_model and self.model not in required_model:
                return False

            return True
        except Exception as ex:
            self.logger.logException(ex)
            return False

    def check_version_patch(self, requirement):
        """检查交换机的版本版本、补丁是否符合要求

        :param requirement: 要求的型号、版本、补丁
        :return:
        """
        required_version_and_patch = requirement.get(self.model)
        if not required_version_and_patch:
            return False
        # 版本大于需求版本，检查通过
        if self.version > required_version_and_patch[0]:
            return True
        # 版本小于需求版本，检查不通过
        if self.version < required_version_and_patch[0]:
            return False
        # 版本相等时，补丁小于需求补丁，不通过
        if required_version_and_patch[1] and \
                self.patch < required_version_and_patch[1]:
            return False

        return True

    def get_model(self):
        """检查交换机的版本型号是否符合要求

        :return:
        """
        return self.model
