#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2021. All rights reserved.

"""
@time: 2020/01/23
@file: execute_on_all_controllers.py
@function: 在所有控制器上执行操作
"""
from com.huawei.ism.tool.obase.utils import DeviceTypeUtil

from cbb.frame.context import contextUtil
from cbb.frame.context.contextUtil import get_base_dev
from cbb.frame.context.contextUtil import createCliConnection
from cbb.frame.base import logger as Logger
from cbb.frame.base.product import Feature
from cbb.frame.base.baseUtil import group_by
from cbb.frame.cli.cliUtil import getProductModel
from cbb.frame.cli.cliUtil import getDevVersion
from cbb.frame.cli.cliUtil import closeCliConnection
from cbb.frame.cli.cliUtil import getMsg
from cbb.frame.cli.cliUtil import exit_heart_beat_mini_sys
from cbb.frame.cli.cliUtil import ssh_remote_ctrl_even_mini_sys
from cbb.frame.cli.cliUtil import getControllerEngineTopography
from cbb.frame.cli.cliUtil import excuteCmdInCliMode
from cbb.frame.cli.cliUtil import getHorizontalCliRet
from cbb.frame.cli.mini_util import get_horizontal_mini_system_ret
from cbb.frame.cli.cliUtil import excuteCmdInMinisystemModel
from cbb.frame.cli.cliUtil import testIsInMinisystemMode
from cbb.frame.cli.cli_helper import exec_mini_system
from cbb.frame.dsl.adapter import get_vertical_ret
from cbb.frame.cli import cliUtil
import re

CLI_STATE_STACK = []


def _parse_logger_info(log, context):
    """
    解析获取日志类
    :param context: 原始工具上下文
    :return: 日志类
    """
    if log is not None:
        return Logger.Logger(log, __file__)
    logger = contextUtil.getLogger(context)
    if logger is not None:
        return Logger.Logger(logger, __file__)
    raise ExeOnAllCtrlException("The context's logger is None.")


def _parse_lang_info(context):
    """
    解析获取语言信息
    :param context: 原始工具上下文
    :return: 语言
    """
    lang = contextUtil.getLang(context)
    if lang is None:
        raise ExeOnAllCtrlException("The context's lang is None.")
    return lang


def ExecuteOnAllControllers(func):
    def inner(*params):
        return ExecuteOnAllControllerInner(func).__call__(*params)

    return inner


class ExecuteOnAllControllerInner:
    """
    遍历控制器类装饰器
    1.支持遍历所有控制器（支持提前终止：设置函数返回FuncResult中need_continue_exe
    为False)
    2.支持单个目标控制器（ExeOnAllCtrlContext中的set_target_engine_ctrl方法，
    默认为主控）
    """

    def __init__(self, func):
        """
        :param func:被装饰函数
        """
        self._func = func
        self._context = None
        self._con_manager = None
        self._need_continue_exe_func = True
        self.__origin_context = None

    def __call__(self, *params):
        """
        装饰器入口
        :param context(ExeOnAllCtrlContext): 遍历控制器上下文
        :return: 执行汇总结果
        """
        self.__origin_context = params
        if len(params) == 1:
            self._context = params[0]
        else:
            self._context = params[1]
        self._context.logger.info("enter ExecuteOnAllControllerInner.")
        if not isinstance(self._context, ExeOnAllCtrlContext):
            raise ExeOnAllCtrlException("The Context of Execute On All "
                                        "Controller is not correct type.")
        self._con_manager = ConnectionManager(
            self._context.lang, self._context.logger,
            self._context.original_context)
        try:
            # 对所有引擎执行操作函数
            exe_result_dict = self._exe_all_engine_func()
        except ExeOnAllCtrlException as exception:
            self._context.logger.error("Exe on all ctrl occur exception:{}.".
                                       format(exception.exception_msg))
            raise exception
        except Exception as exception:
            self._context.logger.error("Exe on all ctrl occur exception:{}.".
                                       format(exception.message))
            raise ExeOnAllCtrlException(exception.message)
        finally:
            cli_ret = self._con_manager.exit_all_connection()
            self._context.append_cli_ret(cli_ret)
        return self._summary_exe_result(exe_result_dict)

    def create_cli_to_controller(self):
        """
        创建直连控制器cli连接
        """
        # 如果是svp设备，遍历所有控制ip，创建cli连接
        engine_ip_dict = self._context.dev_info.engine_ip_dict
        for engine_id in engine_ip_dict:
            engine_ip_list = engine_ip_dict.get(engine_id)
            # 遍历某一引擎的所有控制器ip
            for ip in engine_ip_list:
                cli, cli_ret = self._con_manager.create_cli_connection(ip)
                if cli is not None:
                    self._context.dev_info.cli = cli
                    self._context.logger.info("Create cli for svp successful")
                    self._context.append_cli_ret(cli_ret)
                    self._context.update_current_controller_info(cli)
                    return
        self._context.logger.error("Create cli for svp failed")
        failed_cli_ret = "Failed to create cli connection to ctrl for svp"
        self._context.append_cli_ret(failed_cli_ret)

    def _exe_all_engine_func(self):
        """
        对所有引擎执行被装饰函数
        :return: 结果
        """
        result = dict()
        # 先执行当前引擎
        cur_engine_id = self._context.dev_info.current_engine
        self._context.logger.info(
            "engine_ctrl_dict={}".format(
                self._context.dev_info.engine_ctrl_dict))
        result[cur_engine_id] = self._judge_link_ways_to_exe_engine_func(
            cur_engine_id)
        if not self._need_continue_exe_func:
            return result
        for engine_id in self._context.dev_info.engine_ctrl_dict:
            if engine_id == cur_engine_id:
                continue
            result[engine_id] = self._judge_link_ways_to_exe_engine_func(
                engine_id)
            if not self._need_continue_exe_func:
                break
        return result

    def _judge_link_ways_to_exe_engine_func(self, engine_id):
        """
        判断去连接引擎的方式
        :param engine_id: 需要执行操作的引擎ID
        :return: 引擎执行结果
        """
        # 若是选择了目标控制器，而当前需要执行的控制器不是目标控制器直接返回
        if not self._context.is_target_engine(engine_id):
            return

        if engine_id == self._context.dev_info.current_engine:
            cli = self._context.dev_info.cli
            result_dict = self._exe_one_engine_func(engine_id, cli)
        else:
            # 是否支持跨引擎ssh连接
            if self._context.dev_info.is_support_cross_engine_ssh():
                result_dict = self._create_ssh_to_exe_engine_func(engine_id)
            # 如果跨引擎ssh不支持，则创建cli进行跳转
            else:
                self._context.logger.info("using create cli to jump engine "
                                          "%s" % engine_id)
                result_dict = self._create_cli_to_exe_engine_func(engine_id)

            # 同一个引擎中的所有控制器均连接失败
            if len(result_dict) == 0:
                result_dict = self._create_ssh_to_exe_engine_func(engine_id)
                # 再用心跳试一次,可能是心跳异常或者判断跨引擎有问题
                if len(result_dict) == 0:
                    self._context.exec_result = ResultType.NOT_FINISHED
                    err_msg = getMsg(self._context.lang,
                                     "create.connect.failed",
                                     engine_id)
                    self._context.append_err_msg(err_msg)
        return result_dict

    def _create_cli_to_exe_engine_func(self, engine_id):
        """
        通过创建cli连接的方式连接引擎
        :param engine_id: 目标引擎的id
        :return: 目标引擎的执行的结果
        """
        engine_result_dict = dict()
        engine_ip_list = self._context.dev_info.engine_ip_dict.get(engine_id)
        if not engine_ip_list:
            return engine_result_dict
        for engine_ip in engine_ip_list:
            cli, cli_ret = self._con_manager.create_cli_connection(engine_ip)
            if cli is None:
                self._context.logger.info(
                    "Create cli to %s engine failed, engine ip:%s." % (
                        engine_id, engine_ip))
            else:
                # 缓存当前cli链接
                if self._context.original_context.get('cli', None):
                    self._context.original_context['temp_cli'] = self._context.original_context.get('cli')
                self._context.original_context['cli'] = cli
                try:
                    self._context.append_cli_ret(cli_ret)
                    # 保存当前控制器id
                    self._context.logger.info(
                        "Create cli to %s engine successful, engine ip:%s." % (
                            engine_id, engine_ip))
                    engine_result_dict = self._exe_one_engine_func(engine_id, cli)
                    cli_ret = self._con_manager.close_cli_connection(cli)
                    self._context.append_cli_ret(cli_ret)
                finally:
                    # 复原 原始cli缓存
                    self._context.original_context['cli'] = self._context.original_context.get('temp_cli', None)
                break
        return engine_result_dict

    def _find_opposite_engine_target_ctrl(self, engine_id):
        """
        计算需要连接的控制器节点，4+2控时会出现没有对端节点问题，需要跳转当前节点,
        到当前引擎下有对端节点的控制器节点。
        ssh连接到对端引擎，考虑最严格的场景直连组网，只能对端跳，也就是例如2引擎4控，
        0跳2的场景。
        :param engine_id: 目标引擎id
        :return: 对端目标节点id， ssh连接次数
        """
        dev_info = self._context.dev_info
        target_ctrl_list = dev_info.engine_ctrl_dict.get(engine_id)

        curr_ctrl_list = list(sorted(dev_info.engine_ctrl_dict.get(
            dev_info.current_engine)))
        # 先尝试当前所在控制器
        if not dev_info.current_ctrl:
            curr_ctrl_list.remove(dev_info.current_ctrl)
            curr_ctrl_list.insert(0, dev_info.current_ctrl)
        self._context.logger.info("current engine ctrls %s." %
                                  str(curr_ctrl_list))
        # 遍历当前引擎所有控制器，计算其对端节点是否在目标引擎中
        for ctrl_id in curr_ctrl_list:
            target_ctrl_id = self._get_opposite_ctrl_id(ctrl_id, engine_id)
            if str(target_ctrl_id) in target_ctrl_list:
                if ctrl_id == dev_info.current_ctrl:
                    return target_ctrl_id, 0
                result = self._ssh_to_other_ctrl(dev_info.cli, ctrl_id)
                if result.result_type == ResultType.SUCCESS:
                    self._context.append_cli_ret(result.cli_ret)
                    return target_ctrl_id, 1
        self._context.logger.error("Not find ctrl id able to jump engine.")
        return None, 0

    def _create_ssh_to_exe_engine_func(self, engine_id):
        """
        根据计算的可达目标引擎控制器，进行连接，执行引擎操作
        :param engine_id: m目标引擎
        :return: 目标引擎操作结果
        """
        # 根据计算的对端控制器id进行跨引擎跳转
        engine_result = dict()
        self._context.logger.info(
            "using ssh to jump engine %s" % engine_id)
        # 计算可连接的对端节点
        target_ctrl_id, create_num = \
            self._find_opposite_engine_target_ctrl(engine_id)
        self._context.logger.info("jump engine target ctrl id %s" %
                                  target_ctrl_id)
        # 通过ssh方式连接引擎
        if target_ctrl_id is None:
            return engine_result

        cli = self._context.dev_info.cli
        ssh_result = self._ssh_to_other_ctrl(cli, target_ctrl_id, True)
        self._context.append_cli_ret(ssh_result.cli_ret)
        if ssh_result.result_type == ResultType.SUCCESS:
            create_num += 1
            engine_result = self._exe_one_engine_func(engine_id, cli)

        # 退出连接
        for count in range(create_num):
            cli_ret = self._con_manager.exit_ssh_connection(cli)
            self._context.append_cli_ret(cli_ret)
        return engine_result

    def _get_opposite_ctrl_id(self, ctrl_id, engine_id):
        """
        根据当前控制器id和目标引擎id计算到该引擎的对端控制器节点id
        :param ctrl_id: 当前引擎某控制器id
        :param engine_id: 目标引擎id
        :return: 对端控制器节点id
        """
        max_ctrl_num = self._get_engine_max_ctrl_num()
        ctrl_id = int(ctrl_id) % max_ctrl_num + int(engine_id) * max_ctrl_num
        return str(ctrl_id)

    def _get_engine_max_ctrl_num(self):
        """
        获取所有引擎的最大控制器数
        :return: 最大控制器数
        """
        max_ctrl_num = 0
        dev_info = self._context.dev_info
        engine_ctrl_dict = dev_info.engine_ctrl_dict
        for engine_id in engine_ctrl_dict:
            ctrl_num_list = engine_ctrl_dict.get(engine_id, list())
            max_ctrl_num = max(len(ctrl_num_list), max_ctrl_num)
        return max_ctrl_num

    def _exe_one_engine_func(self, engine_id, cli):
        """
        对该引擎的所有控制器执行操作
        :param engine_id: 该引擎ID
        :return: 执行结果
        """
        dev_info = self._context.dev_info
        old_cur_ctrl = dev_info.current_ctrl
        # 更新当前控制器信息
        if engine_id != dev_info.current_engine:
            self._context.update_current_controller_info(cli)
        result_dict = dict()
        # 先执行当前控制器
        cur_ctrl = dev_info.current_ctrl
        result_dict[cur_ctrl] = self._exe_one_controller_func(cur_ctrl, cli)
        self._context.logger.info("result_dict={}".format(result_dict))
        if not self._need_continue_exe_func:
            self._context.logger.info("not need continue")
            return result_dict
        ctrl_dict = dev_info.engine_ctrl_dict.get(engine_id)
        self._context.logger.info("ctrl_dict={}".format(ctrl_dict))
        # 极端开工失败场景，没匹配到其他控制器信息的情况
        if not ctrl_dict:
            return result_dict
        for ctrl in ctrl_dict:
            # 先清除当前cli ret list
            self._context.clear_current_cli_ret()
            if ctrl == cur_ctrl:
                continue
            result_dict[ctrl] = self._exe_one_controller_func(ctrl, cli)
            if not self._need_continue_exe_func:
                break
        # 还原当前控制器id
        dev_info.current_ctrl = old_cur_ctrl
        return result_dict

    def _exe_one_controller_func(self, ctrl_id, cli):
        """
        对该控制器执行操作
        :param ctrl_id: 该控制器ID
        :param cli: cli连接
        :return:执行结果
        """
        # 若是选择了目标控制器，而当前需要执行的控制器不是目标控制器直接返回
        self._context._cur_ctrl_id = ctrl_id
        if not self._context.is_target_ctrl(ctrl_id):
            return
        func_result = FuncResult()
        if ctrl_id != self._context.dev_info.current_ctrl:
            ssh_result = self._ssh_to_other_ctrl(cli, ctrl_id)
            self._context.append_cli_ret(ssh_result.cli_ret)
            if ssh_result.result_type != ResultType.SUCCESS:
                func_result.err_msg = ssh_result.err_msg
                func_result.result_type = ssh_result.result_type
            else:
                func_result = self._exe_decorated_func(cli)
                cli_ret = self._con_manager.exit_ssh_connection(cli)
                self._context.append_cli_ret(func_result.cli_ret)
                self._context.append_cli_ret(cli_ret)
        else:
            func_result = self._exe_decorated_func(cli)
            self._context.append_cli_ret(func_result.cli_ret)

        self._context.exec_result = func_result.result_type
        # 错误信息中拼装控制器id
        if func_result.err_msg:
            err_msg = "Controller [id:%s]: %s" % (ctrl_id, func_result.err_msg)
            self._context.append_err_msg(err_msg)
        return func_result

    def _exe_decorated_func(self, cli):
        """
        执行连接需要执行的功能函数
        :return: 处理后的结果
        """
        old_cli = self._context.dev_info.cli
        self._context.dev_info.cli = cli
        func_result = self._func(*self.__origin_context)

        self._context.dev_info.cli = old_cli
        if not isinstance(func_result, FuncResult):
            raise ExeOnAllCtrlException("Func return type is not FuncResult.")
        if not ResultType.is_self_result_type(func_result.result_type):
            raise ExeOnAllCtrlException(
                "Func return result_type is not ResultType.")
        self._need_continue_exe_func = func_result.need_continue_exe
        self._context.logger.info(
            "Need to continue exe? %s" % self._need_continue_exe_func)
        return func_result

    def _ssh_to_other_ctrl(self, cli, controller_id, is_cross_engine=False):
        """
        连接到同一引擎下其他控制器
        :param cli: 引擎cli连接
        :param controller_id: 连接目标控制器ID
        :param is_cross_engine: 是否是跨引擎ssh
        :return: 结果
        """
        base_dev_node = self._context.dev_info.base_dev
        password = base_dev_node.getLoginUser().getPassword()
        attempt_cmd_list = self._get_attempt_ssh_cmd_list(controller_id,
                                                          is_cross_engine)
        ssh_result = self._attempt_all_ssh_cmd(cli, password, attempt_cmd_list,
                                               controller_id)
        # 删除本地密码
        del password
        return ssh_result

    def _get_attempt_ssh_cmd_list(self, controller_id, is_cross_engine):
        """
        获取需要尝试的ssh命令
        :param controller_id: 目标控制器
        :param is_cross_engine: 是否跨引擎
        :return: 需要尝试的shh命令列表
        """
        attempt_cmd_list = list()
        attempt_cmd_list.append("sshtoremoteExt %s" % str(controller_id))
        if not is_cross_engine:
            new_ctrl_id = self._trans_to_engine_inside_ctrl_id(controller_id)
            attempt_cmd_list.append("sshtoremoteExt %s" % str(new_ctrl_id))
        attempt_cmd_list.append("sshtoremote")
        return attempt_cmd_list

    def _attempt_all_ssh_cmd(self, cli, password, attempt_cmd_list, ctrl_id):
        """
        尝试所有ssh命令
        :param cli: cli连接
        :param password: 密码
        :param attempt_cmd_list: 需要尝试的ssh命令
        :param ctrl_id: 目标控制器id
        :return: 尝试结果
        """
        ssh_result = FuncResult()
        for cmd in attempt_cmd_list:
            is_success, cli_ret, err_msg = \
                self._con_manager.ssh_to_remote_ctrl(cli, cmd, password,
                                                     ctrl_id)
            ssh_result.append_cli_ret(cli_ret)
            if is_success:
                self._context.logger.info("Ssh to %s controller successful." %
                                          ctrl_id)
                return ssh_result
            ssh_result.err_msg = err_msg
        self._context.logger.error("Ssh to %s controller failed." % ctrl_id)
        ssh_result.result_type = ResultType.NOT_FINISHED
        return ssh_result

    def _had_ssh_to_target_ctrl(self, cli, target_ctrl):
        """
        是否已经ssh到目标控制器
        :param cli: cli连接
        :param target_ctrl: 目标控制器
        :return: 是否
        """
        is_target_ctrl = False
        is_success, cli_ret, err_msg, engine_controller_info_tuple = \
            getControllerEngineTopography(cli, self._context.lang)
        self._context.append_cli_ret(cli_ret)
        if is_success:
            is_target_ctrl = target_ctrl == engine_controller_info_tuple[1]
        self._context.logger.info("Had ssh to target ctrl?: %s." %
                                  is_target_ctrl)
        return is_target_ctrl

    def _trans_to_engine_inside_ctrl_id(self, controller_id):
        """
        转换控制器id为引擎内部控制器id，例如该引擎4控，改控制器id为5,转换为1
        :param controller_id: 当前控制器id
        :return: 转换后的新的控制器id
        """
        engine_ctrl_dict = self._context.dev_info.engine_ctrl_dict
        for engine_id in engine_ctrl_dict:
            ctrl_list = engine_ctrl_dict.get(engine_id)
            if controller_id in ctrl_list:
                engine_ctrl_num = len(ctrl_list)
                new_controller_id = int(controller_id) % engine_ctrl_num
                self._context.logger.info(
                    "trans oldctrl and newctrl: %s %s" % (
                        controller_id, new_controller_id))
                return str(new_controller_id)
        return controller_id

    def _summary_exe_result(self, exe_result_dict):
        """
        汇总所有执行结果
        :param exe_result_dict: 执行结果
        :return: 汇总结果
        """
        total_result = FuncResult()
        total_result.cli_ret = self._context.cli_ret
        total_result.err_msg = self._context.err_msg
        total_result.result_type = self._context.exec_result
        total_result.other_result = exe_result_dict
        return total_result


class ExeOnAllCtrlContext(object):
    """
    遍历控制器执行上下文
    """

    def __init__(self, context, cli=None, log=None, dev_type=None,
                 dev_ver=None):
        self.logger = self._parse_logger_info(log, context)
        self.lang = self._parse_lang_info(context)

        self._target_ctrl_dict = dict()
        self._have_target_ctrl = False
        self._cli_ret_list = list()
        # 当前命令返回，仅在当前执行函数期间有效，每次跳控制器前会被clear掉
        self._current_cli_ret_list = list()
        self._err_msg_list = list()
        self._exec_result = ResultType.SUCCESS
        self._original_context = context
        self._dev_info = self._parse_dev_info(cli, context, dev_type, dev_ver)
        self._cur_ctrl_id = ""
        self.info = None
        self.params_map = {}

    def _parse_dev_info(self, cli, context, dev_type, dev_ver):
        """
        解析生成设备信息
        :param cli: cli连接
        :param context: 原始上下文
        :param dev_type: 设备型号
        :param dev_ver: 设备版本
        :return: dev_info(ExeOnAllCtrlDev)
        """
        try:
            base_dev = get_base_dev(context)
        except Exception as exception:
            raise ExeOnAllCtrlException(exception.message)
        cli = self._parse_cli_info(cli, context)
        dev_info = ExeOnAllCtrlDev(cli, base_dev, context)
        try:
            self._set_dev_type(dev_type, base_dev, dev_info)
            self._set_dev_ver(dev_ver, base_dev, dev_info)
        except Exception as e:
            self.logger.error('error occurs when try to query dev version:{}'.
                              format(e.exception_msg))
        mini_sys_flag = testIsInMinisystemMode(cli)
        parse_result, cli_ret, err_msg = dev_info.parse_engine_controller_info(
            self.lang, self.logger)
        if mini_sys_flag:
            dev_info.parse_control_info_in_mini_system(self.lang, self.logger)
        else:
            cliUtil.enterCliModeFromSomeModel(cli, self.lang)
        self.append_cli_ret(cli_ret)
        self.append_err_msg(err_msg)
        self.exec_result = parse_result

        cli_ret = dev_info.parse_engine_ip_info(self.lang, self.logger)
        self.append_cli_ret(cli_ret)
        return dev_info

    def _set_dev_type(self, dev_type, base_dev, dev_info):
        """
        设置设备型号
        :param dev_type: 设备型号
        :param base_dev: 上下文中设备基础信息
        :param dev_info: 遍历控制器设备信息（ExeOnAllCtrlDev）
        """
        if dev_type is None:
            if hasattr(base_dev, "getDeviceModel"):
                dev_type = base_dev.getDeviceModel()
            if not dev_type:
                dev_type = dev_info.query_dev_type(self.lang, self.logger)
        dev_info.dev_type = dev_type

    def _set_dev_ver(self, dev_ver, base_dev, dev_info):
        """
        设置设备型号
        :param dev_ver: 设备版本
        :param base_dev: 上下文中设备基础信息
        :param dev_info: 遍历控制器设备信息（ExeOnAllCtrlDev）
        """
        if dev_ver is None:
            if hasattr(base_dev, "getProductVersion"):
                dev_ver = base_dev.getProductVersion()
            if not dev_ver:
                dev_ver = dev_info.query_dev_ver(self.lang, self.logger)
        dev_info.dev_ver = dev_ver

    def _parse_logger_info(self, log, context):
        """
        解析获取日志类
        :param context: 原始工具上下文
        :return: 日志类
        """
        if log is not None:
            return Logger.Logger(log, __file__)
        logger = contextUtil.getLogger(context)
        if logger is not None:
            return Logger.Logger(logger, __file__)
        raise ExeOnAllCtrlException("The context's logger is None.")

    def _parse_lang_info(self, context):
        """
        解析获取语言信息
        :param context: 原始工具上下文
        :return: 语言
        """
        lang = contextUtil.getLang(context)
        if lang is None:
            raise ExeOnAllCtrlException("The context's lang is None.")
        return lang

    def _parse_cli_info(self, cli, context):
        """
        解析获取设备cli连接
        :param cli: 输入的cli连接
        :param context: 原始工具上下文
        :return: cli连接
        """
        if cli is not None:
            return cli
        cli = contextUtil.getCliCommon(context)
        if cli is not None:
            return cli
        # fru
        cli = contextUtil.get_new_cli(context)
        if cli is not None:
            return cli
        raise ExeOnAllCtrlException("The cli and context's cli is None.")

    def is_target_engine(self, engine_id):
        """
        是否是目标引擎
        :param engine_id: 目标引擎id
        :return: True/False
        """
        if self._have_target_ctrl:
            if engine_id in self._target_ctrl_dict:
                return True
            return False
        return True

    def is_target_ctrl(self, controller_id):
        """
        是否是目标控制器
        :param controller_id: 目标控制器id
        :return: True/False
        """
        if self._have_target_ctrl:
            for engine_id in self._target_ctrl_dict:
                if controller_id in self._target_ctrl_dict.get(engine_id):
                    return True
            return False
        return True

    def set_target_engine_ctrl(self, engine_id=None, ctrl_id=None):
        """
        设置需只要执行的目标控制器（默认master控）
        :param engine_id: 目标控制器所在引擎
        :param ctrl_id:  目标控制器
        """
        if engine_id is None or ctrl_id is None:
            engine_id, ctrl_id = self._dev_info.get_master_ctrl(self.lang)
            self.logger.info("Set master ctrl as target ctrl.")
        self._have_target_ctrl = True
        engine_id, ctrl_id = str(engine_id), str(ctrl_id)
        target_engine_ctrl_list = self._target_ctrl_dict.get(engine_id, list())
        if ctrl_id not in target_engine_ctrl_list:
            target_engine_ctrl_list.append(ctrl_id)
        self._target_ctrl_dict[engine_id] = target_engine_ctrl_list

    def set_target_ctrl_dict(self, target_ctrl_dict):
        """
        :param target_ctrl_dict: 目标控制器字典
        """
        self._have_target_ctrl = True
        self._target_ctrl_dict = target_ctrl_dict

    def append_cli_ret(self, cli_ret):
        self._cli_ret_list.append(cli_ret)
        self._current_cli_ret_list.append(cli_ret)

    def append_err_msg(self, err_msg):
        self._err_msg_list.append(err_msg)

    def clear_current_cli_ret(self):
        self._current_cli_ret_list = list()

    @property
    def cli_ret(self):
        return "\n\n".join(self._cli_ret_list)

    @property
    def current_cli_ret(self):
        return "\n\n".join(self._current_cli_ret_list)

    @property
    def err_msg(self):
        return "\n".join(self._err_msg_list)

    @property
    def exec_result(self):
        return self._exec_result

    @exec_result.setter
    def exec_result(self, result):
        """
        设置结果（取最大优先级的结果）
        :param result: 结果
        """
        if not ResultType.is_self_result_type(result):
            raise ExeOnAllCtrlException(
                "Wrong exec result value: {}.".format(result))
        if result > self._exec_result:
            self._exec_result = result

    @property
    def dev_info(self):
        return self._dev_info

    @property
    def target_ctrl_dict(self):
        return self._target_ctrl_dict

    @property
    def original_context(self):
        return self._original_context

    @property
    def cur_ctrl_id(self):
        return self._cur_ctrl_id

    def update_current_controller_info(self, cli):
        self._dev_info.update_current_controller_info(cli, self.lang,
                                                      self.logger)

    def is_svp_dev(self):
        return self._dev_info.is_svp_dev(self.logger)

    def set_param(self, key, val):
        self.params_map[key] = val

    def get_param(self, key):
        if key in self.params_map:
            return self.params_map[key]
        return None


class ExeOnAllCtrlDev(object):
    """
    遍历控制器设备信息
    """

    def __init__(self, cli, base_dev, context):
        self._cli = cli
        self._base_dev = base_dev
        self._dev_type = None
        self._dev_ver = None

        self._current_engine_id = None
        self._current_controller_id = None
        self._engine_controller_dict = dict()
        self._engine_ip_dict = dict()
        self._one_engine_ctrl_count = 1
        self._context = context
        self._lang = _parse_lang_info(self._context)
        self._log = _parse_logger_info(None, self._context)

    def query_dev_type(self, lang, logger):
        """
        查询设备型号
        :param lang: 语言
        :param logger: 日志
        :return: 设备型号
        """
        is_success, dev_type, err_msg = getProductModel(self._cli, lang)
        if not is_success:
            logger.error("Query dev type failed.")
            return '--'
        logger.info("Query dev type successful.")
        return dev_type

    def query_dev_ver(self, lang, logger):
        """
        查询设备版本
        :param lang: 语言
        :param logger: 日志
        :return: 设备版本
        """
        is_success, dev_version, err_msg = getDevVersion(self._cli, lang,
                                                         logger)
        if not is_success:
            logger.error("Query dev version failed.")
            raise ExeOnAllCtrlException(err_msg)
        logger.info("Query dev version successful.")
        return dev_version

    def parse_engine_controller_info(self, lang, logger):
        """
        解析设备引擎控制器信息
        :param lang: 语言
        :param logger: 日志
        :return: 解析结果
        """
        parse_result = ResultType.SUCCESS
        flag, cli_ret, err_msg = self.parse_control_info_in_mini_system(
            lang, logger)
        if flag is not True:
            parse_result = ResultType.FAILED
        return parse_result, cli_ret, err_msg

    def parse_engine_ip_info(self, lang, logger):
        """
        解析引擎ip对应信息
        :param lang: 语言
        :param logger: 日志
        :return: 解析回显
        (解析失败不一定会导致遍历控制器执行异常，支持跨引擎不需要ip）
        """
        cmd = "show upgrade package"
        is_success, cli_ret, err_msg = excuteCmdInCliMode(self._cli, cmd, True,
                                                          lang)
        if not is_success:
            logger.error("There is an exception to get the management"
                         " port information.")
        else:
            self._set_all_engine_reachable_ip(cli_ret, logger)
        return cli_ret

    def _set_all_engine_reachable_ip(self, cli_ret, logger):
        """
        设置所有可达的ip
        :param cli_ret: 引擎ip查询回显
        :param logger: 日志
        """
        cli_ret_dict_list = list()
        end_index = cli_ret.find('HotPatch Version')
        # 没找到该字段
        if end_index != -1:
            cli_ret_dict_list = getHorizontalCliRet(cli_ret[:end_index])

        for info in cli_ret_dict_list:
            engine_id = info.get('Name').strip()[:1]
            ip = info.get('IP')
            engine_ip_list = self._engine_ip_dict.get(engine_id, list())
            engine_ip_list.append(ip)
            self._engine_ip_dict[engine_id] = engine_ip_list
        logger.info("[engine ip dict]:%s" % str(self._engine_ip_dict))
        # 解析回显失败
        if len(self._engine_ip_dict) == 0:
            logger.error("Parsing engine ip failed, can't find 'HotPatch"
                         " Version' from cliRet.")

    def parse_control_info_in_mini_system(self, lang, logger):
        """
        登录控制器直接进入小系统的情况下
        解析引擎和控制器id信息
        :param lang:
        :param logger:
        :return:
        """
        cmd = "showsysstatus"
        is_success, cli_ret, err_msg = excuteCmdInMinisystemModel(
            self._cli, cmd, True, lang)
        if not is_success:
            logger.error("fail to parse control info in mini system mode")
        else:
            self.update_control_engine_info_in_mini_sys(cli_ret)
        return is_success, cli_ret, err_msg

    def update_control_engine_info_in_mini_sys(self, cli_ret):
        vertical_res = cliUtil.getVerticalCliRet(cli_ret)[0]
        node_config = int(vertical_res.get('node cfg', '1'))
        self._current_controller_id = vertical_res.get('local node id', '0')
        # 尝试计算单引擎控制器数量
        ctrl_rows = cliUtil.getHorizontalCliRet(cli_ret)
        grouped_rows = group_by(ctrl_rows, 'engine')
        engine_max_ctrl = 2
        for item in grouped_rows.values():
            engine_max_ctrl = max(engine_max_ctrl, len(item))
        self._one_engine_ctrl_count = engine_max_ctrl if engine_max_ctrl % 2 == 0 else engine_max_ctrl + 1
        for engine_id in grouped_rows:
            self._engine_controller_dict[engine_id] = [x.get('id') for x in grouped_rows.get(engine_id)]
        login_engine = '0'
        for row in ctrl_rows:
            if self._current_controller_id == row.get('id'):
                login_engine = row.get('engine', '0')
                break
        if node_config > len(ctrl_rows):
            self._engine_controller_dict = \
                self._gen_ctrl_id_engine_map_mini_sys(
                    node_config, self._one_engine_ctrl_count)
        self._current_engine_id = login_engine

    def _gen_ctrl_id_engine_map_mini_sys(self, node_cfg, group_size):
        """
        小系统下，如果有节点不可见时，通过配置信息，生成引擎-控制器map
        :return:
        """
        engine2ctrl_id = {}
        for idx in range(node_cfg):
            engine_idx = str(int(idx / group_size))
            if engine_idx not in engine2ctrl_id:
                engine2ctrl_id[engine_idx] = []
            engine2ctrl_id[engine_idx].append(str(idx))
        return engine2ctrl_id

    def update_current_controller_info(self, cli, lang, logger):
        """
        更新当前控制器信息
        :param cli: cli连接
        :param lang: 语言
        :param logger: 日志
        """
        is_success, cli_ret, err_msg, engine_controller_info_tuple = \
            getControllerEngineTopography(cli, lang)
        if not is_success:
            logger.error("Update current controller information abnormal.")
            # 更新失败，遍历时无法识别当前控制器，
            # 当前控制器会跳当前控制器，不影响结果
            self._current_controller_id = None
        else:
            self._current_controller_id = engine_controller_info_tuple[1]

    def get_master_ctrl(self, lang):
        """
        获取主控的引擎，控制器id信息
        :param lang: 语言
        :return: 主控所在引擎id, 主控控制器id
        """
        engine_id, ctrl_id, ctrl_num, cur_ctrl_id = self.parses_node_and_engine(lang)
        if engine_id and ctrl_id:
            return engine_id, ctrl_id
        # 没有获取到节点信息，尝试Dorado V6 命令查找
        engine_id, ctrl_id, _, _ = self.parses_node_and_engine(lang, 'sys.sh showcnm')
        if engine_id and ctrl_id:
            return engine_id, ctrl_id

        # 没有获取到节点信息，或在当前的回显没有找到master，根据ctrl_num，重试跳到能够执行命令的设备上去
        # 直到找到mater节点的引擎id和控制器id
        for _ctrl in range(ctrl_num):
            if _ctrl == cur_ctrl_id:
                continue
            engine_id, ctr_id = self.get_ctrl_info_from_target_ctrl(_ctrl, lang)
            if engine_id and ctr_id:
                return engine_id, ctr_id

        raise ExeOnAllCtrlException("Parse master ctrl info failed.")

    def get_ctrl_info_from_target_ctrl(self, target_ctr, lang):
        try:
            engine_id, ctr_id = None, None
            if self.judge_jump_to_target_ctrl(target_ctr):
                engine_id, ctr_id, _, _ = self.parses_node_and_engine(lang)

            return engine_id, ctr_id
        finally:
            exit_heart_beat_mini_sys(self.cli, True, lang)

    def ssh_to_remote_ctrl(self, cli, cmd, password, target_ctrl):
        """
        ssh到目标控制器
        :param cli: cli连接
        :param cmd: ssh命令
        :param password: 登录密码
        :param target_ctrl: 目标控制器id
        :return: 跳转结果
        """
        is_success, ssh_cli_ret, err_msg = ssh_remote_ctrl_even_mini_sys(
            cli, cmd, password, self._lang)
        if is_success:
            self._log.info("Executing [%s] to ctrl[%s] successful."
                           % (cmd, target_ctrl))
            return True, ssh_cli_ret, err_msg
        self._log.info("Executing [%s] to ctrl[%s] failed."
                       % (cmd, target_ctrl))
        return False, ssh_cli_ret, err_msg

    def judge_jump_to_target_ctrl(self, ctrl_id):
        """
        判断是否能够跳到目标控制器
        :param ctrl_id: 目标控制器id
        :return:
        """
        attempt_cmd_list = self._get_attempt_ssh_cmd_list(ctrl_id, False)
        for a_cmd in attempt_cmd_list:
            is_success, cli_ret, err_msg = \
                self.ssh_to_remote_ctrl(self.cli, a_cmd,
                                        self._base_dev.getLoginUser().getPassword(), ctrl_id)
            if is_success:
                return True

        return False

    def parses_node_and_engine(self, lang, cmd='showsysstatus'):
        flag, cli_ret, _ = exec_mini_system(self._cli, cmd, lang, None)
        if not flag:
            return None, None, 0, None
        horizontal_ret = get_horizontal_mini_system_ret(cli_ret)
        vertical_ret = get_vertical_ret(cli_ret)
        ctrl_num = None
        cur_ctrl_id = None
        if vertical_ret:
            ctrl_num = int(vertical_ret.get('node cfg', 2))
            cur_ctrl_id = int(vertical_ret.get('local node id', 0))
        if not horizontal_ret:
            return None, None, ctrl_num, cur_ctrl_id

        for horizontal_info in horizontal_ret:
            role = horizontal_info.get("role", "")
            if "master" in role:
                engine_id = horizontal_info.get("engine")
                controller_id = horizontal_info.get("id")
                return engine_id, controller_id, ctrl_num, cur_ctrl_id
        return None, None, ctrl_num, cur_ctrl_id

    def is_current_controller(self, controller_id):
        return controller_id == self._current_controller_id

    def is_current_engine(self, engine_id):
        return engine_id == self._current_engine_id

    def is_svp_dev(self, logger):
        if DeviceTypeUtil.isDeviceHasSVPModule(self._dev_ver, self._dev_type):
            logger.info("The device is svp device.")
            return True
        logger.info("The device is not svp device.")
        return False

    def is_support_cross_engine_ssh(self):
        return Feature.is_support_cross_engine_ssh(self._dev_ver,
                                                   self._dev_type)

    def _get_attempt_ssh_cmd_list(self, controller_id, is_cross_engine):
        """
        获取需要尝试的ssh命令
        :param controller_id: 目标控制器
        :param is_cross_engine: 是否跨引擎
        :return: 需要尝试的shh命令列表
        """
        attempt_cmd_list = list()
        attempt_cmd_list.append("sshtoremoteExt %s" % str(controller_id))
        if not is_cross_engine:
            new_ctrl_id = self._trans_to_engine_inside_ctrl_id(controller_id)
            attempt_cmd_list.append("sshtoremoteExt %s" % str(new_ctrl_id))
        attempt_cmd_list.append("sshtoremote")
        return tuple(attempt_cmd_list)

    def _trans_to_engine_inside_ctrl_id(self, controller_id):
        """
        转换控制器id为引擎内部控制器id，例如该引擎4控，改控制器id为5,转换为1
        :param controller_id: 当前控制器id
        :return: 转换后的新的控制器id
        """
        for engine_id in self.engine_ctrl_dict:
            ctrl_list = self.engine_ctrl_dict.get(engine_id)
            if controller_id in ctrl_list:
                engine_ctrl_num = len(ctrl_list)
                new_controller_id = int(controller_id) % engine_ctrl_num
                self._context.logger.info(
                    "trans oldctrl and newctrl: %s %s" % (
                        controller_id, new_controller_id))
                return str(new_controller_id)
        return controller_id

    @property
    def dev_type(self):
        return self._dev_type

    @dev_type.setter
    def dev_type(self, dev_type):
        self._dev_type = dev_type

    @property
    def dev_ver(self):
        return self._dev_ver

    @dev_ver.setter
    def dev_ver(self, dev_ver):
        self._dev_ver = dev_ver

    @property
    def current_ctrl(self):
        return self._current_controller_id

    @current_ctrl.setter
    def current_ctrl(self, ctrl_id):
        self._current_controller_id = str(ctrl_id)

    @property
    def current_engine(self):
        return self._current_engine_id

    @property
    def engine_ctrl_dict(self):
        return self._engine_controller_dict

    @property
    def engine_ip_dict(self):
        return self._engine_ip_dict

    @property
    def one_engine_ctrl_count(self):
        return self._one_engine_ctrl_count

    @property
    def cli(self):
        return self._cli

    @cli.setter
    def cli(self, cli):
        self._cli = cli

    @property
    def base_dev(self):
        return self._base_dev


def cli_state_recorder(func):
    def inner(*params):
        CLI_STATE_STACK.append(testIsInMinisystemMode(params[1]))
        return func(*params)

    return inner


class ConnectionManager:
    """
    连接管理
    """

    def __init__(self, lang, logger, context):
        self._cli_list = list()
        self._ssh_list = list()
        self._cli_ip_dict = dict()
        self._lang = lang
        self._logger = logger
        self._context = context

    def create_cli_connection(self, ip):
        """
        创建cli连接
        :param ip: 目标ip
        :return: cli连接
        """
        cli = createCliConnection(self._context, ip)
        if cli is not None:
            self._cli_list.append(cli)
            self._cli_ip_dict[cli] = ip
            return cli, "Create cli[ip:{}] connection.".format(ip)
        return cli, ""

    def close_cli_connection(self, cli):
        """
        关闭cli连接
        :param cli: cli连接
        """
        if len(self._cli_list) == 0:
            return "Close cli connection."
        closeCliConnection(cli)
        ip = self._cli_ip_dict.get(cli)
        if cli in self._cli_list:
            self._cli_list.remove(cli)
        return "Close cli[ip:{}] connection.".format(ip)

    @cli_state_recorder
    def ssh_to_remote_ctrl(self, cli, cmd, password, target_ctrl):
        """
        ssh到目标控制器
        :param cli: cli连接
        :param cmd: ssh命令
        :param password: 登录密码
        :param target_ctrl: 目标控制器id
        :return: 跳转结果
        """
        is_success, ssh_cli_ret, err_msg = ssh_remote_ctrl_even_mini_sys(
            cli, cmd, password, self._lang)
        if is_success:
            self._ssh_list.append(cli)
            self._logger.info("Executing [%s] to ctrl[%s] successful."
                              % (cmd, target_ctrl))
            return True, ssh_cli_ret, err_msg
        self._logger.info("Executing [%s] to ctrl[%s] failed."
                          % (cmd, target_ctrl))
        return False, ssh_cli_ret, err_msg

    def exit_ssh_connection(self, cli):
        """
        退出目前所在ssh连接
        :param cli: cli连接
        :param exit_to_mini_sys
        """
        if len(self._ssh_list) == 0:
            return "Exit controller's ssh connection"
        if cli is not self._ssh_list[-1]:
            raise ExeOnAllCtrlException("Wrong cli connection of recently ssh")
        is_success, old_curr_ctrl, cli_ret = self._get_current_ctrl_id(cli)
        if not is_success:
            raise ExeOnAllCtrlException("Get current ctrl id failed.")
        # 退出
        exit_heart_beat_mini_sys(cli, CLI_STATE_STACK.pop(), self._lang)
        is_success, new_curr_ctrl, cli_ret = self._get_current_ctrl_id(cli)
        if not is_success:
            raise ExeOnAllCtrlException("Get current ctrl id failed.")
        if old_curr_ctrl == new_curr_ctrl:
            self._logger.error("exit [%s] ssh connection failed."
                               % old_curr_ctrl)
            if old_curr_ctrl is not None and new_curr_ctrl is not None:
                raise ExeOnAllCtrlException("exit ssh connection failed.")
        self._ssh_list.pop()
        return "Exit controller[id:{}] ssh connection.".format(old_curr_ctrl)

    def _had_ssh_to_target_ctrl(self, cli, target_ctrl):
        """
        是否到达目标控制器
        :param cli:
        :param target_ctrl:
        :return: True/False
        """
        is_target_ctrl = False
        is_success, current_ctrl, cli_ret = self._get_current_ctrl_id(cli)
        if is_success:
            is_target_ctrl = target_ctrl == current_ctrl
        self._logger.info("Had ssh to target ctrl?: %s." % is_target_ctrl)
        return is_target_ctrl, cli_ret

    def _get_current_ctrl_id(self, cli):
        """
        获取当前当前控制器id
        :param cli: cli连接
        :return: 执行是否成功，当前控制器id，执行回显
        """
        is_success, cli_ret, err_msg, engine_controller_info_tuple = \
            getControllerEngineTopography(cli, self._lang)
        current_ctrl = None
        if is_success:
            current_ctrl = engine_controller_info_tuple[1]
            self._logger.info("success to get current ctrl:%s" % current_ctrl)
            return is_success, current_ctrl, cli_ret
        else:
            self._logger.info("failed to get current ctrl:%s" % current_ctrl)
            return self._get_current_ctrl_id_mini_sys(cli)

    def _get_current_ctrl_id_mini_sys(self, cli):
        """
        小系统下获取当前控制器id
        :param cli:
        :return:
        """
        cmd = 'showsysstatus'
        flag, ret, error_msg = excuteCmdInMinisystemModel(cli, cmd, self._lang)
        current_ctrl = None
        if flag:
            current_ctrl = self._parse_ctrl_id_mini_sys(ret)
        return flag, current_ctrl, error_msg

    def _parse_ctrl_id_mini_sys(self, ret):
        p = re.compile('local node id\\s*:\\s*(\\d+)')
        for line in ret.splitlines():
            match_obj = p.match(line)
            if match_obj:
                return match_obj.group(1)
        return None

    def exit_all_connection(self):
        """
        退出所有连接
        """
        self._logger.info("exit all ssh connection:%s" % str(self._ssh_list))
        self._logger.info("exit all cli connection:%s" % str(self._cli_list))
        cli_ret_list = list()
        while len(self._ssh_list) > 0:
            cli_ret = self.exit_ssh_connection(self._ssh_list[-1])
            cli_ret_list.append(cli_ret)

        while len(self._cli_list) > 0:
            cli_ret = self.close_cli_connection(self._cli_list[-1])
            cli_ret_list.append(cli_ret)
        return "\n".join(cli_ret_list)


class ResultType:
    """
    遍历控制器结果类型
    """
    NOT_SUPPORT = 0
    SUCCESS = 1
    WARNING = 2
    FAILED = 3
    NOT_FINISHED = 4

    @staticmethod
    def get_all_type_list():
        type_list = list()
        type_list.append(ResultType.NOT_SUPPORT)
        type_list.append(ResultType.SUCCESS)
        type_list.append(ResultType.WARNING)
        type_list.append(ResultType.FAILED)
        type_list.append(ResultType.NOT_FINISHED)
        return type_list

    @staticmethod
    def is_self_result_type(result_type):
        return result_type in ResultType.get_all_type_list()


class FuncResult:
    """
    遍历控制器执行结果信息
    """

    def __init__(self, result_type=ResultType.SUCCESS, cli_ret="", err_msg="",
                 need_continue_exe=True, other_result=None):
        self.result_type = result_type
        self.cli_ret = cli_ret
        self.err_msg = err_msg
        self.need_continue_exe = need_continue_exe
        self.other_result = other_result

    def append_cli_ret(self, cli_ret):
        self.cli_ret = "\n".join([self.cli_ret, cli_ret])


class ExeOnAllCtrlException(Exception):
    """
    遍历控制器异常
    """

    def __init__(self, exception_msg):
        self.exception_msg = exception_msg
